/*
 * Copyright (C) 2005-2006 Alex Murray <pragmatine@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SENSORS_APPLET_H
#define SENSORS_APPLET_H

#include <gtk/gtk.h>
#include <panel-applet.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

typedef struct _SensorsApplet SensorsApplet;
typedef struct _ActiveSensor ActiveSensor;

#include "prefs-dialog.h"

#define GRAPH_FRAME_EXTRA_WIDTH 6
#define SENSORS_APPLET_ICON "sensors-applet"

/* device icons */
typedef enum {
        CPU_ICON = 0,
        HDD_ICON,
        BATTERY_ICON,
        MEMORY_ICON,
        GPU_ICON,
        GENERIC_ICON,
        FAN_ICON,
        CASE_ICON,
        NUM_ICONS,
} IconType;

static const gchar * const stock_icons[NUM_ICONS] = {
        "sensors-applet-cpu",
        "sensors-applet-drive-harddisk",
        "sensors-applet-battery",
        "sensors-applet-memory",
        "sensors-applet-gpu",
        "sensors-applet-chip",
        "sensors-applet-fan",
        "sensors-applet-case"
};        

#define DEFAULT_ICON_SIZE 22

#define UNITS_CELSIUS "\302\260C"
#define UNITS_FAHRENHEIT "\302\260F"
#define UNITS_KELVIN ""
#define UNITS_RPM _("RPM")
#define UNITS_VOLTAGE  _("V")
#define UNITS_CURRENT _("A")


/* new interfaces need to be added to the bottom to not break existing
 * configs as cant change the value of SensorInterfaces stored in
 * gconf */
typedef enum {
	UNUSED = 0, /* as a flag to test against later */
	ACPI,
	HDDTEMP,
	I2C_SYS,
	I2C_PROC,
	IBM_ACPI,
	I8K,
	LIBSENSORS,
	OMNIBOOK,
	PMU_SYS,
	SMU_SYS,
	NVIDIA,
        SONYPI,
	N_SENSOR_INTERFACES
} SensorInterface;

/* new interfaces need to be added to the bottom to not break existing
 * configs */
static const gchar *sensor_interface[] = {
	"unused",
	"acpi",
	"hddtemp",
	"i2c-sys",
	"i2c-proc",
	"ibm-acpi",
	"i8k",
	"libsensors",
	"omnibook",
	"pmu-sys",
	"smu-sys",
	"nvidia",
        "sonypi",
};
#define CLAMP_SENSOR_INTERFACE_INDEX(n) CLAMP(n, 0, G_N_ELEMENTS(sensor_interface))

#define SENSORS_INTERFACE_STRING(n) sensor_interface[CLAMP_SENSOR_INTERFACE_INDEX(n)]

/* enumeration used to identify columns in the GtkTreeStore data
 * structure and to access specific gconf keys too.
 */
enum {
	PATH_COLUMN = 0,
	ID_COLUMN,
	LABEL_COLUMN,
	INTERFACE_COLUMN,
	SENSOR_TYPE_COLUMN,
	ENABLE_COLUMN,
	VISIBLE_COLUMN,
       	LOW_VALUE_COLUMN,
	HIGH_VALUE_COLUMN, 
	ALARM_ENABLE_COLUMN,
	LOW_ALARM_COMMAND_COLUMN,
        HIGH_ALARM_COMMAND_COLUMN,
	ALARM_TIMEOUT_COLUMN,
	MULTIPLIER_COLUMN,
	OFFSET_COLUMN,
	ICON_TYPE_COLUMN,
	ICON_PIXBUF_COLUMN,
        GRAPH_COLOR_COLUMN,
	N_COLUMNS
};


typedef enum {
	CURRENT_SENSOR = 0,
	FAN_SENSOR,
	TEMP_SENSOR,
	VOLTAGE_SENSOR
} SensorType;

/* for display mode */
typedef enum {
	DISPLAY_LABEL_WITH_VALUE = 0,
        DISPLAY_ICON_WITH_VALUE, 
	DISPLAY_VALUE,
	DISPLAY_ICON,
	DISPLAY_GRAPH
} DisplayMode;

typedef enum {
        VALUE_BESIDE_LABEL = 0,
        VALUE_BELOW_LABEL
} LayoutMode;

typedef enum {
        KELVIN = 0,
        CELSIUS,
        FAHRENHEIT
} TemperatureScale;

/* types of Notifs - low and high alarm warnings and error conditions*/
typedef enum {
        LOW_ALARM = 0,
        HIGH_ALARM,
        SENSOR_INTERFACE_ERROR,
        GCONF_READ_ERROR,
        GCONF_WRITE_ERROR,
        NUM_NOTIFS
} NotifType;

/* only always two type of alarms - may have more notif types */
#define NUM_ALARMS 2

/* should always return the value in degrees celcius, revolutions per
 * minute, volts, amps, etc */
typedef gdouble (*GetSensorValueFunction)(const gchar *path, 
					  const gchar *id, 
					  SensorType type,
					  GError **error);

struct _SensorsApplet {
	/* the actual applet for this instance */
	PanelApplet* applet;
        gint size;

	GtkTreeStore *sensors;
	GtkTreeSelection *selection;

	GetSensorValueFunction get_sensor_value[N_SENSOR_INTERFACES];

	guint timeout_id;
	/* preferences and about windows (if Gtk < 2.6)*/
	PrefsDialog *prefs_dialog;

	/* primary table to contain the panel dispay - we pack the
	 * list of labels and sensor values into this container */
	GtkWidget *table;
	GList *active_sensors;
	GtkTooltips *tooltips;
#ifdef HAVE_LIBNOTIFY
        NotifyNotification *notification;
#endif // HAVE_LIBNOTIFY
};



/* non-static function prototypes */
void sensors_applet_init(SensorsApplet *sensors_applet);
void sensors_applet_sensor_enabled(SensorsApplet *sensors_applet,
                                   GtkTreePath *path);
void sensors_applet_sensor_disabled(SensorsApplet *sensors_applet,
                                    GtkTreePath *path);
gboolean sensors_applet_update_active_sensors(SensorsApplet *sensors_applet);
void sensors_applet_register_sensors_interface(SensorsApplet *sensors_applet,
                                               SensorInterface interface,
					       GetSensorValueFunction get_sensor_value_function);

gboolean sensors_applet_add_sensor_full_details(SensorsApplet *sensors_applet,
                                                const gchar *path, 
						const gchar *id, 
						const gchar *label, 
						SensorInterface interface, 
						SensorType type, 
						gboolean enable,
						gdouble low_value,
						gdouble high_value,
						gboolean alarm_enable,
						const gchar *low_alarm_command,
						const gchar *high_alarm_command,
						gint alarm_timeout,
						gdouble multiplier,
						gdouble offset,
						IconType icon,
                                                const gchar *graph_color);

gboolean sensors_applet_add_sensor(SensorsApplet *sensors_applet,
                                   const gchar *path, 
				   const gchar *id, 
				   const gchar *label, 
				   SensorInterface interface, 
				   gboolean enable,
				   SensorType type,
				   IconType icon); 

/**
 * to be called by things like prefs dialog to turn off a sensor alarm
 */
void sensors_applet_alarm_off(SensorsApplet *sensors_applet,
                              GtkTreePath *path,
                              NotifType notif_type);
void sensors_applet_all_alarms_off(SensorsApplet *sensors_applet,
                              GtkTreePath *path);
void sensors_applet_icon_changed(SensorsApplet *sensors_applet,
                                 GtkTreePath *path);
void sensors_applet_update_sensor(SensorsApplet *sensors_applet,
                                  GtkTreePath *path);

void sensors_applet_display_layout_changed(SensorsApplet *sensors_applet);
void sensors_applet_reorder_sensors(SensorsApplet *sensors_applet);
gdouble sensors_applet_convert_temperature(gdouble value, 
                                           TemperatureScale old, 
                                           TemperatureScale new);
void sensors_applet_notify(SensorsApplet *sensors_applet,
                           NotifType notif_type);
void sensors_applet_notify_end(ActiveSensor *active_sensor, NotifType notif_type);
void sensors_applet_notify_end_all(SensorsApplet *sensors_applet);
void sensors_applet_notify_active_sensor(ActiveSensor *active_sensor, NotifType notif_type);
GdkPixbuf *sensors_applet_load_icon(IconType icon_type);
void sensors_applet_graph_size_changed(SensorsApplet *sensors_applet);

typedef void SensorsInterfaceTestSensorFunc(SensorsApplet *sensors_applet,
                                            const gchar *path);
void sensors_applet_find_sensors(SensorsApplet *sensors_applet,
                                 const gchar *path,
                                 SensorsInterfaceTestSensorFunc test_sensor);

#endif /* SENSORS_APPLET_H */
