#ifndef  LOG_BUFFER_SIZE
# define LOG_BUFFER_SIZE	65536
#endif
#ifndef  BOOT_LOGFILE
# define BOOT_LOGFILE		"/var/log/boot.msg"
#endif
#include <sys/time.h>
#include <sys/types.h> /* Defines the macros major and minor */
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <time.h>
#include <pty.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dirent.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <limits.h>
#include <errno.h>
#include <paths.h>
#include "libconsole.h"

/*
 * Internal logger
 */
static char *myname = NULL;
static void _logger (const char *fmt, va_list ap)
{
    char buf[strlen(myname)+2+strlen(fmt)+1];
    strcat(strcat(strcpy(buf, myname), ": "), fmt);
    vfprintf(stderr, buf, ap);
    return;
}

/*
 * Cry and exit.
 */
void error (const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    _logger(fmt, ap);
    va_end(ap);
    popd();
    exit (1);
}

/*
 * Warn the user.
 */
void warn (const char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    _logger(fmt, ap);
    va_end(ap);
    return;
}

/*
 * Remove pidfile
 */
static void rmfpid()
{
    char buf[strlen(_PATH_VARRUN)+strlen(myname)+1+3+1];
    strcat(strcat(strcpy(buf, _PATH_VARRUN), myname), ".pid");
    unlink(buf);
}

/*
 * Write pidfile
 */
static void pidfile()
{
    char buf[strlen(_PATH_VARRUN)+strlen(myname)+1+3+1];
    FILE * fpid;

    strcat(strcat(strcpy(buf, _PATH_VARRUN), myname), ".pid");
    if ((fpid = fopen (buf, "w")) == NULL) {
	warn("can not open %s: %s\n", buf, strerror(errno));
	goto out;
    }
    fprintf(fpid, "%d\n", (int)getpid());
    fclose(fpid);
    atexit(rmfpid);
out:
    return;
}

/*
 *  Signal handler
 */
static struct sigaction saved_sigttin;
static struct sigaction saved_sigttou;
static struct sigaction saved_sigtstp;
static struct sigaction saved_sighup;
static struct sigaction saved_sigint;
static struct sigaction saved_sigquit;
static struct sigaction saved_sigterm;
static volatile sig_atomic_t signaled = 0;

static void sighandle(int sig)
{
    signaled = sig;
}

static void set_signal(int sig, struct sigaction *old, sighandler_t handler)
{
    do {
	if (sigaction(sig, NULL, old) == 0)
	    break;
    } while (errno == EINTR);

    if (old->sa_handler != handler) {
	struct sigaction new;
	sigset_t sigset;

	new.sa_handler = handler;
	sigemptyset(&new.sa_mask);
	new.sa_flags = SA_RESTART;
	do {
	    if (sigaction(sig, &new, NULL) == 0)
		break;
	} while (errno == EINTR);

	sigemptyset(&sigset);
	sigaddset(&sigset, sig);
	sigprocmask(SIG_UNBLOCK, &sigset, NULL);
    }
}

static void reset_signal(int sig, struct sigaction *old)
{
    struct sigaction cur;

    do {
	if (sigaction(sig, NULL, &cur) == 0)
	    break;
    } while (errno == EINTR);

    if (old->sa_handler == cur.sa_handler) {
	do {
	    if (sigaction(sig, old, NULL) == 0)
		break;
	} while (errno == EINTR);
    }
}

/*
 * To be able to reconnect to real tty on EIO
 */
static char * tty;
static void reconnect(int fd)
{
    int newfd = -1;

    if ((newfd = open(tty, O_WRONLY|O_NONBLOCK|O_NOCTTY)) < 0)
	error("can not open %s: %s\n", tty, strerror(errno));
 
    if (newfd != 1)
	dup2(newfd,  1);
    if (newfd != 2)
	dup2(newfd,  2);

    if (fd == 1 || fd == 2)
	goto out;
    if (newfd != fd)
	dup2(newfd, fd);
out:
    if (newfd > 2)
	close(newfd);
}

/*
 * Now do the job
 */
int main(int argc, char *argv[])
{
    int fd, flags;
    int ptm, pts, cntrtty = 1;
    pid_t pid, ppid = getppid();
    char ptsname[NAME_MAX+1];
    struct termios t;
    struct winsize w;
    time_t tt;
    char *stt, *name = ttyname(0);

    myname = basename(*argv);

    if (argc > 2)
	error("Too many args; usage: %s [/dev/tty<X>]\n", myname);

    if (argc == 2)
	tty = argv[1];
    else
	tty = fetchtty(getpid(), ppid);

    if (!tty || !*tty)
	error("can not discover real system console tty, boot logging disabled.\n");

    if (!name || !*name) {
	name = strdup("(no tty)");
	if (!name)
	    error("strdup(): %s\n", strerror(errno));
    }

    if (strcmp(tty, name) == 0)
	error("console=%s, stdin=%s, must differ, boot logging disabled\n", tty, name);

    if ((fd = open(tty, O_WRONLY|O_NONBLOCK|O_NOCTTY)) < 0)
	error("can not open %s: %s\n", tty, strerror(errno));

    if ((flags = fcntl(fd, F_GETFL)) < 0)
	error("can not get terminal flags of %s: %s\n", tty, strerror(errno));

    flags &= ~(O_NONBLOCK);
    flags |=   O_NOCTTY;
    if (fcntl(fd, F_SETFL, flags) < 0)
	error("can not set terminal flags of %s: %s\n", tty, strerror(errno));

    if (tcgetattr(fd, &t) < 0)
	error("can not get terminal parameters of %s: %s\n", tty, strerror(errno));
    cfsetispeed(&t, B38400);
    cfsetospeed(&t, B38400);

    if (tcgetpgrp(fd) < 0)
	cntrtty = 0;

    w.ws_row = 0;
    w.ws_col = 0;
    if (ioctl(fd, TIOCGWINSZ, &w) < 0)
	error("can not get window size of %s: %s\n", tty, strerror(errno));

    if (!w.ws_row)
	w.ws_row = 24;
    if (!w.ws_col)
	w.ws_row = 80;

    if (openpty(&ptm, &pts, ptsname, &t, &w) < 0)
	error("can not open pty/tty pair: %s\n", strerror(errno));

    (void)ioctl(0, TIOCCONS, NULL);  /* Undo any current map if any */
    if (ioctl(pts, TIOCCONS, NULL) < 0)
	error("can not set console device to %s: %s\n", ptsname, strerror(errno));

    set_signal(SIGTTIN, &saved_sigttin, SIG_IGN);
    set_signal(SIGTTOU, &saved_sigttou, SIG_IGN);
    set_signal(SIGTSTP, &saved_sigtstp, SIG_IGN);
    set_signal(SIGHUP,  &saved_sighup,  SIG_IGN);
    set_signal(SIGINT,  &saved_sigint,  sighandle);
    set_signal(SIGQUIT, &saved_sigquit, sighandle);
    set_signal(SIGTERM, &saved_sigterm, sighandle);
    (void)siginterrupt(SIGINT,  0);
    (void)siginterrupt(SIGQUIT, 0);
    (void)siginterrupt(SIGTERM, 0);

    switch ((pid = fork())) {
    case 0:
	/* Get our own session */
	setsid();
	/* Reconnect our own terminal I/O */
	dup2(ptm, 0);
	dup2(fd,  1);
	dup2(fd,  2);
	close(ptm);
	close(fd);
	break;
    case -1:
	close(pts);
	close(ptm);
	close(fd);
	error("can not fork to become daemon: %s\n", strerror(errno));
	break;
    default:
	time(&tt);
	stt = ctime(&tt);
	close(pts);
	close(ptm);
	close(fd);
	fprintf(stdout, "\rBoot logging started on %s(%s) at %.24s\n", tty, name, stt);
	fflush(stdout);
	exit(0);
    }
    free(name);
    prepareIO(reconnect, pidfile, 0, 1);
    while (!signaled)
	safeIO();

    if (!cntrtty)
	kill(ppid, SIGSTOP);
    closeIO();
    if (!cntrtty)
	kill(ppid, SIGCONT);

    (void)tcflush(1, TCOFLUSH);
    close(1);
    (void)tcflush(2, TCOFLUSH);
    close(2);

    (void)tcflush(pts, TCIOFLUSH);
    close(pts);
    (void)tcflush(0, TCIFLUSH);
    close(0);
    rmfpid();

    reset_signal(SIGTTIN, &saved_sigttin);
    reset_signal(SIGTTOU, &saved_sigttou);
    reset_signal(SIGTSTP, &saved_sigtstp);
    reset_signal(SIGHUP,  &saved_sighup);
    reset_signal(SIGINT,  &saved_sigint);
    reset_signal(SIGQUIT, &saved_sigquit);
    reset_signal(SIGTERM, &saved_sigterm);

    return 0;
}
