/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the "License").  You may not use this file except
 * in compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://jaxp.dev.java.net/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * HEADER in each file and include the License file at
 * https://jaxp.dev.java.net/CDDLv1.0.html
 * If applicable add the following below this CDDL HEADER
 * with the fields enclosed by brackets "[]" replaced with
 * your own identifying information: Portions Copyright
 * [year] [name of copyright owner]
 */

/*
 * $Id: ReaderToWriterTest.java,v 1.3 2006/11/01 17:05:25 spericas Exp $
 * %W% %E%
 *
 * Copyright 2006 Sun Microsystems, Inc. All Rights Reserved.
 */

package XMLEventWriter;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;

import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.XMLEvent;

import junit.framework.TestCase;
import junit.textui.TestRunner;

public class ReaderToWriterTest
    extends TestCase {

    private static final XMLEventFactory XML_EVENT_FACTORY =
            XMLEventFactory.newInstance();
    private static final XMLInputFactory XML_INPUT_FACTORY =
            XMLInputFactory.newInstance();
    private static final XMLOutputFactory XML_OUTPUT_FACTORY =
            XMLOutputFactory.newInstance();

    private static final String INPUT_FILE = "W2JDLR4002TestService.wsdl";
    private static final String OUTPUT_FILE = "Encoded.wsdl";
    
    /** Creates a new instance of XMLStreamWriterTest. */
    public ReaderToWriterTest(String name) {
        super(name);
    }

    public static void main(String [] args) {
        TestRunner.run(ReaderToWriterTest.class);
    }

    /**
     * Unit test for writing namespaces when namespaceURI == null.
     */
    public void testWriteNamespace() {

        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" encoding=\"UTF-8\"?><prefix:root xmlns=\"\" xmlns:null=\"\"></prefix:root>";

        // new Writer
        XMLEventWriter xmlEventWriter = null;
        ByteArrayOutputStream byteArrayOutputStream =
                new ByteArrayOutputStream();
        try {
            xmlEventWriter =
                    XML_OUTPUT_FACTORY.createXMLEventWriter(
                        byteArrayOutputStream, "UTF-8");
        } catch (XMLStreamException xmlStreamException) {
            xmlStreamException.printStackTrace();
            fail(xmlStreamException.toString());
        }

        try {
            // start a valid event stream
            XMLEvent startDocumentEvent = XML_EVENT_FACTORY.createStartDocument();
            XMLEvent startElementEvent =
                    XML_EVENT_FACTORY.createStartElement(
                        "prefix", "http://example.com", "root");
            xmlEventWriter.add(startDocumentEvent);
            xmlEventWriter.add(startElementEvent);

            // try using a null default namespaceURI
            XMLEvent namespaceEvent = XML_EVENT_FACTORY.createNamespace(null);
            xmlEventWriter.add(namespaceEvent);

            // try using a null prefix'd namespaceURI
            XMLEvent namespacePrefixEvent =
                    XML_EVENT_FACTORY.createNamespace("null", null);
            xmlEventWriter.add(namespacePrefixEvent);

            // close event stream
            XMLEvent endElementEvent =
                    XML_EVENT_FACTORY.createEndElement(
                    "prefix", "http://example.com", "root");
            XMLEvent endDocumentEvent = XML_EVENT_FACTORY.createEndDocument();
            xmlEventWriter.add(endElementEvent);
            xmlEventWriter.add(endDocumentEvent);
            xmlEventWriter.flush();
        } catch (XMLStreamException xmlStreamException) {
            xmlStreamException.printStackTrace();
            fail(xmlStreamException.toString());
        }

        // get XML document as String
        String actualOutput = byteArrayOutputStream.toString();

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Test: 6419687 NPE in XMLEventWriterImpl.
     */
    public void testCR6419687() {

        try {
            InputStream in =
                    new FileInputStream("./tests/XMLEventWriter/ReaderToWriterTest.wsdl");
            OutputStream out =
                    new FileOutputStream("./build/unit-tests/ReaderToWriterTest-out.xml");

            XMLEventReader reader =
                        XML_INPUT_FACTORY.createXMLEventReader(in);
            XMLEventWriter writer =
                    XML_OUTPUT_FACTORY.createXMLEventWriter(out, "UTF-8");
            while (reader.hasNext()) {
                XMLEvent event = reader.nextEvent();
                writer.add(event);
            }
            reader.close();
            writer.close();
        } catch (XMLStreamException xmlStreamException) {
            xmlStreamException.printStackTrace();
            fail(xmlStreamException.toString());
        } catch (FileNotFoundException fileNotFoundException) {
            fileNotFoundException.printStackTrace();
            fail(fileNotFoundException.toString());
        }
    }
    
    /* 
     * Reads UTF-16 encoding file and writes it to UTF-8 encoded format.
     */
    public void testUTF8Encoding() {
        try{
            InputStream in = this.getClass().getResourceAsStream(INPUT_FILE);
            OutputStream out = new FileOutputStream(OUTPUT_FILE);

            XMLEventReader reader = XML_INPUT_FACTORY.createXMLEventReader(in);
            XMLEventWriter writer = XML_OUTPUT_FACTORY.createXMLEventWriter(out, "UTF-8");
            
            writeEvents(reader, writer);
            checkOutput(OUTPUT_FILE);
 
        } catch(Exception e){
            e.printStackTrace();
            fail("Exception occured: "+e.getMessage());
        } finally {
            File file = new File(OUTPUT_FILE);
            if (file.exists())
                file.delete();
        }
    }
    
    private void writeEvents(XMLEventReader reader, XMLEventWriter writer) throws XMLStreamException{
         while(reader.hasNext()) {
                XMLEvent event = reader.nextEvent();
                writer.add(event);
            }
            reader.close();
            writer.close();
    }
    
    private void checkOutput(String output) throws Exception{
        InputStream in = new FileInputStream(output);
        XMLEventReader reader = XML_INPUT_FACTORY.createXMLEventReader(in);
        while(reader.hasNext()) {
            reader.next();
        }
        reader.close();
    }
    
    /*
     * Reads UTF-16 encoding file and writes it with default encoding.
     */
    public void testNoEncoding() {
        try{
            InputStream in = this.getClass().getResourceAsStream(INPUT_FILE);
            OutputStream out = new FileOutputStream(OUTPUT_FILE);

            XMLEventReader reader = XML_INPUT_FACTORY.createXMLEventReader(in);
            XMLEventWriter writer = XML_OUTPUT_FACTORY.createXMLEventWriter(out);
            
            writeEvents(reader, writer);
            checkOutput(OUTPUT_FILE);
 
        } catch(Exception e){
            e.printStackTrace();
            fail("Exception occured: "+e.getMessage());
        } finally {
            File file = new File(OUTPUT_FILE);
            if (file.exists())
                file.delete();
        }
    }
}
