/*
 * $Id: MultiThreadTest.java,v 1.2 2006/04/01 06:01:52 jeffsuttor Exp $
 */

/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://glassfish.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * [Name of File] [ver.__] [Date]
 * 
 * Copyright 2006 Sun Microsystems Inc. All Rights Reserved
 */

package XMLStreamWriter;

import java.io.ByteArrayOutputStream;
import java.util.Random;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamWriter;

import junit.framework.TestCase;

/**
 * Stress test threading XMLStreamWriter.
 *
 * Using a single XMLOutputFactory, give several threads their own
 * XMLStreamWriter and have them reuse it many times.
 * ALso using same XMLOutputFactory, have each thread get a new XMLStreamWriter
 * and use it once many times.
 * Finaly have each Thread create its own XMLOutputFactory and get a
 * new XMLStreamWriter and use it once many times.
 *
 * @author <a href="mailto:Jeff.Sutor@Sun.com">Jeff Suttor</a>
 */
public class MultiThreadTest extends TestCase {
    
    private static final int NO_THREADS = 16;
    private static final int REQS_PER_THREAD = 4096;
    private static int NO_RESPS = 0;

    private static final String EXPECTED_OUTPUT =
            "<?xml version=\"1.0\" ?>"
            + "<hello>world</hello>";
    
    XMLStreamWriter xmlStreamWriter;
    ByteArrayOutputStream byteArrayOutputStream;
    XMLOutputFactory xmlOutputFactory ;

    public void testMultiThread() throws Exception {
        
        xmlOutputFactory = XMLOutputFactory.newInstance();

        // create Threads
        Thread[] threads = new Thread[NO_THREADS];
        for (int i = 0; i < NO_THREADS; i++) {
            ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
            threads[i] = new Thread(
                    new MultiRunnable(
                        xmlOutputFactory.createXMLStreamWriter(byteArrayOutputStream),
                        byteArrayOutputStream));
        }

        // start Threads
        for(int i = 0; i < NO_THREADS; i++) {
            threads[i].start();
        }

        // wait for Threads to complete
        for(int i=0; i < NO_THREADS; i++) {
            threads[i].join();
        }

        // test results
        synchronized(MultiThreadTest.class) {
            assertEquals(NO_THREADS*REQS_PER_THREAD, NO_RESPS);
        }
    }

    public class MultiRunnable implements Runnable {

        XMLStreamWriter xmlStreamWriter = null;
        ByteArrayOutputStream byteArrayOutputStream = null;
        
        public MultiRunnable (
                XMLStreamWriter xmlStreamWriter,
                ByteArrayOutputStream byteArrayOutputStream) {
            
            this.xmlStreamWriter = xmlStreamWriter;
            this.byteArrayOutputStream = byteArrayOutputStream;
        }
        
        public void run() {
            for (int i = 0; i < REQS_PER_THREAD; i++) {
                try {
                    invoke(xmlStreamWriter, byteArrayOutputStream);
                    int rand = new Random(System.currentTimeMillis()).nextInt(32);
                    Thread.sleep(rand);
                } catch (Exception e) {
                    e.printStackTrace();
                    fail(e.toString());
                }
            }
        }
        public void invoke (
                XMLStreamWriter xmlStreamWriter,
                ByteArrayOutputStream byteArrayOutputStream) throws Exception {
            
            // reset the writer
            xmlStreamWriter.close();
            ((com.sun.xml.stream.writers.XMLStreamWriterImpl) xmlStreamWriter).reset();
            byteArrayOutputStream.reset();
            
            // reuse passed in XMLStreamWriter
            xmlStreamWriter.writeStartDocument();
            xmlStreamWriter.writeStartElement("hello");
            xmlStreamWriter.writeCharacters("world");
            xmlStreamWriter.writeEndElement();
            xmlStreamWriter.writeEndDocument();
            xmlStreamWriter.flush();
            String actualOutput = byteArrayOutputStream.toString();
            
            assertEquals(EXPECTED_OUTPUT, actualOutput);
            
            // reuse passed in XMLOutputFactory to create a new XMLStreamWriter
            ByteArrayOutputStream newByteArrayOutputStream = new ByteArrayOutputStream();
            XMLStreamWriter newXMLStreamWriter = xmlOutputFactory.createXMLStreamWriter(newByteArrayOutputStream);
            
            newXMLStreamWriter.writeStartDocument();
            newXMLStreamWriter.writeStartElement("hello");
            newXMLStreamWriter.writeCharacters("world");
            newXMLStreamWriter.writeEndElement();
            newXMLStreamWriter.writeEndDocument();
            newXMLStreamWriter.flush();
            String newActualOutput = newByteArrayOutputStream.toString();
            // reset the writer
            newXMLStreamWriter.close();
            ((com.sun.xml.stream.writers.XMLStreamWriterImpl) newXMLStreamWriter).reset();
            newByteArrayOutputStream.reset();
            
            if (!EXPECTED_OUTPUT.equals(newActualOutput)) {
                fail(EXPECTED_OUTPUT + "!=" + newActualOutput);
            }
            assertEquals(EXPECTED_OUTPUT, newActualOutput);

            // new XMLOutputFactory to create a new XMLStreamWriter
            ByteArrayOutputStream newFactoryByteArrayOutputStream = new ByteArrayOutputStream();
            XMLOutputFactory newXMLOutputFactory = XMLOutputFactory.newInstance();
            XMLStreamWriter newFactoryXMLStreamWriter = newXMLOutputFactory.createXMLStreamWriter(newFactoryByteArrayOutputStream);
            
            newFactoryXMLStreamWriter.writeStartDocument();
            newFactoryXMLStreamWriter.writeStartElement("hello");
            newFactoryXMLStreamWriter.writeCharacters("world");
            newFactoryXMLStreamWriter.writeEndElement();
            newFactoryXMLStreamWriter.writeEndDocument();
            newFactoryXMLStreamWriter.flush();
            String newFactoryActualOutput = newFactoryByteArrayOutputStream.toString();
            // reset the writer
            newFactoryXMLStreamWriter.close();
            ((com.sun.xml.stream.writers.XMLStreamWriterImpl) newFactoryXMLStreamWriter).reset();
            newFactoryByteArrayOutputStream.reset();
            
            assertEquals(EXPECTED_OUTPUT, newFactoryActualOutput);

            // count this run
            synchronized(MultiThreadTest.class) {
                ++NO_RESPS;
            }
        }
    }
}
