package XMLStreamWriter;

import java.io.ByteArrayOutputStream;

import javax.xml.XMLConstants;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import junit.framework.TestCase;
import junit.textui.TestRunner;
/**
 * Test the writing of Namespaces.
 *
 * @author <a href="mailto"Jeff.Suttor@Sun.com>Jeff Suttor</a>,
 *  thanks to Norm Walsh & Sunitha Reddy & Kohsuke Kawaguchi
 */
public class NamespaceTest extends TestCase {
    
    /** debug output? */
    private static final boolean DEBUG = true;
    
    /** Factory to reuse. */
    XMLOutputFactory xmlOutputFactory = null;
    
    /** Writer to reuse. */
    XMLStreamWriter xmlStreamWriter = null;

    /** OutputStream to reuse. */
    ByteArrayOutputStream byteArrayOutputStream = null;
    
    public static void main(String[] args) {
    	TestRunner.run(NamespaceTest.class);
    }
     
    /**
     * All tests need a Writer.
     *
     * Reuse the same Writer.
     */
    public void setUp() {

        // want a Factory that repairs Namespaces
        xmlOutputFactory = XMLOutputFactory.newInstance();
        xmlOutputFactory.setProperty(XMLOutputFactory.IS_REPAIRING_NAMESPACES, Boolean.TRUE);
        
        // new OutputStream
        byteArrayOutputStream = new ByteArrayOutputStream();

        // new Writer
        try {
            xmlStreamWriter = xmlOutputFactory.createXMLStreamWriter(byteArrayOutputStream);
            
        } catch (XMLStreamException xmlStreamException) {
            fail(xmlStreamException.toString());
        }
    }
    
    /**
     * Reset Writer for reuse.
     */
    private void resetWriter() {
        
        // reset the Writer
        try {
            xmlStreamWriter.flush();
            //xmlStreamWriter.close();
            //((com.sun.xml.stream.writers.XMLStreamWriterImpl) xmlStreamWriter).reset();
        } catch (XMLStreamException xmlStreamException) {
            fail(xmlStreamException.toString());
        }
            
        // reset the OutputStream
        byteArrayOutputStream.reset();
    }

    /**
     * See bug 6389310.
     *
     * @author Kohsuke Kawaguchi
     */
    public void testDoubleXmlNs() {
       try{ 
        // reset to known state
        resetWriter();
        
        xmlStreamWriter.writeStartDocument();
        xmlStreamWriter.writeStartElement("foo");
        xmlStreamWriter.writeNamespace("xml", XMLConstants.XML_NS_URI);
        xmlStreamWriter.writeAttribute("xml", XMLConstants.XML_NS_URI, "lang", "ja_JP");
        xmlStreamWriter.writeCharacters("Hello");
        xmlStreamWriter.writeEndElement();
        xmlStreamWriter.writeEndDocument();
        
        xmlStreamWriter.flush();
        String actualOutput = byteArrayOutputStream.toString();

        if (DEBUG) {
            System.out.println("testDoubleXmlNs(): actualOutput: " + actualOutput);
        }

        // there should be no xmlns:xml
        assertTrue(
                "Expected 0 xmlns:xml, actual output: " + actualOutput,
                actualOutput.split("xmlns:xml").length == 1);
       }catch(Exception e){
           e.printStackTrace();
           fail(e.getMessage());
       }
    }
    
    /**
     * See 6395980.
     *
     * @author Kohsuke Kawaguchi
     */
    public void testDuplicateNamespaceURI() throws Exception {

        // reset to known state
        resetWriter();
        
        xmlStreamWriter.writeStartDocument();
        xmlStreamWriter.writeStartElement(new String(""),"localName",new String("nsUri"));
        xmlStreamWriter.writeNamespace(new String(""),new String("nsUri"));
        xmlStreamWriter.writeEndElement();
        xmlStreamWriter.writeEndDocument();
        
        xmlStreamWriter.flush();
        String actualOutput = byteArrayOutputStream.toString();

        if (DEBUG) {
            System.out.println("testDuplicateNamespaceURI(): actualOutput: " + actualOutput);
        }

        // there must be only 1 xmlns=...
        assertTrue(
                "Expected 1 xmlns=, actual output: " + actualOutput,
                actualOutput.split("xmlns").length == 2);
    }
    

    // TODO: test with both "" & null
    // NDW: There's no distinction in XML between a "null" namespace URI and one
    // with a URI of "" (the empty string) so I haven't tried to call out any
    // such distinctions.

    // ---------------- Current default namespace is "" ----------------
    

    private void startDocumentEmptyDefaultNamespace(XMLStreamWriter xmlStreamWriter)
            throws XMLStreamException {
        
        xmlStreamWriter.writeStartDocument();
        xmlStreamWriter.writeStartElement("root");
        xmlStreamWriter.writeDefaultNamespace("");
    }

    private String endDocumentEmptyDefaultNamespace(XMLStreamWriter xmlStreamWriter)
            throws XMLStreamException {
        
        xmlStreamWriter.writeEndDocument();
        
        xmlStreamWriter.flush();
        
        return byteArrayOutputStream.toString();
    }

    /**
     * Current default namespace is "".
     *
     * writeStartElement("", "localName"", "")
     *
     * requires no fixup
     */
    public void testEmptyDefaultEmptyPrefix() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\">"
                + "<localName>"
                + "requires no fixup"
                + "</localName>"
                + "</root>";

        // reset to known state
        resetWriter();        
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);

        xmlStreamWriter.writeStartElement("", "localName", "");
        xmlStreamWriter.writeCharacters("requires no fixup");
        
        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultEmptyPrefix(): actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "".
     *
     * writeStartElement("prefix", "localName", "http://example.org/myURI")
     *
     * requires no fixup, but should generate a declaration for "prefix":
     *   xmlns:prefix="http://example.org/myURI"
     * if necessary
     *
     * necessary to generate a declaration in this test case.
     */
    public void testEmptyDefaultSpecifiedPrefix() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\">"
                + "<prefix:localName xmlns:prefix=\"http://example.org/myURI\">"
                + "generate xmlns:prefix"
                + "</prefix:localName>"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeStartElement("prefix", "localName", "http://example.org/myURI");
        xmlStreamWriter.writeCharacters("generate xmlns:prefix");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultSpecifiedPrefix(): actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }
    
    /**
     * Current default namespace is "".
     *
     * writeStartElement("prefix", "localName", "http://example.org/myURI")
     *
     * requires no fixup, but should generate a declaration for "prefix":
     *   xmlns:prefix="http://example.org/myURI"
     * if necessary
     *
     * not necessary to generate a declaration in this test case.
     */    
    public void testEmptyDefaultSpecifiedPrefixNoDeclarationGeneration() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\""
                +   " xmlns:prefix=\"http://example.org/myURI\">"
                + "<prefix:localName>"
                + "not necessary to generate a declaration"
                + "</prefix:localName>"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeNamespace("prefix", "http://example.org/myURI");

        xmlStreamWriter.writeStartElement("prefix", "localName", "http://example.org/myURI");
        xmlStreamWriter.writeCharacters("not necessary to generate a declaration");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultSpecifiedPrefixNoDeclarationGeneration(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testEmptyDefaultSpecifiedPrefixNoDeclarationGeneration():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "".
     *
     * writeStartElement("", "localName", "http://example.org/myURI")
     *
     * should "fixup" the declaration for the default namespace:
     *   xmlns="http://example.org/myURI"
     */
    public void testEmptyDefaultSpecifiedDefault() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\">"
                + "<localName xmlns=\"http://example.org/myURI\">"
                + "generate xmlns"
                + "</localName>"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeStartElement("", "localName", "http://example.org/myURI");
        xmlStreamWriter.writeCharacters("generate xmlns");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultSpecifiedDefault(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testEmptyDefaultSpecifiedDefault():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "".
     *
     * writeAttribute("", "", "attrName", "value")
     *
     * requires no fixup
     */
    public void testEmptyDefaultEmptyPrefixWriteAttribute() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\" attrName=\"value\">"
                + "requires no fixup"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("", "", "attrName", "value");
        xmlStreamWriter.writeCharacters("requires no fixup");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultEmptyPrefixWriteAttribute(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testEmptyDefaultEmptyPrefixWriteAttribute():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "".
     *
     * writeAttribute("p", "http://example.org/myURI", "attrName", "value")
     *
     * requires no fixup, but should generate a declaration for "p":
     *   xmlns:p="http://example.org/myURI"
     * if necessary
     *
     * necessary to generate a declaration in this test case.
     */
    public void testEmptyDefaultSpecifiedPrefixWriteAttribute() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\" xmlns:p=\"http://example.org/myURI\" p:attrName=\"value\">"
                + "generate xmlns:p=\"http://example.org/myURI\""
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("p", "http://example.org/myURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("generate xmlns:p=\"http://example.org/myURI\"");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultSpecifiedPrefixWriteAttribute(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testEmptyDefaultSpecifiedPrefixWriteAttribute():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "".
     *
     * writeAttribute("p", "http://example.org/myURI", "attrName", "value")
     *
     * requires no fixup, but should generate a declaration for "p":
     *   xmlns:p="http://example.org/myURI"
     * if necessary
     *
     * not necessary to generate a declaration in this test case.
     */
    public void testEmptyDefaultSpecifiedPrefixWriteAttributeNoDeclarationGeneration() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\" xmlns:p=\"http://example.org/myURI\" p:attrName=\"value\">"
                + "not necessary to generate a declaration"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeNamespace("p", "http://example.org/myURI");
        
        xmlStreamWriter.writeAttribute("p", "http://example.org/myURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("not necessary to generate a declaration");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultSpecifiedPrefixWriteAttributeNoDeclarationGeneration(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testEmptyDefaultSpecifiedPrefixWriteAttributeNoDeclarationGeneration():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "".
     *
     * writeAttribute("", "http://example.org/myURI", "attrName", "value")
     *
     * XMLOutputFactory (Javadoc) :
     * "If a writer isRepairingNamespaces it will create a namespace declaration
     * on the current StartElement for any attribute that does not currently
     * have a namespace declaration in scope. If the StartElement has a uri but
     * no prefix specified a prefix will be assigned, if the prefix has not been
     * declared in a parent of the current StartElement it will be declared on
     * the current StartElement. If the defaultNamespace is bound and in scope
     * and the default namespace matches the URI of the attribute or
     * StartElement QName no prefix will be assigned."
     *
     * prefix needs to be assigned for this test case.
     */
    public void testEmptyDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttribute() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\" xmlns:{generated prefix}=\"http://example.org/myURI\" {generated prefix}:attrName=\"value\">"
                + "generate xmlns declaration {generated prefix}=\"http://example.org/myURI\""
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("", "http://example.org/myURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("generate xmlns declaration {generated prefix}=\"http://example.org/myURI\"");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultUnspecifiedPrefixWriteAttribute(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testEmptyDefaultUnspecifiedPrefixWriteAttribute():   actualOutput: " + actualOutput);
        }

        // there must be one xmlns=
        assertTrue(
                "Expected 1 xmlns=, actual output: " + actualOutput,
                actualOutput.split("xmlns=").length == 2);
        
        // there must be one xmlns:{generated prefix}="..."
        assertTrue(
                "Expected 1 xmlns:{generated prefix}=\"\", actual output: " + actualOutput,
                actualOutput.split("xmlns:").length ==2);

        // there must be one {generated prefix}:attrName="value"
        assertTrue(
                "Expected 1 {generated prefix}:attrName=\"value\", actual output: " + actualOutput,
                actualOutput.split(":attrName=\"value\"").length == 2);
    }

    /**
     * Current default namespace is "".
     *
     * writeAttribute("", "http://example.org/myURI", "attrName", "value")
     *
     * XMLOutputFactory (Javadoc) :
     * "If a writer isRepairingNamespaces it will create a namespace declaration
     * on the current StartElement for any attribute that does not currently
     * have a namespace declaration in scope. If the StartElement has a uri but
     * no prefix specified a prefix will be assigned, if the prefix has not been
     * declared in a parent of the current StartElement it will be declared on
     * the current StartElement. If the defaultNamespace is bound and in scope
     * and the default namespace matches the URI of the attribute or
     * StartElement QName no prefix will be assigned."
     *
     * no prefix needs to be assigned for this test case
     */
    public void testEmptyDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttributeNoPrefixGeneration() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\" xmlns:p=\"http://example.org/myURI\" p:attrName=\"value\">"
                + "no prefix generation"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeNamespace("p", "http://example.org/myURI");

        xmlStreamWriter.writeAttribute("", "http://example.org/myURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("no prefix generation");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttributeNoPrefixGeneration(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testEmptyDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttributeNoPrefixGeneration():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    // ---------------- Current default namespace is "http://example.org/uniqueURI" ----------------

    private void startDocumentSpecifiedDefaultNamespace(XMLStreamWriter xmlStreamWriter)
            throws XMLStreamException {
        
        xmlStreamWriter.writeStartDocument();
        xmlStreamWriter.writeStartElement("root");
        xmlStreamWriter.writeDefaultNamespace("http://example.org/uniqueURI");
    }

    private String endDocumentSpecifiedDefaultNamespace(XMLStreamWriter xmlStreamWriter)
            throws XMLStreamException {
        
        xmlStreamWriter.writeEndDocument();
        
        xmlStreamWriter.flush();
        
        return byteArrayOutputStream.toString();
    }

    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeElement("", "localName", "")
     *
     * should "fixup" the declaration for the default namespace:
     *   xmlns=""
     */
    public void testSpecifiedDefaultEmptyPrefix() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\">"
                + "<localName xmlns=\"\">"
                + "generate xmlns=\"\""
                + "</localName>"
                + "</root>";

        // reset to known state
        resetWriter();        
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        xmlStreamWriter.writeStartElement("", "localName", "");
        xmlStreamWriter.writeCharacters("generate xmlns=\"\"");
        
        String actualOutput = endDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultEmptyPrefix(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultEmptyPrefix():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeStartElement("p", "localName", "http://example.org/myURI")
     *
     * requires no fixup, but should generate a declaration for "p":
     *   xmlns:p="http://example.org/myURI"
     * if necessary
     *
     * test case where it is necessary to generate a declaration.
     */
    public void testSpecifiedDefaultSpecifiedPrefix() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\">"
                + "<p:localName xmlns:p=\"http://example.org/myURI\">"
                + "generate xmlns:p=\"http://example.org/myURI\""
                + "</p:localName>"
                + "</root>";

        // reset to known state
        resetWriter();        
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        xmlStreamWriter.writeStartElement("p", "localName", "http://example.org/myURI");
        xmlStreamWriter.writeCharacters("generate xmlns:p=\"http://example.org/myURI\"");
        
        String actualOutput = endDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultSpecifiedPrefix(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultSpecifiedPrefix():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeStartElement("p", "localName", "http://example.org/myURI")
     *
     * requires no fixup, but should generate a declaration for "p":
     *   xmlns:p="http://example.org/myURI"
     * if necessary
     *
     * test case where it is not necessary to generate a declaration.
     */
    public void testSpecifiedDefaultSpecifiedPrefixNoPrefixGeneration() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root"
                +   " xmlns=\"http://example.org/uniqueURI\""
                +   " xmlns:p=\"http://example.org/myURI\">"
                + "<p:localName>"
                + "not necessary to generate a declaration"
                + "</p:localName>"
                + "</root>";

        // reset to known state
        resetWriter();        
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        xmlStreamWriter.writeNamespace("p", "http://example.org/myURI");

        xmlStreamWriter.writeStartElement("p", "localName", "http://example.org/myURI");
        xmlStreamWriter.writeCharacters("not necessary to generate a declaration");
        
        String actualOutput = endDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultSpecifiedPrefixNoPrefixGeneration(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultSpecifiedPrefixNoPrefixGeneration():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeStartElement("", "localName", "http://example.org/myURI")
     *
     * should "fixup" the declaration for the default namespace:
     *   xmlns="http://example.org/myURI"
     */
    public void testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURI() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\">"
                + "<localName xmlns=\"http://example.org/myURI\">"
                + "generate xmlns=\"http://example.org/myURI\""
                + "</localName>"
                + "</root>";

        // reset to known state
        resetWriter();        
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        xmlStreamWriter.writeStartElement("", "localName", "http://example.org/myURI");
        xmlStreamWriter.writeCharacters("generate xmlns=\"http://example.org/myURI\"");
        
        String actualOutput = endDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURI(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURI():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeAttribute("", "", "attrName", "value")
     *
     * requires no fixup
     */
    public void testSpecifiedDefaultEmptyPrefixWriteAttribute() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\" attrName=\"value\">"
                + "requires no fixup"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("", "", "attrName", "value");
        xmlStreamWriter.writeCharacters("requires no fixup");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultEmptyPrefixWriteAttribute(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultEmptyPrefixWriteAttribute():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeAttribute("p", "http://example.org/myURI", "attrName", "value")
     *
     * requires no fixup, but should generate a declaration for "p":
     *   xmlns:p="http://example.org/myURI"
     * if necessary
     *
     * test case where it is necessary to generate a declaration.
     */
    public void testSpecifiedDefaultSpecifiedPrefixWriteAttribute() throws Exception { //want to test
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\" xmlns:p=\"http://example.org/myURI\" p:attrName=\"value\">"
                + "generate xmlns:p=\"http://example.org/myURI\""
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("p", "http://example.org/myURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("generate xmlns:p=\"http://example.org/myURI\"");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultSpecifiedPrefixWriteAttribute(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultSpecifiedPrefixWriteAttribute():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }
    
    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeAttribute("p", "http://example.org/myURI", "attrName", "value")
     *
     * requires no fixup, but should generate a declaration for "p":
     *   xmlns:p="http://example.org/myURI"
     * if necessary
     *
     * test case where it is not necessary to generate a declaration.
     */
    public void testSpecifiedDefaultSpecifiedPrefixWriteAttributeNoDeclarationGeneration() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\" xmlns:p=\"http://example.org/myURI\" p:attrName=\"value\">"
                + "not necessary to generate a declaration"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeNamespace("p", "http://example.org/myURI");

        xmlStreamWriter.writeAttribute("p", "http://example.org/myURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("not necessary to generate a declaration");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultSpecifiedPrefixWriteAttributeNoDeclarationGeneration(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultSpecifiedPrefixWriteAttributeNoDeclarationGeneration():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeAttribute("p", "http://example.org/uniqueURI", "attrName", "value")
     *
     * requires no fixup, but should generate a declaration for "p":
     *   xmlns:p="http://example.org/uniqueURI"
     * if necessary.
     * (Note that this will potentially produce two namespace bindings with the same URI,
     * xmlns="xxx" and xmlns:p="xxx", but that's perfectly legal.)
     */
    public void testSpecifiedDefaultSpecifiedPrefixSpecifiedNamespaceURIWriteAttribute() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\" attrName=\"value\">"
                + "requires no fixup"
                + "</root>";
        final String EXPECTED_OUTPUT_2 =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\" xmlns:p=\"http://example.org/uniqueURI\" p:attrName=\"value\">"
                + "requires no fixup"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("p", "http://example.org/uniqueURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("requires no fixup");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultSpecifiedPrefixSpecifiedNamespaceURIWriteAttribute: expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultSpecifiedPrefixSpecifiedNamespaceURIWriteAttribute: expectedOutput: " + EXPECTED_OUTPUT_2);
            System.out.println("testSpecifiedDefaultSpecifiedPrefixSpecifiedNamespaceURIWriteAttribute:   actualOutput: " + actualOutput);
        }

        assertTrue("Expected: " + EXPECTED_OUTPUT + "\n"
                + "Actual: " + actualOutput,
                actualOutput.equals(EXPECTED_OUTPUT)
                || actualOutput.equals(EXPECTED_OUTPUT_2));
    }

    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeAttribute("", "http://example.org/myURI", "attrName", "value")
     *
     * XMLOutputFactory (Javadoc) :
     * "If a writer isRepairingNamespaces it will create a namespace declaration
     * on the current StartElement for any attribute that does not currently
     * have a namespace declaration in scope. If the StartElement has a uri but
     * no prefix specified a prefix will be assigned, if the prefix has not been
     * declared in a parent of the current StartElement it will be declared on
     * the current StartElement. If the defaultNamespace is bound and in scope
     * and the default namespace matches the URI of the attribute or
     * StartElement QName no prefix will be assigned."
     *
     * test case where prefix needs to be assigned.
     */
    public void testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttribute() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root"
                +   " xmlns=\"http://example.org/uniqueURI\""
                +   " xmlns:{generated prefix}=\"http://example.org/myURI\""
                +   " {generated prefix}:attrName=\"value\">"
                + "generate xmlns declaration {generated prefix}=\"http://example.org/myURI\""
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("", "http://example.org/myURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("generate xmlns declaration {generated prefix}=\"http://example.org/myURI\"");

        String actualOutput = endDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttribute(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttribute():   actualOutput: " + actualOutput);
        }
        
        // there must be one xmlns=
        assertTrue(
                "Expected 1 xmlns=, actual output: " + actualOutput,
                actualOutput.split("xmlns=").length == 2);
        
        // there must be one xmlns:{generated prefix}="..."
        assertTrue(
                "Expected 1 xmlns:{generated prefix}=\"\", actual output: " + actualOutput,
                actualOutput.split("xmlns:").length == 2);

        // there must be one {generated prefix}:attrName="value"
        assertTrue(
                "Expected 1 {generated prefix}:attrName=\"value\", actual output: " + actualOutput,
                actualOutput.split(":attrName=\"value\"").length == 2);
    }
    
    /**
     * Current default namespace is "http://example.org/uniqueURI".
     *
     * writeAttribute("", "http://example.org/myURI", "attrName", "value")
     *
     * XMLOutputFactory (Javadoc) :
     * "If a writer isRepairingNamespaces it will create a namespace declaration
     * on the current StartElement for any attribute that does not currently
     * have a namespace declaration in scope. If the StartElement has a uri but
     * no prefix specified a prefix will be assigned, if the prefix has not been
     * declared in a parent of the current StartElement it will be declared on
     * the current StartElement. If the defaultNamespace is bound and in scope
     * and the default namespace matches the URI of the attribute or
     * StartElement QName no prefix will be assigned."
     *
     * test case where no prefix needs to be assigned.
     */
    public void testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttributeNoPrefixGeneration() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root"
                +   " xmlns=\"http://example.org/uniqueURI\""
                +   " xmlns:p=\"http://example.org/myURI\""
                +   " p:attrName=\"value\">"
                + "no prefix needs to be assigned"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeNamespace("p", "http://example.org/myURI");

        xmlStreamWriter.writeAttribute("", "http://example.org/myURI", "attrName", "value");
        xmlStreamWriter.writeCharacters("no prefix needs to be assigned");

        String actualOutput = endDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttributeNoPrefixGeneration(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultEmptyPrefixSpecifiedNamespaceURIWriteAttributeNoPrefixGeneration():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }

    // --------------- Serializations, sequences ---------------
    
    // Unfortunately, the nature of the StAX API makes it possible for the
    // programmer to generate events that cannot be serialized in XML.

    /**
     * Current default namespace is "".
     *
     * write*("p", "myuri", ...);
     * write*("p", "otheruri", ...);
     *
     * XMLOutputFactory (Javadoc) (If repairing of namespaces is enabled):
     * "If element and/or attribute names in the same start or empty-element tag
     * are bound to different namespace URIs and are using the same prefix then
     * the element or the first occurring attribute retains the original prefix
     * and the following attributes have their prefixes replaced with a new
     * prefix that is bound to the namespace URIs of those attributes."
     */
    public void testSamePrefixDifferentURI() throws Exception {
        
        /**
         * writeAttribute("p", "http://example.org/URI-ONE", "attr1", "value");
         * writeAttribute("p", "http://example.org/URI-TWO", "attr2", "value");
         */
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root"
                +   " xmlns=\"\""
                +   " xmlns:p=\"http://example.org/URI-ONE\""
                +   " p:attr1=\"value\">"
                +   " xmlns:{generated prefix}=\"http://example.org/URI-TWO\""
                +   " {generated prefix}:attr2=\"value\">"
                + "remap xmlns declaration {generated prefix}=\"http://example.org/URI-TWO\""
                + "</root>";
    
        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("p", "http://example.org/URI-ONE", "attr1", "value");
        xmlStreamWriter.writeAttribute("p", "http://example.org/URI-TWO", "attr2", "value");
        xmlStreamWriter.writeCharacters("remap xmlns declaration {generated prefix}=\"http://example.org/URI-TWO\"");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSamePrefixDifferentURI(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSamePrefixDifferentURI():   actualOutput: " + actualOutput);
        }

        // there must be 1 xmlns=
        assertTrue(
                "Expected 1 xmlns=, actual output: " + actualOutput,
                actualOutput.split("xmlns=").length == 2);
        
        // there must be 2 xmlns:
        assertTrue(
                "Expected 2 xmlns:, actual output: " + actualOutput,
                actualOutput.split("xmlns:").length == 3);

        // there must be 2 :attr
        assertTrue(
                "Expected 2 :attr, actual output: " + actualOutput,
                actualOutput.split(":attr").length == 3);
        
        /**
         * writeStartElement("p", "localName", "http://example.org/URI-ONE");
         * writeAttribute("p", "http://example.org/URI-TWO", "attrName", "value");
         */
        final String EXPECTED_OUTPUT_2 =
                "<?xml version=\"1.0\" ?>"
                + "<root"
                +   " xmlns=\"\">"
                + "<p:localName"
                +   " xmlns:p=\"http://example.org/URI-ONE\""
                +   " xmlns:{generated prefix}=\"http://example.org/URI-TWO\""
                +   " {generated prefix}:attrName=\"value\">"
                + "</p:localName>"
                + "</root>";
    
        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeStartElement("p", "localName", "http://example.org/URI-ONE");
        xmlStreamWriter.writeAttribute("p", "http://example.org/URI-TWO", "attrName", "value");

        actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSamePrefixDifferentURI(): expectedOutput: " + EXPECTED_OUTPUT_2);
            System.out.println("testSamePrefixDifferentURI():   actualOutput: " + actualOutput);
        }

        // there must be 1 xmlns=
        assertTrue(
                "Expected 1 xmlns=, actual output: " + actualOutput,
                actualOutput.split("xmlns=").length == 2);
        
        // there must be 2 xmlns:
        assertTrue(
                "Expected 2 xmlns:, actual output: " + actualOutput,
                actualOutput.split("xmlns:").length == 3);

        // there must be 2 p:localName
        assertTrue(
                "Expected 2 p:localName, actual output: " + actualOutput,
                actualOutput.split("p:localName").length == 3);
    
        // there must be 1 :attrName
        assertTrue(
                "Expected 1 :attrName, actual output: " + actualOutput,
                actualOutput.split(":attrName").length == 2);
        
        /**
         * writeNamespace("p", "http://example.org/URI-ONE");
         * writeAttribute("p", "http://example.org/URI-TWO", "attrName", "value");
         */
        final String EXPECTED_OUTPUT_3 =
                "<?xml version=\"1.0\" ?>"
                + "<root"
                +   " xmlns=\"\""
                +   " xmlns:p=\"http://example.org/URI-ONE\""
                +   " xmlns:{generated prefix}=\"http://example.org/URI-TWO\""
                +   " {generated prefix}:attrName=\"value\">"
                + "</root>";
    
        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeNamespace("p", "http://example.org/URI-ONE");
        xmlStreamWriter.writeAttribute("p", "http://example.org/URI-TWO", "attrName", "value");

        actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSamePrefixDifferentURI(): expectedOutput: " + EXPECTED_OUTPUT_3);
            System.out.println("testSamePrefixDifferentURI():   actualOutput: " + actualOutput);
        }

        // there must be 1 xmlns=
        assertTrue(
                "Expected 1 xmlns=, actual output: " + actualOutput,
                actualOutput.split("xmlns=").length == 2);
        
        // there must be 2 xmlns:
        assertTrue(
                "Expected 2 xmlns:, actual output: " + actualOutput,
                actualOutput.split("xmlns:").length == 3);

        // there must be 1 :attrName
        assertTrue(
                "Expected a :attrName, actual output: " + actualOutput,
                actualOutput.split(":attrName").length == 2);
    
        /**
         * writeNamespace("xmlns", "");
         * writeStartElement("", "localName", "http://example.org/URI-TWO");
         */
        final String EXPECTED_OUTPUT_4 =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\">"
                + "<localName xmlns=\"http://example.org/URI-TWO\">"
                + "xmlns declaration =\"http://example.org/URI-TWO\""
                + "</localName"
                + "</root>";
    
        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        // writeNamespace("xmlns", ""); already done by startDocumentEmptyDefaultNamespace above
        xmlStreamWriter.writeStartElement("", "localName", "http://example.org/URI-TWO");
        xmlStreamWriter.writeCharacters("remap xmlns declaration {generated prefix}=\"http://example.org/URI-TWO\"");

        actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSamePrefixDifferentURI(): expectedOutput: " + EXPECTED_OUTPUT_4);
            System.out.println("testSamePrefixDifferentURI():   actualOutput: " + actualOutput);
        }

        // there must be 2 xmlns=
        assertTrue(
                "Expected 2 xmlns=, actual output: " + actualOutput,
                actualOutput.split("xmlns=").length == 3);
        
        // there must be 0 xmlns:
        assertTrue(
                "Expected 0 xmlns:, actual output: " + actualOutput,
                actualOutput.split("xmlns:").length == 1);

        // there must be 0 :localName
        assertTrue(
                "Expected 0 :localName, actual output: " + actualOutput,
                actualOutput.split(":localName").length == 1);
}
    
    
    // ---------------- Misc ----------------
    
    /**
     * The one case where you don't have to worry about fixup is on
     * attributes that do not have a prefix. Irrespective of the current
     * namespace bindings,
     *
     * writeAttribute("", "", "attrName", "value")
     *
     * is always correct and never requires fixup.
     */
    public void testEmptyDefaultEmptyPrefixEmptyNamespaceURIWriteAttribute() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"\" attrName=\"value\">"
                + "never requires fixup"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentEmptyDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("", "", "attrName", "value");
        xmlStreamWriter.writeCharacters("never requires fixup");

        String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testEmptyDefaultEmptyPrefixEmptyNamespaceURIWriteAttribute(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testEmptyDefaultEmptyPrefixEmptyNamespaceURIWriteAttribute():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }
    
    public void testSpecifiedDefaultEmptyPrefixEmptyNamespaceURIWriteAttribute() throws Exception {
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/uniqueURI\" attrName=\"value\">"
                + "never requires fixup"
                + "</root>";

        // reset to known state
        resetWriter();
        startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
        
        xmlStreamWriter.writeAttribute("", "", "attrName", "value");
        xmlStreamWriter.writeCharacters("never requires fixup");

        String actualOutput = endDocumentSpecifiedDefaultNamespace(xmlStreamWriter);

        if (DEBUG) {
            System.out.println("testSpecifiedDefaultEmptyPrefixEmptyNamespaceURIWriteAttribute(): expectedOutput: " + EXPECTED_OUTPUT);
            System.out.println("testSpecifiedDefaultEmptyPrefixEmptyNamespaceURIWriteAttribute():   actualOutput: " + actualOutput);
        }

        assertEquals(EXPECTED_OUTPUT, actualOutput);
    }
    
    /*--------------- Negative tests with isRepairingNamespaces as FALSE ---------------------- */
    
    private void setUpForNoRepair() {

        xmlOutputFactory.setProperty(XMLOutputFactory.IS_REPAIRING_NAMESPACES, Boolean.FALSE);
        
        // new Writer
        try {
            xmlStreamWriter = xmlOutputFactory.createXMLStreamWriter(byteArrayOutputStream);
            
        } catch (XMLStreamException xmlStreamException) {
            xmlStreamException.printStackTrace();
            fail(xmlStreamException.toString());
        }
    }
    
   /* Tries to assign default namespace to empty URI and again to a different uri in element and 
    * attribute. Expects XMLStreamException .
     * writeNamespace("","");
     * writeAttribute("", "http://example.org/myURI", "attrName", "value");
     */
    public void testEmptyDefaultEmptyPrefixSpecifiedURIWriteAttributeNoRepair(){
        try{
            setUpForNoRepair();
            startDocumentEmptyDefaultNamespace(xmlStreamWriter);
            xmlStreamWriter.writeAttribute("", "http://example.org/myURI", "attrName", "value");
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            fail("XMLStreamException is expected, actualOutput: "+actualOutput);
        }catch(Exception e){
            System.out.println("PASS: caught an expected exception"+e.getMessage());
            e.printStackTrace();
        }
    }
    
    /* Tries to assign default namespace to different uris in element and attribute and  
     * expects XMLStreamException.
     * writeNamespace("","http://example.org/uniqueURI");
     * writeAttribute("", "http://example.org/myURI", "attrName", "value");
     */
    public void testSpecifiedDefaultEmptyPrefixSpecifiedURIWriteAttributeNoRepair(){
        try{
            setUpForNoRepair();
            startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
            xmlStreamWriter.writeAttribute("", "http://example.org/uniqueURI", "attrName", "value");
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            fail("XMLStreamException is expected, actualOutput: "+actualOutput);
        }catch(Exception e){
            System.out.println("PASS: caught an expected exception"+e.getMessage());
            e.printStackTrace();
        }
    }
    
    /* Tries to assign default namespace to same uri twice in element and attribute and  
     * expects XMLStreamException.
     * writeNamespace("","http://example.org/uniqueURI");
     * writeAttribute("", "http://example.org/uniqueURI", "attrName", "value");
     */
    public void testSpecifiedDefaultEmptyPrefixSpecifiedDifferentURIWriteAttributeNoRepair(){
       try{
            setUpForNoRepair();
            startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
            xmlStreamWriter.writeAttribute("", "http://example.org/myURI", "attrName", "value");
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            fail("XMLStreamException is expected, actualOutput: "+actualOutput);
        }catch(Exception e){
            System.out.println("PASS: caught an expected exception"+e.getMessage());
            e.printStackTrace();
        } 
    }
    
    /* Tries to assign prefix 'p' to different uris to attributes of the same element and 
     * expects XMLStreamException.
     * writeAttribute("p", "http://example.org/URI-ONE", "attr1", "value");
     * writeAttribute("p", "http://example.org/URI-TWO", "attr2", "value");
     */
    public void testSamePrefixDiffrentURIWriteAttributeNoRepair(){
        try{
            setUpForNoRepair();
            startDocumentEmptyDefaultNamespace(xmlStreamWriter);
            xmlStreamWriter.writeAttribute("p", "http://example.org/URI-ONE", "attr1", "value");
            xmlStreamWriter.writeAttribute("p", "http://example.org/URI-TWO", "attr2", "value");
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            fail("XMLStreamException is expected, actualOutput: "+actualOutput);
        }catch(Exception e){
            System.out.println("PASS: caught an expected exception"+e.getMessage());
            e.printStackTrace();
        } 
    }
    
    /* Tries to assign prefix 'p' to different uris in element and attribute and expects 
     *  XMLStreamException.
     *  writeStartElement("p","localName","http://example.org/URI-ONE")
     *  writeAttribute("p", "http://example.org/URI-TWO", "attrName", "value")
     */
    public void testSamePrefixDiffrentURIWriteElemAndWriteAttributeNoRepair(){
        try{
            setUpForNoRepair();
            startDocumentEmptyDefaultNamespace(xmlStreamWriter);
            xmlStreamWriter.writeStartElement("p","localName", "http://example.org/URI-ONE");
            xmlStreamWriter.writeAttribute("p", "http://example.org/URI-TWO", "attrName", "value");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            fail("XMLStreamException is expected, actualOutput: "+actualOutput);
        }catch(Exception e){
            System.out.println("PASS: caught an expected exception"+e.getMessage());
            e.printStackTrace();
        } 
    }
    
    /* Tries to write following and expects a StreamException.
     * <root xmlns=""http://example.org/uniqueURI"" xmlns=""http://example.org/myURI"" />    
     */
    public void testDefaultNamespaceDiffrentURIWriteElementNoRepair(){
        try{
            System.out.println("+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++");
            setUpForNoRepair();
            startDocumentSpecifiedDefaultNamespace(xmlStreamWriter);
            xmlStreamWriter.writeNamespace("", "http://example.org/myURI");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++");
            fail("XMLStreamException is expected, actualOutput: "+actualOutput);
        }catch(Exception e){
            System.out.println("PASS: caught an expected exception"+e.getMessage());
            e.printStackTrace();
        } 
    }
    
    /*--------------------------------------------------------------------------
     Miscelleneous tests for writeStartElement() & writeAttribute() methods 
     in case of NOREPAIR 
     --------------------------------------------------------------------------*/
     
    private void startDocument(XMLStreamWriter xmlStreamWriter) 
        throws XMLStreamException{
        xmlStreamWriter.writeStartDocument();
        xmlStreamWriter.writeStartElement("root");
     }
     
    public void testSpecifiedPrefixSpecifiedURIWriteElementNoRepair(){
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root>"
                + "<p:localName></p:localName>"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.writeStartElement("p", "localName", "http://example.org/myURI");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            e.printStackTrace();
            fail("Caught an unexpected exception"+e.getMessage());
        } 
    }
     
    public void testSpecifiedPrefixSpecifiedURIWriteAttributeNoRepair(){
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root p:attrName=\"value\">"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.writeAttribute("p","http://example.org/myURI","attrName","value");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            e.printStackTrace();
            fail("Caught an unexpected exception"+e.getMessage());
        } 
    }
    
    public void testSpecifiedPrefixSpecifiedURISpecifiedNamespcaeWriteElementNoRepair(){
        
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root>"
                + "<p:localName xmlns:p=\"http://example.org/myURI\"></p:localName>"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            
            xmlStreamWriter.writeStartElement("p","localName", "http://example.org/myURI");
            xmlStreamWriter.writeNamespace("p","http://example.org/myURI");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            e.printStackTrace();
            fail("Caught an unexpected exception"+e.getMessage());
        } 
    }
    
    /* writeStartElement("p","localName", "http://example.org/myURI")
     * writeNamespace("p","http://example.org/uniqueURI")
     * This sequence of calls should generate an error as prefix 'p' is binded
     * to different namespace URIs in same namespace context and repairing is
     * disabled.
     */
    
    public void testSpecifiedPrefixSpecifiedURISpecifiedDifferentNamespcaeWriteElementNoRepair(){
        
         try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.writeStartElement("p","localName", "http://example.org/myURI");
            xmlStreamWriter.writeNamespace("p","http://example.org/uniqueURI");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            fail("XMLStreamException is expected as 'p' is rebinded to a different URI in same namespace context");
        }catch(Exception e){
            System.out.println("Caught an expected exception"+e.getMessage());
        } 
    }
     
    public void testEmptyPrefixEmptyURIWriteAttributeNoRepair() {
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root>"
                + "<localName attrName=\"value\"></localName>"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.writeStartElement("localName");
            xmlStreamWriter.writeAttribute("","","attrName","value");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            e.printStackTrace();
            fail("Caught an unexpected exception"+e.getMessage());
        } 
    }
    
    public void testEmptyPrefixNullURIWriteAttributeNoRepair() {
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root>"
                + "<localName attrName=\"value\"></localName>"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.writeStartElement("localName");
            xmlStreamWriter.writeAttribute(null,null,"attrName","value");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            fail("XMLStreamException is expected, actualOutput: "+actualOutput);
        }catch(Exception e){
            System.out.println("PASS: caught an expected exception"+e.getMessage());
            e.printStackTrace();
        } 
    }
    
    public void testDoubleXmlNsNoRepair() {
       try{ 
        // reset to known state
        setUpForNoRepair();
        
        xmlStreamWriter.writeStartDocument();
        xmlStreamWriter.writeStartElement("foo");
        xmlStreamWriter.writeNamespace("xml", XMLConstants.XML_NS_URI);
        xmlStreamWriter.writeAttribute("xml", XMLConstants.XML_NS_URI, "lang", "ja_JP");
        xmlStreamWriter.writeCharacters("Hello");
        xmlStreamWriter.writeEndElement();
        xmlStreamWriter.writeEndDocument();
        
        xmlStreamWriter.flush();
        String actualOutput = byteArrayOutputStream.toString();

        if (DEBUG) {
            System.out.println("testDoubleXmlNsNoRepair(): actualOutput: " + actualOutput);
        }

        // there should be no xmlns:xml
        assertTrue(
                "Expected 0 xmlns:xml, actual output: " + actualOutput,
                actualOutput.split("xmlns:xml").length == 1);
       }catch(Exception e){
           e.printStackTrace();
           fail(e.getMessage());
       }
    }
    
    public void testSpecifiedURIWriteAttributeNoRepair(){
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root>"
                + "<p:localName p:attrName=\"value\"></p:localName>"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.writeStartElement("p","localName", "http://example.org/myURI");
            xmlStreamWriter.writeAttribute("http://example.org/myURI","attrName","value");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            System.out.println("Caught an expected exception"+e.getMessage());
        } 
    }
    
    public void testSpecifiedURIWriteAttributeWithRepair() {
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root>"
                + "<p:localName xmlns:p=\"http://example.org/myURI\" p:attrName=\"value\"></p:localName>"
                + "</root>";
        try{
            startDocument(xmlStreamWriter);
            xmlStreamWriter.writeStartElement("p","localName", "http://example.org/myURI");
            xmlStreamWriter.writeNamespace("p","http://example.org/myURI");
            xmlStreamWriter.writeAttribute("http://example.org/myURI","attrName","value");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            e.printStackTrace();
            fail("Exception occured: "+e.getMessage());
        } 
    }
    
    public void testSpecifiedDefaultInDifferentElementsNoRepair() {
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root>"
                + "<localName xmlns=\"http://example.org/myURI\">"
                + "<child xmlns=\"http://example.org/uniqueURI\"></child>"
                + "</localName>"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.writeStartElement("localName");
            xmlStreamWriter.writeDefaultNamespace("http://example.org/myURI");
            xmlStreamWriter.writeStartElement("child");
            xmlStreamWriter.writeDefaultNamespace("http://example.org/uniqueURI");
            xmlStreamWriter.writeEndElement();
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            e.printStackTrace();
            fail("Exception occured: "+e.getMessage());
        } 
    }
    
    /*------------- Tests for setPrefix() and setDefaultNamespace() methods --------------------*/
   
    public void testSetPrefixWriteNamespaceNoRepair(){
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns:p=\"http://example.org/myURI\">"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.setPrefix("p","http://example.org/myURI");
            xmlStreamWriter.writeNamespace("p", "http://example.org/myURI");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            System.out.println("Caught an expected exception"+e.getMessage());
        } 
    }
    
    public void testSetPrefixWriteNamespaceWithRepair(){
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns:p=\"http://example.org/myURI\">"
                + "</root>";
        try{
            startDocument(xmlStreamWriter);
            xmlStreamWriter.setPrefix("p","http://example.org/myURI");
            xmlStreamWriter.writeNamespace("p", "http://example.org/myURI");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            System.out.println("Caught an expected exception"+e.getMessage());
        } 
    }
    
    public void testSetDefaultNamespaceWriteNamespaceNoRepair() {
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/myURI\">"
                + "</root>";
        try{
            setUpForNoRepair();
            startDocument(xmlStreamWriter);
            xmlStreamWriter.setDefaultNamespace("http://example.org/myURI");
            xmlStreamWriter.writeNamespace("", "http://example.org/myURI");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            System.out.println("Caught an expected exception"+e.getMessage());
        } 
    }
    
    public void testSetDefaultNamespaceWriteNamespaceWithRepair() {
        final String EXPECTED_OUTPUT =
                "<?xml version=\"1.0\" ?>"
                + "<root xmlns=\"http://example.org/myURI\">"
                + "</root>";
        try{
            startDocument(xmlStreamWriter);
            xmlStreamWriter.setDefaultNamespace("http://example.org/myURI");
            xmlStreamWriter.writeNamespace("", "http://example.org/myURI");
            xmlStreamWriter.writeEndElement();
            String actualOutput = endDocumentEmptyDefaultNamespace(xmlStreamWriter);
            System.out.println("actualOutput: "+ actualOutput);
            assertEquals(EXPECTED_OUTPUT, actualOutput);
        }catch(Exception e){
            System.out.println("Caught an expected exception"+e.getMessage());
        } 
    }
}
