/****************************************************************/
/* Archive unit                                                 */
/* (c) Christophe CALMEJANE (Ze KiLleR) - 1999-05               */
/****************************************************************/

/*
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifdef SU_USE_ARCH
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef _WIN32
#include <sys/utime.h>
#else /* !_WIN32 */
#include <utime.h>
#endif /* _WIN32 */
#include "skyutils.h"
#ifdef HAVE_ZLIB
#include <zlib.h>
#endif /* HAVE_ZLIB */
#ifdef HAVE_BZLIB
#include <bzlib.h>
#endif /* HAVE_BZLIB */

#ifndef SU_TRACE_INTERNAL
#undef malloc
#undef calloc
#undef realloc
#undef strdup
#undef free
#endif /* !SU_TRACE_INTERNAL */

#define SU_AR_SIGNATURE "SkyArch3"


/* ------------------- INTERNAL FUNCTIONS ------------------- */

bool _SU_AR_CompressFile(SU_PResHdr RH)
{
  FILE *fp;
  struct stat st;

  fp = fopen((char *)RH->Data,"rb");
  if(fp == NULL)
    return false;
  if(stat((char *)RH->Data,&st) != 0)
    return false;
  fseek(fp,0,SEEK_END);
  RH->OrigSize = ftell(fp);
  RH->OrigTime = st.st_ctime;
  fclose(fp);

  switch(RH->CompType)
  {
    case SU_ARCH_COMP_NONE :
      /* RH->Data doesn't change */
      RH->CompSize = RH->OrigSize;
      break;
#ifdef HAVE_ZLIB
    case SU_ARCH_COMP_Z :
      /* Set RH->Data to new FileName */
      /* Set RH->CompSize to correct value */
      break;
#endif /* HAVE_ZLIB */
#ifdef HAVE_BZLIB
    case SU_ARCH_COMP_BZ :
      /* Set RH->Data to new FileName */
      /* Set RH->CompSize to correct value */
      break;
#endif /* HAVE_BZLIB */
    default :
      return false;
  }

  return true;
}

bool _SU_AR_CopyFileToArchive(FILE *fp,SU_PResHdr RH,const char FileName[])
{
  FILE *in;
  char Buf[32768];
  unsigned long int total = 0,len;

  in = fopen(FileName,"rb");
  if(in == NULL)
    return false;

  while(total < RH->CompSize)
  {
    len = RH->CompSize - total;
    if(len > sizeof(Buf))
      len = sizeof(Buf);
    if(fread(Buf,1,len,in) != len)
    {
      fclose(in);
      return false;
    }
    if(fwrite(Buf,1,len,fp) != len)
    {
      fclose(in);
      return false;
    }
    total += len;
  }
  fclose(in);
  return true;
}

bool _SU_AR_CopyFileToDisk(FILE *fp,SU_PResHdr RH,const char FileName[])
{
  FILE *out;
  char Buf[32768];
  unsigned long int total = 0,len;

  out = fopen(FileName,"wb");
  if(out == NULL)
    return false;

  while(total < RH->OrigSize)
  {
    len = RH->OrigSize - total;
    if(len > sizeof(Buf))
      len = sizeof(Buf);
    if(fread(Buf,1,len,fp) != len)
    {
      fclose(out);
      return false;
    }
    if(fwrite(Buf,1,len,out) != len)
    {
      fclose(out);
      return false;
    }
    total += len;
  }
  fclose(out);
  return true;
}

SU_PArch _SU_AR_ReadHeaders(FILE *fp)
{
  SU_PArch Arch;
  unsigned long int NbRes,i;
  char Signature[sizeof(SU_AR_SIGNATURE)-1];

  if(fread(&Signature,1,sizeof(Signature),fp) != sizeof(Signature))
  {
    fclose(fp);
    return NULL;
  }
  if(strncmp(Signature,SU_AR_SIGNATURE,sizeof(Signature)) != 0)
  {
    fclose(fp);
    return NULL;
  }
  if(fread(&NbRes,1,sizeof(NbRes),fp) != sizeof(NbRes))
  {
    fclose(fp);
    return NULL;
  }

  Arch = (SU_PArch) malloc(sizeof(SU_TArch));
  memset(Arch,0,sizeof(SU_TArch));
  Arch->fp = fp;
  Arch->NbRes = NbRes;
  Arch->Resources = (SU_TResHdr *) malloc(sizeof(SU_TResHdr)*NbRes);
  memset(Arch->Resources,0,sizeof(SU_TResHdr)*NbRes);
  for(i=0;i<NbRes;i++)
  {
    if(fread(&Arch->Resources[i].CompSize,1,sizeof(Arch->Resources[i].CompSize),fp) != sizeof(Arch->Resources[i].CompSize))
    {
      SU_AR_CloseArchive(Arch);
      return NULL;
    }
    if(fread(&Arch->Resources[i].CompType,1,sizeof(Arch->Resources[i].CompType),fp) != sizeof(Arch->Resources[i].CompType))
    {
      SU_AR_CloseArchive(Arch);
      return NULL;
    }
    if(fread(&Arch->Resources[i].Reserved,1,sizeof(Arch->Resources[i].Reserved),fp) != sizeof(Arch->Resources[i].Reserved))
    {
      SU_AR_CloseArchive(Arch);
      return NULL;
    }
    if(fread(&Arch->Resources[i].OrigSize,1,sizeof(Arch->Resources[i].OrigSize),fp) != sizeof(Arch->Resources[i].OrigSize))
    {
      SU_AR_CloseArchive(Arch);
      return NULL;
    }
    if(fread(&Arch->Resources[i].OrigTime,1,sizeof(Arch->Resources[i].OrigTime),fp) != sizeof(Arch->Resources[i].OrigTime))
    {
      SU_AR_CloseArchive(Arch);
      return NULL;
    }
    Arch->Resources[i].Offset = ftell(fp);
    if(fseek(fp,Arch->Resources[i].CompSize,SEEK_CUR) != 0)
    {
      SU_AR_CloseArchive(Arch);
      return NULL;
    }
  }

  return Arch;
}

bool _SU_AR_Flush(SU_PArch Arch)
{
  unsigned int i;
  char Signature[sizeof(SU_AR_SIGNATURE)-1];
  bool res = true;
  unsigned long int Ofs = 0;

  strncpy(Signature,SU_AR_SIGNATURE,sizeof(Signature));
  if(fwrite(&Signature,1,sizeof(Signature),Arch->fp) != sizeof(Signature))
    res = false;
  if(fwrite(&Arch->NbRes,1,sizeof(Arch->NbRes),Arch->fp) != sizeof(Arch->NbRes))
    res = false;

  for(i=0;i<Arch->NbRes;i++)
  {
    if(Arch->Resources[i].IsFile)
    {
      if(!_SU_AR_CompressFile(&Arch->Resources[i]))
        res = false;
    }

    if(fwrite(&Arch->Resources[i].CompSize,1,sizeof(Arch->Resources[i].CompSize),Arch->fp) != sizeof(Arch->Resources[i].CompSize))
      res = false;
    if(fwrite(&Arch->Resources[i].CompType,1,sizeof(Arch->Resources[i].CompType),Arch->fp) != sizeof(Arch->Resources[i].CompType))
      res = false;
    if(fwrite(&Arch->Resources[i].Reserved,1,sizeof(Arch->Resources[i].Reserved),Arch->fp) != sizeof(Arch->Resources[i].Reserved))
      res = false;
    if(fwrite(&Arch->Resources[i].OrigSize,1,sizeof(Arch->Resources[i].OrigSize),Arch->fp) != sizeof(Arch->Resources[i].OrigSize))
      res = false;
    if(fwrite(&Arch->Resources[i].OrigTime,1,sizeof(Arch->Resources[i].OrigTime),Arch->fp) != sizeof(Arch->Resources[i].OrigTime))
      res = false;
    if(Arch->Resources[i].Data != NULL)
    {
      if(Arch->Resources[i].IsFile)
      {
        if(!_SU_AR_CopyFileToArchive(Arch->fp,&Arch->Resources[i],(char *)Arch->Resources[i].Data))
          res = false;
      }
      else
      {
        if(fwrite(Arch->Resources[i].Data,1,Arch->Resources[i].CompSize,Arch->fp) != Arch->Resources[i].CompSize)
          res = false;
      }
      free(Arch->Resources[i].Data);
    }
    else
      res = false;
  }
  /* Write ofs to start of archive... should always be 0 */
  if(fwrite(&Ofs,1,sizeof(Ofs),Arch->fp) != sizeof(Ofs))
    res = false;
  return res;
}

/* ------------------- EXPORTED FUNCTIONS ------------------- */

/* Opens a skyutils archive file (or a binary file [exe/dll] if the archive is selfcontained) */
SU_PArch SU_AR_OpenArchive(const char FileName[])
{
  FILE *fp;
  unsigned long int ofs;

  fp = fopen(FileName,"rb");
  if(fp == NULL)
    return NULL;
  if(fseek(fp,-(signed)sizeof(ofs),SEEK_END) != 0)
  {
    fclose(fp);
    return NULL;
  }
  if(fread(&ofs,1,sizeof(ofs),fp) != sizeof(ofs))
  {
    fclose(fp);
    return NULL;
  }
  if(fseek(fp,ofs,SEEK_SET) != 0)
  {
    fclose(fp);
    return NULL;
  }
  return _SU_AR_ReadHeaders(fp);
}

/* Reads resource ResNum (0 is the first one) (NULL if failed) */
SU_PRes SU_AR_ReadRes(SU_PArch Arch,const unsigned int ResNum,bool GetData)
{
  SU_PRes Res;

  if(Arch == NULL)
    return NULL;
  if(ResNum >= Arch->NbRes)
    return NULL;
  Res = (SU_PRes) malloc(sizeof(SU_TRes));
  memset(Res,0,sizeof(SU_TRes));
  Res->Size = Arch->Resources[ResNum].OrigSize;
  Res->Stamp = Arch->Resources[ResNum].OrigTime;
  if(GetData)
  {
    if(fseek(Arch->fp,Arch->Resources[ResNum].Offset,SEEK_SET) != 0)
    {
      SU_AR_FreeRes(Res);
      return NULL;
    }
    switch(Arch->Resources[ResNum].CompType)
    {
      case SU_ARCH_COMP_NONE :
        Res->Data = malloc(Res->Size);
        if(fread(Res->Data,1,Res->Size,Arch->fp) != Res->Size)
        {
          SU_AR_FreeRes(Res);
          return NULL;
        }
        break;
#ifdef HAVE_ZLIB
      case SU_ARCH_COMP_Z :
        uncompress(NULL,NULL,NULL,0);
        break;
#endif /* HAVE_ZLIB */
#ifdef HAVE_BZLIB
      case SU_ARCH_COMP_BZ :
        BZ2_bzBuffToBuffDecompress(NULL,NULL,NULL,0,0,0);
        break;
#endif /* HAVE_BZLIB */
      default :
        SU_AR_FreeRes(Res);
        return NULL;
    }
  }
  return Res;
}

/* Save resource ResNum to FileName (0 is the first one) (true on success) */
bool SU_AR_ReadResFile(SU_PArch Arch,const unsigned int ResNum,const char FileName[])
{
  struct utimbuf ut;
  FILE *out;

  if(Arch == NULL)
    return false;

  if(ResNum >= Arch->NbRes)
    return false;

  out = fopen(FileName,"wb");
  if(out == NULL)
    return false;
  fclose(out);

  if(fseek(Arch->fp,Arch->Resources[ResNum].Offset,SEEK_SET) != 0)
  {
    unlink(FileName);
    return false;
  }

  switch(Arch->Resources[ResNum].CompType)
  {
    case SU_ARCH_COMP_NONE :
      if(!_SU_AR_CopyFileToDisk(Arch->fp,&Arch->Resources[ResNum],FileName))
      {
        unlink(FileName);
        return false;
      }
      break;
#ifdef HAVE_ZLIB
    case SU_ARCH_COMP_Z :
      break;
#endif /* HAVE_ZLIB */
#ifdef HAVE_BZLIB
    case SU_ARCH_COMP_BZ :
      break;
#endif /* HAVE_BZLIB */
    default :
      unlink(FileName);
      return false;
  }

  ut.actime = 0;
  ut.modtime = 0;
  utime(FileName,&ut);
  return true;
}


/* Creates a new archive file. FileName can't be NULL */
SU_PArch SU_AR_CreateArchive(const char FileName[])
{
  SU_PArch Arch;
  FILE *fp;

  fp = fopen(FileName,"wb");
  if(fp == NULL)
    return NULL;

  Arch = (SU_PArch) malloc(sizeof(SU_TArch));
  memset(Arch,0,sizeof(SU_TArch));
  Arch->fp = fp;
  Arch->Flush = true;

  return Arch;
}

/* Adds a resource to the archive (Data can be freed upon return) (true on success) */
bool SU_AR_AddRes(SU_PArch Arch,void *Data,unsigned long int Size,time_t Time,SU_AR_COMP_TYPE Type)
{
  SU_PResHdr RH;

  if(Arch == NULL)
    return false;

  Arch->NbRes++;
  Arch->Resources = (SU_TResHdr *) realloc(Arch->Resources,sizeof(SU_TResHdr)*Arch->NbRes);
  RH = &Arch->Resources[Arch->NbRes-1];
  memset(RH,0,sizeof(SU_TResHdr));
  RH->OrigSize = Size;
  RH->OrigTime = Time;
  RH->CompType = Type;
  if((Size == 0) && (Time == 0)) /* Data represents a FileName */
  {
    RH->Data = strdup(Data);
    RH->IsFile = true;
  }
  else
  {
    switch(Type)
    {
      case SU_ARCH_COMP_NONE :
        RH->Data = malloc(Size);
        memcpy(RH->Data,Data,Size);
        RH->CompSize = Size;
        break;
#ifdef HAVE_ZLIB
      case SU_ARCH_COMP_Z :
        break;
#endif /* HAVE_ZLIB */
#ifdef HAVE_BZLIB
      case SU_ARCH_COMP_BZ :
        break;
#endif /* HAVE_BZLIB */
      default :
        Arch->NbRes--;
        free(RH);
        return false;
    }
  }
  return true;
}

/* Adds a file resource to the archive (true on success) */
bool SU_AR_AddResFile(SU_PArch Arch,const char FileName[],SU_AR_COMP_TYPE Type)
{
  FILE *fp;
  struct stat st;

  if(Arch == NULL)
    return false;

  fp = fopen(FileName,"rb");
  if(fp == NULL)
    return false;
  if(stat(FileName,&st) != 0)
    return false;
  return SU_AR_AddRes(Arch,(void *)FileName,0,0,Type);
}

/* Closes a previous opened/created archive (true on success) */
bool SU_AR_CloseArchive(SU_PArch Arch)
{
  bool res = true;

  if(Arch == NULL)
    return true;

  if(Arch->Flush)
    res = _SU_AR_Flush(Arch);

  if(Arch->Resources != NULL)
    free(Arch->Resources);
  fclose(Arch->fp);
  free(Arch);
  return res;
}

/* Frees a previous returned resource */
void SU_AR_FreeRes(SU_PRes Res)
{
  if(Res == NULL)
    return;

  if(Res->Data != NULL)
    free(Res->Data);
  free(Res);
}

/* Returns supported compression types (as a bit field) */
SU_AR_COMP_TYPE SU_AR_SupportedComps(void)
{
  SU_AR_COMP_TYPE Flags = SU_ARCH_COMP_NONE;
#ifdef HAVE_ZLIB
  /* SetFlag(Flags,SU_ARCH_COMP_Z); Not supported yet */
#endif /* HAVE_ZLIB */
#ifdef HAVE_BZLIB
  /* SetFlag(Flags,SU_ARCH_COMP_BZ); Not supported yet */
#endif /* HAVE_BZLIB */
  return Flags;
}

#endif /* SU_USE_ARCH */

