#include "test_data.hpp"
#include "TriangleMesh.hpp"
#include "GCodeReader.hpp"
#include "Config.hpp"
#include "Print.hpp"
#include <cstdlib>
#include <string>

using namespace std::string_literals;
using namespace std;

namespace Slic3r { namespace Test {

// Mesh enumeration to name mapping
const std::unordered_map<TestMesh, const char*, TestMeshHash> mesh_names {
    std::make_pair<TestMesh, const char*>(TestMesh::A,"A"),
    std::make_pair<TestMesh, const char*>(TestMesh::L,"L"), 
    std::make_pair<TestMesh, const char*>(TestMesh::V,"V"), 
    std::make_pair<TestMesh, const char*>(TestMesh::_40x10,"40x10"), 
    std::make_pair<TestMesh, const char*>(TestMesh::cube_20x20x20,"cube_20x20x20"), 
    std::make_pair<TestMesh, const char*>(TestMesh::sphere_50mm,"sphere_50mm"), 
    std::make_pair<TestMesh, const char*>(TestMesh::bridge,"bridge"), 
    std::make_pair<TestMesh, const char*>(TestMesh::bridge_with_hole,"bridge_with_hole"), 
    std::make_pair<TestMesh, const char*>(TestMesh::cube_with_concave_hole,"cube_with_concave_hole"),
    std::make_pair<TestMesh, const char*>(TestMesh::cube_with_hole,"cube_with_hole"), 
    std::make_pair<TestMesh, const char*>(TestMesh::gt2_teeth,"gt2_teeth"), 
    std::make_pair<TestMesh, const char*>(TestMesh::ipadstand,"ipadstand"), 
    std::make_pair<TestMesh, const char*>(TestMesh::overhang,"overhang"), 
    std::make_pair<TestMesh, const char*>(TestMesh::pyramid,"pyramid"), 
    std::make_pair<TestMesh, const char*>(TestMesh::sloping_hole,"sloping_hole"), 
    std::make_pair<TestMesh, const char*>(TestMesh::slopy_cube,"slopy_cube"), 
    std::make_pair<TestMesh, const char*>(TestMesh::small_dorito,"small_dorito"), 
    std::make_pair<TestMesh, const char*>(TestMesh::step,"step"), 
    std::make_pair<TestMesh, const char*>(TestMesh::two_hollow_squares,"two_hollow_squares")
};



TriangleMesh mesh(TestMesh m) {
    Point3s facets;
    Pointf3s vertices;
    switch(m) {
        case TestMesh::cube_with_hole:
            vertices = { Pointf3(0,0,0), Pointf3(0,0,10), Pointf3(0,20,0), Pointf3(0,20,10), Pointf3(20,0,0), Pointf3(20,0,10), Pointf3(5,5,0), Pointf3(15,5,0), Pointf3(5,15,0), Pointf3(20,20,0), Pointf3(15,15,0), Pointf3(20,20,10), Pointf3(5,5,10), Pointf3(5,15,10), Pointf3(15,5,10), Pointf3(15,15,10) };
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(2,1,3), Point3(1,0,4), Point3(5,1,4), Point3(6,7,4), Point3(8,2,9), Point3(0,2,8), Point3(10,8,9), Point3(0,8,6), Point3(0,6,4), Point3(4,7,9), Point3(7,10,9), Point3(2,3,9), Point3(9,3,11), Point3(12,1,5), Point3(13,3,12), Point3(14,12,5), Point3(3,1,12), Point3(11,3,13), Point3(11,15,5), Point3(11,13,15), Point3(15,14,5), Point3(5,4,9), Point3(11,5,9), Point3(8,13,12), Point3(6,8,12), Point3(10,15,13), Point3(8,10,13), Point3(15,10,14), Point3(14,10,7), Point3(14,7,12), Point3(12,7,6) 
            });
            break;
        case TestMesh::cube_with_concave_hole:
            vertices = std::vector<Pointf3>({
                Pointf3(-10,-10,-5), Pointf3(-10,-10,5), Pointf3(-10,10,-5), Pointf3(-10,10,5), Pointf3(10,-10,-5), Pointf3(10,-10,5), Pointf3(-5,-5,-5), Pointf3(5,-5,-5), Pointf3(5,5,-5), Pointf3(5,10,-5), Pointf3(-5,5,-5), Pointf3(3.06161699911402e-16,5,-5), Pointf3(5,0,-5), Pointf3(0,0,-5), Pointf3(10,5,-5), Pointf3(5,10,5), Pointf3(-5,-5,5), Pointf3(5,0,5), Pointf3(5,-5,5), Pointf3(-5,5,5), Pointf3(10,5,5), Pointf3(5,5,5), Pointf3(3.06161699911402e-16,5,5), Pointf3(0,0,5) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(2,1,3), Point3(1,0,4), Point3(5,1,4), Point3(6,7,4), Point3(8,2,9), Point3(10,2,11), Point3(11,12,13), Point3(0,2,10), Point3(0,10,6), Point3(0,6,4), Point3(11,2,8), Point3(4,7,12), Point3(4,12,8), Point3(12,11,8), Point3(14,4,8), Point3(2,3,9), Point3(9,3,15), Point3(16,1,5), Point3(17,18,5), Point3(19,3,16), Point3(20,21,5), Point3(18,16,5), Point3(3,1,16), Point3(22,3,19), Point3(21,3,22), Point3(21,17,5), Point3(21,22,17), Point3(21,15,3), Point3(23,17,22), Point3(5,4,14), Point3(20,5,14), Point3(20,14,21), Point3(21,14,8), Point3(9,15,21), Point3(8,9,21), Point3(10,19,16), Point3(6,10,16), Point3(11,22,19), Point3(10,11,19), Point3(13,23,11), Point3(11,23,22), Point3(23,13,12), Point3(17,23,12), Point3(17,12,18), Point3(18,12,7), Point3(18,7,16), Point3(16,7,6) 
            });
            break;
        case TestMesh::V:
            vertices = std::vector<Pointf3>({
                Pointf3(-14,0,20), Pointf3(-14,15,20), Pointf3(0,0,0), Pointf3(0,15,0), Pointf3(-4,0,20), Pointf3(-4,15,20), Pointf3(5,0,7.14286), Pointf3(10,0,0), Pointf3(24,0,20), Pointf3(14,0,20), Pointf3(10,15,0), Pointf3(5,15,7.14286), Pointf3(14,15,20), Pointf3(24,15,20) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(2,1,3), Point3(1,0,4), Point3(5,1,4), Point3(4,0,2), Point3(6,4,2), Point3(7,6,2), Point3(8,9,7), Point3(9,6,7), Point3(2,3,7), Point3(7,3,10), Point3(1,5,3), Point3(3,5,11), Point3(11,12,13), Point3(11,13,3), Point3(3,13,10), Point3(5,4,6), Point3(11,5,6), Point3(6,9,11), Point3(11,9,12), Point3(12,9,8), Point3(13,12,8), Point3(8,7,10), Point3(13,8,10) 
            });
            break;
        case TestMesh::L:
            vertices = std::vector<Pointf3>({
                Pointf3(0,10,0), Pointf3(0,10,10), Pointf3(0,20,0), Pointf3(0,20,10), Pointf3(10,10,0), Pointf3(10,10,10), Pointf3(20,20,0), Pointf3(20,0,0), Pointf3(10,0,0), Pointf3(20,20,10), Pointf3(10,0,10), Pointf3(20,0,10) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(2,1,3), Point3(4,5,1), Point3(0,4,1), Point3(0,2,4), Point3(4,2,6), Point3(4,6,7), Point3(4,7,8), Point3(2,3,6), Point3(6,3,9), Point3(3,1,5), Point3(9,3,5), Point3(10,11,5), Point3(11,9,5), Point3(5,4,10), Point3(10,4,8), Point3(10,8,7), Point3(11,10,7), Point3(11,7,6), Point3(9,11,6) 
            });
            break;
        case TestMesh::overhang:
            vertices = std::vector<Pointf3>({
                Pointf3(1364.68505859375,614.398010253906,20.002498626709), Pointf3(1389.68505859375,614.398010253906,20.002498626709), Pointf3(1377.18505859375,589.398986816406,20.002498626709), Pointf3(1389.68505859375,589.398986816406,20.002498626709), Pointf3(1389.68505859375,564.398986816406,20.0014991760254), Pointf3(1364.68505859375,589.398986816406,20.002498626709), Pointf3(1364.68505859375,564.398986816406,20.0014991760254), Pointf3(1360.93505859375,589.398986816406,17.0014991760254), Pointf3(1360.93505859375,585.64697265625,17.0014991760254), Pointf3(1357.18505859375,564.398986816406,17.0014991760254), Pointf3(1364.68505859375,589.398986816406,17.0014991760254), Pointf3(1364.68505859375,571.899963378906,17.0014991760254), Pointf3(1364.68505859375,564.398986816406,17.0014991760254), Pointf3(1348.43603515625,564.398986816406,17.0014991760254), Pointf3(1352.80908203125,589.398986816406,17.0014991760254), Pointf3(1357.18408203125,589.398986816406,17.0014991760254), Pointf3(1357.18310546875,614.398010253906,17.0014991760254), Pointf3(1364.68505859375,606.89599609375,17.0014991760254), Pointf3(1364.68505859375,614.398010253906,17.0014991760254), Pointf3(1352.18603515625,564.398986816406,20.0014991760254), Pointf3(1363.65405273438,589.398986816406,23.3004989624023), Pointf3(1359.46704101562,589.398986816406,23.3004989624023), Pointf3(1358.37109375,564.398986816406,23.3004989624023), Pointf3(1385.56103515625,564.398986816406,23.3004989624023), Pointf3(1373.06311035156,589.398986816406,23.3004989624023), Pointf3(1368.80810546875,564.398986816406,23.3004989624023), Pointf3(1387.623046875,589.398986816406,23.3004989624023), Pointf3(1387.623046875,585.276000976562,23.3004989624023), Pointf3(1389.68505859375,589.398986816406,23.3004989624023), Pointf3(1389.68505859375,572.64599609375,23.3004989624023), Pointf3(1389.68505859375,564.398986816406,23.3004989624023), Pointf3(1367.77709960938,589.398986816406,23.3004989624023), Pointf3(1366.7470703125,564.398986816406,23.3004989624023), Pointf3(1354.31201171875,589.398986816406,23.3004989624023), Pointf3(1352.18603515625,564.398986816406,23.3004989624023), Pointf3(1389.68505859375,614.398010253906,23.3004989624023), Pointf3(1377.31701660156,614.398010253906,23.3004989624023), Pointf3(1381.43908691406,589.398986816406,23.3004989624023), Pointf3(1368.80700683594,614.398010253906,23.3004989624023), Pointf3(1368.80810546875,589.398986816406,23.3004989624023), Pointf3(1356.43908691406,614.398010253906,23.3004989624023), Pointf3(1357.40502929688,589.398986816406,23.3004989624023), Pointf3(1360.56201171875,614.398010253906,23.3004989624023), Pointf3(1348.705078125,614.398010253906,23.3004989624023), Pointf3(1350.44506835938,589.398986816406,23.3004989624023), Pointf3(1389.68505859375,606.153015136719,23.3004989624023), Pointf3(1347.35205078125,589.398986816406,23.3004989624023), Pointf3(1346.56005859375,589.398986816406,23.3004989624023), Pointf3(1346.56005859375,594.159912109375,17.0014991760254), Pointf3(1346.56005859375,589.398986816406,17.0014991760254), Pointf3(1346.56005859375,605.250427246094,23.3004989624023), Pointf3(1346.56005859375,614.398010253906,23.3004989624023), Pointf3(1346.56005859375,614.398010253906,20.8258285522461), Pointf3(1346.56005859375,614.398010253906,17.0014991760254), Pointf3(1346.56005859375,564.398986816406,19.10133934021), Pointf3(1346.56005859375,567.548583984375,23.3004989624023), Pointf3(1346.56005859375,564.398986816406,17.0020332336426), Pointf3(1346.56005859375,564.398986816406,23.0018501281738), Pointf3(1346.56005859375,564.398986816406,23.3004989624023), Pointf3(1346.56005859375,575.118957519531,17.0014991760254), Pointf3(1346.56005859375,574.754028320312,23.3004989624023) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(2,3,4), Point3(2,5,0), Point3(4,6,2), Point3(2,6,5), Point3(2,1,3), Point3(7,8,9), Point3(10,9,8), Point3(11,9,10), Point3(12,9,11), Point3(9,13,14), Point3(7,15,16), Point3(10,17,0), Point3(10,0,5), Point3(12,11,6), Point3(18,16,0), Point3(6,19,13), Point3(6,13,9), Point3(9,12,6), Point3(17,18,0), Point3(11,10,5), Point3(11,5,6), Point3(14,16,15), Point3(17,7,18), Point3(16,18,7), Point3(14,15,9), Point3(7,9,15), Point3(7,17,8), Point3(10,8,17), Point3(20,21,22), Point3(23,24,25), Point3(26,23,27), Point3(28,27,23), Point3(29,28,23), Point3(30,29,23), Point3(25,31,32), Point3(22,33,34), Point3(35,36,37), Point3(24,38,39), Point3(21,40,41), Point3(38,42,20), Point3(33,43,44), Point3(6,4,23), Point3(6,23,25), Point3(36,35,1), Point3(1,0,38), Point3(1,38,36), Point3(29,30,4), Point3(25,32,6), Point3(40,42,0), Point3(35,45,1), Point3(4,3,28), Point3(4,28,29), Point3(3,1,45), Point3(3,45,28), Point3(22,34,19), Point3(19,6,32), Point3(19,32,22), Point3(42,38,0), Point3(30,23,4), Point3(0,16,43), Point3(0,43,40), Point3(24,37,36), Point3(38,24,36), Point3(24,23,37), Point3(37,23,26), Point3(22,32,20), Point3(20,32,31), Point3(33,41,40), Point3(43,33,40), Point3(45,35,26), Point3(37,26,35), Point3(33,44,34), Point3(44,43,46), Point3(20,42,21), Point3(40,21,42), Point3(31,39,38), Point3(20,31,38), Point3(33,22,41), Point3(21,41,22), Point3(31,25,39), Point3(24,39,25), Point3(26,27,45), Point3(28,45,27), Point3(47,48,49), Point3(47,50,48), Point3(51,48,50), Point3(52,48,51), Point3(53,48,52), Point3(54,55,56), Point3(57,55,54), Point3(58,55,57), Point3(49,59,47), Point3(60,56,55), Point3(59,56,60), Point3(60,47,59), Point3(48,53,16), Point3(56,13,19), Point3(54,56,19), Point3(56,59,13), Point3(59,49,14), Point3(59,14,13), Point3(49,48,16), Point3(49,16,14), Point3(44,46,60), Point3(44,60,55), Point3(51,50,43), Point3(19,34,58), Point3(19,58,57), Point3(53,52,16), Point3(43,16,52), Point3(43,52,51), Point3(57,54,19), Point3(47,60,46), Point3(55,58,34), Point3(55,34,44), Point3(50,47,46), Point3(50,46,43) 
            });
            break;
        case TestMesh::_40x10:
            vertices = std::vector<Pointf3>({
                Pointf3(12.8680295944214,29.5799007415771,12), Pointf3(11.7364797592163,29.8480796813965,12), Pointf3(11.1571502685547,29.5300102233887,12), Pointf3(10.5814504623413,29.9830799102783,12), Pointf3(10,29.6000003814697,12), Pointf3(9.41855144500732,29.9830799102783,12), Pointf3(8.84284687042236,29.5300102233887,12), Pointf3(8.26351833343506,29.8480796813965,12), Pointf3(7.70256900787354,29.3210391998291,12), Pointf3(7.13196802139282,29.5799007415771,12), Pointf3(6.59579277038574,28.9761600494385,12), Pointf3(6.03920221328735,29.1821594238281,12), Pointf3(5.53865718841553,28.5003795623779,12), Pointf3(5,28.6602592468262,12), Pointf3(4.54657793045044,27.9006500244141,12), Pointf3(4.02841377258301,28.0212306976318,12), Pointf3(3.63402199745178,27.1856994628906,12), Pointf3(3.13758301734924,27.2737407684326,12), Pointf3(2.81429696083069,26.3659801483154,12), Pointf3(2.33955597877502,26.4278793334961,12), Pointf3(2.0993549823761,25.4534206390381,12), Pointf3(1.64512205123901,25.4950904846191,12), Pointf3(1.49962198734283,24.4613399505615,12), Pointf3(1.0636739730835,24.4879894256592,12), Pointf3(1.02384400367737,23.4042091369629,12), Pointf3(0.603073298931122,23.4202003479004,12), Pointf3(0.678958415985107,22.2974300384521,12), Pointf3(0.269550800323486,22.3061599731445,12), Pointf3(0.469994693994522,21.1571502685547,12), Pointf3(0.067615881562233,21.1609306335449,12), Pointf3(0.399999290704727,20,12), Pointf3(0,20,12), Pointf3(0.399999290704727,5,12), Pointf3(0,5,12), Pointf3(0.456633001565933,4.2804012298584,12), Pointf3(0.0615576282143593,4.21782684326172,12), Pointf3(0.625140011310577,3.5785219669342,12), Pointf3(0.244717106223106,3.45491504669189,12), Pointf3(0.901369392871857,2.91164398193359,12), Pointf3(0.544967114925385,2.73004698753357,12), Pointf3(1.27852201461792,2.29618692398071,12), Pointf3(0.954914808273315,2.06107401847839,12), Pointf3(1.74730801582336,1.74730801582336,12), Pointf3(1.46446597576141,1.46446597576141,12), Pointf3(2.29618692398071,1.27852201461792,12), Pointf3(2.06107401847839,0.954914808273315,12), Pointf3(2.91164398193359,0.901369392871857,12), Pointf3(2.73004698753357,0.544967114925385,12), Pointf3(3.5785219669342,0.625140011310577,12), Pointf3(3.45491504669189,0.244717106223106,12), Pointf3(4.2804012298584,0.456633001565933,12), Pointf3(4.21782684326172,0.0615576282143593,12), Pointf3(5,0.399999290704727,12), Pointf3(5,0,12), Pointf3(19.6000003814697,0.399999290704727,12), Pointf3(20,0,12), Pointf3(19.6000003814697,20,12), Pointf3(20,20,12), Pointf3(19.5300102233887,21.1571502685547,12), Pointf3(19.9323806762695,21.1609306335449,12), Pointf3(19.3210391998291,22.2974300384521,12), Pointf3(19.7304496765137,22.3061599731445,12), Pointf3(18.9761600494385,23.4042091369629,12), Pointf3(19.3969306945801,23.4202003479004,12), Pointf3(18.5003795623779,24.4613399505615,12), Pointf3(18.9363307952881,24.4879894256592,12), Pointf3(17.9006500244141,25.4534206390381,12), Pointf3(18.3548793792725,25.4950904846191,12), Pointf3(17.1856994628906,26.3659801483154,12), Pointf3(17.6604404449463,26.4278793334961,12), Pointf3(16.3659801483154,27.1856994628906,12), Pointf3(16.862419128418,27.2737407684326,12), Pointf3(15.4534196853638,27.9006500244141,12), Pointf3(15.9715900421143,28.0212306976318,12), Pointf3(14.4613399505615,28.5003795623779,12), Pointf3(15,28.6602592468262,12), Pointf3(13.4042100906372,28.9761600494385,12), Pointf3(13.9608001708984,29.1821594238281,12), Pointf3(12.2974300384521,29.3210391998291,12), Pointf3(7.13196802139282,29.5799007415771,0), Pointf3(8.26351833343506,29.8480796813965,0), Pointf3(8.84284687042236,29.5300102233887,0), Pointf3(9.41855144500732,29.9830799102783,0), Pointf3(10,29.6000003814697,0), Pointf3(10.5814504623413,29.9830799102783,0), Pointf3(11.1571502685547,29.5300102233887,0), Pointf3(11.7364797592163,29.8480796813965,0), Pointf3(12.2974300384521,29.3210391998291,0), Pointf3(12.8680295944214,29.5799007415771,0), Pointf3(13.4042100906372,28.9761600494385,0), Pointf3(13.9608001708984,29.1821594238281,0), Pointf3(14.4613399505615,28.5003795623779,0), Pointf3(15,28.6602592468262,0), Pointf3(15.4534196853638,27.9006500244141,0), Pointf3(15.9715900421143,28.0212306976318,0), Pointf3(16.3659801483154,27.1856994628906,0), Pointf3(16.862419128418,27.2737407684326,0), Pointf3(17.1856994628906,26.3659801483154,0), Pointf3(17.6604404449463,26.4278793334961,0), Pointf3(17.9006500244141,25.4534206390381,0), Pointf3(18.3548793792725,25.4950904846191,0), Pointf3(18.5003795623779,24.4613399505615,0), Pointf3(18.9363307952881,24.4879894256592,0), Pointf3(18.9761600494385,23.4042091369629,0), Pointf3(19.3969306945801,23.4202003479004,0), Pointf3(19.3210391998291,22.2974300384521,0), Pointf3(19.7304496765137,22.3061599731445,0), Pointf3(19.5300102233887,21.1571502685547,0), Pointf3(19.9323806762695,21.1609306335449,0), Pointf3(19.6000003814697,20,0), Pointf3(20,20,0), Pointf3(19.6000003814697,0.399999290704727,0), Pointf3(20,0,0), Pointf3(5,0.399999290704727,0), Pointf3(5,0,0), Pointf3(4.2804012298584,0.456633001565933,0), Pointf3(4.21782684326172,0.0615576282143593,0), Pointf3(3.5785219669342,0.625140011310577,0), Pointf3(3.45491504669189,0.244717106223106,0), Pointf3(2.91164398193359,0.901369392871857,0), Pointf3(2.73004698753357,0.544967114925385,0), Pointf3(2.29618692398071,1.27852201461792,0), Pointf3(2.06107401847839,0.954914808273315,0), Pointf3(1.74730801582336,1.74730801582336,0), Pointf3(1.46446597576141,1.46446597576141,0), Pointf3(1.27852201461792,2.29618692398071,0), Pointf3(0.954914808273315,2.06107401847839,0), Pointf3(0.901369392871857,2.91164398193359,0), Pointf3(0.544967114925385,2.73004698753357,0), Pointf3(0.625140011310577,3.5785219669342,0), Pointf3(0.244717106223106,3.45491504669189,0), Pointf3(0.456633001565933,4.2804012298584,0), Pointf3(0.0615576282143593,4.21782684326172,0), Pointf3(0.399999290704727,5,0), Pointf3(0,5,0), Pointf3(0.399999290704727,20,0), Pointf3(0,20,0), Pointf3(0.469994693994522,21.1571502685547,0), Pointf3(0.067615881562233,21.1609306335449,0), Pointf3(0.678958415985107,22.2974300384521,0), Pointf3(0.269550800323486,22.3061599731445,0), Pointf3(1.02384400367737,23.4042091369629,0), Pointf3(0.603073298931122,23.4202003479004,0), Pointf3(1.49962198734283,24.4613399505615,0), Pointf3(1.0636739730835,24.4879894256592,0), Pointf3(2.0993549823761,25.4534206390381,0), Pointf3(1.64512205123901,25.4950904846191,0), Pointf3(2.81429696083069,26.3659801483154,0), Pointf3(2.33955597877502,26.4278793334961,0), Pointf3(3.63402199745178,27.1856994628906,0), Pointf3(3.13758301734924,27.2737407684326,0), Pointf3(4.54657793045044,27.9006500244141,0), Pointf3(4.02841377258301,28.0212306976318,0), Pointf3(5.53865718841553,28.5003795623779,0), Pointf3(5,28.6602592468262,0), Pointf3(6.59579277038574,28.9761600494385,0), Pointf3(6.03920221328735,29.1821594238281,0), Pointf3(7.70256900787354,29.3210391998291,0) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(2,1,3), Point3(2,3,4), Point3(4,3,5), Point3(4,5,6), Point3(6,5,7), Point3(6,7,8), Point3(8,7,9), Point3(8,9,10), Point3(10,9,11), Point3(10,11,12), Point3(12,11,13), Point3(12,13,14), Point3(14,13,15), Point3(14,15,16), Point3(16,15,17), Point3(16,17,18), Point3(18,17,19), Point3(18,19,20), Point3(20,19,21), Point3(20,21,22), Point3(22,21,23), Point3(22,23,24), Point3(24,23,25), Point3(24,25,26), Point3(26,25,27), Point3(26,27,28), Point3(28,27,29), Point3(28,29,30), Point3(30,29,31), Point3(30,31,32), Point3(32,31,33), Point3(32,33,34), Point3(34,33,35), Point3(34,35,36), Point3(36,35,37), Point3(36,37,38), Point3(38,37,39), Point3(38,39,40), Point3(40,39,41), Point3(40,41,42), Point3(42,41,43), Point3(42,43,44), Point3(44,43,45), Point3(44,45,46), Point3(46,45,47), Point3(46,47,48), Point3(48,47,49), Point3(48,49,50), Point3(50,49,51), Point3(50,51,52), Point3(52,51,53), Point3(52,53,54), Point3(54,53,55), Point3(54,55,56), Point3(56,55,57), Point3(56,57,58), Point3(58,57,59), Point3(58,59,60), Point3(60,59,61), Point3(60,61,62), Point3(62,61,63), Point3(62,63,64), Point3(64,63,65), Point3(64,65,66), Point3(66,65,67), Point3(66,67,68), Point3(68,67,69), Point3(68,69,70), Point3(70,69,71), Point3(70,71,72), Point3(72,71,73), Point3(72,73,74), Point3(74,73,75), Point3(74,75,76), Point3(76,75,77), Point3(76,77,78), Point3(78,77,0), Point3(78,0,2), Point3(79,80,81), Point3(81,80,82), Point3(81,82,83), Point3(83,82,84), Point3(83,84,85), Point3(85,84,86), Point3(85,86,87), Point3(87,86,88), Point3(87,88,89), Point3(89,88,90), Point3(89,90,91), Point3(91,90,92), Point3(91,92,93), Point3(93,92,94), Point3(93,94,95), Point3(95,94,96), Point3(95,96,97), Point3(97,96,98), Point3(97,98,99), Point3(99,98,100), Point3(99,100,101), Point3(101,100,102), Point3(101,102,103), Point3(103,102,104), Point3(103,104,105), Point3(105,104,106), Point3(105,106,107), Point3(107,106,108), Point3(107,108,109), Point3(109,108,110), Point3(109,110,111), Point3(111,110,112), Point3(111,112,113), Point3(113,112,114), Point3(113,114,115), Point3(115,114,116), Point3(115,116,117), Point3(117,116,118), Point3(117,118,119), Point3(119,118,120), Point3(119,120,121), Point3(121,120,122), Point3(121,122,123), Point3(123,122,124), Point3(123,124,125), Point3(125,124,126), Point3(125,126,127), Point3(127,126,128), Point3(127,128,129), Point3(129,128,130), Point3(129,130,131), Point3(131,130,132), Point3(131,132,133), Point3(133,132,134), Point3(133,134,135), Point3(135,134,136), Point3(135,136,137), Point3(137,136,138), Point3(137,138,139), Point3(139,138,140), Point3(139,140,141), Point3(141,140,142), Point3(141,142,143), Point3(143,142,144), Point3(143,144,145), Point3(145,144,146), Point3(145,146,147), Point3(147,146,148), Point3(147,148,149), Point3(149,148,150), Point3(149,150,151), Point3(151,150,152), Point3(151,152,153), Point3(153,152,154), Point3(153,154,155), Point3(155,154,156), Point3(155,156,157), Point3(157,156,79), Point3(157,79,81), Point3(57,110,108), Point3(57,108,59), Point3(59,108,106), Point3(59,106,61), Point3(61,106,104), Point3(61,104,63), Point3(63,104,102), Point3(63,102,65), Point3(65,102,100), Point3(65,100,67), Point3(67,100,98), Point3(67,98,69), Point3(69,98,96), Point3(69,96,71), Point3(71,96,94), Point3(71,94,73), Point3(73,94,92), Point3(73,92,75), Point3(75,92,90), Point3(75,90,77), Point3(77,90,88), Point3(77,88,0), Point3(0,88,86), Point3(0,86,1), Point3(1,86,84), Point3(1,84,3), Point3(3,84,82), Point3(3,82,5), Point3(5,82,80), Point3(5,80,7), Point3(7,80,79), Point3(7,79,9), Point3(9,79,156), Point3(9,156,11), Point3(11,156,154), Point3(11,154,13), Point3(13,154,152), Point3(13,152,15), Point3(15,152,150), Point3(15,150,17), Point3(17,150,148), Point3(17,148,19), Point3(19,148,146), Point3(19,146,21), Point3(21,146,144), Point3(21,144,23), Point3(23,144,142), Point3(23,142,25), Point3(25,142,140), Point3(25,140,27), Point3(27,140,138), Point3(27,138,29), Point3(29,138,136), Point3(29,136,31), Point3(33,31,134), Point3(134,31,136), Point3(33,134,132), Point3(33,132,35), Point3(35,132,130), Point3(35,130,37), Point3(37,130,128), Point3(37,128,39), Point3(39,128,126), Point3(39,126,41), Point3(41,126,124), Point3(41,124,43), Point3(43,124,122), Point3(43,122,45), Point3(45,122,120), Point3(45,120,47), Point3(47,120,118), Point3(47,118,49), Point3(49,118,116), Point3(49,116,51), Point3(51,116,114), Point3(51,114,53), Point3(55,53,112), Point3(112,53,114), Point3(57,55,110), Point3(110,55,112), Point3(30,135,137), Point3(30,137,28), Point3(28,137,139), Point3(28,139,26), Point3(26,139,141), Point3(26,141,24), Point3(24,141,143), Point3(24,143,22), Point3(22,143,145), Point3(22,145,20), Point3(20,145,147), Point3(20,147,18), Point3(18,147,149), Point3(18,149,16), Point3(16,149,151), Point3(16,151,14), Point3(14,151,153), Point3(14,153,12), Point3(12,153,155), Point3(12,155,10), Point3(10,155,157), Point3(10,157,8), Point3(8,157,81), Point3(8,81,6), Point3(6,81,83), Point3(6,83,4), Point3(4,83,85), Point3(4,85,2), Point3(2,85,87), Point3(2,87,78), Point3(78,87,89), Point3(78,89,76), Point3(76,89,91), Point3(76,91,74), Point3(74,91,93), Point3(74,93,72), Point3(72,93,95), Point3(72,95,70), Point3(70,95,97), Point3(70,97,68), Point3(68,97,99), Point3(68,99,66), Point3(66,99,101), Point3(66,101,64), Point3(64,101,103), Point3(64,103,62), Point3(62,103,105), Point3(62,105,60), Point3(60,105,107), Point3(60,107,58), Point3(58,107,109), Point3(58,109,56), Point3(30,32,135), Point3(135,32,133), Point3(52,113,115), Point3(52,115,50), Point3(50,115,117), Point3(50,117,48), Point3(48,117,119), Point3(48,119,46), Point3(46,119,121), Point3(46,121,44), Point3(44,121,123), Point3(44,123,42), Point3(42,123,125), Point3(42,125,40), Point3(40,125,127), Point3(40,127,38), Point3(38,127,129), Point3(38,129,36), Point3(36,129,131), Point3(36,131,34), Point3(34,131,133), Point3(34,133,32), Point3(52,54,113), Point3(113,54,111), Point3(54,56,111), Point3(111,56,109) 
            });
            break;
        case TestMesh::sloping_hole:
            vertices = std::vector<Pointf3>({
                Pointf3(-20,-20,-5), Pointf3(-20,-20,5), Pointf3(-20,20,-5), Pointf3(-20,20,5), Pointf3(20,-20,-5), Pointf3(20,-20,5), Pointf3(4.46294021606445,7.43144989013672,-5), Pointf3(20,20,-5), Pointf3(-19.1420993804932,0,-5), Pointf3(-18.8330993652344,-2.07911992073059,-5), Pointf3(-17.9195003509521,-4.06736993789673,-5), Pointf3(-16.4412002563477,-5.87785005569458,-5), Pointf3(-14.4629001617432,-7.43144989013672,-5), Pointf3(-12.0711002349854,-8.66024971008301,-5), Pointf3(-9.37016010284424,-9.51056003570557,-5), Pointf3(-3.5217399597168,-9.94521999359131,-5), Pointf3(-6.4782600402832,-9.94521999359131,-5), Pointf3(-0.629840016365051,-9.51056003570557,-5), Pointf3(2.07106995582581,-8.66024971008301,-5), Pointf3(6.44122982025146,-5.87785005569458,-5), Pointf3(4.46294021606445,-7.43144989013672,-5), Pointf3(-12.0711002349854,8.66024971008301,-5), Pointf3(-9.37016010284424,9.51056003570557,-5), Pointf3(7.91947984695435,-4.06736993789673,-5), Pointf3(8.83310031890869,-2.07911992073059,-5), Pointf3(-6.4782600402832,9.94521999359131,-5), Pointf3(-0.629840016365051,9.51056003570557,-5), Pointf3(2.07106995582581,8.66024971008301,-5), Pointf3(9.14214038848877,0,-5), Pointf3(8.83310031890869,2.07911992073059,-5), Pointf3(-3.5217399597168,9.94521999359131,-5), Pointf3(7.91947984695435,4.06736993789673,-5), Pointf3(6.44122982025146,5.87785005569458,-5), Pointf3(-14.4629001617432,7.43144989013672,-5), Pointf3(-16.4412002563477,5.87785005569458,-5), Pointf3(-17.9195003509521,4.06736993789673,-5), Pointf3(-18.8330993652344,2.07911992073059,-5), Pointf3(20,20,5), Pointf3(3.5217399597168,-9.94521999359131,5), Pointf3(-8.83310031890869,-2.07911992073059,5), Pointf3(-9.14214038848877,0,5), Pointf3(-8.83310031890869,2.07911992073059,5), Pointf3(6.4782600402832,-9.94521999359131,5), Pointf3(-7.91947984695435,4.06736993789673,5), Pointf3(-6.44122982025146,5.87785005569458,5), Pointf3(-4.46294021606445,7.43144989013672,5), Pointf3(-2.07106995582581,8.66024971008301,5), Pointf3(0.629840016365051,9.51056003570557,5), Pointf3(12.0711002349854,-8.66024971008301,5), Pointf3(9.37016010284424,-9.51056003570557,5), Pointf3(3.5217399597168,9.94521999359131,5), Pointf3(6.4782600402832,9.94521999359131,5), Pointf3(9.37016010284424,9.51056003570557,5), Pointf3(12.0711002349854,8.66024971008301,5), Pointf3(14.4629001617432,7.43144989013672,5), Pointf3(16.4412002563477,-5.87785005569458,5), Pointf3(14.4629001617432,-7.43144989013672,5), Pointf3(16.4412002563477,5.87785005569458,5), Pointf3(17.9195003509521,4.06736993789673,5), Pointf3(18.8330993652344,-2.07911992073059,5), Pointf3(17.9195003509521,-4.06736993789673,5), Pointf3(18.8330993652344,2.07911992073059,5), Pointf3(19.1420993804932,0,5), Pointf3(0.629840016365051,-9.51056003570557,5), Pointf3(-2.07106995582581,-8.66024971008301,5), Pointf3(-4.46294021606445,-7.43144989013672,5), Pointf3(-6.44122982025146,-5.87785005569458,5), Pointf3(-7.91947984695435,-4.06736993789673,5) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(2,1,3), Point3(1,0,4), Point3(5,1,4), Point3(6,2,7), Point3(0,2,8), Point3(0,8,9), Point3(0,9,10), Point3(0,10,11), Point3(0,11,12), Point3(0,12,13), Point3(0,13,4), Point3(13,14,4), Point3(15,4,16), Point3(17,4,15), Point3(18,4,17), Point3(19,4,20), Point3(18,20,4), Point3(21,2,22), Point3(4,19,23), Point3(4,23,7), Point3(23,24,7), Point3(22,2,25), Point3(26,2,27), Point3(28,29,7), Point3(25,2,30), Point3(29,31,7), Point3(30,2,26), Point3(31,32,7), Point3(27,2,6), Point3(32,6,7), Point3(28,7,24), Point3(33,2,21), Point3(34,2,33), Point3(35,2,34), Point3(36,2,35), Point3(8,2,36), Point3(16,4,14), Point3(2,3,7), Point3(7,3,37), Point3(38,1,5), Point3(3,1,39), Point3(3,39,40), Point3(3,40,41), Point3(42,38,5), Point3(3,41,43), Point3(3,43,44), Point3(37,3,45), Point3(37,45,46), Point3(37,46,47), Point3(48,49,5), Point3(37,47,50), Point3(49,42,5), Point3(37,50,51), Point3(37,51,52), Point3(37,52,53), Point3(37,53,54), Point3(55,56,5), Point3(37,54,57), Point3(37,57,58), Point3(59,60,5), Point3(37,58,61), Point3(37,62,5), Point3(37,61,62), Point3(62,59,5), Point3(60,55,5), Point3(63,1,38), Point3(64,1,63), Point3(65,1,64), Point3(66,1,65), Point3(67,1,66), Point3(39,1,67), Point3(44,45,3), Point3(56,48,5), Point3(5,4,7), Point3(37,5,7), Point3(41,40,36), Point3(36,40,8), Point3(39,9,40), Point3(40,9,8), Point3(43,41,35), Point3(35,41,36), Point3(44,43,34), Point3(34,43,35), Point3(33,45,44), Point3(34,33,44), Point3(21,46,45), Point3(33,21,45), Point3(22,47,46), Point3(21,22,46), Point3(25,50,47), Point3(22,25,47), Point3(30,51,50), Point3(25,30,50), Point3(26,52,51), Point3(30,26,51), Point3(27,53,52), Point3(26,27,52), Point3(6,54,53), Point3(27,6,53), Point3(32,57,54), Point3(6,32,54), Point3(31,58,57), Point3(32,31,57), Point3(29,61,58), Point3(31,29,58), Point3(28,62,61), Point3(29,28,61), Point3(59,62,28), Point3(24,59,28), Point3(60,59,24), Point3(23,60,24), Point3(55,60,23), Point3(19,55,23), Point3(55,19,56), Point3(56,19,20), Point3(56,20,48), Point3(48,20,18), Point3(48,18,49), Point3(49,18,17), Point3(49,17,42), Point3(42,17,15), Point3(42,15,38), Point3(38,15,16), Point3(38,16,63), Point3(63,16,14), Point3(63,14,64), Point3(64,14,13), Point3(64,13,65), Point3(65,13,12), Point3(65,12,66), Point3(66,12,11), Point3(66,11,67), Point3(67,11,10), Point3(67,10,39), Point3(39,10,9) 
            });
            break;
        case TestMesh::ipadstand:
            vertices = std::vector<Pointf3>({
                Pointf3(17.4344673156738,-2.69879599481136e-16,9.5), Pointf3(14.2814798355103,10,9.5), Pointf3(0,0,9.5), Pointf3(31.7159481048584,10,9.5), Pointf3(62.2344741821289,2.06667568800577e-16,20), Pointf3(31.7159481048584,10,20), Pointf3(17.4344673156738,-2.69879599481136e-16,20), Pointf3(62.2344741821289,10,20), Pointf3(98.2079696655273,10,0), Pointf3(98.2079696655273,8.56525380796383e-16,10), Pointf3(98.2079696655273,0,0), Pointf3(98.2079696655273,10,20), Pointf3(98.2079696655273,0,20), Pointf3(81.6609649658203,-4.39753856997999e-16,10), Pointf3(90.0549850463867,10,10), Pointf3(78.5079803466797,10,10), Pointf3(93.2079696655273,8.56525380796383e-16,10), Pointf3(14.2814798355103,10,20), Pointf3(0,0,20), Pointf3(87.4344711303711,2.81343962782118e-15,20), Pointf3(84.2814788818359,10,20), Pointf3(0,10,20), Pointf3(0,0,0), Pointf3(0,10,0), Pointf3(62.2344741821289,2.06667568800577e-16,30), Pointf3(66.9609756469727,10,30), Pointf3(62.2344741821289,10,30), Pointf3(70.1139602661133,8.5525763717214e-16,30), Pointf3(67.7053375244141,10,28.7107200622559), Pointf3(71.6787109375,1.24046736339707e-15,27.2897701263428) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(1,0,3), Point3(4,5,6), Point3(5,4,7), Point3(8,9,10), Point3(9,11,12), Point3(11,9,8), Point3(13,14,15), Point3(14,13,16), Point3(17,2,1), Point3(2,17,18), Point3(19,11,20), Point3(11,19,12), Point3(17,21,18), Point3(21,2,18), Point3(2,21,22), Point3(22,21,23), Point3(8,22,23), Point3(22,8,10), Point3(24,25,26), Point3(25,24,27), Point3(23,1,8), Point3(1,23,21), Point3(1,21,17), Point3(5,15,3), Point3(15,5,7), Point3(15,7,28), Point3(28,7,26), Point3(28,26,25), Point3(8,14,11), Point3(14,8,3), Point3(3,8,1), Point3(14,3,15), Point3(11,14,20), Point3(26,4,24), Point3(4,26,7), Point3(12,16,9), Point3(16,12,19), Point3(29,4,13), Point3(4,29,24), Point3(24,29,27), Point3(9,22,10), Point3(22,9,0), Point3(0,9,16), Point3(0,16,13), Point3(0,13,6), Point3(6,13,4), Point3(2,22,0), Point3(19,14,16), Point3(14,19,20), Point3(15,29,13), Point3(29,25,27), Point3(25,29,15), Point3(25,15,28), Point3(6,3,0), Point3(3,6,5) 
            });
            break;
        case TestMesh::A:
            vertices = std::vector<Pointf3>({
                Pointf3(513.075988769531,51.6074333190918,36.0009002685547), Pointf3(516.648803710938,51.7324333190918,36.0009002685547), Pointf3(513.495178222656,51.7324333190918,36.0009002685547), Pointf3(489.391204833984,51.4824333190918,24.0011005401611), Pointf3(488.928588867188,51.7324333190918,24.0011005401611), Pointf3(492.06201171875,51.7324333190918,24.0011005401611), Pointf3(496.840393066406,51.2324333190918,24.0011005401611), Pointf3(495.195404052734,51.7324333190918,24.0011005401611), Pointf3(498.981994628906,51.7324333190918,24.0011005401611), Pointf3(506.966613769531,51.6074333190918,24.0011005401611), Pointf3(510.342010498047,51.7324333190918,24.0011005401611), Pointf3(507.163818359375,51.6074333190918,24.0011005401611), Pointf3(512.515380859375,54.7190322875977,36.0009002685547), Pointf3(514.161987304688,54.5058326721191,36.0009002685547), Pointf3(493.06201171875,54.7190322875977,36.0009002685547), Pointf3(495.195404052734,51.7324333190918,36.0009002685547), Pointf3(496.195404052734,54.7190322875977,36.0009002685547), Pointf3(497.195404052734,57.7058334350586,36.0009002685547), Pointf3(500.851989746094,60.2658309936523,36.0009002685547), Pointf3(498.915405273438,62.8258323669434,36.0009002685547), Pointf3(506.701995849609,62.8258323669434,36.0009002685547), Pointf3(503.648590087891,60.2658309936523,36.0009002685547), Pointf3(508.381805419922,57.7058334350586,36.0009002685547), Pointf3(496.418792724609,60.052433013916,36.0009002685547), Pointf3(506.515197753906,72.2124328613281,36.0009002685547), Pointf3(502.808807373047,74.5324325561523,36.0009002685547), Pointf3(503.781982421875,71.6058349609375,36.0009002685547), Pointf3(515.358764648438,55.4658317565918,36.0009002685547), Pointf3(499.375183105469,76.9058380126953,36.0009002685547), Pointf3(501.168792724609,78.0658340454102,36.0009002685547), Pointf3(504.568786621094,78.0658340454102,36.0009002685547), Pointf3(506.32861328125,81.599235534668,36.0009002685547), Pointf3(502.928588867188,81.599235534668,36.0009002685547), Pointf3(499.528594970703,81.599235534668,36.0009002685547), Pointf3(498.20361328125,77.8658294677734,36.0009002685547), Pointf3(495.195404052734,51.7324333190918,30.0011005401611), Pointf3(498.981994628906,51.7324333190918,27.0011005401611), Pointf3(506.555206298828,51.7324333190918,33.0009002685547), Pointf3(506.555206298828,51.7324333190918,36.0009002685547), Pointf3(510.342010498047,51.7324333190918,36.0009002685547), Pointf3(512.515380859375,54.7190322875977,24.0011005401611), Pointf3(509.361999511719,54.7190322875977,24.0011005401611), Pointf3(508.381805419922,57.7058334350586,24.0011005401611), Pointf3(506.701995849609,62.8258323669434,24.0011005401611), Pointf3(509.188812255859,60.052433013916,24.0011005401611), Pointf3(493.06201171875,54.7190322875977,24.0011005401611), Pointf3(503.648590087891,60.2658309936523,24.0011005401611), Pointf3(500.851989746094,60.2658309936523,24.0011005401611), Pointf3(498.915405273438,62.8258323669434,24.0011005401611), Pointf3(502.808807373047,62.8258323669434,24.0011005401611), Pointf3(491.425201416016,54.5058326721191,24.0011005401611), Pointf3(506.421813964844,76.9058380126953,24.0011005401611), Pointf3(502.808807373047,74.5324325561523,24.0011005401611), Pointf3(504.568786621094,78.0658340454102,24.0011005401611), Pointf3(506.32861328125,81.599235534668,24.0011005401611), Pointf3(507.618804931641,77.8658294677734,24.0011005401611), Pointf3(499.221801757812,72.2124328613281,24.0011005401611), Pointf3(501.835388183594,71.6058349609375,24.0011005401611), Pointf3(501.168792724609,78.0658340454102,24.0011005401611), Pointf3(499.528594970703,81.599235534668,24.0011005401611), Pointf3(502.048583984375,79.8324356079102,24.0011005401611), Pointf3(490.253601074219,55.4658317565918,24.0011005401611), Pointf3(488.928588867188,51.7324333190918,30.0011005401611), Pointf3(488.928588867188,51.7324333190918,36.0009002685547), Pointf3(490.253601074219,55.4658317565918,31.5009002685547), Pointf3(498.20361328125,77.8658294677734,34.5009002685547), Pointf3(508.381805419922,57.7058334350586,30.0011005401611), Pointf3(505.585388183594,57.7058334350586,27.0011005401611), Pointf3(502.788818359375,57.7058334350586,36.0009002685547), Pointf3(499.992004394531,57.7058334350586,33.0009002685547), Pointf3(509.851989746094,53.2258338928223,33.0009002685547), Pointf3(509.361999511719,54.7190322875977,36.0009002685547), Pointf3(508.871795654297,56.2124328613281,27.0011005401611), Pointf3(496.695404052734,56.2124328613281,33.0009002685547), Pointf3(495.695404052734,53.2258338928223,27.0011005401611), Pointf3(506.32861328125,81.599235534668,30.0011005401611), Pointf3(507.618804931641,77.8658294677734,25.5011005401611), Pointf3(515.358764648438,55.4658317565918,34.5009002685547), Pointf3(501.228607177734,81.599235534668,33.0009002685547), Pointf3(504.628601074219,81.599235534668,27.0011005401611), Pointf3(503.781982421875,71.6058349609375,33.0009002685547), Pointf3(502.808807373047,74.5324325561523,30.0011005401611), Pointf3(498.915405273438,62.8258323669434,30.0011005401611), Pointf3(500.861999511719,62.8258323669434,27.0011005401611), Pointf3(502.808807373047,62.8258323669434,36.0009002685547), Pointf3(504.755187988281,62.8258323669434,33.0009002685547), Pointf3(501.835388183594,71.6058349609375,33.0009002685547), Pointf3(499.888793945312,65.7524337768555,33.0009002685547), Pointf3(499.888793945312,65.7524337768555,36.0009002685547), Pointf3(513.128601074219,51.4824333190918,36.0009002685547), Pointf3(513.075988769531,51.6074333190918,24.0011005401611), Pointf3(516.648803710938,51.7324333190918,24.0011005401611), Pointf3(513.128601074219,51.4824333190918,24.0011005401611), Pointf3(513.495178222656,51.7324333190918,24.0011005401611), Pointf3(506.966613769531,51.6074333190918,36.0009002685547), Pointf3(507.163818359375,51.6074333190918,36.0009002685547), Pointf3(490.337799072266,51.4824333190918,24.0011005401611), Pointf3(489.391204833984,51.4824333190918,36.0009002685547), Pointf3(492.06201171875,51.7324333190918,36.0009002685547), Pointf3(490.337799072266,51.4824333190918,36.0009002685547), Pointf3(513.233764648438,51.2324333190918,24.0011005401611), Pointf3(513.233764648438,51.2324333190918,36.0009002685547), Pointf3(504.773803710938,51.4824333190918,36.0009002685547), Pointf3(504.773803710938,51.4824333190918,24.0011005401611), Pointf3(489.266998291016,51.2324333190918,24.0011005401611), Pointf3(489.266998291016,51.2324333190918,36.0009002685547), Pointf3(490.253601074219,55.4658317565918,25.5011005401611), Pointf3(499.528594970703,81.599235534668,30.0011005401611), Pointf3(498.20361328125,77.8658294677734,31.5009002685547), Pointf3(515.358764648438,55.4658317565918,28.5011005401611), Pointf3(515.358764648438,55.4658317565918,25.5011005401611), Pointf3(495.246795654297,61.0124320983887,36.0009002685547), Pointf3(490.253601074219,55.4658317565918,34.5009002685547), Pointf3(490.253601074219,55.4658317565918,36.0009002685547), Pointf3(494.228607177734,66.6658325195312,24.0011005401611), Pointf3(499.068786621094,67.5192337036133,24.0011005401611), Pointf3(498.20361328125,77.8658294677734,25.5011005401611), Pointf3(498.20361328125,77.8658294677734,24.0011005401611), Pointf3(506.608795166016,67.5192337036133,36.0009002685547), Pointf3(509.09521484375,64.7458343505859,36.0009002685547), Pointf3(507.618804931641,77.8658294677734,34.5009002685547), Pointf3(507.618804931641,77.8658294677734,36.0009002685547), Pointf3(510.385406494141,61.0124320983887,24.0011005401611), Pointf3(515.358764648438,55.4658317565918,24.0011005401611), Pointf3(489.32861328125,47.7324333190918,31.5009002685547), Pointf3(492.95361328125,47.7324333190918,33.5634994506836), Pointf3(489.32861328125,47.7324333190918,34.5009002685547), Pointf3(489.32861328125,47.7324333190918,28.5011005401611), Pointf3(489.32861328125,47.7324333190918,25.5011005401611), Pointf3(492.95361328125,47.7324333190918,26.4385013580322), Pointf3(492.95361328125,47.7324333190918,30.5635013580322), Pointf3(492.95361328125,47.7324333190918,32.0634994506836), Pointf3(492.95361328125,47.7324333190918,31.3135013580322), Pointf3(492.95361328125,47.7324333190918,35.4384994506836), Pointf3(489.32861328125,47.7324333190918,36.0009002685547), Pointf3(492.95361328125,47.7324333190918,34.3134994506836), Pointf3(492.95361328125,47.7324333190918,34.6884994506836), Pointf3(492.95361328125,47.7324333190918,27.9385013580322), Pointf3(492.95361328125,47.7324333190918,28.6885013580322), Pointf3(492.95361328125,47.7324333190918,29.0635013580322), Pointf3(489.32861328125,47.7324333190918,24.0011005401611), Pointf3(492.95361328125,47.7324333190918,24.5635013580322), Pointf3(492.95361328125,47.7324333190918,25.6885013580322), Pointf3(492.95361328125,47.7324333190918,25.3135013580322), Pointf3(492.95361328125,47.7324333190918,24.1885013580322), Pointf3(492.95361328125,47.7324333190918,24.0011005401611), Pointf3(513.443786621094,50.7324333190918,24.0011005401611), Pointf3(492.95361328125,47.7324333190918,35.8134994506836), Pointf3(492.95361328125,47.7324333190918,36.0009002685547), Pointf3(513.443786621094,50.7324333190918,36.0009002685547), Pointf3(506.350402832031,51.4824333190918,36.0009002685547), Pointf3(506.350402832031,51.4824333190918,24.0011005401611), Pointf3(492.743804931641,48.2324333190918,24.0011005401611), Pointf3(492.638793945312,48.4824333190918,24.0011005401611), Pointf3(492.743804931641,48.2324333190918,36.0009002685547), Pointf3(492.638793945312,48.4824333190918,36.0009002685547), Pointf3(490.089599609375,50.9824333190918,36.0009002685547), Pointf3(490.089599609375,50.9824333190918,24.0011005401611), Pointf3(510.342010498047,51.7324333190918,30.0011005401611), Pointf3(499.068786621094,67.5192337036133,36.0009002685547), Pointf3(494.228607177734,66.6658325195312,36.0009002685547), Pointf3(499.375183105469,76.9058380126953,24.0011005401611), Pointf3(506.421813964844,76.9058380126953,36.0009002685547), Pointf3(506.608795166016,67.5192337036133,24.0011005401611), Pointf3(505.728607177734,65.7524337768555,24.0011005401611), Pointf3(509.09521484375,64.7458343505859,24.0011005401611), Pointf3(506.701995849609,62.8258323669434,30.0011005401611), Pointf3(505.728607177734,65.7524337768555,27.0011005401611), Pointf3(501.835388183594,71.6058349609375,27.0011005401611), Pointf3(499.888793945312,65.7524337768555,27.0011005401611), Pointf3(494.228607177734,66.6658325195312,30.0011005401611), Pointf3(495.553588867188,70.3992309570312,28.5011005401611), Pointf3(492.903594970703,62.9324340820312,28.5011005401611), Pointf3(495.553588867188,70.3992309570312,31.5009002685547), Pointf3(492.903594970703,62.9324340820312,31.5009002685547), Pointf3(511.488800048828,66.6658325195312,24.0011005401611), Pointf3(511.488800048828,66.6658325195312,30.0011005401611), Pointf3(512.778564453125,62.9324340820312,28.5011005401611), Pointf3(515.358764648438,55.4658317565918,31.5009002685547), Pointf3(507.618804931641,77.8658294677734,31.5009002685547), Pointf3(510.198791503906,70.3992309570312,28.5011005401611), Pointf3(511.488800048828,66.6658325195312,36.0009002685547), Pointf3(512.778564453125,62.9324340820312,31.5009002685547), Pointf3(510.198791503906,70.3992309570312,31.5009002685547), Pointf3(502.788818359375,57.7058334350586,24.0011005401611), Pointf3(497.195404052734,57.7058334350586,30.0011005401611), Pointf3(492.903594970703,62.9324340820312,34.5009002685547), Pointf3(492.903594970703,62.9324340820312,36.0009002685547), Pointf3(495.553588867188,70.3992309570312,24.0011005401611), Pointf3(496.725189208984,69.4392318725586,24.0011005401611), Pointf3(495.553588867188,70.3992309570312,25.5011005401611), Pointf3(495.246795654297,61.0124320983887,24.0011005401611), Pointf3(492.903594970703,62.9324340820312,25.5011005401611), Pointf3(492.903594970703,62.9324340820312,24.0011005401611), Pointf3(495.553588867188,70.3992309570312,36.0009002685547), Pointf3(496.725189208984,69.4392318725586,36.0009002685547), Pointf3(495.553588867188,70.3992309570312,34.5009002685547), Pointf3(510.198791503906,70.3992309570312,36.0009002685547), Pointf3(509.002014160156,69.4392318725586,36.0009002685547), Pointf3(510.198791503906,70.3992309570312,34.5009002685547), Pointf3(512.778564453125,62.9324340820312,25.5011005401611), Pointf3(512.778564453125,62.9324340820312,24.0011005401611), Pointf3(510.198791503906,70.3992309570312,24.0011005401611), Pointf3(509.002014160156,69.4392318725586,24.0011005401611), Pointf3(510.198791503906,70.3992309570312,25.5011005401611), Pointf3(510.385406494141,61.0124320983887,36.0009002685547), Pointf3(512.778564453125,62.9324340820312,34.5009002685547), Pointf3(512.778564453125,62.9324340820312,36.0009002685547), Pointf3(496.840393066406,51.2324333190918,36.0009002685547), Pointf3(498.981994628906,51.7324333190918,36.0009002685547), Pointf3(498.981994628906,51.7324333190918,33.0009002685547), Pointf3(506.555206298828,51.7324333190918,24.0011005401611), Pointf3(506.555206298828,51.7324333190918,27.0011005401611), Pointf3(503.82861328125,47.7324333190918,30.7509002685547), Pointf3(507.45361328125,47.7324333190918,32.8134994506836), Pointf3(503.82861328125,47.7324333190918,33.7509002685547), Pointf3(503.82861328125,47.7324333190918,29.2511005401611), Pointf3(503.82861328125,47.7324333190918,26.2511005401611), Pointf3(507.45361328125,47.7324333190918,27.1885013580322), Pointf3(493.921813964844,57.2792320251465,36.0009002685547), Pointf3(491.425201416016,54.5058326721191,36.0009002685547), Pointf3(497.195404052734,57.7058334350586,24.0011005401611), Pointf3(496.418792724609,60.052433013916,24.0011005401611), Pointf3(509.188812255859,60.052433013916,36.0009002685547), Pointf3(511.675415039062,57.2792320251465,24.0011005401611), Pointf3(514.161987304688,54.5058326721191,24.0011005401611), Pointf3(507.45361328125,47.7324333190918,34.3134994506836), Pointf3(503.82861328125,47.7324333190918,35.2509002685547), Pointf3(507.45361328125,47.7324333190918,25.6885013580322), Pointf3(503.82861328125,47.7324333190918,24.7511005401611), Pointf3(500.20361328125,47.7324333190918,31.6885013580322), Pointf3(500.20361328125,47.7324333190918,28.3135013580322), Pointf3(500.20361328125,47.7324333190918,30.1885013580322), Pointf3(507.45361328125,47.7324333190918,29.8135013580322), Pointf3(507.45361328125,47.7324333190918,31.3135013580322), Pointf3(507.45361328125,47.7324333190918,30.5635013580322), Pointf3(503.82861328125,47.7324333190918,36.0009002685547), Pointf3(507.45361328125,47.7324333190918,35.4384994506836), Pointf3(507.45361328125,47.7324333190918,35.0634994506836), Pointf3(507.45361328125,47.7324333190918,28.6885013580322), Pointf3(507.45361328125,47.7324333190918,29.4385013580322), Pointf3(503.82861328125,47.7324333190918,24.0011005401611), Pointf3(507.45361328125,47.7324333190918,24.5635013580322), Pointf3(507.45361328125,47.7324333190918,24.9385013580322), Pointf3(500.20361328125,47.7324333190918,34.6884994506836), Pointf3(500.20361328125,47.7324333190918,33.1884994506836), Pointf3(500.20361328125,47.7324333190918,33.9384994506836), Pointf3(500.20361328125,47.7324333190918,25.3135013580322), Pointf3(500.20361328125,47.7324333190918,26.8135013580322), Pointf3(500.20361328125,47.7324333190918,26.0635013580322), Pointf3(500.20361328125,47.7324333190918,30.9385013580322), Pointf3(500.20361328125,47.7324333190918,35.0634994506836), Pointf3(500.20361328125,47.7324333190918,35.4384994506836), Pointf3(500.20361328125,47.7324333190918,29.0635013580322), Pointf3(500.20361328125,47.7324333190918,29.4385013580322), Pointf3(500.20361328125,47.7324333190918,24.9385013580322), Pointf3(500.20361328125,47.7324333190918,24.5635013580322), Pointf3(507.45361328125,47.7324333190918,24.1885013580322), Pointf3(507.45361328125,47.7324333190918,24.0011005401611), Pointf3(513.86376953125,49.7324333190918,24.0011005401611), Pointf3(507.45361328125,47.7324333190918,35.8134994506836), Pointf3(507.45361328125,47.7324333190918,36.0009002685547), Pointf3(513.86376953125,49.7324333190918,36.0009002685547), Pointf3(500.20361328125,47.7324333190918,24.1885013580322), Pointf3(500.20361328125,47.7324333190918,24.0011005401611), Pointf3(502.988800048828,49.7324333190918,24.0011005401611), Pointf3(500.20361328125,47.7324333190918,35.8134994506836), Pointf3(500.20361328125,47.7324333190918,36.0009002685547), Pointf3(502.988800048828,49.7324333190918,36.0009002685547), Pointf3(504.755187988281,62.8258323669434,27.0011005401611), Pointf3(499.205383300781,51.2324333190918,36.0009002685547), Pointf3(498.786193847656,51.1074333190918,36.0009002685547), Pointf3(502.358795166016,51.2324333190918,36.0009002685547), Pointf3(499.205383300781,51.2324333190918,24.0011005401611), Pointf3(502.358795166016,51.2324333190918,24.0011005401611), Pointf3(498.786193847656,51.1074333190918,24.0011005401611), Pointf3(502.568786621094,50.7324333190918,24.0011005401611), Pointf3(505.931213378906,51.3574333190918,24.0011005401611), Pointf3(509.503601074219,51.4824333190918,24.0011005401611), Pointf3(502.568786621094,50.7324333190918,36.0009002685547), Pointf3(505.931213378906,51.3574333190918,36.0009002685547), Pointf3(509.503601074219,51.4824333190918,36.0009002685547), Pointf3(499.048583984375,50.4824333190918,36.0009002685547), Pointf3(492.428588867188,48.9824333190918,36.0009002685547), Pointf3(499.048583984375,50.4824333190918,24.0011005401611), Pointf3(492.428588867188,48.9824333190918,24.0011005401611), Pointf3(506.088806152344,50.9824333190918,24.0011005401611), Pointf3(506.036010742188,51.1074333190918,24.0011005401611), Pointf3(506.088806152344,50.9824333190918,36.0009002685547), Pointf3(506.036010742188,51.1074333190918,36.0009002685547), Pointf3(498.891204833984,50.8574333190918,36.0009002685547), Pointf3(498.943786621094,50.7324333190918,36.0009002685547), Pointf3(498.891204833984,50.8574333190918,24.0011005401611), Pointf3(498.943786621094,50.7324333190918,24.0011005401611), Pointf3(499.573608398438,49.2324333190918,24.0011005401611), Pointf3(499.783813476562,48.7324333190918,24.0011005401611), Pointf3(499.573608398438,49.2324333190918,36.0009002685547), Pointf3(499.783813476562,48.7324333190918,36.0009002685547), Pointf3(506.403594970703,50.2324333190918,24.0011005401611), Pointf3(506.298797607422,50.4824333190918,24.0011005401611), Pointf3(506.403594970703,50.2324333190918,36.0009002685547), Pointf3(506.298797607422,50.4824333190918,36.0009002685547), Pointf3(501.228607177734,81.599235534668,27.0011005401611), Pointf3(502.928588867188,81.599235534668,24.0011005401611), Pointf3(499.2587890625,49.9824333190918,36.0009002685547), Pointf3(499.363800048828,49.7324333190918,36.0009002685547), Pointf3(499.2587890625,49.9824333190918,24.0011005401611), Pointf3(499.363800048828,49.7324333190918,24.0011005401611), Pointf3(496.695404052734,56.2124328613281,27.0011005401611), Pointf3(496.195404052734,54.7190322875977,24.0011005401611), Pointf3(509.851989746094,53.2258338928223,27.0011005401611), Pointf3(493.464782714844,51.1074333190918,36.0009002685547), Pointf3(493.464782714844,51.1074333190918,24.0011005401611), Pointf3(502.768798828125,51.7324333190918,24.0011005401611), Pointf3(500.215789794922,51.3574333190918,24.0011005401611), Pointf3(497.628601074219,51.2324333190918,24.0011005401611), Pointf3(502.768798828125,51.7324333190918,36.0009002685547), Pointf3(500.215789794922,51.3574333190918,36.0009002685547), Pointf3(497.628601074219,51.2324333190918,36.0009002685547), Pointf3(507.033813476562,48.7324333190918,24.0011005401611), Pointf3(506.823791503906,49.2324333190918,24.0011005401611), Pointf3(507.033813476562,48.7324333190918,36.0009002685547), Pointf3(506.823791503906,49.2324333190918,36.0009002685547), Pointf3(494.4501953125,51.1074333190918,24.0011005401611), Pointf3(494.4501953125,51.1074333190918,36.0009002685547), Pointf3(500.807006835938,51.3574333190918,36.0009002685547), Pointf3(503.591186523438,51.4824333190918,36.0009002685547), Pointf3(503.591186523438,51.4824333190918,24.0011005401611), Pointf3(500.807006835938,51.3574333190918,24.0011005401611), Pointf3(505.728607177734,65.7524337768555,36.0009002685547), Pointf3(505.728607177734,65.7524337768555,33.0009002685547), Pointf3(499.221801757812,72.2124328613281,36.0009002685547), Pointf3(501.835388183594,71.6058349609375,36.0009002685547), Pointf3(506.515197753906,72.2124328613281,24.0011005401611), Pointf3(503.781982421875,71.6058349609375,24.0011005401611), Pointf3(503.781982421875,71.6058349609375,27.0011005401611), Pointf3(499.888793945312,65.7524337768555,24.0011005401611), Pointf3(495.695404052734,53.2258338928223,33.0009002685547), Pointf3(516.648803710938,51.7324333190918,30.0011005401611), Pointf3(498.20361328125,77.8658294677734,28.5011005401611), Pointf3(505.585388183594,57.7058334350586,33.0009002685547), Pointf3(508.871795654297,56.2124328613281,33.0009002685547), Pointf3(499.992004394531,57.7058334350586,27.0011005401611), Pointf3(504.628601074219,81.599235534668,33.0009002685547), Pointf3(500.861999511719,62.8258323669434,33.0009002685547), Pointf3(496.878601074219,74.1324310302734,27.0011005401611), Pointf3(496.878601074219,74.1324310302734,33.0009002685547), Pointf3(491.57861328125,59.199031829834,27.0011005401611), Pointf3(490.253601074219,55.4658317565918,28.5011005401611), Pointf3(491.57861328125,59.199031829834,33.0009002685547), Pointf3(514.068786621094,59.199031829834,27.0011005401611), Pointf3(514.068786621094,59.199031829834,33.0009002685547), Pointf3(508.908813476562,74.1324310302734,27.0011005401611), Pointf3(507.618804931641,77.8658294677734,28.5011005401611), Pointf3(508.908813476562,74.1324310302734,33.0009002685547), Pointf3(491.271789550781,50.9824333190918,36.0009002685547), Pointf3(490.877807617188,50.9824333190918,36.0009002685547), Pointf3(491.271789550781,50.9824333190918,24.0011005401611), Pointf3(490.877807617188,50.9824333190918,24.0011005401611), Pointf3(495.213806152344,50.9824333190918,36.0009002685547), Pointf3(493.636993408203,50.9824333190918,36.0009002685547), Pointf3(495.213806152344,50.9824333190918,24.0011005401611), Pointf3(493.636993408203,50.9824333190918,24.0011005401611), Pointf3(503.985412597656,51.4824333190918,36.0009002685547), Pointf3(503.985412597656,51.4824333190918,24.0011005401611), Pointf3(511.675415039062,57.2792320251465,36.0009002685547), Pointf3(493.921813964844,57.2792320251465,24.0011005401611), Pointf3(502.768798828125,51.7324333190918,30.0011005401611), Pointf3(506.555206298828,51.7324333190918,30.0011005401611), Pointf3(498.981994628906,51.7324333190918,30.0011005401611), Pointf3(492.848815917969,50.9824333190918,24.0011005401611), Pointf3(492.848815917969,50.9824333190918,36.0009002685547), Pointf3(500.861999511719,68.6792297363281,36.0009002685547), Pointf3(500.861999511719,68.6792297363281,24.0011005401611), Pointf3(496.878601074219,74.1324310302734,24.0011005401611), Pointf3(496.878601074219,74.1324310302734,36.0009002685547), Pointf3(504.755187988281,68.6792297363281,24.0011005401611), Pointf3(504.755187988281,68.6792297363281,36.0009002685547), Pointf3(508.908813476562,74.1324310302734,36.0009002685547), Pointf3(508.908813476562,74.1324310302734,24.0011005401611), Pointf3(505.728607177734,65.7524337768555,30.0011005401611), Pointf3(504.755187988281,68.6792297363281,30.0011005401611), Pointf3(503.781982421875,71.6058349609375,30.0011005401611), Pointf3(500.861999511719,68.6792297363281,30.0011005401611), Pointf3(499.888793945312,65.7524337768555,30.0011005401611), Pointf3(501.835388183594,71.6058349609375,30.0011005401611), Pointf3(491.57861328125,59.199031829834,24.0011005401611), Pointf3(491.57861328125,59.199031829834,36.0009002685547), Pointf3(514.068786621094,59.199031829834,36.0009002685547), Pointf3(514.068786621094,59.199031829834,24.0011005401611), Pointf3(511.07861328125,47.7324333190918,34.8759002685547), Pointf3(511.07861328125,47.7324333190918,31.8759002685547), Pointf3(514.70361328125,47.7324333190918,33.9384994506836), Pointf3(511.07861328125,47.7324333190918,25.1261005401611), Pointf3(514.70361328125,47.7324333190918,26.0635013580322), Pointf3(511.07861328125,47.7324333190918,28.1261005401611), Pointf3(502.788818359375,57.7058334350586,30.0011005401611), Pointf3(502.048583984375,79.8324356079102,36.0009002685547), Pointf3(514.70361328125,47.7324333190918,30.9385013580322), Pointf3(511.07861328125,47.7324333190918,30.3759002685547), Pointf3(514.70361328125,47.7324333190918,29.0635013580322), Pointf3(511.07861328125,47.7324333190918,29.6261005401611), Pointf3(496.57861328125,47.7324333190918,31.1259002685547), Pointf3(496.57861328125,47.7324333190918,32.6259002685547), Pointf3(496.57861328125,47.7324333190918,34.1259002685547), Pointf3(496.57861328125,47.7324333190918,28.8761005401611), Pointf3(496.57861328125,47.7324333190918,27.3761005401611), Pointf3(496.57861328125,47.7324333190918,25.8761005401611), Pointf3(496.57861328125,47.7324333190918,29.6261005401611), Pointf3(514.70361328125,47.7324333190918,35.4384994506836), Pointf3(511.07861328125,47.7324333190918,35.6259002685547), Pointf3(514.70361328125,47.7324333190918,24.5635013580322), Pointf3(511.07861328125,47.7324333190918,24.3761005401611), Pointf3(496.57861328125,47.7324333190918,34.8759002685547), Pointf3(496.57861328125,47.7324333190918,25.1261005401611), Pointf3(496.57861328125,47.7324333190918,35.6259002685547), Pointf3(496.57861328125,47.7324333190918,24.3761005401611), Pointf3(511.07861328125,47.7324333190918,36.0009002685547), Pointf3(511.07861328125,47.7324333190918,24.0011005401611), Pointf3(514.70361328125,47.7324333190918,30.1885013580322), Pointf3(514.70361328125,47.7324333190918,35.8134994506836), Pointf3(514.70361328125,47.7324333190918,29.8135013580322), Pointf3(514.70361328125,47.7324333190918,24.1885013580322), Pointf3(496.57861328125,47.7324333190918,36.0009002685547), Pointf3(496.57861328125,47.7324333190918,24.0011005401611), Pointf3(510.238800048828,49.7324333190918,24.0011005401611), Pointf3(510.238800048828,49.7324333190918,36.0009002685547), Pointf3(514.70361328125,47.7324333190918,24.0011005401611), Pointf3(514.70361328125,47.7324333190918,36.0009002685547), Pointf3(496.158813476562,48.7324333190918,36.0009002685547), Pointf3(496.158813476562,48.7324333190918,24.0011005401611), Pointf3(502.808807373047,62.8258323669434,30.0011005401611), Pointf3(509.608795166016,51.2324333190918,24.0011005401611), Pointf3(509.608795166016,51.2324333190918,36.0009002685547), Pointf3(491.641204833984,50.8574333190918,24.0011005401611), Pointf3(495.423797607422,50.4824333190918,36.0009002685547), Pointf3(495.423797607422,50.4824333190918,24.0011005401611), Pointf3(491.641204833984,50.8574333190918,36.0009002685547), Pointf3(495.528594970703,50.2324333190918,24.0011005401611), Pointf3(492.0087890625,49.9824333190918,24.0011005401611), Pointf3(509.818786621094,50.7324333190918,24.0011005401611), Pointf3(495.948608398438,49.2324333190918,36.0009002685547), Pointf3(495.528594970703,50.2324333190918,36.0009002685547), Pointf3(495.948608398438,49.2324333190918,24.0011005401611), Pointf3(509.818786621094,50.7324333190918,36.0009002685547), Pointf3(492.0087890625,49.9824333190918,36.0009002685547), Pointf3(491.956207275391,50.1074333190918,24.0011005401611), Pointf3(491.956207275391,50.1074333190918,36.0009002685547), Pointf3(502.928588867188,81.599235534668,30.0011005401611), Pointf3(491.851013183594,50.3574333190918,36.0009002685547), Pointf3(491.851013183594,50.3574333190918,24.0011005401611), Pointf3(496.195404052734,54.7190322875977,30.0011005401611), Pointf3(509.361999511719,54.7190322875977,30.0011005401611), Pointf3(488.632598876953,51.7256317138672,30.0011005401611), Pointf3(488.632598876953,51.7256317138672,29.5091018676758), Pointf3(488.632598876953,51.7188339233398,24.0011005401611), Pointf3(488.632598876953,51.7256317138672,27.4929008483887), Pointf3(488.632598876953,51.7324333190918,30.0011005401611), Pointf3(488.632598876953,51.7324333190918,29.0175018310547), Pointf3(488.632598876953,51.7324333190918,24.9847011566162), Pointf3(488.632598876953,51.7324333190918,24.0011005401611), Pointf3(488.632598876953,51.7188339233398,30.0011005401611), Pointf3(488.632598876953,51.7176322937012,24.0011005401611), Pointf3(488.632598876953,51.7182312011719,30.0011005401611), Pointf3(488.632598876953,51.7176322937012,30.0011005401611), Pointf3(488.632598876953,51.715030670166,24.0011005401611), Pointf3(488.632598876953,51.7162322998047,30.0011005401611), Pointf3(488.632598876953,50.761833190918,24.0011005401611), Pointf3(488.632598876953,50.7578315734863,24.0011005401611), Pointf3(488.632598876953,50.7598342895508,30.0011005401611), Pointf3(488.632598876953,50.7522315979004,24.0011005401611), Pointf3(488.632598876953,49.7838325500488,24.0011005401611), Pointf3(488.632598876953,50.2680320739746,30.0011005401611), Pointf3(488.632598876953,51.7046318054199,24.0011005401611), Pointf3(488.632598876953,51.709831237793,30.0011005401611), Pointf3(488.632598876953,50.9120330810547,24.0011005401611), Pointf3(488.632598876953,50.8882331848145,24.0011005401611), Pointf3(488.632598876953,50.9002304077148,30.0011005401611), Pointf3(488.632598876953,47.7324333190918,24.0370998382568), Pointf3(488.632598876953,48.5612335205078,30.0011005401611), Pointf3(488.632598876953,47.7324333190918,24.0011005401611), Pointf3(488.632598876953,47.7324333190918,24.1091003417969), Pointf3(488.632598876953,48.5612335205078,30.0189018249512), Pointf3(488.632598876953,47.7324333190918,25.3211002349854), Pointf3(488.632598876953,48.5612335205078,30.0551013946533), Pointf3(488.632598876953,47.7324333190918,25.4651012420654), Pointf3(488.632598876953,48.5612335205078,30.6609001159668), Pointf3(488.632598876953,47.7324333190918,25.5371017456055), Pointf3(488.632598876953,48.5612335205078,30.7329006195068), Pointf3(488.632598876953,47.7324333190918,25.6091003417969), Pointf3(488.632598876953,48.5612335205078,30.7689018249512), Pointf3(488.632598876953,47.7324333190918,25.8971004486084), Pointf3(488.632598876953,48.5612335205078,30.8051013946533), Pointf3(488.632598876953,47.7324333190918,28.321102142334), Pointf3(488.632598876953,48.5612335205078,30.9491004943848), Pointf3(488.632598876953,47.7324333190918,28.4651012420654), Pointf3(488.632598876953,48.5612335205078,32.1609001159668), Pointf3(488.632598876953,47.7324333190918,28.5371017456055), Pointf3(488.632598876953,48.5612335205078,32.2329025268555), Pointf3(488.632598876953,47.7324333190918,28.6811008453369), Pointf3(488.632598876953,48.5612335205078,32.2689018249512), Pointf3(488.632598876953,47.7324333190918,31.1049003601074), Pointf3(488.632598876953,48.5612335205078,32.3411026000977), Pointf3(488.632598876953,47.7324333190918,31.3929004669189), Pointf3(488.632598876953,49.3900299072266,36.0009002685547), Pointf3(488.632598876953,47.7324333190918,31.536901473999), Pointf3(488.632598876953,47.7324333190918,31.6809005737305), Pointf3(488.632598876953,47.7324333190918,34.1049003601074), Pointf3(488.632598876953,47.7324333190918,34.3929023742676), Pointf3(488.632598876953,47.7324333190918,34.464900970459), Pointf3(488.632598876953,47.7324333190918,34.5369033813477), Pointf3(488.632598876953,47.7324333190918,34.6809005737305), Pointf3(488.632598876953,47.7324333190918,35.8929023742676), Pointf3(488.632598876953,47.7324333190918,35.964900970459), Pointf3(488.632598876953,47.7324333190918,36.0009002685547), Pointf3(488.632598876953,50.8816299438477,24.0011005401611), Pointf3(488.632598876953,50.8850326538086,30.0011005401611), Pointf3(488.632598876953,49.7480316162109,24.0011005401611), Pointf3(488.632598876953,49.7426300048828,24.0011005401611), Pointf3(488.632598876953,49.745231628418,30.0011005401611), Pointf3(488.632598876953,49.7592315673828,24.0011005401611), Pointf3(488.632598876953,49.7536315917969,30.0011005401611), Pointf3(488.632598876953,49.3900299072266,24.0011005401611), Pointf3(488.632598876953,49.5664329528809,30.0011005401611), Pointf3(488.632598876953,50.8786315917969,24.0011005401611), Pointf3(488.632598876953,50.7764320373535,24.0011005401611), Pointf3(488.632598876953,50.8274307250977,30.0011005401611), Pointf3(488.632598876953,50.7550315856934,30.0011005401611), Pointf3(488.632598876953,50.7692337036133,30.0011005401611), Pointf3(488.632598876953,50.9284324645996,24.0011005401611), Pointf3(488.632598876953,50.9202308654785,30.0011005401611), Pointf3(488.632598876953,51.1788330078125,24.0011005401611), Pointf3(488.632598876953,51.139232635498,24.0011005401611), Pointf3(488.632598876953,51.1590309143066,30.0011005401611), Pointf3(488.632598876953,51.2324333190918,24.0011005401611), Pointf3(488.632598876953,51.2056312561035,30.0011005401611), Pointf3(488.632598876953,51.4340324401855,24.0011005401611), Pointf3(488.632598876953,51.3946304321289,24.0011005401611), Pointf3(488.632598876953,51.4142303466797,30.0011005401611), Pointf3(488.632598876953,51.4498329162598,24.0011005401611), Pointf3(488.632598876953,51.5772323608398,30.0011005401611), Pointf3(488.632598876953,51.4418334960938,30.0011005401611), Pointf3(488.632598876953,51.3136329650879,30.0011005401611), Pointf3(488.632598876953,49.7714309692383,30.0011005401611), Pointf3(488.632598876953,51.0338325500488,30.0011005401611), Pointf3(488.632598876953,50.8816299438477,30.0011005401611), Pointf3(488.632598876953,50.8800315856934,30.0011005401611), Pointf3(488.632598876953,51.7188339233398,36.0009002685547), Pointf3(488.632598876953,51.7176322937012,36.0009002685547), Pointf3(488.632598876953,49.3900299072266,30.0011005401611), Pointf3(488.632598876953,50.7522315979004,30.0011005401611), Pointf3(488.632598876953,50.7522315979004,36.0009002685547), Pointf3(488.632598876953,49.7426300048828,30.0011005401611), Pointf3(488.632598876953,49.7426300048828,36.0009002685547), Pointf3(488.632598876953,49.7480316162109,30.0011005401611), Pointf3(488.632598876953,49.7480316162109,36.0009002685547), Pointf3(488.632598876953,51.715030670166,30.0011005401611), Pointf3(488.632598876953,51.715030670166,36.0009002685547), Pointf3(488.632598876953,50.7578315734863,30.0011005401611), Pointf3(488.632598876953,50.7578315734863,36.0009002685547), Pointf3(488.632598876953,50.761833190918,30.0011005401611), Pointf3(488.632598876953,50.761833190918,36.0009002685547), Pointf3(488.632598876953,50.8882331848145,30.0011005401611), Pointf3(488.632598876953,50.8882331848145,36.0009002685547), Pointf3(488.632598876953,49.7592315673828,30.0011005401611), Pointf3(488.632598876953,49.7592315673828,36.0009002685547), Pointf3(488.632598876953,51.1788330078125,30.0011005401611), Pointf3(488.632598876953,51.1788330078125,36.0009002685547), Pointf3(488.632598876953,50.9120330810547,30.0011005401611), Pointf3(488.632598876953,50.9120330810547,36.0009002685547), Pointf3(488.632598876953,51.4498329162598,30.0011005401611), Pointf3(488.632598876953,51.4498329162598,36.0009002685547), Pointf3(488.632598876953,51.7046318054199,30.0011005401611), Pointf3(488.632598876953,51.7046318054199,36.0009002685547), Pointf3(488.632598876953,51.2324333190918,30.0011005401611), Pointf3(488.632598876953,51.2324333190918,36.0009002685547), Pointf3(488.632598876953,51.3946304321289,30.0011005401611), Pointf3(488.632598876953,51.3946304321289,36.0009002685547), Pointf3(488.632598876953,51.4340324401855,30.0011005401611), Pointf3(488.632598876953,51.4340324401855,36.0009002685547), Pointf3(488.632598876953,49.7838325500488,30.0011005401611), Pointf3(488.632598876953,49.7838325500488,36.0009002685547), Pointf3(488.632598876953,50.7764320373535,30.0011005401611), Pointf3(488.632598876953,50.7764320373535,36.0009002685547), Pointf3(488.632598876953,51.139232635498,30.0011005401611), Pointf3(488.632598876953,51.139232635498,36.0009002685547), Pointf3(488.632598876953,50.9284324645996,30.0011005401611), Pointf3(488.632598876953,50.9284324645996,36.0009002685547), Pointf3(488.632598876953,50.8816299438477,36.0009002685547), Pointf3(488.632598876953,50.8786315917969,30.0011005401611), Pointf3(488.632598876953,50.8786315917969,36.0009002685547), Pointf3(488.632598876953,51.7324333190918,35.0173034667969), Pointf3(488.632598876953,51.7324333190918,36.0009002685547), Pointf3(488.632598876953,51.7324333190918,30.9847011566162), Pointf3(517.188415527344,51.7140884399414,24.0011005401611), Pointf3(517.188415527344,51.7140884399414,36.0009002685547), Pointf3(517.188415527344,50.4475173950195,24.0011005401611), Pointf3(517.188415527344,51.7324333190918,35.3734130859375), Pointf3(517.188415527344,51.7324333190918,36.0009002685547), Pointf3(517.188415527344,51.7324333190918,34.1185760498047), Pointf3(517.188415527344,51.7324333190918,31.88330078125), Pointf3(517.188415527344,51.7324333190918,30.0011005401611), Pointf3(517.188415527344,51.7324333190918,28.1187744140625), Pointf3(517.188415527344,51.7324333190918,25.8834266662598), Pointf3(517.188415527344,51.7324333190918,24.6285915374756), Pointf3(517.188415527344,51.7324333190918,24.0011005401611), Pointf3(517.188415527344,47.7324333190918,24.0600452423096), Pointf3(517.188415527344,47.7324333190918,24.0011005401611), Pointf3(517.188415527344,50.4475173950195,36.0009002685547), Pointf3(517.188415527344,47.7324333190918,24.1779975891113), Pointf3(517.188415527344,47.7324333190918,24.6498031616211), Pointf3(517.188415527344,47.7324333190918,28.7625770568848), Pointf3(517.188415527344,47.7324333190918,29.7061901092529), Pointf3(517.188415527344,47.7324333190918,29.9420928955078), Pointf3(517.188415527344,47.7324333190918,30.0600452423096), Pointf3(517.188415527344,47.7324333190918,30.2959480285645), Pointf3(517.188415527344,47.7324333190918,31.2395629882812), Pointf3(517.188415527344,47.7324333190918,35.3521995544434), Pointf3(517.188415527344,47.7324333190918,35.8240051269531), Pointf3(517.188415527344,47.7324333190918,35.9419555664062), Pointf3(517.188415527344,47.7324333190918,36.0009002685547) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(3,4,5), Point3(6,7,8), Point3(9,10,11), Point3(12,2,1), Point3(12,1,13), Point3(14,15,16), Point3(17,18,19), Point3(20,21,22), Point3(17,19,23), Point3(24,25,26), Point3(27,13,1), Point3(28,25,29), Point3(30,31,32), Point3(28,33,34), Point3(35,36,7), Point3(37,38,39), Point3(40,10,41), Point3(42,43,44), Point3(45,5,4), Point3(46,47,48), Point3(46,48,49), Point3(45,4,50), Point3(51,52,53), Point3(51,54,55), Point3(56,52,57), Point3(58,59,60), Point3(61,50,4), Point3(62,63,64), Point3(65,34,33), Point3(66,67,42), Point3(68,17,69), Point3(70,71,22), Point3(66,42,72), Point3(73,16,15), Point3(35,7,74), Point3(75,76,54), Point3(77,27,1), Point3(78,32,31), Point3(75,54,79), Point3(80,26,25), Point3(81,80,25), Point3(82,83,48), Point3(84,20,85), Point3(81,25,86), Point3(87,88,19), Point3(0,89,1), Point3(90,91,92), Point3(90,10,93), Point3(38,94,39), Point3(94,95,39), Point3(3,7,96), Point3(97,15,98), Point3(97,99,15), Point3(92,91,100), Point3(89,101,1), Point3(102,39,95), Point3(103,11,10), Point3(104,96,7), Point3(105,15,99), Point3(106,61,4), Point3(107,108,33), Point3(76,55,54), Point3(109,91,110), Point3(111,23,19), Point3(112,63,113), Point3(114,115,48), Point3(116,59,117), Point3(118,20,119), Point3(120,31,121), Point3(122,44,43), Point3(110,91,123), Point3(124,125,126), Point3(127,128,129), Point3(127,130,124), Point3(131,124,132), Point3(126,133,134), Point3(135,136,126), Point3(137,138,127), Point3(139,127,138), Point3(128,140,141), Point3(142,128,143), Point3(144,140,145), Point3(100,91,146), Point3(147,148,134), Point3(101,149,1), Point3(102,150,39), Point3(103,10,151), Point3(145,140,152), Point3(152,140,153), Point3(148,154,134), Point3(154,155,134), Point3(156,15,105), Point3(157,104,7), Point3(36,8,7), Point3(158,37,39), Point3(159,19,88), Point3(160,19,159), Point3(161,59,58), Point3(161,117,59), Point3(162,31,30), Point3(162,121,31), Point3(163,43,164), Point3(163,165,43), Point3(166,167,43), Point3(167,164,43), Point3(168,57,52), Point3(82,48,169), Point3(114,170,171), Point3(108,65,33), Point3(64,63,112), Point3(114,172,170), Point3(160,173,170), Point3(171,170,173), Point3(172,174,170), Point3(160,170,174), Point3(175,176,177), Point3(178,77,1), Point3(179,31,120), Point3(175,180,176), Point3(181,182,176), Point3(177,176,182), Point3(180,183,176), Point3(181,176,183), Point3(184,42,67), Point3(185,69,17), Point3(160,111,19), Point3(186,187,160), Point3(188,189,114), Point3(190,188,114), Point3(114,48,191), Point3(192,114,193), Point3(194,160,195), Point3(196,160,194), Point3(197,198,181), Point3(199,197,181), Point3(122,43,165), Point3(200,201,175), Point3(202,175,203), Point3(204,175,202), Point3(205,119,20), Point3(206,181,207), Point3(208,209,15), Point3(210,15,209), Point3(211,10,9), Point3(212,10,211), Point3(213,214,215), Point3(216,217,218), Point3(219,14,17), Point3(113,63,220), Point3(221,222,48), Point3(191,48,222), Point3(22,223,20), Point3(205,20,223), Point3(224,40,42), Point3(123,91,225), Point3(214,226,215), Point3(227,215,226), Point3(218,217,228), Point3(229,228,217), Point3(215,230,213), Point3(125,135,126), Point3(217,216,231), Point3(129,128,142), Point3(216,213,232), Point3(130,132,124), Point3(213,216,233), Point3(234,213,235), Point3(236,227,237), Point3(238,237,227), Point3(239,240,216), Point3(233,216,240), Point3(241,242,229), Point3(243,229,242), Point3(215,227,244), Point3(245,215,246), Point3(217,247,229), Point3(248,249,217), Point3(232,213,250), Point3(230,250,213), Point3(133,147,134), Point3(244,227,251), Point3(236,252,227), Point3(251,227,252), Point3(231,216,253), Point3(254,253,216), Point3(141,140,144), Point3(247,255,229), Point3(241,229,256), Point3(255,256,229), Point3(257,241,258), Point3(259,146,91), Point3(260,261,236), Point3(262,1,149), Point3(263,264,241), Point3(265,241,264), Point3(266,236,267), Point3(268,267,236), Point3(49,48,83), Point3(166,43,269), Point3(270,271,272), Point3(273,274,275), Point3(276,274,277), Point3(278,151,10), Point3(279,280,272), Point3(281,39,150), Point3(272,282,279), Point3(155,283,134), Point3(274,276,284), Point3(153,140,285), Point3(286,276,287), Point3(265,276,286), Point3(288,289,279), Point3(268,288,279), Point3(290,291,272), Point3(271,290,272), Point3(292,274,293), Point3(275,274,292), Point3(294,265,295), Point3(276,265,294), Point3(296,297,268), Point3(279,296,268), Point3(241,265,298), Point3(298,265,299), Point3(236,300,268), Point3(300,301,268), Point3(107,33,78), Point3(302,303,59), Point3(304,305,279), Point3(282,304,279), Point3(306,276,307), Point3(284,276,306), Point3(185,17,73), Point3(308,309,221), Point3(158,39,70), Point3(310,41,10), Point3(15,311,208), Point3(7,6,312), Point3(313,314,6), Point3(315,6,314), Point3(316,208,317), Point3(318,317,208), Point3(258,241,319), Point3(319,241,320), Point3(261,321,236), Point3(321,322,236), Point3(6,315,323), Point3(208,324,318), Point3(270,325,318), Point3(326,318,325), Point3(327,328,315), Point3(273,315,328), Point3(118,329,20), Point3(330,20,329), Point3(331,332,25), Point3(86,25,332), Point3(333,334,52), Point3(335,52,334), Point3(115,336,48), Point3(169,48,336), Point3(62,106,4), Point3(35,15,210), Point3(35,337,15), Point3(158,10,212), Point3(158,310,10), Point3(338,178,1), Point3(339,59,116), Point3(107,302,59), Point3(66,22,340), Point3(66,341,22), Point3(185,221,342), Point3(185,308,221), Point3(75,31,179), Point3(75,343,31), Point3(166,20,330), Point3(166,85,20), Point3(81,52,335), Point3(81,168,52), Point3(82,19,344), Point3(82,87,19), Point3(108,339,345), Point3(346,108,345), Point3(64,347,348), Point3(349,347,64), Point3(178,109,350), Point3(351,178,350), Point3(179,352,353), Point3(354,352,179), Point3(355,208,356), Point3(356,208,311), Point3(357,358,6), Point3(358,312,6), Point3(68,22,21), Point3(68,340,22), Point3(221,48,47), Point3(184,342,221), Point3(359,270,360), Point3(318,360,270), Point3(361,362,273), Point3(315,273,362), Point3(272,102,270), Point3(363,270,102), Point3(274,273,103), Point3(364,103,273), Point3(21,19,18), Point3(21,20,84), Point3(184,46,42), Point3(43,42,46), Point3(12,22,71), Point3(365,22,12), Point3(14,98,15), Point3(14,220,63), Point3(40,93,10), Point3(40,225,91), Point3(45,221,309), Point3(366,221,45), Point3(313,367,212), Point3(212,367,368), Point3(36,369,367), Point3(313,36,367), Point3(316,37,367), Point3(37,368,367), Point3(210,367,369), Point3(316,367,210), Point3(362,370,315), Point3(370,323,315), Point3(360,318,371), Point3(371,318,324), Point3(372,331,159), Point3(159,195,160), Point3(373,115,56), Point3(115,114,189), Point3(52,56,161), Point3(374,161,56), Point3(25,28,331), Point3(375,331,28), Point3(376,333,163), Point3(163,203,175), Point3(377,118,24), Point3(118,181,198), Point3(25,24,162), Point3(378,162,24), Point3(52,51,333), Point3(379,333,51), Point3(167,380,381), Point3(376,167,381), Point3(377,381,330), Point3(330,381,380), Point3(335,381,382), Point3(376,381,335), Point3(373,383,169), Point3(169,383,384), Point3(168,385,383), Point3(373,168,383), Point3(372,87,383), Point3(87,384,383), Point3(377,80,381), Point3(80,382,381), Point3(86,383,385), Point3(372,383,86), Point3(106,348,347), Point3(386,106,347), Point3(375,65,346), Point3(108,346,65), Point3(64,112,349), Point3(387,349,112), Point3(171,190,114), Point3(346,345,171), Point3(374,190,345), Point3(171,345,190), Point3(349,172,347), Point3(172,114,192), Point3(386,347,192), Point3(172,192,347), Point3(173,160,196), Point3(171,173,346), Point3(375,346,196), Point3(173,196,346), Point3(172,349,174), Point3(174,186,160), Point3(387,186,349), Point3(174,349,186), Point3(64,348,62), Point3(106,62,348), Point3(108,107,339), Point3(59,339,107), Point3(374,345,116), Point3(339,116,345), Point3(76,353,352), Point3(379,76,352), Point3(388,77,351), Point3(178,351,77), Point3(179,120,354), Point3(378,354,120), Point3(177,200,175), Point3(351,350,177), Point3(389,200,350), Point3(177,350,200), Point3(354,180,352), Point3(180,175,204), Point3(379,352,204), Point3(180,204,352), Point3(182,181,206), Point3(177,182,351), Point3(388,351,206), Point3(182,206,351), Point3(180,354,183), Point3(183,199,181), Point3(378,199,354), Point3(183,354,199), Point3(91,109,338), Point3(178,338,109), Point3(76,75,353), Point3(179,353,75), Point3(389,350,110), Point3(109,110,350), Point3(390,391,392), Point3(393,394,395), Point3(224,122,389), Point3(122,175,201), Point3(365,388,205), Point3(205,207,181), Point3(66,340,396), Point3(68,396,340), Point3(184,396,342), Point3(185,342,396), Point3(66,396,67), Point3(184,67,396), Point3(68,69,396), Point3(185,396,69), Point3(219,111,387), Point3(111,160,187), Point3(366,386,191), Point3(191,193,114), Point3(150,272,280), Point3(102,272,150), Point3(151,277,274), Point3(103,151,274), Point3(161,374,117), Point3(116,117,374), Point3(366,61,386), Point3(106,386,61), Point3(111,187,387), Point3(186,387,187), Point3(56,188,374), Point3(190,374,188), Point3(191,386,193), Point3(192,193,386), Point3(331,375,194), Point3(196,194,375), Point3(28,34,375), Point3(65,375,34), Point3(219,387,113), Point3(112,113,387), Point3(224,389,123), Point3(110,123,389), Point3(51,55,379), Point3(76,379,55), Point3(24,197,378), Point3(199,378,197), Point3(122,201,389), Point3(200,389,201), Point3(333,379,202), Point3(204,202,379), Point3(205,388,207), Point3(206,207,388), Point3(365,27,388), Point3(77,388,27), Point3(162,378,121), Point3(120,121,378), Point3(162,30,25), Point3(30,29,25), Point3(51,53,54), Point3(303,60,59), Point3(28,29,33), Point3(29,397,33), Point3(161,58,52), Point3(53,52,58), Point3(21,84,19), Point3(84,344,19), Point3(46,49,43), Point3(49,269,43), Point3(208,316,209), Point3(210,209,316), Point3(327,313,211), Point3(212,211,313), Point3(36,35,369), Point3(210,369,35), Point3(37,158,368), Point3(212,368,158), Point3(6,8,313), Point3(36,313,8), Point3(326,38,316), Point3(37,316,38), Point3(392,391,398), Point3(399,398,391), Point3(394,400,395), Point3(401,395,400), Point3(390,214,391), Point3(214,213,234), Point3(393,395,218), Point3(218,239,216), Point3(402,230,403), Point3(230,215,245), Point3(125,124,131), Point3(404,125,403), Point3(405,406,231), Point3(231,248,217), Point3(129,137,127), Point3(407,406,129), Point3(130,127,139), Point3(402,130,408), Point3(194,195,331), Point3(159,331,195), Point3(115,189,56), Point3(188,56,189), Point3(14,219,220), Point3(113,220,219), Point3(45,50,366), Point3(61,366,50), Point3(221,366,222), Point3(191,222,366), Point3(17,23,219), Point3(111,219,23), Point3(118,198,24), Point3(197,24,198), Point3(202,203,333), Point3(163,333,203), Point3(40,224,225), Point3(123,225,224), Point3(12,13,365), Point3(27,365,13), Point3(22,365,223), Point3(205,223,365), Point3(42,44,224), Point3(122,224,44), Point3(399,391,234), Point3(214,234,391), Point3(401,239,395), Point3(218,395,239), Point3(214,390,226), Point3(226,238,227), Point3(218,228,393), Point3(228,229,243), Point3(401,399,233), Point3(233,235,213), Point3(392,409,390), Point3(410,390,409), Point3(394,393,411), Point3(412,411,393), Point3(402,403,131), Point3(125,131,403), Point3(405,137,406), Point3(129,406,137), Point3(405,408,139), Point3(130,139,408), Point3(230,245,403), Point3(404,403,245), Point3(231,406,248), Point3(407,248,406), Point3(232,254,216), Point3(402,408,232), Point3(413,404,244), Point3(244,246,215), Point3(414,247,407), Point3(247,217,249), Point3(133,126,136), Point3(415,133,413), Point3(141,143,128), Point3(416,414,141), Point3(410,238,390), Point3(226,390,238), Point3(412,393,243), Point3(228,243,393), Point3(233,399,235), Point3(234,235,399), Point3(237,260,236), Point3(238,410,237), Point3(417,260,410), Point3(237,410,260), Point3(239,401,240), Point3(233,240,401), Point3(242,241,257), Point3(243,242,412), Point3(418,412,257), Point3(242,257,412), Point3(401,419,399), Point3(398,399,419), Point3(417,410,420), Point3(409,420,410), Point3(400,421,401), Point3(419,401,421), Point3(418,422,412), Point3(411,412,422), Point3(413,135,404), Point3(125,404,135), Point3(414,407,142), Point3(129,142,407), Point3(130,402,132), Point3(131,132,402), Point3(133,136,413), Point3(135,413,136), Point3(423,147,415), Point3(133,415,147), Point3(137,405,138), Point3(139,138,405), Point3(141,414,143), Point3(142,143,414), Point3(424,416,144), Point3(141,144,416), Point3(405,254,408), Point3(232,408,254), Point3(244,404,246), Point3(245,246,404), Point3(247,249,407), Point3(248,407,249), Point3(232,250,402), Point3(230,402,250), Point3(415,413,251), Point3(244,251,413), Point3(252,236,266), Point3(251,252,415), Point3(423,415,266), Point3(252,266,415), Point3(231,253,405), Point3(254,405,253), Point3(416,255,414), Point3(247,414,255), Point3(256,263,241), Point3(255,416,256), Point3(424,263,416), Point3(256,416,263), Point3(257,258,418), Point3(425,418,258), Point3(260,417,261), Point3(426,261,417), Point3(422,418,427), Point3(427,259,91), Point3(420,428,417), Point3(428,1,262), Point3(147,423,148), Point3(429,148,423), Point3(263,424,264), Point3(264,295,265), Point3(266,267,423), Point3(267,268,297), Point3(144,145,424), Point3(430,424,145), Point3(49,431,269), Point3(166,269,431), Point3(82,431,83), Point3(49,83,431), Point3(84,85,431), Point3(166,431,85), Point3(82,344,431), Point3(84,431,344), Point3(432,278,90), Point3(10,90,278), Point3(433,0,281), Point3(39,281,0), Point3(362,361,434), Point3(435,271,359), Point3(270,359,271), Point3(436,361,275), Point3(273,275,361), Point3(360,437,359), Point3(277,287,276), Point3(151,278,277), Point3(280,279,289), Point3(150,280,281), Point3(436,438,439), Point3(439,285,140), Point3(90,92,432), Point3(440,432,92), Point3(282,272,291), Point3(441,282,442), Point3(284,293,274), Point3(443,438,284), Point3(278,432,286), Point3(286,299,265), Point3(281,288,433), Point3(288,268,301), Point3(0,433,89), Point3(444,89,433), Point3(435,445,442), Point3(445,134,283), Point3(439,446,436), Point3(361,436,446), Point3(442,290,435), Point3(271,435,290), Point3(438,436,292), Point3(275,292,436), Point3(445,435,447), Point3(359,447,435), Point3(286,287,278), Point3(277,278,287), Point3(288,281,289), Point3(280,289,281), Point3(145,152,430), Point3(443,430,152), Point3(148,429,154), Point3(441,154,429), Point3(424,430,294), Point3(294,307,276), Point3(423,296,429), Point3(296,279,305), Point3(425,440,100), Point3(92,100,440), Point3(290,442,291), Point3(282,291,442), Point3(292,293,438), Point3(284,438,293), Point3(298,320,241), Point3(432,440,298), Point3(300,236,322), Point3(433,300,444), Point3(426,101,444), Point3(89,444,101), Point3(107,448,302), Point3(302,79,54), Point3(78,31,343), Point3(107,78,448), Point3(75,79,448), Point3(302,448,79), Point3(78,343,448), Point3(75,448,343), Point3(427,418,259), Point3(425,259,418), Point3(428,262,417), Point3(426,417,262), Point3(437,449,359), Point3(447,359,449), Point3(434,361,450), Point3(446,450,361), Point3(32,33,397), Point3(78,33,32), Point3(53,303,54), Point3(302,54,303), Point3(152,153,443), Point3(438,443,153), Point3(429,304,441), Point3(282,441,304), Point3(430,443,306), Point3(284,306,443), Point3(154,441,155), Point3(442,155,441), Point3(298,299,432), Point3(286,432,299), Point3(300,433,301), Point3(288,301,433), Point3(185,451,308), Point3(308,74,7), Point3(73,15,337), Point3(185,73,451), Point3(35,74,451), Point3(308,451,74), Point3(73,337,451), Point3(35,451,337), Point3(158,452,310), Point3(310,72,42), Point3(70,22,341), Point3(158,70,452), Point3(66,72,452), Point3(310,452,72), Point3(70,341,452), Point3(66,452,341), Point3(313,327,314), Point3(315,314,327), Point3(316,317,326), Point3(318,326,317), Point3(15,156,311), Point3(356,311,156), Point3(7,312,157), Point3(358,157,312), Point3(211,9,327), Point3(364,327,9), Point3(38,326,94), Point3(363,94,326), Point3(294,295,424), Point3(264,424,295), Point3(296,423,297), Point3(267,297,423), Point3(262,149,426), Point3(101,426,149), Point3(258,319,425), Point3(440,425,319), Point3(261,426,321), Point3(444,321,426), Point3(259,425,146), Point3(100,146,425), Point3(306,307,430), Point3(294,430,307), Point3(304,429,305), Point3(296,305,429), Point3(319,320,440), Point3(298,440,320), Point3(321,444,322), Point3(300,322,444), Point3(445,283,442), Point3(155,442,283), Point3(439,438,285), Point3(153,285,438), Point3(17,68,18), Point3(21,18,68), Point3(46,184,47), Point3(221,47,184), Point3(102,95,363), Point3(94,363,95), Point3(9,11,364), Point3(103,364,11), Point3(6,323,357), Point3(370,357,323), Point3(371,324,355), Point3(208,355,324), Point3(270,363,325), Point3(326,325,363), Point3(327,364,328), Point3(273,328,364), Point3(0,2,39), Point3(12,39,2), Point3(90,93,91), Point3(40,91,93), Point3(14,16,17), Point3(73,17,16), Point3(45,309,7), Point3(308,7,309), Point3(12,71,39), Point3(70,39,71), Point3(40,41,42), Point3(310,42,41), Point3(97,98,63), Point3(14,63,98), Point3(3,5,7), Point3(45,7,5), Point3(118,377,329), Point3(330,329,377), Point3(331,372,332), Point3(86,332,372), Point3(333,376,334), Point3(335,334,376), Point3(115,373,336), Point3(169,336,373), Point3(167,166,380), Point3(330,380,166), Point3(80,81,382), Point3(335,382,81), Point3(86,385,81), Point3(168,81,385), Point3(169,384,82), Point3(87,82,384), Point3(159,88,372), Point3(87,372,88), Point3(163,164,376), Point3(167,376,164), Point3(24,26,377), Point3(80,377,26), Point3(56,57,373), Point3(168,373,57), Point3(32,397,30), Point3(29,30,397), Point3(58,60,53), Point3(303,53,60), Point3(205,181,119), Point3(118,119,181), Point3(163,175,165), Point3(122,165,175), Point3(453,454,455), Point3(454,456,455), Point3(457,455,456), Point3(458,455,457), Point3(459,455,458), Point3(460,455,459), Point3(461,462,463), Point3(464,465,466), Point3(467,468,469), Point3(470,471,472), Point3(465,473,474), Point3(475,476,477), Point3(478,479,480), Point3(481,482,478), Point3(483,484,481), Point3(485,486,483), Point3(487,488,485), Point3(489,490,487), Point3(491,492,489), Point3(493,494,491), Point3(495,496,493), Point3(497,498,495), Point3(499,500,497), Point3(501,502,499), Point3(503,504,501), Point3(505,504,503), Point3(506,504,505), Point3(507,504,506), Point3(508,504,507), Point3(509,504,508), Point3(510,504,509), Point3(511,504,510), Point3(512,504,511), Point3(513,504,512), Point3(514,504,513), Point3(476,515,516), Point3(517,518,519), Point3(520,517,521), Point3(518,522,523), Point3(522,480,479), Point3(524,525,526), Point3(468,470,527), Point3(525,467,528), Point3(529,475,530), Point3(531,532,533), Point3(534,531,535), Point3(536,537,538), Point3(473,539,540), Point3(539,536,541), Point3(537,534,542), Point3(471,520,543), Point3(532,529,544), Point3(545,524,546), Point3(453,461,547), Point3(463,464,548), Point3(523,549,504), Point3(527,550,551), Point3(519,552,553), Point3(521,554,555), Point3(466,556,557), Point3(469,558,559), Point3(528,560,561), Point3(477,562,563), Point3(543,564,565), Point3(535,566,567), Point3(530,568,569), Point3(540,570,571), Point3(474,572,573), Point3(542,574,575), Point3(538,576,577), Point3(541,578,579), Point3(472,580,581), Point3(526,582,583), Point3(533,584,585), Point3(544,586,587), Point3(516,545,588), Point3(588,589,590), Point3(455,460,4), Point3(591,592,63), Point3(462,455,4), Point3(592,547,63), Point3(547,548,63), Point3(465,462,4), Point3(548,557,63), Point3(127,124,501), Point3(127,501,499), Point3(505,503,124), Point3(124,126,507), Point3(124,507,506), Point3(509,508,126), Point3(126,134,512), Point3(126,512,511), Point3(510,509,126), Point3(128,127,493), Point3(128,493,491), Point3(497,495,127), Point3(489,487,128), Point3(140,128,483), Point3(140,483,481), Point3(487,485,128), Point3(478,480,140), Point3(480,522,140), Point3(514,513,134), Point3(504,514,134), Point3(551,581,437), Point3(471,470,434), Point3(445,447,555), Point3(445,555,553), Point3(134,445,553), Point3(134,553,504), Point3(446,439,518), Point3(446,518,517), Point3(439,140,522), Point3(439,522,518), Point3(515,476,358), Point3(563,588,356), Point3(557,573,63), Point3(473,465,4), Point3(437,360,559), Point3(437,559,551), Point3(360,371,561), Point3(360,561,559), Point3(362,434,470), Point3(362,470,468), Point3(370,362,468), Point3(370,468,467), Point3(499,497,127), Point3(506,505,124), Point3(495,493,127), Point3(513,512,134), Point3(481,478,140), Point3(447,449,565), Point3(447,565,555), Point3(450,446,517), Point3(450,517,520), Point3(356,156,569), Point3(356,569,563), Point3(157,358,476), Point3(157,476,475), Point3(357,370,467), Point3(357,467,525), Point3(371,355,583), Point3(371,583,561), Point3(460,459,4), Point3(63,62,593), Point3(63,593,591), Point3(62,4,459), Point3(62,459,458), Point3(532,531,104), Point3(531,534,104), Point3(567,585,105), Point3(575,567,105), Point3(4,3,539), Point3(4,539,473), Point3(536,539,3), Point3(97,63,573), Point3(97,573,571), Point3(571,579,97), Point3(99,97,579), Point3(99,579,577), Point3(105,99,577), Point3(105,577,575), Point3(96,104,534), Point3(96,534,537), Point3(3,96,537), Point3(3,537,536), Point3(503,501,124), Point3(508,507,126), Point3(491,489,128), Point3(511,510,126), Point3(485,483,128), Point3(434,450,520), Point3(434,520,471), Point3(449,437,581), Point3(449,581,565), Point3(156,105,585), Point3(156,585,587), Point3(587,569,156), Point3(104,157,529), Point3(104,529,532), Point3(475,529,157), Point3(590,583,355), Point3(355,356,588), Point3(355,588,590), Point3(358,357,524), Point3(358,524,515), Point3(525,524,357), Point3(458,457,62), Point3(457,593,62), Point3(479,478,482), Point3(479,504,549), Point3(479,482,504), Point3(482,481,484), Point3(472,551,550), Point3(581,551,472), Point3(482,484,504), Point3(484,483,486), Point3(523,553,552), Point3(504,553,523), Point3(540,573,572), Point3(571,573,540), Point3(544,585,584), Point3(587,585,544), Point3(542,577,576), Point3(575,577,542), Point3(526,590,589), Point3(583,590,526), Point3(535,575,574), Point3(567,575,535), Point3(533,567,566), Point3(585,567,533), Point3(538,579,578), Point3(577,579,538), Point3(543,581,580), Point3(565,581,543), Point3(477,569,568), Point3(563,569,477), Point3(530,587,586), Point3(569,587,530), Point3(541,571,570), Point3(579,571,541), Point3(528,583,582), Point3(561,583,528), Point3(591,453,592), Point3(547,592,453), Point3(521,565,564), Point3(555,565,521), Point3(474,557,556), Point3(573,557,474), Point3(516,563,562), Point3(588,563,516), Point3(519,555,554), Point3(553,555,519), Point3(527,559,558), Point3(551,559,527), Point3(469,561,560), Point3(559,561,469), Point3(462,461,455), Point3(453,455,461), Point3(461,463,547), Point3(548,547,463), Point3(465,464,462), Point3(463,462,464), Point3(464,466,548), Point3(557,548,466), Point3(469,560,467), Point3(528,467,560), Point3(472,550,470), Point3(527,470,550), Point3(474,556,465), Point3(466,465,556), Point3(477,568,475), Point3(530,475,568), Point3(516,562,476), Point3(477,476,562), Point3(519,554,517), Point3(521,517,554), Point3(521,564,520), Point3(543,520,564), Point3(523,552,518), Point3(519,518,552), Point3(479,549,522), Point3(523,522,549), Point3(526,589,524), Point3(589,546,524), Point3(527,558,468), Point3(469,468,558), Point3(528,582,525), Point3(526,525,582), Point3(530,586,529), Point3(544,529,586), Point3(533,566,531), Point3(535,531,566), Point3(535,574,534), Point3(542,534,574), Point3(538,578,536), Point3(541,536,578), Point3(540,572,473), Point3(474,473,572), Point3(541,570,539), Point3(540,539,570), Point3(542,576,537), Point3(538,537,576), Point3(543,580,471), Point3(472,471,580), Point3(544,584,532), Point3(533,532,584), Point3(524,545,515), Point3(516,515,545), Point3(545,546,588), Point3(589,588,546), Point3(453,591,454), Point3(593,454,591), Point3(484,486,504), Point3(486,485,488), Point3(486,488,504), Point3(488,487,490), Point3(488,490,504), Point3(490,489,492), Point3(490,492,504), Point3(492,491,494), Point3(492,494,504), Point3(494,493,496), Point3(494,496,504), Point3(496,495,498), Point3(496,498,504), Point3(498,497,500), Point3(498,500,504), Point3(500,499,502), Point3(500,502,504), Point3(501,504,502), Point3(454,593,456), Point3(457,456,593), Point3(594,595,596), Point3(597,598,594), Point3(599,597,594), Point3(600,599,594), Point3(601,600,594), Point3(602,601,594), Point3(603,602,594), Point3(604,603,594), Point3(605,604,594), Point3(606,607,608), Point3(609,606,608), Point3(610,609,608), Point3(611,610,608), Point3(612,611,608), Point3(613,612,608), Point3(614,613,608), Point3(615,614,608), Point3(616,615,608), Point3(617,616,608), Point3(618,617,608), Point3(619,618,608), Point3(620,619,608), Point3(596,608,607), Point3(595,594,598), Point3(608,596,595), Point3(605,594,91), Point3(91,338,602), Point3(91,602,603), Point3(598,597,1), Point3(594,596,91), Point3(608,595,1), Point3(595,598,1), Point3(616,617,392), Point3(610,611,394), Point3(419,421,613), Point3(419,613,614), Point3(422,427,607), Point3(422,607,606), Point3(427,91,596), Point3(427,596,607), Point3(428,420,619), Point3(428,619,620), Point3(1,428,620), Point3(1,620,608), Point3(420,409,618), Point3(420,618,619), Point3(411,422,606), Point3(411,606,609), Point3(398,419,614), Point3(398,614,615), Point3(421,400,612), Point3(421,612,613), Point3(409,392,617), Point3(409,617,618), Point3(394,411,609), Point3(394,609,610), Point3(604,605,91), Point3(338,1,599), Point3(338,599,600), Point3(392,398,615), Point3(392,615,616), Point3(400,394,611), Point3(400,611,612), Point3(603,604,91), Point3(601,602,338), Point3(597,599,1), Point3(600,601,338) 
            });
            break;
        case TestMesh::gt2_teeth:
            vertices = std::vector<Pointf3>({
                Pointf3(15.8899993896484,19.444055557251,2.67489433288574), Pointf3(15.9129991531372,19.1590557098389,2.67489433288574), Pointf3(15.9039993286133,19.1500549316406,2.67489433288574), Pointf3(15.9489994049072,19.2490558624268,2.67489433288574), Pointf3(15.9579992294312,19.3570556640625,2.67489433288574), Pointf3(15.8819999694824,18.690055847168,2.67489433288574), Pointf3(15.8319997787476,17.7460556030273,2.67489433288574), Pointf3(15.8489999771118,18.819055557251,2.67489433288574), Pointf3(15.8589992523193,17.7190551757812,2.67489433288574), Pointf3(15.8769998550415,19.0490550994873,2.67489433288574), Pointf3(15.7529993057251,17.8080558776855,2.67489433288574), Pointf3(15.7869997024536,19.5010547637939,2.67489433288574), Pointf3(14.0329990386963,18.7170543670654,2.67489433288574), Pointf3(13.9599990844727,18.7460556030273,2.67489433288574), Pointf3(13.9869995117188,20.2840557098389,2.67489433288574), Pointf3(14.2029991149902,20.149055480957,2.67489433288574), Pointf3(14.1939992904663,19.9560546875,2.67489433288574), Pointf3(14.1939992904663,20.1670551300049,2.67489433288574), Pointf3(14.2119998931885,20.0590553283691,2.67489433288574), Pointf3(12.1899995803833,19.1840553283691,2.67489433288574), Pointf3(12.096999168396,19.1950550079346,2.67489433288574), Pointf3(12.1099996566772,20.6690559387207,2.67489433288574), Pointf3(11.382999420166,19.9750556945801,2.67489433288574), Pointf3(11.2599992752075,19.2490558624268,2.67489433288574), Pointf3(11.2369995117188,19.9320545196533,2.67489433288574), Pointf3(11.5349998474121,20.0640544891357,2.67489433288574), Pointf3(11.6259994506836,20.1550559997559,2.67489433288574), Pointf3(11.6829986572266,20.2390556335449,2.67489433288574), Pointf3(11.7369995117188,20.3570556640625,2.67489433288574), Pointf3(11.8449993133545,20.645055770874,2.67489433288574), Pointf3(11.7729988098145,20.4640560150146,2.67489433288574), Pointf3(11.7799987792969,20.5370559692383,9.41389465332031), Pointf3(11.7639999389648,20.4470558166504,2.67489433288574), Pointf3(11.9559993743896,20.6810550689697,2.67489433288574), Pointf3(12.3079996109009,20.6020545959473,2.67489433288574), Pointf3(12.1959991455078,19.1860542297363,2.67489433288574), Pointf3(12.2059993743896,20.6540546417236,2.67489433288574), Pointf3(12.3489990234375,20.3740558624268,2.67489433288574), Pointf3(12.3579998016357,20.2750549316406,2.67489433288574), Pointf3(12.3669996261597,20.266056060791,2.67489433288574), Pointf3(12.3849992752075,20.1670551300049,2.67489433288574), Pointf3(12.4269990921021,20.0680541992188,2.67489433288574), Pointf3(12.5029993057251,19.9540557861328,2.67489433288574), Pointf3(12.6169996261597,19.8550548553467,2.67489433288574), Pointf3(12.7449989318848,19.7800559997559,2.67489433288574), Pointf3(12.7629995346069,19.7800559997559,2.67489433288574), Pointf3(12.8799991607666,19.7350559234619,2.67489433288574), Pointf3(13.0369997024536,19.7250556945801,2.67489433288574), Pointf3(13.0149993896484,19.0340557098389,2.67489433288574), Pointf3(11.1699991226196,19.2580547332764,2.67489433288574), Pointf3(11.0959987640381,19.2580547332764,2.67489433288574), Pointf3(11.1209993362427,19.9230556488037,2.67489433288574), Pointf3(13.0599994659424,19.024055480957,2.67489433288574), Pointf3(14.9049997329712,18.3170547485352,2.67489433288574), Pointf3(14.8779993057251,18.3400554656982,2.67489433288574), Pointf3(14.8779993057251,19.149055480957,2.67489433288574), Pointf3(13.3039989471436,19.77805519104,2.67489433288574), Pointf3(13.1589994430542,18.9890556335449,2.67489433288574), Pointf3(13.1559991836548,19.7350559234619,2.67489433288574), Pointf3(13.4269990921021,19.8600559234619,2.67489433288574), Pointf3(13.5339994430542,19.9700546264648,2.67389440536499), Pointf3(13.6359996795654,20.1220550537109,2.67489433288574), Pointf3(13.6359996795654,20.1400547027588,2.67489433288574), Pointf3(13.6719989776611,20.2210559844971,2.67489433288574), Pointf3(13.6899995803833,20.2300548553467,2.67489433288574), Pointf3(13.7509994506836,20.3010559082031,2.67489433288574), Pointf3(13.8539991378784,20.3180541992188,2.67489433288574), Pointf3(14.8329992294312,18.3580551147461,2.67489433288574), Pointf3(14.1849994659424,19.8530559539795,2.67489433288574), Pointf3(14.0769996643066,18.7000541687012,2.67489433288574), Pointf3(14.1099996566772,20.2400550842285,2.67489433288574), Pointf3(14.2009992599487,19.6230545043945,2.67489433288574), Pointf3(14.2729997634888,19.4670543670654,2.67489433288574), Pointf3(14.3379993438721,19.3790550231934,2.67489433288574), Pointf3(14.4549999237061,19.2770557403564,2.67489433288574), Pointf3(14.5899991989136,19.2040557861328,2.67489433288574), Pointf3(14.6079998016357,19.2040557861328,2.67489433288574), Pointf3(14.7209997177124,19.1600551605225,2.67489433288574), Pointf3(15.1379995346069,19.210054397583,2.67489433288574), Pointf3(14.9949998855591,18.2680549621582,2.67489433288574), Pointf3(15.0029993057251,19.1580543518066,2.67489433288574), Pointf3(15.2369995117188,19.2760543823242,2.67489433288574), Pointf3(15.3779993057251,19.4060554504395,2.67489433288574), Pointf3(15.4539995193481,19.520055770874,2.67489433288574), Pointf3(15.471999168396,19.52805519104,2.67489433288574), Pointf3(15.5449991226196,19.5830554962158,2.67489433288574), Pointf3(15.6529998779297,19.573055267334,2.67489433288574), Pointf3(15.7059993743896,17.8360557556152,2.67489433288574), Pointf3(15.9449996948242,18.5560550689697,2.67489433288574), Pointf3(15.8589992523193,18.9380550384521,2.67489433288574), Pointf3(14.9589996337891,18.2950553894043,2.67489433288574), Pointf3(15.7779998779297,19.5100555419922,2.67489433288574), Pointf3(14.0049991607666,20.2750549316406,2.67489433288574), Pointf3(12.3489990234375,20.5000553131104,2.67489433288574), Pointf3(13.0689992904663,19.0150547027588,2.67489433288574), Pointf3(13.0999994277954,19.0100555419922,2.67489433288574), Pointf3(15.9489994049072,19.3670558929443,9.41489505767822), Pointf3(15.9489994049072,19.2490558624268,9.41489505767822), Pointf3(15.75,17.8080558776855,9.41489505767822), Pointf3(15.6639995574951,19.5710544586182,9.41489505767822), Pointf3(15.5709991455078,17.9260559082031,9.41489505767822), Pointf3(15.8769998550415,18.690055847168,9.41489505767822), Pointf3(15.8499994277954,18.8170547485352,9.41489505767822), Pointf3(15.9459991455078,18.5520553588867,9.41489505767822), Pointf3(15.914999961853,17.6890544891357,9.41489505767822), Pointf3(15.3999996185303,19.4290542602539,9.41489505767822), Pointf3(15.3099994659424,19.339054107666,9.41489505767822), Pointf3(15.3729991912842,18.0440559387207,9.41489505767822), Pointf3(15.4579992294312,19.5170555114746,9.41489505767822), Pointf3(15.5469999313354,19.5820541381836,9.41489505767822), Pointf3(13.2309989929199,19.7610549926758,9.41489505767822), Pointf3(13.168999671936,19.7360553741455,9.41489505767822), Pointf3(13.096999168396,19.0140552520752,9.41489505767822), Pointf3(13.1999988555908,18.9870548248291,9.41489505767822), Pointf3(15.1399993896484,19.2080554962158,9.41489505767822), Pointf3(15.0159997940063,19.1600551605225,9.41489505767822), Pointf3(14.9859991073608,18.2770557403564,9.41489505767822), Pointf3(15.1749992370605,18.1690559387207,9.41489505767822), Pointf3(15.9039993286133,19.1320552825928,9.41489505767822), Pointf3(15.8949995040894,19.4460544586182,9.41489505767822), Pointf3(15.8769998550415,19.0420551300049,9.41489505767822), Pointf3(12.2169990539551,20.6500549316406,9.41489505767822), Pointf3(11.9379997253418,20.6810550689697,9.41489505767822), Pointf3(11.8629989624023,19.2130546569824,9.41489505767822), Pointf3(12.096999168396,19.1950550079346,9.41489505767822), Pointf3(14.1669998168945,18.6640548706055,9.41489505767822), Pointf3(14.1039991378784,20.2460556030273,9.41489505767822), Pointf3(13.9849996566772,18.7360553741455,9.41489505767822), Pointf3(14.7349996566772,19.1590557098389,9.41489505767822), Pointf3(14.5849990844727,19.2050552368164,9.41489505767822), Pointf3(14.5719995498657,18.4850559234619,9.41489505767822), Pointf3(14.1939992904663,19.6760559082031,9.41489505767822), Pointf3(14.1849994659424,19.9330558776855,9.41489505767822), Pointf3(14.1759996414185,18.6640548706055,9.41489505767822), Pointf3(14.261999130249,19.4890556335449,9.41489505767822), Pointf3(14.3539991378784,19.3610553741455,9.41489505767822), Pointf3(14.3559989929199,18.5830554962158,9.41489505767822), Pointf3(11.6039991378784,20.1250553131104,9.41489505767822), Pointf3(11.5209999084473,20.0520553588867,9.41489505767822), Pointf3(11.4209995269775,19.2480545043945,9.41489505767822), Pointf3(11.6989994049072,20.2690544128418,9.41389465332031), Pointf3(11.7609996795654,20.4310550689697,9.41489505767822), Pointf3(11.8359994888306,19.2130546569824,9.41489505767822), Pointf3(14.1889991760254,20.1710548400879,9.41489505767822), Pointf3(13.9689998626709,20.2840557098389,9.41489505767822), Pointf3(13.8739995956421,20.315055847168,9.41489505767822), Pointf3(13.7799997329712,18.8080558776855,9.41489505767822), Pointf3(13.9869995117188,20.2750549316406,9.41489505767822), Pointf3(12.3129997253418,20.5980548858643,9.41489505767822), Pointf3(12.3399991989136,20.5090560913086,9.41489505767822), Pointf3(12.3489990234375,20.3830547332764,9.41489505767822), Pointf3(12.3599996566772,20.2680549621582,9.41489505767822), Pointf3(12.3849992752075,20.1850547790527,9.41489505767822), Pointf3(12.3849992752075,20.1670551300049,9.41489505767822), Pointf3(12.4249992370605,20.065055847168,9.41489505767822), Pointf3(12.4729995727539,19.1350555419922,9.41489505767822), Pointf3(14.4399995803833,19.2900543212891,9.41489505767822), Pointf3(14.3649997711182,18.5740547180176,9.41489505767822), Pointf3(13.5729999542236,20.0310554504395,9.41489505767822), Pointf3(13.4889993667603,19.9140548706055,9.41489505767822), Pointf3(13.5639991760254,18.8710556030273,9.41489505767822), Pointf3(13.6389999389648,20.1310558319092,9.41489505767822), Pointf3(13.6719989776611,20.2130546569824,9.41489505767822), Pointf3(13.75,20.3020553588867,9.41489505767822), Pointf3(12.7399997711182,19.7810554504395,9.41489505767822), Pointf3(12.6189994812012,19.8520545959473,9.41489505767822), Pointf3(12.5799999237061,19.1200542449951,9.41489505767822), Pointf3(12.8349990844727,19.069055557251,9.41489505767822), Pointf3(11.2669992446899,19.9350547790527,9.41489505767822), Pointf3(11.1029987335205,19.9230556488037,9.41489505767822), Pointf3(11.0209999084473,19.2600555419922,9.41489505767822), Pointf3(11.3819999694824,19.9710559844971,9.41489505767822), Pointf3(13.418999671936,19.8530559539795,9.41489505767822), Pointf3(13.4329996109009,18.9160556793213,9.41489505767822), Pointf3(11.8399991989136,20.6430549621582,9.41489505767822), Pointf3(13.3119993209839,19.7800559997559,9.41489505767822), Pointf3(15.2189998626709,19.2600555419922,9.41489505767822), Pointf3(15.1839990615845,18.1600551605225,9.41489505767822), Pointf3(15.3639993667603,18.0520553588867,9.41489505767822), Pointf3(13.0189990997314,19.7250556945801,9.41489505767822), Pointf3(12.8949995040894,19.7350559234619,9.41489505767822), Pointf3(15.9039993286133,19.1500549316406,9.41489505767822), Pointf3(15.7699995040894,19.5140552520752,9.41489505767822), Pointf3(15.8589992523193,18.9340553283691,9.41489505767822), Pointf3(14.1939992904663,19.9510555267334,9.41489505767822), Pointf3(14.2119998931885,20.0630550384521,9.41489505767822), Pointf3(14.8589992523193,19.149055480957,9.41489505767822), Pointf3(14.8159999847412,18.3670558929443,9.41489505767822), Pointf3(14.8959999084473,18.3220558166504,9.41489505767822), Pointf3(12.5189990997314,19.9360542297363,9.41489505767822), Pointf3(11.0209999084473,19.9290542602539,9.41489505767822), Pointf3(11.0209999084473,19.2530555725098,2.67489433288574), Pointf3(11.0209999084473,19.9300556182861,2.67489433288574), Pointf3(15.9799995422363,18.505931854248,5.58724021911621), Pointf3(15.9799995422363,18.5044555664062,9.41489505767822), Pointf3(15.9799995422363,18.5041732788086,2.67489433288574), Pointf3(15.9799995422363,18.1684837341309,2.67489433288574), Pointf3(15.9799995422363,18.1288299560547,9.41489505767822), Pointf3(15.9799995422363,17.9876575469971,2.67489433288574), Pointf3(15.9799995422363,17.6247596740723,3.91620373725891), Pointf3(15.9799995422363,17.6247596740723,2.67489433288574), Pointf3(15.9799995422363,17.6254329681396,4.32245063781738), Pointf3(15.9799995422363,17.8920269012451,9.41489505767822), Pointf3(15.9799995422363,17.8795108795166,2.67489433288574), Pointf3(15.9799995422363,17.629810333252,4.58585262298584), Pointf3(15.9799995422363,17.6336059570312,5.27938556671143), Pointf3(15.9799995422363,17.8311748504639,2.67489433288574), Pointf3(15.9799995422363,17.638355255127,9.41489505767822), Pointf3(15.9799995422363,17.6346111297607,5.98653984069824), Pointf3(15.9799995422363,17.8728256225586,2.67489433288574), Pointf3(15.9799995422363,18.2221603393555,2.67489433288574) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(0,3,1), Point3(0,4,3), Point3(5,6,7), Point3(8,6,5), Point3(2,9,0), Point3(6,10,11), Point3(12,13,14), Point3(15,16,17), Point3(18,16,15), Point3(19,20,21), Point3(22,23,24), Point3(25,23,22), Point3(26,23,25), Point3(27,23,26), Point3(28,23,27), Point3(29,30,31), Point3(29,32,30), Point3(29,28,32), Point3(33,28,29), Point3(33,23,28), Point3(21,23,33), Point3(20,23,21), Point3(34,35,36), Point3(37,35,34), Point3(38,35,37), Point3(39,35,38), Point3(40,35,39), Point3(41,35,40), Point3(42,35,41), Point3(43,35,42), Point3(44,35,43), Point3(45,35,44), Point3(46,35,45), Point3(47,35,46), Point3(48,35,47), Point3(49,50,51), Point3(52,48,47), Point3(23,49,24), Point3(53,54,55), Point3(56,57,58), Point3(59,57,56), Point3(60,57,59), Point3(61,57,60), Point3(62,57,61), Point3(63,57,62), Point3(64,57,63), Point3(65,57,64), Point3(66,57,65), Point3(13,57,66), Point3(54,67,55), Point3(68,69,70), Point3(71,69,68), Point3(72,69,71), Point3(73,69,72), Point3(74,69,73), Point3(75,69,74), Point3(76,69,75), Point3(77,69,76), Point3(67,69,77), Point3(70,16,68), Point3(70,17,16), Point3(78,79,80), Point3(81,79,78), Point3(82,79,81), Point3(83,79,82), Point3(84,79,83), Point3(85,79,84), Point3(86,79,85), Point3(87,79,86), Point3(88,8,5), Point3(11,7,6), Point3(11,89,7), Point3(11,9,89), Point3(11,0,9), Point3(55,90,53), Point3(55,79,90), Point3(55,80,79), Point3(91,11,10), Point3(92,69,12), Point3(92,70,69), Point3(34,93,37), Point3(47,94,52), Point3(47,95,94), Point3(47,57,95), Point3(47,58,57), Point3(51,24,49), Point3(21,35,19), Point3(21,36,35), Point3(14,92,12), Point3(86,10,87), Point3(86,91,10), Point3(77,55,67), Point3(66,14,13), Point3(96,97,4), Point3(98,99,100), Point3(101,102,98), Point3(103,101,98), Point3(104,103,98), Point3(105,106,107), Point3(108,105,107), Point3(109,108,107), Point3(100,109,107), Point3(110,111,112), Point3(113,110,112), Point3(114,115,116), Point3(117,114,116), Point3(118,119,120), Point3(121,122,123), Point3(124,121,123), Point3(125,126,127), Point3(128,129,130), Point3(131,132,133), Point3(71,131,133), Point3(134,71,133), Point3(135,134,133), Point3(136,135,133), Point3(137,138,139), Point3(140,137,139), Point3(141,140,139), Point3(142,31,141), Point3(142,141,139), Point3(143,126,132), Point3(144,145,146), Point3(147,144,146), Point3(127,147,146), Point3(148,121,124), Point3(149,148,124), Point3(150,149,124), Point3(151,150,124), Point3(152,151,124), Point3(153,152,124), Point3(154,153,124), Point3(155,154,124), Point3(129,156,157), Point3(130,129,157), Point3(158,159,160), Point3(161,158,160), Point3(162,161,160), Point3(163,162,160), Point3(146,163,160), Point3(164,165,166), Point3(167,164,166), Point3(168,169,170), Point3(171,168,170), Point3(139,171,170), Point3(159,172,173), Point3(123,174,142), Point3(175,110,113), Point3(173,175,113), Point3(106,176,177), Point3(178,106,177), Point3(179,180,167), Point3(112,179,167), Point3(175,173,172), Point3(119,118,181), Point3(119,181,97), Point3(119,97,96), Point3(182,98,102), Point3(182,102,183), Point3(182,183,120), Point3(182,120,119), Point3(143,132,184), Point3(184,185,143), Point3(147,127,126), Point3(174,123,122), Point3(159,173,160), Point3(126,125,133), Point3(126,133,132), Point3(186,187,188), Point3(186,188,116), Point3(186,116,115), Point3(99,98,182), Point3(109,100,99), Point3(106,178,107), Point3(114,117,177), Point3(114,177,176), Point3(128,130,187), Point3(128,187,186), Point3(135,136,157), Point3(135,157,156), Point3(163,146,145), Point3(164,167,180), Point3(179,112,111), Point3(171,139,138), Point3(189,155,166), Point3(189,166,165), Point3(149,150,93), Point3(154,155,189), Point3(31,142,174), Point3(114,176,78), Point3(81,78,176), Point3(7,89,183), Point3(89,9,120), Point3(89,120,183), Point3(78,80,114), Point3(176,106,81), Point3(88,5,103), Point3(183,102,7), Point3(118,120,9), Point3(9,2,181), Point3(9,181,118), Point3(115,114,80), Point3(82,81,106), Point3(101,103,5), Point3(102,101,5), Point3(5,7,102), Point3(97,181,2), Point3(2,1,97), Point3(1,3,97), Point3(80,55,115), Point3(172,159,59), Point3(59,56,172), Point3(3,4,97), Point3(4,0,96), Point3(105,108,82), Point3(186,115,55), Point3(82,106,105), Point3(83,82,108), Point3(60,59,159), Point3(175,172,56), Point3(119,96,0), Point3(0,11,119), Point3(108,109,84), Point3(84,83,108), Point3(55,77,186), Point3(56,58,110), Point3(56,110,175), Point3(60,159,158), Point3(11,91,182), Point3(182,119,11), Point3(91,86,182), Point3(85,84,109), Point3(86,85,99), Point3(128,186,77), Point3(58,111,110), Point3(158,161,60), Point3(26,25,137), Point3(138,137,25), Point3(99,182,86), Point3(109,99,85), Point3(77,76,128), Point3(58,47,111), Point3(61,60,161), Point3(137,140,26), Point3(27,26,140), Point3(25,22,138), Point3(129,128,76), Point3(76,75,129), Point3(75,74,129), Point3(74,73,156), Point3(73,72,135), Point3(68,16,184), Point3(68,184,132), Point3(16,18,185), Point3(161,162,62), Point3(62,61,161), Point3(179,111,47), Point3(171,138,22), Point3(156,129,74), Point3(135,156,73), Point3(134,135,72), Point3(72,71,134), Point3(68,132,131), Point3(185,184,16), Point3(18,15,185), Point3(63,62,162), Point3(28,27,140), Point3(22,24,171), Point3(71,68,131), Point3(15,17,143), Point3(15,143,185), Point3(17,70,143), Point3(70,92,126), Point3(162,163,64), Point3(64,63,162), Point3(180,179,47), Point3(47,46,180), Point3(140,141,28), Point3(168,171,24), Point3(126,143,70), Point3(92,14,147), Point3(147,126,92), Point3(14,66,144), Point3(14,144,147), Point3(65,64,163), Point3(66,65,145), Point3(46,45,180), Point3(32,28,141), Point3(24,51,168), Point3(145,144,66), Point3(163,145,65), Point3(164,180,45), Point3(45,44,164), Point3(44,43,164), Point3(43,42,165), Point3(38,37,151), Point3(150,151,37), Point3(37,93,150), Point3(141,31,30), Point3(30,32,141), Point3(169,168,51), Point3(165,164,43), Point3(189,165,42), Point3(42,41,189), Point3(40,39,152), Point3(40,152,153), Point3(151,152,39), Point3(39,38,151), Point3(93,34,149), Point3(154,189,41), Point3(153,154,41), Point3(41,40,153), Point3(148,149,34), Point3(34,36,148), Point3(36,21,121), Point3(31,174,29), Point3(121,148,36), Point3(21,33,122), Point3(21,122,121), Point3(33,29,122), Point3(174,122,29), Point3(116,188,53), Point3(104,98,10), Point3(87,10,98), Point3(98,100,87), Point3(79,87,100), Point3(79,100,107), Point3(90,79,107), Point3(90,107,178), Point3(178,177,90), Point3(53,90,177), Point3(53,177,117), Point3(117,116,53), Point3(54,53,188), Point3(54,188,187), Point3(67,54,187), Point3(67,187,130), Point3(69,67,130), Point3(69,130,157), Point3(12,69,157), Point3(12,157,136), Point3(136,133,12), Point3(12,133,125), Point3(125,127,12), Point3(13,12,127), Point3(127,146,13), Point3(57,13,146), Point3(57,146,160), Point3(95,57,160), Point3(95,160,173), Point3(173,113,95), Point3(94,95,113), Point3(113,112,94), Point3(52,94,112), Point3(48,52,112), Point3(112,167,48), Point3(35,48,167), Point3(35,167,166), Point3(19,35,166), Point3(139,170,50), Point3(50,49,139), Point3(166,155,19), Point3(20,19,155), Point3(155,124,20), Point3(23,20,124), Point3(23,124,123), Point3(49,23,123), Point3(49,123,142), Point3(142,139,49), Point3(190,191,170), Point3(192,191,190), Point3(191,192,51), Point3(191,51,50), Point3(170,169,190), Point3(169,51,192), Point3(169,192,190), Point3(170,191,50), Point3(193,194,195), Point3(196,197,198), Point3(199,200,201), Point3(198,202,203), Point3(204,201,200), Point3(205,204,200), Point3(206,207,208), Point3(206,208,205), Point3(206,205,200), Point3(207,206,209), Point3(207,209,203), Point3(207,203,202), Point3(202,198,197), Point3(197,196,210), Point3(197,210,195), Point3(197,195,194), Point3(8,88,195), Point3(8,195,210), Point3(210,196,8), Point3(196,198,8), Point3(198,203,8), Point3(203,209,8), Point3(209,206,8), Point3(206,200,8), Point3(202,197,104), Point3(207,202,104), Point3(103,104,197), Point3(103,197,194), Point3(193,195,88), Point3(88,103,194), Point3(88,194,193), Point3(200,199,8), Point3(199,201,8), Point3(204,205,6), Point3(6,8,201), Point3(6,201,204), Point3(10,6,205), Point3(10,205,208), Point3(104,10,208), Point3(104,208,207) 
            });
            break;
        case TestMesh::pyramid:
            vertices = std::vector<Pointf3>({
                Pointf3(10,10,40), Pointf3(0,0,0), Pointf3(20,0,0), Pointf3(20,20,0), Pointf3(0,20,0)
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(0,3,4), Point3(3,1,4), Point3(1,3,2), Point3(3,0,2), Point3(4,1,0)
            });
            break;
        case TestMesh::two_hollow_squares:
            vertices = std::vector<Pointf3>({
                Pointf3(66.7133483886719,104.286666870117,0), Pointf3(66.7133483886719,95.7133331298828,0), Pointf3(65.6666870117188,94.6666717529297,0), Pointf3(75.2866821289062,95.7133331298828,0), Pointf3(76.3333435058594,105.333335876465,0), Pointf3(76.3333435058594,94.6666717529297,0), Pointf3(65.6666870117188,105.33332824707,0), Pointf3(75.2866821289062,104.286666870117,0), Pointf3(71.1066818237305,104.58666229248,2.79999995231628), Pointf3(66.4133529663086,104.58666229248,2.79999995231628), Pointf3(75.5866851806641,104.58666229248,2.79999995231628), Pointf3(66.4133529663086,99.8933334350586,2.79999995231628), Pointf3(66.4133529663086,95.4133377075195,2.79999995231628), Pointf3(71.1066818237305,95.4133377075195,2.79999995231628), Pointf3(75.5866851806641,95.4133377075195,2.79999995231628), Pointf3(75.5866851806641,100.106666564941,2.79999995231628), Pointf3(74.5400161743164,103.540000915527,2.79999995231628), Pointf3(70.0320129394531,103.540000915527,2.79999995231628), Pointf3(67.4600067138672,103.540000915527,2.79999995231628), Pointf3(67.4600067138672,100.968002319336,2.79999995231628), Pointf3(67.4600067138672,96.4599990844727,2.79999995231628), Pointf3(74.5400161743164,99.0319976806641,2.79999995231628), Pointf3(74.5400161743164,96.4599990844727,2.79999995231628), Pointf3(70.0320129394531,96.4599990844727,2.79999995231628), Pointf3(123.666717529297,94.6666717529297,0), Pointf3(134.333312988281,94.6666717529297,0), Pointf3(124.413360595703,95.4133377075195,2.79999995231628), Pointf3(129.106674194336,95.4133377075195,2.79999995231628), Pointf3(133.586669921875,95.4133377075195,2.79999995231628), Pointf3(123.666717529297,105.33332824707,0), Pointf3(124.413360595703,104.58666229248,2.79999995231628), Pointf3(124.413360595703,99.8933334350586,2.79999995231628), Pointf3(134.333312988281,105.33332824707,0), Pointf3(129.106674194336,104.58666229248,2.79999995231628), Pointf3(133.586669921875,104.58666229248,2.79999995231628), Pointf3(133.586669921875,100.106666564941,2.79999995231628), Pointf3(124.713317871094,104.286666870117,0), Pointf3(124.713317871094,95.7133331298828,0), Pointf3(133.286712646484,95.7133331298828,0), Pointf3(133.286712646484,104.286666870117,0), Pointf3(132.540023803711,103.540000915527,2.79999995231628), Pointf3(128.032028198242,103.540008544922,2.79999995231628), Pointf3(125.460006713867,103.540000915527,2.79999995231628), Pointf3(125.460006713867,100.968002319336,2.79999995231628), Pointf3(125.460006713867,96.4599990844727,2.79999995231628), Pointf3(132.540023803711,99.0319976806641,2.79999995231628), Pointf3(132.540023803711,96.4599990844727,2.79999995231628), Pointf3(128.032028198242,96.4599990844727,2.79999995231628)
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(3,4,5), Point3(6,4,0), Point3(6,0,2), Point3(2,1,5), Point3(7,4,3), Point3(1,3,5), Point3(0,4,7), Point3(4,6,8), Point3(6,9,8), Point3(4,8,10), Point3(6,2,9), Point3(2,11,9), Point3(2,12,11), Point3(2,5,12), Point3(5,13,12), Point3(5,14,13), Point3(4,10,15), Point3(5,4,14), Point3(4,15,14), Point3(7,16,17), Point3(0,7,18), Point3(7,17,18), Point3(1,19,20), Point3(1,0,19), Point3(0,18,19), Point3(7,3,21), Point3(3,22,21), Point3(7,21,16), Point3(3,23,22), Point3(3,1,23), Point3(1,20,23), Point3(24,25,26), Point3(25,27,26), Point3(25,28,27), Point3(29,24,30), Point3(24,31,30), Point3(24,26,31), Point3(32,29,33), Point3(29,30,33), Point3(32,33,34), Point3(32,34,35), Point3(25,32,28), Point3(32,35,28), Point3(36,37,24), Point3(38,32,25), Point3(29,32,36), Point3(29,36,24), Point3(24,37,25), Point3(39,32,38), Point3(37,38,25), Point3(36,32,39), Point3(39,40,41), Point3(36,39,42), Point3(39,41,42), Point3(37,43,44), Point3(37,36,43), Point3(36,42,43), Point3(39,38,45), Point3(38,46,45), Point3(39,45,40), Point3(38,47,46), Point3(38,37,47), Point3(37,44,47), Point3(16,8,9), Point3(16,10,8), Point3(10,16,15), Point3(15,16,21), Point3(22,15,21), Point3(15,22,14), Point3(22,23,14), Point3(23,20,14), Point3(17,16,9), Point3(18,17,9), Point3(19,18,9), Point3(19,9,11), Point3(19,11,20), Point3(13,14,20), Point3(20,11,12), Point3(13,20,12), Point3(41,40,30), Point3(42,41,30), Point3(43,42,30), Point3(43,30,31), Point3(43,31,44), Point3(27,28,44), Point3(44,31,26), Point3(27,44,26), Point3(40,33,30), Point3(40,34,33), Point3(34,40,35), Point3(35,40,45), Point3(46,35,45), Point3(35,46,28), Point3(46,47,28), Point3(47,44,28) ,
            });
            break;
        case TestMesh::small_dorito:
            vertices = std::vector<Pointf3>({
                Pointf3(6.00058937072754,-22.9982089996338,0), Pointf3(22.0010242462158,-49.9998741149902,0), Pointf3(-9.99957847595215,-49.999870300293,0), Pointf3(6.00071382522583,-32.2371635437012,28.0019245147705), Pointf3(11.1670551300049,-37.9727020263672,18.9601669311523), Pointf3(6.00060224533081,-26.5392456054688,10.7321853637695) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(3,4,5), Point3(2,1,4), Point3(2,4,3), Point3(2,3,5), Point3(2,5,0), Point3(5,4,1), Point3(5,1,0) 
            });
            break;
        case TestMesh::bridge:
            vertices = std::vector<Pointf3>({
                Pointf3(75,84.5,8), Pointf3(125,84.5,8), Pointf3(75,94.5,8), Pointf3(120,84.5,5), Pointf3(125,94.5,8), Pointf3(75,84.5,0), Pointf3(80,84.5,5), Pointf3(125,84.5,0), Pointf3(125,94.5,0), Pointf3(80,94.5,5), Pointf3(75,94.5,0), Pointf3(120,94.5,5), Pointf3(120,84.5,0), Pointf3(80,94.5,0), Pointf3(80,84.5,0), Pointf3(120,94.5,0) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(1,0,3), Point3(2,1,4), Point3(2,5,0), Point3(0,6,3), Point3(1,3,7), Point3(1,8,4), Point3(4,9,2), Point3(10,5,2), Point3(5,6,0), Point3(6,11,3), Point3(3,12,7), Point3(7,8,1), Point3(4,8,11), Point3(4,11,9), Point3(9,10,2), Point3(10,13,5), Point3(14,6,5), Point3(9,11,6), Point3(11,12,3), Point3(12,8,7), Point3(11,8,15), Point3(13,10,9), Point3(5,13,14), Point3(14,13,6), Point3(6,13,9), Point3(15,12,11), Point3(15,8,12) 
            });
            break;
        case TestMesh::bridge_with_hole:
            vertices = std::vector<Pointf3>({
                Pointf3(75,69.5,8), Pointf3(80,76.9091644287109,8), Pointf3(75,94.5,8), Pointf3(125,69.5,8), Pointf3(120,76.9091644287109,8), Pointf3(120,87.0908355712891,8), Pointf3(80,87.0908355712891,8), Pointf3(125,94.5,8), Pointf3(80,87.0908355712891,5), Pointf3(120,87.0908355712891,5), Pointf3(125,94.5,0), Pointf3(120,69.5,0), Pointf3(120,94.5,0), Pointf3(125,69.5,0), Pointf3(120,94.5,5), Pointf3(80,94.5,5), Pointf3(80,94.5,0), Pointf3(75,94.5,0), Pointf3(80,69.5,5), Pointf3(80,69.5,0), Pointf3(80,76.9091644287109,5), Pointf3(120,69.5,5), Pointf3(75,69.5,0), Pointf3(120,76.9091644287109,5) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(1,0,3), Point3(1,3,4), Point3(4,3,5), Point3(2,6,7), Point3(6,2,1), Point3(7,6,5), Point3(7,5,3), Point3(5,8,9), Point3(8,5,6), Point3(10,11,12), Point3(11,10,13), Point3(14,8,15), Point3(8,14,9), Point3(2,16,17), Point3(16,2,15), Point3(15,2,14), Point3(14,10,12), Point3(10,14,7), Point3(7,14,2), Point3(16,18,19), Point3(18,16,20), Point3(20,16,1), Point3(1,16,8), Point3(8,16,15), Point3(6,1,8), Point3(3,11,13), Point3(11,3,21), Point3(21,3,18), Point3(18,22,19), Point3(22,18,0), Point3(0,18,3), Point3(16,22,17), Point3(22,16,19), Point3(2,22,0), Point3(22,2,17), Point3(5,23,4), Point3(23,11,21), Point3(11,23,12), Point3(12,23,9), Point3(9,23,5), Point3(12,9,14), Point3(23,18,20), Point3(18,23,21), Point3(10,3,13), Point3(3,10,7), Point3(1,23,20), Point3(23,1,4) 
            });
            break;
        case TestMesh::step:
            vertices = std::vector<Pointf3>({
                Pointf3(0,20,5), Pointf3(0,20,0), Pointf3(0,0,5), Pointf3(0,0,0), Pointf3(20,0,0), Pointf3(20,0,5), Pointf3(1,19,5), Pointf3(1,1,5), Pointf3(19,1,5), Pointf3(20,20,5), Pointf3(19,19,5), Pointf3(20,20,0), Pointf3(19,19,10), Pointf3(1,19,10), Pointf3(1,1,10), Pointf3(19,1,10) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2), Point3(1,3,2), Point3(3,4,5), Point3(2,3,5), Point3(6,0,2), Point3(6,2,7), Point3(5,8,7), Point3(5,7,2), Point3(9,10,8), Point3(9,8,5), Point3(9,0,6), Point3(9,6,10), Point3(9,11,1), Point3(9,1,0), Point3(3,1,11), Point3(4,3,11), Point3(5,11,9), Point3(5,4,11), Point3(12,10,6), Point3(12,6,13), Point3(6,7,14), Point3(13,6,14), Point3(7,8,15), Point3(14,7,15), Point3(15,8,10), Point3(15,10,12), Point3(12,13,14), Point3(12,14,15) 
            });
            break;
        case TestMesh::slopy_cube:
            vertices = std::vector<Pointf3>({
                Pointf3(-10,-10,0) , Pointf3(-10,-10,20) , Pointf3(-10,10,0) , Pointf3(-10,10,20) , Pointf3(0,-10,10) , Pointf3(10,-10,0) , Pointf3(2.92893,-10,10) , Pointf3(10,-10,2.92893) , 
                Pointf3(0,-10,20) , Pointf3(10,10,0) , Pointf3(0,10,10) , Pointf3(0,10,20) , Pointf3(2.92893,10,10) , Pointf3(10,10,2.92893) 
            });
            facets = std::vector<Point3>({
                Point3(0,1,2) , Point3(2,1,3) , Point3(4,0,5) , Point3(4,1,0) , Point3(6,4,7) , Point3(7,4,5) , Point3(4,8,1) , Point3(0,2,5) , Point3(5,2,9) , Point3(2,10,9) , Point3(10,3,11) , 
                Point3(2,3,10) , Point3(9,10,12) , Point3(13,9,12) , Point3(3,1,8) , Point3(11,3,8) , Point3(10,11,8) , Point3(4,10,8) , Point3(6,12,10) , Point3(4,6,10) , 
                Point3(7,13,12) , Point3(6,7,12) , Point3(7,5,9) , Point3(13,7,9) 
            });
            break;        
        default:
            break;        
    }
    TriangleMesh _mesh;
    if (vertices.size() == 0) {
        switch(m) {
            case TestMesh::cube_20x20x20:
                _mesh = Slic3r::TriangleMesh::make_cube(20,20,20);
                break;
            case TestMesh::sphere_50mm:
                _mesh = Slic3r::TriangleMesh::make_sphere(50, PI / 243.0);
                break;
            default:
                break;
        }
    } else {
        _mesh = TriangleMesh(vertices, facets);
    }

    _mesh.repair();
    return _mesh;
}


shared_Print init_print(std::initializer_list<TestMesh> meshes, Slic3r::Model& model, config_ptr _config, bool comments) {
    auto config {Slic3r::Config::new_from_defaults()};
    config->apply(_config);

    const char* v {std::getenv("SLIC3R_TESTS_GCODE")};
    auto tests_gcode {(v == nullptr ? ""s : std::string(v))};

    if (tests_gcode != ""s)
        config->set("gcode_comments", 1);

    shared_Print print {std::make_shared<Slic3r::Print>()};
    print->apply_config(config);
    for (auto t : meshes) {
        auto* object {model.add_object()};
        object->name += std::string(mesh_names.at(t)) + ".stl"s;
        object->add_volume(mesh(t));

        auto* inst {object->add_instance()};
        inst->rotation = 0;
        inst->scaling_factor = 1.0;
    }

    model.arrange_objects(print->config.min_object_distance());
    model.center_instances_around_point(Slic3r::Pointf(100,100));
    for (auto* mo : model.objects) {
        print->auto_assign_extruders(mo);
        print->add_model_object(mo);
    }

    print->validate();

    return print;
}
shared_Print init_print(std::initializer_list<TriangleMesh> meshes, Slic3r::Model& model, config_ptr _config, bool comments) {
    auto config {Slic3r::Config::new_from_defaults()};
    config->apply(_config);

    const char* v {std::getenv("SLIC3R_TESTS_GCODE")};
    auto tests_gcode {(v == nullptr ? ""s : std::string(v))};

    if (tests_gcode != ""s)
        config->set("gcode_comments", 1);

    shared_Print print {std::make_shared<Slic3r::Print>()};
    print->apply_config(config);
    for (auto& t : meshes) {
        auto* object {model.add_object()};
        object->name += "object.stl"s;
        object->add_volume(t);

        auto* inst {object->add_instance()};
        inst->rotation = 0;
        inst->scaling_factor = 1.0;
    }

    model.arrange_objects(print->config.min_object_distance());
    model.center_instances_around_point(Slic3r::Pointf(100,100));
    for (auto* mo : model.objects) {
        print->auto_assign_extruders(mo);
        print->add_model_object(mo);
    }

    print->validate();

    return print;
}

void gcode(std::stringstream& gcode, shared_Print _print) {
    _print->export_gcode(gcode, true);
}

Slic3r::Model model(const std::string& model_name, TriangleMesh&& _mesh) {
    Slic3r::Model result;

    auto* object {result.add_object()};
    object->name += model_name + ".stl"s;
    object->add_volume(_mesh);

    auto* inst {object->add_instance()};
    inst->rotation = 0;
    inst->scaling_factor = 1.0;

    return result;
}
void add_testmesh_to_model(Slic3r::Model& result, const std::string& model_name, TriangleMesh&& _mesh) {
    auto* object {result.add_object()};
    object->name += model_name + ".stl"s;
    object->add_volume(_mesh);

    auto* inst {object->add_instance()};
    inst->rotation = 0;
    inst->scaling_factor = 1.0;
}

} } // namespace Slic3r::Test
