/*
  spamdyke -- a filter for stopping spam at connection time.
  Copyright (C) 2014 Sam Clippinger (samc (at) silence (dot) org)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config-qrv.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#include <pwd.h>
#include <errno.h>
#include "spamdyke-qrv.h"
#include "validate-qrv.h"
#include "log-qrv.h"
#include "fs-qrv.h"
#include "cdb-qrv.h"
#include "exec-qrv.h"

/*
 * RETURN:
 *   DECISION_ERROR: error occurred
 *   DECISION_UNKNOWN: no decision reached -- is this an error?
 *   DECISION_VALID: address is valid
 *   DECISION_INVALID: address is invalid
 *   DECISION_UNAVAILABLE: address is "unavailable"
 */
int validate(struct qrv_settings *current_settings, char *target_recipient_username, int strlen_recipient_username, char *target_recipient_domain, int forwarded)
  {
  int return_value;
  int i;
  int j;
  int k;
  int continue_processing;
  int current_step;
  char tmp_username[MAX_ADDRESS + 1];
  int tmp_strlen_username;
  char tmp_domain[MAX_ADDRESS + 1];
  int tmp_strlen_domain;
  int found_match;
  int tmp_loc;
  int num_loop;
  char tmp_line[MAX_FILE_BUF + 1];
  int tmp_strlen;
  char qmail_prefix[MAX_PATH + 1];
  int strlen_qmail_prefix;
  char qmail_home[MAX_PATH + 1];
  char qmail_dash[MAX_PATH + 1];
  char qmail_ext[MAX_PATH + 1];
  int strlen_qmail_ext;
  char tmp_filename[MAX_PATH + 1];
  char tmp_name[MAX_ADDRESS + 1];
  int strlen_tmp_name;
  char tmp_path[MAX_PATH + 1];
  char tmp_unreal_path[MAX_PATH + 1];
  struct stat tmp_stat;
  int tmp_return;
  int max_line;
  struct passwd *tmp_passwd;
  int forward_only;
  char tmp_qmail_filename[MAX_PATH + 1];
  char tmp_qmail_path[MAX_PATH + 1];
  char **qmail_lines;
  int current_line;
  int num_lines;
  int last_line;
  int total_lines;
  char *tmp_file_line;
  int strlen_found_key;
  char tmp_address[MAX_ADDRESS + 1];
  int tmp_uid;
  int tmp_gid;
  int strlen_recipient_domain;

#ifdef WITH_VPOPMAIL_SUPPORT

  int tmp_strlen_command;
  char *tmp_argv[4];
  char vpopmail_output[MAX_BUF + 1];
  int tmp_strlen_output;
  int tmp_status;

#endif /* WITH_VPOPMAIL_SUPPORT */

  return_value = DECISION_UNKNOWN;
  continue_processing = 1;
  num_loop = 0;
  i = 0;
  current_line = 0;

  snprintf(tmp_username, MAX_ADDRESS, "%.*s%n", strlen_recipient_username, target_recipient_username, &tmp_strlen_username);
  snprintf(tmp_domain, MAX_ADDRESS, "%s%n", target_recipient_domain, &tmp_strlen_domain);
  strlen_recipient_domain = tmp_strlen_domain;

  if (forwarded)
    current_step = 2;
  else
    {
    current_step = 7;

    if ((current_settings->relayclient != NULL) &&
        (current_settings->relayclient[0] != '\0'))
      {
      if (current_settings->recipient_domain[0] != '\0')
        snprintf(tmp_domain, MAX_ADDRESS, "%s%s%n", target_recipient_domain, current_settings->relayclient, &tmp_strlen_domain);
      else
        snprintf(tmp_username, MAX_ADDRESS, "%s%s%n", target_recipient_username, current_settings->relayclient, &tmp_strlen_username);
      }
    }

  tmp_loc = 0;
  max_line = -1;
  forward_only = 0;
  tmp_line[0] = '\0';
  tmp_strlen = 0;
  qmail_prefix[0] = '\0';
  strlen_qmail_prefix = 0;
  qmail_home[0] = '\0';
  qmail_dash[0] = '\0';
  qmail_ext[0] = '\0';
  strlen_qmail_ext = 0;
  tmp_filename[0] = '\0';
  tmp_name[0] = '\0';
  strlen_tmp_name = 0;
  last_line = 0;
  qmail_lines = NULL;
  tmp_file_line = NULL;
  total_lines = 0;
  num_lines = 0;
  strlen_found_key = 0;
  tmp_uid = -1;
  tmp_gid = -1;

  /*
   * This process is implemented as a switch statement because doing it all with
   * if/else statements was just too hairy and goto is considered harmful, right?
   *
   * This doesn't seem to make a lot of sense by itself, but it all becomes much
   * clearer when compared to the recipient validation flowchart.  The step
   * numbers correspond to the labels in the flowchart.
   */
  if (forwarded < MAX_VALIDATE_DEPTH)
    {
    while (continue_processing)
      {
      QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_STEP, current_step, tmp_strlen_username, tmp_username, tmp_domain, tmp_name, tmp_filename, tmp_path, qmail_dash, qmail_ext);

      num_loop++;
      if (num_loop > 1000)
        {
        QRV_LOG_ERROR(current_settings, LOG_ERROR_VALIDATE_LOOP, NULL);
        break;
        }

      switch (current_step)
        {
        case 2:
          if (forwarded)
            current_step = (tmp_strlen_domain > 0) ? 7 : 3;
          else
            {
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }

          break;
        case 3:
          if (current_settings->qmail_envnoathost_file != NULL)
            if ((tmp_return = check_path_perms(current_settings, current_settings->qmail_envnoathost_file, S_IFREG, FILE_PERMISSION_READ, NULL, -1, -1)) == 1)
              {
              if ((tmp_strlen = read_file_first_line(current_settings, current_settings->qmail_envnoathost_file, &tmp_file_line)) > 0)
                {
                tmp_strlen_domain = MINVAL(MAX_ADDRESS, tmp_strlen);
                memcpy(tmp_domain, tmp_file_line, tmp_strlen_domain);
                tmp_domain[tmp_strlen_domain] = '\0';

                current_step = 7;
                }
              else if (tmp_strlen == 0)
                current_step = 4;
              else
                {
                return_value = DECISION_ERROR;
                continue_processing = 0;
                }

              if (tmp_file_line != NULL)
                {
                free(tmp_file_line);
                tmp_file_line = NULL;
                }
              }
            else if (tmp_return == 0)
              current_step = 4;
            else
              {
              return_value = DECISION_ERROR;
              continue_processing = 0;
              }
          else
            current_step = 4;

          break;
        case 4:
          if (current_settings->qmail_me_file != NULL)
            if ((tmp_return = check_path_perms(current_settings, current_settings->qmail_me_file, S_IFREG, FILE_PERMISSION_READ, NULL, -1, -1)) == 1)
              {
              if ((tmp_strlen = read_file_first_line(current_settings, current_settings->qmail_me_file, &tmp_file_line)) > 0)
                {
                tmp_strlen_domain = MINVAL(MAX_ADDRESS, tmp_strlen);
                memcpy(tmp_domain, tmp_file_line, tmp_strlen_domain);
                tmp_domain[tmp_strlen_domain] = '\0';

                current_step = 7;
                }
              else if (tmp_strlen == 0)
                {
                QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
                return_value = DECISION_INVALID;
                continue_processing = 0;
                }
              else
                {
                return_value = DECISION_ERROR;
                continue_processing = 0;
                }

              if (tmp_file_line != NULL)
                {
                free(tmp_file_line);
                tmp_file_line = NULL;
                }
              }
            else if (tmp_return == 0)
              {
              QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
              return_value = DECISION_INVALID;
              continue_processing = 0;
              }
            else
              {
              return_value = DECISION_ERROR;
              continue_processing = 0;
              }
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_INVALID;
            continue_processing = 0;
            }

          break;
        case 7:
          found_match = 0;
          for (tmp_loc = (tmp_strlen_username - 1); tmp_loc >= 0; tmp_loc--)
            if (tmp_username[tmp_loc] == QMAIL_PERCENTHACK_TARGET)
              {
              QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_PERCENT_FOUND, tmp_username);
              found_match = 1;
              break;
              }

          current_step = (found_match) ? 8 : 9;
          break;
        case 8:
          found_match = 0;
          if ((tmp_loc >= 0) &&
              (tmp_loc < tmp_strlen_username) &&
              (current_settings->qmail_percenthack_file != NULL))
            for (i = 0; current_settings->qmail_percenthack_file[i] != NULL; i++)
              {
              if ((tmp_return = search_file(current_settings, current_settings->qmail_percenthack_file[i], tmp_domain, tmp_strlen_domain, '\0', NULL, '\0', NULL)) > 0)
                {
                tmp_strlen_domain = (tmp_strlen_username - tmp_loc) - 1;
                memcpy(tmp_domain, tmp_username + tmp_loc + 1, tmp_strlen_domain);
                tmp_domain[tmp_strlen_domain] = '\0';

                tmp_username[tmp_loc] = '\0';
                tmp_strlen_username = tmp_loc;

                QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_PERCENTHACK_FOUND, current_settings->qmail_percenthack_file[i], tmp_username, tmp_domain);
                found_match = 1;
                break;
                }
              else if (tmp_return == -1)
                {
                continue_processing = 0;
                break;
                }
              }

          current_step = (found_match) ? 7 : 9;
          break;
        case 9:
          found_match = 0;
          if (current_settings->qmail_locals_file != NULL)
            for (i = 0; current_settings->qmail_locals_file[i] != NULL; i++)
              {
              if ((tmp_return = search_file(current_settings, current_settings->qmail_locals_file[i], tmp_domain, tmp_strlen_domain, '\0', NULL, '\0', NULL)) > 0)
                {
                QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_LOCALS_FILE, current_settings->qmail_locals_file[i], tmp_domain);
                found_match = 1;
                break;
                }
              else if (tmp_return == -1)
                {
                continue_processing = 0;
                break;
                }
              }

          current_step = (found_match) ? 11 : 10;
          break;
        case 10:
          found_match = 0;
          if (current_settings->qmail_virtualdomains_file != NULL)
            for (i = 0; current_settings->qmail_virtualdomains_file[i] != NULL; i++)
              {
              tmp_strlen = MAX_FILE_BUF;
              if ((tmp_return = search_virtualdomains_file(current_settings, current_settings->qmail_virtualdomains_file[i], tmp_domain, tmp_strlen_domain, tmp_line, &tmp_strlen)) > 0)
                {
                memcpy(tmp_address, tmp_username, tmp_strlen_username);
                tmp_address[tmp_strlen_username] = '\0';
                snprintf(tmp_username, MAX_ADDRESS, "%s-%s%n", tmp_line, tmp_address, &tmp_strlen_username);

                QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_VIRTUALDOMAIN, tmp_domain, current_settings->qmail_virtualdomains_file[i], tmp_line, tmp_username);
                found_match = 1;
                break;
                }
              else if (tmp_return == -1)
                {
                continue_processing = 0;
                break;
                }
              else
                QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_VIRTUALDOMAIN_NONE, tmp_domain, current_settings->qmail_virtualdomains_file[i]);
              }

          if (found_match)
            current_step = 11;
          else
            continue_processing = 0;

          break;
        case 11:
          found_match = 0;
          tmp_line[0] = '\0';
          tmp_strlen = 0;
          if (current_settings->qmail_assign_cdb != NULL)
            for (i = 0; current_settings->qmail_assign_cdb[i] != NULL; i++)
              if ((tmp_strlen = search_assign_cdb(current_settings, tmp_line, MAX_FILE_BUF, current_settings->qmail_assign_cdb[i], tmp_username, tmp_strlen_username, &strlen_found_key)) > 0)
                {
                QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_ASSIGN, tmp_username, current_settings->qmail_assign_cdb[i], tmp_line);
                found_match = 1;
                break;
                }

          if (found_match)
            {
            j = 0;
            k = 0;
            qmail_home[0] = '\0';
            qmail_dash[0] = '\0';
            qmail_ext[0] = '\0';
            strlen_qmail_ext = 0;

            strlen_qmail_prefix = (strlen_found_key >= 0) ? MINVAL(MAX_PATH, strlen_found_key) : MINVAL(MAX_PATH, tmp_strlen_username);
            memcpy(qmail_prefix, tmp_username, strlen_qmail_prefix);
            qmail_prefix[strlen_qmail_prefix] = '\0';

            for (i = 1; i < tmp_strlen; i++)
              if (tmp_line[i] == QMAIL_ASSIGN_DELIMITER)
                {
                j++;
                k = 0;
                }
              else if (j == 1)
                {
                if ((i > 0) &&
                    (tmp_line[i - 1] == QMAIL_ASSIGN_DELIMITER) &&
                    ((sscanf(tmp_line + i, "%d", &tmp_uid) != 1) ||
                     (tmp_uid < 0)))
                  tmp_uid = -1;
                }
              else if (j == 2)
                {
                if ((i > 0) &&
                    (tmp_line[i - 1] == QMAIL_ASSIGN_DELIMITER) &&
                    ((sscanf(tmp_line + i, "%d", &tmp_gid) != 1) ||
                     (tmp_gid < 0)))
                  tmp_gid = -1;
                }
              else if (j == 3)
                {
                if (k < MAX_PATH)
                  {
                  qmail_home[k++] = tmp_line[i];
                  qmail_home[k] = '\0';
                  }
                }
              else if (j == 4)
                {
                if (k < MAX_PATH)
                  {
                  qmail_dash[k++] = tmp_line[i];
                  qmail_dash[k] = '\0';
                  }
                }
              else if (j == 5)
                {
                if (strlen_qmail_ext < MAX_PATH)
                  {
                  qmail_ext[strlen_qmail_ext++] = (tmp_line[i] == QMAIL_REPLACE_EXT_TARGET) ? QMAIL_REPLACE_EXT_REPLACEMENT : tolower((int)tmp_line[i]);
                  qmail_ext[strlen_qmail_ext] = '\0';
                  }
                }

            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_ASSIGN_VALUES, qmail_prefix, qmail_home, qmail_dash, qmail_ext);

            if (strlen_found_key == -1)
              {
              snprintf(tmp_filename, MAX_PATH, ".qmail%s%s", qmail_dash, qmail_ext);
              current_step = 13;
              }
            else
              current_step = 12;
            }
          else
            current_step = 12;

          break;
        case 12:
          if ((tmp_strlen > 0) &&
              (strlen_found_key >= 0))
            {
            for (i = strlen_qmail_prefix; (i < tmp_strlen_username) && (strlen_qmail_ext < MAX_PATH); i++)
              {
              qmail_ext[strlen_qmail_ext] = (tmp_username[i] == QMAIL_REPLACE_EXT_TARGET) ? QMAIL_REPLACE_EXT_REPLACEMENT : tolower((int)tmp_username[i]);
              strlen_qmail_ext++;
              }
            qmail_ext[strlen_qmail_ext] = '\0';

            snprintf(tmp_filename, MAX_PATH, ".qmail%s%s", qmail_dash, qmail_ext);

            current_step = 20;
            }
          else
            {
            strlen_tmp_name = tmp_strlen_username;
            memcpy(tmp_name, tmp_username, tmp_strlen_username);
            tmp_name[strlen_tmp_name] = '\0';

            current_step = 28;
            }

          break;
        case 13:
        case 20:
          if ((tmp_return = check_path_perms(current_settings, qmail_home, S_IFDIR, 0, &tmp_stat, tmp_uid, tmp_gid)) == 1)
            current_step = (current_step == 13) ? 14 : 21;
          else if (tmp_return == 0)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_UNAVAILABLE;
            continue_processing = 0;
            }
          else
            {
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }

          break;
        case 14:
        case 21:
          if ((tmp_return = check_path_perms(current_settings, qmail_home, S_IFDIR, FILE_PERMISSION_READ | FILE_PERMISSION_EXECUTE, &tmp_stat, tmp_uid, tmp_gid)) == 1)
            current_step = (current_step == 14) ? 15 : 22;
          else if (tmp_return == 0)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_UNAVAILABLE;
            continue_processing = 0;
            }
          else
            {
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }

          break;
        case 15:
        case 22:
          if ((tmp_stat.st_mode & S_IWOTH) == 0)
            current_step = (current_step == 15) ? 16 : 23;
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_HOME_WRITEABLE, qmail_home);
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_UNAVAILABLE;
            continue_processing = 0;
            }

          break;
        case 16:
        case 23:
          if ((tmp_return = check_path_perms(current_settings, qmail_home, S_IFDIR, FILE_PERMISSION_STICKY | FILE_PERMISSION_EXECUTE, &tmp_stat, tmp_uid, tmp_gid)) == 0)
            current_step = (current_step == 16) ? 17 : 24;
          else if (tmp_return == 1)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_UNAVAILABLE;
            continue_processing = 0;
            }
          else
            {
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }

          break;
        case 17:
          snprintf(tmp_path, MAX_PATH, "%s/%s", qmail_home, tmp_filename);
          if ((tmp_return = check_path_perms(current_settings, tmp_path, S_IFREG, 0, &tmp_stat, tmp_uid, tmp_gid)) == 1)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_FILE_EXISTS, tmp_path);
            current_step = 32;
            }
          else if (tmp_return == 0)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_FILE_DOES_NOT_EXIST, tmp_path);
            snprintf(tmp_filename, MAX_PATH, ".qmail%sdefault", qmail_dash);
            current_step = 18;
            }
          else
            {
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }

          break;
        case 18:
        case 24:
          snprintf(tmp_path, MAX_PATH, "%s/%s", qmail_home, tmp_filename);
          if ((tmp_return = check_path_perms(current_settings, tmp_path, S_IFREG, 0, &tmp_stat, tmp_uid, tmp_gid)) == 1)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_FILE_EXISTS, tmp_path);
            current_step = 32;
            }
          else if (tmp_return == 0)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_FILE_DOES_NOT_EXIST, tmp_path);
            current_step = (current_step == 18) ? 19 : 25;
            }
          else
            {
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }

          break;
        case 19:
          if ((qmail_dash[0] == '\0') &&
              (current_settings->qmail_defaultdelivery_file != NULL))
            {
            snprintf(tmp_path, MAX_PATH, "%s", current_settings->qmail_defaultdelivery_file);
            max_line = 1;
            current_step = 37;
            }
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_NO_DEFAULTDELIVERY, tmp_path);
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_INVALID;
            continue_processing = 0;
            }

          break;
        case 25:
          found_match = 0;
          for (i = strlen_qmail_ext - 1; i >= 0; i--)
            if (qmail_ext[i] == QMAIL_EXT_TRUNCATE_TARGET)
              {
              found_match = 1;
              break;
              }

          if (found_match)
            {
            strlen_qmail_ext = i;
            qmail_ext[i] = '\0';
            snprintf(tmp_filename, MAX_PATH, ".qmail%s%s-default", qmail_dash, qmail_ext);
            current_step = 24;
            }
          else
            current_step = 26;

          break;
        case 26:
          if (strlen_qmail_ext == 0)
            current_step = 27;
          else
            {
            strlen_qmail_ext = 0;
            qmail_ext[0] = '\0';
            snprintf(tmp_filename, MAX_PATH, ".qmail%sdefault", qmail_dash);
            current_step = 24;
            }

          break;
        case 27:
          if ((qmail_dash[0] == '\0') &&
              (current_settings->qmail_defaultdelivery_file != NULL))
            {
            snprintf(tmp_path, MAX_PATH, "%s", current_settings->qmail_defaultdelivery_file);
            max_line = 1;
            current_step = 37;
            }
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_NO_DEFAULTDELIVERY, tmp_path);
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_INVALID;
            continue_processing = 0;
            }

          break;
        case 28:
          if (strlen_tmp_name > 31)
            {
            strlen_tmp_name = 34;
            tmp_name[strlen_tmp_name] = '\0';
            current_step = 30;
            }
          else
            current_step = 29;

          break;
        case 29:
          found_match = 0;
          errno = 0;
          if ((tmp_passwd = getpwnam(tmp_name)) != NULL)
            {
            tmp_uid = tmp_passwd->pw_uid;
            tmp_gid = tmp_passwd->pw_gid;

            if (!stat(tmp_passwd->pw_dir, &tmp_stat))
              {
              if (tmp_stat.st_uid == tmp_passwd->pw_uid)
                {
                snprintf(qmail_home, MAX_PATH, "%s", tmp_passwd->pw_dir);

                if (strlen_tmp_name == tmp_strlen_username)
                  qmail_dash[0] = '\0';
                else
                  {
                  memcpy(qmail_dash, QMAIL_DASH_USER_FOUND, STRLEN(QMAIL_DASH_USER_FOUND));
                  qmail_dash[STRLEN(QMAIL_DASH_USER_FOUND)] = '\0';
                  }

                strlen_qmail_ext = 0;
                for (i = strlen_tmp_name + 1; i < tmp_strlen_username; i++)
                  {
                  qmail_ext[strlen_qmail_ext] = (tmp_username[i] == QMAIL_REPLACE_EXT_TARGET) ? QMAIL_REPLACE_EXT_REPLACEMENT : tolower((int)tmp_username[i]);
                  strlen_qmail_ext++;
                  }
                qmail_ext[strlen_qmail_ext] = '\0';

                snprintf(tmp_filename, MAX_PATH, ".qmail%s%s", qmail_dash, qmail_ext);

                current_step = 24;
                found_match = 1;
                }
              else
                QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_HOME_NOT_OWNED, tmp_passwd->pw_dir, tmp_name, tmp_passwd->pw_uid, tmp_stat.st_uid);
              }
            else
              QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_HOME_NOT_FOUND, tmp_passwd->pw_dir);
            }
          else if (errno != 0)
            {
            QRV_LOG_ERROR(current_settings, LOG_ERROR_GETUSER_ERRNO, tmp_name, strerror(errno));
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }
          else
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_USER_NOT_FOUND, tmp_name);

          if (!found_match)
            current_step = 30;

          break;
        case 30:
          found_match = 0;
          for (i = strlen_tmp_name - 1; i >= 0; i--)
            if (tmp_name[i] == QMAIL_USER_TRUNCATE_TARGET)
              {
              strlen_tmp_name = i;
              tmp_name[strlen_tmp_name] = '\0';
              found_match = 1;
              break;
              }

          current_step = (found_match) ? 29 : 31;
          break;
        case 31:
          errno = 0;
          if ((tmp_passwd = getpwnam(QMAIL_USER_ALIAS)) != NULL)
            {
            tmp_uid = tmp_passwd->pw_uid;
            tmp_gid = tmp_passwd->pw_gid;

            snprintf(qmail_home, MAX_PATH, "%s", tmp_passwd->pw_dir);

            memcpy(qmail_dash, QMAIL_DASH_USER_FOUND, STRLEN(QMAIL_DASH_USER_FOUND));
            qmail_dash[STRLEN(QMAIL_DASH_USER_FOUND)] = '\0';

            for (i = 0; i < tmp_strlen_username; i++)
              qmail_ext[i] = (tmp_username[i] == QMAIL_REPLACE_EXT_TARGET) ? QMAIL_REPLACE_EXT_REPLACEMENT : tolower((int)tmp_username[i]);
            strlen_qmail_ext = tmp_strlen_username;
            qmail_ext[strlen_qmail_ext] = '\0';

            snprintf(tmp_filename, MAX_PATH, ".qmail%s%s", qmail_dash, qmail_ext);
            current_step = 24;
            }
          else if (errno != 0)
            {
            QRV_LOG_ERROR(current_settings, LOG_ERROR_GETUSER_ERRNO, tmp_name, strerror(errno));
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_USER_NOT_FOUND, tmp_name);
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_INVALID;
            continue_processing = 0;
            }

          break;
        case 32:
          if ((tmp_return = check_path_perms(current_settings, tmp_path, S_IFREG, FILE_PERMISSION_READ, &tmp_stat, tmp_uid, tmp_gid)) == 1)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_FILE_READABLE, tmp_uid, tmp_gid, tmp_path);
            current_step = 33;
            }
          else if (tmp_return == 0)
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_FILE_UNREADABLE, tmp_uid, tmp_gid, tmp_path);
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_UNAVAILABLE;
            continue_processing = 0;
            }
          else
            {
            return_value = DECISION_ERROR;
            continue_processing = 0;
            }

          break;
        case 33:
          if ((tmp_stat.st_mode & S_IWOTH) == 0)
            current_step = 34;
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_FILE_WRITEABLE, tmp_path);
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_UNAVAILABLE;
            continue_processing = 0;
            }

          break;
        case 34:
          if (tmp_stat.st_size > 0)
            current_step = 35;
          else if (current_settings->qmail_defaultdelivery_file != NULL)
            {
            snprintf(tmp_path, MAX_PATH, "%s", current_settings->qmail_defaultdelivery_file);
            max_line = 1;
            current_step = 37;
            }
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIDATE_NO_DEFAULTDELIVERY, tmp_path);
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_INVALID;
            continue_processing = 0;
            }

          break;
        case 35:
          if (tmp_stat.st_mode & S_IXUSR)
            forward_only = 1;

          current_step = 36;
          break;
        case 36:
          if (qmail_lines == NULL)
            {
            if ((qmail_lines = (char **)malloc(sizeof(char *) * QMAIL_LINES_PER_READ)) != NULL)
              for (i = 0; i < QMAIL_LINES_PER_READ; i++)
                qmail_lines[i] = NULL;
            else
              {
              QRV_LOG_ERROR(current_settings, LOG_ERROR_MALLOC, sizeof(char *) * QMAIL_LINES_PER_READ);
              return_value = DECISION_ERROR;
              continue_processing = 0;
              break;
              }
            }
              
          found_match = 0;
          if (((num_lines = read_file(current_settings, tmp_path, &qmail_lines, 0, 1, QMAIL_LINES_PER_READ, 1)) > 1) &&
              (qmail_lines[0] != NULL))
            {
            for (i = 0; qmail_lines[0][i] != '\0'; i++)
              if (!isspace((int)qmail_lines[0][i]))
                {
                num_lines--;
                found_match = 1;
                break;
                }
            }

          if (found_match)
            {
            current_line = 0;
            last_line = num_lines;
            current_step = 37;
            }
          else if (num_lines == -1)
            {
            QRV_LOG_VERBOSE(current_settings, LOG_VERBOSE_RECIPIENT_PERMISSION, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_path);
            continue_processing = 0;
            }
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_UNAVAILABLE;
            continue_processing = 0;
            }

          break;
        case 37:
          if (qmail_lines == NULL)
            {
            if ((qmail_lines = (char **)malloc(sizeof(char *) * QMAIL_LINES_PER_READ)) != NULL)
              for (i = 0; i < QMAIL_LINES_PER_READ; i++)
                qmail_lines[i] = NULL;
            else
              {
              QRV_LOG_ERROR(current_settings, LOG_ERROR_MALLOC, sizeof(char *) * QMAIL_LINES_PER_READ);
              return_value = DECISION_ERROR;
              continue_processing = 0;
              break;
              }
            }

          if ((max_line == -1) ||
              (total_lines < max_line))
            if (current_line >= num_lines)
              {
              for (i = 0; i < QMAIL_LINES_PER_READ; i++)
                if (qmail_lines[i] != NULL)
                  {
                  free(qmail_lines[i]);
                  qmail_lines[i] = NULL;
                  }

              if (((num_lines = read_file(current_settings, tmp_path, (char ***)&qmail_lines, 0, last_line + 1, QMAIL_LINES_PER_READ, 1)) > 1) &&
                  (qmail_lines[0] != NULL))
                {
                num_lines--;
                current_line = 0;
                last_line += num_lines;
                current_step = 38;
                }
              else if (num_lines == -1)
                {
                QRV_LOG_VERBOSE(current_settings, LOG_VERBOSE_RECIPIENT_PERMISSION, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_path);
                continue_processing = 0;
                }
              else
                {
                if (return_value == DECISION_UNKNOWN)
                  {
                  QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
                  return_value = DECISION_INVALID;
                  }

                continue_processing = 0;
                }
              }
            else
              current_step = 38;
          else
            {
            QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
            return_value = DECISION_INVALID;
            continue_processing = 0;
            }

          break;
        case 38:
          found_match = 0;
          if (qmail_lines[current_line][0] != QMAIL_COMMENT)
            for (i = 0; qmail_lines[current_line][i] != '\0'; i++)
              if (!isspace((int)qmail_lines[current_line][i]))
                {
                found_match = 1;
                break;
                }

          if (found_match)
            current_step = 39;
          else
            {
            current_line++;
            current_step = 37;
            }

          break;
        case 39:
          for (i = 0; qmail_lines[current_line][i] != '\0'; i++);
          if ((strchr(QMAIL_MBOX_START_CHARS, qmail_lines[current_line][0]) != NULL) &&
              (i > 0) &&
              (qmail_lines[current_line][i - 1] != QMAIL_MBOX_END_NOT_CHAR))
            {
            snprintf(tmp_qmail_filename, MAX_PATH, "%s", qmail_lines[current_line]);
            current_step = 40;
            }
          else
            current_step = 41;

          break;
        case 40:
          if (!forward_only)
            {
            if (tmp_qmail_filename[0] == DIR_DELIMITER)
              tmp_return = check_path_perms(current_settings, tmp_qmail_filename, S_IFREG, FILE_PERMISSION_WRITE, NULL, tmp_uid, tmp_gid);
            else
              {
              snprintf(tmp_unreal_path, MAX_PATH, "%s/%s", qmail_home, tmp_qmail_filename);
              realpath(tmp_unreal_path, tmp_qmail_path);
              tmp_return = check_path_perms(current_settings, tmp_qmail_path, S_IFREG, FILE_PERMISSION_WRITE, NULL, tmp_uid, tmp_gid);
              }

            if (tmp_return == 1)
              {
              return_value = DECISION_VALID;
              continue_processing = 0;
              }
            else if (tmp_return == 0)
              {
              QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
              return_value = DECISION_UNAVAILABLE;

              current_line++;
              current_step = 37;
              }
            else
              {
              return_value = DECISION_ERROR;
              continue_processing = 0;
              }
            }
          else
            {
            current_line++;
            current_step = 37;
            }

          break;
        case 41:
          for (i = 0; qmail_lines[current_line][i] != '\0'; i++);
          if ((strchr(QMAIL_MAILDIR_START_CHARS, qmail_lines[current_line][0]) != NULL) &&
              (i > 0) &&
              (qmail_lines[current_line][i - 1] == QMAIL_MAILDIR_END_CHAR))
            {
            snprintf(tmp_qmail_filename, MAX_PATH, "%s", qmail_lines[current_line]);
            current_step = 42;
            }
          else
            current_step = 43;

          break;
        case 42:
          if (!forward_only)
            {
            if (tmp_qmail_filename[0] == DIR_DELIMITER)
              snprintf(tmp_qmail_path, MAX_PATH, "%snew", tmp_qmail_filename);
            else
              {
              snprintf(tmp_unreal_path, MAX_PATH, "%s/%snew", qmail_home, tmp_qmail_filename);
              realpath(tmp_unreal_path, tmp_qmail_path);
              }

            if ((tmp_return = check_path_perms(current_settings, tmp_qmail_path, S_IFDIR, FILE_PERMISSION_WRITE, NULL, tmp_uid, tmp_gid)) == 1)
              {
              return_value = DECISION_VALID;
              continue_processing = 0;
              }
            else if (tmp_return == 0)
              {
              QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
              return_value = DECISION_UNAVAILABLE;

              current_line++;
              current_step = 37;
              }
            else
              {
              return_value = DECISION_ERROR;
              continue_processing = 0;
              }
            }
          else
            {
            current_line++;
            current_step = 37;
            }

          break;
        case 43:
          if (qmail_lines[current_line][0] == QMAIL_PROGRAM_START_CHAR)
            {
            find_command(qmail_lines[current_line] + 1, tmp_qmail_filename, MAX_PATH);
            current_step = 44;
            }
          else
            current_step = 45;

          break;
        case 44:
          if (!forward_only)
            {
            if ((tmp_return = find_path_perms(current_settings, tmp_qmail_filename, S_IFREG, FILE_PERMISSION_EXECUTE, tmp_uid, tmp_gid)) == 1)
              {

#ifdef WITH_VPOPMAIL_SUPPORT

              tmp_strlen_command = strlen(tmp_qmail_filename);
              if ((STRLEN(VPOPMAIL_VALIAS_PATH) > 0) &&
                  (STRLEN(VPOPMAIL_VUSERINFO_PATH) > 0) &&
                  !strncmp(tmp_qmail_filename + (tmp_strlen_command - STRLEN(VPOPMAIL_VDELIVERMAIL)), VPOPMAIL_VDELIVERMAIL, STRLEN(VPOPMAIL_VDELIVERMAIL)))
                {
                QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VPOPMAIL_FILE, current_line);
                tmp_strlen = strlen(qmail_lines[current_line]);
                found_match = 0;
                for (i = tmp_strlen - 1; i >= (tmp_strlen_command + 1); i--)
                  if (isspace((int)qmail_lines[current_line][i]))
                    {
                    found_match = 1;
                    break;
                    }

                if (found_match)
                  {
                  tmp_argv[0] = VPOPMAIL_VALIAS;
                  tmp_argv[1] = VPOPMAIL_VALIAS_ARG;
                  tmp_argv[2] = reassemble_address(target_recipient_username, strlen_recipient_username, target_recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL);
                  tmp_argv[3] = NULL;

                  if ((exec_command_argv(current_settings, VPOPMAIL_VALIAS_PATH, tmp_argv, vpopmail_output, MAX_BUF, &tmp_status, tmp_uid, tmp_gid) != -1) &&
                      (tmp_status == 0))
                    {
                    tmp_strlen_output = strlen(vpopmail_output);
                    for (i = 0; i < ((tmp_strlen_output - STRLEN(VPOPMAIL_VALIAS_DELIMITER)) - 1); i++)
                      if ((vpopmail_output[i] == VPOPMAIL_VALIAS_DELIMITER_START) &&
                          !strncmp(vpopmail_output + i, VPOPMAIL_VALIAS_DELIMITER, STRLEN(VPOPMAIL_VALIAS_DELIMITER)))
                        {
                        QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_VALIAS_ADDRESS, i, vpopmail_output + i + STRLEN(VPOPMAIL_VALIAS_DELIMITER));

                        i += STRLEN(VPOPMAIL_VALIAS_DELIMITER);
                        if (vpopmail_output[i] == QMAIL_FORWARD_START_CHAR)
                          i++;

                        k = -1;
                        for (j = i; (j < tmp_strlen_output) && (vpopmail_output[j] != '\n'); j++)
                          if (vpopmail_output[j] == '@')
                            k = j;
                        if (j < tmp_strlen_output)
                          vpopmail_output[j] = '\0';

                        if ((k > i) &&
                            (k < tmp_strlen_output))
                          {
                          if (((return_value = validate(current_settings, vpopmail_output + i, k - i, vpopmail_output + k + 1, forwarded + 1)) == DECISION_VALID) ||
                              (return_value == DECISION_ERROR))
                            {
                            continue_processing = 0;
                            break;
                            }
                          }

                        i = j + 1;
                        }
                    }
                  }

                if (found_match &&
                    continue_processing)
                  {
                  tmp_argv[0] = VPOPMAIL_VUSERINFO;
                  tmp_argv[1] = VPOPMAIL_VUSERINFO_ARG;
                  tmp_argv[2] = reassemble_address(target_recipient_username, strlen_recipient_username, target_recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL);
                  tmp_argv[3] = NULL;

                  if (exec_command_argv(current_settings, VPOPMAIL_VUSERINFO_PATH, tmp_argv, vpopmail_output, MAX_BUF, &tmp_status, tmp_uid, tmp_gid) != -1)
                    {
                    if (tmp_status == 0)
                      {
                      return_value = DECISION_VALID;
                      continue_processing = 0;
                      }
                    else if (!strncmp(qmail_lines[current_line] + i + 1, VPOPMAIL_BOUNCE, STRLEN(VPOPMAIL_BOUNCE)))
                      {
                      QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_INVALID_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
                      return_value = DECISION_INVALID;

                      current_line++;
                      current_step = 37;
                      }
                    else if (qmail_lines[current_line][i + 1] == DIR_DELIMETER)
                      {
                      snprintf(tmp_qmail_path, MAX_PATH, "%snew", qmail_lines[current_line] + i + 1);

                      if ((tmp_return = check_path_perms(current_settings, tmp_qmail_path, S_IFDIR, FILE_PERMISSION_WRITE, NULL, tmp_uid, tmp_gid)) == 1)
                        {
                        return_value = DECISION_VALID;
                        continue_processing = 0;
                        }
                      else
                        {
                        QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
                        return_value = DECISION_UNAVAILABLE;

                        current_line++;
                        current_step = 37;
                        }
                      }
                    else
                      {
                      //FIXME: Should this use find_address() instead?
                      for (tmp_strlen = 0; qmail_lines[current_line][i + tmp_strlen + 1] != '\0'; tmp_strlen++)
                        if (qmail_lines[current_line][i + tmp_strlen + 1] == '@')
                          break;

                      if (((return_value = validate(current_settings, qmail_lines[current_line] + i + 1, tmp_strlen, (qmail_lines[current_line][i + tmp_strlen + 1] == '@') ? (qmail_lines[current_line] + i + tmp_strlen + 2) : "", forwarded + 1)) == DECISION_VALID) ||
                          (return_value == DECISION_ERROR))
                        continue_processing = 0;
                      else
                        {
                        current_line++;
                        current_step = 37;
                        }
                      }
                    }
                  else
                    {
                    return_value = DECISION_VALID;
                    continue_processing = 0;
                    }
                  }
                else
                  {
                  return_value = DECISION_VALID;
                  continue_processing = 0;
                  }
                }
              else
                {
                return_value = DECISION_VALID;
                continue_processing = 0;
                }

#else /* WITH_VPOPMAIL_SUPPORT */

              return_value = DECISION_VALID;
              continue_processing = 0;

#endif /* WITH_VPOPMAIL_SUPPORT */

              }
            else if (tmp_return == 0)
              {
              QRV_LOG_EXCESSIVE(current_settings, LOG_EXCESSIVE_UNAVAILABLE_RECIPIENT, reassemble_address(current_settings->recipient_username, -1, current_settings->recipient_domain, NULL, tmp_address, MAX_ADDRESS, NULL), tmp_username);
              return_value = DECISION_UNAVAILABLE;

              current_line++;
              current_step = 37;
              }
            else
              {
              return_value = DECISION_ERROR;
              continue_processing = 0;
              }
            }
          else
            {
            current_line++;
            current_step = 37;
            }

          break;
        case 45:
          if ((qmail_lines[current_line][0] == QMAIL_FORWARD_START_CHAR) ||
              isalnum((int)qmail_lines[current_line][0]))
            current_step = 46;
          else
            {
            current_line++;
            current_step = 37;
            }

          break;
        case 46:
          //FIXME: Should this use find_address() instead?
          for (i = 0; (qmail_lines[current_line][i] != '\0') && (qmail_lines[current_line][i] != '@'); i++);
          tmp_strlen = (qmail_lines[current_line][i] == '@') ? strlen(qmail_lines[current_line] + i + 1) : 0;

          if ((strlen_recipient_username != (i - ((qmail_lines[current_line][0] == QMAIL_FORWARD_START_CHAR) ? 1 : 0))) ||
              (strlen_recipient_domain != tmp_strlen) ||
              strncmp(target_recipient_username, qmail_lines[current_line] + ((qmail_lines[current_line][0] == QMAIL_FORWARD_START_CHAR) ? 1 : 0), strlen_recipient_username) ||
              (strlen_recipient_domain != tmp_strlen) ||
              ((strlen_recipient_domain > 0) &&
               strncmp(target_recipient_domain, qmail_lines[current_line] + i + 1, strlen_recipient_domain)))
            current_step = 47;
          else
            {
            current_line++;
            current_step = 37;
            }

          break;
        case 47:
          if (((return_value = validate(current_settings, qmail_lines[current_line] + ((qmail_lines[current_line][0] == QMAIL_FORWARD_START_CHAR) ? 1 : 0), (qmail_lines[current_line][i] == '@') ? (i - 1) : i, (qmail_lines[current_line][i] == '@') ? (qmail_lines[current_line] + i + 1) : "", forwarded + 1)) == DECISION_VALID) ||
              (return_value == DECISION_ERROR))
            continue_processing = 0;
          else
            {
            current_line++;
            current_step = 37;
            }

          break;
        default:
          return_value = DECISION_ERROR;
          continue_processing = 0;
        }
      }

    endpwent();

    if (qmail_lines != NULL)
      {
      for (i = 0; i < QMAIL_LINES_PER_READ; i++)
        if (qmail_lines[i] != NULL)
          free(qmail_lines[i]);

      free(qmail_lines);
      }

    if (tmp_file_line != NULL)
      free(tmp_file_line);
    }
  else
    QRV_LOG_VERBOSE(current_settings, LOG_VERBOSE_VALIDATE_DEPTH, reassemble_address(target_recipient_username, -1, target_recipient_domain, LOG_DATA_NULL, tmp_address, MAX_ADDRESS, NULL), forwarded);

  return(return_value);
  }
