#include <linux/module.h>

#include <linux/kernel.h> /* printk() */
#include <linux/slab.h> /* kmalloc() */
#include <linux/fs.h>     /* everything... */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/stddef.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>        /* O_ACCMODE */
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/devfs_fs_kernel.h>

#include <linux/sched.h>
#include <linux/wait.h>

#include "kernel25compat.h"

#include <asm/uaccess.h>
#include <asm/system.h>   /* cli(), *_flags */
#include <asm/segment.h>  /* memcpy and such */
#include <asm/io.h>
#include <asm/pgtable.h>
#include <asm/irq.h>

#include "svgalib_helper.h"

#include "mixed/vgaversion.h"

#include "i810.h"
#include "interrupt.h"

int num_devices=0;

static char *sdev_id="svgalib_helper";

static struct sh_pci_device *sh_pci_devs[MAX_NR_DEVICES];

static int irqs[MAX_NR_DEVICES];

#ifdef CONFIG_DEVFS_FS
static devfs_handle_t devfs_handle;
#endif

static int check_io_range(port,device) {
    return 1;
}

static struct pci_dev *get_pci_dev(int pcipos, int minor) {
    
    if(minor>=num_devices) return NULL;
    if(minor>0) {
        return sh_pci_devs[minor]->dev;
    } else {
        if(pcipos>0 && pcipos<num_devices)
            return sh_pci_devs[pcipos]->dev;
    }
    return NULL;

}

static int get_dev(int pcipos, int minor) {
    
    if(minor>=num_devices) return 0;
    if(minor>0) {
        return minor;
    } else {
        int i;
        int b, d;

        d=pcipos&0xff;
        b=(pcipos>>8)&0xff;
        for(i=1;i<num_devices;i++) {
            if((b==sh_pci_devs[i]->dev->bus->number) && (d==sh_pci_devs[i]->dev->devfn)) 
                return i;
        }
        return 0;
    }
}
static volatile int vsync=0;
static wait_queue_head_t vsync_wait;

static void vsync_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
    struct sh_pci_device *dev = (struct sh_pci_device *)dev_id;

    if((char *)dev==sdev_id) {
        if(!vga_test_vsync(dev))return;    
        vga_ack_vsync(dev);
    } else {
        if(!dev->test_vsync(dev)) return;
        dev->ack_vsync(dev);
    }

    vsync=0;
    wake_up_interruptible(&vsync_wait);
}


static int svgalib_helper_ioctl( struct inode *inode, struct file *filp, 
                          unsigned int cmd, unsigned long arg) {

    io_t iov;
    pcic_t pciv;
    int minor = minor(inode->i_rdev);
    struct pci_dev *pdev;
    io_string_t iostr;
    int i = 0, ret;
    u8 pb;
    u16 pw;
    u32 pl;
    void *dev_id;
    unsigned char *outb_str;

    if(_IOC_TYPE(cmd)!=SVGALIB_HELPER_IOC_MAGIC) {
        return -EINVAL;
    }
    
    ret=0;

    switch(_IOC_NR(cmd)) {

        case _IOC_NR(SVGALIB_HELPER_IOCSREPOUTB):
            copy_from_user(&iostr,(char *)arg,sizeof(iostr));
            if (iostr.length>4096) return -EINVAL;
	    if ( (outb_str = kmalloc(iostr.length, GFP_KERNEL )) == NULL ) return -ENOMEM;
	    copy_from_user(outb_str,iostr.string,iostr.length);
	    if(check_io_range(iostr.port,minor))
	        for(i=0; i<iostr.length; i++)
                    outb(outb_str[i], iostr.port);
                else ret = -EPERM;
            kfree (outb_str);
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCSOUTB):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           if(check_io_range(iov.port,minor))
               outb(iov.val,iov.port);
               else ret = -EPERM;
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCSOUTW):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           if(check_io_range(iov.port,minor))
               outw(iov.val,iov.port);
               else ret = -EPERM;
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCSOUTL):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           if(check_io_range(iov.port,minor))
               outl(iov.val,iov.port);
               else ret = -EPERM;
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGINB):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           if(check_io_range(iov.port,minor))
               iov.val=inb(iov.port);
               else ret = -EPERM;
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGINW):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           if(check_io_range(iov.port,minor))
               iov.val=inw(iov.port);
               else ret = -EPERM;
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGINL):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           if(check_io_range(iov.port,minor))
               iov.val=inl(iov.port);
               else ret = -EPERM;
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCSWRITEB):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           writeb(iov.val,iov.port);
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCSWRITEW):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           writew(iov.val,iov.port);
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCSWRITEL):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           writel(iov.val,iov.port);
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGREADB):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           iov.val=readb(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGREADW):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           iov.val=readw(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGREADL):
           copy_from_user(&iov,(char *)arg,sizeof(iov));
           iov.val=readl(iov.port);
           copy_to_user((char *)arg,&iov,sizeof(iov));
        break;

        case _IOC_NR(SVGALIB_HELPER_IOCGPCIINB):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pci_read_config_byte(pdev, pciv.address, &pb);
            pciv.val=pb;
            copy_to_user((char *)arg,&pciv,sizeof(pciv));
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCGPCIINW):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pci_read_config_word(pdev, pciv.address, &pw);
            pciv.val=pw;
            copy_to_user((char *)arg,&pciv,sizeof(pciv));
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCGPCIINL):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pci_read_config_dword(pdev, pciv.address, &pl);
            pciv.val=pl;
            copy_to_user((char *)arg,&pciv,sizeof(pciv));
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCGPCIAPLEN):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            i = get_dev(pciv.pcipos, minor);
            if((i==0) | (pciv.address>5)) return -EINVAL;
            pciv.val=sh_pci_devs[i]->len[pciv.address];
            copy_to_user((char *)arg,&pciv,sizeof(pciv));
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCSPCIOUTB):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pb=pciv.val;
            pci_write_config_byte(pdev, pciv.address, pb);
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCSPCIOUTW):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pw=pciv.val;
            pci_write_config_word(pdev, pciv.address, pw);
            break;

        case _IOC_NR(SVGALIB_HELPER_IOCSPCIOUTL):
            copy_from_user(&pciv,(char *)arg,sizeof(pciv));
            pdev = get_pci_dev(pciv.pcipos, minor);
            if(!pdev) return -EINVAL;
            pl=pciv.val;
            pci_write_config_dword(pdev, pciv.address, pl);
            break;
#ifdef __386__
        case _IOC_NR(SVGALIB_HELPER_IOCGI810GTT):
            i=i810_make_gtt();
            copy_to_user((char *)arg, &i, sizeof(unsigned int));
            break;
        case _IOC_NR(SVGALIB_HELPER_IOCGI810GTTE):
            copy_from_user(&i, (char *)arg, sizeof(unsigned int));
            if(i<I810_SIZE)
                copy_to_user((char *)arg, &i810_gttes[i], sizeof(unsigned long));
                else return -EINVAL;
            break;
#endif
        case _IOC_NR(SVGALIB_HELPER_IOCWAITRETRACE):

            /* Workaround for nvidia cards, which are not vga compatible */
            if(!minor && num_devices==2) minor=1;

            if(minor) {
                i=sh_pci_devs[minor]->dev->irq;
                dev_id = sh_pci_devs[minor];
                if(i==0 || i==-1 || i==255) return -EINVAL;
            } else dev_id = sdev_id;
                        
            vsync=1;

            if(minor) {
                request_irq(i, vsync_interrupt, SA_SHIRQ, "svgalib_helper", dev_id);
            } else {
                i=0;
                while(irqs[i]!=-1)
                    request_irq(irqs[i++], vsync_interrupt, SA_SHIRQ, "svgalib_helper", dev_id);
            }

            if(minor) {
                sh_pci_devs[minor]->enable_vsync(sh_pci_devs[minor]);
            } else {
                vga_enable_vsync(sh_pci_devs[minor]);
            }

            interruptible_sleep_on(&vsync_wait);
            
            if(minor) {
                if(vsync) sh_pci_devs[minor]->ack_vsync(dev_id);
                free_irq(i, dev_id);
            } else {
                i=0;
                if(vsync) vga_ack_vsync(dev_id);
                while(irqs[i]!=-1)
                    free_irq(irqs[i++], dev_id);
            }
            
            if(vsync) return -ERESTARTSYS;
            break;
        default: 
            return -EINVAL;
    }
    return ret;
}   			


static int svgalib_helper_open( struct inode *inode, struct file * filp) {

   int minor = minor(inode->i_rdev);
   
   if(minor>=num_devices) return -ENODEV;
   
   MOD_INC_USE_COUNT;
   
   return 0;
}

static int svgalib_helper_release( struct inode *inode, struct file *filp) {
   MOD_DEC_USE_COUNT;
   
   return 0;
}

int remap_cache(struct vm_area_struct *vma, unsigned long ofs,
    		unsigned long start, unsigned long end) {

#if defined(__powerpc__)
	  pgprot_val(vma->vm_page_prot) |= _PAGE_NO_CACHE|_PAGE_GUARDED;
#elif defined(__i386__)
	  if (boot_cpu_data.x86 > 3) pgprot_val(vma->vm_page_prot) |= _PAGE_PCD;
#endif
          return my_remap_page_range(vma, start, ofs, end-start, vma->vm_page_prot);
}

int check_mem(int card, unsigned long start, unsigned long len) {
    int j;
    unsigned long rstart, rlen;
#ifdef __alpha__
    int type;
#endif
        
    rstart=start;
    rlen=len;
#ifdef __alpha__
    type = start>>32;
    switch(type) {
        case 2:
            start = start & 0xffffffff;
            start = start >> 5;
            len = len >> 5;
            break;
        case 3:
            start = start & 0xffffffff;
            break;
        default:
            return -1;
    }
#endif

    if(!card) {
        if( (start<0x110000) && (start+len<0x110000) ) return 0;
        for(j=1;j<num_devices;j++)
            if(!check_mem(j, rstart, rlen)) return 0;
    } else if(card<num_devices) {
        for(j=0;j<6;j++) if(sh_pci_devs[card]->mem[j])
            if((start>=sh_pci_devs[card]->mem[j])&&
               (start+len<=sh_pci_devs[card]->mem[j]+sh_pci_devs[card]->len[j])) {
               return 0;
           }
    }
    return -2;
}

static int svgalib_helper_mmap(struct file *filp, struct vm_area_struct *vma) {
   unsigned long start=vma->vm_start;
   unsigned long end=vma->vm_end;
   unsigned long minor = minor(filp->f_dentry->d_inode->i_rdev);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
   unsigned long ofs=vma->vm_pgoff*PAGE_SIZE;
#else
   unsigned long ofs=vma->vm_offset;
#endif

   if(check_mem(minor, ofs, end-start)) return -EPERM;
   if(remap_cache(vma, ofs, start, end)) return -EAGAIN;

   printk(KERN_INFO "svgalib_helper: remapped address (0x%x-0x%x) for PID: %d, UID: %d\n",
       start, end, current->pid, current->uid);

   return 0;
}

struct file_operations svgalib_helper_fops = {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
   .owner	= THIS_MODULE,
#endif
   .ioctl	= svgalib_helper_ioctl,
   .mmap	= svgalib_helper_mmap,
   .open	= svgalib_helper_open,
   .release	= svgalib_helper_release,
};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
#define base_address(i) dev->resource[i].start
#else
#define base_address(i) (dev->base_address[i]&PCI_BASE_ADDRESS_MEM_MASK)
#endif

int init_module(void)
{
    int result, i, j;
    struct pci_dev *dev=NULL;
#ifdef CONFIG_DEVFS_FS
    devfs_handle_t slave_handle;
#endif
    /*
     * Register your major, and accept a dynamic number
     */
     
    printk(KERN_INFO "svgalib_helper: Initializing, version %s\n", versionstr);
    result = devfs_register_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper", &svgalib_helper_fops);
    if (result < 0) {
        printk(KERN_WARNING "svgalib_helper: can't get major %d\n",SVGALIB_HELPER_MAJOR);
        return result;
    }

    if((sh_pci_devs[0]=kmalloc(sizeof(struct sh_pci_device),GFP_KERNEL))==NULL) {
        goto nomem_error;
    }
    memset(sh_pci_devs[0],0,sizeof(struct sh_pci_device));
    num_devices=1;
    for(i=1;i<MAX_NR_DEVICES;i++) sh_pci_devs[i]=NULL;

#ifdef CONFIG_DEVFS_FS
    devfs_handle = devfs_mk_dir ( NULL, "svga_helper", NULL );
    devfs_register_series( devfs_handle,
               "%u", 8, DEVFS_FL_DEFAULT, SVGALIB_HELPER_MAJOR, 0,
                       S_IFCHR |  S_IRUGO | S_IRWXU, &svgalib_helper_fops, NULL ) ;
    devfs_mk_symlink( NULL, "svga", 0, "svga_helper/0", &slave_handle, NULL );
    devfs_auto_unregister( devfs_handle, slave_handle );
#endif /* devfsd support */

    if(pci_present()) {
        while((dev=pci_find_class(PCI_CLASS_DISPLAY_VGA<<8,dev)) && 
              (num_devices<=MAX_NR_DEVICES)) {
            if((sh_pci_devs[num_devices]=kmalloc(sizeof(struct sh_pci_device),GFP_KERNEL))==NULL) {
                goto nomem_error;
            }
            memset(sh_pci_devs[num_devices],0,sizeof(struct sh_pci_device));
            sh_pci_devs[num_devices]->dev=dev;
            pci_read_config_word(dev,0,&sh_pci_devs[num_devices]->vendor);
            pci_read_config_word(dev,2,&sh_pci_devs[num_devices]->id);
            pci_read_config_byte(dev,8,&sh_pci_devs[num_devices]->revision);
            printk(KERN_DEBUG "svgalib_helper: device%d: vendor:%.4x id:%.4x\n",num_devices,
            sh_pci_devs[num_devices]->vendor,sh_pci_devs[num_devices]->id);
            for(i=0;i<6;i++){
                unsigned int t; 
                int len;
                pci_read_config_dword(dev,16+4*i,&result);
                if(result) {
                    pci_write_config_dword(dev,16+4*i,0xffffffff);
                    pci_read_config_dword(dev,16+4*i,&t);
                    pci_write_config_dword(dev,16+4*i,result);
                    len = ~(t&~0xf)+1;
                    if (len){
                       sh_pci_devs[num_devices]->mem[i]=result&~0xf;
                       sh_pci_devs[num_devices]->flags[i]=0x80 | (result&0xf);
                       sh_pci_devs[num_devices]->len[i]=len;
                       sh_pci_devs[num_devices]->mask[i]=t&~0xf;
                       printk(KERN_DEBUG "device%d: region%d, base=%.8x len=%d type=%d\n",
                       num_devices,i, result&(~0xf), len, result&0xf);
                    }
                }
            }
            vga_init_vsync(sh_pci_devs[num_devices]);
            num_devices++;
        }
    }
    
    j=0;
    for(i=1; i<num_devices;i++) {
        int k, l;
        k=sh_pci_devs[i]->dev->irq;
        if(k>0 && k<255) {
            for(l=0;l<j;l++) if(k==irqs[l])k=-1;
            if(k>0) irqs[j]=k;
            j++;
        }
    }
    irqs[j]=-1;

    init_waitqueue_head(&vsync_wait);

    EXPORT_NO_SYMBOLS;

    return 0; /* succeed */

    nomem_error:
    for(i=0;i<MAX_NR_DEVICES;i++) 
        if(sh_pci_devs[i])kfree(sh_pci_devs[i]);     
    devfs_unregister_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper");
    return result;
}

void cleanup_module(void)
{
    int i;
    for(i=0;i<MAX_NR_DEVICES;i++) 
        if(sh_pci_devs[i]) {
            kfree(sh_pci_devs[i]);
        }
     
#ifdef CONFIG_DEVFS_FS
    devfs_unregister(devfs_handle);
#endif
    devfs_unregister_chrdev(SVGALIB_HELPER_MAJOR, "svgalib_helper");
}

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL and additional rights");
#endif

MODULE_AUTHOR("Matan Ziv-Av <matan@svgalib.org>");
MODULE_DESCRIPTION("Generic hardware access to vga cards");
