/*-*- Mode: C; c-basic-offset: 8; indent-tabs-mode: nil -*-*/

/***
  This file is part of systemd.

  Copyright 2012 Lennart Poettering

  systemd is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 2.1 of the License, or
  (at your option) any later version.

  systemd is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with systemd; If not, see <http://www.gnu.org/licenses/>.
***/

#include <unistd.h>

#include <systemd/sd-messages.h>

#include "conf-parser.h"
#include "special.h"
#include "dbus-common.h"
#include "logind-action.h"

int manager_handle_action(
                Manager *m,
                InhibitWhat inhibit_key,
                HandleAction handle,
                bool ignore_inhibited,
                bool is_edge) {

        static const char * const message_table[_HANDLE_ACTION_MAX] = {
                [HANDLE_POWEROFF] = "Powering Off...",
                [HANDLE_REBOOT] = "Rebooting...",
                [HANDLE_HALT] = "Halting...",
                [HANDLE_KEXEC] = "Rebooting via kexec...",
                [HANDLE_SUSPEND] = "Suspending...",
                [HANDLE_HIBERNATE] = "Hibernating...",
                [HANDLE_HYBRID_SLEEP] = "Hibernating and suspending..."
        };

        static const char * const target_table[_HANDLE_ACTION_MAX] = {
                [HANDLE_POWEROFF] = SPECIAL_POWEROFF_TARGET,
                [HANDLE_REBOOT] = SPECIAL_REBOOT_TARGET,
                [HANDLE_HALT] = SPECIAL_HALT_TARGET,
                [HANDLE_KEXEC] = SPECIAL_KEXEC_TARGET,
                [HANDLE_SUSPEND] = SPECIAL_SUSPEND_TARGET,
                [HANDLE_HIBERNATE] = SPECIAL_HIBERNATE_TARGET,
                [HANDLE_HYBRID_SLEEP] = SPECIAL_HYBRID_SLEEP_TARGET
        };

        DBusError error;
        int r;
        InhibitWhat inhibit_operation;
        bool supported;

        assert(m);

        if (m->action_what) {
                log_debug("Action already in progress, ignoring.");
                return -EALREADY;
        }

        /* If the key handling is turned off, don't do anything */
        if (handle == HANDLE_IGNORE) {
                log_debug("Refusing operation, as it is turned off.");
                return 0;
        }

        if (handle == HANDLE_SUSPEND)
                supported = can_sleep("mem") > 0;
        else if (handle == HANDLE_HIBERNATE)
                supported = can_sleep("disk") > 0;
        else if (handle == HANDLE_HYBRID_SLEEP)
                supported = can_sleep("disk") > 0 && can_sleep_disk("suspend") > 0;
        else if (handle == HANDLE_KEXEC)
                supported = access("/sbin/kexec", X_OK) >= 0;
        else
                supported = true;

        if (!supported) {
                log_warning("Requested operation not supported, ignoring.");
                return -ENOTSUP;
        }

        /* If the key handling is inhibited, don't do anything */
        if (inhibit_key > 0) {
                if (manager_is_inhibited(m, inhibit_key, INHIBIT_BLOCK, NULL, true, false, 0)) {
                        log_debug("Refusing operation, %s is inhibited.", inhibit_what_to_string(inhibit_key));
                        return 0;
                }
        }

        /* Locking is handled differently from the rest. */
        if (handle == HANDLE_LOCK) {
                log_info("Locking sessions...");
                session_send_lock_all(m, true);
                return 1;
        }

        inhibit_operation = handle == HANDLE_SUSPEND || handle == HANDLE_HIBERNATE || handle == HANDLE_HYBRID_SLEEP ? INHIBIT_SLEEP : INHIBIT_SHUTDOWN;

        /* If the actual operation is inhibited, warn and fail */
        if (!ignore_inhibited &&
            manager_is_inhibited(m, inhibit_operation, INHIBIT_BLOCK, NULL, false, false, 0)) {

                /* If this is just a recheck of the lid switch then don't warn about anything */
                if (!is_edge) {
                        log_debug("Refusing operation, %s is inhibited.", inhibit_what_to_string(inhibit_operation));
                        return 0;
                }

                log_error("Refusing operation, %s is inhibited.", inhibit_what_to_string(inhibit_operation));
                warn_melody();
                return -EPERM;
        }

        log_info("%s", message_table[handle]);

        dbus_error_init(&error);
        r = bus_manager_shutdown_or_sleep_now_or_later(m, target_table[handle], inhibit_operation, &error);
        if (r < 0) {
                log_error("Failed to execute operation: %s", bus_error_message(&error));
                dbus_error_free(&error);
                return r;
        }

        return 1;
}

static const char* const handle_action_table[_HANDLE_ACTION_MAX] = {
        [HANDLE_IGNORE] = "ignore",
        [HANDLE_POWEROFF] = "poweroff",
        [HANDLE_REBOOT] = "reboot",
        [HANDLE_HALT] = "halt",
        [HANDLE_KEXEC] = "kexec",
        [HANDLE_SUSPEND] = "suspend",
        [HANDLE_HIBERNATE] = "hibernate",
        [HANDLE_HYBRID_SLEEP] = "hybrid-sleep",
        [HANDLE_LOCK] = "lock"
};

DEFINE_STRING_TABLE_LOOKUP(handle_action, HandleAction);
DEFINE_CONFIG_PARSE_ENUM(config_parse_handle_action, handle_action, HandleAction, "Failed to parse handle action setting");
