# $Id: TLPDB.pm 10016 2008-08-02 22:12:46Z preining $
# TeXLive::TLPDB.pm - module for using tlpdb files
# Copyright 2007, 2008 Norbert Preining
#
# This file is licensed under the GNU General Public License version 2
# or any later version.

package TeXLive::TLPDB;

=pod

=head1 NAME

C<TeXLive::TLPDB> -- A database of TeX Live Packages

=head1 SYNOPSIS

  use TeXLive::TLPDB;

  TeXLive::TLPDB->new ();
  TeXLive::TLPDB->new (root => "/path/to/texlive/installation/root");

  $tlpdb->root("/path/to/root/of/texlive/installation");
  $tlpdb->copy;
  $tlpdb->from_file($filename);
  $tlpdb->writeout;
  $tlpdb->writeout(FILEHANDLE);
  $tlpdb->save;
  $tlpdb->available_architectures();
  $tlpdb->add_tlpcontainer($pkg, $ziploc [, $archrefs [, $dest ]] );
  $tlpdb->add_tlpobj($tlpobj);
  $tlpdb->needed_by($pkg);
  $tlpdb->remove_package($pkg);
  $tlpdb->get_package("packagename");
  $tlpdb->list_packages;
  $tlpdb->find_file("filename");
  $tlpdb->collections;
  $tlpdb->schemes;
  $tlpdb->updmap_cfg_lines;
  $tlpdb->fmtutil_cnf_lines;
  $tlpdb->language_dat_lines;
  $tlpdb->language_def_lines;
  $tlpdb->package_revision("packagename");
  $tlpdb->location;
  $tlpdb->config_src_container;
  $tlpdb->config_doc_container;
  $tlpdb->config_container_format;
  $tlpdb->config_release;
  $tlpdb->config_revision;

  TeXLive::TLPDB->listdir([$dir]);
  $tlpdb->generate_listfiles([$destdir]);

=head1 DESCRIPTION

=cut

use TeXLive::TLConfig qw($CategoriesRegexp $DefaultCategory $InfraLocation
      $DatabaseName $MetaCategoriesRegexp);
use TeXLive::TLUtils qw(dirname mkdirhier member win32 info debug ddebug tlwarn);
use TeXLive::TLPOBJ;

my $_listdir;

=pod

=over 6

=item C<< TeXLive::TLPDB->new >>

=item C<< TeXLive::TLPDB->new( [root => "$path"] ) >>

C<< TeXLive::TLPDB->new >> creates a new C<TLPDB> object. If the
argument C<root> is given it will be initialized from the respective
location within $path. If
C<$path> begins with C<http://> or C<ftp://>, the program C<wget>
is used to download the file.
The C<$path> can start with C<file:/> in which case it is treated as
a file on the filesystem in the usual way.

Returns either an object of type C<TeXLive::TLPDB>, or undef if the root
was given but no package could be read from that location.

=cut

sub new { 
  my $class = shift;
  my %params = @_;
  my $self = {
    root => $params{'root'},
    tlps => $params{'tlps'}
  };
  $_listdir = $params{'listdir'} if defined($params{'listdir'});
  bless $self, $class;
  if (defined($self->{'root'})) {
    my $nr_packages_read = $self->from_file("$self->{'root'}/$InfraLocation/$DatabaseName");
    if ($nr_packages_read == 0) {
      # that is bad, we cannot read anything, so return undef
      return(undef);
    }
  }
  return $self;
}


sub copy {
  my $self = shift;
  my $bla = {};
  %$bla = %$self;
  bless $bla, "TeXLive::TLPDB";
  return $bla;
}

=pod

=item C<< $tlpdb->add_tlpobj($tlpobj) >>

The C<add_tlpobj> adds an object of the type TLPOBJ to the TLPDB.

=cut

sub add_tlpobj {
  my ($self,$tlp) = @_;
  $self->{'tlps'}{$tlp->name} = $tlp;
}

=pod

=item C<< $tlpdb->needed_by($pkg) >>

Returns an array of package names depending on $pkg.

=cut

sub needed_by {
  my ($self,$pkg) = @_;
  my @ret;
  # we only check collections and schemes ...
  foreach my $p ($self->list_packages) {
    my $tlp = $self->get_package($p);
    if ($tlp->category =~ m/$MetaCategoriesRegexp/) {
      foreach my $d ($tlp->depends) {
        if ($d =~ m/^(.*)\.win32$/) {
          if (member("win32", $self->available_architectures)) {
            if ($d eq $pkg) {
              push @ret, $p;
              last;
            }
          }
        } else {
          if ($d eq $pkg) {
            push @ret, $p;
            last;
          }
        }
      }
    }
  }
  return @ret;
}

=pod

=item C<< $tlpdb->remove_package($pkg) >>

Remove the package named C<$pkg> from the tlpdb. Gives a warning if the
package is not present

=cut

sub remove_package {
  my ($self,$pkg) = @_;
  if (defined($self->{'tlps'}{$pkg})) {
    delete $self->{'tlps'}{$pkg};
  } else {
    tlwarn("Cannot find package $pkg for removal in tlpdb!");
  }
}

=pod

=item C<< $tlpdb->from_file($filename) >>

The C<from_file> function initializes the C<TLPDB> in case the
root was not given at generation time.  If C<$filename> begins
with C<http://> or C<ftp://>, the program C<wget> is used to download
the file. The prefix C<file:/> is treated the expected way.

It returns the actual number of packages (TLPOBJs) read from C<$filename>.

=cut

sub from_file {
  my ($self, $path) = @_;
  if (@_ != 2) {
    die("Need a filename for initialization!");
  }
  my $root_from_path = dirname(dirname($path));
  if (defined($self->{'root'})) {
    if ($self->{'root'} ne $root_from_path) {
      tlwarn("root=$self->{'root'}, root_from_path=$root_from_path\n");
      tlwarn("Initialisation from different location as originally given.\nHope you are sure!\n");
    }
  } else {
    $self->root($root_from_path);
  }
  my $retfh;
  if ($path =~ m;^((http|ftp)://|file:\/\/*);) {
    debug("TLPDB.pm: trying to initialize from $path\n");
    # if we have lzmadec available we try the lzma file
    if (defined($::progs{'lzmadec'})) {
      # we first try the lzma compressed file
      my $tmpdir = TeXLive::TLUtils::get_system_tmpdir();
      my $bn = TeXLive::TLUtils::basename("$path");
      my $lzmafile = "$tmpdir/$bn.lzma";
      my $lzmafile_quote = $lzmafile;
      my $tlpdbfile = "$tmpdir/$bn";
      my $tlpdbfile_quote = $tlpdbfile;
      if (win32()) {
        $lzmafile  =~ s!/!\\!g;
        $tlpdbfile =~ s!/!\\!g;
      }
      $lzmafile_quote = "\"$lzmafile\"";
      $tlpdbfile_quite = "\"$tlpdbfile\"";
      debug("trying to download $path.lzma to $lzmafile\n");
      my $ret = TeXLive::TLUtils::download_file("$path.lzma", "$lzmafile");
      # better to check both, the return value AND the existence of the file
      if ($ret && (-r "$lzmafile")) {
        # ok, let the fun begin
        debug("Un-lzmaing $lzmafile to $tlpdbfile\n");
        # lzmadec *hopefully* returns 0 on success and anything else on failure
        if (!system("$::progs{'lzmadec'} < $lzmafile_quote > $tlpdbfile_quote")) {
          debug("Un-lzmaing $lzmafile did not succeed, try normally\n");
        } else {
          unlink($lzmafile);
          open $retfh, "<$tlpdbfile" or die"Cannot open $tlpdbfile!";
          debug("Ok, found the uncompressed lzma file!\n");
        }
      } 
    } else {
      debug("no lzmadec defined, not trying tlpdb.lzma ...\n");
    }
    if (!defined($retfh)) {
      debug("TLPDB: downloading $path.lzma didn't succeed, try $path\n");
      # lzma did not succeed, so try the normal file
      $retfh = TeXLive::TLUtils::download_file($path, "|");
      if (!$retfh) {
        die("Cannot open tlpdb file: $path");
      }
    }
  } else {
    open(TMP,"<$path") || die("Cannot open tlpdb file: $path");
    $retfh = \*TMP;
  }
  my $found = 0;
  my $ret = 0;
  do {
    my $tlp = TeXLive::TLPOBJ->new;
    ddebug("creating tlp from $path...\n");
    $ret = $tlp->from_fh($retfh,1);
    if ($ret) {
      $self->add_tlpobj($tlp);
      $found++;
    }
  } until (!$ret);
  if (!$found) {
    tlwarn("Cannot read any package from $path, seems not to be a TLPDB!");
  }
  return($found);
}

=pod

=item C<< $tlpdb->writeout >>

=item C<< $tlpdb->writeout(FILEHANDLE) >>

The C<writeout> function writes the database to C<STDOUT>, or 
the file handle given as argument.

=cut

sub writeout {
  my $self = shift;
  my $fd = (@_ ? $_[0] : STDOUT);
  foreach (sort keys %{$self->{'tlps'}}) {
    ddebug("tlpname = $_\n");
    ddebug("foo: ", $self->{'tlps'}{$_}->name, "\n");
    $self->{'tlps'}{$_}->writeout($fd);
    print $fd "\n";
  }
}

=pod

=item C<< $tlpdb->save >>

The C<save> functions saves the C<TLPDB> to the file which has been set
as location. If the location is undefined, die.

=cut

sub save {
  my $self = shift;
  my $path = $self->location;
  mkdirhier(dirname($path));
  open(FOO,">$path") || die("Cannot open $path for writing: $!");
  $self->writeout(\*FOO);
  close(FOO);
}

=pod

=item C<< $tlpdb->available_architectures >>

The C<available_architectures> functions returns the list of available 
architectures as set in the options section 
(i.e., using option_available_architectures)

=cut

sub available_architectures {
  my $self = shift;
  my @archs = $self->option_available_architectures;
  if (! @archs) {
    # fall back to the old method checking bin-tex\.*
    my @packs = $self->list_packages;
    map { s/^bin-tex\.// ; push @archs, $_ ; } grep(/^bin-tex\.(.*)$/, @packs);
  }
  return @archs;
}

=pod

=item C<< $tlpdb->add_tlpcontainer($pkg, $ziploc [, $archrefs [, $dest ]] ) >>

Installs the package C<$pkg> from the container files in C<$ziploc>. If
C<$archrefs> is given then it must be a reference to a list of 
architectures to be installed. If the normal (arch=all) package is
architecture dependent then all arch packages in this list are installed.
If C<$dest> is given then the files are
installed into it, otherwise into the location of the TLPDB.

Note that this procedure does NOT check for dependencies. So if your package
adds new dependencies they are not necessarily fulfilled.

=cut

sub add_tlpcontainer {
  my ($self, $package, $ziplocation, $archrefs, $dest) = @_;
  my @archs;
  require Cwd;
  if (defined($archrefs)) {
    @archs = @$archrefs;
  }
  my $cwd = getcwd();
  if ($ziplocation !~ m,^/,) {
    $ziplocation = "$cwd/$ziplocation";
  }
  my $tlpobj = $self->_add_tlpcontainer($package, $ziplocation, "all", $dest);
  if ($tlpobj->is_arch_dependent) {
    foreach (@$archrefs) {
      $self->_add_tlpcontainer($package, $ziplocation, $_, $dest);
    }
  }
}

sub _add_tlpcontainer {
  my ($self, $package, $ziplocation, $arch, $dest) = @_;
  my $unpackprog;
  my $args;
  require Cwd;
  # WARNING: If you change the location of the texlive.tlpdb this
  # has to be changed, too!!
  if (not(defined($dest))) { 
    $dest = $self->{'root'};
  }
  my $container = "$ziplocation/$package";
  if ($arch ne "all") {
    $container .= ".$arch";
  }
  if (-r "$container.zip") {
    $container .= ".zip";
    $unpackprog="unzip";
    $args="-o -qq $container -d $dest";
  } elsif (-r "$container.lzma") {
    $container .= ".lzma";
    $unpackprog="NO_IDEA_HOW_TO_UNPACK_LZMA";
    $args="NO IDEA WHAT ARGS IT NEEDS";
    die("lzma is checked for but not implemented, please edit TLPDB.pm\n");
  } else {
    die "Cannot find a package $container (.zip or .lzma) in $ziplocation\n";
  }
  tlwarn("Huuu, this needs testing and error checking!\n");
  tlwarn("Should we use -a -- adapt line endings etc?\n");
  `$unpackprog $args`;
  # we only create/add tlpobj for arch eq "all"
  if ($arch eq "all") {
    my $tlpobj = new TeXLive::TLPOBJ;
    $tlpobj->from_file("$dest/$TeXLive::TLConfig::InfraLocation/tlpobj/$package.tlpobj");
    $self->add_tlpobj($tlpobj);
    return $tlpobj;
  }
}


=pod

=item C<< $tlpdb->get_package("packagename") >> 

The C<get_package> function returns a reference to a C<TLPOBJ> object
in case its name the the argument name coincide.

=cut

sub get_package {
  my ($self,$pkg) = @_;
  if (defined($self->{'tlps'}{$pkg})) {
    return($self->{'tlps'}{$pkg});
  } else {
    return(undef);
  }
}

=pod

=item C<< $tlpdb->list_packages >>

The C<list_packages> function returns the list of all included packages.

=cut

sub list_packages {
  my $self = shift;
  return (sort keys %{$self->{'tlps'}});
}

=pod

=item C<< $tlpdb->find_file("filename") >>

The C<find_file> returns a list of packages:filename
containing a file named C<filename>.

=cut

sub find_file {
  my ($self,$fn) = @_;
  my @ret;
  foreach my $pkg ($self->list_packages) {
    my @foo = $self->get_package($pkg)->contains_file($fn);
    foreach my $f ($self->get_package($pkg)->contains_file($fn)) {
      push @ret, "$pkg:$f";
    }
  }
  return(@ret);
}

=pod

=item C<< $tlpdb->collections >>

The C<collections> function returns the list of all collections.

=cut

sub collections {
  my $self = shift;
  my @ret;
  foreach my $p ($self->list_packages) {
    if ($self->get_package($p)->category eq "Collection") {
      push @ret, $p;
    }
  }
  return @ret;
}

=pod

=item C<< $tlpdb->schemes >>

The C<collections> function returns the list of all schemes.

=cut

sub schemes {
  my $self = shift;
  my @ret;
  foreach my $p ($self->list_packages) {
    if ($self->get_package($p)->category eq "Scheme") {
      push @ret, $p;
    }
  }
  return @ret;
}



=pod

=item C<< $tlpdb->package_revision("packagename") >>

The C<package_revision> function returns the revision number of the
package named in the first argument.

=cut

sub package_revision {
  my ($self,$pkg) = @_;
  if (defined($self->{'tlps'}{$pkg})) {
    return($self->{'tlps'}{$pkg}->revision);
  } else {
    return(undef);
  }
}

=pod

=item C<< $tlpdb->generate_packagelist >>

The C<generate_packagelist> prints TeX Live package names in the object
database, together with their revisions, to the file handle given in the
first (optional) argument, or C<STDOUT> by default.  It also outputs all
available architectures as packages with revision number -1.

=cut

sub generate_packagelist {
  my $self = shift;
  my $fd = (@_ ? $_[0] : STDOUT);
  foreach (sort keys %{$self->{'tlps'}}) {
    print $fd $self->{'tlps'}{$_}->name, " ",
              $self->{'tlps'}{$_}->revision, "\n";
  }
  foreach ($self->available_architectures) {
    print $fd "$_ -1\n";
  }
}

=pod

=item C<< $tlpdb->generate_listfiles >>

=item C<< $tlpdb->generate_listfiles($destdir) >>

The C<generate_listfiles> generates the list files for the old 
installers. This function will probably go away.

=cut

sub generate_listfiles {
  my ($self,$destdir) = @_;
  if (not(defined($destdir))) {
    $destdir = TeXLive::TLPDB->listdir;
  }
  foreach (sort keys %{$self->{'tlps'}}) {
    $tlp = $self->{'tlps'}{$_};
    $self->_generate_listfile($tlp, $destdir);
  }
}

sub _generate_listfile {
  my ($self,$tlp,$destdir) = @_;
  my $listname = $tlp->name;
  my @files = $tlp->all_files;
  @files = TeXLive::TLUtils::sort_uniq(@files);
  &mkpath("$destdir") if (! -d "$destdir");
  my (@lop, @lot);
  foreach my $d ($tlp->depends) {
    my $subtlp = $self->get_package($d);
    if (defined($subtlp)) {
      if ($subtlp->is_meta_package) {
        push @lot, $d;
      } else {
        push @lop, $d;
      }
    } else {
      # speudo dependencies on $Package.ARCH can be ignored
      if ($d !~ m/\.ARCH$/) {
        tlwarn("Strange: $tlp->name depends on $d, but this package does not exists!\n");
      }
    }
  }
  open(TMP, ">$destdir/$listname") or die "Cannot open $destdir/$listname!";
  # title and size information for collections and schemes in the
  # first two lines, marked with *
	if ($tlp->category eq "Collection") {
    print TMP "*Title: ", $tlp->shortdesc, "\n";
    # collections references Packages, we have to collect the sizes of
    # all the Package-tlps included
    # What is unclear for me is HOW the size is computed for bin-*
    # packages. The collection-basicbin contains quite a lot of
    # bin-files, but the sizes for the different archs differ.
    # I guess we have to take the maximum?
    my $s = 0;
    foreach my $p (@lop) {
      my $subtlp = $self->get_package($p);
      if (!defined($subtlp)) {
        tlwarn("Strange: $listname references $p, but it is not in tlpdb");
      }
      $s += $subtlp->total_size;
    }
    # in case the collection itself ships files ...
    $s += $tlp->runsize + $tlp->srcsize + $tlp->docsize;
    print TMP "*Size: $s\n";
  } elsif ($tlp->category eq "Scheme") {
    print TMP "*Title: ", $tlp->shortdesc, "\n";
    my $s = 0;
    # schemes size includes ONLY those packages which are directly
    # included and direclty included files, not the size of the
    # included collections. But if a package is included in one of
    # the called for collections AND listed directly, we don't want
    # to count its size two times
    my (@inccol,@incpkg,@collpkg);
    # first we add all the packages tlps that are directly included
    @incpkg = @lop;
    # now we select all collections, and for all collections we
    # again select all packages of type Documentation and Package
    foreach my $c (@lot) {
      my $coll = $self->get_package($c);
      foreach my $d ($coll->depends) {
        my $subtlp = $self->get_package($d);
        if (defined($subtlp)) {
          if (!($subtlp->is_meta_package)) {
            TeXLive::TLUtils::push_uniq(\@collpkg,$d);
          }
        } else {
          tlwarn("Strange: $coll->name depends on $d, but this does not exists!\n");
        }
      }
    }
    # finally go through all packages and add the ->total_size
    foreach my $p (@incpkg) {
      if (!TeXLive::TLUtils::member($p,@collpkg)) {
        $s += $self->get_package($p)->total_size;
      }
    } 
    $s += $tlp->runsize + $tlp->srcsize + $tlp->docsize;
    print TMP "*Size: $s\n";
  }
  # dependencies and inclusion of packages
  foreach my $t (@lot) {
    # strange, schemes mark included collections via -, while collections
    # themself mark deps on other collections with +. collection are
    # never referenced in Packages
    if ($listname =~ m/^scheme/) {
      print TMP "-";
    } else {
      print TMP "+";
    }
    print TMP "$t\n";
  }
  foreach my $t (@lop) { print TMP "+$t\n"; }
  # included files
  foreach my $f (@files) { print TMP "$f\n"; }
  # also print the listfile itself
  print TMP "$destdir/$listname\n";
  # execute statements
  foreach my $e ($tlp->executes) {
    print TMP "!$e\n";
  }
  # finish
  close(TMP);
}

=pod

=item C<< $tlpdb->root([ "/path/to/installation" ]) >>

The function C<root> allows to read and set the root of the
installation. 

=cut

sub root {
  my $self = shift;
  if (@_) { $self->{'root'} = shift }
  return $self->{'root'};
}

=pod

=item C<< $tlpdb->location >>

The function C<location> returns the location of the actual C<texlive.tlpdb>
file used. Note that this is a read-only function, you cannot change 
the root of the TLPDB using this function.

=cut

sub location {
  my $self = shift;
  return "$self->{'root'}/$InfraLocation/$DatabaseName";
}

=pod

=item C<< $tlpdb->listdir >>

The function C<listdir> allows to read and set the packages variable
specifiying where generated list files are created.

=cut

sub listdir {
  my $self = shift;
  if (@_) { $_listdir = $_[0] }
  return $_listdir;
}

=pod

=item C<< $tlpdb->config_src_container >>

Returns 1 if the the texlive config option for src files splitting on 
container level is set. See Options below.

=cut

sub config_src_container {
  my $self = shift;
  if (defined($self->{'tlps'}{'00texlive.config'})) {
    foreach my $d ($self->{'tlps'}{'00texlive.config'}->depends) {
      if ($d eq "container_split_src_files") {
        return 1;
      }
    }
  }
  return 0;
}

=pod

=item C<< $tlpdb->config_doc_container >>

Returns 1 if the the texlive config option for doc files splitting on 
container level is set. See Options below.

=cut

sub config_doc_container {
  my $self = shift;
  if (defined($self->{'tlps'}{'00texlive.config'})) {
    foreach my $d ($self->{'tlps'}{'00texlive.config'}->depends) {
      if ($d eq "container_split_doc_files") {
        return 1;
      }
    }
  }
  return 0;
}

=pod

=item C<< $tlpdb->config_doc_container >>

Returns the currently set default container format. See Options below.

=cut

sub config_container_format {
  my $self = shift;
  if (defined($self->{'tlps'}{'00texlive.config'})) {
    foreach my $d ($self->{'tlps'}{'00texlive.config'}->depends) {
      if ($d =~ m!^container_format/(.*)$!) {
        return "$1";
      }
    }
  }
  return "";
}

=pod

=item C<< $tlpdb->config_release >>

Returns the currently set release. See Options below.

=cut

sub config_release {
  my $self = shift;
  if (defined($self->{'tlps'}{'00texlive.config'})) {
    foreach my $d ($self->{'tlps'}{'00texlive.config'}->depends) {
      if ($d =~ m!^release/(.*)$!) {
        return "$1";
      }
    }
  }
  return "";
}

=pod

=item C<< $tlpdb->config_revision >>

Returns the currently set revision. See Options below.

=cut

sub config_revision {
  my $self = shift;
  if (defined($self->{'tlps'}{'00texlive.config'})) {
    foreach my $d ($self->{'tlps'}{'00texlive.config'}->depends) {
      if ($d =~ m!^revision/(.*)$!) {
        return "$1";
      }
    }
  }
  return "";
}

=pod

=item C<< $tlpdb->option_XXXXX >>

Need to be documented

=cut

sub _set_option_value {
  my ($self,$key,$value) = @_;
  my $pkg = $self->{'tlps'}{'00texlive-installation.config'};
  my @newdeps;
  if (!defined($pkg)) {
    $pkg = new TeXLive::TLPOBJ;
    $pkg->name("00texlive-installation.config");
    $pkg->category("TLCore");
    push @newdeps, "$key:$value";
  } else {
    my $found = 0;
    foreach my $d ($pkg->depends) {
      if ($d =~ m!^$key:!) {
        $found = 1;
        push @newdeps, "$key:$value";
      } else {
        push @newdeps, $d;
      }
    }
    if (!$found) {
      push @newdeps, "$key:$value";
    }
  }
  $pkg->depends(@newdeps);
  $self->{'tlps'}{'00texlive-installation.config'} = $pkg;
}

sub _option_value {
  my ($self,$key) = @_;
  if (defined($self->{'tlps'}{'00texlive-installation.config'})) {
    foreach my $d ($self->{'tlps'}{'00texlive-installation.config'}->depends) {
      if ($d =~ m!^$key:(.*)$!) {
        return "$1";
      }
    }
    return "";
  }
  return;
}

sub option_available_architectures {
  my $self = shift;
  if (@_) { $self->_set_option_value("available_architectures","@_"); }
  my @archs = split ' ', $self->_option_value("available_architectures");
  return @archs;
}
sub option_create_symlinks { 
  my $self = shift; 
  if (@_) { $self->_set_option_value("opt_create_symlinks", shift); }
  return $self->_option_value("opt_create_symlinks"); 
}
sub option_install_docfiles { 
  my $self = shift;
  if (@_) { $self->_set_option_value("opt_install_docfiles", shift); }
  return $self->_option_value("opt_install_docfiles"); 
}
sub option_install_srcfiles {
  my $self = shift;
  if (@_) { $self->_set_option_value("opt_install_srcfiles", shift); }
  return $self->_option_value("opt_install_srcfiles");
}
sub option_create_formats { 
  my $self = shift; 
  if (@_) { $self->_set_option_value("opt_create_formats", shift); }
  return $self->_option_value("opt_create_formats"); 
}
sub option_paper { 
  my $self = shift; 
  if (@_) { $self->_set_option_value("opt_paper", shift); }
  return $self->_option_value("opt_paper"); 
}
sub option_location { 
  my $self = shift; 
  if (@_) { $self->_set_option_value("location", shift); }
  my $loc = $self->_option_value("location");
  if ($loc eq "__MASTER__") {
    return $self->root;
  }
  return $self->_option_value("location");
}
sub option_sys_bin {
  my $self = shift;
  if (@_) { $self->_set_option_value("opt_sys_bin", shift); }
  return $self->_option_value("opt_sys_bin");
}
sub option_sys_man {
  my $self = shift;
  if (@_) { $self->_set_option_value("opt_sys_man", shift); }
  return $self->_option_value("opt_sys_man");
}
sub option_sys_info {
  my $self = shift;
  if (@_) { $self->_set_option_value("opt_sys_info", shift); }
  return $self->_option_value("opt_sys_info");
}
sub option_platform {
  my $self = shift;
  if (@_) { $self->_set_option_value("platform", shift); }
  return $self->_option_value("platform");
}

=pod

=item C<< $tlpdb->fmtutil_cnf_lines >>

The function C<fmtutil_cnf_lines> returns the list of a fmtutil.cnf file
containing only those formats present in the installation.

=cut
sub fmtutil_cnf_lines {
  my $self = shift;
  my %fmtcnffiles;
  foreach my $p ($self->list_packages) {
    my $obj = $self->get_package ($p);
    die "No TeX Live package named $p, strange" if ! $obj;
    foreach my $e ($obj->executes) {
      if ($e =~ m/BuildFormat (.*)$/) {
        $fmtcnffiles{$1} = 1;
      } 
      # others are ignored here
    }
  }
  my @formatlines;
  foreach my $f (sort keys %fmtcnffiles) {
    open(INFILE,"<$self->{'root'}/texmf/fmtutil/format.$f.cnf")
      or tlwarn("Cannot open $self->{'root'}/texmf/fmtutil/format.$f.cnf\nThe generated fmtutil.cnf file might be incomplete!\nError: $!\n");
    @tmp = <INFILE>;
    close(INFILE);
    push @formatlines, @tmp;
  }
  return(@formatlines);
}

=item C<< $tlpdb->updmap_cfg_lines >>

The function C<updmap_cfg_lines> returns the list of a updmap.cfg file
containing only those maps present in the installation.

=cut
sub updmap_cfg_lines {
  my $self = shift;
  my %maps;
  foreach my $p ($self->list_packages) {
    my $obj = $self->get_package ($p);
    die "No TeX Live package named $p, strange" if ! $obj;
    foreach my $e ($obj->executes) {
      if ($e =~ m/addMap (.*)$/) {
        $maps{$1} = 1;
      } elsif ($e =~ m/addMixedMap (.*)$/) {
        $maps{$1} = 2;
      }
      # others are ignored here
    }
  }
  my @updmaplines;
  foreach (sort keys %maps) {
    if ($maps{$_} == 2) {
      push @updmaplines, "MixedMap $_\n";
    } else {
      push @updmaplines, "Map $_\n";
    }
  }
  return(@updmaplines);
}

=item C<< $tlpdb->language_dat_lines >>

The function C<language_dat_lines> returns the list of all
lines for language.dat that can be generated from the tlpdb.

=cut

sub language_dat_lines {
  sub make_dat_lines {
    my ($name, $lhm, $rhm, $file, @syn) = @_;
    my @ret;
    push @ret, "$name $file\n";
    foreach (@syn) {
      push @ret, "=$_\n";
    }
    return(@ret);
  }
  my $self = shift;
  my @lines = $self->_parse_hyphen_execute(\&make_dat_lines);
  return(@lines);
}

=item C<< $tlpdb->language_def_lines >>

The function C<language_def_lines> returns the list of all
lines for language.def that can be generated from the tlpdb.

=cut

sub language_def_lines {
  sub make_def_lines {
    my ($name, $lhm, $rhm, $file, @syn) = @_;
    my $exc = "";
    my @ret;
    push @ret, "\\addlanguage\{$name\}\{$file\}\{$exc\}\{$lhm\}\{$rhm\}\n";
    foreach (@syn) {
      # synonyms in language.def ???
      push @ret, "\\addlanguage\{$_\}\{$file\}\{$exc\}\{$lhm\}\{$rhm\}\n";
      #debug("Ignoring synonym $_ for $name when creating language.def\n");
    }
    return(@ret);
  }
  my $self = shift;
  my @lines = $self->_parse_hyphen_execute(\&make_def_lines);
  return(@lines);
}
    


sub _parse_hyphen_execute {
  my ($self, $coderef) = @_;
  my @langlines = ();
  
  foreach my $pkg ($self->list_packages) {
    my $obj = $self->get_package ($pkg);
    die "No TeX Live package named $pkg, too strange" if ! $obj;
    foreach my $e ($obj->executes) {
      my $first = 1;
      if ($e =~ m/AddHyphen\s+(.*)\s*/) {
        my $name;
        my $lefthyphenmin;
        my $righthyphenmin;
        my $file;
        my @synonyms;
        if ($first) {
          push @langlines, "% from $pkg:\n";
          $first = 0;
        }
        foreach my $p (split(' ', $1)) {
          my ($a, $b) = split /=/, $p;
          if ($a eq "name") { 
            die "AddHyphen line needs name=something: $pkg, $e" unless $b;
            $name = $b; next; 
          }
          if ($a eq "lefthyphenmin") { 
            # lefthyphenmin default to 3
            $lefthyphenmin = ( $b ? $b : 2 );
            next;
          }
          if ($a eq "righthyphenmin") { 
            $righthyphenmin = ( $b ? $b : 3); 
            next; 
          }
          if ($a eq "file") { 
            die "AddHyphen line needs file=something: $pkg, $e" unless $b;
            $file = $b;
            next;
          }
          if ($a eq "synonyms") {
            @synonyms = split /,/, $b;
            next;
          }
          die "Unknown language directive in $pkg: $e";
        }
        my @foo = &$coderef ($name, $lefthyphenmin, $righthyphenmin, $file, @synonyms);
        push @langlines, @foo;
      }
    }
  }
  return @langlines;
}

=back

=pod

=head1 OPTIONS

Options regarding the full TeX Live installation to be described are saved
in a package C<00texlive.config> as values of C<depend> lines. This special
package C<00texlive.config> does not contain any files, only depend lines
which set one or more of the following options:

=over 6

=item C<container_split_src_files>

=item C<container_split_doc_files>

These options specify that at container generation time the source and
documentation files for a package have been put into a separate container
named C<package.source.extension> and C<package.doc.extension>.

=item C<container_format/I<format>>

This option specifies a format for containers. The currently supported 
formats are C<lzma> and C<zip>. But note that C<zip> is untested.

=back

To set these options the respective lines should be added to
C<00texlive.config.tlpsrc>.

=head1 SEE ALSO

The modules L<TeXLive::TLPSRC>, L<TeXLive::TLPOBJ>, 
L<TeXLive::TLTREE>, L<TeXLive::TLUtils> and the
document L<Perl-API.txt> and the specification in the TeX Live
repository trunk/Master/tlpkg/doc/.

=head1 AUTHORS AND COPYRIGHT

This script and its documentation were written for the TeX Live
distribution (L<http://tug.org/texlive>) and both are licensed under the
GNU General Public License Version 2 or later.

=cut

1;

### Local Variables:
### perl-indent-level: 2
### tab-width: 2
### indent-tabs-mode: nil
### End:
# vim:set tabstop=2 expandtab: #
