/* 

tp-scroll.c

AUTHOR:
Daniel Grobe Sachs <sachs@uiuc.edu>	1 Apr 2002

MODIFIED BY:
Christoph Thiel <ct@kki.org>		14 Dec 2002


LICENSE:  
The following is a standard BSD-style license.

Copyright (c) 2002, Daniel Grobe Sachs 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.  

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <getopt.h>
#include <math.h>
#include <string.h>

#define DEF_XY_EXP 	1.6
#define DEF_Z_EXP 	1.2
#define DEF_Z_MULT 	0.02

#define DEF_INDEV	"/dev/mouse"
#define DEF_OUTDEV	"/dev/imouse"

void usage();

main(int argc, char *argv[])
{
    FILE *in, *out;

    int state;
    int b, prev_b;
    signed char x, y;
    int scroll;

    double f;
    double x_scroll_accum;
    double y_scroll_accum;

    double xy_accel_exp = DEF_XY_EXP; 
    double z_accel_exp = DEF_Z_EXP;
    double z_accel_mult = DEF_Z_MULT;

    char *indev = DEF_INDEV;
    char *outdev = DEF_OUTDEV;

    int c;
    int ch;


    while (1) {
	static struct option long_options[] =
	  {
	      {"input",  required_argument, 0, 'i'},
	      {"output",  required_argument, 0, 'o'},
	      {"vx", required_argument, 0, 'x'},
	      {"vz", required_argument, 0, 'z'},
	      {"mz", required_argument, 0, 'm'},
	      {"help", no_argument, 0, 'h'},
	      {0, 0, 0, 0}
	  };
	int option_index = 0;
	
	ch = getopt_long (argc, argv, "i:o:x:z:m:h", long_options, &option_index);
	
	/* Detect the end of the options. */
	if (ch == -1)
	    break;
	
	switch (ch) {
	    
	    case 'i':
		indev = strdup(optarg);
		break;
	  
	    case 'o':
		outdev = strdup(optarg);
		break;

	    case 'x':
		xy_accel_exp = atof(optarg);
		break;

	    case 'z':
		z_accel_exp = atof(optarg);
		break;

	    case 'm':
		z_accel_mult = atof(optarg);
		break;
	    case 'h':
		usage();
		exit(1);
	    case '?':
		usage();
		exit(1);
		break;
	    default:
		usage();
		exit(1);
	}
    }

    signal(SIGPIPE, SIG_IGN);

    while (1) {
	out = fopen(outdev, "w");
	in = fopen(indev, "r");

	if ((in == NULL) || (out == NULL)) {
	    printf("%s: unable to open files\n", argv[0]);
	    exit(1);
	}

	state = 0;
	prev_b = 0;
	scroll = 0;

	while (!feof(in) && !feof(out)) {
	    state++;
	    c = getc(in);

	    if ((state == 1) & !(c & 0x08))
		state++;

	    if (state == 1)
		b = c;

	    if (!(b & 0x04)) {
		if ((prev_b & 0x04) && (!scroll)) {
		    putc(prev_b & 0xF, out);
		    putc(0, out);
		    putc(0, out);
		    putc(0, out);
		}

		scroll = 0;
		x_scroll_accum = 0;
		y_scroll_accum = 0;
		prev_b = b;

	        if( (state == 2) || (state == 3) )
		{
			x = (signed char)c;
			y = abs(x);

			f = pow( (double)y, (double)xy_accel_exp);

			f = (f > 127) ? 127 : f;

			if( x < 0 ) 
			    y = (signed char)(-f);
			else
		    	    y = (signed char) f;

#ifdef DEBUG
			printf("%2x %i %f %i %i\n",b,state,f,(int)x,(int)y);
#endif

			putc(y, out);
		}
		else
			putc(c, out);

		if (state == 3) {
		    putc(0, out);
		    state = 0;
		    fflush(out);
		}
	    } else {
		prev_b = b;

		x = getc(in);
		y = getc(in);
		state = 0;

		if( y != 0 )
		{
		    scroll = 1;

		    f = (double)z_accel_mult * pow( (double)(abs(y)), (double)z_accel_exp);
#ifdef DEBUG
		    printf("y scroll %i %f accum %f\n",y,f,y_scroll_accum);
#endif
			
		    for( y_scroll_accum += f; y_scroll_accum > 1.0; y_scroll_accum -= 1.0 )
		    {
			putc(b & 0x0B, out);
			putc(0, out);
			putc(0, out);
		        y < 0 ? putc(0x01, out) : putc(0x0F, out);
		    }

		    fflush(out);
		}

		if( x != 0 )
		{
		    scroll = 1;

		    f = (double)z_accel_mult * pow( (double)(abs(y)), (double)z_accel_exp);
#ifdef DEBUG
		    printf("x scroll %i %f accum %f\n",y,f,x_scroll_accum);
#endif
			
		    for( x_scroll_accum += f; x_scroll_accum > 1.0; x_scroll_accum -= 1.0 )
		    {
			putc(b & 0x0B, out);
			putc(0, out);
			putc(0, out);
		        y < 0 ? putc(0x10, out) : putc(0x20, out);
		    }

		    fflush(out);
		}
	    }
	}
	fclose(in);
	fclose(out);
    }
}

void usage() {

    printf("usage:\n");
    printf("-h | --help \t output this help message\n");
    printf("-i | --input \t input-device (default: %s)\n",DEF_INDEV);
    printf("-o | --output \t pseudo-mouse-device (default: %s)\n",DEF_OUTDEV);
    printf("Acceleration:\n");
    printf("-x | --vx\t XY_ACCEL_EXP (default: %.1f)\n",DEF_XY_EXP);
    printf("-z | --vz\t Z_ACCEL_EXP (default: %.1f)\n",DEF_Z_EXP);
    printf("-m | --mz\t Z_ACCEL_MULT (default: %.2f)\n",DEF_Z_MULT);

}
