//*****************************************************************************************
// Truevision - a 3d modeler for gnome and povray
//
// scriptobj.h
//
// Christian Spoer <spoer@users.sourceforge.net>
// Copyright (C) 2005 Christian Spoer
// Copyright (C) 2000-2005 Vincent LE PRINCE
// This file is part of the TRUEVISION Package

//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */ 
//*******************************************************************************************
#ifndef TV_SCRIPTOBJ_H
#define TV_SCRIPTOBJ_H
using namespace std;
#include "object3d.h"
#include "pyengine.h"
#include <vector>

// Definition
enum _ScriptType {
        ST_INT = 0,
        ST_FLOAT,
		ST_DOUBLE,
		ST_STRING,
};
typedef _ScriptType ScriptType;

typedef struct _ScriptValue {
	ScriptType type;
	gpointer value;
	char* name;
} ScriptValue;

// Triangle
typedef struct _GLTriangle {
	float x1; float y1; float z1;
	float x2; float y2; float z2;
	float x3; float y3; float z3;
} GLTriangle;

typedef struct _GLQuad {
	float x1; float y1; float z1;
	float x2; float y2; float z2;
	float x3; float y3; float z3;
	float x4; float y4; float z4;
} GLQuad;

// ScriptObj object
class ScriptObj : public Object3D_with_material {
	#define SENDER ((ScriptObj*)data)
	friend void sign_scriptobj_edit( GtkWidget *wid, gpointer data ) { SENDER->edit(); }
	#undef SENDER

	private:
		ObjParam_point *location;
		ObjParam_scale *size;
		ObjParam_rotation *rotation;
		
		char* m_script_path;
		char* pov_pass1_string;
		char* pov_pass2_string;

		bool has_location;
		bool has_rotation;
		bool has_scale;
		bool has_material;
		bool has_editbutton;
		
		PyObject *my_pyobject;

		vector<GLTriangle*> *triangle_list;
		vector<GLQuad*> *quad_list;
		vector<ScriptValue*> *value_list;
		
	public:
		ScriptObj( app_objs *appref );
		ScriptObj( ScriptObj & ref );
		~ScriptObj();
		Object3D *duplicate_yourself() { ScriptObj *res = new ScriptObj( *this ); return res; }
		void add_to_tree( GtkWidget *view, GtkTreeStore *store, GtkTreeSelection *sel, GtkTreeIter *parent, GtkTreeIter *sibling, const gchar *pixmap = NULL ) { Object3D::add_to_tree( view, store, sel, parent, sibling, "object_default.xpm" ); }

		void display( glview *view, bool set_color = true );
		void edit_widget( GtkWidget *wid );
		void destroy_editor();
		void pref_changed();
		
		void mouse_drag( struct drag_info *drag );
		float *get_location() { return location->get(); }
		Rotation *get_rotation() { return rotation->get_rotation(); }
		
		void output_to_povray_pass1( ofstream & file );
		void save( ofstream & file );
		bool load( ifstream & file, char *tag );

		// Access methods
		virtual void set_location(float *data) { location->set(data[0], data[1], data[2]); location->update_widget();}
		void set_rotation(float *data) {rotation->set(data[0], data[1], data[2]); rotation->update_widget(); rotation->flush();}
		void set_size(float *data) { size->set(data[0], data[1], data[2]); size->update_widget();}
		float* get_rot(float* val) { float _x, _y, _z; rotation->get(_x, _y, _z); val[0] = _x; val[1] = _y; val[2] = _z; return (float*)val;}
		float* get_size() { return size->get(); }

		void set_script_path( char* sp ) { m_script_path = sp; }
		char* get_script_path() { return m_script_path; }

		void edit();
		
		void add_triangle( float, float, float,
						   float, float, float,
						   float, float, float );

		void add_quad( float, float, float,
					   float, float, float,
					   float, float, float,
					   float, float, float );

		void clear_triangles();
		void clear_quads();

		void new_int( char* _name, int _i );
		void new_float( char* _name, float _f );
		void new_double( char* _name, double _d );
		void new_string( char* _name, char* _str );

		void set_int( char* _name, int _i );
		void set_float( char* _name, float _f );
		void set_double( char* _name, double _d );
		void set_string( char* _name, char* _str );

		int get_int( char* _name );
		float get_float( char* _name );
		double get_double( char* _name );
		char* get_string( char* _name );

		void add_value( ScriptValue* v );
		ScriptValue* get_value( char* _name );
		void set_value( char* _name, gpointer v );

		void save_values( ofstream & file );
		bool load_values( ifstream & file, char* tag );
		
		void save_triangles( ofstream & file );
		bool load_triangles( ifstream & file, char* tag );

		void save_quads( ofstream & file );
		bool load_quads( ifstream & file, char* tag );

		void save_has_transform( ofstream & file );
		bool load_has_transform( ifstream & file, char* tag );
		
		void save_script_path( ofstream & file );
		bool load_script_path( ifstream & file, char* tag );

		bool load_value_name(ifstream & file, char* tag, char** _name );
		bool load_value_type(ifstream & file, char* tag, ScriptType *t );
		bool load_value_value(ifstream & file, char* tag, char** val );

		void set_pov_pass1_string( char* pp1 )  { pov_pass1_string = pp1; }
		char* get_pov_pass1_string() { return pov_pass1_string; }

		void set_pov_pass2_string( char* pp2 ) { pov_pass2_string = pp2; }
		char* get_pov_pass2_string() { return pov_pass2_string; }

		void set_has_rotation( bool yn ) { has_rotation = yn; }
		void set_has_location( bool yn ) { has_location = yn; }
		void set_has_scale( bool yn ) { has_scale = yn; }
		void set_has_material( bool yn ) { has_material = yn; }
		void set_has_editbutton( bool yn ) { has_editbutton = yn; }
		bool is_has_rotation() { return has_rotation; }
		bool is_has_location() { return has_location; }
		bool is_has_scale() { return has_scale; }
		bool is_has_material() { return has_material; }
		bool is_has_editbutton() { return has_editbutton; }
};

#endif // TV_SCRIPTOBJ_H 
