//*****************************************************************************************
// Truevision - a 3d modeler for gnome and povray
//
// cone.cc
//
// Vincent LE PRINCE <vincentleprince@users.sourceforge.net>
// Copyright (C) 2000-2005 Vincent LE PRINCE
// This file is part of the TRUEVISION Package

//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */ 
//*******************************************************************************************
#include "include/cone.h"
#include "include/viewmanager.h"
#include "include/objectlist.h"
#include "include/tvio.h"
#include "include/preferences.h"
#include <GL/glu.h>
#include <math.h>

 /* Some <math.h> files do not define M_PI... */
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

//**************************************
// Constructeur
//**************************************
Cone::Cone( app_objs *appref ) : Object3D_with_material( appref )
{
type = TV_OBJ3D_CONE;
category = TV_OBJ3D_OBJECTS;
set_name( "Cone" );

// Base
base_point = new ObjParam_point_virtual( N_("Base point"), "BPOINT", NULL, app_ref, true );
base_point->set( 0, 0, 0 );
cap_point = new ObjParam_point_virtual( N_("Cap point"), "CPOINT", NULL, app_ref, true );
cap_point->set( 0, 0.4, 0 );
base = new ObjParam_float( N_("Base radius"), "BASE", NULL, app_ref, true, 0.2 );
base->set_range( 1000, -1000, 0.1, 5 );
cap = new ObjParam_float( N_("Cap radius"), "CAP", NULL, app_ref, true, 0 );
cap->set_range( 1000, -1000, 0.1, 5 );
location = new ObjParam_point( N_("Translation"), "LOC", NULL, app_ref, true );
location->set( 0, 0, 0 );
size = new ObjParam_scale( N_("Scale"), "SIZE", NULL, app_ref, true );
size->set( 1, 1, 1 );
rotation = new ObjParam_rotation( N_("Rotation"), "ROT", NULL, app_ref, true );
rotation->set( 0, 0, 0 );
open = new ObjParam_bool( N_("Open"), "OPEN", NULL, app_ref, true, false );location = new ObjParam_point( N_("Location"), "LOC", NULL, app_ref, true );
}

Cone::Cone( Cone & ref ) : Object3D_with_material( ref )
{
location = new ObjParam_point( *ref.location );
base_point = new ObjParam_point_virtual( *ref.base_point );
cap_point = new ObjParam_point_virtual( *ref.cap_point );
base = new ObjParam_float( *ref.base );
cap = new ObjParam_float( *ref.cap );
size = new ObjParam_scale( *ref.size );
rotation = new ObjParam_rotation( *ref.rotation );
open = new ObjParam_bool( *ref.open );
}

Cone::~Cone()
{
delete location;
delete base_point;
delete cap_point;
delete base;
delete cap;
delete size;
delete rotation;
delete open;
}

//**************************************
// Display
// Dessin de la boite
//**************************************
void Cone::display( glview *view, bool set_col )
{
if ( hidden->value() ) return;
if ( location->changed() || size->changed() || rotation->changed() || base->changed()  || cap->changed() || open->changed() || base_point->changed() || cap_point->changed() )
	list.invalidate();

Object3D::display( view );		
if ( set_col ) set_color();
if ( ! list.exec() )
	{
	// mise  zero des parametres
	location->unchange();
	size->unchange();
	rotation->unchange();
	base->unchange();
	cap->unchange();
	open->unchange();
	base_point->unchange();
	cap_point->unchange();

	// creation de la liste si necessaire
	list.begin();
	glPushMatrix();

	// Position et direction
	float x, y, z, a, b, c;
	location->get( x, y, z );
	glTranslatef( x, y, z );
	size->get( x, y, z );
	glScalef( x, y, z );
	rotation->gl_rotate();

    // Def of points
	base_point->get( x, y, z );
	cap_point->get( a, b, c );
	glTranslatef( x, y, z );	
	glRotatef( atan2( a-x, c-z )*180.0/M_PI, 0, 1, 0 );	
	glRotatef( -atan2( b-y, sqrt( (c-z)*(c-z) + (a-x)*(a-x) ) )*180.0/M_PI, 1, 0, 0 );

	PREF_DEF
	int quality = 6 + pref->preview_quality->value()*6;

	float dist = sqrt( (x-a)*(x-a) + (y-b)*(y-b) + (z-c)*(z-c) );
	GLUquadricObj *quadobj = gluNewQuadric();
	if ( !quadobj ) { app_warning( N_("Out of memory" ) ); return; }	
	gluQuadricDrawStyle( quadobj, GLU_FILL );
	gluQuadricNormals( quadobj, GLU_SMOOTH );
	gluQuadricOrientation( quadobj, GLU_OUTSIDE );
    gluCylinder(quadobj, base->value(), cap->value(), dist, quality, quality );

	if ( ! open->value() )
		{
		GLUquadricObj *disk1 = gluNewQuadric();
		if ( !quadobj ) { app_warning( N_("Out of memory" ) ); return; }	
		gluQuadricDrawStyle( disk1, GLU_FILL );
		gluQuadricNormals( disk1, GLU_SMOOTH );
		gluQuadricOrientation( disk1, GLU_OUTSIDE );
	    gluDisk( disk1, 0, base->value(), quality, 6 );

	   	glTranslatef( 0, 0, dist );
		GLUquadricObj *disk2 = gluNewQuadric();
		if ( !quadobj ) { app_warning( N_("Out of memory" ) ); return; }	
		gluQuadricDrawStyle( disk2, GLU_FILL );
		gluQuadricNormals( disk2, GLU_SMOOTH );
		gluQuadricOrientation( disk2, GLU_OUTSIDE );
	    gluDisk( disk2, 0, cap->value(), 12, 6 );    	
	    }
	
	glEnd();
	glPopMatrix();
	list.end();
	}
}


//***********************************************
// Edit
//***********************************************
void Cone::edit_widget( GtkWidget *wid )
{
PREF_DEF
bool tt = pref->tooltips->value();
// Options communes
Object3D::edit_widget( wid );

// Options de geometrie
new_table( edit_cont, _("General settings"), 8 );
	base_point->get_widget( table, tt, 1 );
	cap_point->get_widget( table, tt, 2 );
	base->get_widget( table, tt, 3 );
	cap->get_widget( table, tt, 4 );
	location->get_widget( table, tt, 5 );
	size->get_widget( table, tt, 6 );
	rotation->get_widget( table, tt, 7 );
	open->get_widget( table, tt, 8 );
	
get_texture_widgets( edit_cont, tt );
gtk_widget_show_all( wid );
}

//***********************************************
// Mouse drag
//***********************************************
void Cone::mouse_drag( struct drag_info *drag )
{
VMAN_DEF
OBJLIST_DEF

switch( vmanager->get_pointer_mode() )
	{
	case TV_PMODE_SELECT:
	case TV_PMODE_TRANSLATE:
		location->mouse_drag( drag );
		break;

	case TV_PMODE_SCALE:
		{ size->mouse_drag( drag ); }
		break;

	case TV_PMODE_ROTATE:
		{ rotation->mouse_drag( drag ); break; }

	case TV_PMODE_CUSTOM:
		((ObjParam_point*)(objlist->get_current_param()))->mouse_drag( drag );
		break;

	default: 
		break;
	}
}

//***********************************************
// Pref_changed
//***********************************************
void Cone::pref_changed()
{
Object3D::pref_changed();
location->pref_changed();
size->pref_changed();
base->pref_changed();
cap->pref_changed();
rotation->pref_changed();
open->pref_changed();
base_point->pref_changed();
cap_point->pref_changed();
}

//***********************************************
// Destroy editor
//***********************************************
void Cone::destroy_editor()
{
Object3D::destroy_editor();
location->clear_widget();
size->clear_widget();
texture->clear_widget();
rotation->clear_widget();
base->clear_widget();
cap->clear_widget();
open->clear_widget();
base_point->clear_widget();
cap_point->clear_widget();
}

//***********************************************
// Output to povray
//***********************************************
void Cone::output_to_povray_pass1( ofstream & file )
{
file << "\n\n// Cone : " << name->value();
file << "\n#declare "; get_underscore_name( file ); file << " ="; 

file << "\ncone {\n\t";
float x,y,z;
base_point->get( x, y, z );
file << "<" << x << ',' << y << ',' << -z << ">, "<< base->value();
cap_point->get( x, y, z );
file << " ,<" << x << ',' << y << ',' << -z << ">, " << cap->value();
if ( open->value() ) file << "open\n\t";
Object3D_with_material::output_to_povray_pass1( file );

file <<"\t";
rotation->output_to_povray( file );
size->get( x, y, z );
file << "\n\tscale <" << x << ',' << y << ',' << z << ">";
location->get( x, y, z );
file << "\n\ttranslate <" << x << "," << y << "," << -z << ">";

file << "\n}";
}


void Cone::save( ofstream & file )
{
file << "\nCONE{\n";
save_basics( file );
base_point->save( file );
cap_point->save( file );
base->save( file );
cap->save( file );
size->save( file );
rotation->save( file );
location->save( file );
open->save( file );
texture->save( file );
file << "\n}";
}

bool Cone::load( ifstream & file, char *ltag )
{
if ( strcmp( ltag, "CONE" ) ) return false;
set_load_progress( file );

char * tag = NULL;
do {
	tag = tvio_get_next_tag( file );
	if ( tag == NULL ) break;
	
	if ( load_basics( file, tag ) ) continue;
 	if ( base_point->load( file, tag ) ) continue;
 	if ( cap_point->load( file, tag ) ) continue;	
 	if ( location->load( file, tag ) ) continue;
 	if ( size->load( file, tag ) ) continue;
 	if ( rotation->load( file, tag ) ) continue;
 	if ( base->load( file, tag ) ) continue;
 	if ( cap->load( file, tag ) ) continue;
 	if ( open->load( file, tag ) ) continue;
		
	tvio_skip_section(file );
	} while ( tag != NULL );

return true;
}
