//*****************************************************************************************
// Truevision - a 3d modeler for gnome and povray
//
// main.cc
//
// Vincent LE PRINCE <vincentleprince@users.sourceforge.net>
// Copyright (C) 2000-2005 Vincent LE PRINCE
// This file is part of the TRUEVISION Package

//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */ 
//*******************************************************************************************
#include "include/interface.h"
#include "include/preferences.h"
#include "include/scene.h"
#include "include/povfe.h"
#include "include/matlib.h"
#include "include/undo.h"
#include "include/dlgutils.h"
#include "config.h"

#ifdef HAVE_TV_PYTHON
#include "include/pyengine.h"
#include "include/pyscriptdb.h"
#endif

//*****************************************************
// Objets globaux
//*****************************************************
app_objs obj_ref;
Interface *Panel;
Preferences *Pref;
Scene *scene;
PovrayFE *povfe;
MatLib *matlib;
UndoRedoManager *undoman;

#ifdef HAVE_TV_PYTHON
PyEngine *python_engine;
PyScriptDB *pyscript_db;
#endif
//*****************************************************
// Fonction Main
//*****************************************************
int main( int argc, char * argv[], char *envp[] )
{
// Global objects references initialisations
	obj_ref.argc = argc;
	obj_ref.argv = argv;
	obj_ref.envp = envp;
	obj_ref.interf = NULL;
	obj_ref.vmanager = NULL;
	obj_ref.scene = NULL;
	obj_ref.python_engine = NULL;
	obj_ref.pyscript_db = NULL;
	
// Internationalisation
#ifdef ENABLE_NLS
	bindtextdomain( PACKAGE, PACKAGE_LOCALE_DIR );
	textdomain( PACKAGE );
#endif
	
// Command line options
	char *file_to_load = NULL;
	static const struct poptOption options[] = {
		{ "file", 'f', POPT_ARG_STRING, &file_to_load, 0,	 N_("Open file"), N_("FILE") },
		{NULL, '\0', 0, NULL, 0} /* end the list */
};
	
	
// Init
	g_thread_init(NULL);
	gdk_threads_init();
	obj_ref.app = gnome_program_init( PACKAGE, VERSION, LIBGNOMEUI_MODULE, argc, argv,  GNOME_PARAM_POPT_TABLE, options,  GNOME_PROGRAM_STANDARD_PROPERTIES, GNOME_PARAM_NONE );	
	obj_ref.file_to_load = file_to_load;
	setlocale( LC_NUMERIC, "C" );
	gtk_gl_init( &argc, &argv );
	
// Load preferences
	Pref = new Preferences( &obj_ref );
	bool disp_splash = Pref->display_splash->value();
	
// Splash screen & data loading
	SplashScreen *splash = NULL;
	if ( disp_splash ) {
		splash = new SplashScreen( obj_ref.app );	
		splash->set_status( _("Loading material library...") );
	}
	matlib = new MatLib( &obj_ref );
	if ( disp_splash )
		splash->set_status( _("Loading interface...") );
	undoman = new UndoRedoManager( &obj_ref );
	povfe = new PovrayFE( &obj_ref );
	obj_ref.povfe = povfe;
	scene = new Scene( &obj_ref );
#ifdef HAVE_TV_PYTHON
	if ( disp_splash )
		splash->set_status( _("Starting script engine...") );
	python_engine = new PyEngine( &obj_ref );

	if ( disp_splash )
		splash->set_status( _("Loading plug-ins...") );

	pyscript_db = new PyScriptDB( &obj_ref );
	pyscript_db->scan_directory(PACKAGE_PYTHON_PLUGIN_DIR);
#endif
	Panel = new Interface( &obj_ref );
	undoman->init();
	if ( disp_splash ) {
		sleep(1);  // shall not be needed in the future but it loads really to fast now !
		delete splash;
	}
	
// Main gtk loop
	if ( file_to_load != NULL )
		scene->load( file_to_load );
	gtk_main();
	return(-1);
}



//******************************************************
// Fonction app_exit()
//******************************************************
gboolean app_exit(GtkWidget *widget, GdkEvent *event, gpointer data) 
{
	if ( scene->is_modified() )
	{
		GtkWidget *dialog =  gtk_message_dialog_new ( GTK_WINDOW(Panel->get_gtkwin()), GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO, _("Current scene is unsaved, quit anyway ?"));
		gint button = gtk_dialog_run( GTK_DIALOG(dialog));
		gtk_widget_destroy( dialog );
		if ( button == GTK_RESPONSE_NO ) {
			return TRUE;
		}
	}
	scene->clear_scene();
	delete obj_ref.proppanel;
	Pref->save();
	delete obj_ref.matlist;
#ifdef HAVE_TV_PYTHON
	delete obj_ref.python_engine;
	delete obj_ref.pyscript_db;
#endif
	//if(data != NULL)
		app_destroy(NULL, NULL);
	
	return FALSE;
}

void app_destroy( GtkWidget *widget, gpointer data)
{
	gtk_main_quit();
}

void app_fatal_err( const char *message )
{
cout << "\n[TrueVision]: Fatal error -> " << message << endl;
exit(-1);
}

//*******************************************************
// Message boxes : with 1 or 2 strings
//*******************************************************
void app_warning( const char *message )
{
GtkWidget *warning_box =  gtk_message_dialog_new ( NULL, GTK_DIALOG_MODAL , GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, message);
gtk_dialog_run( GTK_DIALOG(warning_box ) );
gtk_widget_destroy( warning_box );
}

void app_warning( const char *mess1, const char *mess2 )
{
char *message = new char[ strlen(mess1) + strlen(mess2) + 1 ];
if ( message == NULL ) return;
strcpy( message, mess1 );
strcat( message, mess2 );
app_warning( message );
delete message;
}


//********************************************************
// About Box
//********************************************************
GtkWidget *dlg = NULL;;
GdkPixbuf *buffer = NULL;

void close_about( GtkWidget *widget, gpointer data ) 
{ 
gtk_widget_destroy(dlg); dlg = NULL;
if ( buffer != NULL ) g_object_unref( buffer );
buffer = NULL;
}

void about()
{
if ( GTK_IS_DIALOG(dlg) ) return;
const gchar *authors[] = APP_AUTHORS;
char *fname = tv_get_pixmap( "splash.png" );
buffer = gdk_pixbuf_new_from_file( fname, NULL );
dlg = (GtkWidget*)g_object_new (GTK_TYPE_ABOUT_DIALOG,
               "name", APP_NAME, 
                "version", VERSION,
			    "copyright", APP_COPY,
				"comments", APP_DESCR,
			    "authors", authors,
                "logo", buffer,  NULL);

g_signal_connect( G_OBJECT(dlg), "close", G_CALLBACK(close_about), NULL );
delete fname;
gtk_dialog_run( GTK_DIALOG(dlg) );
}


//********************************************************************************
// Help & homepage display
//********************************************************************************
void help_index()
{
truevision_help_display( "truevision.xml",  NULL, NULL );	
return;
}


void display_homepage() 
{ 
gnome_url_show( "http://truevision.sourceforge.net", NULL ); 
}

// Access to the Object refs
app_objs* tv_get_obj_ref()
{
	return &obj_ref;
}


void truevision_help_display( char *file, char *sect, char *sub )
{
	gnome_help_display( file, sect, NULL );
}
