//*****************************************************************************************
// Truevision - a 3d modeler for gnome and povray
//
// plane.cc
//
// Vincent LE PRINCE <vincentleprince@users.sourceforge.net>
// Copyright (C) 2000-2005 Vincent LE PRINCE
// This file is part of the TRUEVISION Package

//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
//*******************************************************************************************
#include "include/plane.h"
#include "include/viewmanager.h"
#include "include/objectlist.h"
#include "include/tvio.h"
#include "include/preferences.h"


//**************************************
// Constructeur
//**************************************
Plane::Plane( app_objs *appref ) : Object3D_with_material( appref )
{
type = TV_OBJ3D_PLANE;
category = TV_OBJ3D_OBJECTS;
set_name( "Plane" );

// Base
location = new ObjParam_point( N_("Location"), "LOC", NULL, app_ref, true );
location->set( 0, 0, 0 );
glsize = new ObjParam_int( N_("GLSize"), "SIZE", NULL, app_ref, true, 4 );
glsize->set_range( 100, 1, 1 );
rotation = new ObjParam_rotation( N_("Rotation"), "ROT", NULL, app_ref, true );
rotation->set( 0, 0, 0 );
}

Plane::Plane( Plane & ref ) : Object3D_with_material( ref )
{
location = new ObjParam_point( *ref.location );
glsize = new ObjParam_int( *ref.glsize );
rotation = new ObjParam_rotation( *ref.rotation );
}

Plane::~Plane()
{
delete location;
delete glsize;
delete rotation;
}

//**************************************
// Display
// Dessin de la boite
//**************************************
void Plane::display( glview *view, bool set_col )
{
if ( hidden->value() ) return;
if ( location->changed() || glsize->changed() || rotation->changed() )
	list.invalidate();
	
Object3D::display( view );	
if ( set_col ) set_color();
if ( ! list.exec() )
	{
	// mise  zero des parametres
	location->unchange();
	glsize->unchange();
	rotation->unchange();

	// creation de la liste si necessaire
	list.begin();
	glPushMatrix();

	
	// Position et direction
	gfloat x, y, z;
	//gfloat a, b, c;
	location->get( x, y, z );
	glTranslatef( x, y, z );
	rotation->gl_rotate();
	glBegin( GL_QUADS );

	const float pas = 0.2;	
	float size =  (float)glsize->value() / 2.0;
	for ( float i = -size ; i < size ; i += pas )
		for ( float j = -size ; j < size ; j += pas )
			{
			glNormal3f( 0, 1, 0 );
			glVertex3f( i, 0, j );
			//glNormal3f( 0, 1, 0 );
			glVertex3f( i, 0, j+pas );
			//glNormal3f( 0, 1, 0 );
			glVertex3f( i+pas, 0, j+pas );
			//glNormal3f( 0, 1, 0 );
			glVertex3f( i+pas, 0, j );
			}
	

    glEnd();
	glPopMatrix();
	list.end();
	}
}


//***********************************************
// Edit
//***********************************************
void Plane::edit_widget( GtkWidget *wid )
{
PREF_DEF
bool tt = pref->tooltips->value();
// Options communes
Object3D::edit_widget( wid );

// Options de geometrie
new_table( edit_cont, _("General settings"), 3 );
	location->get_widget( table, tt, 1 );
	rotation->get_widget( table, tt, 2 );
	glsize->get_widget( table, tt, 3 );

get_texture_widgets( edit_cont, tt );
gtk_widget_show_all( wid );
}

//***********************************************
// Mouse drag
//***********************************************
void Plane::mouse_drag( struct drag_info *drag )
{
VMAN_DEF
OBJLIST_DEF

switch( vmanager->get_pointer_mode() )
	{
	case TV_PMODE_SELECT:
	case TV_PMODE_TRANSLATE:
		location->mouse_drag( drag );
		break;

	case TV_PMODE_SCALE:
		{ glsize->mouse_drag( drag ); }
		break;

	case TV_PMODE_ROTATE:
		{ rotation->mouse_drag( drag ); break; }

	case TV_PMODE_CUSTOM:
		((ObjParam_point*)(objlist->get_current_param()))->mouse_drag( drag );
		break;

	default:
		break;
	}
}

//***********************************************
// Pref_changed
//***********************************************
void Plane::pref_changed()
{
Object3D::pref_changed();
location->pref_changed();
glsize->pref_changed();
rotation->pref_changed();
}

//***********************************************
// Destroy editor
//***********************************************
void Plane::destroy_editor()
{
Object3D::destroy_editor();
location->clear_widget();
glsize->clear_widget();
texture->clear_widget();
rotation->clear_widget();
}

//***********************************************
// Output to povray
//***********************************************
void Plane::output_to_povray_pass1( ofstream & file )
{
file << "\n\n// Plane : " << name->value();
file << "\n#declare "; get_underscore_name( file ); file << " ="; 
file << "\n\nplane {\n\t<0,1,0>, 0\n\t";
float x, y, z;
file << '\n';
Object3D_with_material::output_to_povray_pass1( file );
rotation->output_to_povray( file );
location->get( x, y, z );
file << "\n\ttranslate <" << x << "," << y << "," << -z << "> \n\t";
file << "\n}\n";
}


void Plane::save( ofstream & file )
{
file << "\nPLANE{\n";
save_basics( file );
location->save( file );
glsize->save( file );
rotation->save( file );
texture->save( file );
file << "\n}";
}

bool Plane::load( ifstream & file, char *ltag )
{
if ( strcmp( ltag, "PLANE" ) ) return false;
set_load_progress( file );

char * tag = NULL;
do {
	tag = tvio_get_next_tag( file );
	if ( tag == NULL ) break;
	
	if ( load_basics( file, tag ) ) continue;
 	if ( location->load( file, tag ) ) continue;
 	if ( glsize->load( file, tag ) ) continue;
 	if ( rotation->load( file, tag ) ) continue;
		
	tvio_skip_section(file );
	} while ( tag != NULL );

return true;
}
