//*****************************************************************************************
// Truevision - a 3d modeler for gnome and povray
//
// pylights.cc
//
// Christian Spoer <spoer@users.sourceforge.net>
// Copyright (C) 2000-2005 Vincent LE PRINCE
// This file is part of the TRUEVISION Package

//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
//*******************************************************************************************

#include "include/pyutils.h"
#include "include/pylights.h"
#include "include/main.h"

#include <string>

////////////////////////////////////////////////////////////////////
// PyPointLight::
PyObject*
PyPointLight::new_pointlight(PyObject* self, PyObject* args)
{
	PyObject *obj;
    PointLight *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "PointLight#%d",box_cnt++);	
    result = (PointLight*)new PointLight(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyPointLight::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	PointLight* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		vec = c->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyPointLight::set_location(PyObject* self, PyObject* args)
{
	PointLight* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_location( vec );
	}
	return PYVAL_NONE;
}

PyObject*
PyPointLight::get_color(PyObject* self, PyObject* args)
{
	double *vec;
	PointLight* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		vec = c->get_light_color();
		result = ConvertTuple3( (float*)vec );
	}
	Py_DECREF(object);
	return result;
}

PyObject*
PyPointLight::set_color(PyObject* self, PyObject* args)
{
	PointLight* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	double vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV_double( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_light_color( vec );
	}
	return PYVAL_NONE;
}


PyObject*
PyPointLight::set_fade(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_fade(b);
	}
	else
		printf("Object3d::set_fade: PyCObject_Check failed!\n");fflush(stdout);

	return PYVAL_NONE;	
}

PyObject*
PyPointLight::set_refraction(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_refraction(b);
	}
	else
		printf("Object3d::set_refraction: PyCObject_Check failed!\n");fflush(stdout);
	
	return PYVAL_NONE;	
}
PyObject*
PyPointLight::set_media_attenuation(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_media_attenuation(b);
	}
	else
		printf("Object3d::set_media_attenuation: PyCObject_Check failed!\n");fflush(stdout);
	return PYVAL_NONE;	
}
PyObject*
PyPointLight::set_media_interaction(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_media_interaction(b);
	}
	else
		printf("Object3d::set_media_interaction: PyCObject_Check failed!\n");fflush(stdout);
	return PYVAL_NONE;	
}
PyObject*
PyPointLight::set_parallel(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	bool b;

	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_parallel(b);
	}
	else
		printf("Object3d::set_parallel: PyCObject_Check failed!\n");fflush(stdout);
	return PYVAL_NONE;	
}

PyObject*
PyPointLight:: set_area(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_area(b);
	}
	else
		printf("Object3d::set_area: PyCObject_Check failed!\n");fflush(stdout);
	return PYVAL_NONE;	
}

PyObject*
PyPointLight::set_reflection(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_reflection(b);
	}
	else
		printf("Object3d::set_reflection: PyCObject_Check failed!\n");fflush(stdout);
	return PYVAL_NONE;	
}

PyObject*
PyPointLight::is_fade(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_fade() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	else
		printf("Object3d::is_fade: PyCObject_Check failed!\n");fflush(stdout);

	return retobj;	
}
PyObject*
PyPointLight::is_media_attenuation(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_media_attenuation() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	else
		printf("Object3d::is_media_attenuation: PyCObject_Check failed!\n");fflush(stdout);

	return retobj;	
}
PyObject*
PyPointLight::is_refraction(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_refraction() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	else
		printf("Object3d::is_refraction: PyCObject_Check failed!\n");fflush(stdout);

	return retobj;	
}	

PyObject*
PyPointLight::is_media_interaction(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_media_interaction() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	else
		printf("Object3d::is_media_interaction: PyCObject_Check failed!\n");fflush(stdout);

	return retobj;		
}

PyObject*
PyPointLight::is_parallel(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_parallel() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	else
		printf("Object3d::is_parallel: PyCObject_Check failed!\n");fflush(stdout);

	return retobj;		
}

PyObject*
PyPointLight::is_area(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_area() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	else
		printf("Object3d::is_area: PyCObject_Check failed!\n");fflush(stdout);

	return retobj;		
}

PyObject*
PyPointLight::is_reflection(PyObject* self, PyObject* args)
{
	PointLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_reflection() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	else
		printf("Object3d::is_reflection: PyCObject_Check failed!\n");fflush(stdout);

	return retobj;	
}


PyObject*
PyPointLight::get_fade_power(PyObject* self, PyObject* args)
{
	PointLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_fade_power());
	}
	return retobj;
}

PyObject*
PyPointLight::set_fade_power(PyObject* self, PyObject* args)
{
	PointLight* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		c->set_fade_power( radius );
	}
	return PYVAL_NONE;
}

PyObject*
PyPointLight::get_fade_distance(PyObject* self, PyObject* args)
{
	PointLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_fade_distance());
	}
	return retobj;
}

PyObject*
PyPointLight::set_fade_distance(PyObject* self, PyObject* args)
{
	PointLight* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<PointLight*>(PyCObject_AsVoidPtr(object));
		c->set_fade_distance( radius );
	}
	return PYVAL_NONE;
}


PyMethodDef
PyPointLight::python_methods[] = {
	{"new_pointlight", new_pointlight, METH_VARARGS, "New pointlight"},
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_color", get_color, METH_VARARGS, "Get color" },
	{"set_color", set_color, METH_VARARGS, "Set color" },
	{"is_fade", is_fade, METH_VARARGS, "Check fading"},
	{"set_fade", set_fade, METH_VARARGS, "Set if object is fade"},
	{"is_media_interaction", is_media_interaction, METH_VARARGS, "Check media_interactionness"},
	{"set_media_interaction", set_media_interaction, METH_VARARGS, "Set if object is hidden"},
	{"is_media_attenuation", is_media_attenuation, METH_VARARGS, "Check "},
	{"set_media_attenuation", set_media_attenuation, METH_VARARGS, "Set if object is hidden"},
	{"is_area", is_area, METH_VARARGS, "Check "},
	{"set_area", set_area, METH_VARARGS, "Set if object is hidden"},
	{"is_refraction", is_refraction, METH_VARARGS, "Check "},
	{"set_refraction", set_refraction, METH_VARARGS, "Set if object is hidden"},
	{"is_reflection", is_reflection, METH_VARARGS, "Check "},
	{"set_reflection", set_reflection, METH_VARARGS, "Set if object is hidden"},
	{"is_parallel", is_parallel, METH_VARARGS, "Check "},
	{"set_parallel", set_parallel, METH_VARARGS, "Set if object is hidden"},
	{"get_fade_distance", get_fade_distance, METH_VARARGS, "Get fade distance"},
	{"set_fade_distance", set_fade_distance, METH_VARARGS, "Set fade distance"},
	{"get_fade_power", get_fade_power, METH_VARARGS, "Get fade power"},
	{"set_fade_power", set_fade_power, METH_VARARGS, "Set fade power"},
	{NULL, NULL, 0, NULL}
};

PyObject*
PyAreaLight::new_arealight(PyObject* self, PyObject* args)
{
	PyObject *obj;
    AreaLight *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "AreaLight#%d",box_cnt++);	
    result = (AreaLight*)new AreaLight(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyAreaLight::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	AreaLight* b = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		b->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}
	
PyObject*
PyAreaLight::set_rotation(PyObject* self, PyObject* args)
{
	AreaLight* b = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		b->set_rotation( vec );
	}
	return PYVAL_NONE;
}

PyObject* 
PyAreaLight::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	AreaLight* b = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		vec = b->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject* 
PyAreaLight::set_size(PyObject* self, PyObject* args)
{
	AreaLight* b = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		b->set_size( vec );
	}
	return PYVAL_NONE;
}

PyObject*
PyAreaLight::get_size_z(PyObject* self, PyObject* args)
{
	AreaLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("i", c->get_size_z());
	}
	return retobj;
}

PyObject*
PyAreaLight::set_size_z(PyObject* self, PyObject* args)
{
	AreaLight* c = NULL;
	PyObject *object;
	int sz;
	PyArg_ParseTuple(args, "Oi", &object, &sz);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		c->set_size_z( sz );
	}
	return PYVAL_NONE;
}
PyObject* 
PyAreaLight::get_size_x(PyObject* self, PyObject* args)
{
	AreaLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("i", c->get_size_x());
	}
	return retobj;
}

PyObject* 
PyAreaLight::set_size_x(PyObject* self, PyObject* args)
{
	AreaLight* c = NULL;
	PyObject *object;
	int sx;
	PyArg_ParseTuple(args, "Oi", &object, &sx);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		c->set_size_x( sx );
	}
	return PYVAL_NONE;
}
PyObject* 
PyAreaLight::get_adaptive(PyObject* self, PyObject* args)
{
	AreaLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("i", c->get_adaptive());
	}
	return retobj;
}

PyObject* 
PyAreaLight::set_adaptive(PyObject* self, PyObject* args)
{
	AreaLight* c = NULL;
	PyObject *object;
	int adap;
	PyArg_ParseTuple(args, "Oi", &object, &adap);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		c->set_adaptive( adap );
	}
	return PYVAL_NONE;
}
PyObject* 
PyAreaLight::is_jitter(PyObject* self, PyObject* args)
{
	AreaLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_jitter() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;		
}

PyObject* 
PyAreaLight::set_jitter(PyObject* self, PyObject* args)
{
	AreaLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_jitter(b);
	}
	return PYVAL_NONE;	
}
	
PyObject* 
PyAreaLight::is_circular(PyObject* self, PyObject* args)
{
	AreaLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_circular() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;		
}

PyObject* 
PyAreaLight::set_circular(PyObject* self, PyObject* args)
{
	AreaLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_circular(b);
	}
	return PYVAL_NONE;	
}
	
PyObject* 
PyAreaLight::is_parallel(PyObject* self, PyObject* args)
{
	AreaLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_parallel() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;		
}

PyObject* 
PyAreaLight::set_parallel(PyObject* self, PyObject* args)
{
	AreaLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_parallel(b);
	}
	return PYVAL_NONE;	
}
	
PyObject* 
PyAreaLight::is_orient(PyObject* self, PyObject* args)
{
	AreaLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_orient() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;		
}
PyObject* 
PyAreaLight::set_orient(PyObject* self, PyObject* args)

{
	AreaLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<AreaLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_orient(b);
	}
	return PYVAL_NONE;	
}

PyMethodDef
PyAreaLight::python_methods[] = {
	{"new_arealight", new_arealight, METH_VARARGS, "New arealight"},
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"get_size_z", get_size_z, METH_VARARGS, "Get size z" },
	{"set_size_z", set_size_z, METH_VARARGS, "Set size z" },
	{"get_size_x", get_size_x, METH_VARARGS, "Get size x" },
	{"set_size_x", set_size_x, METH_VARARGS, "Set size x" },
	{"get_adaptive", get_adaptive, METH_VARARGS, "Get adaptive" },
	{"set_adaptive", set_adaptive, METH_VARARGS, "Set adaptive" },
	{"is_jitter", is_jitter, METH_VARARGS, "Check jitter"},
	{"set_jitter", set_jitter, METH_VARARGS, "Set jitter"},
	{"is_orient", is_orient, METH_VARARGS, "Check orient"},
	{"set_orient", set_orient, METH_VARARGS, "Set orient"},
	{"is_circular", is_circular, METH_VARARGS, "Check circular"},
	{"set_circular", set_circular, METH_VARARGS, "Set circular"},
	{"is_parallel", is_parallel, METH_VARARGS, "Check parallel"},
	{"set_parallel", set_parallel, METH_VARARGS, "Set if object is parallel"},
	{NULL, NULL, 0, NULL}
};

////////////////////////////////////////////////////////////////////
// PySpotLight::

PyObject*
PySpotLight::new_spotlight(PyObject* self, PyObject* args)
{
	PyObject *obj;
    SpotLight *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "SpotLight#%d",box_cnt++);	
    result = (SpotLight*)new SpotLight(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PySpotLight::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	SpotLight* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		vec = c->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PySpotLight::set_location(PyObject* self, PyObject* args)
{
	SpotLight* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_location( vec );
	}
	return PYVAL_NONE;
}

PyObject*
PySpotLight::get_color(PyObject* self, PyObject* args)
{
	double *vec;
	SpotLight* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		vec = c->get_light_color();
		result = ConvertTuple3( (float*)vec );
	}
	Py_DECREF(object);
	return result;
}

PyObject*
PySpotLight::set_color(PyObject* self, PyObject* args)
{
	SpotLight* c = NULL;
	PyObject *object;
	PyObject *pyvec;
	double vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV_double( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_light_color( vec );
	}
	return PYVAL_NONE;
}

PyObject*
PySpotLight::set_fade(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_fade(b);
	}
	return PYVAL_NONE;	
}

PyObject*
PySpotLight::set_refraction(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_refraction(b);
	}	
	return PYVAL_NONE;	
}
PyObject*
PySpotLight::set_media_attenuation(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	bool b;
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_media_attenuation(b);
	}
	return PYVAL_NONE;	
}
PyObject*
PySpotLight::set_media_interaction(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	bool b;
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_media_interaction(b);
	}
	return PYVAL_NONE;	
}
PyObject*
PySpotLight::set_parallel(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	bool b;
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_parallel(b);
	}
	return PYVAL_NONE;	
}

PyObject*
PySpotLight::set_area(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	bool b;
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_area(b);
	}
	return PYVAL_NONE;	
}

PyObject*
PySpotLight::set_reflection(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	bool b;
		PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_reflection(b);
	}
	return PYVAL_NONE;	
}

PyObject*
PySpotLight::is_fade(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_fade() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PySpotLight::is_media_attenuation(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_media_attenuation() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}
PyObject*
PySpotLight::is_refraction(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_refraction() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}	

PyObject*
PySpotLight::is_media_interaction(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_media_interaction() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;		
}

PyObject*
PySpotLight::is_parallel(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_parallel() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;		
}

PyObject*
PySpotLight::is_area(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_area() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;		
}

PyObject*
PySpotLight::is_reflection(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_reflection() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}


PyObject*
PySpotLight::get_fade_power(PyObject* self, PyObject* args)
{
	SpotLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_fade_power());
	}
	return retobj;
}

PyObject*
PySpotLight::set_fade_power(PyObject* self, PyObject* args)
{
	SpotLight* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		c->set_fade_power( radius );
	}
	return PYVAL_NONE;
}

PyObject*
PySpotLight::get_fade_distance(PyObject* self, PyObject* args)
{
	SpotLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_fade_distance());
	}
	return retobj;
}

PyObject*
PySpotLight::set_fade_distance(PyObject* self, PyObject* args)
{
	SpotLight* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		c->set_fade_distance( radius );
	}
	return PYVAL_NONE;
}

PyObject*
PySpotLight::get_tightness(PyObject* self, PyObject* args)
{
	SpotLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("i", c->get_tightness());
	}
	return retobj;
}

PyObject*
PySpotLight::set_tightness(PyObject* self, PyObject* args)
{
	SpotLight* c = NULL;
	PyObject *object;
	int tig;
	PyArg_ParseTuple(args, "Oi", &object, &tig);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		c->set_tightness( tig );
	}
	return PYVAL_NONE;
}

PyObject*
PySpotLight::get_point_at(PyObject* self, PyObject* args)
{
	Vector3D vec;
	SpotLight* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		vec = c->get_point_at();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PySpotLight::set_point_at(PyObject* self, PyObject* args)
{
	SpotLight* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_point_at( vec );
	}
	return PYVAL_NONE;
}

PyObject*
PySpotLight::is_show_frustum(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_show_frustum() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PySpotLight::set_show_frustum(PyObject* self, PyObject* args)
{
	SpotLight* obj3d;
	bool b;
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		obj3d->set_show_frustum(b);
	}
	return PYVAL_NONE;	
}

PyObject*
PySpotLight::get_radius(PyObject* self, PyObject* args)
{
	SpotLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_radius());
	}
	return retobj;
}

PyObject*
PySpotLight::set_radius(PyObject* self, PyObject* args)
{
	SpotLight* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		c->set_radius( radius );
	}
	return PYVAL_NONE;
}


PyObject*
PySpotLight::get_falloff(PyObject* self, PyObject* args)
{
	SpotLight *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_falloff());
	}
	return retobj;
}

PyObject*
PySpotLight::set_falloff(PyObject* self, PyObject* args)
{
	SpotLight* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<SpotLight*>(PyCObject_AsVoidPtr(object));
		c->set_falloff( radius );
	}
	return PYVAL_NONE;
}

PyMethodDef
PySpotLight::python_methods[] = {
	{"new_spotlight", new_spotlight, METH_VARARGS, "New spotlight"},
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_color", get_color, METH_VARARGS, "Get color" },
	{"set_color", set_color, METH_VARARGS, "Set color" },
	{"is_fade", is_fade, METH_VARARGS, "Is object is fade"},
	{"set_fade", set_fade, METH_VARARGS, "Set if object is fade"},
	{"is_media_interaction", is_media_interaction, METH_VARARGS, "Check media_interactionness"},
	{"set_media_interaction", set_media_interaction, METH_VARARGS, "Set if object is hidden"},
	{"is_media_attenuation", is_media_attenuation, METH_VARARGS, "Check "},
	{"set_media_attenuation", set_media_attenuation, METH_VARARGS, "Set if object is hidden"},
	{"is_area", is_area, METH_VARARGS, "Check "},
	{"set_area", set_area, METH_VARARGS, "Set if object is hidden"},
	{"is_refraction", is_refraction, METH_VARARGS, "Check "},
	{"set_refraction", set_refraction, METH_VARARGS, "Set if object is hidden"},
	{"is_reflection", is_reflection, METH_VARARGS, "Check "},
	{"set_reflection", set_reflection, METH_VARARGS, "Set if object is hidden"},
	{"is_parallel", is_parallel, METH_VARARGS, "Check "},
	{"set_parallel", set_parallel, METH_VARARGS, "Set if object is hidden"},
	{"is_show_frustum", is_show_frustum, METH_VARARGS, "Check "},
	{"set_ahow_frustum", set_show_frustum, METH_VARARGS, "Set if object is hidden"},
	{"get_fade_distance", get_fade_distance, METH_VARARGS, "Get fade distance"},
	{"set_fade_distance", set_fade_distance, METH_VARARGS, "Set fade distance"},
	{"get_fade_power", get_fade_power, METH_VARARGS, "Get fade power"},
	{"set_fade_power", set_fade_power, METH_VARARGS, "Set fade power"},
	{"get_radius", get_radius, METH_VARARGS, "Get radius"},
	{"set_radius", set_radius, METH_VARARGS, "Set radius"},
	{"get_falloff", get_falloff, METH_VARARGS, "Get falloff"},
	{"set_falloff", set_falloff, METH_VARARGS, "Set falloff"},
	{"get_point_at", get_point_at, METH_VARARGS, "Get point_at"},
	{"set_point_at", set_point_at, METH_VARARGS, "Set point_at"},
	{"get_tightness", get_tightness, METH_VARARGS, "Get tightness"},
	{"set_tightness", set_tightness, METH_VARARGS, "Set tightness"},
	{NULL, NULL, 0, NULL}
};


////////////////////////////////////////////////////////////////////
// PySpotLight::

PyObject*
PyCylindricalLight::new_cylindricallight(PyObject* self, PyObject* args)
{
	PyObject *obj;
    CylindricalLight *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "CylLight#%d",box_cnt++);	
    result = (CylindricalLight*)new CylindricalLight(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}
PyMethodDef
PyCylindricalLight::python_methods[] = {
	{"new_cylindricallight", new_cylindricallight, METH_VARARGS, "New cylindrical light"},
	{NULL, NULL, 0, NULL}
};
