//*****************************************************************************************
// Truevision - a 3d modeler for gnome and povray
//
// pyobjects.cc
//
// Christian Spoer <spoer@users.sourceforge.net>
// Copyright (C) 2000-2005 Vincent LE PRINCE
// This file is part of the TRUEVISION Package

//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
//*******************************************************************************************

#include "include/pyutils.h"
#include "include/pyobjects.h"
#include "include/main.h"
#include "include/pyscriptdb.h"

#include <string>

PyObject*
PyApp::exit_app(PyObject* self, PyObject* args)
{
	printf("Exit called by Python\n");
	exit(0);
	return NULL;
}

PyObject*
PyApp::warning_msg( PyObject* self, PyObject* args )
{
	char *msg;
	
	if ( PyArg_ParseTuple( args, "s", &msg ) )
        app_warning( msg );

	return Py_BuildValue("s", msg);
}

PyObject*
PyApp::load(PyObject* self, PyObject* args)
{
	Scene *sc;
	char* fname;
	app_objs* object_ref = tv_get_obj_ref();
	
	if ( !PyArg_ParseTuple( args, "s", &fname ) ) {
        app_warning( "Python: Could not parse filename." );
		return NULL;
	}
	sc = (Scene*)object_ref->scene;
	sc->load( fname );
	//return NULL;
	return Py_BuildValue("s", fname);
}

PyObject*
PyApp::save(PyObject* self, PyObject* args)
{
	return NULL;
}

PyObject*
PyApp::register_script( PyObject* self, PyObject* args )
{
	char* script_path;
	char* menu_path;
	char* description;
	char* author;
	char* copyright;
	char* year;
	char* object_types;
	char* help_message;
	PyScriptDB *m_pyscript_db = NULL;
	app_objs* object_ref;
	object_ref = tv_get_obj_ref();

	m_pyscript_db = (PyScriptDB*)(object_ref->pyscript_db);
	
	if ( PyArg_ParseTuple( args, "ssssssss", &script_path, &description, &help_message, &author,
						   &copyright, &year, &object_types, &menu_path) ) {
		PyScript* ps = new PyScript();
		ps->set_script_path( strdup( script_path ) );
		ps->set_description( strdup( description ) );
		ps->set_help_message( strdup( help_message ) );
		ps->set_author( strdup( author ) );
		ps->set_copyright( strdup( copyright ) );
		ps->set_year( strdup( year ) );
		ps->set_object_types( strdup( object_types ) );
		ps->set_menupath( strdup( menu_path ) );
		
		m_pyscript_db->register_script( ps );
	}else {
		app_warning( "Register: Could not parse script_name or menu_path." );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyMethodDef
PyApp::python_methods[] = {
	{"exit_app", exit_app, METH_VARARGS, "Exit application" },
	{"warning_msg", warning_msg, METH_VARARGS, "Warning message" },
	{"load", load, METH_VARARGS, "Load the scene" },
	{"save", save, METH_VARARGS, "Save the scene" },
	{"register_script", register_script, METH_VARARGS, "Register a script" },
	{NULL, NULL, 0, NULL}
};

////////////////////////////////////////////////////////////////////
// PyObject3D::
PyObject*
PyObject3D::set_name(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	char* objname;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Os", &object, &objname);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_name(objname);
	}
	else
		printf("Object3d::get_name: PyCObject_Check failed!\n");fflush(stdout);
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::get_name(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = StringToPyVal(obj3d->get_name());
	}
	else
		printf("Object3d::get_name: PyCObject_Check failed!\n");fflush(stdout);

	return retobj;	
}

PyObject*
PyObject3D::set_hidden(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_hidden(b);
	}
	else {
		printf("Object3d::set_hidden: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::set_no_shadow(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_no_shadow(b);
	}
	else {
		printf("Object3d::set_no_shadow: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}
PyObject*
PyObject3D::set_render(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_render(b);
	} else {
		printf("Object3d::set_render: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}
PyObject*
PyObject3D::set_hollow(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_hollow(b);
	} else {
		printf("Object3d::set_hollow: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}
PyObject*
PyObject3D::set_inverse(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	bool b;

	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_inverse(b);
	} else {
		printf("Object3d::set_inverse: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D:: set_no_image(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_no_image(b);
	} else {
		printf("Object3d::set_no_image: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::set_no_reflection(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_no_reflection(b);
	} else {
		printf("Object3d::set_no_reflection: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::set_double_illuminate(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->set_double_illuminate(b);
	} else {
		printf("Object3d::set_double_illuminate: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}


PyObject*
PyObject3D::is_hidden(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	Py_INCREF(object);
	PyArg_ParseTuple(args, "O", &object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_hidden() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_hidden: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;	
}
PyObject*
PyObject3D::is_render(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_render() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_render: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;	
}
PyObject*
PyObject3D::is_no_shadow(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_no_shadow() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_no_shadow: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;	
}	

PyObject*
PyObject3D::is_hollow(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_hollow() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_hollow: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;		
}

PyObject*
PyObject3D::is_inverse(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_inverse() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_inverse: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;		
}

PyObject*
PyObject3D::is_no_image(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_no_image() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_no_image: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;		
}

PyObject*
PyObject3D::is_no_reflection(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_no_reflection() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_no_reflection: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;	
}

PyObject*
PyObject3D::is_double_illuminate(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_double_illuminate() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_double_illuminate: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;		
}
PyObject*
PyObject3D::select(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *object;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->select();
	} else {
		printf("Object3d::select: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyObject3D::unselect(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *object;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		obj3d->unselect();
	} else {
		printf("Object3d::unselect: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyObject3D::is_selected(PyObject* self, PyObject* args)
{
	Object3D* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_selected() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3d::is_selected: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;		
}

PyObject*
PyObject3D::set_parent(PyObject* self, PyObject* args)
{
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::get_parent(PyObject* self, PyObject* args)
{
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::set_layer(PyObject* self, PyObject* args)
{
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::get_layer(PyObject* self, PyObject* args)
{
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}	

PyObject*
PyObject3D::set_type(PyObject* self, PyObject* args)
{
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::get_type(PyObject* self, PyObject* args)
{
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::set_category(PyObject* self, PyObject* args)
{
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3D::get_category(PyObject* self, PyObject* args)
{
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyMethodDef
PyObject3D::python_methods[] = {
	{"set_name", set_name, METH_VARARGS, "Set name of the object" },
	{"get_name", get_name, METH_VARARGS, "Get name of the object"},
	{"is_hidden", is_hidden, METH_VARARGS, "Check if object is hidden"},
	{"set_hidden", set_hidden, METH_VARARGS, "Set if object is hidden"},
	{"is_hollow", is_hollow, METH_VARARGS, "Check hollowness"},
	{"set_hollow", set_hollow, METH_VARARGS, "Set if object is hidden"},
	{"is_render", is_render, METH_VARARGS, "Check "},
	{"set_render", set_render, METH_VARARGS, "Set if object is hidden"},
	{"is_no_image", is_no_image, METH_VARARGS, "Check "},
	{"set_no_image", set_no_image, METH_VARARGS, "Set if object is hidden"},
	{"is_no_shadow", is_no_shadow, METH_VARARGS, "Check "},
	{"set_no_shadow", set_no_shadow, METH_VARARGS, "Set if object is hidden"},
	{"is_double_illuminate", is_double_illuminate, METH_VARARGS, "Check "},
	{"set_double_illuminate", set_double_illuminate, METH_VARARGS, "Set if object is hidden"},
	{"is_no_reflection", is_no_reflection, METH_VARARGS, "Check "},
	{"set_no_reflection", set_no_reflection, METH_VARARGS, "Set if object is hidden"},
	{"is_inverse", is_inverse, METH_VARARGS, "Check "},
	{"set_inverse", set_inverse, METH_VARARGS, "Set if object is hidden"},
	{"is_selected", is_selected, METH_VARARGS, "Is selected"},
	{"select", select, METH_VARARGS, "select"},
	{"unselect", unselect, METH_VARARGS, "unselect"},
	{NULL, NULL, 0, NULL}
};

//////////////////////////////////////////////////
// PyObject3DMaterial

PyObject*
PyObject3DMaterial::is_target(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_target() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3dMaterial::is_target: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;	
}

PyObject* 
PyObject3DMaterial::is_refraction(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_refraction() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3dMaterial::is_refraction: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;
}

PyObject*
PyObject3DMaterial::is_reflection(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_reflection() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3dMaterial::is_reflection: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;
}

PyObject* 
PyObject3DMaterial::is_collect(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_collect() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3dMaterial::is_collect: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;
}

PyObject* 
PyObject3DMaterial::is_pass_through(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		retobj = obj3d->is_pass_through() ? PYVAL_TRUE : PYVAL_FALSE;
	} else {
		printf("Object3dMaterial::is_pass_through: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;
}

PyObject*
PyObject3DMaterial:: get_spacing(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", obj3d->get_spacing());
	} else {
		printf("Object3dMaterial::is_pass_through: PyCObject_Check failed!\n");fflush(stdout);
	}
	return retobj;
}

PyObject*
PyObject3DMaterial:: set_target(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		obj3d->set_target(b);
	} else {
		printf("Object3dMaterial::set_target: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}
PyObject* 
PyObject3DMaterial::set_refraction(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		obj3d->set_refraction(b);
	} else {
		printf("Object3dMaterial::set_refraction: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject* 
PyObject3DMaterial::set_reflection(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		obj3d->set_reflection(b);
	} else {
		printf("Object3dMaterial::set_reflection: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3DMaterial:: set_collect(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		obj3d->set_collect(b);
	} else {
		printf("Object3dMaterial::set_collect: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject* 
PyObject3DMaterial::set_pass_through(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		obj3d = reinterpret_cast<Object3D_with_material*>(PyCObject_AsVoidPtr(object));
		obj3d->set_pass_through(b);
	} else {
		printf("Object3dMaterial::set_pass_through: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyObject3DMaterial::set_spacing(PyObject* self, PyObject* args)
{
	Object3D_with_material* obj3d = NULL;
	PyObject *object;
	float spacing;
	PyArg_ParseTuple(args, "Of", &object, &spacing);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		obj3d = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		obj3d->set_spacing( spacing );
	} else {
		printf("Object3DMaterial::set_spacing: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}
	
PyMethodDef
PyObject3DMaterial::python_methods[]= {
	{"is_target", is_target, METH_VARARGS, "Check if object is target"},
	{"is_refraction", is_refraction, METH_VARARGS, "Check if object refracts"},
	{"is_reflection", is_reflection, METH_VARARGS, "Check if object reflects"},
	{"is_collect", is_collect, METH_VARARGS, "Check if object collects photons"},
	{"is_pass_through", is_pass_through, METH_VARARGS, "Check if photons pass through object"},
	{"set_target", set_target, METH_VARARGS, "Set if object is target"},
	{"set_refraction", set_refraction, METH_VARARGS, "Set if object refracts"},
	{"set_reflection", set_reflection, METH_VARARGS, "Set if object reflects"},
	{"set_collect", set_collect, METH_VARARGS, "Set if object collects photons"},
	{"set_pass_through", set_pass_through, METH_VARARGS, "Set if photons pass through object"},
	{"set_spacing", set_spacing, METH_VARARGS, "Set spacing"},
	{"get_spacing", get_spacing, METH_VARARGS, "Get spacing"},
	{NULL, NULL, 0, NULL}
};




/////////////////////////////////////////////////////////////////
//PyBox
PyObject*
PyBox::new_box(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Box *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "Box#%d",box_cnt++);	
    result = (Box*)new Box(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyBox::access_box(PyObject* self, PyObject* args)
{
	PyObject* object;
	
	PyArg_ParseTuple(args, "O", &object);
	Box* b = NULL;
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
	}
	return object;
}

PyObject*
PyBox::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Box* b = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		vec = b->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyBox::set_location(PyObject* self, PyObject* args)
{
	Box* b = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		b->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBox::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Box* b = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		vec = b->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyBox::set_size(PyObject* self, PyObject* args)
{
	Box* b = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		//printf("Set Size: Vec %f, %f, %f\n",vec[0], vec[1], vec[2]);fflush(stdout);
		b->set_size( vec );
	} else {
		printf("PyBox::set_size: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBox::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Box* b = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		b->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
 	} else { 
		printf("PyBox::get_rotation: PyCObject_Check failed!\n");fflush(stdout);
	}
	return result;
}

PyObject*
PyBox::set_rotation(PyObject* self, PyObject* args)
{
	Box* b = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		b->set_rotation( vec );
	} else {
		printf("PyBox::set_rotation: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBox::get_corner1(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Box* b = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		vec = b->get_corner1();
		result = ConvertTuple3( vec );
	} else {
		printf("PyBox::get_corner1: PyCObject_Check failed!\n");fflush(stdout);
	}
	return result;
}

PyObject*
PyBox::set_corner1(PyObject* self, PyObject* args)
{
	Box* b = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		b->set_corner1( vec );
	} else {
		printf("PyBox::set_corner1: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBox::get_corner2(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Box* b = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		vec = b->get_corner2();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyBox::set_corner2(PyObject* self, PyObject* args)
{
	Box* b = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		b = reinterpret_cast<Box*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		b->set_corner2( vec );
	} else {
		printf("PyBox::set_corner2: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyMethodDef
PyBox::python_methods[] = {
	{"new_box", new_box, METH_VARARGS, "Create new box object" },
	{"access_box", access_box, METH_VARARGS, "Warning message" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_corner1", get_corner1, METH_VARARGS, "Get first corner" },
	{"get_corner2", get_corner2, METH_VARARGS, "Get second corner" },
	{"set_corner1", set_corner1, METH_VARARGS, "Set first corner" },
	{"set_corner2", set_corner2, METH_VARARGS, "Set second corner" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{NULL, NULL, 0, NULL}
};

///////////////////////////////////////////////////////////
// PySphere

PyObject*
PySphere::new_sphere(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Sphere *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "Sphere#%d",box_cnt++);	
    result = (Sphere*)new Sphere(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PySphere::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Sphere* s = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		s = reinterpret_cast<Sphere*>(PyCObject_AsVoidPtr(object));
		vec = s->get_location();
		printf("Get: Vec %f, %f, %f\n",vec[0], vec[1], vec[2]);fflush(stdout);
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PySphere::set_location(PyObject* self, PyObject* args)
{
	Sphere* s = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		s = reinterpret_cast<Sphere*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		s->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PySphere::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Sphere* s = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		s = reinterpret_cast<Sphere*>(PyCObject_AsVoidPtr(object));
		vec = s->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PySphere::set_size(PyObject* self, PyObject* args)
{
	Sphere* s = NULL;
	PyObject *object;
	PyObject *pyvec;
	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		s = reinterpret_cast<Sphere*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		s->set_size( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PySphere::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Sphere* s = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		s = reinterpret_cast<Sphere*>(PyCObject_AsVoidPtr(object));
		s->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PySphere::set_rotation(PyObject* self, PyObject* args)
{
	Sphere* s = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		s = reinterpret_cast<Sphere*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		s->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PySphere::get_radius(PyObject* self, PyObject* args)
{
	Sphere *s = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		s = reinterpret_cast<Sphere*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", s->get_radius());
	}
	return retobj;
}

PyObject*
PySphere::set_radius(PyObject* self, PyObject* args)
{
	Sphere* s = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		s = reinterpret_cast<Sphere*>(PyCObject_AsVoidPtr(object));
		s->set_radius( radius );
	} else {
		printf("Object3DMaterial::set_spacing: PyCObject_Check failed!\n");fflush(stdout);
	}
	Py_INCREF(PYVAL_NONE);	
	return PYVAL_NONE;
}

PyMethodDef
PySphere:: python_methods[] = {
	{"new_sphere", new_sphere, METH_VARARGS, "Create new sphere object" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_radius", get_radius, METH_VARARGS, "Get radius" },
	{"set_radius", set_radius, METH_VARARGS, "Set radius" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{NULL, NULL, 0, NULL}
};


PyObject*
PyCone::new_cone(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Cone *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "Cone#%d",box_cnt++);	
    result = (Cone*)new Cone(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyCone::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Cone* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		vec = c->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyCone::set_location(PyObject* self, PyObject* args)
{
	Cone* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCone::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Cone* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		vec = c->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyCone::set_size(PyObject* self, PyObject* args)
{
	Cone* c = NULL;
	PyObject *object;
	PyObject *pyvec;
	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_size( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCone::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Cone* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		c->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PyCone::set_rotation(PyObject* self, PyObject* args)
{
	Cone* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCone::get_cap_radius(PyObject* self, PyObject* args)
{
	Cone *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_cap_radius());
	}
	return retobj;
}

PyObject*
PyCone::set_cap_radius(PyObject* self, PyObject* args)
{
	Cone* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		c->set_cap_radius( radius );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject* PyCone::get_base_radius(PyObject* self, PyObject* args)
{
	Cone *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_base_radius());
	}
	return retobj;
}

PyObject* PyCone::set_base_radius(PyObject* self, PyObject* args)
{
	Cone* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		c->set_base_radius( radius );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject* PyCone::get_cap_point(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Cone* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		vec = c->get_cap_point();
		result = ConvertTuple3( vec );
	}
	return result;
}
		
PyObject* PyCone::set_cap_point(PyObject* self, PyObject* args)
{
	Cone* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_cap_point( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject* PyCone::get_base_point(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Cone* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		vec = c->get_base_point();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject* PyCone::set_base_point(PyObject* self, PyObject* args)
{
	Cone* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_base_point( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCone::is_open(PyObject* self, PyObject* args)
{
	Cone* c;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		retobj = c->is_open() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PyCone::set_open(PyObject* self, PyObject* args)
{
	Cone* c;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cone*>(PyCObject_AsVoidPtr(object));
		c->set_open(b);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyMethodDef
PyCone::python_methods[] = {
	{"new_cone", new_cone, METH_VARARGS, "Create new cone object" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_base_radius", get_base_radius, METH_VARARGS, "Get base radius" },
	{"set_base_radius", set_base_radius, METH_VARARGS, "Set base radius" },
	{"get_cap_radius", get_cap_radius, METH_VARARGS, "Get cap radius" },
	{"set_cap_radius", set_cap_radius, METH_VARARGS, "Set cap radius" },
	{"get_cap_point", get_cap_point, METH_VARARGS, "Get cap point" },
	{"set_cap_point", set_cap_point, METH_VARARGS, "Set cap point" },
	{"get_base_point", get_base_point, METH_VARARGS, "Get base point" },
	{"set_base_point", set_base_point, METH_VARARGS, "Set base point" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"is_open", is_open, METH_VARARGS, "Is open" },
	{"set_open", set_open, METH_VARARGS, "Set caps" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{NULL, NULL, 0, NULL}
};

//////////////////////
// PyCylinder

PyObject*
PyCylinder::new_cylinder(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Cylinder *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "Cylinder#%d",box_cnt++);	
    result = (Cylinder*)new Cylinder(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyCylinder::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		vec = c->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyCylinder::set_location(PyObject* self, PyObject* args)
{
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCylinder::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		vec = c->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyCylinder::set_size(PyObject* self, PyObject* args)
{
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *pyvec;
	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_size( vec );
	}
	Py_INCREF(PYVAL_NONE);	
	return PYVAL_NONE;
}

PyObject*
PyCylinder::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		c->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PyCylinder::set_rotation(PyObject* self, PyObject* args)
{
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCylinder::get_radius(PyObject* self, PyObject* args)
{
	Cylinder *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_radius());
	}
	return retobj;
}

PyObject*
PyCylinder::set_radius(PyObject* self, PyObject* args)
{
	Cylinder* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		c->set_radius( radius );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCylinder::get_cap_point(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		vec = c->get_cap_point();
		result = ConvertTuple3( vec );
	}
	return result;
}
		
PyObject*
PyCylinder::set_cap_point(PyObject* self, PyObject* args)
{
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_cap_point( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCylinder::get_base_point(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		vec = c->get_base_point();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyCylinder::set_base_point(PyObject* self, PyObject* args)
{
	Cylinder* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_base_point( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCylinder::is_open(PyObject* self, PyObject* args)
{
	Cylinder* c;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		retobj = c->is_open() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PyCylinder::set_open(PyObject* self, PyObject* args)
{
	Cylinder* c;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Cylinder*>(PyCObject_AsVoidPtr(object));
		c->set_open(b);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyMethodDef
PyCylinder::python_methods[] = {
	{"new_cylinder", new_cylinder, METH_VARARGS, "Create new cylinder object" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_radius", get_radius, METH_VARARGS, "Get radius" },
	{"set_radius", set_radius, METH_VARARGS, "Set radius" },
	{"get_cap_point", get_cap_point, METH_VARARGS, "Get cap point" },
	{"set_cap_point", set_cap_point, METH_VARARGS, "Set cap point" },
	{"get_base_point", get_base_point, METH_VARARGS, "Get base point" },
	{"set_base_point", set_base_point, METH_VARARGS, "Set base point" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"is_open", is_open, METH_VARARGS, "Is open" },
	{"set_open", set_open, METH_VARARGS, "Set caps" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{NULL, NULL, 0, NULL}
};


///////////////////////////
//PyPlane

PyObject*
PyPlane::new_plane(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Plane *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "Plane#%d",box_cnt++);	
    result = (Plane*)new Plane(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyPlane::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Plane* p = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		p = reinterpret_cast<Plane*>(PyCObject_AsVoidPtr(object));
		vec = p->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyPlane::set_location(PyObject* self, PyObject* args)
{
	Plane* p = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		p = reinterpret_cast<Plane*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		p->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyPlane::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Plane* p = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		p = reinterpret_cast<Plane*>(PyCObject_AsVoidPtr(object));
		p->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PyPlane::set_rotation(PyObject* self, PyObject* args)
{
	Plane* p = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		p = reinterpret_cast<Plane*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		p->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyMethodDef
PyPlane::python_methods[]={
	{"new_plane", new_plane, METH_VARARGS, "Create new plane object" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{NULL, NULL, 0, NULL}
};

// PyLayer
PyObject*
PyLayer::new_layer(PyObject* self, PyObject* args)
{
	PyObject *l;
    ObjectLayer *result;
    result = (ObjectLayer*)new ObjectLayer(tv_get_obj_ref());
	l = PyCObject_FromVoidPtr( result, NULL );
    return l;
}

PyObject*
PyLayer::add_object(PyObject* self, PyObject* args)
{
	Object3D* obj = NULL;
	ObjectLayer* lay = NULL;
	
	PyObject* layer;
	PyObject* object;
	PyArg_ParseTuple(args, "OO", &layer, &object);
	Py_INCREF(object);
	Py_INCREF(layer);
	if (PyCObject_Check(object) && PyCObject_Check(layer)) {
		obj = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		lay = reinterpret_cast<ObjectLayer*>(PyCObject_AsVoidPtr(layer));
		lay->append_object( obj );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyLayer::set_name(PyObject* self, PyObject* args)
{
	ObjectLayer* layer;
	char* layer_name;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Os", &object, &layer_name);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		layer = reinterpret_cast<ObjectLayer*>(PyCObject_AsVoidPtr(object));
		layer->set_name(layer_name);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyLayer::get_name(PyObject* self, PyObject* args)
{
	ObjectLayer* layer;
	PyObject* retobj = PYVAL_NONE;
	
	PyObject* object;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		layer = reinterpret_cast<ObjectLayer*>(PyCObject_AsVoidPtr(object));
		retobj = StringToPyVal(layer->get_name());
	}
	return retobj;	
}

PyMethodDef
PyLayer::python_methods[] = {
	{"add_object", add_object, METH_VARARGS, "Adding object to layer" },
	{"new_layer", new_layer, METH_VARARGS, "Creating new layer" },
	{"get_name", get_name, METH_VARARGS, "Get layer name" },
	{"set_name", set_name, METH_VARARGS, "Set layer name" },
	{NULL, NULL, 0, NULL}
};


PyMethodDef
PyScene::python_methods[] = {
	{"add_layer", add_layer, METH_VARARGS, "Adding layer to scene" },
	{"add_object", add_object, METH_VARARGS, "Adding object to current layer" },
	{NULL, NULL, 0, NULL}
};

PyObject*
PyScene::add_layer(PyObject* self, PyObject* args)
{	
	PyObject* object;
	ObjectLayer* lay = NULL;
	ObjectList *objlist;
	
	app_objs* object_ref = tv_get_obj_ref();
	
	objlist = (ObjectList*)object_ref->obj3dlist;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		lay = reinterpret_cast<ObjectLayer*>(PyCObject_AsVoidPtr(object));
		objlist->insert_layer(lay, 1);
	}
//	return object;
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyScene::add_object(PyObject* self, PyObject* args)
{	
	PyObject* object;
	ObjectLayer* lay = NULL;
	Object3D* obj = NULL;
	ObjectList *objlist;
	
	app_objs* object_ref = tv_get_obj_ref();
	
	objlist = (ObjectList*)object_ref->obj3dlist;
	lay = (ObjectLayer*)objlist->get_current_layer();
	if ( lay == NULL ) {
		printf("PyScene::add_object: No current layer!\n"); fflush(stdout);
	} else {
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		obj = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		lay->append_object(obj);
	}
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}


PyObject*
PySuperEllipsoid::new_superellipsoid(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Superellipsoid *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "SuperE#%d",box_cnt++);	
    result = (Superellipsoid*)new Superellipsoid(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PySuperEllipsoid::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Superellipsoid* csg = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		vec = csg->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PySuperEllipsoid::set_location(PyObject* self, PyObject* args)
{
	Superellipsoid* csg = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		csg->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PySuperEllipsoid::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Superellipsoid* s = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		s = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		s->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PySuperEllipsoid::set_rotation(PyObject* self, PyObject* args)
{
	Superellipsoid* csg = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		csg->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PySuperEllipsoid::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Superellipsoid* csg = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		vec = csg->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PySuperEllipsoid::set_size(PyObject* self, PyObject* args)
{
	Superellipsoid* csg = NULL;
	PyObject *object;
	PyObject *pyvec;
	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		csg->set_size( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PySuperEllipsoid::set_east(PyObject* self, PyObject* args)
{
	Superellipsoid* csg;
	float b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Of", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		csg->set_east(b);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PySuperEllipsoid::set_north(PyObject* self, PyObject* args)
{
	Superellipsoid* csg;
	float b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Of", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		csg->set_north(b);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PySuperEllipsoid::get_east(PyObject* self, PyObject* args)
{
	Superellipsoid* csg;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		retobj = csg->get_east() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PySuperEllipsoid::get_north(PyObject* self, PyObject* args)
{
	Superellipsoid* csg;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<Superellipsoid*>(PyCObject_AsVoidPtr(object));
		retobj = csg->get_north() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyMethodDef PySuperEllipsoid::python_methods[] = {
	{"new_superellipsoid", new_superellipsoid, METH_VARARGS, "New superellipsoid" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"get_north", get_north, METH_VARARGS, "Get north value" },
	{"get_east", get_east, METH_VARARGS, "Get east value" },
	{"set_north", set_north, METH_VARARGS, "Set north value" },
	{"set_east", set_east, METH_VARARGS, "Set east value" },
	{NULL, NULL, 0, NULL}
};



PyObject*
PyCSGObject::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	ObjCSG* csg = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(object));
		vec = csg->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyCSGObject::set_location(PyObject* self, PyObject* args)
{
	ObjCSG* csg = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		csg->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCSGObject::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	ObjCSG* csg = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(object));
		csg->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PyCSGObject::set_rotation(PyObject* self, PyObject* args)
{
	ObjCSG* csg = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		csg->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCSGObject::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	ObjCSG* csg = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(object));
		vec = csg->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyCSGObject::set_size(PyObject* self, PyObject* args)
{
	ObjCSG* csg = NULL;
	PyObject *object;
	PyObject *pyvec;
	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		csg->set_size( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyCSGObject::is_evaluate(PyObject* self, PyObject* args)
{
	ObjCSG* csg;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(object));
		retobj = csg->is_evaluate() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}


PyObject*
PyCSGObject::set_evaluate(PyObject* self, PyObject* args)
{
	ObjCSG* csg;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(object));
		csg->set_evaluate(b);
	}
	Py_INCREF(PYVAL_NONE);	
	return PYVAL_NONE;
}

PyObject*
PyCSGObject::add_object(PyObject* self, PyObject* args)
{
	Object3D* obj = NULL;
	ObjCSG* csg = NULL;
	
	PyObject* csgobj;
	PyObject* object;

	PyArg_ParseTuple(args, "OO", &csgobj, &object);
	Py_INCREF(object);
	Py_INCREF(csgobj);
	if (PyCObject_Check(object) && PyCObject_Check(csgobj)) {
		obj = reinterpret_cast<Object3D*>(PyCObject_AsVoidPtr(object));
		csg = reinterpret_cast<ObjCSG*>(PyCObject_AsVoidPtr(csgobj));
		csg->add_object( obj );
	}	
	return object;
}

PyMethodDef
PyCSGObject::python_methods[] = {
	{"add_object", add_object, METH_VARARGS, "Adding object to CSG" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"is_evaluate",  is_evaluate,  METH_VARARGS, "Evaluated" },
	{"set_evaluate",  set_evaluate,  METH_VARARGS, "Set evaluated" },
	{NULL, NULL, 0, NULL}
};

PyObject*
PyCSGUnion::new_csgunion(PyObject* self, PyObject* args)
{
	PyObject *obj;
    CsgUnion *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "CSG_Union#%d",box_cnt++);	
    result = (CsgUnion*)new CsgUnion(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyMethodDef
PyCSGUnion::python_methods[] = {
	{"new_csgunion", new_csgunion, METH_VARARGS, "Create new union object" },
	{NULL, NULL, 0, NULL}
};


PyObject*
PyCSGCutAway::is_cutaway_textures(PyObject* self, PyObject* args)
{
	CsgCutAway* csg;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<CsgCutAway*>(PyCObject_AsVoidPtr(object));
		retobj = csg->is_cutaway() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PyCSGCutAway::set_cutaway_textures(PyObject* self, PyObject* args)
{
	CsgCutAway* csg;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		csg = reinterpret_cast<CsgCutAway*>(PyCObject_AsVoidPtr(object));
		csg->set_cutaway(b);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyMethodDef
PyCSGCutAway::python_methods[]={
	{"is_cutaway_textures", is_cutaway_textures, METH_VARARGS, "Cutaway textures" },
	{"set_cutaway_textures", set_cutaway_textures, METH_VARARGS, "Set cutaway textures" },
	{NULL, NULL, 0, NULL}
};

PyObject*
PyCSGDifference::new_csgdifference(PyObject* self, PyObject* args)
{
	PyObject *obj;
    CsgUnion *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "CSG_Diff#%d",box_cnt++);	
    result = (CsgUnion*)new CsgDifference(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyMethodDef
PyCSGDifference::python_methods[] = {
	{"new_csgdifference", new_csgdifference, METH_VARARGS, "Create new difference object" },
	{NULL, NULL, 0, NULL}
};

PyObject*
PyCSGMerge::new_csgmerge(PyObject* self, PyObject* args)
{
	PyObject *obj;
    CsgMerge *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "CSG_Merge#%d",box_cnt++);	
    result = (CsgMerge*)new CsgMerge(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyMethodDef
PyCSGMerge::python_methods[] = {
	{"new_csgmerge", new_csgmerge, METH_VARARGS, "Create new merge object" },
	{NULL, NULL, 0, NULL}
};


PyObject*
PyCSGIntersection::new_csgintersection(PyObject* self, PyObject* args)
{
	PyObject *obj;
    CsgIntersection *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "CSG_Intersec#%d",box_cnt++);	
    result = (CsgIntersection*)new CsgIntersection(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyMethodDef
PyCSGIntersection::python_methods[] = {
	{"new_csgintersection", new_csgintersection, METH_VARARGS, "Create new intersect object" },
	{NULL, NULL, 0, NULL}
};

/////////////////////////////
// PyTorus

PyObject*
PyTorus::new_torus(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Torus *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "Torus#%d",box_cnt++);	
    result = (Torus*)new Torus(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyTorus::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Torus* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		vec = c->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyTorus::set_location(PyObject* self, PyObject* args)
{
	Torus* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyTorus::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Torus* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		vec = c->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyTorus::set_size(PyObject* self, PyObject* args)
{
	Torus* c = NULL;
	PyObject *object;
	PyObject *pyvec;
	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_size( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyTorus::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Torus* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		c->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PyTorus::set_rotation(PyObject* self, PyObject* args)
{
	Torus* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyTorus::get_minor_radius(PyObject* self, PyObject* args)
{
	Torus *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_minor_radius());
	}
	return retobj;
}

PyObject*
PyTorus::set_minor_radius(PyObject* self, PyObject* args)
{
	Torus* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		c->set_minor_radius( radius );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject* PyTorus::get_major_radius(PyObject* self, PyObject* args)
{
	Torus *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_major_radius());
	}
	return retobj;
}

PyObject* PyTorus::set_major_radius(PyObject* self, PyObject* args)
{
	Torus* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		c->set_major_radius( radius );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyTorus::is_sturm(PyObject* self, PyObject* args)
{
	Torus* c;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		retobj = c->is_sturm() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PyTorus::set_sturm(PyObject* self, PyObject* args)
{
	Torus* c;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Torus*>(PyCObject_AsVoidPtr(object));
		c->set_sturm(b);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyMethodDef
PyTorus::python_methods[] = {
	{"new_torus", new_torus, METH_VARARGS, "Create new torus object" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_minor_radius", get_minor_radius, METH_VARARGS, "Get minor radius" },
	{"set_minor_radius", set_minor_radius, METH_VARARGS, "Set minor radius" },
	{"get_major_radius", get_major_radius, METH_VARARGS, "Get major radius" },
	{"set_major_radius", set_major_radius, METH_VARARGS, "Set major radius" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"is_sturm", is_sturm, METH_VARARGS, "Is sturm" },
	{"set_sturm", set_sturm, METH_VARARGS, "Set sturm" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{NULL, NULL, 0, NULL}
};

/////////////////////////////
// PyDisc

PyObject*
PyDisc::new_disc(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Disc *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "Disc#%d",box_cnt++);	
    result = (Disc*)new Disc(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyDisc::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Disc* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		vec = c->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyDisc::set_location(PyObject* self, PyObject* args)
{
	Disc* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyDisc::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Disc* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		vec = c->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyDisc::set_size(PyObject* self, PyObject* args)
{
	Disc* c = NULL;
	PyObject *object;
	PyObject *pyvec;
	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_size( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyDisc::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Disc* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		c->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PyDisc::set_rotation(PyObject* self, PyObject* args)
{
	Disc* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyDisc::get_radius(PyObject* self, PyObject* args)
{
	Disc *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_radius());
	}
	return retobj;
}

PyObject*
PyDisc::set_radius(PyObject* self, PyObject* args)
{
	Disc* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		c->set_radius( radius );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyDisc::get_hole_radius(PyObject* self, PyObject* args)
{
	Disc *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_hole_radius());
	}
	return retobj;
}

PyObject*
PyDisc::set_hole_radius(PyObject* self, PyObject* args)
{
	Disc* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Disc*>(PyCObject_AsVoidPtr(object));
		c->set_hole_radius( radius );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyMethodDef
PyDisc::python_methods[] = {
	{"new_disc", new_disc, METH_VARARGS, "Create new disc object" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_radius", get_radius, METH_VARARGS, "Get radius" },
	{"set_radius", set_radius, METH_VARARGS, "Set radius" },
	{"get_hole_radius", get_hole_radius, METH_VARARGS, "Get hole radius" },
	{"set_hole_radius", set_hole_radius, METH_VARARGS, "Set hole radius" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{NULL, NULL, 0, NULL}
};

////////////////////////////////////
// PyBicubic

PyObject*
PyBicubic::new_bicubic(PyObject* self, PyObject* args)
{
	PyObject *obj;
    Bicubic *result;	
	static int box_cnt = 0;
	char new_name[16];
	sprintf(new_name, "Bicubic#%d",box_cnt++);	
    result = (Bicubic*)new Bicubic(tv_get_obj_ref());
	result->set_name(new_name);
	obj = PyCObject_FromVoidPtr( result, NULL );
    return obj;
}

PyObject*
PyBicubic::get_location(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Bicubic* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		vec = c->get_location();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyBicubic::set_location(PyObject* self, PyObject* args)
{
	Bicubic* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_location( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBicubic::get_size(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Bicubic* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		vec = c->get_size();
		result = ConvertTuple3( vec );
	}
	return result;
}

PyObject*
PyBicubic::set_size(PyObject* self, PyObject* args)
{
	Bicubic* c = NULL;
	PyObject *object;
	PyObject *pyvec;
	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_size( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBicubic::get_rotation(PyObject* self, PyObject* args)
{
	float vec[3];
	Bicubic* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		c->get_rot( vec );
		result = ConvertTuple3( (float*)vec );
	}
	return result;
}

PyObject*
PyBicubic::set_rotation(PyObject* self, PyObject* args)
{
	Bicubic* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	PyArg_ParseTuple(args, "OO", &object, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_rotation( vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBicubic::get_flatness(PyObject* self, PyObject* args)
{
	Bicubic *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;
	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("f", c->get_flatness());
	}
	return retobj;
}

PyObject*
PyBicubic::set_flatness(PyObject* self, PyObject* args)
{
	Bicubic* c = NULL;
	PyObject *object;
	float radius;
	PyArg_ParseTuple(args, "Of", &object, &radius);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		c->set_flatness( radius );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}


PyObject*
PyBicubic::get_ustep(PyObject* self, PyObject* args)
{
	Bicubic *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("i", c->get_ustep());
	}
	return retobj;
}

PyObject*
PyBicubic::get_vstep(PyObject* self, PyObject* args)
{
	Bicubic *c = NULL;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		retobj = Py_BuildValue("i", c->get_vstep());
	}
	return retobj;
}

PyObject*
PyBicubic::set_ustep(PyObject* self, PyObject* args)
{
	Bicubic* c = NULL;
	PyObject *object;
	int step;
	PyArg_ParseTuple(args, "Oi", &object, &step);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		c->set_ustep( step );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBicubic::set_vstep(PyObject* self, PyObject* args)
{
	Bicubic* c = NULL;
	PyObject *object;
	int step;
	PyArg_ParseTuple(args, "Oi", &object, &step);
	Py_INCREF(object);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		c->set_vstep( step );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}

PyObject*
PyBicubic::get_control_point(PyObject* self, PyObject* args)
{
	Vector3D vec;
	Bicubic* c = NULL;
	PyObject *object;
	PyObject *result = PYVAL_NONE;
	int pindex;
	PyArg_ParseTuple(args, "Oi", &object, &pindex);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		vec = c->get_control_point(pindex);
		result = ConvertTuple3( vec );
	}
	return result;
}
		
PyObject*
PyBicubic::set_control_point(PyObject* self, PyObject* args)
{
	Bicubic* c = NULL;
	PyObject *object;
	PyObject *pyvec;

	float vec[3];
	int pindex;
	PyArg_ParseTuple(args, "OiO", &object, &pindex, &pyvec);
	Py_INCREF(object);
	Py_INCREF(pyvec);
	if (PyCObject_Check(object)){
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		ConvertVector3DV( pyvec, &vec[0], &vec[1], &vec[2] );
		c->set_control_point( pindex, vec );
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;
}


PyObject*
PyBicubic::is_edit(PyObject* self, PyObject* args)
{
	Bicubic* c;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		retobj = c->is_edit() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PyBicubic::set_edit(PyObject* self, PyObject* args)
{
	Bicubic* c;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		c->set_edit(b);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyObject*
PyBicubic::is_preprocess(PyObject* self, PyObject* args)
{
	Bicubic* c;
	PyObject *retobj, *object;
	retobj = PYVAL_NONE;

	PyArg_ParseTuple(args, "O", &object);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		retobj = c->is_preprocess() ? PYVAL_TRUE : PYVAL_FALSE;
	}
	return retobj;	
}

PyObject*
PyBicubic::set_preprocess(PyObject* self, PyObject* args)
{
	Bicubic* c;
	bool b;
	
	PyObject* object;
	PyArg_ParseTuple(args, "Oi", &object, &b);
	Py_INCREF(object);
	if (PyCObject_Check(object)) {
		c = reinterpret_cast<Bicubic*>(PyCObject_AsVoidPtr(object));
		c->set_preprocess(b);
	}
	Py_INCREF(PYVAL_NONE);
	return PYVAL_NONE;	
}

PyMethodDef
PyBicubic::python_methods[] = {
	{"new_bicubic", new_bicubic, METH_VARARGS, "Create new cone object" },
	{"get_location", get_location, METH_VARARGS, "Get loacation" },
	{"set_location", set_location, METH_VARARGS, "Set loacation" },
	{"get_flatness", get_flatness, METH_VARARGS, "Get cap radius" },
	{"set_flatness", set_flatness, METH_VARARGS, "Set cap radius" },
	{"get_ustep", get_ustep, METH_VARARGS, "Get ustep" },
	{"set_ustep", set_ustep, METH_VARARGS, "Set ustep" },
	{"get_vstep", get_vstep, METH_VARARGS, "Get vstep" },
	{"set_vstep", set_vstep, METH_VARARGS, "Set vstep" },
	{"get_control_point", get_control_point, METH_VARARGS, "Get control point" },
	{"set_control_point", set_control_point, METH_VARARGS, "Set control point" },
	{"get_size", get_size, METH_VARARGS, "Get size" },
	{"set_size", set_size, METH_VARARGS, "Set size" },
	{"is_preprocess", is_preprocess, METH_VARARGS, "Is preprocess" },
	{"set_preprocess", set_preprocess, METH_VARARGS, "Set preprocess" },
	{"is_edit", is_edit, METH_VARARGS, "Is edit" },
	{"set_edit", set_edit, METH_VARARGS, "Set edit" },
	{"set_rotation", set_rotation, METH_VARARGS, "Set rotation" },
	{"get_rotation", get_rotation, METH_VARARGS, "Get rotation" },
	{NULL, NULL, 0, NULL}
};

