//*****************************************************************************************
// Truevision - a 3d modeler for gnome and povray
//
// texpigment.cc
//
// Vincent LE PRINCE <vincentleprince@users.sourceforge.net>
// Copyright (C) 2000-2005 Vincent LE PRINCE
// This file is part of the TRUEVISION Package

//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */ 
//*******************************************************************************************
#include "include/dlgutils.h"
#include "include/texpigment.h"
#include "include/tvio.h"
#include "include/matlist.h"
#include "include/preferences.h"

//**************************************
// Dfinitions
//**************************************
const int pig_type_num = 6;
const char *pig_type_list[pig_type_num] = { 
	N_("Solid color"),
	N_("Color List"),
	N_("Pigment List"),
	N_("Color map"),
	N_("Pigment map"),
	N_("Image map") };

MapItem *mapedit_feeder_pig( gpointer data )
{ return (MapItem*)( new MapItem_pigment( 0, 0, ((PovTexPigment*)data)->app_ref ) ); }

MapItem *mapedit_feeder_col( gpointer data )
{ return (MapItem*)( new MapItem_color( 0, 0, ((PovTexPigment*)data)->app_ref ) ); }
	
//**************************************
// Constructeur
//**************************************
PovTexPigment::PovTexPigment( app_objs *appref, char *tagname, bool mother  ) : MaterialItem( appref, tagname )
{
set_name( _("Pigment") );
item_type = TV_MITEM_PIGMENT;
is_mother = mother;
changing_box = NULL;
changing_box2 = NULL;
blockpig1 = blockpig2 = blockpig3 = NULL;
mapedit = NULL;
blendmap = NULL;
blendmap_mod = NULL;
blockpattern = NULL;

used = new TvWidget_bool_activator( N_("Use pigment"), "USED", NULL, app_ref, is_mother ? false : true );
type = new TvWidget_option_combo( N_("Type"), "TYPE", NULL, app_ref );
type->set_list( pig_type_list, pig_type_num, 0 );
transform = new TvWidget_transformation_rb( N_("Transformation"), "TRANSFO", NULL, app_ref );
solid1 = new TvWidget_pigment( N_("Solid color 1"), "SOLCOL1", NULL, app_ref );
solid2 = new TvWidget_pigment( N_("Solid color 2"), "SOLCOL2", NULL, app_ref );
solid3 = new TvWidget_pigment( N_("Solid color 3"), "SOLCOL3", NULL, app_ref );
noise = new TvWidget_noise_rb( N_("Noise"), "NOISE", NULL, app_ref );
warp = new TvWidget_warp_rb( N_("Warp"), "WARP", NULL, app_ref );
bitmap_def = new TvWidget_bitmap( N_("Image"), "BITMAP", NULL, app_ref );
}

PovTexPigment::PovTexPigment( PovTexPigment & ref ) : MaterialItem( ref )
{
set_name( ref.get_name() );
item_type = TV_MITEM_PIGMENT;
is_mother = true;
changing_box = NULL;
changing_box2 = NULL;

if ( ref.blockpig1 != NULL ) blockpig1 = new PovTexPigment( *ref.blockpig1 ); else blockpig1 = NULL;
if ( ref.blockpig2 != NULL ) blockpig2 = new PovTexPigment( *ref.blockpig2 ); else blockpig2 = NULL;
if ( ref.blockpig3 != NULL ) blockpig3 = new PovTexPigment( *ref.blockpig3 ); else blockpig3 = NULL;
if ( ref.mapedit != NULL ) mapedit = new TvWidget_map_editor( *ref.mapedit ); else mapedit = NULL;
if ( ref.blendmap != NULL ) blendmap = new TvWidget_blendmap( *ref.blendmap ); else blendmap = NULL;
if ( ref.blendmap_mod != NULL ) blendmap_mod = new TvWidget_blendmap_mod( *ref.blendmap_mod ); else blendmap_mod = NULL;
if ( ref.blockpattern != NULL ) blockpattern = new TvWidget_blockpattern( *ref.blockpattern ); else blockpattern = NULL;

used = new TvWidget_bool_activator( *ref.used );
type = new TvWidget_option_combo( *ref.type );
transform = new TvWidget_transformation_rb( *ref.transform );
solid1 = new TvWidget_pigment( *ref.solid1 );
solid2 = new TvWidget_pigment( *ref.solid2 );
solid3 = new TvWidget_pigment( *ref.solid3 );
noise = new TvWidget_noise_rb( *ref.noise );
warp = new TvWidget_warp_rb( *ref.warp );
bitmap_def = new TvWidget_bitmap( *ref.bitmap_def );
}


PovTexPigment::~PovTexPigment()
{
delete used;
delete type;
delete solid1;
delete solid2;
delete solid3;
delete transform;
delete noise;
delete warp;
delete bitmap_def;
if ( blockpattern != NULL ) delete blockpattern;
if ( blendmap != NULL ) delete blendmap;
if ( blendmap_mod != NULL ) delete blendmap_mod;
if ( mapedit != NULL ) delete mapedit;
if ( blockpig1 != NULL )
	{
	delete blockpig1;
	delete blockpig2;
	delete blockpig3;	
	}
}


void PovTexPigment::add_to_tree( GtkTreeView *view, GtkTreeStore *store, GtkTreeSelection *selection, GtkTreeIter *parent, GtkTreeIter *sibling )
{
MaterialItem::add_to_tree( view, store, selection, parent, sibling );
if ( blockpig1 != NULL )
	{
	blockpig1->add_to_tree( view, store, selection, &node_iter );
	blockpig2->add_to_tree( view, store, selection, &node_iter );
	if ( blockpattern->value() == 2 ) 
		blockpig3->add_to_tree( view, store, selection, &node_iter );
	}
if ( mapedit != NULL ) mapedit->add_to_tree( view, store, selection, &node_iter );
set_node_state();
}

void PovTexPigment::remove_from_tree()
{
if ( blockpig1 != NULL ) blockpig1->remove_from_tree();
if ( blockpig2 != NULL ) blockpig2->remove_from_tree();
if ( blockpig3 != NULL ) blockpig3->remove_from_tree();
if ( mapedit != NULL ) mapedit->remove_from_tree();
MaterialItem::remove_from_tree();
}

void PovTexPigment::save_node_state()
{
if ( blockpig1 != NULL ) blockpig1->save_node_state();
if ( blockpig2 != NULL ) blockpig2->save_node_state();
if ( blockpig3 != NULL ) blockpig3->save_node_state();
if ( mapedit != NULL ) mapedit->save_node_state();
}

void PovTexPigment::edit_widget( GtkWidget *box )
{
PREF_DEF
bool tt = pref->tooltips->value();
MaterialItem::edit_widget( box, _("Pigment") );

// Bool activator
activated_box = gtk_vbox_new( FALSE, 0 );

if ( is_mother )
	{
	used->set_target( activated_box );
	used->get_widget_no_toggle( edit_cont, tt );
	}
	
// Activated box
gtk_box_pack_start( GTK_BOX(edit_cont), activated_box, TRUE, TRUE, 0 );
	type->get_widget( activated_box, tt );
	type->connect_signal( GTK_SIGNAL_FUNC(sign_pigtype_changed), this );
	set_changing_widget();
	
gtk_widget_show_all( widget );
if ( is_mother ) used->update_widget();
}

void PovTexPigment::destroy_widget()
{
MaterialItem::destroy_widget();
	used->clear_widget();
	type->clear_widget();
	solid1->clear_widget();
	solid2->clear_widget();
	solid3->clear_widget();	
	transform->clear_widget();
	noise->clear_widget();
	warp->clear_widget();
	bitmap_def->clear_widget();
	if ( blockpattern != NULL ) blockpattern->clear_widget();
	if ( blendmap != NULL ) blendmap->clear_widget();
	if ( blendmap_mod != NULL ) blendmap_mod->clear_widget();
	if ( mapedit != NULL ) mapedit->clear_widget();
changing_box = NULL;
changing_box2 = NULL;
}

void PovTexPigment::flush()
{
used->flush();
type->flush();
solid1->flush();
solid2->flush();
solid3->flush();
transform->flush();
noise->flush();
warp->flush();
bitmap_def->flush();
if ( blockpattern != NULL ) blockpattern->flush();
if ( blendmap != NULL ) blendmap->flush();
if ( blendmap_mod != NULL ) blendmap_mod->flush();
if ( mapedit != NULL ) mapedit->flush();
}



//**************************************
// Type changed
//**************************************
void PovTexPigment::type_changed()
{
type->flush();
if ( blockpig1 != NULL && type->value() != 2 )
	{
	blockpig1->remove_from_tree(); delete blockpig1;
	blockpig2->remove_from_tree(); delete blockpig2;
	blockpig3->remove_from_tree(); delete blockpig3;
	blockpig1 = blockpig2 = blockpig3 = NULL;
	}

if ( blockpig1 == NULL && type->value() == 2 )
	{
	blockpig1 = new PovTexPigment( app_ref, false );
	blockpig2 = new PovTexPigment( app_ref, false );
	blockpig3 = new PovTexPigment( app_ref, false );
	blockpig1->add_to_tree(  tree_view, tree_store, tree_selection, &node_iter  );
	blockpig2->add_to_tree(  tree_view, tree_store, tree_selection, &node_iter  );
	}
if ( blockpattern != NULL && type->value() != 1 && type->value() != 2 )
	{
	delete blockpattern;
	blockpattern = NULL;
	}
if ( blendmap != NULL )
	{
	delete blendmap;
	blendmap = NULL;
	}
if ( blendmap_mod != NULL && type->value() != 3 && type->value() != 4 )
	{
	delete blendmap_mod;
	blendmap_mod = NULL;
	}
if ( mapedit != NULL )
	{
	mapedit->remove_from_tree();
	delete mapedit;
	mapedit = NULL;
	}
set_changing_widget();
}

void PovTexPigment::set_changing_widget()
{
PREF_DEF
bool tt = pref->tooltips->value();
//type->flush();

// Changing box 1
if ( changing_box != NULL ) gtk_widget_destroy( changing_box );
solid1->clear_widget();
solid2->clear_widget();
solid3->clear_widget();
bitmap_def->clear_widget();
changing_box  = gtk_vbox_new( FALSE, 0 );
gtk_box_pack_start( GTK_BOX(activated_box), changing_box, FALSE, FALSE, 0 );

switch ( type->value() )
	{
	case 0:
		solid1->get_widget( changing_box, tt );
		break;

	case 1:
		{
		if ( blockpattern == NULL ) blockpattern = new TvWidget_blockpattern( N_("Block Pattern"), "BLKPAT", NULL, app_ref );
		blockpattern->get_widget( changing_box, tt );
		blockpattern->connect_signal( sign_colblocktype_changed, this );
		solid1->get_widget( changing_box, tt );
		solid2->get_widget( changing_box, tt );
		solid3->get_widget( changing_box, tt );
		if ( blockpattern->value() != 2 ) solid3->set_widget_inactive();
		}
		break;

	case 2:
		{
		if ( blockpattern == NULL ) blockpattern = new TvWidget_blockpattern( N_("Block Pattern"), "BLKPAT", NULL, app_ref );
		blockpattern->get_widget( changing_box, tt );
		blockpattern->connect_signal( sign_pigblocktype_changed, this );
		set_pigblock_items();
		}
		break;

	case 3:
		{
		if ( blendmap == NULL ) blendmap = new TvWidget_blendmap( N_("Map definition"), "BMAP", NULL, app_ref, true );
		blendmap->get_widget( changing_box, tt );
		if ( mapedit == NULL ) 
			{
			mapedit = new TvWidget_map_editor( N_("Map editor"), "MAPEDIT", NULL, app_ref, mapedit_feeder_col, this );
			MapItem_color *col = new MapItem_color( 0.0, 0, app_ref );
			mapedit->add( col );
			col = new  MapItem_color( 1.0, 255, app_ref );
			mapedit->add( col );		
			}
		mapedit->get_widget( changing_box, tt );
		}
		break;

	case 4:
		{
		if ( blendmap == NULL ) blendmap = new TvWidget_blendmap( N_("Map definition"), "BMAP", NULL, app_ref );		
		blendmap->get_widget( changing_box, tt );
		if ( mapedit == NULL ) 
			{
			mapedit = new TvWidget_map_editor( N_("Map editor"), "MAPEDIT", NULL, app_ref, mapedit_feeder_pig, this );
			MapItem_pigment *pig = new MapItem_pigment( 0.0, 0, app_ref );
			mapedit->add( pig );
			pig = new  MapItem_pigment( 1.0, 255, app_ref );
			mapedit->add( pig );		
			mapedit->add_to_tree( tree_view, tree_store, tree_selection, &node_iter );
			}
		mapedit->get_widget( changing_box, tt );
		}
		break;

	case 5:
		bitmap_def->get_widget( changing_box, tt );
		break;
	}

// Changing box 2
if ( changing_box2 != NULL ) gtk_widget_destroy( changing_box2 );
changing_box2  = gtk_vbox_new( FALSE, 0 );
gtk_box_pack_start( GTK_BOX(activated_box), changing_box2, FALSE, FALSE, 0 );
if ( type->value() == 3 || type->value() == 4 )
	{ 
	if ( blendmap_mod == NULL ) blendmap_mod = new TvWidget_blendmap_mod( N_("Map modifiers"), "BMAPMOD", NULL, app_ref );
	blendmap_mod->get_widget( changing_box2, tt );
	}
transform->get_widget_rb( changing_box2, tt );
noise->get_widget_rb( changing_box2, tt );
warp->get_widget_rb( changing_box2, tt );

gtk_widget_show_all( changing_box );
gtk_widget_show_all( changing_box2 );
}


void PovTexPigment::set_pigblock_items()
{
if ( blockpig3 == NULL ) return;
blockpattern->flush();
if ( blockpattern->value() == 2 )
		blockpig3->add_to_tree( tree_view, tree_store, tree_selection, &node_iter );
else
		blockpig3->remove_from_tree();

}

void PovTexPigment::set_colblock_items()
{
blockpattern->flush(); 
if ( blockpattern->value() == 2 )
	solid3->set_widget_active();
else
	solid3->set_widget_inactive();
}


void PovTexPigment::output_to_povray( ofstream & file, bool in_map )
{
if ( !used->value() ) return;
if ( !in_map ) file << "\n\t\tpigment {\n\t\t\t";

switch ( type->value() )
	{
	case 0:
		solid1->output_to_povray( file );
		break;

	case 1:
		blockpattern->output_to_povray( file );
		file << "\n\t\t\t\t";
		solid1->output_to_povray( file );
		file << ",\n\t\t\t\t";
		solid2->output_to_povray( file );		
		if ( blockpattern->value() == 2 )
			{
			file << ",\n\t\t\t\t";
			solid3->output_to_povray( file );
			}
		blockpattern->output_to_povray_options( file );
		break;

	case 2:
		blockpattern->output_to_povray( file );
		file << "\n\t\t\t\t";
		blockpig1->output_to_povray( file );
		file << ",\n\t\t\t\t";
		blockpig2->output_to_povray( file );		
		if ( blockpattern->value() == 2 )
			{
			file << ",\n\t\t\t\t";
			blockpig3->output_to_povray( file );
			}
		blockpattern->output_to_povray_options( file );				
		break;

	case 3:
		blendmap->output_to_povray( file );
		file << "\n\t\t\tcolor_map {";
		mapedit->output_to_povray( file );
		file << "\n\t\t\t}\n\t\t\t";
		if ( blendmap_mod != NULL ) blendmap_mod->output_to_povray( file );
		break;

	case 4:
		blendmap->output_to_povray( file );
		file << "\n\t\t\tpigment_map {";
		mapedit->output_to_povray( file );
		file << "\n\t\t\t}\n\t\t\t";
		if ( blendmap_mod != NULL ) blendmap_mod->output_to_povray( file );
		break;

	case 5:
		bitmap_def->output_to_povray( file );
		break;
	}
noise->output_to_povray( file );
warp->output_to_povray( file );
transform->output_to_povray( file );
if ( !in_map ) file << "\n\t\t}";
}


//**********************************************
// Sauvegarde
//**********************************************
void PovTexPigment::save( ofstream & file )
{
if ( !used->value() ) return;
flush();
file << "\nPIGMENT{\n";
used->save( file );
name->save( file ); file << "\n";
expand->save(file);
type->save( file ); file << "\n";
transform->save( file ); file << "\n";
noise->save( file );
warp->save( file );

switch ( type->value() )
	{
	case 0:
		solid1->save( file ); file << "\n";
		break;

	case 1:
		blockpattern->save( file ); file << "\n";
		solid1->save( file );
		solid2->save( file );		
		if ( blockpattern->value() == 2 )
			solid3->save( file );
		break;

	case 2:
		blockpattern->save( file );
		blockpig1->save( file );
		blockpig2->save( file );		
		if ( blockpattern->value() == 2 )
			blockpig3->save( file );
		break;

	case 3:	
		blendmap->save( file );
		mapedit->save( file );
		if ( blendmap_mod != NULL ) blendmap_mod->save( file );
		break;

	case 4:
		blendmap->save( file );
		mapedit->save( file );
		if ( blendmap_mod != NULL ) blendmap_mod->save( file );
		break;

	case 5:
		bitmap_def->save( file );
		break;
	}
file << "\n}\n\n";
}


bool PovTexPigment::load( ifstream & file, char *ltag )
{
if ( strcmp( "PIGMENT", ltag ) ) return false;
char * tag = NULL;
used->set( true );

do
	{
	tag = tvio_get_next_tag( file );
	if ( tag == NULL ) break;
	if (used->load( file , tag ) ) continue;	
	if (type->load( file , tag ) ) continue;	
	if (expand->load( file , tag ) ) continue;	
	if (name->load( file , tag ) ) continue;	
	if (noise->load( file , tag ) ) continue;	
	if (warp->load( file , tag ) ) continue;	
	if (transform->load( file , tag ) ) continue;
	if (solid1->load( file , tag ) ) continue;
	if (solid2->load( file , tag ) ) continue;
	if (solid3->load( file , tag ) ) continue;
	if (bitmap_def->load( file , tag ) ) continue;
	
	if ( !strcmp( tag, "BMAP" ) )
		{
		if ( blendmap == NULL ) blendmap = new TvWidget_blendmap( N_("Pattern"), "BMAP", NULL, app_ref );
		if ( blendmap_mod == NULL ) blendmap_mod = new TvWidget_blendmap_mod( N_("Pattern"), "BMAPMOD", NULL, app_ref );		
		blendmap->load( file, tag );
		continue;
		}
		
	if ( !strcmp( tag, "BMAPMOD" ) )
		{
		if ( blendmap_mod == NULL ) blendmap_mod = new TvWidget_blendmap_mod( N_("Pattern"), "BMAPMOD", NULL, app_ref );
		blendmap_mod->load( file, tag );
		continue;
		}

	if ( !strcmp( tag, "BLKPAT" ) )
		{
		if ( blockpattern == NULL ) blockpattern = new TvWidget_blockpattern( N_("Block Pattern"), "BLKPAT", NULL, app_ref );
		blockpattern->load( file, tag );
		continue;
		}

	if ( !strcmp( tag, "PIGMENT" ) )
		{
		if ( blockpig1 == NULL ) { blockpig1 = new PovTexPigment( app_ref, false ); blockpig1->load( file, tag ); continue; }
		if ( blockpig2 == NULL ) { blockpig2 = new PovTexPigment( app_ref, false ); blockpig2->load( file, tag ); continue; }
		if ( blockpig3 == NULL ) { blockpig3 = new PovTexPigment( app_ref, false ); blockpig3->load( file, tag ); continue; }
		tvio_skip_section( file );
		}

	if ( !strcmp( tag, "MAPEDIT" ) )
		{
		if ( mapedit != NULL ) delete mapedit;
		if ( type->value() == 3 )
			mapedit = new TvWidget_map_editor( N_("Map editor"), "MAPEDIT", NULL, app_ref, mapedit_feeder_col, this );
		else
			mapedit = new TvWidget_map_editor( N_("Map editor"), "MAPEDIT", NULL, app_ref, mapedit_feeder_pig, this );
		mapedit->load( file, tag );
		continue;
		}

	tvio_skip_section( file );
	}
while( tag != NULL );
return true;
}


bool PovTexPigment::paste( MaterialItem *item )
{
	if ( item->get_type() != TV_MITEM_PIGMENT ) return false;
	set_name( item->get_name() );

	PovTexPigment *pig = (PovTexPigment*)item;
	if ( blockpig1 != NULL ) { delete blockpig1; blockpig1 = NULL; }
	if ( blockpig2 != NULL ) { delete blockpig2; blockpig2 = NULL; }
	if ( blockpig3 != NULL ) { delete blockpig3; blockpig3 = NULL; }
	if ( blendmap != NULL ) { delete blendmap; blendmap = NULL; }
	if ( blendmap_mod != NULL ) { delete blendmap_mod; blendmap_mod = NULL; }
	if ( blockpattern != NULL ) { delete blockpattern; blockpattern = NULL; }
	if ( mapedit != NULL ) { delete mapedit; mapedit = NULL; }

	if ( GTK_IS_WIDGET(widget) ) {
		gtk_widget_destroy( edit_cont );
		used->clear_widget();
		type->clear_widget();
		transform->clear_widget();
		noise->clear_widget();
		warp->clear_widget();
		solid1->clear_widget();
		solid2->clear_widget();
		solid3->clear_widget();
		bitmap_def->clear_widget();
	}
	used->copy( pig->used );
	type->copy( pig->type );
	transform->copy( pig->transform );
	noise->copy( pig->noise );
	warp->copy( pig->warp );
	solid1->copy( pig->solid1 );
	solid2->copy( pig->solid2 );
	solid3->copy( pig->solid3 );
	bitmap_def->copy( pig->bitmap_def );

	if ( pig->blockpig1 != NULL ) blockpig1 = new PovTexPigment( *pig->blockpig1 ); else blockpig1 = NULL;
	if ( pig->blockpig2 != NULL ) blockpig2 = new PovTexPigment( *pig->blockpig2 ); else blockpig2 = NULL;
	if ( pig->blockpig3 != NULL ) blockpig3 = new PovTexPigment( *pig->blockpig3 ); else blockpig3 = NULL;
	if ( pig->blendmap != NULL ) blendmap = new TvWidget_blendmap( *pig->blendmap );
	if ( pig->blendmap_mod != NULL ) blendmap_mod = new TvWidget_blendmap_mod( *pig->blendmap_mod );
	if ( pig->blockpattern != NULL ) blockpattern = new TvWidget_blockpattern( *pig->blockpattern );
	if ( pig->mapedit != NULL ) mapedit = new TvWidget_map_editor( *pig->mapedit );

	//TEXLIST_DEF
	GtkTreeIter *old_node = gtk_tree_iter_copy( &(pig->node_iter) );
	add_to_tree( tree_view, tree_store, tree_selection, tree_node_parent, old_node );
	gtk_tree_selection_unselect_iter( tree_selection, &node_iter );
	select_tree_row();
	gtk_tree_iter_free( old_node );
	return true;
}


//**********************************************
// Map Item Pigment
//**********************************************
int MapItem_pigment::obj_count = 1;

MapItem_pigment::MapItem_pigment( float val, guint8 c, app_objs *appref ) : MapItem( _("Pigment"), val )
{
color[0]=color[1]=color[2]=c; color[3]=255;
pigment = new PovTexPigment( appref, false );
char text[25];
sprintf( text, "Map pigment #%u", obj_count++ );
pigment->set_name( text );
}

MapItem_pigment::MapItem_pigment( MapItem_pigment & ref  ) : MapItem( ref )
{
for ( int i = 0 ; i < 3 ; i ++ ) color[i] = ref.color[i];
color[3]=255;
pigment = new PovTexPigment( *ref.pigment );
pigment->set_name( ref.pigment->get_name() );
}

void MapItem_pigment::save( ofstream & file )
{
MapItem::save( file );
pigment->save( file );
file << "} ";
}

bool MapItem_pigment::load( ifstream & file, char *ltag )
{
if ( strcmp( "MAPITEM", ltag ) ) return false;
char * tag = NULL;

do
	{
	tag = tvio_get_next_tag( file );
	if ( tag == NULL ) break;
	if ( pigment->load( file , tag ) ) continue;
	if ( MapItem::load( file , tag ) ) continue;
	tvio_skip_section( file );
	}
while( tag != NULL );
return true;
}

//**********************************************
// Map Item Color
//**********************************************
int MapItem_color::obj_count = 1;

MapItem_color::MapItem_color( float val, guint8 c, app_objs *appref ) : MapItem( _("Color"), val ) 
{
color[0]=color[1]=color[2]=c; 
color[3]=255;
label = new char[ strlen(name) + 15 ];
sprintf( label, "%s %u", name, obj_count++ );
name = label;
solcol = new TvWidget_pigment( label, "MAPPIG", NULL, appref );
double col = (float)c /255.0 ;
solcol->set( col, col, col, 0.0 );
func = NULL;
}

MapItem_color::MapItem_color( MapItem_color & ref ) : MapItem( ref )
{
for ( int i = 0 ; i < 4 ; i++ ) color[i] = ref.color[i];
label = new char[ strlen(name) + 15 ];
sprintf( label, "Color %u",  obj_count++ );
name = label;
solcol = new TvWidget_pigment( *ref.solcol );
func = NULL;
}

/*
MapItem_color::MapItem_color( MapItem_color & ref ) : MapItem( ref )
{
for ( int i = 0 ; i < 4 ; i++ ) color[i] = ref.color[i];
label = new char[ strlen(name) + 15 ];
sprintf( label, "%s %u", name, obj_count++ );
name = label;
solcol = new TvWidget_pigment( *ref.solcol );
func = NULL;
}
*/

void MapItem_color::get_editor( GtkWidget *tbox, bool tt )
{
solcol->get_widget( tbox, tt) ;
solcol->connect_signal( sign_mapitemcol_changed, this );
}

void MapItem_color::destroy_editor()
{
solcol->destroy_widget();
func = NULL;
}

void MapItem_color::changed()
{
color[0] = (guint8)(solcol->get_color(0)*255.0);
color[1] = (guint8)(solcol->get_color(1)*255.0);
color[2] =(guint8)( solcol->get_color(2)*255.0);
color[3] = (guint8)( (1.0 - solcol->get_alpha()) * 255.0 );
if ( func != NULL ) (func)(func_data);
}

void MapItem_color::save( ofstream & file )
{
MapItem::save( file );
solcol->save( file );
file << "} ";
}

bool MapItem_color::load( ifstream & file, char *ltag )
{
if ( strcmp( "MAPITEM", ltag ) ) return false;
char * tag = NULL;

do
	{
	tag = tvio_get_next_tag( file );
	if ( tag == NULL ) break;
	if ( solcol->load( file , tag ) ) continue;
	if ( MapItem::load( file , tag ) ) continue;
	tvio_skip_section( file );
	}
while( tag != NULL );
return true;
}
