//*****************************************************************************************
// Truevision - a 3d modeler for gnome and povray
//
// text.cc
//
// Vincent LE PRINCE <vincentleprince@users.sourceforge.net>
// Copyright (C) 2000-2005 Vincent LE PRINCE
// This file is part of the TRUEVISION Package

//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */ 
//*******************************************************************************************
#include "include/text.h"
#include "include/viewmanager.h"
#include "include/objectlist.h"
#include "include/tvio.h"
#include "include/preferences.h"
#include "include/scene.h"
#include "../ftgl/FTGLExtrdFont.h"

//**************************************
// Constructeur
//**************************************
Text::Text( app_objs *appref ) : Object3D_with_material( appref )
{
type = TV_OBJ3D_TEXT;
category = TV_OBJ3D_OBJECTS;
set_name( "Text" );

// Base
location = new ObjParam_point( N_("Location"), "LOC", NULL, app_ref, true );
location->set( 0, 0, 0 );
size = new ObjParam_scale( N_("Scale"), "SCALE", NULL, app_ref, true );
size->set( 0.2, 0.2, 0.2 );
rotation = new ObjParam_rotation( N_("Rotation"), "ROT", NULL, app_ref, true );
rotation->set( 0, 0, 0 );
thickness = new ObjParam_float( N_("Thickness"), "THICK", NULL, app_ref, true, 0.2 );
thickness->set_range( 10, 0, 0.1 );
offset = new ObjParam_float( N_("Offset"), "OFFS", NULL, app_ref, true, 0 );
offset->set_range( 10, 0, 0.1 );
entry = new ObjParam_entry( N_("Text"), "STRING", NULL, app_ref, true, "Text" );
font = new ObjParam_path( N_("Font"), "FONT", NULL, app_ref, true );
PREF_DEF
font->set( pref->fonts_path->value() );   
}

Text::Text( Text & ref ) : Object3D_with_material( ref )
{
location = new ObjParam_point( *ref.location );
size = new ObjParam_scale( *ref.size );
rotation = new ObjParam_rotation( *ref.rotation );
thickness = new ObjParam_float( *ref.thickness );
offset = new ObjParam_float( *ref.offset );
entry = new ObjParam_entry( *ref.entry );
font = new ObjParam_path( *ref.font );
}

Text::~Text()
{
delete location;
delete size;
delete rotation;
delete thickness;
delete offset;
delete entry;
delete font;
}

//**************************************
// Display
//**************************************
void Text::display( glview *view, bool set_col )
{
if ( hidden->value() ) return;

//Object3D::display( view );	
	glPushMatrix();
	if ( set_col ) set_color();
	
	// Position et direction
	gfloat x, y, z;
	size->get( x, y, z );
	glScalef( x, y, z );
	location->get( x, y, z );
	glTranslatef( x, y, z );
	rotation->gl_rotate();


	//FTGLExtrdFont glfont( "/usr/lib/povray-3.5/include/timrom.ttf");   
	FTGLExtrdFont glfont( font->value() );
	if( glfont.Error() == 0 ) 
	{
    glfont.FaceSize( 1);
	glfont.Offset( offset->value() );
	glfont.Depth( thickness->value() );
    glfont.Render( entry->value() );
	}
	
	glPopMatrix();
}


//***********************************************
// Edit
//***********************************************
void Text::edit_widget( GtkWidget *wid )
{
PREF_DEF
bool tt = pref->tooltips->value();
// Options communes
Object3D::edit_widget( wid );

// Options de geometrie
new_table( edit_cont, _("General settings"), 7 );
	entry->get_widget( table, tt, 1 );
	font->get_widget( table, tt, 2 );
	thickness->get_widget( table, tt, 3 );
	offset->get_widget( table, tt, 4 );
	location->get_widget( table, tt, 5 );
	size->get_widget( table, tt, 6 );
	rotation->get_widget( table, tt, 7 );

get_texture_widgets( edit_cont, tt );
gtk_widget_show_all( wid );
}

//***********************************************
// Mouse drag
//***********************************************
void Text::mouse_drag( struct drag_info *drag )
{
VMAN_DEF
OBJLIST_DEF

switch( vmanager->get_pointer_mode() )
	{
	case TV_PMODE_SELECT:
	case TV_PMODE_TRANSLATE:
		location->mouse_drag( drag );
		break;

	case TV_PMODE_SCALE:
		{ size->mouse_drag( drag ); }
		break;

	case TV_PMODE_ROTATE:
		{ rotation->mouse_drag( drag ); break; }

	case TV_PMODE_CUSTOM:
		((ObjParam_point*)(objlist->get_current_param()))->mouse_drag( drag );
		break;

	default: 
		break;
	}
}

//***********************************************
// Pref_changed
//***********************************************
void Text::pref_changed()
{
Object3D::pref_changed();
location->pref_changed();
size->pref_changed();
rotation->pref_changed();
thickness->pref_changed();
offset->pref_changed();
entry->pref_changed();
font->pref_changed();
}

//***********************************************
// Destroy editor
//***********************************************
void Text::destroy_editor()
{
Object3D::destroy_editor();
location->clear_widget();
size->clear_widget();
texture->clear_widget();
rotation->clear_widget();
thickness->clear_widget();
offset->clear_widget();
entry->clear_widget();
font->clear_widget();
}

//***********************************************
// Output to povray
//***********************************************
void Text::output_to_povray_pass1( ofstream & file )
{
file << "\n\n// Text : " << name->value();
file << "\n#declare "; get_underscore_name( file ); file << " ="; 

file << "\ntext {\n\t";
file << "ttf \"" << font->value();
file << "\" \"" <<  entry->value() << "\" ";
file << thickness->value() << ", " << offset->value();
//texture->output_to_povray( file );
Object3D_with_material::output_to_povray_pass1( file );
file << "\n\t";
rotation->output_to_povray( file );
float x,y,z;
location->get( x, y, z );
file << "\n\ttranslate <" << x << "," << y << "," << -z << "> \n\t";
size->get( x, y, z );
file << "\n\tscale <" << x << "," << y << "," << z << "> \n\t";

file << "\n}";
}


//***********************************************
// Save & load
//***********************************************
void Text::save( ofstream & file )
{
file << "\nTEXT{\n";
save_basics( file );
entry->save( file );
font->save( file );
thickness->save( file );
offset->save( file );
location->save( file );
size->save( file );
rotation->save( file );
texture->save( file );
file << "\n}";
}

bool Text::load( ifstream & file, char *ltag )
{
	if ( strcmp( ltag, "TEXT" ) ) 
		return false;
	set_load_progress( file );

	char * tag = NULL;
	do {
		tag = tvio_get_next_tag( file );
		if ( tag == NULL ) break;
	
		if ( load_basics( file, tag ) ) continue;
 		if ( location->load( file, tag ) ) continue;
 		if ( size->load( file, tag ) ) continue;
 		if ( rotation->load( file, tag ) ) continue;
 		if ( thickness->load( file, tag ) ) continue;
 		if ( offset->load( file, tag ) ) continue;
 		if ( entry->load( file, tag ) ) continue;
 		if ( font->load( file, tag ) ) continue;		
		
		tvio_skip_section(file );
		} while ( tag != NULL );

	// Check if font path is valid 
	char *path = font->value();
	ifstream font_file( path, ios::binary );
	if ( font_file == NULL ) {
		SCENE_DEF
		PREF_DEF
		scene->invalid_font_path_detected();
		font->set( pref->fonts_path->value() );   
	} else
		font_file.close();
	
	return true;
}
