/*
 * apitest.c - URLfilterDB
 *
 * ufdbGuard API is copyrighted (C) 2005,2006,2007 by URLfilterDB with all rights reserved.
 *
 * ufdbGuard API is used to integrate the functionality of ufdbGuard into
 * programs of 3rd parties.
 *
 * This API test skeleton assumes that the program that uses the API is
 * multi-threaded.
 * In case of a single-threaded application, the macro UFDB_API_NO_THREADS must
 * be defined in the Makefile:
 * CFLAGS= -DUFDB_API_NO_THREADS ...
 *
 * RCS $Id: apitest.c,v 1.19 2007/11/20 16:36:32 root Exp root $
 */

#define UFDB_API_CHECK_PROXY_TUNNELS


#include "ufdb-api.h"

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>

#ifdef UFDB_API_CHECK_PROXY_TUNNELS
#include <pthread.h>
#include "ufdbchkport.h"
#endif


/*
 *  Change this macro to point to the local database directory.
 *  ===========================================================
 */
#define DBDIR "./urlfilterdb"



/*
 * Global data structure for the database categories:
 */
static UFDBusedCategory usedCategory[UFDB_MAX_CATEGORIES];
static int numUsedCategories;

/*
 * Each thread needs an administration area.  Apitest has only 1 thread.
 */
static UFDBthreadAdmin * admin;



/*
 * Initialise an aray with meta data of all known categories.
 * NOTE: *all* categories must be loaded if you want to upload
 * 'uncategorised URLs' to URLfilterDB since a URL can only be
 * uploaded if it is not in any existing category.  This is to
 * prevent the upload of a URLs that is in a category that is
 * allowed by a local iinternet access policy.
 */
void initialiseCategories( char * databaseDirectory )
{
   int     i;
   char    filename1[1024];
   char    filename2[1024];
   /*************************************************************************
   ***
   *** NOTE: the order of categories is important because URLs can be part of
   ***       more than 1 category.  E.g. 
   ***       www.usatoday.com         is in news
   ***       www.usatoday.com/sports  is in news and sport
   ***                                and depending on the order, it is
   ***                                classified as "sport" or "news".
   **************************************************************************/
   char *  categories[] = { 
		"always_deny",		/* "always_deny" must be the 1st ! */
		"always_allow", 	/* "always_allow" must be the 2nd ! */
   		"ads", 
		"proxies",
		"adult", 
		"warez", 
		"illegal", 
		"violence", 
		"gambling",
		"drugs", 
		"webmail",
		"dating", 
		"chat", 
		"forum", 
		"private", 
		"audio-video", 
		"sports", 
		"finance", 
		"jobs", 
		"games", 
		"entertain",
		"shops", 
		"travel", 
		"news", 
		"toolbars", 
		"checked",		/* "checked" MUST be the last category ! */
		NULL
	 };

   numUsedCategories = 0;
   for (i = 0; categories[i] != NULL; i++)
   {
      usedCategory[i].name = categories[i];
      usedCategory[i].block = 0;
      usedCategory[i].userData = NULL;
      sprintf( filename1, "%s/%s/domains.ufdb", databaseDirectory, categories[i] );
      sprintf( filename2, "%s/%s/expressions",  databaseDirectory, categories[i] );

      if (UFDBloadCategory( filename1, filename2, &usedCategory[i].handle ) != 0)
      {
	 fprintf( stderr, "ERROR loading category %s\n", usedCategory[i].name );
      }
      else
         printf( "loaded category %s\n", usedCategory[i].name );
   }
   numUsedCategories = i;
}


/*
 * Mark a category that it must be used to block URLs.
 */
void blockCategory( char * category )
{
   int i;

   for (i = 0; i < numUsedCategories; i++)
   {
      if (strcmp( category, usedCategory[i].name ) == 0)
      {
         usedCategory[i].block = 1;
	 return;
      }
   }

   fprintf( stderr, "ERROR bad category to block: %s\n", category );
}


void applInit( void )
{
   /*
    * Load all categories into memory.
    */
   initialiseCategories( DBDIR );

   /*
    * per-thread initialisation:
    */
   admin = UFDBallocThreadAdmin();

#if defined(UFDB_API_CHECK_PROXY_TUNNELS)
   /*********************************************************************************
    *  When https proxy tunnel checks are performed in a non-agressive way, 
    *  the threads that actually do te verifications must be initialised and
    *  the verification method must be set: agressive/queued.
    *  The HTTPS/SLL code needs to be initialised with UFDBinitHTTPSchecker().
    ********************************************************************************/
   {
      pthread_t      th[8];
      pthread_attr_t attr;

      UFDBinitHTTPSchecker();
      UFDBsetTunnelCheckMethod( UFDB_API_HTTPS_CHECK_AGRESSIVE );

      pthread_attr_init( &attr );
      pthread_attr_setscope( &attr, PTHREAD_SCOPE_SYSTEM );
      pthread_attr_setstacksize( &attr, 256 * 1024 );

      (void) pthread_create( &th[0], &attr, UFDBhttpsTunnelVerifier, NULL );
      (void) pthread_create( &th[1], &attr, UFDBhttpsTunnelVerifier, NULL );
      (void) pthread_create( &th[2], &attr, UFDBhttpsTunnelVerifier, NULL );
      (void) pthread_create( &th[3], &attr, UFDBhttpsTunnelVerifier, NULL );
      (void) pthread_create( &th[4], &attr, UFDBhttpsTunnelVerifier, NULL );
      (void) pthread_create( &th[5], &attr, UFDBhttpsTunnelVerifier, NULL );
      (void) pthread_create( &th[6], &attr, UFDBhttpsTunnelVerifier, NULL );
      (void) pthread_create( &th[7], &attr, UFDBhttpsTunnelVerifier, NULL );
   }
#endif
}


void applTerminate( void )
{
   int i;

   for (i = 0; i < numUsedCategories; i++)
   {
      UFDBunloadCategory( &usedCategory[i].handle );
      printf( "unloaded category %s\n", usedCategory[i].name );
   }

   /* IF UPLOAD_UNCATEGORISED_URLS: */
   (void) UFDBuploadUncategorisedURLs( "apitest" );
}


void MYAPPverifyURL(
   int          feedback,
   char *       URL )	 /* e.g. http://user:password@www.asite.com:8080/cgi-bin/x.pl?a=0 */
{
   int          i;
   int          blockurl;
   UFDBrevURL * revUrl;
   int		port;
   char         protocol[16];
   char         strippedUrl[1024];
   char         domain[1024];

   blockurl = 0;
   UFDBstripURL( URL, strippedUrl, domain, protocol, &port );

   if (feedback)
      printf( "\nMYAPPverifyURL  URL=%s  stripped=%s\n", URL, strippedUrl );

   /* revURL conversion is outside the loop */
   revUrl = UFDBgenRevURL( admin, (unsigned char *) strippedUrl );

   for (i = 0; i < numUsedCategories; i++)
   {
      if (usedCategory[i].block)
      {
         if (UFDBverifyUrlCategory( strippedUrl, revUrl, &usedCategory[i] ))
	 {
	    blockurl = 1;
	    break;
	 }
#ifdef UFDB_API_CHECK_PROXY_TUNNELS
	 if (strcmp( usedCategory[i].name , "proxies" ) == 0  &&
             (port == 443 || port == 563 || strcasecmp( protocol, "https" ) == 0)  &&
	     UFDBcheckForHTTPStunnel( domain, port, 0 ) == UFDB_API_ERR_TUNNEL)
	 {
	    blockurl = 1;
	    break;
	 }
#endif
      }
   }

   if (feedback)
   {
      if (blockurl)
	 printf( "URL %s blocked since it is in category %s\n", URL, usedCategory[i].name );
      else
	 printf( "URL %s is OK\n", URL );
   }

   /* IF UPLOAD_UNCATEGORISED_URLS: */
   if (!blockurl)
   {
      static unsigned int sample = 0;

      if (++sample % 11 == 0)	/* a sample of 9% of URLs are verified */
      {
         if (UFDBVerifyURLisUncategorised( strippedUrl, revUrl, usedCategory, numUsedCategories ))
	 {
	    char * slash;
	    slash = strchr( strippedUrl, '/' );
	    if (slash != NULL)
	       *slash = '\0';
	    UFDBsaveUncategorisedURL( strippedUrl );
	 }
      }
   }

   UFDBfreeRevURL( admin, revUrl );
}


int main( int argc, char * argv[] )
{
   applInit();

   /*
    * Implement the local internet access policy.
    */
   blockCategory( "ads" );
   blockCategory( "adult" );
   blockCategory( "news" );
   blockCategory( "proxies" );
   blockCategory( "sports" );
   blockCategory( "warez" );
   blockCategory( "webmail" );

   if (argc > 1  &&  strcmp(argv[1],"stdin")==0)	/* test many URLs from stdin */
   {
      int        n;
      struct tms t;
      char       line[1024];

      UFDBtimerInit( &t );
      n = 0;

      while (fgets(line,1024,stdin) != NULL)
      {
	 MYAPPverifyURL( 0, line );
	 n++;
      }

      UFDBtimerStop( &t );
      UFDBtimerPrintString( line, &t, "" );
      printf( "%d URL verifications%s\n", n, line );
   }
   else							/* test a few hardcoded URLs */
   {
      MYAPPverifyURL( 1, "http://www.google.com" );
      MYAPPverifyURL( 1, "http://www.usatoday.com/index.html" );
      MYAPPverifyURL( 1, "http://www.usatoday.com/sports/index.html" );
      MYAPPverifyURL( 1, "http://www.usatoday.com//sports/index.html" );
      MYAPPverifyURL( 1, "http://www.usatoday.com/./sports/index.html" );
      MYAPPverifyURL( 1, "http://www.usatoday.com/.//.///sports/index.html" );
      MYAPPverifyURL( 1, "http://www.sex.com" );
      MYAPPverifyURL( 1, "http://www.sex.com." );
      MYAPPverifyURL( 1, "http://www.sex.com./index.html" );
      MYAPPverifyURL( 1, "http://www.sex.com.:8000" );
      MYAPPverifyURL( 1, "http://ads.asite.com/foobar/index.html" );
      MYAPPverifyURL( 1, "http://Ads.asite.com:8080/foobar2/index.html" );
      MYAPPverifyURL( 1, "http://user:pw@ads.asite.com:8080/foobar2/index.html" );
      MYAPPverifyURL( 1, "http://%41ds.asite.com/foobar2/index.html" );

      MYAPPverifyURL( 1, "https://www.usatoday.com/index.html" );
      MYAPPverifyURL( 1, "https://webmail.xs4all.nl/src/login.php" );
      MYAPPverifyURL( 1, "https://www.urlfilterdb.com:9443/abc" );

      /* test upload uncategorised URLs */
      MYAPPverifyURL( 1, "http://www.urlfilterdb.atld1" );
      MYAPPverifyURL( 1, "http://www.urlfilterdb.atld2" );
      MYAPPverifyURL( 1, "http://www.urlfilterdb.atld3" );
      MYAPPverifyURL( 1, "http://www.urlfilterdb.atld4" );
      MYAPPverifyURL( 1, "http://www.urlfilterdb.atld5" );
      MYAPPverifyURL( 1, "http://www.urlfilterdb.atld6" );
      MYAPPverifyURL( 1, "http://www.urlfilterdb.atld7" );
      MYAPPverifyURL( 1, "http://www.urlfilterdb.atld8" );
      MYAPPverifyURL( 1, "http://www.nonexistent-domain.de.eu/fr/index.shtml" );
      MYAPPverifyURL( 1, "http://www.nonexistent-domain.nl.eu/fr/index.shtml" );
      MYAPPverifyURL( 1, "http://www.nonexistent-domain.fr.eu/fr/index.shtml" );
      MYAPPverifyURL( 1, "http://www.nonexistent-domain.es.eu/fr/index.shtml" );
      MYAPPverifyURL( 1, "http://www.nonexistent-domain.pt.eu/fr/index.shtml" );
      MYAPPverifyURL( 1, "http://www.nonexistent-domain.be.eu/fr/index.shtml" );
      MYAPPverifyURL( 1, "http://www.nonexistent-domain.dk.eu/fr/index.shtml" );
      MYAPPverifyURL( 1, "http://www.nonexistent-domain.it.eu/fr/index.shtml" );

      printf( "\n" );
   }

   applTerminate();

   return 0;
}

