/* 
 * httpsQueue.c
 * 
 * communication between the worker and other threads is done via
 * RW-locks and data is passed via a queue.
 *
 * Since this queue is a FIFO we could use a tail queue. But we don't
 * like the malloc() overhead, so therefore we use a static array.
 *
 * RCS $Id: httpsQueue.c,v 1.8 2007/11/13 16:41:00 root Exp root $
 */

#include "ufdb.h"
#include "ufdblib.h"
#include "sg.h"
#include "httpsQueue.h"

#include <unistd.h>
#include <string.h>
#include <pthread.h>

#define UFDB_HTTPS_QUEUE_SIZE	 128

static struct {
   int    portnumber;
   char   hostname[1028];
} q[UFDB_HTTPS_QUEUE_SIZE];

static int    ihead = 0;                    /* array index for the head */
static int    itail = 0;                    /* array index for the tail */
volatile int  _ufdb_httpsq_n_queued = 0;
static pthread_mutex_t httpsq_lock = UFDB_STATIC_MUTEX_INIT;
static pthread_cond_t  empty = PTHREAD_COND_INITIALIZER;


/*
 * enqueue a hostname:portnumber for future https tunnel detection.
 */
int ufdbHttpsQueueRequest( char * hostname, int portnumber )
{
   int i;
   int rv;

   rv = pthread_mutex_lock( &httpsq_lock );
#ifdef UFDB_DEBUG
   if (rv != 0)
      ufdbLogError( "ufdbHttpsQueueRequest: mutex_lock failed rv=%d *****", rv );
#endif

   /*
    * prevent duplicates: check if the request is already queued.
    */
   for (i = ihead; i != itail; i = (i+1) % UFDB_HTTPS_QUEUE_SIZE)
   {
      if (strcmp( hostname, q[i].hostname ) == 0  &&
          q[i].portnumber == portnumber)
      {
	 rv = pthread_mutex_unlock( &httpsq_lock );
#ifdef UFDB_DEBUG
	 if (rv != 0)
	    ufdbLogError( "ufdbHttpsQueueRequest: mutex_unlock failed rv=%d *****", rv );
#endif
         return 0;
      }
   }

   if (_ufdb_httpsq_n_queued < UFDB_HTTPS_QUEUE_SIZE)
   {
      /*
       * insert the request at the tail
       */
      q[itail].portnumber = portnumber;
      strcpy( q[itail].hostname, hostname );

      itail = (itail + 1) % UFDB_HTTPS_QUEUE_SIZE;
      _ufdb_httpsq_n_queued++;

      /*
       * leave the critical section
       */
      rv = pthread_mutex_unlock( &httpsq_lock );
#ifdef UFDB_DEBUG
      if (rv != 0)
	 ufdbLogError( "ufdbHttpsQueueRequest: mutex_unlock failed rv=%d *****", rv );
#endif

      /*
       * signal anyone who is waiting
       */
      pthread_cond_broadcast( &empty );
   } 
   else    
   {
      /* queue is full; just drop the request */
      ufdbLogError( "ufdbHttpsQueueRequest: https security/tunnel detection queue is full." );

      rv = pthread_mutex_unlock( &httpsq_lock );
#ifdef UFDB_DEBUG
      if (rv != 0)
	 ufdbLogError( "ufdbHttpsQueueRequest: (full) mutex_unlock failed rv=%d *****", rv );
#endif
      return -1;
   }

   return 0;
}


/*
 * Get a queued https tunnel check request.
 */
void ufdbGetHttpsRequest( char * hostname, int * portnumber )
{
   int rv;

allover:
   rv = pthread_mutex_lock( &httpsq_lock );
#ifdef UFDB_DEBUG
   if (rv != 0)
      ufdbLogError( "ufdbGetHttpsRequest: mutex_lock failed with rv=%d *****", rv );
#endif

   while (1)
   {
      /*
       * if there are requests in the queue
       */
      if (_ufdb_httpsq_n_queued > 0)
      {
         _ufdb_httpsq_n_queued--;
         /*
          * get the request at the head
          */
#if 0
	 ufdbLogMessage( "ufdbGetHttpsRequest: ihead=%d  %s:%d", 
	                 ihead, q[ihead].hostname, q[ihead].portnumber );
#endif
         *portnumber = q[ihead].portnumber;
	 strcpy( hostname, q[ihead].hostname );
         ihead = (ihead + 1) % UFDB_HTTPS_QUEUE_SIZE;

         pthread_mutex_unlock( &httpsq_lock );

	 return;
      } 
      else   /* otherwise wait until there are fds available */ 
      {
         pthread_cond_wait( &empty, &httpsq_lock );
         /*
          * I have been signaled because the queue
          * is not empty.  Go try again.
	  * usleep() to try to prevent a race with old pthread libraries.
          */
         rv = pthread_mutex_unlock( &httpsq_lock );
#ifdef UFDB_DEBUG
	 if (rv != 0)
	    ufdbLogError( "ufdbGetHttpsRequest: mutex_unlock failed with rv=%d *****", rv );
#endif
	 usleep( ((unsigned long) pthread_self()) % 1025 );
	 goto allover;
      }
   }
}

