/* 
 *  ufdbguardd.c  -  multithreaded ufdbGuard daemon
 *
 *  $Id: ufdbguardd.c,v 1.44 2007/11/26 15:31:39 root Exp root $
 */

#include "ufdb.h"
#include "sg.h"
#include "ufdbchkport.h"
#include "httpsQueue.h"
#include "httpserver.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>
#include <ctype.h>
#include <signal.h>
#include <libgen.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/tcp.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <sys/utsname.h>
#ifdef HAVE_SYS_SYSCALL_H
#include <sys/syscall.h>
#endif
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include <sys/socket.h>
#if HAVE_UNIX_SOCKETS
#include <sys/un.h>
#endif
#include <netinet/tcp.h>

#ifdef __linux__
#include <execinfo.h>
#endif

#define FULL 1
#define EMPTY 0

pthread_mutex_t sighup_mutex = UFDB_STATIC_MUTEX_INIT;

#define UFDB_MAX_WORKERS	UFDB_MAX_THREADS
static int     n_workers = 33;

static int     my_argc;
static char ** my_argv;
static char ** my_envp;
static char *  configFile = NULL;
static struct timeval  start_time;

#define UFDBGUARDD_PID_FILE	"/var/tmp/ufdbguardd.pid"
char * globalPidFile = UFDBGUARDD_PID_FILE;

extern int globalPid;

extern int globalFatalError;
extern int globalDebugRedirect;
extern int globalDebugTimeDelta;
extern int globalDebugRedirect;
extern int sig_other;
extern char ** globalArgv;
extern char ** globalEnvp;

extern int analyseUncategorised;
extern int httpsConnectionCacheSize;

extern struct UFDBmemTable knownDb;

static int        portNumCmdLine = -1;
extern int        portNum;

static int        runAsDaemon = 1;
static int        noRedirects = 0;
static int        parseOnly = 0;
static pthread_t  HUPhandler;
static pthread_t  sockethandler;
static pthread_t  workers[UFDB_MAX_WORKERS];
static pthread_t  httpsthr[UFDB_NUM_HTTPS_VERIFIERS];
static pid_t      httpDaemonPid = 0;
#define PS_DIED       1
#define PS_STARTED    2
#define PS_RUNNING    3
#define PS_TOBEKILLED 4
static int        httpDaemonStatus = PS_DIED;

volatile int UFDBreconfig = 1;

volatile unsigned long UFDBlookupCounter = 0;
volatile unsigned long UFDBlookupHttpsCounter = 0;
volatile unsigned long UFDBblockCounter = 0;
volatile unsigned long UFDBtestBlockCounter = 0;
volatile unsigned long UFDBuncategorisedCounter = 0;
volatile unsigned long UFDBsafesearchCounter = 0;
volatile unsigned long UFDBuploadSeqNo = 0;

static void saveUncategorisedURLs( void );
static void * worker_main( void * ptr );

#ifdef UFDB_DEBUG
extern int yydebug;
#endif


static void usage( char option )
{
   if (option != '\0')
      fprintf( stderr, "unknown option '-%c'\n", option );

   fprintf( stderr, "usage: ufdbguardd [-A] [-d] [-v] [-r] [-T] [-h] [-w 8-64] [-p port] [-c <configfile>]\n" );
   fprintf( stderr, "-v      print version\n" );
   fprintf( stderr, "-c F    use configuration file F\n" );
   fprintf( stderr, "-T      test mode; do not block, only log which URLs might be blocked\n" );
   fprintf( stderr, "-w N    use N worker threads (default: 32)\n" );
   fprintf( stderr, "-p N    use TCP port N to listen on\n" );
   fprintf( stderr, "-U user run with privileges of \"user\"\n" );
   fprintf( stderr, "-d      extra debug information in log file\n" );
   fprintf( stderr, "-r      log all redirections\n" );
   fprintf( stderr, "-N      do not analyse uncategorised URLs\n" );
   fprintf( stderr, "-h      print this help text\n" );

   if (option != '\0')
      exit( 1 );
   else
      exit( 0 );
}


static void writePidFile( void )
{
   FILE * fp;

   (void) unlink( globalPidFile );
   fp = fopen( globalPidFile, "w" );
   if (fp == NULL)
      ufdbLogError( "cannot write to PID file %s", globalPidFile );
   else
   {
      fprintf( fp, "%d\n", globalPid );
      fclose( fp );
   }
}


static void removePidFile( void )
{
   FILE * fp;

#if HAVE_UNIX_SOCKETS
   char   unixsocket[64];

   sprintf( unixsocket, "/tmp/ufdbguardd-%05d", portNum );
   if (unlink( unixsocket ) < 0  && errno != ENOENT)
      ufdbLogError( "cannot remove socket %s: %s", unixsocket, strerror(errno) );
#endif

   (void) unlink( globalPidFile );

   if ((httpDaemonStatus == PS_STARTED || httpDaemonStatus == PS_RUNNING)  &&  UFDBglobalHttpdPort > 0)
   {
      httpDaemonStatus = PS_TOBEKILLED;
      fp = fopen( UFDBHTTPD_PID_FILE, "r" );
      if (fp != NULL)
      {
         if (fscanf( fp, "%d", &httpDaemonPid ) == 1)
	 {
	    ufdbLogMessage( "sending TERM signal to the http daemon (pid=%d)", httpDaemonPid );
	    if (kill( httpDaemonPid, SIGTERM ) < 0)
	       ufdbLogError( "cannot kill pid %d: %s", httpDaemonPid, strerror(errno) );
	 }
         fclose( fp );
      }
   }
}


static void catchHUPSignal( int signal )
{
   pthread_kill( HUPhandler, SIGHUP );
}


static void catchUSR1Signal( int signal )
{
   pthread_kill( HUPhandler, SIGUSR1 );
}


static void catchChildSignal( int signal )
{
   int   status;
   pid_t pid;

   /* a child process died.  This can only be the ufdbhttpd process... */

   while ((pid = waitpid( -1, &status, WNOHANG )) > 0)
   {
      if (pid == httpDaemonPid)
      {
	 if (WIFSIGNALED(status))
	 {
	    ufdbLogError( "ufdbhttpd unexpectedly died by a signal with exit status %d  *****", WEXITSTATUS(status) );
	 }
         else if (WIFEXITED(status))
	 {
	    /* ufdbhttpd daemonizes itself */
	    if (httpDaemonStatus == PS_STARTED)
	       httpDaemonStatus = PS_RUNNING;
	    /* were we expected the child to die ? if not, report an error */
	    else if (httpDaemonStatus != PS_TOBEKILLED)
	    {
	       ufdbLogError( "ufdbhttpd unexpectedly died with exit status %d  *****", WEXITSTATUS(status) );
	    }
	    else
	       httpDaemonStatus = PS_DIED;
	 }
      }
   }
}


/* 
 * queue.c
 * 
 * communication between the worker and other threads is done via
 * RW-locks and data is passed via a queue.
 *
 * Since this queue is a FIFO we could use a tail queue. But we don't
 * like the malloc() overhead, so therefore we use a static array.
 *
 * RCS $Id: ufdbguardd.c,v 1.44 2007/11/26 15:31:39 root Exp root $
 */

#define UFDB_MAX_FD_QUEUE   UFDB_MAX_THREADS


static struct {
   int    fd;
} q[UFDB_MAX_FD_QUEUE];

static int    ihead = 0;                    /* array index for the head */
static int    itail = 0;                    /* array index for the tail */
volatile static int    n_queued = 0;
static pthread_mutex_t fdq_lock = UFDB_STATIC_MUTEX_INIT;
static pthread_cond_t  empty = PTHREAD_COND_INITIALIZER;


/*
 * enqueue a new fd.
 */
static void putFdQueue( int fd )
{
   int ret;

   ret = pthread_mutex_lock( &fdq_lock );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "putFdQueue: mutex_lock failed with code %d *****", ret );
#endif

   if (n_queued < UFDB_MAX_FD_QUEUE)
   {
      /*
       * insert it at the tail
       */
      q[itail].fd = fd;

      itail = (itail + 1) % UFDB_MAX_FD_QUEUE;
      n_queued++;

      /*
       * leave the critical section
       */
      ret = pthread_mutex_unlock( &fdq_lock );
#ifdef UFDB_DEBUG
      if (ret != 0)
	 ufdbLogError( "putFdQueue: mutex_unlock failed with code %d *****", ret );
#endif

      /*
       * signal anyone who is waiting
       */
      pthread_cond_broadcast( &empty );
   } 
   else    
   {
      ufdbLogFatalError( "FATAL ERROR: connection queue is full." );
      ufdbLogFatalError( "There are too many ufdbgclient processes !!" );
      ufdbLogFatalError( "the maximum is %d when the -w option is used", UFDB_MAX_FD_QUEUE );
      ufdbLogMessage( "closing fd %d", fd );
      /* there is no other option than to close the connection */
      /* TO-DO: maybe: close all queued fd's ? */
      close( fd );
   }
}


/*
 * get a fd where there is new content.
 */
static void getFdQueue( int * fd )
{
   int ret;

allover:
   ret = pthread_mutex_lock( &fdq_lock );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "getFdQueue: mutex_lock failed with code %d *****", ret );
#endif

   while (1)
   {
      /*
       * if there are jobs in the queue
       */
      if (n_queued > 0)
      {
         n_queued--;
         /*
          * get the one at the head
          */
         *fd = q[ihead].fd;
         ihead = (ihead + 1) % UFDB_MAX_FD_QUEUE;

         ret = pthread_mutex_unlock( &fdq_lock );
#ifdef UFDB_DEBUG
	 if (ret != 0)
	    ufdbLogError( "getFdQueue: mutex_unlock failed with code %d *****", ret );
#endif

	 return;
      } 
      else   /* otherwise wait until there are fds available */ 
      {
         pthread_cond_wait( &empty, &fdq_lock );
         /*
          * when I'm here, I've been signaled because there
          * are jobs in the queue.  Go try and get one.
          */
	 ret = pthread_mutex_unlock( &fdq_lock );
#ifdef UFDB_DEBUG
	 if (ret != 0)
	    ufdbLogError( "getFdQueue: mutex_unlock failed with code %d *****", ret );
#endif
	 usleep( ((unsigned long) pthread_self()) % 821 );
	 goto allover;
      }
   }
}


static void daemon_accept_connections( int s, int protocol )
{
   int newfd;
   int size;
   int sock_parm;
   struct timeval tv;
   struct linger  linger;

   /*
    * Allow that this thread can be killed without delays at any time.
    */
   pthread_setcancelstate( PTHREAD_CANCEL_ENABLE, NULL );
   pthread_setcanceltype( PTHREAD_CANCEL_ASYNCHRONOUS, NULL );

   size = sizeof(struct sockaddr);
   while (!sig_other)
   {
      errno = 0;
      newfd = accept( s, NULL, NULL );
      if (newfd < 0)
      {
	 if (errno == EINTR)
	 {
	    continue;
	 }
	 ufdbLogError( "SRV: accept on socket failed: %s", strerror(errno) );
	 return;
      }
      linger.l_onoff = 1;
      linger.l_linger = 1;
      setsockopt( newfd, SOL_SOCKET, SO_LINGER, (void *) &linger, sizeof(linger) );

      if (protocol == AF_INET)
      {
	 sock_parm = 1;
	 setsockopt( newfd, IPPROTO_TCP, TCP_NODELAY, (void *) &sock_parm, sizeof(sock_parm) );
	 sock_parm = 1;
	 setsockopt( newfd, SOL_SOCKET, SO_KEEPALIVE, (void *) &sock_parm, sizeof(sock_parm) );
	 sock_parm = 16384;
	 setsockopt( newfd, SOL_SOCKET, SO_SNDBUF, (void *) &sock_parm, sizeof(sock_parm) );
	 sock_parm = 16384;
	 setsockopt( newfd, SOL_SOCKET, SO_RCVBUF, (void *) &sock_parm, sizeof(sock_parm) );
      }

      /*
       *  In case of troubles, the timeout prevents that a thread will block for ever.
       */
#if 0
      tv.tv_sec = 16 * 60;
      tv.tv_usec = 0;
      setsockopt( newfd, SOL_SOCKET, SO_RCVTIMEO, (void *) &tv, sizeof(tv) );
#endif
      tv.tv_sec = 1 * 30;	/* 30 seconds */
      tv.tv_usec = 0;
      setsockopt( newfd, SOL_SOCKET, SO_SNDTIMEO, (void *) &tv, sizeof(tv) );

      if (UFDBglobalDebug)
	 ufdbLogMessage( "SRV: accept new fd %2d", newfd );

      putFdQueue( newfd );
   }
}


static void adjustNumberOfThreads( void )
{
   int i;
   pthread_attr_t      attr;

   /* IF the check_https_tunnel_option is agressive we need more worker threads */
   if (UFDBgetTunnelCheckMethod() == UFDB_API_HTTPS_CHECK_AGRESSIVE  &&  n_workers < 32)
   {
      /* create more worker threads */
      i = n_workers;
      n_workers = 32;
      pthread_attr_init( &attr );
      pthread_attr_setscope( &attr, PTHREAD_SCOPE_SYSTEM );
      pthread_attr_setstacksize( &attr, 256 * 1024 );
      for (; i < n_workers; i++)
	 pthread_create( &workers[i], &attr, worker_main, (void *) ((long) i) );
      ufdbLogMessage( "#threads is increased to %d due to agressive proxy tunnel verifications", n_workers );
   }
}


static void startHttpDaemon( void )
{
   int    i;
   char * argv[24];
   struct stat statbuf;
   char   portStr[16];
   char   httpdbin[1024];

   sprintf( httpdbin, "%s/%s", DEFAULT_BINDIR, "ufdbhttpd" );
   if (stat( httpdbin, &statbuf ) < 0)
   {
      ufdbLogError( "cannot access http daemon executable (%s): %s  *****", httpdbin, strerror(errno) );
      return;
   }

   if (UFDBglobalUserName[0] != '\0')
   {
      /* go back to user root to allow ufdbhttpd to use privileged ports */
      if (seteuid( 0 ) != 0)	  
         ufdbLogError( "pre-fork setuid(0) failed: %s", strerror(errno) );
      else if (UFDBglobalDebug)
         ufdbLogMessage( "pre-fork: setuid(0) was OK" );
   }

   httpDaemonStatus = PS_STARTED;
   httpDaemonPid = fork();
   if (httpDaemonPid == 0)	/* child */
   {
      i = 0;
      argv[i++] = httpdbin;
      if (UFDBglobalDebug)
         argv[i++] = "-d";
      argv[i++] = "-p";
      sprintf( portStr, "%d", UFDBglobalHttpdPort );
      argv[i++] = portStr;
      argv[i++] = "-l";
      if (UFDBglobalLogDir == NULL)
	 argv[i++] = DEFAULT_LOGDIR;
      else
	 argv[i++] = UFDBglobalLogDir;
      argv[i++] = "-I";
      if (UFDBglobalHttpdImagesDirectory[0] == '\0')
	 strcpy( UFDBglobalHttpdImagesDirectory, DEFAULT_IMAGESDIR );
      argv[i++] = UFDBglobalHttpdImagesDirectory;
      argv[i++] = "-i";
      argv[i++] = UFDBglobalHttpdInterface;
      if (UFDBglobalUserName[0] != '\0')
      {
	 argv[i++] = "-U";
	 argv[i++] = UFDBglobalUserName;
	 if (UFDBglobalHttpdPort < 1024)
	 {
	    /* go back to user root to allow ufdbhttpd to use privileged ports */
	    if (setuid( 0 ) != 0)	  
	    {
	       ufdbLogError( "Cannot change userid to root to start ufdbhttpd.  "
			     "Privileged ports (< 1024) cannot be used." );
	    }
	 }
      }
      argv[i] = NULL;
#ifdef _POSIX_PRIORITY_SCHEDULING
      sched_yield();	/* allow parent to go first */
#endif
      ufdbLogMessage( "starting ufdbhttpd ..." );
      execv( httpdbin, argv );
      ufdbLogError( "failed starting http daemon: execv failed: %s", strerror(errno) );
      _exit( 2 );
   }

   if (UFDBglobalUserName[0] != '\0')
      UFDBdropPrivileges( UFDBglobalUserName );

   if (httpDaemonPid < 0)
   {
      ufdbLogError( "cannot start http daemon: fork failed: %s", strerror(errno) );
      httpDaemonPid = 0;
      httpDaemonStatus = PS_DIED;
   }
   else
   {
      ufdbLogMessage( "ufdbhttpd started.  port=%d interface=%s images=%s",
                      UFDBglobalHttpdPort, UFDBglobalHttpdInterface, UFDBglobalHttpdImagesDirectory );
   }
}


void * accept_main( void * ptr )
{
   int                 protocol;
   sigset_t            signals;
   int                 i;
   int                 s;
   int                 sock_parm;
   int                 retval;
   pthread_attr_t      attr;
   struct sockaddr_in  addr_in;
   struct linger       linger;
   struct tms 	       timer;
   char                timeString[256];
#if HAVE_UNIX_SOCKETS
   struct sockaddr_un  addr_un;
#endif

   /* The HUP signal must be blocked.
    * This is a requirement to use sigwait() in a thread.
    */
   sigemptyset( &signals );
   sigaddset( &signals, SIGHUP );
   pthread_sigmask( SIG_BLOCK, &signals, NULL );

   UFDBtimerInit( &timer );
   gettimeofday( &start_time, NULL );

   globalArgv = my_argv;
   globalEnvp = my_envp;
   ufdbSetGlobalErrorLogFile();
   globalFatalError = 0;
#ifdef UFDB_DEBUG
   yydebug = 1;
#endif
   sgReadConfig( configFile );
   /* ufdbSetGlobalErrorLogFile(); */

   adjustNumberOfThreads();

   UFDBinitHTTPSchecker();
   usleep( 100000 );
   pthread_attr_init( &attr );
   pthread_attr_setscope( &attr, PTHREAD_SCOPE_SYSTEM );
   pthread_attr_setstacksize( &attr, 256 * 1024 );
   for (i = 0; i < UFDB_NUM_HTTPS_VERIFIERS; i++)
   {
      pthread_create( &httpsthr[i], &attr, UFDBhttpsTunnelVerifier, (void *) ((long) i) );
      usleep( 10000 );
   }

   if (noRedirects)
      ufdbLogMessage( "-T option is used.  In test mode no URLs are ever blocked." );
   if (globalFatalError)
      ufdbLogError( "A FATAL ERROR OCCURRED: ALL REQUESTS ARE ANSWERED WITH \"OK\" (see previous lines)  *****" );

   /*
    * Create the daemon socket that the it accepts connections on.
    * if available, first try a UNIX socket and then an IP socket.
    */
#if HAVE_UNIX_SOCKETS
   protocol = AF_UNIX;
   s = socket( protocol, SOCK_STREAM, 0 );
   if (s < 0)
   {
      protocol = AF_INET;
      s = socket( protocol, SOCK_STREAM, 0 );
   }
#else
   protocol = AF_INET;
   s = socket( protocol, SOCK_STREAM, 0 );
#endif

   if (s < 0)
   {
      ufdbLogError( "cannot create daemon socket: %s", strerror(errno) );
      pthread_exit( (void *) 1 );    /* TO-DO */
   }

   /* 
    * The port number can be specified in the config file (portNum is set by the parser)
    * or a command line parameter (portNumCmdLine is set in main).
    * The command line setting has preference.
    */
   if (portNumCmdLine >= 0)
      portNum = portNumCmdLine;

#if HAVE_UNIX_SOCKETS
   if (protocol == AF_UNIX)
   {
      sprintf( addr_un.sun_path, "/tmp/ufdbguardd-%05d", portNum );
#if 0
      if (unlink( addr_un.sun_path ) < 0  && errno != ENOENT)
         ufdbLogError( "cannot remove socket %s: %s", addr_un.sun_path, strerror(errno) );
#endif
      addr_un.sun_family = AF_UNIX;
      retval = bind( s, (struct sockaddr *) &addr_un, sizeof(addr_un) );
      /* allow anybody to connect to the socket */
      if (retval >= 0)
         chmod( addr_un.sun_path, S_IRUSR|S_IWUSR| S_IRGRP|S_IWGRP| S_IROTH|S_IWOTH );
   }
   else
#endif
   {
      /*
       * Allow server-side addresses to be reused (don't have to wait for timeout).
       */
      sock_parm = 1;
      setsockopt( s, SOL_SOCKET, SO_REUSEADDR, (void *) &sock_parm, sizeof(sock_parm) );

      addr_in.sin_family = AF_INET;
      addr_in.sin_addr.s_addr = htonl( INADDR_ANY );
      addr_in.sin_port = htons( portNum );
      retval = bind( s, (struct sockaddr *) &addr_in, sizeof(addr_in) );
   }

   if (retval < 0)
   {
      ufdbLogError( "cannot bind daemon socket: %s (protocol=%s)", strerror(errno), 
                    protocol==AF_INET?"IP":"UNIX" );
      fprintf( stderr, "cannot bind daemon socket: %s (protocol=%s)\n", strerror(errno), 
               protocol==AF_INET?"IP":"UNIX" );
      ufdbLogMessage( "check for and kill old daemon processes" );
      fprintf( stderr, "check for and kill old daemon processes\n" );
#if HAVE_UNIX_SOCKETS
      ufdbLogMessage( "and remove UNIX socket file \"%s\"", addr_un.sun_path );
      fprintf( stderr, "and remove UNIX socket file \"%s\"\n", addr_un.sun_path );
#endif
      close( s );
      pthread_exit( (void *) 1 );
   }
   
   /*
    * According to comment in the Apache httpd source code, these socket
    * options should only be set after a successful bind....
    */
   sock_parm = 1;
   setsockopt( s, SOL_SOCKET, SO_KEEPALIVE, (void *) &sock_parm, sizeof(sock_parm) );

   linger.l_onoff = 1;
   linger.l_linger = 2;
   setsockopt( s, SOL_SOCKET, SO_LINGER, (void *) &linger, sizeof(linger) );

   if (protocol == AF_INET)
   {
      sock_parm = 1;
      setsockopt( s, IPPROTO_TCP, TCP_NODELAY, (void *) &sock_parm, sizeof(sock_parm) );
   }

   if (listen( s, UFDB_MAX_WORKERS ) < 0)
   {
      ufdbLogError( "cannot listen on daemon socket: %s", strerror(errno) );
      fprintf( stderr, "cannot listen on daemon socket: %s\n", strerror(errno) );
      close( s );
      pthread_exit( (void *) 1 );
   }

   /* Now is the right moment to write out main PID to the pid file
    * that is used by /etc/init.d/ufdb to send signals to ufdbguardd.
    */
   writePidFile();
   atexit( removePidFile );

   ufdbLogMessage( "ufdbguardd " VERSION " started with %d verification threads", n_workers );
   if (UFDBglobalHttpdPort > 0)
      startHttpDaemon();

   UFDBreconfig = 0;

   daemon_accept_connections( s, protocol );

   UFDBtimerStop( &timer );
   UFDBtimerPrintString( timeString, &timer, "S" );
   ufdbLogMessage( timeString );

   removePidFile();
   pthread_exit( NULL );

   return NULL;
}


/*
 * the only task of this thread is to wait for the HUP, USR1 and CHLD signals.
 */
void * waitforsignals( void * ptr )
{
   FILE *     fp;
   sigset_t   signals;
   int        sig;
   struct tms timer;
   char       timeString[96];

   ufdbSetThreadCPUaffinity( 0 );

   /* The HUP, USR1 and CHLD signals must be blocked.
    * This is a requirement to use sigwait() in a thread.
    */
   sigemptyset( &signals );
   sigaddset( &signals, SIGHUP );
   sigaddset( &signals, SIGUSR1 );
   sigaddset( &signals, SIGCHLD );
   pthread_sigmask( SIG_BLOCK, &signals, NULL );

   UFDBtimerInit( &timer );

   while (1)
   {
      sig = 0;
      if (0 != sigwait( &signals, &sig ))
         ufdbLogError( "sigwait() failed with %s", strerror(errno) );

      if (sig == SIGHUP)
      {
	 unsigned int sleep_period;
	 time_t       t;
	 struct tm    thetime;

	 if (pthread_mutex_trylock( &sighup_mutex ) != 0)
	 {
	    ufdbLogMessage( "HUP signal is already being processed" );
	    continue;
	 }
         ufdbLogMessage( "HUP signal received to reload the configuration and database" );
	 UFDBtimerStop( &timer );

	 UFDBreconfig = 1;
	 if (analyseUncategorised)
	    saveUncategorisedURLs();

	 ufdbLogMessage( "statistics: %lu URL lookups (%lu https). %lu URLs blocked. "
	 	         "%lu tunnels detected. %lu safe searches. %lu clients.", 
	                 UFDBlookupCounter, 
		         UFDBlookupHttpsCounter,
		         UFDBblockCounter + UFDBtestBlockCounter,
		         UFDBglobalTunnelCounter,
		         UFDBsafesearchCounter,
		         UFDBgetNumberOfRegisteredIPs() );
	 UFDBtimerPrintString( timeString, &timer, "CPU time by all threads:" );
	 ufdbLogMessage( timeString );

	 /* To prevent counting 'old' clients: 
	  * if today is Wednesday, we reset the IP counters.
	  */
	 t = time( NULL );
	 localtime_r( &t, &thetime );
	 if (thetime.tm_wday == 3)
	    UFDBinitializeIPcounters();

	 sleep( 3 );
	 /* after this short sleep, it is very unlikely that ufdbguardd still
	  * sends redirects, so it effectively means that ufdbhttpd can be restarted safely.
	  */
	 if (UFDBglobalHttpdPort > 0)
	 {
	    fp = fopen( UFDBHTTPD_PID_FILE, "r" );
	    if (fp != NULL)
	    {
	       httpDaemonPid = -1;
	       fscanf( fp, "%d", &httpDaemonPid );
	       fclose( fp );
	    }
	    else
	       httpDaemonPid = -1;

	    if (httpDaemonPid > 0)
	    {
	       int retval;

	       httpDaemonStatus = PS_TOBEKILLED;
	       ufdbLogMessage( "killing ufdbhttpd (pid=%d)", httpDaemonPid );
	       retval = kill( httpDaemonPid, SIGTERM );
	       if (retval != 0)
		  ufdbLogError( "cannot kill httpd daemon: %s", strerror( errno ) );
	    }
	 }

	 sleep( 1 );
	 /* be sure that the daemon is killed. */
	 if (UFDBglobalHttpdPort > 0   &&  httpDaemonPid > 0)
	    (void) kill( httpDaemonPid, SIGKILL );

	 sleep( 1 );
	 if (UFDBglobalHttpdPort > 0  &&  httpDaemonPid > 0  &&  kill( httpDaemonPid, 0 ) < 0)
	    ufdbLogMessage( "http daemon died to be restarted (OK)" );

#ifdef UFDB_HAVE_NATIVE_RWLOCK_MUTEX
	 /* 
	  * Synchronise with all worker threads.
	  * UFDBreconfig is set for the last 4 seconds so the worker threads have 
	  * most likely no read-locks any more.  Try to get a write lock within 1 second
	  * and if this fails, we will sleep 30 seconds more and hope for the best.
	  *
	  * We use a 1-second timeout since we do not want to block readers (workers)
	  * too long because the flow of messages back to Squid may not be interrupted too long.
	  */
	 DatabaseLockTimeout.tv_sec = 1;
	 DatabaseLockTimeout.tv_nsec = 0;
	 ret = pthread_rwlock_timedwrlock( &TheDatabaseLock, &DatabaseLockTimeout );
	 if (ret != 0)
	 {
	    if (ret == ETIMEDOUT)
	    {
	       ufdbLogMessage( "waitforsignals: sleeping 16 seconds before acquiring the write lock" );
	       sleep( 21-3-1-1 );
	    }
	    else 
	       ufdbLogError( "waitforsignals: pthread_rwlock_timedwrlock failed with code %d", ret );
	    /* wait indefinitely for a write lock */
	    ret = pthread_rwlock_wrlock( &TheDatabaseLock );
#ifdef UFDB_DEBUG
	    if (ret != 0)
	       ufdbLogError( "waitforsignals: pthread_rwlock_wrlock failed with code %d", ret );
#endif
	 }
	 (void) pthread_rwlock_unlock( &TheDatabaseLock );
#else
	 /* sleep() is a very crude way to allow worker threads to finish what
	  * they were doing, before we remove the database from memory,
	  * but it allows use not to use mutexes for a many-reader-one-writer problem.
	  */
	 sleep_period = n_workers / 2;
	 if (sleep_period < 12)
	    sleep_period = 12;
	 if (sleep_period > 21)
	    sleep_period = 21;
	 sleep_period -= 3+1+1;		/* we already slept a few seconds */
	 ufdbLogMessage( "waitforsignals: sleeping %d seconds to allow workers to read the database", sleep_period );
	 sleep( sleep_period );		
#endif

         ufdbLogMessage( "freeing all memory..." );
	 ufdbFreeAllMemory();
         ufdbLogMessage( "reading new config..." );
	 globalFatalError = 0;
	 UFDBglobalHttpdPort = 0;
	 httpDaemonPid = 0;
	 sgReadConfig( configFile );
	 if (globalFatalError)
	 {
	    ufdbLogError( "A FATAL ERROR OCCURRED: ALL REQUESTS ARE ANSWERED WITH \"OK\" (see previous lines)  *****" );
	 }
	 ufdbLogMessage( "the configuration is reloaded" );
	 adjustNumberOfThreads();

	 if (UFDBglobalHttpdPort > 0)
	    startHttpDaemon();

	 UFDBlookupCounter = 0;
	 UFDBlookupHttpsCounter = 0;
	 UFDBblockCounter = 0;
	 UFDBtestBlockCounter = 0;
	 UFDBuncategorisedCounter = 0;
	 UFDBglobalTunnelCounter = 0;
	 UFDBsafesearchCounter = 0;
	 UFDBuploadSeqNo++;

	 UFDBreconfig = 0;
	 pthread_mutex_unlock( &sighup_mutex );
      }
      else if (sig == SIGUSR1)
      {
         ufdbLogMessage( "USR1 signal received for logfile rotation" );
         UFDBrotateLogfile();
	 if (UFDBglobalHttpdPort > 0)
	 {
	    fp = fopen( UFDBHTTPD_PID_FILE, "r" );
	    if (fp != NULL)
	    {
	       httpDaemonPid = -1;
	       fscanf( fp, "%d", &httpDaemonPid );
	       fclose( fp );
	    }
	    else
	       httpDaemonPid = -1;

	    if (httpDaemonPid > 0)
	    {
	       int retval;

	       ufdbLogMessage( "sending USR1 signal to ufdbhttpd (pid=%d)", httpDaemonPid );
	       retval = kill( httpDaemonPid, SIGUSR1 );
	       if (retval != 0)
		  ufdbLogError( "cannot send USR1 signal to httpd daemon: %s", strerror( errno ) );
	    }
	 }
         ufdbLogMessage( "USR1 signal received for logfile rotation" );
      }
      else if (sig == SIGCHLD)
      {
         catchChildSignal( sig );
      }
      else 
         ufdbLogMessage( "signal %d received", sig );
   }
}


extern struct Category * Dest;


/*
 * A pseudo-random sample of URLs is checked against all known categories
 * and if it appears to be uncategorised, it is buffered to be sent
 * to URLfilterDB at a later date (when SIGHUP is received).
 *
 * return 0 iff the URL is uncategorised, 1 otherwise.
 */
int ufdbVerifyURLallCats( UFDBrevURL * revURL )
{
   struct Category *     cat;
   struct UFDBmemTable * mt;

   if (knownDb.table.nNextLevels > 0)
   {
      if (UFDBlookupRevUrl( &(knownDb.table.nextLevel[0]), revURL ))
         return 1;
   }

   for (cat = Dest; cat != NULL; cat = cat->next)
   {
      if (UFDBreconfig)
         return 0;
      if (cat->domainlistDb != NULL)
      {
	 mt = (struct UFDBmemTable *) cat->domainlistDb->dbcp;
	 if (UFDBlookupRevUrl( &(mt->table.nextLevel[0]), revURL ))
	    return 1;
      }
   }

   return 0;
}


static volatile int sample = 1;

/*
 * close a socket, read first all input.
 */
static void linger_close( void * fdp )
{
   int  fd;

   fd = *( (int *) fdp);
#ifdef UFDB_USE_MY_LINGER_CLOSE
   if (fd != -1)
   {
      struct timeval tv;
      char           buffer[1024];

      tv.tv_sec = 1;
      tv.tv_usec = 0;
      setsockopt( fd, SOL_SOCKET, SO_RCVTIMEO, (void *) &tv, sizeof(tv) );
   
      while (read( fd, buffer, 1024 ) == 1024)
	 ;

      ufdbLogMessage( "linger_close: fd = %d", fd );
      close( fd );
   }
#endif
}


static int UFDBcheckSafeSearchRedirection( struct SquidInfo * si )
{
   si->orig[MAX_BUF-28] = '\0';

   if (strstr( si->domain, "google." ) != NULL  &&		/* Google */
       strstr( si->surl, "q=") != NULL)
   {
      /* append &safe=active to the URL and send a redirection */
      strcat( si->orig, "&safe=active" );
      return 1;
   }
   else
   if (strstr( si->domain, "ask.com" ) != NULL  &&
       strchr( si->surl, '?' ) != NULL)    			/* ask */
   {
      strcat( si->orig, "&adt=0" );
      return 1;
   }
   else
   if (strstr( si->domain, "search.yahoo." ) != NULL  &&
       strstr( si->surl, "p=" ) != NULL)			/* Yahoo */
   {
      strcat( si->orig, "&vm=r" );
      return 1;
   }
   else
   if (strstr( si->domain, "excite.com" ) != NULL  &&
       strchr( si->surl, '?' ) != NULL)  			/* Excite */
   {
      strcat( si->orig, "&familyfilter=1&splash=filtered" );
      return 1;
   }
   else
   if (strncmp( si->domain, "search.msn.", 11 ) == 0)		/* MSN */
   {
      strcat( si->orig, "&adlt=strict" );
      return 1;
   }
   else
   if (strncmp( si->domain, "search.live.", 12 ) == 0)		/* Live */
   {
      strcat( si->orig, "&adlt=strict" );
      return 1;
   }
   else
   if (strncmp( si->domain, "search.lycos.", 13 ) == 0)		/* Lycos .com */
   {
      strcat( si->orig, "&contentFilter=strict&family=on" );
      return 1;
   }
   else
   if (strncmp( si->domain, "suche.lycos.", 12 ) == 0)		/* Lycos .de .at  .ch */
   {
      strcat( si->orig, "&family=on" );
      return 1;
   }
   else
   if (strcmp( si->domain, "buscador.lycos.es" ) == 0)		/* Lycos .es */
   {
      strcat( si->orig, "&family=on" );
      return 1;
   }
   else
   if (strcmp( si->domain, "http://vachercher.lycos.fr" ) == 0)	/* Lycos .fr */
   {
      strcat( si->orig, "&family=on" );
      return 1;
   }
   else
   if (strcmp( si->domain, "cerca.lycos.it" ) == 0)		/* Lycos .it */
   {
      strcat( si->orig, "&family=on" );
      return 1;
   }
   else
   if (strcmp( si->domain, "alltheweb.com" ) == 0)		/* alltheweb */
   {
      strcat( si->orig, "&copt_offensive=on&nooc=on" );
      return 1;
   }
   else
   if (strstr( si->domain, "dogpile.com" ) != NULL  ||
       strstr( si->domain, "dogpile.co.uk" ) != NULL)		/* dogpile */
   {
      strcat( si->orig, "&adultfilter=heavy" );
      return 1;
   }
   else
   if (strcmp( si->domain, "a9.com" ) == 0)			/* A9 */
   {
      strcat( si->orig, "&qsafe=high" );
      return 1;
   }
   else
   if (strcmp( si->domain, "hotbot.com" ) == 0)			/* hotbot */
   {
      strcat( si->orig, "&adf=on&family=on" );
      return 1;
   }
   else
   if (strstr( si->domain, "infospace.com" ) != NULL)		/* infospace */
   {
      strcat( si->orig, "&familyfilter=1" );
      return 1;
   }
   else
   if (strstr( si->domain, "metacrawler.com" ) != NULL)		/* metacrawler */
   {
      strcat( si->orig, "&familyfilter=1" );
      return 1;
   }
   else
   if (strstr( si->domain, "metaspy.com" ) != NULL)		/* metaspy */
   {
      strcat( si->orig, "&familyfilter=1" );
      return 1;
   }
   else
   if (strstr( si->domain, "webfetch.com" ) != NULL)		/* webfetch */
   {
      strcat( si->orig, "&familyfilter=1" );
      return 1;
   }
   else
   if (strstr( si->domain, "webcrawler.com" ) != NULL)		/* webcrawler */
   {
      strcat( si->orig, "&familyfilter=1" );
      return 1;
   }
   else
   if (strncmp( si->domain, "search", 6 ) == 0  &&
       strstr( si->domain, "foxnews.com" ) != NULL)		/* foxnews */
   {
      strcat( si->orig, "&familyfilter=1" );
      return 1;
   }
   /* TODO altavista NADA :-( */

   return 0;
}


/*
 * main() of the worker threads.
 */
static void * worker_main( void * ptr )
{
   sigset_t                signals;
   UFDBthreadAdmin *       admin;
   int                     tnum;
   int                     fd;
   int                     nbytes;
   unsigned long           num_reqs;
   struct Source *         src;
   struct Acl *            acl;
   char * 	 	   redirect;
   struct SquidInfo        squidInfo;
   char *                  line;
   char *                  answer;
   char *                  tmp;

   extern struct Source *  Source;

   tnum = (int) ((long) ptr);

   /* The HUP signal must be blocked.
    * This is a requirement to use sigwait() in a thread.
    */
   sigemptyset( &signals );
   sigaddset( &signals, SIGHUP );
   sigaddset( &signals, SIGPIPE );
   pthread_sigmask( SIG_BLOCK, &signals, NULL );

   ufdbSetThreadCPUaffinity( tnum );

   admin = UFDBallocThreadAdmin();

   line = ufdbCalloc( 1, MAX_BUF );
   answer = ufdbCalloc( 1, MAX_BUF );
   tmp = ufdbCalloc( 1, MAX_BUF );

   fd = -1;
   pthread_cleanup_push( linger_close, (void *) &fd );

   while (1)
   {
      getFdQueue( &fd );
      num_reqs = 0UL;
      if (UFDBglobalDebug)
	 ufdbLogMessage( "W%02d: open fd %2d", tnum, fd );

      while ((nbytes = read( fd, line, MAX_BUF-2 )) > 0)
      {
         line[nbytes] = '\0';
         num_reqs++;
	 UFDBlookupCounter++;	/* no mutex: faster and we do not care about a few increments less */

         if (UFDBglobalDebug > 1)
	    ufdbLogMessage( "W%02d: line=%s", tnum, line );

	 /* The strstr() is to prevent loops with redirect 302:... ads and other URLs
	  * that are matched by regular expressions 
	  */
	 if (globalFatalError || UFDBreconfig  || strstr( line, "/URLblocked.cgi?" ) != NULL)
	 {
	    if (write( fd, "\n", 1 ) < 0)
	    {
	       ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
	       goto write_error;
	    }
#ifdef _POSIX_PRIORITY_SCHEDULING
	    if (UFDBreconfig)
	       sched_yield();	/* be nice to the reconfiguration thread */
#endif
	    continue;
	 }

	 squidInfo.revUrl = NULL;
	 if (parseLine(admin, line, &squidInfo) != 1) 	/* put input line in struct squidInfo */
	 {
	    ufdbLogError( "W%02d: error parsing input line: %s", tnum, line );
	    if (write( fd, "\n", 1 ) < 0)
	    {
	       ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
	       UFDBfreeRevURL( admin, squidInfo.revUrl );
	       goto write_error;
	    }
	 }
	 else
	 {
            if (strcmp( squidInfo.protocol, "https" ) == 0)
	       UFDBlookupHttpsCounter++;

	    UFDBregisterCountedIP( squidInfo.src );

#ifdef UFDB_HAVE_NATIVE_RWLOCK_MUTEX
            ret = pthread_rwlock_rdlock( &TheDatabaseLock );
#ifdef UFDB_DEBUG
            if (ret != 0)
	       ufdbLogError( "worker_main: pthread_rwlock_rdlock failed with code %d", ret );
#endif
#endif

	    src = Source;
	    for (;;)
	    {
	       /* be extra careful with NOT using the in-memory database on a reload. */
	       if (UFDBreconfig) 
	       {
		  if (write( fd, "\n", 1 ) < 0)
		  {
		     ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
		     UFDBfreeRevURL( admin, squidInfo.revUrl );
		     goto write_error;
		  }
		  break;
	       }

	       strcpy( tmp, squidInfo.src );	/* TODO: find out if strcpy() is necessary... */
	       src = sgFindSource( src, tmp, &squidInfo );

	       if (UFDBreconfig)
	       {
		  if (write( fd, "\n", 1 ) < 0)
		  {
		     ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
		     UFDBfreeRevURL( admin, squidInfo.revUrl );
		     goto write_error;
		  }
		  break;
	       }
	       else
		  acl = sgAclCheckSource( src );

	       if (parseOnly || UFDBreconfig)
	       {
		  if (write( fd, "\n", 1 ) < 0)
		  {
		     ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
		     UFDBfreeRevURL( admin, squidInfo.revUrl );
		     goto write_error;
		  }
		  break;
	       }

	       redirect = sgAclAccess( src, acl, &squidInfo, tmp );
	       if (redirect == NULL) 
	       {
		  /* src not found */
		  if (src == NULL || src->cont_search == 0) 
		  {
		     /* safe search */
		     if (UFDBglobalSafeSearch  &&
		         UFDBcheckSafeSearchRedirection( &squidInfo ))
		     {
			int len;

			UFDBsafesearchCounter++;
			if (UFDBglobalLogAllRequests || globalDebugRedirect)
			{
			   if (squidInfo.ident[0] == '\0') 
			   {
			      squidInfo.ident[0] = '-';
			      squidInfo.ident[1] = '\0';
			   }
			   ufdbLogMessage( "PASS %s %s %s %s %s %s", 
					   squidInfo.ident, squidInfo.src, acl->name, 
					   "SAFESEARCH", squidInfo.orig, squidInfo.urlgroup );
			}

			/* send a REDIRECT to Squid */
			len = strlen( squidInfo.orig );
			squidInfo.orig[len] = '\n';
			len++;
			squidInfo.orig[len] = '\0';
			if (write( fd, squidInfo.orig, len ) < 0)
			{
			   ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
			   UFDBfreeRevURL( admin, squidInfo.revUrl );
			   goto write_error;
			}
		     }
		     else
		     {
			if (write( fd, "\n", 1 ) < 0)
			{
			   ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
			   UFDBfreeRevURL( admin, squidInfo.revUrl );
			   goto write_error;
			}

			if (UFDBglobalLogAllRequests && !UFDBreconfig)
			{
			   if (squidInfo.ident[0] == '\0') 
			   {
			      squidInfo.ident[0] = '-';
			      squidInfo.ident[1] = '\0';
			   }
			   ufdbLogMessage( "PASS %s %s %s %s %s %s",
					   squidInfo.ident, squidInfo.src, acl->name, "-", squidInfo.orig, squidInfo.urlgroup );
			}
		     }

		     if (analyseUncategorised && !UFDBreconfig)
		     {
	 		sample++;
			if (sample % 11 == 0)	/* approx. 9% is sampled */
			{
			   if (isdigit( squidInfo.domain[0] )  &&
			       ( (strncmp( squidInfo.domain, "10.", 3 )==0) ||
				 (strncmp( squidInfo.domain, "127.", 4 )==0) ||
			         (strncmp( squidInfo.domain, "192.168.", 8 )==0) ))
			   {
			      /* do NOT check 10.*, 127.* and 192.168.*  with uncatgeorised domains */
			   }
			   else
			   {
			      if (!ufdbVerifyURLallCats( squidInfo.revUrl ))
			      {
				 ufdbRegisterUnknownURL( squidInfo.domain );
				 UFDBuncategorisedCounter++;
			      }
			   }
			}
		     }

		     break;
		  }
		  else
		  {
		     if (!UFDBreconfig  &&  src->next != NULL) 
		     {
		        src = src->next;
		        continue;
		     }
		     else
		     {
		        if (write( fd, "\n", 1 ) < 0)
			{
			   ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
			   UFDBfreeRevURL( admin, squidInfo.revUrl );
			   goto write_error;
			}
		        break;
		     }
		  }
	       }
	       else
	       {
		  char * category;

		  /* src found, so block the URL and send the redirect string */
		  if (squidInfo.srcDomain[0] == '\0') 
		  {
		     squidInfo.srcDomain[0] = '-';
		     squidInfo.srcDomain[1] = '\0';
		  }

		  if (squidInfo.ident[0] == '\0') 
		  {
		     squidInfo.ident[0] = '-';
		     squidInfo.ident[1] = '\0';
		  }

		  if (globalDebugRedirect)
		  {
		      ufdbLogMessage( "W%02d:   REDIRECT %s %s/%s %s %s %s", tnum,
				      redirect, squidInfo.src, squidInfo.srcDomain, 
				      squidInfo.ident, "GET", squidInfo.urlgroup );
		  }

		  if (squidInfo.aclpass == NULL)
		     category = "error";
		  else if (squidInfo.aclpass->name == NULL)
		     category = "error";
		  else 
		     category = squidInfo.aclpass->name;

		  if (noRedirects)
		  {
		     ufdbLogMessage( "TESTBLOCK %s %s %s %s %s %s",
		                     squidInfo.ident, squidInfo.src, acl->name, 
				     category, squidInfo.orig, squidInfo.urlgroup );
		     if (write( fd, "\n", 1 ) < 0)	/* send back an "OK" message */
		     {
			ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
	                UFDBfreeRevURL( admin, squidInfo.revUrl );
			goto write_error;
		     }
		     UFDBtestBlockCounter++;
		  }
		  else
		  {
		     char urlgrp[64];

		     if (squidInfo.urlgroup[0] == '#')
			urlgrp[0] = '\0';
		     else
		     {
			urlgrp[0] = ' ';
			strncpy( &urlgrp[1], squidInfo.urlgroup, 62 );
			urlgrp[63] = '\0';
		     }

		     if (UFDBglobalLogBlock || UFDBglobalLogAllRequests)
			ufdbLogMessage( "BLOCK %s %s %s %s %s %s",
			                squidInfo.ident, squidInfo.src, acl->name, 
					category, squidInfo.orig, squidInfo.urlgroup );
		     sprintf( answer, "%s %s/%s %s %s%s\n", redirect, squidInfo.src,
			      squidInfo.srcDomain, squidInfo.ident, "GET", urlgrp );
		     if (write( fd, answer, strlen(answer) ) < 0)
		     {
			ufdbLogError( "W%02d: write failed: fd=%d %s", tnum, fd, strerror(errno) );
	                UFDBfreeRevURL( admin, squidInfo.revUrl );
			goto write_error;
		     }
		     UFDBblockCounter++;
		  }
		  break;
	       }
	    }
#ifdef UFDB_HAVE_NATIVE_RWLOCK_MUTEX
            (void) pthread_rwlock_unlock( &TheDatabaseLock );
#endif
	    UFDBfreeRevURL( admin, squidInfo.revUrl );
	 }
	 if (sig_other)
	    break;
         pthread_testcancel();
      }
      if (nbytes < 0)
      {
         ufdbLogError( "W%02d: read failed: fd=%d %s", tnum, fd, strerror(errno) );
      }
write_error:

#ifdef UFDB_HAVE_NATIVE_RWLOCK_MUTEX
      (void) pthread_rwlock_unlock( &TheDatabaseLock );
#endif

      if (num_reqs > 1UL)
	 ufdbLogMessage( "W%02d: %lu URL verifications", tnum, num_reqs );

      if (UFDBglobalDebug)
	 ufdbLogMessage( "W%02d: close fd %2d", tnum, fd );
      close( fd );
      fd = -1;
   }

   /*NOTREACHED*/
   pthread_cleanup_pop( 0 );

   pthread_exit( NULL );
}


void ufdbCatchBadSignal( int signum )
{
   int       i;
   int       num;
   int       i_am_thread;
   pthread_t self;
   char      me[32];
   
   UFDBreconfig = 1;		/* be extra careful and set UFDBreconfig */

   self = pthread_self();

   /* find out which thread has a signal */
   i_am_thread = 0;
   sprintf( me, "pid %d", (int) getpid() );
   if (self == HUPhandler)
   {
      strcpy( me, "thread HUP-handler" );
      i_am_thread = 1;
   }
   else if (self == sockethandler)
   {
      strcpy( me, "thread socket-handler" );
      i_am_thread = 1;
   }
   else 
   {
      num = UFDB_NUM_HTTPS_VERIFIERS;
      for (i = 0; i < num; i++)
         if (self == httpsthr[i])
	 {
	    sprintf( me, "thread https-verifier-%02d", i );
	    i_am_thread = 1;
	    break;
	 }
      num = n_workers;
      for (i = 0; i < num; i++)
         if (self == workers[i])
	 {
	    sprintf( me, "thread worker-%02d", i );
	    i_am_thread = 1;
	    break;
	 }
   }
   ufdbLogMessage( "\"%s\" caught signal %d.", me, signum );

#if defined(__linux__) && 0
   if (signum != SIGINT && signum != SIGTERM)
   {
      void *    functions[32];
      char **   function_names;

      num = backtrace( functions, 32 );
      function_names = backtrace_symbols( functions, num );
      for (i = 0; i < num; i++)
	 ufdbLogMessage( "   function %s", function_names[i] );
      free( function_names );
   }
#endif

   if (signum == SIGINT || signum == SIGTERM)
   {
#if defined(UFDB_FREE_MEMORY)
      /* usleep( 200000 ); */
      sleep( 1 );
      ufdbLogMessage( "freeing memory ..." );
      ufdbFreeAllMemory();
      ufdbLogMessage( "done freeing memory." );
#endif
      /* implement our own linger_close() */
      removePidFile();
      pthread_cancel( sockethandler );
      if (analyseUncategorised)
	 saveUncategorisedURLs();
      sleep( 1 );
      exit( 0 );
   }
   else
   {
      removePidFile();

      kill( globalPid, SIGABRT );
      sleep( 3 );
      exit( 2 );
   }
}

extern char ** environ;


int main( 
   int            argc,
   char **        argv )
{
   int            i;
   pid_t          pid;
   pthread_attr_t attr;
   time_t 	  t;
   char           niso_str[22];

   char ** envp = environ;

   strcpy( progname, "ufdbguardd" );
   UFDBinitializeIPcounters();

   /* TODO: unset LC variables; regex cannot deal with multibyte characters */

   while ((i = getopt(argc, argv, "ATdDPh?rt:c:L:p:U:vw:")) != EOF)
   {
      switch (i) {
      case 'A':
	 fprintf( stderr, "-A option is obsolete\n");
	 break;
      case 'N':
         analyseUncategorised = 0;
	 break;
      case 'D':
	 runAsDaemon = 0;	/* undocumented -D option to prevent running as daemon */
	 break;
      case 'd':
	 UFDBglobalDebug++;
	 break;
      case 'c':
	 configFile = optarg;
	 break;
      case 'p':
      	 portNumCmdLine = atoi( optarg );
	 if (portNumCmdLine <= 0) {
	    fprintf( stderr, "port number must be > 0\n" );
	    exit( 1 );
	 }
	 break;
      case 'P':			/* undocumented -P option for development purposes */
	 parseOnly = 1;
	 break;
      case 'r':
	 globalDebugRedirect = 1;
	 break;
      case 't':			/* undocumented -t option for time-related testing */
	 if ((t = iso2sec(optarg)) == -1) {
	    fprintf( stderr, "-t dateformat error, should be yyyy-mm-ddTHH:MM:SS\n" );
	    exit( 1 );
	 }
	 if (t < 0) {
	    fprintf( stderr, "-t date have to after 1970-01-01T01:00:00\n" );
	    exit( 1 );
	 }
	 niso( t, niso_str );
	 ufdbLogMessage( "ufdbguardd emulating date %s", niso_str );
	 globalDebugTimeDelta = t - start_time.tv_sec;
	 start_time.tv_sec = start_time.tv_sec + globalDebugTimeDelta;
	 break;
      case 'T':
	 noRedirects = 1;		/* Test mode means noRedirects */
	 break;
      case 'U':
	 if (strlen(optarg) <= 31)
	    strcpy( UFDBglobalUserName, optarg );
	 else
	    ufdbLogFatalError( "username supplied with -U option is too long" );
         break;
      case 'L':				/* undocumented -L option for alternate PID file */
	 globalPidFile = optarg;
         break;
      case 'v':
	 fprintf( stderr, "ufdbguardd: %s\n", VERSION );
	 exit( 0 );
	 break;
      case 'w':
         n_workers = atoi( optarg );
	 if (n_workers < 8)
	    n_workers = 8;
	 else if (n_workers > UFDB_MAX_WORKERS)
	    n_workers = UFDB_MAX_WORKERS;
	 break;
      case '?':
      case 'h':
         usage( '\0' );
	 break;
      default:
         usage( i );
      }
   }

   UFDBdropPrivileges( UFDBglobalUserName );

   if (runAsDaemon)
   {
      if ((pid = fork()) != 0)
	 exit( 0 );        
#ifndef UFDB_DEBUG
      close( 0 );
      close( 1 );
#endif
      setsid();
   }
   globalPid = getpid();

   /*
    * Initialise signal handlers.
    * The HUP signal is dealt with on a per thread basis.
    */
#ifndef UFDB_CORE_DUMP_SEGV
   ufdbSetSignalHandler( SIGILL,  ufdbCatchBadSignal );
   ufdbSetSignalHandler( SIGBUS,  ufdbCatchBadSignal );
   ufdbSetSignalHandler( SIGSEGV, ufdbCatchBadSignal );
#endif

   ufdbSetSignalHandler( SIGINT,  ufdbCatchBadSignal );
   ufdbSetSignalHandler( SIGTERM, ufdbCatchBadSignal );

   ufdbSetSignalHandler( SIGPIPE, SIG_IGN );
   ufdbSetSignalHandler( SIGHUP,  catchHUPSignal );
   ufdbSetSignalHandler( SIGUSR1, catchUSR1Signal );

   ufdbSetSignalHandler( SIGCHLD, catchChildSignal );

   my_argc = argc;
   my_argv = argv;
   my_envp = envp;

   /*
    * create the threads.
    */
   pthread_attr_init( &attr );
   pthread_attr_setscope( &attr, PTHREAD_SCOPE_SYSTEM );
   pthread_attr_setstacksize( &attr, 256 * 1024 );

   pthread_create( &HUPhandler, &attr, waitforsignals, (void *) 0 );
   usleep( 10000 );

   pthread_create( &sockethandler, &attr, accept_main, (void *) 0 );
   usleep( 100000 );

   if (UFDBglobalDebug)
      ufdbLogMessage( "using %d worker threads", n_workers );
   for (i = 0; i < n_workers; i++)
   {
      pthread_create( &workers[i], &attr, worker_main, (void *) ((long) i) );
      usleep( 10000 );
   }

   /*
    * exit if sockethandler thread dies.
    */
   pthread_join( sockethandler, 0 );

   /*
    * cleanup fast. cancel all other threads and wait a short while.
    */
   UFDBreconfig = 1;
   pthread_cancel( HUPhandler );
   for (i = 0; i < n_workers; i++)
      pthread_cancel( workers[i] );
   for (i = 0; i < UFDB_NUM_HTTPS_VERIFIERS; i++)
      pthread_cancel( httpsthr[i] );
   /* usleep( 100000 ); */
   sleep( 1 );
   exit( 0 );

   return 0;
}



static void saveUncategorisedURLs( void )
{
   char * URLs;
   char * message;
   int    length;
   int    s;
   int    nbytes;
   int    written;
   long   num_cpus;
   struct utsname sysinfo;

   if (UFDBglobalDebug)
      ufdbLogMessage( "saveUncategorisedURLs" );

   URLs = ufdbGetUnknownURLs();
   length = strlen( URLs );
   if (length == 0)
   {
      ufdbResetUnknownURLs();
      return;
   }

   message = ufdbMalloc( 1024 + length );
   if (message == NULL)
      return;

   s = UFDBopenSocket( UFDB_UPLOAD_UNCATEGORISED_URLS_WEBSITE, 80 );
   if (s < 0)
   {
      ufdbResetUnknownURLs();
      ufdbLogError( "cannot upload uncategorised URLs  *****" );
      ufdbLogError( "cannot open a communication socket to %s:%d (%s)", 
                    UFDB_UPLOAD_UNCATEGORISED_URLS_WEBSITE, 80, strerror(errno) );
      return;
   }

   if (uname( &sysinfo ) != 0)
   {
      strcpy( sysinfo.machine, "M?" );
      strcpy( sysinfo.release, "R?" );
      strcpy( sysinfo.nodename, "unknown" );
      strcpy( sysinfo.sysname, "sysname" );
   }
   else
   {
      sysinfo.machine[ sizeof(sysinfo.machine)-1 ] = '\0';
      sysinfo.release[ sizeof(sysinfo.release)-1 ] = '\0';
      sysinfo.nodename[ sizeof(sysinfo.nodename)-1 ] = '\0';
      sysinfo.sysname[ sizeof(sysinfo.sysname)-1 ] = '\0';
   }

#if defined(_SC_NPROCESSORS_ONLN)
   num_cpus = sysconf( _SC_NPROCESSORS_ONLN );

#elif defined(__NR_sched_getaffinity)
   /* sched_setaffinity() is buggy on linux 2.4.x so we use syscall() instead */
   cpu = syscall( __NR_sched_getaffinity, getpid(), 4, &cpu_mask );
   /* printf( "sched_getaffinity returned %d %08lx\n", cpu, cpu_mask ); */
   if (cpu >= 0)
   {
      num_cpus = 0;
      for (cpu = 0; cpu < 32; cpu++)
         if (cpu_mask & (1 << cpu))
            num_cpus++;
      /* printf( "   found %d CPUs in the cpu mask\n", num_cpus ); */
   }
   else
#else
      num_cpus = 0;
#endif

   sprintf( message, 
	    "POST /cgi-bin/uncat.pl HTTP/1.1\r\n"
	    "Host: " UFDB_UPLOAD_UNCATEGORISED_URLS_WEBSITE "\r\n"
	    "User-Agent: ufdbGuardd-" VERSION "\r\n"
	    "Content-Type: text/plain\r\n"
	    "Content-Length: %d\r\n"
	    "Connection: close\r\n"
	    "X-SiteInfo: %s %ld %s %s\r\n"
	    "X-NodeName: %s\r\n"
	    "X-NumClients: %lu\r\n"
	    "X-NumLookups: %lu\r\n"
	    "X-NumHttpsLookups: %lu\r\n"
	    "X-NumTunnelsDetected: %lu\r\n"
	    "X-NumTestBlock: %lu\r\n"
	    "X-NumBlock: %lu\r\n"
	    "X-NumSafeSearch: %lu\r\n"
	    "X-Uncategorised: %lu\r\n"
	    "X-UploadSeqNo: %lu\r\n"
	    "\r\n"
	    "%s\r\n",
            length, 
	    sysinfo.machine, num_cpus, sysinfo.sysname, sysinfo.release, 
	    sysinfo.nodename,
	    UFDBgetNumberOfRegisteredIPs(),
	    UFDBlookupCounter,
	    UFDBlookupHttpsCounter,
	    UFDBglobalTunnelCounter,
	    UFDBtestBlockCounter,
	    UFDBblockCounter,
	    UFDBsafesearchCounter,
	    UFDBuncategorisedCounter,
	    UFDBuploadSeqNo,
	    URLs );
   length = strlen( message );
   written = 0;
   while (length > 0)
   {
      nbytes = write( s, message+written, length );
      if (nbytes == 0)
         break;
      if (nbytes < 0)
      {
         if (errno != EINTR)
	    break;
      }
      else
      {
         written += nbytes;
	 length -= nbytes;
      }
   }

   close( s );

   ufdbFree( message );

   ufdbResetUnknownURLs();
   ufdbLogMessage( "uncategorised domains have been uploaded to URLfilterDB" );
}

