%{

#ifdef UFDB_DEBUG
#define YYDEBUG 1
#endif

/*
  By accepting this notice, you agree to be bound by the following
  agreements:
  
  ufdbGuard is copyrighted (C) 2005,2006,2007 by URLfilterDB with all rights reserved.
  ufdbGuard is based on squidGuard.

  squidGuard is copyrighted (C) 1998 by
  ElTele st AS, Oslo, Norway, with all rights reserved.
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License (version 2) as
  published by the Free Software Foundation.  It is distributed in the
  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License (GPL) for more details.
  
  You should have received a copy of the GNU General Public License
  (GPL) along with this program.
*/


/* #define YYDEBUG 1 */

#include "sg.h"
#include "ufdb.h"
#include "ufdblib.h"
#include "ufdbchkport.h"

#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <grp.h>
#include <syslog.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/stat.h>


FILE * yyin;
FILE * yyout;
char * configFile;

extern struct ufdbSetting * lastSetting ;
extern struct ufdbSetting * Setting;

extern struct Source * lastSource ;
extern struct Source * Source ;

extern struct Category * lastDest ;
extern struct Category * Dest ;

extern struct sgRewrite * lastRewrite;
extern struct sgRewrite * Rewrite;
extern struct ufdbRegExp *  lastRewriteRegExec;

extern struct Time * lastTime;
extern struct Time * Time;

extern struct LogFile * globalLogFile;

extern struct LogFileStat * lastLogFileStat;
extern struct LogFileStat * LogFileStat;

extern struct TimeElement * lastTimeElement;
extern struct TimeElement * TimeElement;

int   numTimeElements;
int * TimeElementsEvents = NULL;

extern struct Acl * lastAcl;
extern struct Acl * defaultAcl;
extern struct Acl * Acl;
extern struct AclDest * lastAclDest;
 
extern struct UFDBmemTable knownDb;

extern struct ufdbRegExp * lastRegExpDest;

extern int    analyseUncategorised;
extern int    httpsConnectionCacheSize;

extern int globalDebugTimeDelta;

static int time_switch = 0;
static int date_switch = 0;

int numSource = 0;
int portNum = UFDB_DAEMON_PORT;

static void ufdbSourceGroup( int groupType, char * groupName );
static void ufdbSourceUserQuota( char * seconds, char * sporadic, char * renew );

%}

%union {
  char * string;
  char * tval;
  char * dval;
  char * dvalcron;
  int    integer;
}

%token ADMINISTRATOR QSTRING
%token CHECK_PROXY_TUNNELS QUEUE_CHECKS AGRESSIVE LOG_ONLY 
%token ON OFF
%token COMMA EQUAL PORT
%token HTTP_SERVER INTERFACE IMAGES 
%token HTTPS_CONNECTION_CACHE_SIZE
%token WORD END START_BRACKET STOP_BRACKET WEEKDAY
%token CATEGORY REWRITE ACL CPUS TIME TVAL DVAL DVALCRON
%token SOURCE CIDR IPCLASS CONTINUE
%token IPADDR DBHOME DOMAINLIST URLLIST EXPRESSIONLIST IPLIST
%token DOMAIN UNIX LDAP USER USERLIST USERQUOTA GROUP 
%token IP NL NUMBER NUMBERS
%token PASS REDIRECT SUBST CHAR MINUTELY HOURLY DAILY WEEKLY DATE
%token WITHIN OUTSIDE ELSE ANONYMOUS CONTINIOUS SPORADIC
%token LOGFILE LOGDIR LOGBLOCK LOGALL
%token OPTION ENFORCE_HTTPS_WITH_HOSTNAME ENFORCE_HTTPS_OFFICAL_CERTIFICATE
%token ANALYSE_UNCATEGORISED
%token SAFE_SEARCH MAX_LOGFILE_SIZE


%type <string> WORD 
%type <string> QSTRING
%type <string> WEEKDAY
%type <string> NUMBER
%type <string> NUMBERS
%type <tval>   TVAL
%type <string> DVAL
%type <string> DVALCRON
%type <string> CHAR
%type <string> SUBST 
%type <string> IPADDR
%type <string> DBHOME LOGDIR
%type <string> CIDR
%type <string> IPCLASS
%type <string> acl_content
%type <string> acl_header
%type <string> dval
%type <string> dvalcron
%type <string> tval
%type <string> date
%type <string> ttime

%type <integer> check_proxy_tunnel_option
%type <integer> on_or_off

%%

start: statements
       ; 

https_cache_size:
           HTTPS_CONNECTION_CACHE_SIZE NUMBER   { httpsConnectionCacheSize = atoi($2); }
	 ;

on_or_off:	
	   ON    { $$ = 1; }
	 | OFF   { $$ = 0; }
	 ;

log_block:
	   LOGBLOCK on_or_off   { UFDBglobalLogBlock = $2; }
	 ;

log_all:
	   LOGALL on_or_off     { UFDBglobalLogAllRequests = $2; }
	 ;

enforce_https_with_hostname:
	   ENFORCE_HTTPS_WITH_HOSTNAME on_or_off         { UFDBglobalHttpsWithHostname = $2; }
	 ;

enforce_https_offical_certificate:
           ENFORCE_HTTPS_OFFICAL_CERTIFICATE on_or_off   { UFDBglobalHttpsOfficialCertificate = $2; }
	 ;


check_proxy_tunnel_option:
	   OFF           { $$ = UFDB_API_HTTPS_CHECK_OFF; }
	 | QUEUE_CHECKS  { $$ = UFDB_API_HTTPS_CHECK_QUEUE_CHECKS; }
	 | AGRESSIVE     { $$ = UFDB_API_HTTPS_CHECK_AGRESSIVE; }
	 | LOG_ONLY      { $$ = UFDB_API_HTTPS_LOG_ONLY; }
	 ;

check_proxy_tunnels:
           	  CHECK_PROXY_TUNNELS check_proxy_tunnel_option { UFDBsetTunnelCheckMethod( $2 ); }
	 	;

admin_spec:    
		  ADMINISTRATOR QSTRING  { ufdbSetting("administrator",$2); }
	   	;

dbhome:    
		  DBHOME WORD { ufdbSetting("dbhome",$2); }
           	;

logdir:    
		  LOGDIR WORD { ufdbSetting("logdir",$2); }
            	;

port:	   
		  PORT NUMBER { ufdbSetting("port",$2); }
                ;

cpus:	   
		  CPUS NUMBER  { ufdbSetCPU( $2 ); }
         	| CPUS NUMBERS { ufdbSetCPU( $2 ); }
           	;

analyse_uncategorised: 
               	  ANALYSE_UNCATEGORISED on_or_off
		     { analyseUncategorised = $2;   }
	        |
               	  ANALYSE_UNCATEGORISED NUMBER
		     { analyseUncategorised = atoi( $2 );   }
	       	;

safe_search:
		  SAFE_SEARCH on_or_off
		     { UFDBglobalSafeSearch = $2; }
	        ;

max_logfile_size:
		  MAX_LOGFILE_SIZE NUMBER
		     { UFDBglobalMaxLogfileSize = strtoul( $2, NULL, 10 );
		       if (UFDBglobalMaxLogfileSize < 2 * 1024 * 1024)		/* minimum is 2 MB */
		          UFDBglobalMaxLogfileSize = 2 * 1024 * 1024;
		       if (UFDBglobalMaxLogfileSize > 2000000000)		/* maximum is 2 GB */
		          UFDBglobalMaxLogfileSize = 2000000000;
		     }
	        ;

httpd_option:
		  PORT EQUAL NUMBER        { UFDBglobalHttpdPort = atoi( $3 );             }
	        | INTERFACE EQUAL WORD     { if (strcmp($3,"all")== 0)
						strcpy( UFDBglobalHttpdInterface, "all" );    
					     else
					        ufdbLogFatalError( "http-server interface must be \"all\" or IP address" );
					   }
	        | INTERFACE EQUAL IPADDR   { strcpy( UFDBglobalHttpdInterface, $3 );       }
		| IMAGES EQUAL WORD        { strcpy( UFDBglobalHttpdImagesDirectory, $3 ); }
		| IMAGES EQUAL QSTRING     { strcpy( UFDBglobalHttpdImagesDirectory, $3 ); }
		;

httpd_options:
		  httpd_option COMMA httpd_options
		| httpd_option
	        ;

http_server_def:
		  HTTP_SERVER START_BRACKET httpd_options STOP_BRACKET
	        ;

category: 
		  CATEGORY WORD { ufdbCategory($2); }
             	;

category_block: 
		  category START_BRACKET category_contents STOP_BRACKET 
                       { ufdbCategoryEnd(); }
                ;

category_contents:
                | category_contents category_content
		;

category_content:  
 	      	  DOMAINLIST WORD         { ufdbCategoryDomainList( $2 ); }
            	| DOMAINLIST '-'          { ufdbCategoryDomainList( NULL ); }
            	| URLLIST WORD            { ufdbCategoryUrlList( $2 ); }
            	| URLLIST '-'             { ufdbCategoryUrlList( NULL ); }
            	| EXPRESSIONLIST '-'      { ufdbCategoryExpressionList( NULL, NULL ); }
            	| EXPRESSIONLIST 'i' WORD { ufdbCategoryExpressionList( $3, "i" ); }
            	| EXPRESSIONLIST WORD     { ufdbCategoryExpressionList( $2, "n" ); }
            	| REDIRECT WORD           { ufdbCategoryRedirect( $2 ); }
            	| REDIRECT QSTRING        { ufdbCategoryRedirect( $2 ); }
            	| REWRITE WORD            { ufdbCategoryRewrite( $2 ); }
            	| WITHIN WORD             { ufdbCategoryTime( $2, WITHIN ); }
            	| OUTSIDE WORD            { ufdbCategoryTime( $2, OUTSIDE ); }
		| OPTION ENFORCE_HTTPS_WITH_HOSTNAME         { ufdbCategoryOption( 1, UFDB_OPT_HTTPS_WITH_HOSTNAME );  }
		| OPTION ENFORCE_HTTPS_OFFICAL_CERTIFICATE   { ufdbCategoryOption( 1, UFDB_OPT_HTTPS_OFFICAL_CERTIFICATE );  }
            	| LOGFILE ANONYMOUS WORD  { ufdbLogError( "unsupported logfile context for %s", $3 ); }
            	| LOGFILE WORD            { ufdbLogError( "unsupported logfile context for %s", $2 ); }
            	;

source:      
		  SOURCE WORD { sgSource($2); }
             	;

source_block: 
		  source START_BRACKET source_contents STOP_BRACKET { sgSourceEnd(); }
             	;

source_contents:
		| source_contents source_content
		;

source_content:       
		  DOMAIN domain
                | USER user 	
                | UNIX USER user 		     
                | USERLIST WORD 		 { ufdbSourceUserList( $2 ); } 
                | USERLIST QSTRING 		 { ufdbSourceUserList( $2 ); } 
                | UNIX USERLIST WORD 	         { ufdbSourceUserList( $3 ); } 
                | UNIX USERLIST QSTRING 	 { ufdbSourceUserList( $3 ); } 
		| UNIX GROUP WORD                { ufdbSourceGroup( UFDB_GROUPTYPE_UNIX, $3 ); }
		| UNIX GROUP QSTRING             { ufdbSourceGroup( UFDB_GROUPTYPE_UNIX, $3 ); }
                | USERQUOTA NUMBER NUMBER HOURLY { ufdbSourceUserQuota( $2, $3, "3600" ); }  
                | USERQUOTA NUMBER NUMBER DAILY  { ufdbSourceUserQuota( $2, $3, "86400" ); }  
                | USERQUOTA NUMBER NUMBER WEEKLY { ufdbSourceUserQuota( $2, $3, "604800" ); } 
                | USERQUOTA NUMBER NUMBER NUMBER { ufdbSourceUserQuota( $2, $3, $4 ); } 
                | IP ips
                | IPLIST WORD             { sgSourceIpList($2); }
                | WITHIN WORD             { sgSourceTime($2,WITHIN); }
                | OUTSIDE WORD            { sgSourceTime($2,OUTSIDE); }
                | LOGFILE ANONYMOUS WORD  { ufdbLogError( "unsupported logfile context for %s", $3 ); }
                | LOGFILE WORD            { ufdbLogError( "unsupported logfile context for %s", $2 ); }
                | CONTINUE                { lastSource->cont_search = 1; }
                ;


domain:		    
		| domain WORD  { sgSourceDomain( $2 ); }
                | domain COMMA
		;

user:		    
		| user WORD    { ufdbSourceUser( $2 ); }
		| user QSTRING { ufdbSourceUser( $2 ); }
                | user COMMA
		;

acl_block: 
		  ACL START_BRACKET acl_contents STOP_BRACKET 
                ;

acl_contents:     /* empty */
                | acl_contents acl_content
	        ;

acl_header:       WORD              { sgAcl( $1, NULL, 0 );     }
                | QSTRING           { sgAcl( $1, NULL, 0 );     }
		| WORD WITHIN WORD  { sgAcl( $1, $3, WITHIN );  }
                | WORD OUTSIDE WORD { sgAcl( $1, $3, OUTSIDE ); }
                ;

acl_content:      acl_header START_BRACKET access_contents STOP_BRACKET
                | acl_header START_BRACKET access_contents STOP_BRACKET ELSE
                     { sgAcl( NULL, NULL, ELSE );               } 
                     START_BRACKET access_contents STOP_BRACKET
                ;

access_contents:  /* empty */
                | access_contents access_content
                ;

access_content:     
		  PASS access_pass       { }
                | REWRITE WORD           { sgAclSetValue("rewrite",$2,0); }
                | REDIRECT WORD          { sgAclSetValue("redirect",$2,0); }
                | LOGFILE ANONYMOUS WORD { ufdbLogError( "unsupported logfile context for %s", $3 ); }
                | LOGFILE WORD           { ufdbLogError( "unsupported logfile context for %s", $2 ); }
                ;

access_pass:      /* empty */
                | access_pass WORD     { sgAclSetValue("pass",$2,1); }
                | access_pass '!' WORD { sgAclSetValue("pass",$3,0); }
		| access_pass COMMA
                ;

cidr:             CIDR { sgIp($1); }
                ;

ipclass:          IPCLASS { sgIp($1); }
                ;

ips: 		  /* empty */
                | ips ip         { sgIp("255.255.255.255"); sgSetIpType(SG_IPTYPE_HOST,NULL,0); }
                | ips ip cidr    { sgSetIpType(SG_IPTYPE_CIDR,NULL,0); }
                | ips ip ipclass { sgSetIpType(SG_IPTYPE_CLASS,NULL,0); }
                | ips ip '-' ip  { sgSetIpType(SG_IPTYPE_RANGE,NULL,0); }
                | ips COMMA
		;

ip:  		  IPADDR { sgIp($1);}
     		;

rew:       	  REWRITE WORD { sgRewrite($2); } 
		;

rew_block:  	  rew START_BRACKET rew_contents STOP_BRACKET 
             	;

rew_contents:     /* empty */
		| rew_contents rew_content
		;


rew_content:      SUBST                  { sgRewriteSubstitute($1); }
                | WITHIN WORD            { sgRewriteTime($2,WITHIN); }
                | OUTSIDE WORD           { sgRewriteTime($2,OUTSIDE); }
                | LOGFILE ANONYMOUS WORD { ufdbLogError( "unsupported logfile context for %s", $3 ); }
                | LOGFILE WORD           { ufdbLogError( "unsupported logfile context for %s", $2 ); }
                ;


time:       	  TIME WORD { sgTime($2); } 
		;

time_block:  	  time START_BRACKET time_contents STOP_BRACKET 
                ;

time_contents:    /* empty */
		| time_contents time_content
		;


time_content:     WEEKLY { sgTimeElementInit(); } WORD    { sgTimeElementAdd($3,T_WEEKLY); } ttime
                | WEEKLY { sgTimeElementInit(); } WEEKDAY { sgTimeElementAdd($3,T_WEEKDAY); } ttime
                | DATE   { sgTimeElementInit(); } date    { sgTimeElementEnd(); }
                ;

ttime:            ttime { sgTimeElementClone(); }  tval '-' tval
		| tval '-' tval 
                ;

date:             dval ttime
                | dval 
                | dval '-' dval ttime
                | dval '-' dval 
                | dvalcron ttime
                | dvalcron
                ;

dval:		  DVAL { sgTimeElementAdd($1,T_DVAL); }
                ;

tval:		  TVAL { sgTimeElementAdd($1,T_TVAL); }
                ;

dvalcron:	  DVALCRON { sgTimeElementAdd($1,T_DVALCRON); }
                ;

an_error:
		  error  
		     {  yyerror( "syntax error" );  }
	        ;

statements:       /* empty */
       		| statements statement
       		;

statement:   
                  category
	     	| source_block
	     	| category_block
		| http_server_def
	     	| admin_spec
	     	| https_cache_size
	     	| check_proxy_tunnels
	     	| log_block
	     	| log_all
		| enforce_https_with_hostname
		| enforce_https_offical_certificate
             	| dbhome
	     	| logdir
	     	| port
	     	| cpus
	     	| analyse_uncategorised
		| safe_search
		| max_logfile_size
	     	| acl_block
	     	| rew_block
	     	| time_block
	     	| an_error
	     	| NL
             	;

%%

void sgReadConfig( char * file )
{
  char *      text;
  char *      defaultFile = DEFAULT_CONFIGFILE;
  extern int  analyseUncategorised;

  lineno = 1;
  ufdbResetCPUs();

  configFile = file;
  if (configFile == NULL)
    configFile = defaultFile;

  yyin = fopen(configFile,"r");
  if (yyin == NULL) 
  {
    syslog( LOG_ALERT, "%s: cannot open configuration file %s", progname, configFile );
    ufdbLogFatalError( "%s: cannot open configuration file %s", progname, configFile );
    return;
  }

  ufdbLogError( "configuration file: %s", configFile );

  /* UFDBglobalDebug = 0; */
  UFDBglobalLogBlock = 0;
  UFDBglobalLogAllRequests = 0;
  UFDBglobalSafeSearch = 1;
  UFDBglobalHttpsWithHostname = 0;
  UFDBglobalHttpsOfficialCertificate = 0;
  UFDBglobalHttpdPort = 0;
  strcpy( UFDBglobalHttpdInterface, "all" );
  strcpy( UFDBglobalHttpdImagesDirectory, "." );

  (void) yyparse();
  fclose( yyin );

  /*
   * For analysis of uncategorised URLs we also load a "known" database
   * that contains URLs that are reviewed and known.
   */
  if (analyseUncategorised)
  {
     char * dbhome;
     char   dbfname[1024];

     dbhome = ufdbSettingGetValue("dbhome");
     if (dbhome == NULL)
       dbhome = DEFAULT_DBHOME;
     sprintf( dbfname, "%s/checked/domains.ufdb", dbhome );
     if (UFDBloadDatabase( &knownDb, dbfname ) != UFDB_API_OK)
     {
        knownDb.table.nNextLevels = 0;
	analyseUncategorised = 0;
        ufdbLogError( "uncategorised URLs will NOT be analysed since there is no proper database" );
     }
     else
        ufdbLogError( "a sample of uncategorised URLs will be analysed." );
  }

  switch (httpsChecks)
  {
     case UFDB_API_HTTPS_CHECK_OFF:   		text = "off";  break;
     case UFDB_API_HTTPS_CHECK_QUEUE_CHECKS:   	text = "queue-checks";  break;
     case UFDB_API_HTTPS_CHECK_AGRESSIVE:   	text = "agressive";  break;
     case UFDB_API_HTTPS_LOG_ONLY:   		text = "log-only";  break;
     default:  					text = "unknown";   break;
  }
  ufdbLogError( "proxy tunnel detection is set to \"%s\"", text );
  ufdbLogError( "safe-search is set to \"%s\"", UFDBglobalSafeSearch? "on" : "off" );

  if (defaultAcl == NULL)
     ufdbLogFatalError( "%s: default acl not defined in configuration file  %s", progname, configFile );
}



/*
 * Source functions
 */

void sgSource( char * source )
{
  struct Source * sp;

  if (Source != NULL)  
  {
    if ((struct Source *) sgSourceFindName(source) != NULL)
    {
      ufdbLogFatalError( "%s: source %s is defined in configuration file %s",
		       progname, source, configFile );
      return;
    }
  }

  sp = (struct Source *) ufdbMalloc( sizeof(struct Source) );
  sp->name = (char *) ufdbMalloc( strlen(source)+1 );
  strcpy( sp->name, source );
  sp->active = 1;
  sp->ip = NULL;
  sp->lastip = NULL;
  sp->domainDb = NULL;
  sp->userDb = NULL;
  sp->time = NULL;
  sp->within = 0;
  sp->cont_search = 0;
  sp->next = NULL;
  sp->userquota.seconds = 0;
  sp->userquota.renew = 0;
  sp->userquota.sporadic = 0;

  if (Source == NULL) {
    Source = sp;
    lastSource = sp;
  } else {
    lastSource->next = sp;
    lastSource = sp;
  }
}


void sgSourceEnd( void )
{
  struct Source * s;

  s = lastSource;
  if (s->ip == NULL && s->domainDb == NULL && s->userDb == NULL)
  {
    ufdbLogError( "sourceblock %s missing active content, set inactive", s->name );
    s->time = NULL;
    s->active = 0;
  }
}


void ufdbSourceUser( char * user )
{
  struct Source * sp;
  char * lc;
  struct UserQuotaInfo q;

  sp = lastSource;
  if (sp->userDb == NULL)
    sp->userDb = (void *) UFDBmemDBinit();

  for (lc = user; *lc != '\0'; lc++)    /* convert username to lowercase chars */
  {
    if (*lc >= 'A'  &&  *lc <= 'Z')
       *lc += 'a' - 'A';
  }

  UFDBmemDBinsert( (struct UFDBmemDB *) sp->userDb, user, (char *) setuserquota(&q),
	           sizeof(struct UserQuotaInfo) );
}


static void ufdbSourceUnixGroup(
   char *          groupName  )
{
   struct Source * sp;
   struct group *  grp;
   int             n;
   char *          user;

   sp = lastSource;
   if (sp->userDb == NULL) 
      sp->userDb = (void *) UFDBmemDBinit();
 
   /* walk through the /etc/group file with getgrent() to be sure that groups
    * that are on multiple lines are completely parsed.
    */
   n = 0;
   setgrent();
   while ((grp = getgrent()) != NULL)
   {
      if (strcmp( grp->gr_name, groupName ) == 0)
      {
         int i;

         n++;
	 for (i = 0; (user = grp->gr_mem[i]) != NULL; i++)
	 {
	    char * lc;
	    struct UserQuotaInfo q;

	    for (lc = user; *lc != '\0'; lc++)  	/* convert username to lowercase chars */
	    {
	      if (*lc >= 'A'  &&  *lc <= 'Z')
		 *lc += 'a' - 'A';
	    }
	    UFDBmemDBinsert( (struct UFDBmemDB *) sp->userDb, user, (char *) setuserquota(&q),
			     sizeof(struct UserQuotaInfo));
	 }
      }
   }
   endgrent();

   if (n == 0)
   {
      ufdbLogFatalError( "%s: %s is not a unix group", configFile, groupName );
      return;
   }
}


static void ufdbSourceGroup( 
   int    groupType,
   char * groupName  )
{
   switch (groupType)
   {
   case UFDB_GROUPTYPE_UNIX:
      ufdbSourceUnixGroup( groupName );
      break;
   default:     
      ufdbLogFatalError( "ufdbSourceGroup: unknown group type %d", groupType );
   }
}


void ufdbSourceUserList( 
  char * file  )
{
  char * dbhome;
  char * f;
  FILE * fp;
  char * p;
  char * c;
  char * s;
  char * lc;
  char * lineptr;
  struct Source * sp;
  struct UserQuotaInfo q;
  char   line[MAX_BUF];

  sp = lastSource;
  if (sp->userDb == NULL) 
    sp->userDb = (void *) UFDBmemDBinit();
 
  dbhome = ufdbSettingGetValue("dbhome");
  if (dbhome == NULL)
    dbhome = DEFAULT_DBHOME;

  if (file[0] == '/')
    f = ufdbStrdup( file );
  else
  {
    f = (char *) ufdbMalloc( strlen(dbhome) + strlen(file) + 2 );
    strcpy( f, dbhome );
    strcat( f, "/" );
    strcat( f, file );
  }

  if ((fp = fopen(f,"r")) == NULL)
  {
    ufdbLogError("%s: can't open userlist %s: %s", progname, f, strerror(errno) );
    ufdbFree( f );
    return;
  }

  while (fgets(line,sizeof(line),fp) != NULL)
  {
    if (line[0] == '#')			/* skip comments */
      continue;

    p = strchr( line, '\n' );
    if (p != NULL  &&  p != line) 
    {
      if (*(p - 1) == '\r') 		/* remove ^M  */
	p--;
      *p = '\0';
    }

    c = strchr( line, '#' );		/* remove comment */
    if (c != NULL)
       *c = '\0';

    p = strtok_r( line, " \t,", &lineptr );
    if ((s = strchr(line,':')) != NULL) 
    {
      *s = '\0';
      for (lc = line; *lc != '\0'; lc++)  /* convert username to lowercase chars */
      {
	if (*lc >= 'A'  &&  *lc <= 'Z')
	   *lc += 'a' - 'A';
      }
      UFDBmemDBinsert( (struct UFDBmemDB *) sp->userDb, line, (char *) setuserquota(&q),
                       sizeof(struct UserQuotaInfo));
    }
    else 
    {
      do {
	for (lc=p; *lc != '\0'; lc++)   /* convert username to lowercase chars */
        {
	  if (*lc >= 'A'  &&  *lc <= 'Z')
	     *lc += 'a' - 'A';
        }
        UFDBmemDBinsert( (struct UFDBmemDB *) sp->userDb, p, (char *) setuserquota(&q),
	                 sizeof(struct UserQuotaInfo));
      } while ((p = strtok_r(NULL," \t,",&lineptr)) != NULL);
    }
  }

  fclose( fp );
  ufdbFree( f );
}


static void ufdbSourceUserQuota( char * seconds, char * sporadic, char * renew )
{
   int s;
   struct UserQuota * uq;
   struct Source * sp;
 
   sp = lastSource;
   uq = &sp->userquota;
   s = atoi(seconds);
   if (s <= 0)
      ufdbLogError("Userquota seconds sporadic hourly|daily|weekly");
   uq->seconds = s; 
   s = atoi(sporadic);
   if (s <= 0)
      ufdbLogError("Userquota seconds sporadic hourly|daily|weekly");
   uq->sporadic = s; 
   s = atoi(renew);
   if (s <= 0)
      ufdbLogError("Userquota seconds sporadic hourly|daily|weekly");
   uq->renew = s;
}


void sgSourceDomain( char * domain )
{
   struct Source * sp;

   sp = lastSource;
   if (sp->domainDb == NULL) 
      sp->domainDb = (struct sgDb *) UFDBmemDBinit();
   UFDBmemDBinsert( (struct UFDBmemDB *) sp->domainDb, domain, NULL, 0 );
}


void sgSourceTime( char * name, int within )
{
   struct Time *   time = NULL;
   struct Source * sp;

   sp = lastSource;
   if ((time = sgTimeFindName(name)) == NULL)
   {
      ufdbLogFatalError( "%s: Time %s is not defined in configuration file %s",
		         progname, name, configFile );
      return;
   }
   sp->within = within;
   sp->time = time;
}


struct Source * sgSourceFindName( char * name )
{
   struct Source * p;

   for (p = Source; p != NULL; p = p->next)
   {
     if (strcmp(name,p->name) == 0)
        return p;
   }

   return NULL;
}


void sgSourceIpList( char * file )
{
  char * dbhome = NULL, *f;
  FILE * fd;
  char * p,*c,*cidr;
  int    i, l=0;
  char * lineptr;
  char   line[MAX_BUF];

  dbhome = ufdbSettingGetValue("dbhome");
  if (dbhome == NULL)
    dbhome = DEFAULT_DBHOME;

  if (file[0] == '/') {
    f = ufdbStrdup(file);
  } else {
    f = (char *) ufdbMalloc( strlen(dbhome) + strlen(file) + 2 );
    strcpy(f,dbhome);
    strcat(f,"/");
    strcat(f,file);
  }
  if ((fd = fopen(f,"r")) == NULL) {
    ufdbLogError("%s: can't open iplist %s: %s",progname, f,strerror(errno));
    return;
  }
  ufdbLogError("init iplist %s",f);
  while (fgets(line,sizeof(line),fd) != NULL)
  {
    l++;
    if (*line == '#')
      continue;
    p = strchr(line,'\n');
    if (p != NULL && p != line) {
      if (*(p - 1) == '\r') /* removing ^M  */
	p--;
      *p = '\0';
    }
    c = strchr(line,'#');
    p = strtok_r( line, " \t,", &lineptr );
    do {
      if (c != NULL && p >= c) /*find the comment */
	break;
      i=strspn(p,".0123456789/-");
      if (i == 0)
	break;
      *(p + i ) = '\0';
      if ((cidr = strchr(p,'/')) != NULL) {
	*cidr = '\0';
	cidr++;
	sgIp(p);
	sgIp(cidr);
	if (strchr(cidr,'.') == NULL)
	  sgSetIpType(SG_IPTYPE_CIDR,f,l);
	else 
	  sgSetIpType(SG_IPTYPE_CLASS,f,l);
      } else if ((cidr = strchr(p,'-')) != NULL) {
	*cidr = '\0';
	cidr++;
	sgIp(p);
	sgIp(cidr);
	sgSetIpType(SG_IPTYPE_RANGE,f,l);
      } else {
	sgIp(p);
	sgIp(ufdbStrdup("255.255.255.255"));
	sgSetIpType(SG_IPTYPE_HOST,f,l);
      }
    } while ((p=strtok_r(NULL," \t,",&lineptr)) != NULL);
  }

  fclose(fd);
}


/*
 *  called by main loop: find a source definition based on the user ident/domain/ip
 */
struct Source * sgFindSource( 
  struct Source *    allSrcs, 
  char *             net, 
  struct SquidInfo * si )
{
  struct Source *    s;
  struct Ip *        ip;
  int                foundip, founduser, founddomain, unblockeduser;
  unsigned long      i, octet, op;

#ifdef UFDB_USERQUOTA_SUPPORT
  struct UserQuotaInfo * q = NULL;
#endif

#if 0
  ufdbLogError( "  sgFindSource( %s, %s, ... )", allSrcs->name, net );
#endif

  octet = 0;
  if (net != NULL)
  {
    if (sgConvDot(net,&op) != NULL)
      octet = op;
  }

  for (s = allSrcs; s != NULL; s = s->next)
  {
    if (s->active == 0)
      continue;

    foundip = founduser = founddomain = 0;
    unblockeduser = 1;

    if (s->ip != NULL)
    {
      if (net == NULL)
	foundip = 0;
      else 
      {
	for (ip = s->ip; ip != NULL; ip = ip->next)
	{
	  if (ip->net_is_set == 0)
	    continue;

	  if (ip->type == SG_IPTYPE_RANGE) 
	  {
	    if (octet >= ip->net && octet <= ip->mask) {
	      foundip = 1;
	      break;
	    }
	  }
	  else				/* CIDR or HOST */
	  { 
	    i = octet & ip->mask;
	    if (i == ip->net) {
	      foundip = 1;
	      break;
	    }
	  }
	}
      }
    }
    else
      foundip = 1;

    if (s->userDb != NULL)
    {
      if (si->ident[0] == '\0')
	founduser = 0;
      else 
      {
#ifdef UFDB_USERQUOTA_SUPPORT
	/* see if the user has a quota definition */
	if (UFDBmemDBfind( (struct UFDBmemDB *) s->userDb, si->ident, (char **) (char *) &q ) == 1)
	{
	  founduser = 1;
	  unblockeduser = 1;
	  if (s->userquota.seconds != 0) 
	  {
	    time_t t = time(NULL) + globalDebugTimeDelta;
#if 0
	    ufdbLogError( "status %d time %d lasttime %d consumed %d", 
	                  q->status, q->time, q->last, q->consumed );
	    ufdbLogError( "renew %d seconds %d", s->userquota.renew, s->userquota.seconds );
#endif
	    if (q->status == 0)   /* first time */
	    {
	      q->status = 1;
	      q->time = t;
	      q->last = t;
	      /* ufdbLogError("user %s first time %d", si->ident, q->time); */
	    }
	    else if (q->status == 1) 
	    {
	      /* ufdbLogError("user %s other time %d %d", si->ident, q->time, t ); */
	      if (s->userquota.sporadic > 0) 
	      {
		if (t - q->last  < s->userquota.sporadic) {
		  q->consumed += t - q->last;
		  q->time = t;
		}
		if (q->consumed > s->userquota.seconds) {
		  q->status = 2; /* block this user, time is up */
		  unblockeduser = 0;
		}
		q->last = t;
		/* ufdbLogError( "user %s consumed %d %d", si->ident, q->consumed, q->last ); */
	      }
	      else if (q->time + s->userquota.seconds < t) 
	      {
		ufdbLogError( "time is up user %s blocket", si->ident);
		q->status = 2; /* block this user, time is up */
		unblockeduser = 0;
	      } 
	    }
	    else
	    {
	      ufdbLogError( "user %s blocked %d %d %d %d", 
	         si->ident, q->status, 
		 q->time, t, (q->time + s->userquota.renew) - t);
	      if (q->time + s->userquota.renew < t)  /* new chance */
	      {
		/* ufdbLogError( "user %s new chance", si->ident ); */
		unblockeduser = 1;
		q->status = 1;
		q->time = t;
		q->consumed = 0;
	      }
	      else 
		unblockeduser = 0;
	    }
	    UFDBmemDBinsert( (struct UFDBmemDB *) s->userDb, si->ident, (char *) q,
		             sizeof(struct UserQuotaInfo) );
	  }
	}
#endif  /* UFDB_USERQUOTA_SUPPORT */
      }
    }
    else
      founduser = 1;

    if (s->domainDb != NULL)
    {
      if (si->srcDomain[0] == '\0')
	founddomain = 0;
      else 
      {
	if (UFDBmemDBfind( (struct UFDBmemDB *) s->domainDb, si->srcDomain, (char **) NULL) == 1)
	  founddomain = 1;
      }
    }
    else
      founddomain = 1;

    if (founduser && foundip && founddomain)
    {
      if (unblockeduser)
	return s;
      else 
      {
	si->lastActiveSource = s;
	return NULL;
      }
    }
  }

  return NULL;
}


/* category block funtions */

void ufdbCategory( char * dest )
{
  struct Category * sp;

  if (Dest != NULL) 
  {
    if ((struct Category *) ufdbCategoryFindName(dest) != NULL)
    {
      ufdbLogFatalError( "%s: category %s is defined in configuration file %s",
		         progname, dest, configFile );
      return;
    }
  }

  sp = (struct Category *) ufdbMalloc( sizeof(struct Category) );
  sp->name = (char  *) ufdbMalloc( strlen(dest) + 1 );
  strcpy( sp->name, dest );
  sp->active = 1;
  sp->domainlist = NULL;
  sp->domainlistDb = NULL;
  sp->expressionlist = NULL;
  sp->regExp = NULL;
  sp->rewrite = NULL;
  sp->redirect = NULL;
  sp->options = 0;
  sp->time = NULL;
  sp->within = 0;
  sp->next = NULL;

  if (Dest == NULL) {
    Dest = sp;
    lastDest = sp;
  } else {
    lastDest->next = sp;
    lastDest = sp;
  }
}


void ufdbCategoryEnd( void )
{
  struct Category * d;
 
  d = lastDest;
  if (d->domainlist == NULL && d->expressionlist == NULL
      && d->redirect == NULL && d->rewrite == NULL)
  {
    ufdbLogError( "dest block \"%s\" is missing content, set inactive", d->name );
    d->time = NULL;
    d->active = 0;
  }
}


void ufdbCategoryOption( int value, int option )
{
   struct Category * sp;

   sp = lastDest;
   if (value)
      sp->options = sp->options | option;
   else
      sp->options = sp->options & (~option);

   if (option == UFDB_OPT_HTTPS_WITH_HOSTNAME)
      UFDBglobalHttpsWithHostname = 1;
   if (option == UFDB_OPT_HTTPS_OFFICAL_CERTIFICATE)
      UFDBglobalHttpsOfficialCertificate = 1;
}


void ufdbCategoryDomainList( char * domainlist )
{
  struct Category * sp;
  char * dbhome = NULL;
  char * dl = domainlist;
  char * name;

  dbhome = ufdbSettingGetValue("dbhome");
  sp = lastDest;
  if (dbhome == NULL)
    dbhome = DEFAULT_DBHOME;

  if (domainlist == NULL)
  {
    name = sp->name;
    dl = (char *) ufdbMalloc( sizeof("/dest/") + strlen(name) + sizeof("/domainlist") + 2 );
    strcpy(dl,"/dest/");
    strcat(dl,name);
    strcat(dl,"/domainlist");

    sp->domainlist = (char *) ufdbMalloc( strlen(dbhome) + strlen(dl) + 2 );
    strcpy(sp->domainlist,dbhome);
    strcat(sp->domainlist,"/");
    strcat(sp->domainlist,dl);
  }
  else
  {
    if (domainlist[0] == '/') 
      sp->domainlist = ufdbStrdup( domainlist );
    else
    {
       sp->domainlist = (char *) ufdbMalloc( strlen(dbhome) + strlen(domainlist) + 2 );
       strcpy( sp->domainlist, dbhome );
       strcat( sp->domainlist, "/" );
       strcat( sp->domainlist, domainlist );
    }
  }

  sp->domainlistDb = (struct sgDb *) ufdbCalloc( 1, sizeof(struct sgDb) );
  sp->domainlistDb->type = SGDBTYPE_DOMAINLIST;
  sgDbInit( sp->domainlistDb, sp->domainlist );
  ufdbLogError( "init domainlist %s", sp->domainlist );
  if (sp->domainlistDb->entries == 0)  /* empty database */
  {
    ufdbLogError( "domainlist empty, removed from memory" );
    ufdbFree( sp->domainlistDb );
    sp->domainlistDb = NULL;
  }
}


void ufdbFreeDomainDb( struct sgDb * dbp )
{
   struct UFDBmemTable * mt;

   if (dbp == NULL)
      return;

   mt = (struct UFDBmemTable *) dbp->dbcp;
   UFDBfreeTableIndex( &(mt->table) ); 		/*.nextLevel[0]) ); */
   ufdbFree( (void *) mt->mem );
   ufdbFree( mt );
   ufdbFree( dbp );
}


void ufdbFreeIpList( struct Ip * p )
{
   if (p == NULL)
      return;

   ufdbFreeIpList( p->next );
   if (p->str != NULL)
      ufdbFree( p->str );
   ufdbFree( p );
}


void ufdbFreeSourceList( struct Source * p )
{
   if (p == NULL)
      return;

   ufdbFreeSourceList( p->next );

   ufdbFreeIpList( p->ip );
   ufdbFree( p->name );
   UFDBmemDBdeleteDB( (struct UFDBmemDB *) p->domainDb );
   UFDBmemDBdeleteDB( (struct UFDBmemDB *) p->userDb );

   ufdbFree( p );
}


void ufdbFreeAclDestList( struct AclDest * p )
{
   if (p == NULL)
      return;

   ufdbFreeAclDestList( p->next );
   ufdbFree( p->name );
   ufdbFree( p );
}


void ufdbFreeAclList( struct Acl * p )
{
   if (p == NULL)
      return;

   ufdbFreeAclList( p->next );

   ufdbFree( p->name );
   ufdbFreeAclDestList( p->pass );
   ufdbFree( p->redirect );

   ufdbFree( p );
}


void ufdbFreeSettingList( struct ufdbSetting * p )
{
   if (p == NULL)
      return;

   ufdbFreeSettingList( p->next );
   ufdbFree( p->name );
   ufdbFree( p->value );
   ufdbFree( p );
}


void ufdbFreeDestDomainList( struct Category * p )
{
   if (p == NULL)
      return;

   ufdbFreeDestDomainList( p->next );

   ufdbFree( p->name );
   ufdbFree( p->domainlist );
   ufdbFreeDomainDb( p->domainlistDb );
   ufdbFree( p->expressionlist );
   ufdbFreeRegExprList( p->regExp );
   ufdbFree( p->redirect );
   ufdbFree( p->rewrite );
   ufdbFree( p );
}


void ufdbFreeRewriteList( struct sgRewrite * p )
{
   if (p == NULL)
      return;

   ufdbFreeRegExprList( p->rewrite );
   ufdbFreeRewriteList( p->next );
   ufdbFree( p->name );
   ufdbFree( p );
}


void ufdbFreeTimeElement( struct TimeElement * t )
{
   if (t == NULL)
      return;

   ufdbFreeTimeElement( t->next );
   ufdbFree( t );
}


void ufdbFreeTime( struct Time * t )
{
   if (t == NULL)
      return;

   ufdbFreeTime( t->next );
   ufdbFree( t->name );
   ufdbFreeTimeElement( t->element );
   ufdbFree( t );
}


void ufdbFreeAllMemory( void )
{
   if (UFDBglobalLogDir != NULL)
      ufdbFree( UFDBglobalLogDir );
   UFDBglobalLogDir = NULL;
   globalLogFile = NULL;

   ufdbFreeSettingList( Setting );
   ufdbFreeRewriteList( Rewrite );
   ufdbFreeDestDomainList( Dest );
   ufdbFreeAclList( Acl );
   ufdbFreeSourceList( Source );

   /* free knownDb */
   if (knownDb.table.nNextLevels > 0  &&  knownDb.mem != NULL)
   {
      UFDBfreeTableIndex( &(knownDb.table) ); 		/* .nextLevel[0]) ); */
      ufdbFree( (void *) knownDb.mem );
      knownDb.table.nNextLevels = 0;
      knownDb.mem = NULL;
   }

   /* set all pointers back to NULL */

   lastSetting = NULL;
   Setting = NULL;

   lastSource = NULL;
   Source = NULL;
   /*** saveSource = NULL; ***/

   lastDest = NULL;
   Dest = NULL;

   lastRewrite = NULL;
   Rewrite = NULL;
   lastRewriteRegExec = NULL;

   ufdbFreeTime( Time );
   Time = NULL;
   lastTime = NULL;

   ufdbFree( TimeElementsEvents );
   TimeElementsEvents = NULL;

   lastTimeElement = NULL;
   TimeElement = NULL;

   lastAcl = NULL;
   defaultAcl = NULL;
   Acl = NULL;
   lastAclDest = NULL;

   lastRegExpDest = NULL;
}


void ufdbCategoryUrlList( char * urllist )
{
   if (urllist == NULL)
      urllist = "-";
   ufdbLogError( "\"urllist %s\" is deprecated and ignored *****", urllist );
}


void ufdbCategoryExpressionList( char * exprlist, char * chcase  )
{
  FILE * fp;
  struct Category * sp;
  struct ufdbRegExp * regexp;
  char * dbhome;
  char * dl;
  char * name, *p;
  int    flags        = REG_EXTENDED | REG_NOSUB;
  char   buf[MAX_BUF];
  char   errbuf[256];

  dbhome = ufdbSettingGetValue("dbhome");
  sp = lastDest;
  if (dbhome == NULL)
    dbhome = DEFAULT_DBHOME;

  if (exprlist == NULL)
  {
    name = sp->name;
    dl = (char *) ufdbMalloc( sizeof("/dest/") + strlen(name) + sizeof("/expressionlist") + 2 );
    strcpy(dl,"/dest/");
    strcat(dl,name);
    strcat(dl,"/expressionlist");

    flags |= REG_ICASE; 	/* default case insensitive */

    sp->expressionlist = (char *) ufdbMalloc( strlen(dbhome) + strlen(dl) + 2 );
    strcpy(sp->expressionlist,dbhome);
    strcat(sp->expressionlist,"/");
    strcat(sp->expressionlist,dl);
  }
  else
  {
    if (exprlist[0] == '/') {
      sp->expressionlist = ufdbStrdup(exprlist);
    } else {
       sp->expressionlist = (char *) ufdbMalloc( strlen(dbhome) + strlen(exprlist) + 2 );
       strcpy(sp->expressionlist,dbhome);
       strcat(sp->expressionlist,"/");
       strcat(sp->expressionlist,exprlist);
    }
    if (*chcase == 'i')
       flags |= REG_ICASE;     /* set case insensitive */
  }

  ufdbLogError( "init expressionlist %s", sp->expressionlist );
  if ((fp = fopen(sp->expressionlist, "r")) == NULL) 
  {
    ufdbLogFatalError( "expression list %s: %s", sp->expressionlist, strerror(errno) );
    return;
  }

  while (!feof(fp)  &&  fgets(buf, sizeof(buf), fp) != NULL) 
  {
    p = (char *) strchr(buf,'\n');
    if (p != NULL && p != buf) {
      if (*(p-1) == '\r') 	/* removing ^M  */
	p--;
      *p = '\0';
    }
    regexp = ufdbNewPatternBuffer( buf, flags );
    if (regexp->error) 
    {
      regerror( regexp->error, regexp->compiled, errbuf, sizeof(errbuf) );
      ufdbLogError( "%s: %s: %s", sp->expressionlist, strerror(errno), buf );
    }
    if (lastDest->regExp == NULL) 
    {
      lastDest->regExp = regexp;
      lastRegExpDest = regexp;
    }
    else
    {
      lastRegExpDest->next = regexp;
      lastRegExpDest = regexp;
    }
  }

  fclose(fp);
}


void ufdbCategoryRedirect( char * value )
{
  struct Category * sp;

  sp = lastDest;
  sp->redirect = (char *) ufdbMalloc( strlen(value) + 1 );
  strcpy( sp->redirect, value );
}


void ufdbCategoryRewrite( char * value )
{
  struct sgRewrite *   rewrite;
  struct Category * sp;

  sp = lastDest;
  if ((rewrite = sgRewriteFindName(value)) == NULL)
  {
    ufdbLogFatalError( "%s: Rewrite %s is not defined in configuration file %s",
		     progname, value, configFile );
    return;
  }
  sp->rewrite = rewrite;
}



void ufdbCategoryTime( char * name, int within )
{
  struct Time * time = NULL;
  struct Category * sp;

  sp = lastDest;
  if ((time = sgTimeFindName(name)) == NULL) 
  {
    ufdbLogFatalError( "%s: Time %s is not defined in configuration file %s",
		     progname, name, configFile );
    return;
  }
  sp->within = within;
  sp->time = time;
}


struct Category * ufdbCategoryFindName( 
  char *               name )
{
  struct Category * p;

  for (p = Dest; p != NULL; p = p->next)
  {
    if (strcmp(name,p->name) == 0)
      return p;
  }
  return NULL;
}


/*
 * Setting functions
 */
void ufdbSetting( char * name, char * value )
{
  struct ufdbSetting * sp;

  if (strcmp( name, "administrator" ) == 0)
  {
     char * p;

     while ((p = strchr( value, '?' )) != NULL)
        *p = '_';
     while ((p = strchr( value, '&' )) != NULL)
        *p = '_';
  }

  if (strcmp( name, "port" ) == 0)
  {
     portNum = atoi( value );
     if (portNum <= 0)
     {
        ufdbLogError( "port number must be > 0, using default port %d", UFDB_DAEMON_PORT );
	portNum = UFDB_DAEMON_PORT;
     }
     return;
  }

  if (Setting != NULL)
  {
    if (ufdbSettingFindName(name) != NULL)
    {
      ufdbLogFatalError( "%s: setting %s is defined multiple times in configuration file %s",
		         progname, name, configFile );
      return;
    }
  }

  sp = (struct ufdbSetting *) ufdbMalloc( sizeof(struct ufdbSetting) );
  sp->name = ufdbStrdup(name);
  sp->value = ufdbStrdup(value);
  sp->next = NULL;

  if (Setting == NULL) 
  {
    Setting = sp;
    lastSetting = sp;
  }
  else
  {
    lastSetting->next = sp;
    lastSetting = sp;
  }

  if (strcmp(name,"dbhome") == 0) 
  {
     struct stat dirbuf;

     if (stat( value, &dirbuf ) != 0)
     {
        ufdbLogError( "dbhome: directory does not exist or "
	              "access rights are insufficient (value is \"%s\")", 
		      value );
     }
     else
     {
        if (!S_ISDIR(dirbuf.st_mode))
	{
	   ufdbLogError( "dbhome: %s is not a directory", value );
	}
     }
  }

  if (strcmp(name,"logdir") == 0) 
  {
     UFDBglobalLogDir = ufdbStrdup( value );
     ufdbSetGlobalErrorLogFile();
     ufdbLogError( "configuration file: %s", configFile );
  }
}


struct ufdbSetting * ufdbSettingFindName( char * name )
{
  struct ufdbSetting * p;

  for (p = Setting; p != NULL; p = p->next)
  {
    if (strcmp(name,p->name) == 0)
      return p;
  }
  return NULL;
}


char * ufdbSettingGetValue( char * name )
{
  struct ufdbSetting * p;

  p = ufdbSettingFindName( name );
  if (p != NULL)
    return p->value;
  return NULL;
}


/*
 * sgRewrite function
 */

void sgRewrite( char * rewrite )
{
  struct sgRewrite * rew;

  if (Rewrite != NULL)
  {
    if ((struct sgRewrite *) sgRewriteFindName(rewrite) != NULL)
    {
      ufdbLogFatalError( "%s: rewrite %s is defined in configuration file %s",
		       progname, rewrite, configFile );
      return;
    }
  }
  rew = (struct sgRewrite *) ufdbMalloc( sizeof(struct sgRewrite) );
  rew->name = ufdbStrdup(rewrite);
  rew->active = 1;
  rew->rewrite = NULL;
  rew->time = NULL;
  rew->within = 0;
  rew->next = NULL;

  if (Rewrite == NULL) 
  {
    Rewrite = rew;
    lastRewrite = rew;
  }
  else
  {
    lastRewrite->next = rew;
    lastRewrite = rew;
  }
}


void sgRewriteTime( char * name, int within )
{
  struct Time * time = NULL;
  struct sgRewrite * sp;

  sp = lastRewrite;
  if ((time = sgTimeFindName(name)) == NULL)
  {
    ufdbLogFatalError( "%s: Time %s is not defined in configuration file %s",
		     progname, name, configFile );
    return;
  }
  sp->within = within;
  sp->time = time;
}


void sgRewriteSubstitute( char * string )
{
  char * pattern, 
       * subst = NULL, 
       * p;
  int    flags = REG_EXTENDED;
  int    global = 0;
  char * httpcode = NULL;
  struct ufdbRegExp * regexp;
  char   errbuf[256];

  pattern = string + 2; 	/* skipping s@ */
  p = pattern;
  while ((p = strchr(p,'@')) != NULL)
  {
    if (*( p - 1) != '\\') {
      *p = '\0';
      subst = p + 1;
      break;
    }
    p++;
  }

  p = strrchr(subst,'@');
  while (p != NULL && *p != '\0')
  {
    if (*p == 'r' )
      httpcode =  REDIRECT_TEMPORARILY;
    if (*p == 'R' )
      httpcode =  REDIRECT_PERMANENT;
    if (*p == 'i' || *p == 'I')
      flags |= REG_ICASE;
    if (*p == 'g')
      global = 1;
    *p = '\0'; 		/* removes @i from string */
    p++;
  } 
  regexp = ufdbNewPatternBuffer(pattern,flags);
  if (regexp->error) 
  {
      regerror(regexp->error,regexp->compiled, errbuf,sizeof(errbuf));
      ufdbLogError("Error in regexp %s: %s",pattern,errbuf);
  }
  else {
    regexp->substitute = ufdbStrdup( subst );
  }

  if (lastRewrite->rewrite == NULL)
    lastRewrite->rewrite = regexp;
  else 
    lastRewriteRegExec->next = regexp;
  regexp->httpcode = httpcode;
  regexp->global = global;
  lastRewriteRegExec = regexp;
}


struct sgRewrite * sgRewriteFindName( char * name )
{
  struct sgRewrite * p;

  for (p = Rewrite; p != NULL; p = p->next)
  {
    if (strcmp(name,p->name) == 0)
      return p;
  }

  return NULL;
}



/*
 * Time functions
 */

void sgTime( char * name )
{
  struct Time * t;

  if (Time != NULL)
  {
    if ((struct Time *) sgTimeFindName(name) != NULL)
    {
      ufdbLogFatalError( "%s: time %s is defined in configuration file %s",
		         progname, name, configFile );
      return;
    }
  }
  else 
    numTimeElements = 0;

  t = (struct Time *) ufdbMalloc( sizeof(struct Time) );
  t->name = ufdbStrdup( name );
  t->active = 1;
  t->element = NULL;
  t->next = NULL;

  TimeElement = NULL;
  lastTimeElement = NULL;
  if (Time == NULL) 
  {
    Time = t;
    lastTime = t;
  }
  else
  {
    lastTime->next = t;
    lastTime = t;
  }
}


void sgTimeElementInit( void )
{
  struct TimeElement * te;

  te = (struct TimeElement *) ufdbCalloc( 1, sizeof(struct TimeElement) );
  numTimeElements++;
  if (lastTime->element == NULL)
    lastTime->element = te;
  if (lastTimeElement != NULL)
    lastTimeElement->next = te;
  lastTimeElement = te;
}


void sgTimeElementEnd( void )
{
  time_switch = 0;
  date_switch = 0;

  if (lastTimeElement->fromdate != 0)
  {
    if (lastTimeElement->todate == 0)
      lastTimeElement->todate = lastTimeElement->fromdate + 86399;
    else 
      lastTimeElement->todate = lastTimeElement->todate + 86399;
  }

  if (lastTimeElement->from == 0  &&  lastTimeElement->to == 0)
    lastTimeElement->to = 1439;  /* set time to 23:59 */
}


void sgTimeElementAdd( char * element, char type ) 
{
  struct TimeElement * te;
  char * p;
  char   wday = 0;
  int    h, m, Y, M = 0, D = -1;
  time_t sec;
  char * lineptr;

  te = lastTimeElement;
  switch (type)
  {
  case T_WEEKDAY:
    p = strtok_r( element, " \t,", &lineptr );
    do {
      if (*p == '*') {
	wday = 127;
      } else if (!strncmp(p,"sun",3)) {
	wday = wday | 0x01;
      } else if (!strncmp(p,"mon",3)) {
	wday = wday | 0x02;
      } else if (!strncmp(p,"tue",3)) {
	wday = wday | 0x04;
      } else if (!strncmp(p,"wed",3)) {
	wday = wday | 0x08;
      } else if (!strncmp(p,"thu",3)) {
	wday = wday | 0x10;
      } else if (!strncmp(p,"fri",3)) {
	wday = wday | 0x20;
      } else if (!strncmp(p,"sat",3)) {
	wday = wday | 0x40;
      }
      p = strtok_r( NULL, " \t,", &lineptr );
    } while (p != NULL);
    te->wday = wday;
    break;

  case T_TVAL:
    sscanf( element, "%d:%d", &h, &m );
    if ((h < 0 && h > 24) && (m < 0 && m > 59))
    {
      ufdbLogFatalError( "%s: time format error in %s line %d",
		         progname, configFile, lineno );
      h = 0;
      m = 0;
    }
    if (time_switch == 0) {
      time_switch++;
      te->from = (h * 60) + m ;
    } else {
      time_switch=0;
      te->to = (h * 60) + m ;
    }
    break;

  case T_DVAL:
    sec = date2sec(element);
    if (sec == -1) 
    {
      ufdbLogFatalError( "%s: date format error in %s line %d",
		         progname, configFile, lineno );
      sec = 1;
    }
    if (date_switch == 0) {
      date_switch++;
      te->fromdate = sec;
    } else {
      date_switch = 0;
      te->todate = sec;
    }
    break;

  case T_DVALCRON:
    p = strtok_r( element, "-.", &lineptr );
    Y = atoi(p);
    if (*p == '*')
      Y = -1;
    else
      Y = atoi(p);
    while ((p=strtok_r(NULL,"-.",&lineptr)) != NULL) {
      if (*p == '*')
	if (M == 0)
	  M = -1;
	else 
	  D = -1;
      else
	if (M == 0)
	  M = atoi(p);
	else
	  D = atoi(p);
    }
    te->y=Y; te->m=M; te->d=D;
    break;

  case T_WEEKLY:
    p = element;
    while (*p != '\0') {
      switch (*p) {
      case 'S':
      case 's':
	wday = wday | 0x01;
	break;
      case 'M':
      case 'm':
	wday = wday | 0x02;
	break;
      case 'T':
      case 't':
	wday = wday | 0x04;
	break;
      case 'W':
      case 'w':
	wday = wday | 0x08;
	break;
      case 'H':
      case 'h':
	wday = wday | 0x10;
	break;
      case 'F':
      case 'f':
	wday = wday | 0x20;
	break;
      case 'A':
      case 'a':
	wday = wday | 0x40;
	break;
      default:
	ufdbLogFatalError( "%s: weekday format error in %s line %d",
			   progname, configFile, lineno );
	break;
      }
      p++;
    }
    te->wday = wday;
    break;
  }
}



struct Time *sgTimeFindName(char *name)
{
  struct Time *p;

  for (p=Time; p != NULL; p = p->next)
  {
    if (strcmp(name,p->name) == 0)
      return p;
  }
  return NULL;
}


int sgTimeCmp( const void * a, const void * b )
{
   const int * aa = (const int *) a;
   const int * bb = (const int *) b;
   return *aa - *bb;
}


void sgTimeElementSortEvents( void )
{
  struct Time * p;
  struct TimeElement * te;
  int i = 0, j;
  int * t;
 
  /* TO-DO: analyze this code */
  if (Time != NULL)
  {
    TimeElementsEvents = (int *) ufdbCalloc( numTimeElements * 2 , sizeof(int) ); 
    for (p = Time; p != NULL; p = p->next) 
    {
      for (te = p->element; te != NULL; te = te->next) 
      {
        TimeElementsEvents[i++] = te->from == 0 ? 1440 : te->from;
        TimeElementsEvents[i++] = te->to == 0   ? 1440 : te->to;
      }
    }
 
    qsort( TimeElementsEvents, numTimeElements * 2, sizeof(int), sgTimeCmp );

    t = (int *) ufdbCalloc( numTimeElements * 2, sizeof(int) ); 
    for (i=0,j=0; i < numTimeElements * 2; i++) 
    {
      if (j==0)
      {
        t[j++] = TimeElementsEvents[i];
      }
      else
      {
        if (t[j-1] != TimeElementsEvents[i]) {
 	 t[j++] = TimeElementsEvents[i];
        }
      }
    }
    ufdbFree( TimeElementsEvents );
    numTimeElements = j;
    TimeElementsEvents = t;
  }
}


int sgTimeNextEvent( void )
{
  time_t     t;
  struct tm  lt;
  int        m = 0; 
#if HAVE_SIGACTION
  struct sigaction act;
#endif
  int index = 0;
  int lastval = 0;

  if (Time == NULL)
    return 0;
  t = time(NULL) + globalDebugTimeDelta;

  localtime_r( &t, &lt ); 
  m = (lt.tm_hour * 60) + lt.tm_min ;
  
  for (index=0; index < numTimeElements; index++)
  {
    if (TimeElementsEvents[index] >= m) {
      break;
    }
  }
  lastval = TimeElementsEvents[index];

#if HAVE_SIGACTION
#ifndef SA_NODEFER
#define SA_NODEFER 0
#endif
  act.sa_handler = sgAlarm;
  act.sa_flags = SA_NODEFER | SA_RESTART;
  sigemptyset( &act.sa_mask );
  sigaction(SIGALRM, &act, NULL);
#else
#if HAVE_SIGNAL
  signal(SIGALRM, &sgAlarm);
#else
#endif
#endif

  if (lastval < m)
    m = (((1440 - m ) + lastval) * 60) - lt.tm_sec;
  else
    m = ((lastval - m) * 60) - lt.tm_sec;
  if (m <= 0)
    m = 30;

  ufdbLogError( "recalculating alarm in %d seconds", (unsigned int) m );
  alarm( (unsigned int) m );
  sgTimeCheck( &lt, t );
  sgTimeSetAcl();

  return 0;
}


int sgTimeCheck(struct tm *lt, time_t t)
{
  struct Time *sg;
  struct TimeElement *te;
  int min;

  if (Time == NULL)
    return -1;

  for (sg = Time; sg != NULL; sg = sg->next)
  {
    sg->active = 0;
    for (te = sg->element; te != NULL ; te = te->next)
    {
      if (te->wday != 0) {
	if (((1 << lt->tm_wday ) & te->wday) != 0) {
	  min = (lt->tm_hour * 60 ) + lt->tm_min;
	  if (min >= te->from && min < te->to) {
	    sg->active = 1;
	    break;
	  }
	}
      } else { /* date */
	if (te->fromdate != 0) {
	  if (t >= te->fromdate && t <= te->todate) {
	    min = (lt->tm_hour * 60 ) + lt->tm_min;
	    if (min >= te->from && min < te->to) {
	      sg->active =1;
	      break;
	    }
	  }
	} else { /* cron */
	  if (te->y == -1 || te->y == (lt->tm_year + 1900)) {
	    if (te->m == -1 || te->m == (lt->tm_mon + 1)) {
	      if (te->d == -1 || te->d == (lt->tm_mday)) {
		min = (lt->tm_hour * 60 ) + lt->tm_min;
		if (min >= te->from && min < te->to) {
		  sg->active =1;
		  break;
		}
	      }
	    }
	  }
	}
      }
    }
  }

  return 0;
}


void sgTimeSetAcl( void )
{
  struct Acl *         acl = defaultAcl;
  struct Category * d;
  struct Source *      s;
  struct sgRewrite *   rew;

  for (acl=Acl; acl != NULL; acl = acl->next)
  {
    if (acl->time != NULL) {
      acl->active = acl->time->active;
      if (acl->within == OUTSIDE)
      {
	if (acl->active)
	  acl->active = 0;
	else
	  acl->active = 1;
      }
      if (acl->next != NULL && acl->next->within == ELSE) 
      {
	if (acl->active == 0) {
	  acl->next->active = 1;
	} else {
	  acl->next->active = 0;
	}
      }
    }
  }

  for (d = Dest; d != NULL; d = d->next)
  {
    if (d->time != NULL) 
    {
      d->active = d->time->active;
      if (d->within == OUTSIDE)
      {
	if (d->active)
	  d->active = 0;
	else
	  d->active = 1;
      }
    }
  }

  for (s = Source; s != NULL; s = s->next)
  {
    if (s->time != NULL) {
      s->active = s->time->active;
      if (s->within == OUTSIDE)
      {
	if (s->active)
	  s->active = 0;
	else
	  s->active = 1;
      }
    }
  }

  for (rew = Rewrite; rew != NULL; rew = rew->next)
  {
    if (rew->time != NULL) {
      rew->active = rew->time->active;
      if (rew->within == OUTSIDE)
      {
	if (rew->active)
	  rew->active = 0;
	else
	  rew->active = 1;
      }
    }
  }
}


void sgTimeElementClone( void )
{
  struct TimeElement * te;
  struct TimeElement * tmp;

  te = lastTimeElement;
  if (te == NULL) 
  {
    ufdbLogFatalError( "No previous TimeElement in sgTimeElementClone !" );
    return;
  }
  else
  {
    sgTimeElementInit();
    lastTimeElement->wday = te->wday;
    lastTimeElement->from = te->from;
    lastTimeElement->to = te->to;
    lastTimeElement->y = te->y;
    lastTimeElement->m = te->m;
    lastTimeElement->d = te->d;
    lastTimeElement->fromdate = te->fromdate;
    lastTimeElement->todate = te->todate;
    tmp = lastTimeElement;
    lastTimeElement = te;
    sgTimeElementEnd();
    lastTimeElement = tmp;
  }
}


void sgTimePrint( void )
{
  struct Time * t;
  struct TimeElement * te;
  char   niso_str[22];

  for (t = Time; t != NULL; t = t->next)
  {
    printf( "Time %s is %s\n", t->name, t->active ? "active" : "inactive" );
    for (te = t->element; te != NULL; te = te->next)
    {
      printf("\tte->wday     = %x\n",te->wday);
      printf("\tte->from     = %d\n",te->from);
      printf("\tte->to       = %d\n",te->to);
      printf("\tte->y,m,d    = %d-%d-%d\n",te->y,te->m,te->d);
      niso( te->fromdate, niso_str );
      printf("\tte->fromdate = %s\n", te->fromdate == 0 ?  "0" : niso_str );
      niso( te->todate, niso_str );
      printf("\tte->todate   = %s\n\n", te->todate == 0 ?  "0" : niso_str );
    }
  }
}


/*
 * IP functions
 */

void sgSetIpType( 
  int         type, 
  char *      file, 
  int         line )
{
  struct Ip * ip;
  char *      p;
  char *      f;
  int         l;
  unsigned long octet;
  unsigned long op;

  ip = sgIpLast( lastSource );
  f = (file == NULL) ? configFile : file;
  l = (line == 0) ? lineno : line;
  if (type == SG_IPTYPE_HOST)
    ip->mask = 0xffffffff;

  if (type == SG_IPTYPE_RANGE)
  {
    if (sgConvDot(ip->str,&op) == NULL)
    {
      ufdbLogFatalError( "%s: address error in %s line %d", progname, f, l );
      ip->mask = 0;
    }
    else 
      ip->mask = op;
    if (ip->net > ip->mask)
    {
      ufdbLogFatalError( "%s: IP range error in %s line %d", progname, f, l );
    }
  }

  if (type == SG_IPTYPE_CLASS)
  {
    p = ip->str;
    if (*p == '/')
      p++;
    if (sgConvDot(p,&op) == NULL)
    {
      ufdbLogFatalError( "%s: address error in %s line %d", progname, f, l );
      ip->mask = 0;
    }
    else 
      ip->mask = op;
  }

  if (type == SG_IPTYPE_CIDR)
  {
    p = ip->str;
    if (*p == '/')
      p++;
    octet = atoi( p );
    if (octet < 0 || octet > 32) 
    {
      ufdbLogFatalError( "%s: prefix error %s in %s line %d", progname, p, f, l );
      octet = 0;
    }
    if (octet == 32)
      ip->mask = 0xffffffff;
    else
      ip->mask = 0xffffffff ^ (0xffffffff >> octet);
    ip->net = ip->net & ip->mask;
  }

  ip->type = type;
  ip->next = (struct Ip *) ufdbCalloc( 1, sizeof(struct Ip) );

  /* TO-DO: fix messy code where the last struct must contain zeros */
}


void sgIp( 
  char *      name )
{
  struct Ip * ip;
  ulong       op;

  if (lastSource->ip == NULL) 
  {
    ip = (struct Ip *) ufdbCalloc( 1, sizeof(struct Ip) );
    ip->next = NULL;
    ip->str = NULL;
    lastSource->ip = ip;
    lastSource->lastip = ip;
  }
  else
  {
    ip = sgIpLast( lastSource );
  }

  if (ip->net_is_set == 0) 
  {
    ip->net_is_set = 1;
    if (sgConvDot(name,&op) == NULL) 
    {
      ufdbLogFatalError( "%s: address error in %s line %d", progname, configFile, lineno );
      ip->net = 0;
    }
    else 
      ip->net = op;
  }
  else
  {
    ip->str = (char *) ufdbMalloc( strlen(name) + 1 );
    strcpy( ip->str, name );
  }
}


struct Ip * sgIpLast( struct Source * s )
{
  struct Ip * ip, *ret;

  ret = NULL;
  for (ip = s->ip; ip != NULL; ip = ip->next)
    ret = ip;

  return ret;
}


/*
 * ACL functions
 */


void sgAcl( 
  char *          name,
  char *          value, 
  int             within )
{
  struct Acl *    acl;
  struct Source * source = NULL;
  struct Time *   time = NULL;
  int             def;
  char *          s;

#if UFDB_DEBUG
  ufdbLogMessage( "sgAcl name=%s value=%s within=%d", name==NULL?"NULL":name,  value==NULL?"NULL":value, within );
#endif

  def = 0;
  if (name != NULL)
  {
    /* due to some strange things in the yacc code */
    if ((s=(char *) strchr(name,' ')) != NULL)
      *s = '\0';    
    if ((s=(char *) strchr(name,'\t')) != NULL)
      *s = '\0';    
					/* TO-DO: find out why commented out */
#if UFDB_DEBUG
    if (Acl != NULL) 
    {
      if (sgAclFindName(name) != NULL)
      {
	ufdbLogFatalError( "%s: ACL \"%s\" is already defined in configuration file %s",
			   progname, name, configFile );
      }
    }
#endif
  }

  if (lastAcl != NULL  &&  name == NULL  &&  within == ELSE) 
    name = lastAcl->name;
  acl = (struct Acl *) ufdbMalloc( sizeof(struct Acl) );

  if (strcmp(name,"default") == 0)
  {
    defaultAcl = acl;
    def++;
  }
  else
  {
    if ((source = sgSourceFindName(name)) == NULL && !def) 
    {
      ufdbLogFatalError( "%s: ACL source \"%s\" is not defined in configuration file %s",
		         progname, name, configFile );
      return;
    }
  }

  acl->name = ufdbMalloc( strlen(name) + 1 );
  strcpy( acl->name, name );
  acl->active = within == ELSE ? 0 : 1;
  acl->source = source;
  acl->pass = NULL;
  acl->rewriteDefault = 1;
  acl->rewrite = NULL;
  acl->redirect = NULL;
  acl->time = NULL;
  acl->within = within;
  acl->next = NULL;
  if (value != NULL) 
  {
    if ((time = sgTimeFindName(value)) == NULL) 
    {
      ufdbLogFatalError( "%s: ACL time %s is not defined in configuration file %s",
		       progname, value, configFile );
      return;
    }
    acl->time = time;
  }

  if (Acl == NULL) {
    Acl = acl;
    lastAcl = acl;
  } else {
    lastAcl->next = acl;
    lastAcl = acl;
  }
}


void sgAclSetValue( 
  char *               what,
  char *               value, 
  int                  allowed ) 
{
  struct Category *    dest = NULL;
  struct sgRewrite *   rewrite = NULL;
  struct AclDest *     acldest;
  int                  type;
  
  type = ACL_TYPE_TERMINATOR;

  if (strcmp(what,"pass") == 0)
  {
    if (strcmp(value,"any")==0 || strcmp(value,"all")==0)
      allowed = 1;
    else if (strcmp(value,"none") == 0)
      allowed = 0;
    else if (strcmp(value,"in-addr") == 0) {
      type = ACL_TYPE_INADDR;
    }
    else
    {
      if ((dest = ufdbCategoryFindName(value)) == NULL) 
      {
	ufdbLogFatalError( "%s: ACL category %s is not defined in configuration file %s",
			   progname, value, configFile );
        return;
      } 
      type = ACL_TYPE_DEFAULT;
    }

    acldest = ufdbMalloc( sizeof(struct AclDest) );
    acldest->name = (char *) ufdbMalloc( strlen(value) + 1 );
    strcpy( acldest->name, value );
    acldest->dest = dest;
    acldest->access = allowed;
    acldest->type = type;
    acldest->next = NULL;

    if (lastAcl->pass == NULL) {
      lastAcl->pass = acldest;
    } else {
      lastAclDest->next = acldest;
    }
    lastAclDest = acldest;
  }

  if (strcmp(what,"rewrite") == 0)
  {
    if (strcmp(value,"none") == 0)
    {
      lastAcl->rewriteDefault = 0;
      lastAcl->rewrite = NULL;
    }
    else
    {
      if ((rewrite = sgRewriteFindName(value)) == NULL) 
      {
	ufdbLogFatalError( "%s: Rewrite %s is not defined in configuration file %s",
			   progname, value, configFile );
      }
      lastAcl->rewriteDefault = 0;
      lastAcl->rewrite = rewrite;
    }
  }

  if (strcmp(what,"redirect") == 0)
  {
    if (strcmp(value,"default") != 0)
    {
      lastAcl->redirect = (char *) ufdbMalloc( strlen(value) + 1 );
      strcpy( lastAcl->redirect, value );
    } else {
      lastAcl->redirect = NULL;
    }
  }
}


struct Acl * sgAclFindName( 
  char *       name )
{
  struct Acl * p;

  if (name == NULL)
     return NULL;

  for (p=Acl; p != NULL; p = p->next)
  {
    if (strcmp(name,p->name) == 0)
      return p;
  }
  return NULL;
}



/*
 * called by main loop: find the ACL that matches a source
 */

struct Acl * sgAclCheckSource( 
   struct Source * source )
{
   struct Acl *    acl = defaultAcl;
   int             found = 0;

   if (source != NULL) 
   {
     for (acl = Acl; acl != NULL; acl = acl->next)
     {
       if (acl->source == source)
       {
	  if (acl->active) 
	  {
	    found++;
	    break;
	  }
	  else
	  {
	    if (acl->next->source == source  &&  acl->next->active != 0)
	    {
	      found++;
	      acl = acl->next;
	      break;
	    }
	  }
        }
      }
   }

   if (!found)
     acl = defaultAcl;

   return acl;
}


/*
 * called by main loop: scan the full access matrix for the URL in parameter 'req'
 * return NULL if access is granted, otherwise a redirection URL string.
 */
char * sgAclAccess( 
  struct Source *    src, 
  struct Acl *       acl, 
  struct SquidInfo * req, 
  char *             buffer )
{
  int    access;
  int    result;
  char * redirect = NULL;
  char * dbdata = NULL;
  struct sgRewrite *    rewrite = NULL;
  struct AclDest *      aclpass = NULL;
  struct Category *     category;
  struct UFDBmemTable * mt;
  char   newstring[MAX_BUF];
  char   newredir[MAX_BUF + MAX_BUF];

  if (acl == NULL)
    return NULL;

  if (acl->pass == NULL)
    acl->pass = defaultAcl->pass;

  access = 1; 			/* let's start optimistic and grant access */
  if (acl->pass != NULL)
  {
    for (aclpass = acl->pass; aclpass != NULL; aclpass = aclpass->next)
    {
      category = aclpass->dest;
      if (category != NULL && !category->active)
	continue;

      if (aclpass->type == ACL_TYPE_TERMINATOR) 
      {
	access = aclpass->access;
	break;
      }

      if (aclpass->type == ACL_TYPE_INADDR) 
      {
	if (req->dot) {
	  access = aclpass->access;
	  break;
	}
	continue;
      }

#if 0
      /* be very careful not to touch table memory during a reconfigure */
      if (UFDBreconfig)
      {
	 *buffer = '\0';
	 return NULL;
      }
#endif

      /* domain name lookup for this source */
      if (category->domainlistDb != NULL) 
      {
        mt = (struct UFDBmemTable *) category->domainlistDb->dbcp;
        result = UFDBlookupRevUrl( &(mt->table.nextLevel[0]), req->revUrl );

	if (result) 
	{
	  if (aclpass->access) 
	  {
	     access++;
	     break; 
	  }
	  else 
	  {
	     access = 0;
	     break;
	  }
	}
      }

      if (req->dot &&
          (category->options & UFDB_OPT_HTTPS_WITH_HOSTNAME)  &&
          (req->port == 443 || req->port == 563 || strcmp( req->protocol, "https" ) == 0) )
      {
         /* hostname may not by an IP address */
	 ufdbLogError( "Security violation: https protocol used to access IP %s", req->domain );
	 access = 0;
	 break;
      }

      if ((category->options & UFDB_OPT_HTTPS_OFFICAL_CERTIFICATE)  &&
	  (UFDBgetTunnelCheckMethod() != UFDB_API_HTTPS_CHECK_OFF)  &&
          (req->port == 443 || req->port == 563 || strcmp( req->protocol, "https" ) == 0) )
      {
         int httpsStatus;

	 httpsStatus = UFDBcheckForHTTPStunnel( req->orig_domain, req->port,
	    UFDBgetTunnelCheckMethod() == UFDB_API_HTTPS_CHECK_AGRESSIVE ? 0 : UFDB_API_ALLOW_QUEUING );
	 if (httpsStatus == UFDB_API_ERR_INVALID_CERT)
	 {
	    access = 0;
	    break;
	 }
      }

      if ((UFDBgetTunnelCheckMethod() != UFDB_API_HTTPS_CHECK_OFF)  &&  
	  (strcmp( aclpass->name, "proxies" ) == 0)  &&
          (req->port == 443 || req->port == 563 || strcmp( req->protocol, "https" ) == 0) )
      {
         int httpsStatus;

	 httpsStatus = UFDBcheckForHTTPStunnel( req->orig_domain, req->port,
	    UFDBgetTunnelCheckMethod() == UFDB_API_HTTPS_CHECK_AGRESSIVE ? 0 : UFDB_API_ALLOW_QUEUING );
	 if (httpsStatus == UFDB_API_ERR_TUNNEL)
	 {
	    UFDBglobalTunnelCounter++;
	    if (UFDBgetTunnelCheckMethod() != UFDB_API_HTTPS_LOG_ONLY)
	    {
	       access = 0;
	       break;
	    }
	 }
      }

      /* expr lookup for this source */
      if (category->regExp != NULL  &&  access) 
      {
	result = ufdbRegExpMatch( category->regExp, req->url );
#if 0
	ufdbLogError( "ufdbRegExpMatch( %s  %s ) = %d", category->regExp->pattern, req->url, result );
#endif
	if (result) 
	{
	  if (aclpass->access) 
	  {
	     access++;
	     break;
	  } 
	  else 
	  {
	     access = 0;
	     break;
	  }
	}
      }

#ifdef UFDB_DOMAIN_IP_LOOKUP
      /* new: IP address lookup in the domains database */
      if (category->domainlistDb != NULL  &&  access) 
      {
	struct hostent thishost;
	int rv;

	rv = gethostbyname_r( req->domain );	/* NOT USED because potentially blocking !!! */
	if (rv != 0)
	{
	   /* future: do it for that whole array h_addr_list[] */
	   result = UFDBmemDBfind( category->domainlistDb, thishost->h_addr_list[0], &dbdata );
	   if (result) 
	   {
	     if (aclpass->access) 
	     {
	       access++;
	       break; 
	     } else {
	       access = 0;
	       break;
	     }
	   }
        }
      }
#endif

    }

    if (!access) 
    {
      if (dbdata != NULL)
	redirect = dbdata;
      else if (category != NULL && category->redirect != NULL)
	redirect = category->redirect;
      else if (category != NULL && category->rewrite != NULL &&
	      (redirect = 
	       sgRewriteExpression(category->rewrite,req->orig,newstring)) != NULL) 
      {
	;
      }
      else if (acl->redirect == NULL)
	redirect = defaultAcl->redirect;
      else
	redirect = acl->redirect;

      /* in case there is no redirect rule defined... */
      if (redirect == NULL)
      {
	 ufdbLogError( "!!!!! there is no \"redirect\" rule for ACL %s/%s  *****", 
	 	       acl->name, aclpass->name );
         redirect = "http://www.urlfilterdb.com/cgi-bin/URLblocked.cgi?"
	            "clientgroup=%s&targetgroup=%t&url=%u";
      }
    }
  }
  else
  {  
    /* acl->pass == NULL, probably defaultAcl->pass == NULL */
    access = 0;
    redirect = defaultAcl->redirect;
  }

  if (acl->rewrite == NULL)
    rewrite = defaultAcl->rewrite;
  else
    rewrite = acl->rewrite;

  if (rewrite != NULL && access)
  {
    redirect = sgRewriteExpression( rewrite, req->orig, newstring );
  }
  else if (redirect != NULL) 
  {
    redirect = sgParseRedirect( redirect, req, acl, aclpass, newredir, strcmp(req->protocol,"https")==0 );
  }

  if (redirect != NULL)
  {
     strcpy( buffer, redirect );
     req->aclpass = aclpass;
     return buffer;
  }

  *buffer = '\0';
  return NULL;
}


void yyerror( char * s )
{
   ufdbLogFatalError( "%s in configuration file %s line %d", s, configFile, lineno );
}


int yywrap()
{
  return 1;
}

