/*
 * ufdbchkport.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005,2006,2007 by URLfilterDB with all rights reserved.
 * ufdbGuard is based on squidGuard.
 *
 * RCS $Id: ufdbchkport.c,v 1.18 2007/11/20 16:34:04 root Exp root $
 */

#include "ufdb.h"
#include "ufdblib.h"
#include "sg.h"
#include "ufdbchkport.h"
#include "httpsQueue.h"

#include "hashtable.h"

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <pthread.h>
#include <signal.h>
#include <syslog.h>
#include <sys/socket.h>
#include <netinet/tcp.h>

#if defined(__FreeBSD__)
#include <sys/select.h>
#endif

#include <openssl/ssl.h>
#include <openssl/x509v3.h>
#include <openssl/err.h>


#if defined(UFDB_DO_DEBUG) || 0
#define DEBUG(x) fprintf x 
#else
#define DEBUG(x) 
#endif


int httpsChecks = UFDB_API_HTTPS_CHECK_OFF;

static pthread_mutex_t hashtable_mutex = UFDB_STATIC_MUTEX_INIT;
static int num_static_ssl_locks = 0;
static pthread_mutex_t * crypto_mutexes = NULL;
static SSL_CTX * ssl_ctx = NULL;
static struct hashtable * myht = NULL;
static char cacerts[1024];
static int ufdbCacertsLoaded = 0;

static int  lookupHTTPScache( char * hostname, int portnumber );
static int  UFDBverifyPortHasHTTPS( char * hostname, int portnumber, int flags );
static void addHTTPScache( char * hostname, int portnumber, int status );

static int ssl_connect( int fd, SSL ** ssl );
static int ssl_check_certificate( SSL * ssl, const char * hostname );
static int openssl_write( int fd, char * buf, int bufsize, SSL * ssl );
static int openssl_read( int fd, char * buf, int bufsize, SSL * ssl );
static void openssl_close( int fd, SSL * ssl );


#ifdef UFDB_DEBUG

static pthread_mutex_t https_mutex = UFDB_STATIC_MUTEX_INIT;

#define SSL_MUTEX_FN_INIT 	int _mutex_retval;
#define SSL_MUTEX_LOCK(fn) \
{                                                                \
   _mutex_retval = pthread_mutex_unlock( &https_mutex );         \
   if (_mutex_retval != 0)                                       \
      ufdbLogError( fn ": mutex_unlock failed with code %d", _mutex_retval );  \
}
#define SSL_MUTEX_UNLOCK(fn) \
{                                                                \
   _mutex_retval = pthread_mutex_unlock( &https_mutex );         \
   if (_mutex_retval != 0)                                       \
      ufdbLogError( fn ": mutex_unlock failed with code %d", _mutex_retval );  \
}

#else
#define SSL_MUTEX_FN_INIT
#define SSL_MUTEX_LOCK(fn)
#define SSL_MUTEX_UNLOCK(fn)
#endif


int UFDBloadAllowedHTTPSsites( char * filename )
{
   return UFDB_API_OK;
}


static unsigned int hostname2hash( void * key )
{
   int n;
   unsigned int hash;
   char * hostname = (char *) key;

   hash = 0;
   n = 15;
   while (n > 0  &&  *hostname != '\0')
   {
      hash = (hash << 2) + ((unsigned int) (*hostname));
      hash ^= hash << 11;
      hostname++;
      n--;
   }

   return hash;
}


static int mystrcmp( void * a, void * b )
{
   return strcmp( (char *) a, (char *) b ) == 0;
}


void UFDBsetTunnelCheckMethod( int method )
{
   httpsChecks = method;
}


static void ufdb_pthread_locking_callback( int mode, int type, const char * file, int line )
{
   int ret;

#if 0
   if (UFDBglobalDebug)
      ufdbLogMessage( "      CRYPTO %6s %2d", (mode & CRYPTO_LOCK) ? "lock" : "unlock", type );
#endif

   if (type < 0  || type > num_static_ssl_locks)
   {
      ufdbLogMessage( "CRYPTO type %d out of range (max=%d)  *****", type, num_static_ssl_locks );
      return;
   }

   if (mode & CRYPTO_LOCK)
   {
#if 0
      if (type == CRYPTO_LOCK_MALLOC || type == CRYPTO_LOCK_MALLOC2)
         ufdbGetMallocMutex( "ufdb_pthread_locking_callback" );
#endif
      ret = pthread_mutex_lock( &crypto_mutexes[type] );
#ifdef UFDB_DEBUG
      if (ret != 0)
	 ufdbLogError( "ufdb_pthread_locking_callback: mutex_lock[%d] failed with code %d", type, ret );
#endif
   }
   else if (mode & CRYPTO_UNLOCK)
   {
#if 0
      if (type == CRYPTO_LOCK_MALLOC || type == CRYPTO_LOCK_MALLOC2)
         ufdbReleaseMallocMutex( "ufdb_pthread_locking_callback" );
#endif
      ret = pthread_mutex_unlock( &crypto_mutexes[type] );
#ifdef UFDB_DEBUG
      if (ret != 0)
	 ufdbLogError( "ufdb_pthread_locking_callback: mutex_unlock[%d] failed with code %d", type, ret );
#endif
   }
   else
      ufdbLogError( "ufdb_pthread_locking_callback: no LOCK|UNLOCK for type %d", type );
}


static unsigned long ufdb_pthread_id_callback( void )
{
   unsigned long id = (unsigned long) pthread_self();

   return id;
}


int UFDBinitHTTPSchecker( void )
{
   int                    ret;
   static int             inited = 0;
   static pthread_mutex_t init_mutex = UFDB_STATIC_MUTEX_INIT;
   SSL_MUTEX_FN_INIT 	

   ret = pthread_mutex_lock( &init_mutex );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "UFDBinitHTTPSchecker: mutex_lock failed for init with code %d", ret );
#endif

   if (!inited)
   {
      int    i;
      char * dbdirectory;

      inited = 1;

      SSL_MUTEX_LOCK( "UFDBinitHTTPSchecker" )

      ret = pthread_mutex_lock( &hashtable_mutex );
#ifdef UFDB_DEBUG
      if (ret != 0)
	 ufdbLogError( "UFDBinitHTTPSchecker: mutex_lock for hashtable failed with code %d", ret );
#endif
      myht = create_hashtable( 800, hostname2hash, mystrcmp );
      ret = pthread_mutex_unlock( &hashtable_mutex );
#ifdef UFDB_DEBUG
      if (ret != 0)
	 ufdbLogError( "UFDBinitHTTPSchecker: mutex_unlock for hashtable failed with code %d", ret );
#endif

      srandom( ((getpid() << 8) ^ time(NULL)) + ((unsigned long)myht << 26) );

      ufdbGetMallocMutex( "UFDBinitHTTPSchecker 1" );
      SSL_load_error_strings();
      SSL_library_init();
      ufdbReleaseMallocMutex( "UFDBinitHTTPSchecker 1" );

      CRYPTO_set_id_callback( ufdb_pthread_id_callback );
      num_static_ssl_locks = CRYPTO_num_locks();
      crypto_mutexes = ufdbMalloc( (num_static_ssl_locks+1) * sizeof(pthread_mutex_t) );
      for (i = 0; i <= num_static_ssl_locks; i++)
         pthread_mutex_init( &crypto_mutexes[i], NULL );
      CRYPTO_set_locking_callback( ufdb_pthread_locking_callback );

      ufdbGetMallocMutex( "UFDBinitHTTPSchecker 2" );
      ssl_ctx = SSL_CTX_new( SSLv23_client_method() );
      ufdbReleaseMallocMutex( "UFDBinitHTTPSchecker 2" );
      if (ssl_ctx == NULL)
      {
         SSL_MUTEX_UNLOCK( "UFDBinitHTTPSchecker" )
	 errno = EINVAL;
         return UFDB_API_ERR_ERRNO;
      }

      ufdbGetMallocMutex( "UFDBinitHTTPSchecker 3" );
      SSL_CTX_set_options( ssl_ctx, SSL_OP_ALL );
      SSL_CTX_set_session_cache_mode( ssl_ctx, SSL_SESS_CACHE_OFF );
      SSL_CTX_set_timeout( ssl_ctx, 150 );
      SSL_CTX_set_default_verify_paths( ssl_ctx );
      ufdbReleaseMallocMutex( "UFDBinitHTTPSchecker 3" );
      
      dbdirectory = ufdbSettingGetValue( "dbhome" );
      if (dbdirectory == NULL)
         dbdirectory = DEFAULT_DBHOME;
      strcpy( cacerts, dbdirectory );
      strcat( cacerts, "/security/cacerts" );
      ufdbGetMallocMutex( "UFDBinitHTTPSchecker 4" );
      ufdbCacertsLoaded = SSL_CTX_load_verify_locations( ssl_ctx, cacerts, NULL ); 
      ufdbReleaseMallocMutex( "UFDBinitHTTPSchecker 4" );
      if (ufdbCacertsLoaded != 1)
         ufdbLogError( "SSL_CTX_load_verify_locations failed. cannot verify CA authorities *****" );

      /* use SSL_VERIFY_NONE and verify the certificate in ssl_check_certificate */
      SSL_CTX_set_verify( ssl_ctx, SSL_VERIFY_NONE, NULL );
#if 0
      SSL_CTX_set_mode( ssl_ctx, SSL_MODE_ENABLE_PARTIAL_WRITE );
#endif
      SSL_CTX_set_mode( ssl_ctx, SSL_MODE_AUTO_RETRY );

      SSL_MUTEX_UNLOCK( "UFDBinitHTTPSchecker" )

      ufdbLogMessage( "https/SSL verification initializer is finished" );
   }

   ret = pthread_mutex_unlock( &init_mutex );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "UFDBinitHTTPSchecker: mutex_unlock failed for init with code %d", ret );
#endif

   return UFDB_API_OK;
}


static char * httpsGETroot( int s, char * hostname, int portnumber, int * status )
{
   SSL_MUTEX_FN_INIT
   SSL *       ssl;
   int         n;
   char        request[2048];
   char        reply[4096+4];

   /*******************************
   GET / HTTP/1.0
   User-Agent: Wget/1.8.2
   Host: www.urlfilterdb.com:9443
   Accept: * / *
   Connection: Keep-Alive
   ********************************/

   if (ssl_ctx == NULL)
   {
      *status = UFDB_API_ERR_NULL;
      ufdbLogError( "httpsGETroot: ssl_ctx is NULL" );
      return NULL;
   }

   SSL_MUTEX_LOCK( "httpsGETroot" )

   n = ssl_connect( s, &ssl );
   if (n != UFDB_API_OK)
   {
      SSL_MUTEX_UNLOCK( "httpsGETroot" )
      ufdbLogError( "SSL connect failure to %s:%d fd=%d", hostname, portnumber, s );
      *status = n;
      return NULL;
   }
   if (UFDBglobalDebug)
      ufdbLogMessage( "SSL connect to %s:%d OK", hostname, portnumber );

   n = ssl_check_certificate( ssl, hostname );

   SSL_MUTEX_UNLOCK( "httpsGETroot" )

   if (n != UFDB_API_OK)
   {
      *status = UFDB_API_ERR_INVALID_CERT;
      openssl_close( s, ssl );
      return NULL;
   }

   *status = UFDB_API_OK;

   /* TO-DO: if there is a proxy, "CONNECT" must be used */
   sprintf( request, "GET / HTTP/1.0\r\n"
                     "User-Agent: Mozilla/4.0 "
		        "(compatible; MSIE 6.0; Windows NT 5.0; .NET CLR 1.0.3705)\r\n"
		     "Host: %s:%d\r\n"
		     "Accept: */*\r\n"
		     "Connection: Close\r\n"
		     "\r\n",
		     hostname, portnumber );
   n = openssl_write( s, request, strlen(request), ssl );
   if (n < 0)
   {
      *status = UFDB_API_ERR_SOCKET;
      strcpy( reply, "failed to send SSL message" );
   }
   else
   {
      reply[0] = '\0';
      n = openssl_read( s, reply, sizeof(reply)-4, ssl );
      if (n > 0)
      {
	 reply[n] = '\0';
#if 0
	 reply[n+1] = '\0';	/* to make valgrind happy */
	 reply[n+2] = '\0';
	 reply[n+3] = '\0';
#endif
      }
      else
         *status = UFDB_API_ERR_SOCKET;
   }
   openssl_close( s, ssl );

   return ufdbStrdup( reply );
}


/* Check for a tunnel.
 *
 * Valid flags are:
 * UFDB_API_ALLOW_QUEUING
 * UFDB_API_VERBOSE_OUTPUT
 *
 * return values are:
 * UFDB_API_OK:              regular https traffic
 * UFDB_API_REQ_QUEUED:      request is queued for an other thread
 * UFDB_API_ERR_TUNNEL:      https channel is tunneled
 * UFDB_API_ERR_CERTIFICATE: SSL certificate is self-signed or has wrong common name
 * UFDB_API_ERR_IP_ADDRESS:  hostname is an IP address (only when flag UFDB_API_NO_IP is used)
 * UFDB_API_ERR_NULL:        tunnel verification is OFF.
 */
int UFDBcheckForHTTPStunnel( char * hostname, int portnumber, int flags )
{
   if (httpsChecks == UFDB_API_HTTPS_CHECK_OFF)
      return UFDB_API_ERR_NULL;

   if (flags & UFDB_API_ALLOW_QUEUING)
   {
      int status;

      status = lookupHTTPScache( hostname, portnumber );
      if (status == UFDB_API_ERR_NULL)
      {
	 int                ret;
	 struct httpsInfo * info;
	 char               key[1024+16];

	 sprintf( key, "%s:%d", hostname, portnumber );
	 info = ufdbMalloc( sizeof(struct httpsInfo) );
	 info->t = time( NULL );
	 info->status = UFDB_API_REQ_QUEUED;

	 ufdbHttpsQueueRequest( hostname, portnumber );

	 ret = pthread_mutex_lock( &hashtable_mutex );
#ifdef UFDB_DEBUG
	 if (ret != 0)
	    ufdbLogError( "UFDBcheckForHTTPStunnel: mutex_lock failed with code %d", ret );
#endif
	 hashtable_insert( myht, (void *) ufdbStrdup(key), (void *) info );
	 ret = pthread_mutex_unlock( &hashtable_mutex );
#ifdef UFDB_DEBUG
	 if (ret != 0)
	    ufdbLogError( "UFDBcheckForHTTPStunnel: mutex_unlock failed with code %d", ret );
#endif

	 return UFDB_API_REQ_QUEUED;
      }
      return status;
   }

   /* TODO: when agressive lookups, also add hostname:portnumber to the hash NOW */
   return UFDBverifyPortHasHTTPS( hostname, portnumber, flags );
}


/*
 * in case that the caller of UFDBcheckForHTTPStunnel uses the UFDB_API_ALLOW_QUEUING flag,
 * the caller MUST have created 1-8 threads with UFDBhttpsTunnelVerifier() as main.
 */
void * UFDBhttpsTunnelVerifier( void * ptr )
{
   sigset_t  signals;
   int       portnumber;
   char      hostname[1028];

   /* 
    * The HUP signal must be blocked.
    */
   sigemptyset( &signals );
   sigaddset( &signals, SIGHUP );
   pthread_sigmask( SIG_BLOCK, &signals, NULL );

   ufdbSetThreadCPUaffinity( (int) ptr );

   UFDBinitHTTPSchecker();

   while (1)
   {
      ufdbGetHttpsRequest( hostname, &portnumber );

      if (UFDBglobalDebug)
	 ufdbLogMessage( "T%02ld: UFDBhttpsTunnelVerifier: start SSL verification %s:%d ...", ((long) ptr), hostname, portnumber );
      (void) UFDBverifyPortHasHTTPS( hostname, portnumber, 0 );
   }

   /*NOTREACHED*/
   return NULL;
}


/* TODO: hostnameIsIP() check can be done immediately, it does not have be be done in a queue */
static int hostnameIsIP( const char * hostname )
{
   while (*hostname != '\0')
   {
      if (*hostname != '.'  &&  !isdigit(*hostname))
         return 0;
      hostname++;
   }

   return 1;
}


/* Check that a port uses the https protocol.
 *
 * NOTE: This function may take 1-20 seconds to complete !
 */
static int UFDBverifyPortHasHTTPS( char * hostname, int portnumber, int flags )
{
   /* Hmmm. challenging task.
    * For now, we check that the port does not reply with 
    * - a SSH string indicating sshd and the use of proxytunnel
    */
   int            s;
   int            status;
   char *         content;
#if defined(UFDB_INITIAL_SOCKET_CHECKS)
   struct timeval tv;
   int            n;
   char           line[132];
#endif

   if (UFDBglobalDebug  ||  flags & UFDB_API_VERBOSE_OUTPUT)
      ufdbLogMessage( "UFDBverifyPortHasHTTPS( %s, %d )", hostname, portnumber );

   status = lookupHTTPScache( hostname, portnumber );
   if (status != UFDB_API_ERR_NULL  &&  status != UFDB_API_REQ_QUEUED)
      return status;

   if (UFDBglobalHttpsWithHostname && hostnameIsIP(hostname))
   {
      ufdbLogError( "invalid certificate: IP address is not allowed for the https protocol. IP=%s  *****", hostname );
      status = UFDB_API_ERR_INVALID_CERT;
      addHTTPScache( hostname, portnumber, status );
      return status;
   }

   s = UFDBopenSocket( hostname, portnumber );
   if (s < 0)
   {
      ufdbLogError( "https protocol verification for %s:%d failed: cannot open communication socket",
                     hostname, portnumber );
      addHTTPScache( hostname, portnumber, UFDB_API_ERR_SOCKET );
      return UFDB_API_ERR_SOCKET;
   }

   if (UFDBglobalDebug  ||  flags & UFDB_API_VERBOSE_OUTPUT)
      ufdbLogMessage( "UFDBverifyPortHasHTTPS: socket to %s is opened successfully. fd=%d", hostname, s );

   content = NULL;
   status = UFDB_API_OK;

#if defined(UFDB_INITIAL_SOCKET_CHECKS)

   /* set the timeout on the socket: we wait for a few seconds to find out if it sends "SSH-*" */
   tv.tv_sec = 3;
   tv.tv_usec = 0;
   n = setsockopt( s, SOL_SOCKET, SO_RCVTIMEO, (void *) &tv, sizeof(tv) );
   if (n < 0)
      ufdbLogError( "UFDBverifyPortHasHTTPS: setsockopt for %d second timeout failed: %s", tv.tv_sec, strerror(errno) );

ufdbLogMessage( "UFDBverifyPortHasHTTPS: socket opened for %s:%d and timeout set nsec=%d fd=%d", hostname, portnumber, tv.tv_sec, s );

   n = read( s, line, 8 );
   if (n > 0)
   {
      if (n >= 3  && strncmp( line, "SSH", 3 ) == 0)
      {
         ufdbLogError( "TUNNEL https protocol on %s:%d uses an SSH proxy tunnel", hostname, portnumber );
	 status = UFDB_API_ERR_TUNNEL;
      }
      else if (n >= 3  && strncmp( line, "RFB", 3 ) == 0) 
      {
         ufdbLogError( "TUNNEL https protocol on %s:%d uses a VNC proxy tunnel", hostname, portnumber );
	 status = UFDB_API_ERR_TUNNEL;
      }
      else
      {
	 int   i, j;
	 char  debuginfo[80];

	 /* very unlikely to be HTTPS since a normal HTTPS server does not start a handshake ! */
	 ufdbLogError( "TUNNEL https protocol on %s:%d does not encapsulate HTTP and "
	               "is considered a proxy tunnel",
	               hostname, portnumber );
	 if (n > 64)
	    n = 64;
	 for (j = 0, i = 0; i < n; i++)
	 {
	    if (line[i] == '\r'  ||  line[i] == '\n')
	       debuginfo[j++] = '_';
	    else if (isprint(line[i]))
	       debuginfo[j++] = line[i];
	    else
	       debuginfo[j++] = '.';
	 }
	 ufdbLogMessage( "   server started handshake with %s", debuginfo );
	 status = UFDB_API_ERR_TUNNEL;
      }
   }
#endif

   if (status == UFDB_API_OK)
   {
      /* TODO: check for vtunnel */

#if defined(UFDB_INITIAL_SOCKET_CHECKS)
      /* Increase the timeout: 2-4 seconds is not enough for SSL... */
      tv.tv_sec = 20;	/* TODO: experiment with this value */
      tv.tv_usec = 0;
      n = setsockopt( s, SOL_SOCKET, SO_RCVTIMEO, (void *) &tv, sizeof(tv) );
      if (n < 0)
         ufdbLogError( "UFDBverifyPortHasHTTPS: cannot set socket timeout to %d seconds: %s", tv.tv_sec, strerror(errno) );
#endif

      /* Setup a TLS connection to connect to the HTTPS port, do a "GET /", 
       * and see what the server has to say.
       */
      status = UFDB_API_OK;
      content = httpsGETroot( s, hostname, portnumber, &status );
      if (content == NULL  ||  *content == '\0')
      {
	 if (status == UFDB_API_OK)
	 {
	    /* We did not read anything from the server...  so we cannot draw a conclusion.
	     * Therefore we return "OK" and hope that the next check gives an answer.
	     */
	    ufdbLogError( "https server %s:%d is lame. close fd %d", hostname, portnumber, s );
	    close( s );
	    ufdbFree( content );
	    return UFDB_API_OK;
	 }
      }
      else
      if (status != UFDB_API_OK)
      {
         /* we have already an error condition; do no more checks */
	 ;
      }
      else
      if (strncmp( content, "HTTP/", 5 ) == 0)
      {
         if (UFDBglobalDebug)
	 {
	    int   i, j;
	    char  debuginfo[1080];
	    /* TODO: put this debug code in a seperate function */

	    for (j = 0, i = 0; content[i] != '\0' && i < 1024; i++)
	    {
	       if (content[i] == '\r'  ||  content[i] == '\n')
		  debuginfo[j++] = '_';
	       else if (isprint(content[i]))
		  debuginfo[j++] = content[i];
	       else
		  debuginfo[j++] = '.';
	    }
	    debuginfo[j] = '\0';
	    ufdbLogMessage( "   https protocol reply: %s", debuginfo );
	 }

	 /* TODO: investigate nomachine.com */
	 if (strstr( content, "Set-Cookie: SSLX_SSESHID=" ) != NULL)
	 {
	    ufdbLogError( "TUNNEL https protocol on %s:%d uses an SSL-Explorer tunnel", 
	                  hostname, portnumber );
	    status = UFDB_API_ERR_TUNNEL;
	 }
	 else
	 if (strstr( content, "BarracudaServer.com" ) != NULL   ||
	     strstr( content, "barracudaserver.com" ) != NULL   ||
	     strstr( content, "BarracudaDrive" ) != NULL)
	 {
	    ufdbLogError( "TUNNEL https protocol on %s:%d uses a BARRACUDA proxy tunnel",
	     		  hostname, portnumber );
	    status = UFDB_API_ERR_TUNNEL;
	 }
	 else
	 if (strstr( content, "  index.vnc -" ) != NULL   ||
	     strstr( content, "  used with Xvnc" ) != NULL   ||
	     strstr( content, "TightVNC Java viewer applet" ) != NULL)
	 {
	    ufdbLogError( "TUNNEL https protocol on %s:%d uses a VNC proxy tunnel",
	                  hostname, portnumber );
	    status = UFDB_API_ERR_TUNNEL;
	 }
      }
      else
      {
	 int   i,j;
	 char  debuginfo[80];

	 ufdbLogError( "TUNNEL https protocol on %s:%d encapsulates a non-HTTP protocol and "
		       "is considered a PROXY TUNNEL", 
		       hostname, portnumber );
	 /* TODO: put this code in a separate function */
	 for (j = 0, i = 0; content[i] != '\0' && i < 64; i++)
	 {
	    if (content[i] == '\r'  ||  content[i] == '\n')
	       debuginfo[j++] = '_';
	    else if (isprint(content[i]))
	       debuginfo[j++] = content[i];
	    else
	       debuginfo[j++] = '.';
	 }
	 debuginfo[j] = '\0';
	 ufdbLogMessage( "   https protocol reply: %s", debuginfo );

	 status = UFDB_API_ERR_TUNNEL;
      }
   }

   close( s );

   if (UFDBglobalDebug  ||  UFDBglobalLogAllRequests  ||  httpsChecks == UFDB_API_HTTPS_LOG_ONLY)
   {
      if (status == UFDB_API_OK)
	 ufdbLogMessage( "https protocol on %s:%d is checked", hostname, portnumber );
      if (httpsChecks == UFDB_API_HTTPS_LOG_ONLY)
	 status = UFDB_API_OK;
   }

   addHTTPScache( hostname, portnumber, status );

   ufdbFree( content );
   return status;
}


static int lookupHTTPScache( char * hostname, int portnumber )
{
   void * value;
   int    status;
   int    ret;
   time_t now;
   char   key[1024+16];

   sprintf( key, "%s:%d", hostname, portnumber );

   ret = pthread_mutex_lock( &hashtable_mutex );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "lookupHTTPScache: mutex_lock failed with code %d", ret );
#endif

   value = hashtable_search( myht, (void *) key );
   if (value == NULL)
   {
      ret = pthread_mutex_unlock( &hashtable_mutex );
#ifdef UFDB_DEBUG
      if (ret != 0)
	 ufdbLogError( "lookupHTTPScache: mutex_unlock failed with code %d", ret );
#endif
      return UFDB_API_ERR_NULL;
   }

   now = time( NULL );
   ((struct httpsInfo *) value)->t = now;
   status = ((struct httpsInfo *) value)->status;

   if (status == UFDB_API_ERR_SOCKET  &&
       ((struct httpsInfo *) value)->t < now - 600)
   {
      value = hashtable_remove( myht, key );
      ufdbFree( value );
      ret = pthread_mutex_unlock( &hashtable_mutex );
#ifdef UFDB_DEBUG
      if (ret != 0)
	 ufdbLogError( "lookupHTTPScache: mutex_unlock failed with code %d", ret );
#endif
      return UFDB_API_ERR_NULL;
   }

   ret = pthread_mutex_unlock( &hashtable_mutex );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "lookupHTTPScache: mutex_unlock failed with code %d", ret );
#endif
   return status;
}


static void addHTTPScache( char * hostname, int portnumber, int status )
{
   int                ret;
   struct httpsInfo * info;
   char               key[1024+16];

#if 0
   ufdbLogMessage( "   addHTTPScache  %s:%d %d", hostname, portnumber, status );
#endif

   sprintf( key, "%s:%d", hostname, portnumber );

   /* The hostname:portnumber is most likely already in the hashlist with status QUEUED.
    * So we only have to update the status...
    * Unless the cache was rebuilt, then we need to add the entry to the hash.
    */
   ret = pthread_mutex_lock( &hashtable_mutex );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "addHTTPScache: mutex_lock failed with code %d", ret );
#endif

   info = hashtable_search( myht, (void *) key );
   if (info == NULL)
   {
      sprintf( key, "%s:%d", hostname, portnumber );
      info = ufdbMalloc( sizeof(struct httpsInfo) );
      info->t = time( NULL );
      info->status = status;
      hashtable_insert( myht, (void *) ufdbStrdup(key), (void *) info );
   }
   else
   {
      info->t = time( NULL );
      info->status = status;
   }
   ret = pthread_mutex_unlock( &hashtable_mutex );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "addHTTPScache: mutex_unlock failed with code %d", ret );
#endif

   /* UFDB_API_ERR_SOCKET: cache this for 20 minutes only */
}


int select_fd( int fd, double maxtime, int wait_for )
{
   int      result;
   fd_set   fdset;
   fd_set * rd;
   fd_set * wr;
   struct timeval tmout;

   FD_ZERO( &fdset );
   FD_SET( fd, &fdset );
   rd = &fdset;
   wr = &fdset;

   tmout.tv_sec = (long) maxtime;
   tmout.tv_usec = 1000000 * (maxtime - (long) maxtime);

   errno = 0;
   do
      result = select( fd+1, rd, wr, NULL, &tmout );
   while (result < 0  &&  errno == EINTR);

   return result;
}


static void print_errors( void )
{
   unsigned long curerr = 0;
   int header = 0;

   ufdbGetMallocMutex( "print_errors" );
   while ((curerr = ERR_get_error()) != 0)
   {
      if (!header)
      {
	 ufdbLogError( "   https/SSL/tunnel verification failed with OpenSSL error:" );
	 header = 1;
      }
      ufdbLogError( "   %s", ERR_error_string(curerr,NULL) );
   }
   ufdbReleaseMallocMutex( "print_errors" );
}


static int openssl_read( int fd, char * buf, int bufsize, SSL * ssl )
{
   SSL_MUTEX_FN_INIT
   int  ret;

   SSL_MUTEX_LOCK( "openssl_read" )

   errno = 0;
   ret = -2;
   do
   {
      ret = SSL_read( ssl, buf, bufsize );
   }
   while (ret == -1  &&
	  SSL_get_error(ssl, ret) == SSL_ERROR_SYSCALL  &&
	  errno == EINTR);

   SSL_MUTEX_UNLOCK( "openssl_read" )

   return ret;
}


static int openssl_write( int fd, char * buf, int bufsize, SSL * ssl )
{
   SSL_MUTEX_FN_INIT
   int  ret;

   SSL_MUTEX_LOCK( "openssl_write" )

   errno = 0;
   do
   {
      ret = SSL_write( ssl, buf, bufsize );
   }
   while (ret == -1  &&
	  SSL_get_error(ssl,ret) == SSL_ERROR_SYSCALL  &&
	  errno == EINTR);

   SSL_MUTEX_UNLOCK( "openssl_write" )

   return ret;
}


static void openssl_close( int fd, SSL * ssl )
{
   SSL_MUTEX_FN_INIT

   SSL_MUTEX_LOCK( "openssl_close" )

   ufdbGetMallocMutex( "openssl_close" );
   if (SSL_shutdown( ssl ) == 0)
      SSL_shutdown( ssl );
   ERR_remove_state( 0 );
   SSL_clear( ssl );
   SSL_free( ssl );
   ufdbReleaseMallocMutex( "openssl_close" );

   SSL_MUTEX_UNLOCK( "openssl_close" )
}


/* Perform the SSL handshake on file descriptor FD, which is assumed
   to be connected to an SSL server.  The SSL handle provided by
   OpenSSL is registered with the file descriptor FD using
   fd_register_transport, so that subsequent calls to fd_read,
   fd_write, etc., will use the corresponding SSL functions.

   Returns UFDB_API_OK on success  */

static int ssl_connect( int fd, SSL ** ssl ) 
{
   int   ret;
   int   connect_status;
   long  state;

   if (ssl_ctx == NULL) 
   {
      ufdbLogError( "ssl_connect: ssl_ctx is NULL" );
      return UFDB_API_ERR_NULL;
   }

   connect_status = UFDB_API_OK;
   ERR_clear_error();

   ufdbGetMallocMutex( "openssl_close 1" );
   *ssl = SSL_new( ssl_ctx );
   ufdbReleaseMallocMutex( "openssl_close 1" );
   if (*ssl == NULL)
   {
      ufdbLogError( "ssl_connect: SSL_new failed fd=%d", fd );
      connect_status = UFDB_API_ERR_NULL;
      goto error;
   }

   ufdbGetMallocMutex( "openssl_close 2" );
   ret = SSL_set_fd( *ssl, fd );
   ufdbReleaseMallocMutex( "openssl_close 2" );
   if (ret == 0)
   {
      ufdbLogError( "ssl_connect: SSL_set_fd failed fd=%d", fd );
      connect_status = UFDB_API_ERR_NULL;
      goto error;
   }

   errno = 0;
   ERR_remove_state( 0 );
   ufdbGetMallocMutex( "openssl_close 3" );
   SSL_set_connect_state( *ssl );
   ret = SSL_connect( *ssl );
   state = (*ssl)->state;
   ufdbReleaseMallocMutex( "openssl_close 3" );
   if (ret <= 0  ||  state != SSL_ST_OK)
   {
      int SSL_error;

      connect_status = UFDB_API_ERR_NULL;
      SSL_error = SSL_get_error( *ssl, ret );
      ufdbLogError( "https/SSL connection: SSL_connect failed. ret=%d error=%d state=0x%08x", 
                    ret, SSL_error, state );
      if (SSL_error == SSL_ERROR_SYSCALL)
         ufdbLogMessage( "   system call by SSL produced error %d: %s", errno, strerror(errno) );
      else if (SSL_error == SSL_ERROR_SSL)
      {
	 ufdbLogMessage( "   SSL_connect detected a non-HTTP(s) protocol so it is a TUNNEL *****" );
         connect_status = UFDB_API_ERR_TUNNEL;
      }
      else
      {
	 ufdbLogMessage( "ssl_connect: SSL connection failed with SSL error %d", SSL_error );
      }
      goto error;
   }

   return UFDB_API_OK;

error:
   print_errors();

   if (*ssl != NULL)
   {
      ufdbGetMallocMutex( "openssl_close 4" );
      if (SSL_shutdown( *ssl ) == 0)
	 SSL_shutdown( *ssl );
      ERR_remove_state( 0 );
      SSL_clear( *ssl );
      SSL_free( *ssl );
      ufdbReleaseMallocMutex( "openssl_close 4" );
   }

   return connect_status;
}


/* Return 1 if hostname (case-insensitively) matches common, 0
   otherwise.  The recognized wildcard character is "*", which matches
   any character in common except ".".  

   This is used to match of hosts as indicated in rfc2818: "Names may
   contain the wildcard character * which is considered to match any
   single domain name component or component fragment. E.g., *.a.com
   matches foo.a.com but not bar.foo.a.com. f*.com matches foo.com but
   not bar.com [or foo.bar.com]."

   If the pattern contain no wildcards, matchHostname(a,b) is
   equivalent to !strcasecmp(a,b)
*/

static int matchHostname( const char * common, const char * hostname )
{
   const char * c = common;
   const char * h = hostname;
   char         ch;

#if 0
   if (UFDBglobalDebug)
      ufdbLogMessage( "      matchHostname( %s, %s )", common, hostname );
#endif

   for (; (ch = tolower(*c++)) != '\0'; h++)
   {
      if (ch == '*')
      {
	 for (ch = tolower(*c); ch == '*'; ch = tolower(*++c))
	    ;
	 for (; *h != '\0'; h++)
	 {
	    if (tolower(*h) == ch  &&  matchHostname(c,h))
	       return 1;
	    else if (*h == '.')
	       return 0;
	 }
	 return ch == '\0';
      }
      else
      {
	 if (ch != tolower(*h))
	    return 0;
      }
   }

   return (*h == '\0');
}


static int ssl_check_certificate( SSL * ssl, const char * hostname )
{
   const char * altPtr;
   STACK_OF(GENERAL_NAME) * altNames;
   int    altNameSeen;
   X509 * cert;
   long   vresult;
   int    success;
   int    match;
   char   issuer[1024];

   ufdbGetMallocMutex( "ssl_check_certificate 1" );
   cert = SSL_get_peer_certificate( ssl );
   ufdbReleaseMallocMutex( "ssl_check_certificate 1" );

   success = 1;
   if (cert == NULL)
   {
      success = 0;
      ufdbLogError( "host %s has no SSL certificate", hostname );
      goto no_cert;		/* must bail out since CERT is NULL */
   }

   issuer[0] = '\0';
   ufdbGetMallocMutex( "ssl_check_certificate 2" );
   (void) X509_NAME_oneline( X509_get_issuer_name(cert), issuer, 1023 );
   ufdbReleaseMallocMutex( "ssl_check_certificate 2" );
   issuer[1023] = '\0';
   if (issuer[0] == '\0')
   {
      strcpy( issuer, "--no-issuer-found-in-certificate--" );
      success = 0;
   }
   if (UFDBglobalDebug)
      ufdbLogMessage( "   SSL certificate for %s has issuer '%s'", hostname, issuer );

   ufdbGetMallocMutex( "ssl_check_certificate 3" );
   vresult = SSL_get_verify_result( ssl );
   ufdbReleaseMallocMutex( "ssl_check_certificate 3" );
   if (vresult == X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY)
   {
      if (ufdbCacertsLoaded)
      {
         ufdbLogError( "SSL certificate for %s: unrecognised issuer", hostname );
	 ufdbLogMessage( "   issuer: %s *****", issuer );
	 ufdbLogMessage( "   this issuer is not a recognised certificate authority" );
	 success = 0;
      }
      else
	 if (UFDBglobalDebug)
	 {
	    ufdbLogError( "SSL_get_verify_result() is X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY (ignored)" );
	    ufdbLogMessage( "Check that you have .../blacklists/security/cacerts" );
	 }
   }
   else if (vresult != X509_V_OK)
   {
      switch (vresult)
      {
      case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY:
	 ufdbLogError( "SSL certificate for %s: unrecognised issuer *****", hostname );
	 ufdbLogMessage( "   issuer: %s *****", issuer );
	 ufdbLogMessage( "   this issuer is not a recognised certificate authority" );
	 break;
      case X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN:
	 ufdbLogError( "SSL certificate for %s: self-signed certificate in chain *****", hostname );
	 ufdbLogMessage( "   issuer: %s *****", issuer );
	 break;
      case X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT:
	 ufdbLogError( "SSL certificate for %s: self-signed certificate *****", hostname );
	 ufdbLogMessage( "   issuer: %s *****", issuer );
	 ufdbLogMessage( "   the SSL certificate of the server is not signed by an authority" );
	 break;
      case X509_V_ERR_CERT_NOT_YET_VALID:
	 ufdbLogError( "SSL certificate for %s: certificate date is not yet valid *****", hostname );
	 break;
      case X509_V_ERR_CERT_HAS_EXPIRED:
	 ufdbLogError( "SSL certificate for %s: certificate has expired *****", hostname );
	 break;
      default:
	 ufdbLogError( "Certificate verification error for %s: %s *****",
		       hostname, X509_verify_cert_error_string(vresult) );
      }
      success = 0;
      /* Fall through */
   }

   if (success && UFDBglobalDebug)
      ufdbLogMessage( "   SSL certificate for `%s' is signed by a CA", hostname );

   /* 
    * Check that hostname matches the common name in the certificate.
    * rfc2818:
    * "If a subjectAltName extension of type dNSName is present, 
    * that MUST be used as the identity."
    * If not, we will use the commonName as the identity.
    */
   match = 0;
   
   altNameSeen = 0;

   altNames = X509_get_ext_d2i( cert, NID_subject_alt_name, NULL, NULL );
   if (altNames)
   {
      const GENERAL_NAME * check;
      int numAltNames;
      int i;

      numAltNames = sk_GENERAL_NAME_num( altNames );
      for (i = 0; i < numAltNames; i++)
      {
	 check = sk_GENERAL_NAME_value( altNames, i );
	 altPtr = (char *) ASN1_STRING_data( check->d.ia5 );
	 if (check->type == GEN_DNS)
	 {
	    altNameSeen = 1;
	    if (matchHostname( altPtr, hostname ))
	    {
	       match = 1;
	       break;
	    }
	 }
	 else if (check->type == GEN_IPADD)
	 {
	    altNameSeen = 1;
	    if (!UFDBglobalHttpsWithHostname)
	    {
	       int altLen = ASN1_STRING_length( check->d.ia5 );
	       if (strncmp( altPtr, hostname, altLen ) == 0)
	       {
	          match = 1;
		  break;
	       }
	    }
	 }
      }

      if (altNameSeen)
      {
	 if (!match)
	 {
	    ufdbLogError( "SSL certificate has subjectAltName which does not match hostname name %s", hostname );
	    success = 0;
	 }
	 else
	    if (UFDBglobalDebug)
	       ufdbLogMessage( "   certificate subjectAltName %s matches hostname %s", altPtr, hostname );
      }
      GENERAL_NAMES_free( altNames );
   }
   
   if (!altNameSeen)
   {
      int             i, j;
      X509_NAME *     name;
      unsigned char * nulstr = (unsigned char *) "";
      unsigned char * commonName = nulstr;

      if (UFDBglobalDebug)
	 ufdbLogMessage( "   SSL certificate has no subjectAltName" );

      i = -1;
      ufdbGetMallocMutex( "ssl_check_certificate 4" );
      name = X509_get_subject_name( cert );
      ufdbReleaseMallocMutex( "ssl_check_certificate 4" );
      if (name != NULL)
         while ((j=X509_NAME_get_index_by_NID(name,NID_commonName,i)) >= 0)
	    i = j;

      /* now we have the name entry and convert it to a string */
      if (i >= 0)
      {
         ASN1_STRING * tmp;

	 ufdbGetMallocMutex( "ssl_check_certificate 5" );
	 tmp = X509_NAME_ENTRY_get_data( X509_NAME_get_entry(name,i) );
	 ufdbReleaseMallocMutex( "ssl_check_certificate 5" );

	 /* In OpenSSL 0.9.7d and earlier ASN1_STRING_to_UTF8 fails if string is already UTF8 :-) */
#if defined(OPENSSL_VERSION_NUMBER)
#if OPENSSL_VERSION_NUMBER <= 0x0090704fL
	 if (UFDBglobalDebug)
	    ufdbLogMessage( "   OpenSSL library version is %08X", OPENSSL_VERSION_NUMBER );
	 if (tmp != NULL  &&  ASN1_STRING_type(tmp) == V_ASN1_UTF8STRING)
	 {
	    j = ASN1_STRING_length( tmp );
	    if (j >= 0)
	    {
	       commonName = ufdbMalloc( j+1 );
	       memcpy( commonName, ASN1_STRING_data(tmp), j );
	       commonName[j] = '\0';
	    }
	 }
	 else
	 {
	    ufdbGetMallocMutex( "ssl_check_certificate 6" );
	    j = ASN1_STRING_to_UTF8( &commonName, tmp );
	    ufdbReleaseMallocMutex( "ssl_check_certificate 6" );
	 }
#else
	 ufdbGetMallocMutex( "ssl_check_certificate 7" );
         j = ASN1_STRING_to_UTF8( &commonName, tmp );
	 ufdbReleaseMallocMutex( "ssl_check_certificate 7" );
#endif
#else
	 ufdbGetMallocMutex( "ssl_check_certificate 8" );
         j = ASN1_STRING_to_UTF8( &commonName, tmp );
	 ufdbReleaseMallocMutex( "ssl_check_certificate 8" );
#endif
      }

      if (commonName == nulstr)
	 commonName = NULL;
      if (commonName == NULL)
      {
	 ufdbLogError( "SSL certificate for `%s' has no common name *****", hostname );
	 success = 0;
      }
      else
      {
	 if ( !matchHostname( (char *) commonName, hostname ))
	 {
	    ufdbLogError( "SSL certificate common name `%s' doesn't match hostname `%s' *****",
			  commonName, hostname );
	    success = 0;
	 }
         ufdbFree( commonName );
      }
   }

   print_errors();

   if (success && UFDBglobalDebug)
      ufdbLogMessage( "SSL certificate successfully verified and matches hostname %s", hostname );

   ufdbGetMallocMutex( "ssl_check_certificate 9" );
   X509_free( cert );
   ufdbReleaseMallocMutex( "ssl_check_certificate 9" );

no_cert:
   return success ? UFDB_API_OK : UFDB_API_ERR_INVALID_CERT;
}

