/*
 * ufdblib.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005,2006,2007 by URLfilterDB with all rights reserved.
 *
 * RCS $Id: ufdblib.c,v 1.40 2007/11/26 15:31:03 root Exp root $
 */

/* to inline string functions with gcc : */
#if defined(__OPTIMIZE__) && defined(__GNUC__) && defined(GCC_INLINE_STRING_FUNCTIONS_ARE_FASTER)
#define __USE_STRING_INLINES
#endif

#include "ufdb.h"
#include "sg.h"
#include "ufdblib.h"

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <syslog.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#if HAVE_UNIX_SOCKETS
#include <sys/un.h>
#endif
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>

#include "bzlib.h"

#if defined(UFDB_DO_DEBUG) || 0
#define DEBUG(x) fprintf x 
#else
#define DEBUG(x) 
#endif

static pthread_mutex_t gethostbyname_mutex = UFDB_STATIC_MUTEX_INIT;


static unsigned char never[] =
{
   "UFDB 1.1 never 1 key=FGrH-PE5U-ljDA-8jVV date=20050724.1140 -------- \n\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0never\001never\020"
};


UFDBthreadAdmin * UFDBallocThreadAdmin( void )
{
   UFDBthreadAdmin * admin;

   admin = (UFDBthreadAdmin *) ufdbCalloc( 1, sizeof(UFDBthreadAdmin) );

   return admin;
}


/* Mutex locks for _allocRevURL() are a thread synchronisation bottleneck */
/* so it is better that each thread has its own array of UFDBrevURL* */

static __inline__ UFDBrevURL * _allocRevURL( 
   UFDBthreadAdmin * admin )
{
   int               i;
   UFDBrevURL *      new;
   
   for (i = 0; i < admin->nAlloced; i++)
   {
      if (admin->myArrayUsage[i] == 0)
      {
         admin->myArrayUsage[i] = 1;
	 new = &(admin->myArray[i]);
	 goto end;
      }
   }

   if (admin->nAlloced == MAX_REVURLS)
   {
      new = ufdbMalloc( sizeof(UFDBrevURL) );
   }
   else
   {
      admin->myArrayUsage[admin->nAlloced] = 1;
      new = &(admin->myArray[admin->nAlloced]);
      admin->nAlloced++;
   }

end:
   return new;
}


static __inline__ void _freeRevURL( 
   UFDBthreadAdmin * admin, 
   UFDBrevURL *      old )
{
   int               i;

   for (i = 0; i < admin->nAlloced; i++)
   {
      if (old == &(admin->myArray[i]))
      {
         admin->myArrayUsage[i] = 0;
	 return;
      }
   }

   ufdbFree( old );
}


static UFDBrevURL * parseURLrecursive( 
   UFDBthreadAdmin * admin,
   UFDBrevURL *      prevHead, 
   unsigned char *   lastPiece )
{
   char *            dot;
   UFDBrevURL *      newHead;

#if 0
   DEBUG(( stderr, "   parseURLrecursive  0x%08x 0x%08x %s\n", prevHead, lastPiece, lastPiece ));  /* */
#endif

   dot = rindex( (char *) lastPiece, '.' );
   if (dot == NULL)
   {
      newHead = _allocRevURL( admin );

      strcpy( (char *) newHead->part, (char *) lastPiece );
      newHead->next = prevHead;
   }
   else
   {
      *dot = '\0';
      newHead = _allocRevURL( admin );

      strcpy( (char *) newHead->part, dot+1 );
      newHead->next = parseURLrecursive( admin, prevHead, lastPiece );
   }

   return newHead;
}


/* Parse the URL and reverse it into a linked list 
 * e.g. my.sex.com becomes "com" -> "sex" -> "my" -> NULL
 *
 * The input parameter, URL, must be a writable array of characters (size = UFDBmaxURLsize).
 */
UFDBrevURL * UFDBgenRevURL( 
   UFDBthreadAdmin * admin,
   unsigned char *   URL )
{
   char *            slash;
   UFDBurlPart       URLpart;
   char              newStr[MAX_BUF];

   URL = (unsigned char *) strcpy( newStr, (char *) URL );

   URLpart[0] = '\0';
   slash = index( newStr, '/' );
   if (slash != NULL)
   {
      UFDBrevURL *      head;
      UFDBrevURL *      ptr;

      *slash = '\0';
      if (*(slash-1) == '.')		/* also match www.sex.com./index.html */
         *(slash-1) = '\0';
      slash++;

      head = parseURLrecursive( admin, NULL, URL );

      if (*slash != '\0')
      {
	 ptr = head;
	 while (ptr->next != NULL)
	    ptr = ptr->next;

	 ptr->next = _allocRevURL( admin );
	 ptr = ptr->next;
	 ptr->next = NULL;
	 ptr->part[0] = '/';
	 strncpy( (char *) ptr->part + 1, slash, sizeof(UFDBurlPart)-2 );
	 ptr->part[ sizeof(UFDBurlPart)-2 ] = '\0';
      }
      return head;
   }
   else
   {
      unsigned char * end;

      for (end = URL; *end != '\0'; end++)
         ;
      if (*(end-1) == '.')		/* also match www.sex.com. */
         *(end-1) = '\0';
      return parseURLrecursive( admin, NULL, URL );
   }
}


/* DEBUG: print a revURL to stderr
 */
void UFDBprintRevURL( UFDBrevURL * revURL )
{
   fputs( "   P ", stderr );
   while (revURL != NULL)
   {
      fputs( (char *) revURL->part, stderr );
      if (revURL->next != NULL)
         fputs( " . ", stderr );
      revURL = revURL->next;
   }
   fputs( " \n", stderr );
}


__inline__ void UFDBfreeRevURL( 
   UFDBthreadAdmin * admin,
   UFDBrevURL *      revURL )
{
   UFDBrevURL *      next;

   while (revURL != NULL)
   {
      next = revURL->next;
      _freeRevURL( admin, revURL );
      revURL = next;
   }
}


#define ROUNDUPBY      4
#define ROUNDUP(i)     ( (i) + (ROUNDUPBY - ((i)%ROUNDUPBY) ) )

#define BIGROUNDUPBY   (4 * ROUNDUPBY)
#define BIGROUNDUP(i)  ( (i) + (BIGROUNDUPBY - ((i)%BIGROUNDUPBY) ) )

__inline__ static struct UFDBtable * _reallocTableArray( 
   struct UFDBtable *  ptr,
   int                 nElem )
{
   if (nElem <= ROUNDUPBY)
   {
      ptr = ufdbRealloc( ptr, (size_t) nElem * sizeof(struct UFDBtable) );
   }
   else if (nElem < 4*BIGROUNDUPBY)
   {
      if (nElem % ROUNDUPBY == 1)
	 ptr = ufdbRealloc( ptr, (size_t) ROUNDUP(nElem) * sizeof(struct UFDBtable) );
   }
   else
   {
      if (nElem % BIGROUNDUPBY == 1)
	 ptr = ufdbRealloc( ptr, (size_t) BIGROUNDUP(nElem) * sizeof(struct UFDBtable) );
   }

   return ptr;
}


void UFDBfreeTableIndex( struct UFDBtable * t )
{
   int i;

   if (t == NULL)
      return;

   for (i = 0; i < t->nNextLevels; i++)
   {
      UFDBfreeTableIndex( &(t->nextLevel[i]) );
   }

   ufdbFree( &(t->nextLevel[0]) );
}


void ufdbDecompressTable( struct UFDBmemTable * memTable )
{
   char * new_mem;
   int    rv;

#if 0
   fprintf( stderr, "decompress table from %ld original size of %d\n", 
            memTable->tableSize, memTable->numEntries );
#endif

   new_mem = ufdbMalloc( sizeof(struct UFDBfileHeader) + memTable->numEntries );
   memcpy( new_mem, memTable->mem, sizeof(struct UFDBfileHeader) );

   rv = BZ2_bzBuffToBuffDecompress( new_mem + sizeof(struct UFDBfileHeader),
                                    (unsigned int *) &(memTable->numEntries),
			            (char *) memTable->mem + sizeof(struct UFDBfileHeader),
			            memTable->tableSize,
			            0,
			            0 );
   if (rv != BZ_OK)
   {
      fprintf( stderr, "ufdbDecompressTable: decompression failed with code %d\n", rv );
      exit( 1 );
   }

   ufdbFree( memTable->mem );
   memTable->mem = new_mem;
}


/* parse a binary table header from a memory table.
 */
void UFDBparseTableHeader( struct UFDBmemTable * memTable )
{
   int  error = 0;
   int  n;
   char prefix[32];
   char tableName[32];
   char key[32];
   char date[32];
   char flags[8+1];
   ufdbCrypt uc;
   unsigned char * mem;

   strcpy( date, "nodate" );
   strcpy( flags, "--------" );
   n = sscanf( memTable->mem, "%5s %7s %20s %11d key=%30s date=%20s %8s", 
               prefix, &(memTable->version[0]), tableName, &(memTable->numEntries), 
	       key, date, flags );
   if (n < 5  ||  strcmp(prefix,"UFDB") != 0)
   {
      fprintf( stderr, "invalid UFDB header\n" );
      fprintf( stderr, "contact support@urlfilterdb.com\n" );
      exit( 1 );
   }

   if (strcmp( memTable->version, UFDBdbVersion ) > 0)
   {
      fprintf( stderr, "UFDB file for table \"%s\" has data format version \"%s\" while "
      		       "this program\n"
                       "does not support versions higher than \"%s\"\n", 
		       tableName, memTable->version, UFDBdbVersion );
      fprintf( stderr, "Download/install a new version of this program.\n" );
      fprintf( stderr, "Go to http://www.urlfilterdb.com\n" );
      exit( 1 );
   }

   if (strlen( key ) < 19)
   {
      fprintf( stderr, "UFDB file for table \"%s\" has an invalid key\n", tableName );
      fprintf( stderr, "contact support@urlfilterdb.com\n" );
      exit( 1 );
   }
   memTable->key[0] = key[0];
   memTable->key[1] = key[1];
   memTable->key[2] = key[2];
   memTable->key[3] = key[3];
   /* skip '-' */
   memTable->key[4] = key[5];
   memTable->key[5] = key[6];
   memTable->key[6] = key[7];
   memTable->key[7] = key[8];
   /* skip '-' */
   memTable->key[8] = key[10];
   memTable->key[9] = key[11];
   memTable->key[10] = key[12];
   memTable->key[11] = key[13];
   /* skip '-' */
   memTable->key[12] = key[15];
   memTable->key[13] = key[16];
   memTable->key[14] = key[17];
   memTable->key[15] = key[18];

   strncpy( memTable->flags, flags, 8 );

   if (strcmp( date, "nodate" ) != 0)
   {
      struct tm   tm;

      if (5 != sscanf( date, "%4d%2d%2d.%2d%2d", 
                       &tm.tm_year, &tm.tm_mon, &tm.tm_mday, &tm.tm_hour, &tm.tm_min ))
      {
         fprintf( stderr, "table %s has an invalid date (date=%13.13s)\n", tableName, date );
         fprintf( stderr, "contact support@urlfilterdb.com\n" );
	 syslog( LOG_ALERT, "table %s has an invalid date", tableName );
	 error = 1;
      }
      else
      {
	 time_t      t_now;
	 time_t      t_db;
	 time_t      diff;

	 tm.tm_year -= 1900;
	 tm.tm_mon  -= 1;
	 tm.tm_isdst = 0;
	 tm.tm_sec   = 0;

	 t_db = mktime( &tm );
	 t_now = time( NULL );
	 diff = t_now - t_db;
	 if (diff < -24 * 60 * 60)	/* allow 1 day back for various time zones */
	 {
	    fprintf( stderr, "table %s has an invalid date (%13.13s)\n", tableName, date );
	    fprintf( stderr, "the difference between current system time and database timestamp is %ld seconds\n", diff );
            fprintf( stderr, "contact support@urlfilterdb.com\n" );
	    syslog( LOG_ALERT, "table %s has an invalid date", tableName );
	    error = 1;
	 }
	 else if (diff > 61 * 24 * 60 * 60  &&  flags[1] == 'P')
	 {
	    fprintf( stderr, "table %s is dated %13.13s and is too old\n", tableName, date );
            fprintf( stderr, "contact support@urlfilterdb.com\n" );
	    syslog( LOG_ALERT, "table %s is too old", tableName );
	    error = 1;
	 }
#if 0
	 fprintf( stderr, "t_db  %12ld\n", t_db );
	 fprintf( stderr, "t_now %12ld\n", t_now );
	 fprintf( stderr, "diff  %12ld\n", t_now - t_db );
#endif
      }
   }

   if (error)
   {
      memTable->mem = never;
      memTable->tableSize = 113;
   }
   else
   {
      if (flags[2] == 'Q'  ||  strcmp( memTable->version, "1.2" ) >= 0)
      {
	 mem = (unsigned char *) memTable->mem + sizeof( struct UFDBfileHeader );
	 ufdbCryptInit( &uc, (unsigned char *) memTable->key, 16 );
	 ufdbEncryptText( &uc, mem, mem, memTable->tableSize );
      }

      if (memTable->flags[0] == 'C')
      {
	 ufdbDecompressTable( memTable );
      }
   }
}


#include "strcmpurlpart.static.c"


static __inline__ unsigned char * parseNextTag( 
   struct UFDBtable * parent, 
   unsigned char *    mem )
{
   unsigned char *    tag;
   int                tagType;
   int                n;

   while (mem != NULL)
   {
      tag = mem;
      while (*mem >= ' ')
	 mem++;

      tagType = *mem;
      *mem++ = '\0';

      switch (tagType)
      {
      case UFDBsubLevel:
	    /* DEBUG(( stderr, "   parse  tag = %-10s  sub-level\n", tag )); */

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel = _reallocTableArray( parent->nextLevel, parent->nNextLevels );
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = NULL;

	    mem = parseNextTag( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsameLevel:
	    /* DEBUG(( stderr, "   parse  tag = %-10s  same-level\n", tag )); */

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel = _reallocTableArray( parent->nextLevel, parent->nNextLevels );
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = NULL;

	    break;

      case UFDBprevLevel:
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       /* DEBUG(( stderr, "   parse  tag = %-10s  prev-level\n", tag )); */

	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel = _reallocTableArray( parent->nextLevel, parent->nNextLevels );
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    else 
	    {
	       /* DEBUG(( stderr, "   parse  tag = %-10s  prev-level\n", "*" )); */
	       ;
	    }
	    return mem;
	    break;

      case UFDBendTable:
	    /* DEBUG(( stderr, "   parse  tag = %-10s  end-of-table\n", "NONE" )); */
	    return NULL;
      }
   }

   return NULL;
}


/* parse a binary table that is loaded into memory.
 */
void UFDBparseTable( struct UFDBmemTable * memTable )
{
   memTable->table.nNextLevels = 0;
   memTable->table.nextLevel = NULL;
   (void) parseNextTag( &(memTable->table), (unsigned char *) memTable->mem + sizeof(struct UFDBfileHeader) );
}


void UFDBappInit( void )
{
   ;
}


void UFDBtimerInit( struct tms * t )
{
   (void) times( t );
}


void UFDBtimerStop( struct tms * t )
{
   struct tms te;

   (void) times( &te );

   t->tms_utime  = te.tms_utime  - t->tms_utime;
   t->tms_stime  = te.tms_stime  - t->tms_stime;
   t->tms_cutime = te.tms_cutime - t->tms_cutime;
   t->tms_cstime = te.tms_cstime - t->tms_cstime;
}


void UFDBtimerPrintString( char * line, struct tms * t, char * tag )
{
   double  numTicks;

   numTicks = (double) sysconf( _SC_CLK_TCK );

   if (tag == NULL)
      tag = "UFDB timer";

   sprintf( line, "%s:  %5.2f user  %5.2f sys  %5.2f total", 
	    tag,
            (double) t->tms_utime / numTicks, 
	    (double) t->tms_stime / numTicks, 
	    (double) (t->tms_utime+t->tms_stime) / numTicks );
}


void UFDBtimerPrint( struct tms * t, char * tag )
{
   char    line[256];

   UFDBtimerPrintString( line, t, tag );
   fprintf( stderr, "%s\n", line );
   fflush( stderr );
}


/* perform lookup of revUrl in table t.
 * return 1 iff found, 0 otherwise.
 */
int UFDBlookupRevUrl( 
   struct UFDBtable * t, 
   UFDBrevURL *       revUrl )
{
   int b, e, i;
   int cmp;

begin:
   DEBUG(( stderr, "    UFDBlookupRevUrl:  table %10s  %s\n", t->tag, revUrl->part ));  /* */

   {
      /* binary search */
      b = 0;
      e = t->nNextLevels - 1;
      while (b <= e)
      {
	 i = (b + e) / 2;
	 cmp = strcmpURLpart( (char *) revUrl->part, (char *) t->nextLevel[i].tag );
         DEBUG(( stderr, "      strcmpURLpart[%d]  %s  %s  = %d\n", i,
	                 revUrl->part, t->nextLevel[i].tag, cmp ));
	 if (cmp == 0)
	 {
	    t = &(t->nextLevel[i]);
	    if (t->nNextLevels == 0)		/* no more levels in table -> MATCH */
	       return 1;
	    revUrl = revUrl->next;
	    if (revUrl == NULL)			/* no more levels in URL -> NO match */
	       return 0;

	    /* optimise the recursive call : */
	    /* return UFDBlookupRevUrl( t, revUrl ); */
	    goto begin;
	 }
	 if (cmp < 0)
	    e = i - 1;
	 else
	    b = i + 1;
      }
   }

   return 0;  /* not found */
}


/*
 *  UFDBlookup: lookup a domain/URL in domain and URL databases.  
 *  return 1 if found.
 */
int UFDBlookup( 
   UFDBthreadAdmin *     admin,
   struct UFDBmemTable * mt, 
   char *                request )
{
   int                   result;
   UFDBrevURL *          revUrl;
   
   if (admin == NULL)
   {
      fprintf( stderr, "UFDBlookup admin=NULL\n" );
      return 0;
   }

   revUrl = UFDBgenRevURL( admin, (unsigned char *) request );
   result = UFDBlookupRevUrl( &(mt->table.nextLevel[0]), revUrl );
   DEBUG(( stderr, "  UFDBlookup( %s, %s ) = %d\n", mt->table.nextLevel[0].tag, request, result ));

   UFDBfreeRevURL( admin, revUrl );

   return result;
}


static pthread_mutex_t mutex_url_history = UFDB_STATIC_MUTEX_INIT;
static char            url_history[URL_HIST_SIZE] = "";
static volatile int    url_history_index = 0;


/* Store the name of a webserver in the url_history buffer.
 * Return 1 if the buffer is full, 0 otherwise.
 */
int ufdbRegisterUnknownURL( char * webserver )
{
   int  length;
   char b[200];

   if (url_history_index >= sizeof(url_history) - 2)
      return 1;

   /* webserver names that do not have a dot (.) have no domain name          */
   /* and can never be incorporated in the URL database and are skipped here. */
   if (webserver == NULL  ||  strchr( webserver, '.' ) == NULL)
      return 0;

   length = strlen( webserver );
   pthread_mutex_lock( &mutex_url_history );

   /* check for buffer overflow */
   if (url_history_index + length >= sizeof(url_history) - 2)
   {
      url_history_index = sizeof(url_history);
      pthread_mutex_unlock( &mutex_url_history );
      return 1;
   }

   /* almost perfect optimisation for duplicates */
   if (length <= 196)
   {
      sprintf( b, "|%s|", webserver );
      if (strstr( url_history, b ) != NULL)
      {
	 pthread_mutex_unlock( &mutex_url_history );
	 return 0;
      }
   }

   strcpy( &url_history[url_history_index], webserver );
   url_history_index += length;
   url_history[url_history_index] = '|';
   url_history_index++;
   url_history[url_history_index] = '\0';

   pthread_mutex_unlock( &mutex_url_history );

   return 0;
}


/*
 * Retrieve all registered uncategorised URLs.
 * This must be followed by a ufdbResetUnknownURLs().
 */
char * ufdbGetUnknownURLs( void )
{
   pthread_mutex_lock( &mutex_url_history );
   url_history_index = sizeof(url_history) + 2;		/* prevent additions until reset */
   pthread_mutex_unlock( &mutex_url_history );

   return url_history;
}


void ufdbResetUnknownURLs( void )
{
   pthread_mutex_lock( &mutex_url_history );
   url_history[0] = '\0';
   url_history_index = 0;
   pthread_mutex_unlock( &mutex_url_history );
}


int UFDBopenSocket( char * serverName, int port )
{
   int                 s;
   int                 my_errno;
   int                 ret;
   struct hostent      server;
   struct hostent *    server_ptr;
   struct sockaddr_in  addr;
#if HAVE_GETADDRINFO==0  &&  defined(__linux__) && HAVE_GETHOSTBYNAME_R>0
   char                buffer[1024];
   struct hostent *    result;
#endif
#if HAVE_GETADDRINFO
   struct addrinfo     addrinfo_hints;
   struct addrinfo *   addrinfo;
#endif

   ufdbGetMallocMutex( "UFDBopenSocket" );

#if HAVE_GETADDRINFO
   addrinfo = NULL;
   addrinfo_hints.ai_flags = 0;
   addrinfo_hints.ai_family = AF_INET;
   addrinfo_hints.ai_socktype = SOCK_STREAM;
   addrinfo_hints.ai_protocol = IPPROTO_TCP;
   addrinfo_hints.ai_addrlen = 0;
   addrinfo_hints.ai_addr = NULL;
   addrinfo_hints.ai_canonname = NULL;
   addrinfo_hints.ai_next = NULL;

   ret = getaddrinfo( serverName, NULL, &addrinfo_hints, &addrinfo );
   if (ret != 0)
   {
      syslog( LOG_WARNING, "cannot resolve hostname %s: %s", serverName, gai_strerror(ret) );
      ufdbReleaseMallocMutex( "UFDBopenSocket" );
      return -1;
   }

   memcpy( (void *) &addr, (void *) addrinfo->ai_addr, sizeof(addr) );  
   addr.sin_family = AF_INET;
   addr.sin_port = htons( port );

   freeaddrinfo( addrinfo );

#else

#if defined(__linux__) && HAVE_GETHOSTBYNAME_R>0
   /* 
    * check the server name.
    */
   my_errno = 0;
   ret = gethostbyname_r( serverName, &server, buffer, sizeof(buffer)-1, &result, &my_errno );
   if (ret != 0)
   {
      syslog( LOG_WARNING, "cannot resolve hostname %s: %s", serverName, strerror(my_errno) );
      ufdbReleaseMallocMutex( "UFDBopenSocket" );
      return -1;
   }

   addr.sin_family = AF_INET;
   memcpy( (void *) &addr.sin_addr, (void *) server.h_addr_list[0], sizeof(addr.sin_addr) );  
   addr.sin_port = htons( port );

#else

#if HAVE_GETIPNODEBYNAME
   my_errno = 0;
   server_ptr = getipnodebyname( serverName, AF_INET, 0, &my_errno );
   if (server_ptr == NULL)
   {
      syslog( LOG_WARNING, "cannot resolve hostname %s: %s", serverName, strerror(my_errno) );
      ufdbReleaseMallocMutex( "UFDBopenSocket" );
      return -1;
   }

   addr.sin_family = AF_INET;
   memcpy( (void *) &addr.sin_addr, (void *) server_ptr->h_addr_list[0], sizeof(addr.sin_addr) );  
   addr.sin_port = htons( port );

   freehostent( server_ptr );
   #error *USE* HAVE_GETIPNODEBYNAME   /* TODO */

#else

   ret = pthread_mutex_lock( &gethostbyname_mutex );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "UFDBopenSocket: mutex_lock failed" );
#endif

   /*
    * check the server name.
    */
   errno = 0;
   server_ptr = gethostbyname( serverName );
   if (server_ptr == NULL)
   {
      syslog( LOG_WARNING, "cannot resolve hostname %s: %s", serverName, hstrerror(h_errno) );
      ufdbReleaseMallocMutex( "UFDBopenSocket" );
      return -1;
   }

   addr.sin_family = AF_INET;
   memcpy( (void *) &addr.sin_addr, (void *) server_ptr->h_addr_list[0], sizeof(addr.sin_addr) );  
   addr.sin_port = htons( port );

   ret = pthread_mutex_unlock( &gethostbyname_mutex );
#ifdef UFDB_DEBUG
   if (ret != 0)
      ufdbLogError( "UFDBopenSocket: mutex_unlock failed" );
#endif

#endif
#endif
#endif

   ufdbReleaseMallocMutex( "UFDBopenSocket" );

   /*
    * create the socket to connect to the daemon.
    */
   s = socket( AF_INET, SOCK_STREAM, 0 );
   if (s < 0)
      return -1;

   if (connect( s, (struct sockaddr *) &addr, sizeof(addr) ) < 0)
   {
      syslog( LOG_WARNING, "cannot connect to %s port %d: %s", serverName, port,strerror(errno) );
      close( s );
      s = -1;
   }
   else
   {
      int            sock_parm;
      struct timeval tv;
      struct linger  linger;

      /*
       * Allow server-side addresses to be reused (don't have to wait for timeout).
       * Turn off NAGLE.
       */
      sock_parm = 1;
      setsockopt( s, SOL_SOCKET, SO_REUSEADDR, (void *) &sock_parm, sizeof(sock_parm) );

      linger.l_onoff = 1;
      linger.l_linger = 1;
      setsockopt( s, SOL_SOCKET, SO_LINGER, (void *) &linger, sizeof(linger) );

      sock_parm = 1;
      setsockopt( s, IPPROTO_TCP, TCP_NODELAY, (void *) &sock_parm, sizeof(sock_parm) );

      /*
       *  Prevent long blocking on communication with the daemon.
       */
      tv.tv_sec = 20;
      tv.tv_usec = 0;
      setsockopt( s, SOL_SOCKET, SO_RCVTIMEO, (void *) &tv, sizeof(tv) );

      tv.tv_sec = 20;
      tv.tv_usec = 0;
      setsockopt( s, SOL_SOCKET, SO_SNDTIMEO, (void *) &tv, sizeof(tv) );

      sock_parm = 64 * 1024;
      setsockopt( s, SOL_SOCKET, SO_SNDBUF, (void *) &sock_parm, sizeof(sock_parm) );
   }

   return s;
}

