#!/usr/bin/perl
use strict;
use warnings;

# Copyright 1999-2005 Gentoo Foundation
# Distributed under the terms of the GNU General Public License v2
# $Header: /var/cvsroot/gentoo-src/ufed/ufed.pl,v 1.33 2005/05/01 10:00:47 truedfx Exp $

use File::Temp qw(tempfile);
use Term::ReadKey;

my $version = '0.39';

my %environment;
$environment{$_}={} for qw(USE); # INCREMENTALS, except we only need USE

my %packages;
my @profiles;
my %use_masked_flags;
my %use_defaults_flags;
my %make_defaults_flags;
my %default_flags;
my %make_conf_flags;
my %archs;
my %all_flags;
my %use_descriptions;

my @portagedirs;

sub finalise(%);
sub flags_dialog();
sub have_package($);
sub merge(\%%);
sub merge_env(\%);
sub noncomments($);
sub norm_path($$);
sub read_archs();
sub read_make_conf();
sub read_make_defaults();
sub read_make_globals();
sub read_packages();
sub read_profiles();
sub read_sh($);
sub read_use_defaults();
sub read_use_descs();
sub read_use_mask();
sub save_flags(@);
sub show_help();

read_packages;
read_profiles;
read_use_mask;
read_use_defaults;
read_make_globals;
read_make_defaults;
read_make_conf;
read_archs;
read_use_descs;

%default_flags = %make_defaults_flags;
merge %default_flags, %use_defaults_flags;

%all_flags = %default_flags;
merge %all_flags, %make_conf_flags;
delete $all_flags{'*'};
$all_flags{'-*'} = 1 if exists $make_conf_flags{'*'} && !$make_conf_flags{'*'};

merge %use_masked_flags, %archs;

for(keys %all_flags) {
	$use_descriptions{$_} = "(Unknown)"
	if not exists $use_descriptions{$_};
}
$use_descriptions{'-*'} = 'Never enable any flags other than those specified in /etc/make.conf';

for(keys %use_masked_flags)
{ delete $use_descriptions{$_} if $use_masked_flags{$_} }

my %flags;
DIALOG: {
	my $rc = flags_dialog;
	if($rc==-1) {
		# error
		print STDERR "fatal error: the dialog couldn't be opened\n";
		exit 1
	} elsif($rc==0) {
		# save
		# we don't check for use.masked flags here anymore
		# the checks were broken. they were filtered out earlier anyway
		if(-w '/etc/' || -w '/etc/make.conf' && -w '/etc/make.conf.old') {
			my @flags = finalise %flags;
			save_flags @flags;
		}
		exit
	} elsif($rc==1) {
		# cancel
		exit
	} elsif($rc==2) {
		# help
		show_help;
		redo DIALOG
	}
}

sub finalise(%) {
	my (%flags) = @_;
	if(exists $flags{'-*'}) {
		return sort keys %flags;
	} else {
		my(@enabled, @disabled);
		my %all_flags;
		@all_flags{keys %flags, keys %default_flags} = ();
		for(sort keys %all_flags) {
			next if $_ eq '*';
			push @enabled,    $_  if $flags{$_} && !$default_flags{$_};
			push @disabled, "-$_" if $default_flags{$_} && !$flags{$_};
		}
		return @enabled, @disabled;
	}
}

sub have_package($) {
	my ($cp) = @_;
	return $packages{$cp};
}

sub merge(\%%) {
	my ($env, %env) = @_;
	%{$env} = () if(exists $env{'*'});
	$env->{$_} = $env{$_} for(keys %env);
}

sub merge_env(\%) {
	my ($env) = @_;
	for(keys %environment) {
		if(ref $environment{$_} eq 'HASH') {
			if(exists $env->{$_}) {
				my %split;
				for(split ' ', $env->{$_}) {
					my $off = s/^-//;
					%split = () if($_ eq '*');
					$split{$_} = !$off;
				}
				$env->{$_} = { %split };
				merge %{$environment{$_}}, %{$env->{$_}};
			}
		}
	}
	for(keys %{$env}) {
		if(ref $environment{$_} ne 'HASH') {
			$environment{$_} = $env->{$_};
		}
	}
}

sub noncomments($) {
	my ($fname) = @_;
	my @result;
	local $/;
	if(open my $file, '<', $fname) {
		@result = split /(?:[^\S\n]*(?:#.*)?\n)+/, <$file>."\n";
		shift @result if @result>0 && $result[0] eq '';
		close $file;
	}
	return @result;
}

sub norm_path($$) {
	my ($base, $path) = @_;
	my @pathcomp = ($path !~ m!^/! && split(m!/!, $base), split(m!/!, $path));
	for(my $i=0;;$i++) {
		last if $i == @pathcomp; # don't want to skip this with redo
		if($pathcomp[$i] eq '' || $pathcomp[$i] eq '.') {
			splice @pathcomp, $i, 1;
			redo;
		}
		if($pathcomp[$i] eq '..') {
			if($i==0) {
				splice @pathcomp, 0, 1;
			} else {
				splice @pathcomp, --$i, 2;
			}
			redo;
		}
	}
	return '/'.join '/', @pathcomp;
}

sub read_archs() {
	for my $dir(@portagedirs) {
		for(noncomments "$dir/profiles/arch.list") {
			$archs{$_} = 1;
		}
	}
}

sub read_make_conf() {
	my %env = read_sh "/etc/make.conf";
	merge %make_conf_flags, %{$env{USE}} if exists $env{USE};
	@portagedirs = $environment{PORTDIR};
	push @portagedirs, split ' ', $environment{PORTDIR_OVERLAY} if defined $environment{PORTDIR_OVERLAY};
}

sub read_make_defaults() {
	for my $dir(@profiles) {
		my %env = read_sh "$dir/make.defaults";
		merge %make_defaults_flags, %{$env{USE}} if exists $env{USE};
	}
}

sub read_make_globals() {
	for my $dir(@profiles, '/etc') {
		read_sh "$dir/make.globals";
	}
}

sub read_packages() {
	chdir "/var/db/pkg";
	for(glob "*/*") {
		if(open my $provide, '<', "$_/PROVIDE") {
			if(open my $use, '<', "$_/USE") {
				# could be shortened, but make sure not to strip off part of the name
				s/-\d+(?:\.\d+)*\w?(?:_(?:alpha|beta|pre|rc|p)\d*)?(?:-r\d+)?$//;
				$packages{$_} = 1;
				local $/;
				my @provide = split ' ', <$provide>;
				my @use = split ' ', <$use>;
				for(my $i=0; $i<@provide; $i++) {
					my $pkg = $provide[$i];
					next if $pkg eq '(' || $pkg eq ')';
					if($pkg !~ s/\?$//) {
						$pkg =~ s/-\d+(?:\.\d+)*\w?(?:_(?:alpha|beta|pre|rc|p)\d*)?(?:-r\d+)?$//;
						$packages{$pkg} = 1;
					} else {
						my $musthave = $pkg !~ s/^!//;
						my $have = 0;
						for(@use) {
							if($pkg eq $_)
							{ $have = 1; last }
						}
						if($musthave != $have) {
							my $level = 0;
							for($i++;$i<@provide;$i++) {
								$level++ if $provide[$i] eq '(';
								$level-- if $provide[$i] eq ')';
								last if $level==0;
							}
						}
					}
				}
				close $use;
			}
			close $provide;
		}
	}
}

sub read_profiles() {
	$_ = readlink '/etc/make.profile';
	die "/etc/make.profile is not a symlink\n" if not defined $_;
	@profiles = norm_path '/etc', $_;
	PARENT: {
		for(noncomments "$profiles[0]/parent") {
			unshift @profiles, norm_path $profiles[0], $_;
			redo PARENT;
		}
	}
}

sub read_sh($) {
	my $BLANK = qr{(?:[ \n\t]+|#.*)+};         # whitespace and comments
	my $IDENT = qr{([^ \\\n\t'"{}=]+)};        # identifiers
	my $ASSIG = qr{=};                         # assignment operator
	my $UQVAL = qr{((?:[^ \\\n\t'"]+|\\.)+)}s; # unquoted value
	my $SQVAL = qr{'([^']*)'};                 # singlequoted value
	my $DQVAL = qr{"((?:[^\\"]|\\.)*)"}s;      # doublequoted value

	my ($fname) = @_;
	my %env;
	if(open my $file, '<', $fname) {
		{ local $/; $_ = <$file> }
		eval {
			for(;;) {
				/\G$BLANK/gc;
				last if pos == length;
				/\G$IDENT/gc or die;
				my $name = $1;
				/\G$BLANK/gc;
				/\G$ASSIG/gc or die;
				/\G$BLANK/gc;
				die if pos == length;
				my $value = '';
				for(;;) {
					if(/\G$UQVAL/gc || /\G$DQVAL/gc) {
						my $addvalue = $1;
						$addvalue =~ s[
							\\\n       | # backspace-newline
							\\(.)      | # other escaped characters
							\$({)?       # $
							$IDENT       # followed by an identifier
							(?(2)})      # optionally enclosed in braces
						][
							defined $3 ? $env{$3} || '' : # replace envvars
							defined $1 ? $1             : # unescape escaped characters
							             ''               # delete backspace-newlines
						]gex;
						$value .= $addvalue
					} elsif(/\G$SQVAL/gc) {
						$value .= $1
					} else {
						last
					}
				}
				$env{$name} = $value;
			}
		};
		die "Parse error in $fname\n" if $@;
		close $file;
	}
	merge_env %env;
	return %env if wantarray;
}

sub read_use_defaults() {
	for my $dir(@profiles) {
		for(noncomments "$dir/use.defaults") {
			my ($flag, @packages) = split;
			for(@packages)
			{ $use_defaults_flags{$flag} = 1 if have_package $_ }
		}
	}
}

sub read_use_descs() {
	for my $dir(@portagedirs) {
		for(noncomments "$dir/profiles/use.local.desc") {
			s/([\\"])/\\$1/g;
			my ($pkg, $flag, $desc) = /^(.*?):(.*?)\s+-\s+(.*)$/ or next;
			$use_descriptions{$flag} = "Local Flag: $desc ($pkg)";
		}
		for(noncomments "$dir/profiles/use.desc") {
			s/([\\"])/\\$1/g;
			my ($flag, $desc) = /^(.*?)\s+-\s+(.*)$/ or next;
			$use_descriptions{$flag} = $desc;
		}
	}
}

sub read_use_mask() {
	for my $dir(@profiles) {
		for(noncomments "$dir/use.mask") {
			my $off = s/^-//;
			$use_masked_flags{$_} = !$off;
		}
	}
}

sub flags_dialog() {
	my $cols     = 80;
	my $lines    = 20;
	my @termsize = GetTerminalSize();
	if(@termsize == 4) {
		$cols  = $termsize[0];
		$lines = $termsize[1] - 4;
	}

	my ($tempfh, $tempfile) = tempfile('use.XXXXXX', DIR => '/tmp', UNLINK => 1);

	my $save;
	if(-w '/etc/' || -w '/etc/make.conf' && -w '/etc/make.conf.old')
	{ $save = "Save" }
	else
	{ $save = "Read Only/No Saving" }

	my $items;
	for my $flag(sort { uc $a cmp uc $b } keys %use_descriptions) {
		$items .= $flag . ' " ';

		$items .= exists $make_defaults_flags{$flag} ? $make_defaults_flags{$flag} ?'(+' :'(-' :'( ' ;
		$items .= exists  $use_defaults_flags{$flag} ?  $use_defaults_flags{$flag} ? '+' : '-' : ' ' ;
		$items .= exists     $make_conf_flags{$flag} ?     $make_conf_flags{$flag} ? '+)': '-)': ' )';

		$items .= ' ' . $use_descriptions{$flag} . '" ';

		if($all_flags{$flag})
		{ $items .= 'on' }
		else
		{ $items .= 'off' }

		$items .= ' "'.$use_descriptions{$flag}.'" ';
	}
	# bug 51781, in some cased dialog was outputting to stderr and it was messing up
	# the expected results from dialog.  Brandon Edens provided a patch so that the
	# stderr output was not messing up the parsing of the results anymore.
	# Thanks Brandon
	my ($cmdfh, $cmdfile) = tempfile 'dialog.XXXXXX', DIR => '/tmp', UNLINK => 1;
	print $cmdfh '--output-fd 3 --separate-output --visit-items '
	           . '--no-shadow --backtitle "Gentoo Linux USE flags editor '.$version.'" '
	           . '--ok-label "'.$save.'" --cancel-label Exit --help-label "What are USE flags?/Help" '
	           . '--item-help --help-button --help-status --checklist "Select desired set of USE '
	           . 'flags from the list below:\\n(press SPACE to toggle, cursor keys to select)" '
	           . $lines . ' ' . $cols . ' ' . ($lines - 8) . ' ' . $items;
	my $rc = system('exec 3> '.$tempfile.' ; DIALOG_ESC="" dialog --file '.$cmdfile) >> 8;
	if($rc==0) {
		# OK
		open my $file, '<', $tempfile or die "Couldn't open temporary file\n";
		local $/;
		%flags = ();
		$flags{$_} = 1 for split ' ', <$file>;
		close $file;
		return 0;
	} elsif($rc==1||$rc==255) {
		# Cancel, Esc
		return 1;
	} elsif($rc==2) {
		# Help
		if(open my $file, '<', $tempfile) {
			<$file>; # skip past the first line which is just HELP and the flag description
			local $/;
			%all_flags = ();
			$all_flags{$_} = 1 for split ' ', <$file>;
			close $file;
		}
		return 2;
	} else {
		return -1;
	}
}

sub show_help() {
	my $cols     = 80;
	my $lines    = 20;
	my @termsize = GetTerminalSize();

	if(@termsize == 4) {
		$cols  = $termsize[0];
		$lines = $termsize[1] - 4;
	}

	my ($tempfh, $tempfile) = tempfile 'use.XXXXXX', DIR => '/tmp', UNLINK => 1;

	# bug 50112 fixed, url for howto changed
	open my $file, '>', $tempfile or return;
	print $file qq{(press UP/DOWN to scroll, RETURN to go back)

UFED is a simple program designed to help you configure the systems
USE flags (see below) to your liking.  To select of unselect a flag
highlight it and hit space.

UFED attempts to show you where a  particular use setting came from.
Each USE flag has a 3 character descriptor that represents the three
ways a use flag can be set.

The 1st char is the setting from the /etc/make.profile/make.defaults
file. These are the defaults for Gentoo as a whole. These should not
be changed.

The 2nd char is the setting from the /etc/make.profile/use.defaults
file. These will change as packages are added and removes from the
system.

The 3rd char is the settings from the /etc/make.conf file. these are
the only ones that should be changed by the user and these are the
ones that UFED changes.

If the character is a + then that USE flag was set in that file, if
it is a space then the flag was not mentioned in that file and if it
is a - then that flag was unset in that file.

------------------- What Are USE Flags -----------------------------

The USE settings system is a flexible way to enable or disable various
features at package build-time on a global level and for individual
packages. This allows an administrator control over how packages are built
in regards to the optional features which can be compiled into those
packages.


For instance, packages with optional GNOME support can have this support
disabled at compile time by disabling the "gnome" USE setting. Enabling
the "gnome" USE setting would enable GNOME support in these same packages.

The effect of USE settings on packages is dependent on whether both the
software itself and the package ebuild supports the USE setting as an
optional feature. If the software does not have support for an optional
feature then the corresponding USE setting will obviously have no effect.

Also many package dependencies are not considered optional by the software
and thus USE settings will have no effect on those mandatory dependencies.

A list of USE keywords used by a particular package can be found by
checking the IUSE line in any ebuild file.


See http://www.gentoo.org/doc/en/handbook/handbook-x86.xml?part=2&chap=1
for more information on USE flags.


Please also note that if UFED describes a flag as (Unknown) it generally means
that it is either a spelling error in one of the three configuration files or
it is not an ofically sanctioned USE flag. Sanctioned USE flags can be found in
${portagedirs[0]}/profiles/use.desc and in ${portagedirs[0]}/profiles/use.local.desc.


* * * * *

ufed was originally written by Maik Schreiber <blizzy\@blizzy.de>.
ufed was previously maintained by Robin Johnson <robbat2\@gentoo.org>,
Fred Van Andel <fava\@gentoo.org> and Arun Bhanu <codebear\@gentoo.org>.
ufed is currently maintained by Harald van Dijk <truedfx\@gentoo.org>.

Copyright 1999-2005 Gentoo Foundation
Distributed under the terms of the GNU General Public License v2
};
	close $file;

	system('dialog --exit-label Back --no-shadow --title "What are USE flags?" '
	     . '--backtitle "Gentoo Linux USE flags editor '
	     . $version
	     . ' - Help" '
	     . '--textbox '
	     . $tempfile . ' '
	     . $lines . ' '
	     . $cols);
}

sub save_flags(@) {
	my $BLANK = qr{(?:[ \n\t]+|#.*)+};              # whitespace and comments
	my $UBLNK = qr{(?:                              # as above, but scan for #USE=
		[ \n\t]+ |
		\#[ \t]*USE[ \t]*=.*(\n?) | # place capture after USE=... line
		\#.*)+}x;
	my $IDENT = qr{([^ \\\n\t'"{}=]+)};             # identifiers
	my $ASSIG = qr{=};                              # assignment operator
	my $UQVAL = qr{(?:[^ \\\n\t'"]+|\\.)+}s;        # unquoted value
	my $SQVAL = qr{'[^']*'};                        # singlequoted value
	my $DQVAL = qr{"(?:[^\\"]|\\.)*"}s;             # doublequoted value
	my $BNUQV = qr{(?:[^ \\\n\t'"]+|\\\n()|\\.)+}s; # unquoted value (scan for \\\n)
	my $BNDQV = qr{"(?:[^\\"]|\\\n()|\\.)*"}s;      # doublequoted value (scan for \\\n)

	my (@flags) = @_;
	my $contents;

	{
		open my $makeconf, '<', '/etc/make.conf' or die "Couldn't open /etc/make.conf\n";
		open my $makeconfold, '>', '/etc/make.conf.old' or die "Couldn't open /etc/make.conf.old\n";
		local $/;
		$_ = <$makeconf>;
		print $makeconfold $_;
		close $makeconfold;
		close $makeconf;
	}

	eval {
		# USE comment start/end (start/end of newline character at the end, specifically)
		# default to end of make.conf, to handle make.confs without #USE=
		my($ucs, $uce) = (length, length);
		my $flags = '';
		for(;;) {
			if(/\G$UBLNK/gc) {
				($ucs, $uce) = ($-[1], $+[1]) if defined $1;
			}
			last if pos == length;
			my $flagatstartofline = do {
				my $linestart = 1+rindex $_, "\n", pos()-1;
				my $line = substr($_, $linestart, pos()-$linestart);
				$line !~ /[^ \t]/;
			};
			/\G$IDENT/gc or die;
			my $name = $1;
			/\G$BLANK/gc;
			/\G$ASSIG/gc or die;
			/\G$BLANK/gc;
			die if pos == length;
			if($name ne 'USE') {
				/\G(?:$UQVAL|$SQVAL|$DQVAL)+/gc or die;
			} else {
				my $start = pos;
				/\G(?:$BNUQV|$SQVAL|$BNDQV)+/gc or die;
				my $end = pos;
				# save whether user uses backslash-newline
				my $bsnl = defined $1 || defined $2;
				# start of the line is one past the last newline; also handles first line
				my $linestart = 1+rindex $_, "\n", $start-1;
				# everything on the current line before the USE flags, plus one for the "
				my $line = substr($_, $linestart, $start-$linestart).' ';
				# only indent if USE starts a line
				my $blank = $flagatstartofline ? $line : "";
				$blank =~ s/[^ \t]/ /g;
				# word wrap
				if(@flags != 0) {
					my $length = 0;
					while($line =~ /(.)/g) {
						if($1 ne "\t") {
							$length++;
						} else {
							# no best tab size discussions, please. terminals use ts=8.
							$length&=~8;
							$length+=8;
						}
					}
					my $blanklength = $blank ne '' ? $length : 0;
					# new line, using backslash-newline if the user did that
					my $nl = ($bsnl ? " \\\n" : "\n").$blank;
					my $linelength = $bsnl ? 76 : 78;
					my $flag = $flags[0];
					if($blanklength != 0 || length $flag <= $linelength) {
						$flags   = $flag;
						$length += length $flag;
					} else {
						$flags   = $nl.$flag;
						$length  = length $flag;
					}
					for $flag(@flags[1..$#flags]) {
						if($length + 1 + length $flag <= $linelength) {
							$flags  .= " $flag";
							$length += 1+length $flag;
						} else {
							$flags  .= $nl.$flag;
							$length  = $blanklength + length $flag;
						}
					}
				}
				# replace the current USE flags with the modified ones
				substr($_, $start, $end-$start) = "\"$flags\"";
				# and have the next search start after our new flags
				pos = $start + 2 + length $flags;
				# and end this
				undef $flags;
				last;
			}
		}
		if(defined $flags) { # if we didn't replace the flags, tack them after the last #USE= or at the end
			$flags = '';
			if(@flags != 0) {
				$flags = $flags[0];
				my $length = 5 + length $flags[0];
				for my $flag(@flags[1..$#flags]) {
					if($length + 1 + length $flag <= 78) {
						$flags  .= " $flag";
						$length += 1+length $flag;
					} else {
						$flags  .= "\n     $flag";
						$length  = 5+length $flag;
					}
				}
			}
			substr($_, $ucs, $uce-$ucs) = "\nUSE=\"$flags\"\n";
		} else { # if we replaced the flags, delete any further overrides
			for(;;) {
				my $start = pos;
				/\G$BLANK/gc;
				last if pos == length;
				/\G$IDENT/gc or die;
				my $name = $1;
				/\G$BLANK/gc;
				/\G$ASSIG/gc or die;
				/\G$BLANK/gc;
				/\G(?:$UQVAL|$SQVAL|$DQVAL)+/gc or die;
				my $end = pos;
				if($name eq 'USE') {
					substr($_, $start, $end-$start) = '';
					pos = $start;
				}
			}
		}
	};
	die "Parse error when writing make.conf - did you modify it while ufed was running?\n" if $@;

	{
		open my $makeconf, '>', '/etc/make.conf' or die "Couldn't open /etc/make.conf\n";
		print $makeconf $_;
		close $makeconf;
	}
}
