#include "unionfs.h"

/*
 * export operations.
 * unionfs cannot handle disconnected dentry, since it has no hidden dentries.
 */
/* un-tested 64 bit environment (pointer and inode number) */

#define is_anon(d) ((d)->d_flags & DCACHE_DISCONNECTED)
extern struct export_operations export_op_default;

static void prepend_path(char **path, const char *name, int len)
{
	*path -= len;
	memcpy(*path, name, len);
	(*path)--;
	**path = '/';
}

struct filldir_arg {
	int found, called;
	char *path;
	ino_t ino, parent_ino;
};

static int filldir(void *arg, const char *name, int len, loff_t pos, u64 ino,
		unsigned int d_type)
{
	struct filldir_arg *a = arg;

	a->called++;
	if (len == 2 && !strncmp(name, "..", 2)) {
		a->parent_ino = ino;
		a->found++;
	} else if (ino == a->ino) {
		if (len != 1 || *name != '.')
			prepend_path(&a->path, name, len);
		a->found++;
	}
	return (a->found == 2) ? 1 : 0;
}

static struct dentry *get_hidden_parent(struct super_block *hidden_sb,
		ino_t hidden_parent_ino)
{
	__u32 fh[2];

	if (hidden_sb->s_root->d_inode->i_ino == hidden_parent_ino)
		return DGET(hidden_sb->s_root);

	fh[0] = hidden_parent_ino;
	fh[1] = 0;
	return export_op_default.get_dentry(hidden_sb, fh);
}

static struct dentry *do_get_dentry(struct super_block *sb, ino_t ino,
		__u32 gen, struct dentry *hidden_root,
		ino_t hidden_ino, ino_t hidden_parent_ino)
{
	struct dentry *dentry, *hidden_parent, *parent;
	char *path, *p;
	struct filldir_arg arg = {
		.ino = hidden_ino,
		.parent_ino = hidden_parent_ino
	};
	int open_flags, err, bindex, bend, found;
	struct file *hidden_file;
	struct super_block *hidden_sb;

	print_entry("hr%p, hi%lu, hpi%lu",
			hidden_root, hidden_ino, hidden_parent_ino);

	dentry = ERR_PTR(-ENOMEM);
	path = __getname();
	if (!path)
		goto out;
	arg.path = path + PATH_MAX - 1;
	*arg.path = 0;

	open_flags = O_RDONLY | O_DIRECTORY /* | O_NOATIME */ ;
	if (force_o_largefile())
		open_flags |= O_LARGEFILE;

	dentry = ERR_PTR(-ESTALE);
	unionfs_read_lock(sb);
	lock_dentry(sb->s_root);
	bend = dbend(sb->s_root);
	found = -1;
	for (bindex = 0; found == -1 && bindex <= bend; bindex++)
		if (hidden_root == dtohd_index(sb->s_root, bindex))
			found = bindex;
	unlock_dentry(sb->s_root);
	if (found == -1)
		goto out_unlock;

	bindex = found;
	hidden_sb = stohs_index(sb, bindex);
	while (1) {
		hidden_parent = get_hidden_parent(hidden_sb, hidden_parent_ino);
		dentry = hidden_parent;
		if (IS_ERR(hidden_parent))
			goto out_unlock;

		branchget(sb, bindex);
		hidden_file = DENTRY_OPEN(DGET(hidden_parent), NULL,
				open_flags);
		if (IS_ERR(hidden_file)) {
			dentry = (void *)hidden_file;
			DPUT(hidden_parent);
			branchput(sb, bindex);
			goto out_unlock;
		}

		arg.found = 0;
		while (arg.found != 2) {
			arg.called = 0;
			err = vfs_readdir(hidden_file, filldir, &arg);
			if (!arg.called || err < 0)
				break;
		}
		fput(hidden_file);
		branchput(sb, bindex);
		if (arg.found != 2) {
			dentry = ERR_PTR(-ESTALE);
			DPUT(hidden_parent);
			goto out_unlock;
		}

		DPUT(hidden_parent);
		if (hidden_parent_ino == hidden_root->d_inode->i_ino)
			break;
		arg.ino = hidden_parent_ino;
		hidden_parent_ino = arg.parent_ino;
	}
	BUG_ON(arg.path < path);

	parent = DGET(sb->s_root);
	p = strchr(++arg.path, '/');
	while (p) {
		mutex_lock(&parent->d_inode->i_mutex);
		dentry = LOOKUP_ONE_LEN(arg.path, parent, p - arg.path);
		mutex_unlock(&parent->d_inode->i_mutex);
		DPUT(parent);
		if (IS_ERR(dentry))
			goto out_unlock;
		if (!dentry->d_inode || !S_ISDIR(dentry->d_inode->i_mode)) {
			DPUT(dentry);
			dentry = ERR_PTR(-ESTALE);
			goto out_unlock;
		}
		parent = dentry;
		arg.path = p + 1;
		p = strchr(arg.path, '/');
	}
	mutex_lock(&parent->d_inode->i_mutex);
	dentry = LOOKUP_ONE_LEN(arg.path, parent, strlen(arg.path));
	mutex_unlock(&parent->d_inode->i_mutex);
	DPUT(parent);
	if (!IS_ERR(dentry)
			&& (!dentry->d_inode
				|| dentry->d_inode->i_ino != ino
				|| dentry->d_inode->i_generation != gen)) {
		DPUT(dentry);
		dentry = ERR_PTR(-ESTALE);
	}

out_unlock:
	unionfs_read_unlock(sb);
	__putname(path);
out:
	print_exit_pointer(dentry);
	return dentry;
}

enum {
	FhHead = 4, FhHRoot1 = FhHead, FhHRoot2,
	FhHIno1, FhHIno2, FhHPIno1, FhHPIno2,
	FhTail
};

static void do_decode(__u32 * fh, struct dentry **hidden_root,
		ino_t * hidden_ino, ino_t * hidden_parent_ino)
{
	unsigned long root;

	root = fh[FhHRoot2];
	*hidden_ino = fh[FhHIno2];
	*hidden_parent_ino = fh[FhHPIno2];
#if BITS_PER_LONG == 64
	root |= ((unsigned long)fh[FhHRoot1]) << 32;
	*hidden_ino |= ((unsigned long) fh[FhHIno1]) << 32;
	*hidden_parent_ino |= ((unsigned long) fh[FhHPIno1]) << 32;
#elif BITS_PER_LONG == 32
	/* ok */
#else
#error unknown size
#endif

	*hidden_root = (struct dentry*) root;
}

static int unionfs_encode_fh(struct dentry *dentry, __u32 * fh, int *max_len,
		int connectable)
{
	int type, len, bindex;
	struct super_block *sb;
	struct dentry *h_root;
	ino_t h_ino, hp_ino;
	static int warn;

	print_entry("dentry %p", dentry);
	BUG_ON(is_anon(dentry) || !dentry->d_inode
			|| is_anon(dentry->d_parent));

#ifdef UNIONFS_IMAP
	if (!warn && stopd(dentry->d_sb)->usi_persistent)
		warn++;
#endif
	if (!warn) {
		printk(KERN_WARNING "Exporting Unionfs without imap"
				" option may stop your NFS server or client");
		warn++;
	}

	sb = dentry->d_sb;
	unionfs_read_lock(sb);
	lock_dentry(dentry);

	len = *max_len;
	type = export_op_default.encode_fh(dentry, fh, max_len, connectable);
	if (type == 255 || *max_len > FhHead || len < FhTail) {
		type = 255;
		goto out;
	}

	*max_len = FhTail;
	bindex = dbstart(dentry);
	lock_dentry(sb->s_root);
	h_root = dtohd_index(sb->s_root, bindex);
	unlock_dentry(sb->s_root);
	h_ino = itohi_index(dentry->d_inode, bindex)->i_ino;
	hp_ino = parent_ino(dtohd(dentry));
	fh[FhHRoot2] = (unsigned long) h_root;
	fh[FhHIno2] = h_ino;
	fh[FhHPIno2] = hp_ino;
#if BITS_PER_LONG == 64
	fh[FhHRoot1] = ((unsigned long) h_root) >> 32;
	fh[FhHIno1] = h_ino >> 32;
	fh[FhHPIno1] = hp_ino >> 32;
#endif
out:
	unionfs_print(PRINT_MAIN_EXIT, "%d, fh{i%u, g%d, hr%x, hi%u, hpi%u}\n",
			type, fh[0], fh[1], fh[FhHRoot2], fh[FhHIno2],
			fh[FhHPIno2]);
	unlock_dentry(dentry);
	unionfs_read_unlock(sb);
	return type;
}

static struct dentry *unionfs_decode_fh(struct super_block *sb, __u32 * fh,
		int fh_len, int fh_type,
		int (*acceptable) (void *context,
			struct dentry * de),
		void *context)
{
	struct dentry *dentry, *hidden_root;
	ino_t hidden_ino, hidden_parent_ino;

	print_entry("%d, fh{i%u, g%d, hr%x, hi%u, hpi%u}",
			fh_type, fh[0], fh[1], fh[FhHRoot2], fh[FhHIno2],
			fh[FhHPIno2]);

	dentry = export_op_default.get_dentry(sb, fh);
	if (!dentry || IS_ERR(dentry) || (dentry->d_inode && !is_anon(dentry)))
		return dentry;

	d_drop(dentry);
	DPUT(dentry);
	do_decode(fh, &hidden_root, &hidden_ino, &hidden_parent_ino);
	dentry = do_get_dentry(sb, fh[0], fh[1], hidden_root, hidden_ino,
			hidden_parent_ino);
	if (!IS_ERR(dentry)) {
		if (acceptable(context, dentry))
			return dentry;  /* success */
		DPUT(dentry);
		dentry = NULL;
	}
	return dentry;
}

struct export_operations unionfs_export_ops = {
	.decode_fh = unionfs_decode_fh,
	.encode_fh = unionfs_encode_fh
};

