/* Copyright (C) Bram Avontuur (bram@avontuur.org) */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include	<stdio.h>
#include	<unistd.h>
#ifdef		HAVE_STDLIB_H
#include	<stdlib.h>
#endif		/* HAVE_STDLIB_H */
#include	<errno.h>
#include	<ctype.h>
#include <string.h>
#include <sys/time.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include	<fcntl.h>
#include <iostream>
/* Custom headerfiles */

#include "vcr.h"

#include <creators.h>
using namespace Creators;
using namespace std;

#include <signal.h>

#include <getopt.h>
#include <avifile.h>
#include <videoencoder.h>
#include "capproc.h"
#include "v4lxif.h"
#include "global.h"
#include "parseconfig.h"
#include "channel.h"
#include "frequencies.h"

#include VCR_SOUNDCARD_H

using namespace std;

/* External variables/functions */

/* Function prototypes */

void init_v4l();
void vcr_debug(const char*);
void usage();
void parse_arguments(int, char*argv[]);
void init_debug();
void init_globals();
void verify_globals();
void end_program(int);
short add_attribute(const char*);
short set_attributes();
int get_codec_index(int);
char **split_string(const char *);
short parse_config_file();
short set_error(const char *);
void show_error();
void vcr_warning(const char *);
void vcr_notice(const char *);
void vcr_notice(string&);
void vcr_notice(ostringstream&);
int set_rectime(const char *);
void clear_attributes();
void list_attributes(int fourcc, int indent_level = 0);
void list_codecs();

/* Global variables */

//enum cmdopts { OPT_
struct codec_attr {
	char *name;
	char *val;
} *attributes;

static short mp3_bitrates[2][2][6] = 
{
	{ //22 Khz
		{ //mono
			32, 48, 56, 64, 96, 112
		},
		{ //stereo
			64, 80, 96, 112, 128, 160
		}
	},
	{ //44 Khz
		{	//mono
			56, 64, 96, 112, 128, 160
		},
		{ //stereo
			112, 128, 160, 192, 256, 320
		}
	}
};

static struct STRTAB norms[] = {
    {  0, "PAL" },
    {  1, "NTSC" },
    {  2, "SECAM" },
    {  3, "AUTO" },
    { -1, NULL }
};

CaptureProcess *cap = NULL;
v4lxif *v4l = NULL;
FILE
	*vcr_debug_file = NULL;
short
	want_debug,
	verbose;
char
	*prog_name = NULL,
	*codec_name = NULL,
	*rectime_str = NULL,
	*attributes_str = NULL,
	*input_source,
	*preset,
	*filename,
	*config_file,
	*norm,
	*audio_mode,
	*freqtab,
	*error = NULL,
	*debugfile = NULL,
	*window_str = NULL,
	*codec_preset = NULL,
	*init_sound = NULL,
	*sound_device = NULL,
	*grabdevice;
int
	rectime,
	codecID,
	keyframes,
	quality,
	normID,
	audio_freq,
	audio_size,
	audio_bitrate = -1,
	vertical_flip = -1,
	resolution = 0,
	splitsize = -1,
	attr_count = 0,
	resolution_width = 0,
	resolution_height = 0;
avm::vector<AttributeInfo>
	encoder_info;
float
	fps;

window_t window;

enum Sound_Freqs soundfreq;
enum Sample_Sizes samplesize;
enum Sound_Chans soundchan;
enum Resolutions videores;

void stop_record_handler(int arg)
{
	cerr << "Caught CTRL+C: Stopping capture.\n" << endl;
	cap->stopRecord();
	signal(SIGINT, SIG_IGN); //otherwise this function will be called again
}

void
end_program(int exit_status)
{
	if (cap)
		delete cap;

	if (v4l)
	{
		v4l->setCapture(0);
		v4l->setAudioMute(1);
		delete v4l;
	}	
	exit(exit_status);
}

/* Kinda quick & dirty ... */
void
list_audiobitrates()
{
	cerr << "Available mp3 bitrates: " << endl << endl;

	int f_index = 0, m_index = 0, b_index = 0;

	for (f_index = 0; f_index < 2; f_index++)
	{
		const char *khz = (f_index ? "44 Khz" : "22 Khz");

		for (m_index = 0; m_index < 2; m_index++)
		{
			if (!m_index)
				cerr << khz << " mono  : ";
			else
				cerr << khz << " stereo: ";

			for (b_index = 0; b_index < 6; b_index++)
				cerr << mp3_bitrates[f_index][m_index][b_index] << " ";
			
			cerr << endl;
		}
	}
}

void
list_norms()
{
	cerr << "Available norms: ";
	int i = 0;

	while (norms[i].nr > -1)
	{
		cerr << norms[i++].str << " ";
	}
	
	cerr << endl;
}

void
list_resolutions()
{
	int i = 0;

	cerr << "Possible video sizes:" << endl;

	while (restable[i].res != WNONE)
	{
		cerr << restable[i].width << "x" << restable[i].height << 
			" (config value: " << restable[i].width << ")" << endl;
		i++;
	}
}

void
list_presets()
{
	cerr << "Available presets are:" << endl << endl;

	char **list;

	for (list = cfg_list_sections(); *list != NULL; list++)
	{
		if (
			!strcmp(*list, "defaults") ||
			!strcmp(*list, "global") ||
			!strcmp(*list, "launch") ||
			cfg_get_str(*list, "codec") //codec => codec preset
		)
			continue;
		cerr << "\"" << *list << "\" ";
	}
	cerr << endl;
}

void
list_codec_presets()
{
	cerr << "Available codec presets are:" << endl << endl;

	char **list;

	for (list = cfg_list_sections(); *list != NULL; list++)
	{
		if (
			!strcmp(*list, "defaults") ||
			!strcmp(*list, "global") ||
			!strcmp(*list, "launch") ||
			!cfg_get_str(*list, "codec") //codec => codec preset
		)
			continue;
		cerr << "\"" << *list << "\" ";
	}
	cerr << endl;
}

void
list_sources()
{
	int 
		i,
		nrsources;

	if (!v4l)
	{
		init_v4l();
		if (!v4l)
		{
			cerr << "Error opening video4linux device; cannot show input sources." <<
				endl;
			end_program(1);
		}
	}

	cerr << "Available input sources: ";
 
 	nrsources = v4l->capCapChannelC();
	for (i = 0; i < nrsources; i++)
	{
		cerr << v4l->capChannelName(i);
		cerr << (i != nrsources - 1 ? ", " : ".");
	}
	cerr << endl;
}

void
list_codecs()
{
	avm::vector<const CodecInfo*> codecList;
	avm::vector<const CodecInfo*>::const_iterator it;

	cerr << "Available codecs: " << endl << endl;

	CodecInfo::Get(codecList);

	for ( it = codecList.begin(); it != codecList.end(); it++)
	{
		if ((*it)->kind == CodecInfo::DShow_Dec)
			continue; //probably not a usable codec..
		
		if (!((*it)->direction & CodecInfo::Encode))
			continue; //only interested in codecs that can encode
		const char *cname = (*it)->GetName();

		cerr << "\"" << cname << "\"" << endl;

		list_attributes((*it)->fourcc, 1);
	}
	cerr << endl;
}

void
list_attributes(int fourcc, int indent_level)
{
	int defval;
	const char *def_str;
	avm::vector<const CodecInfo*> codecList;
	int idx;
	string indent(indent_level, ' ');

	idx = get_codec_index(fourcc);
	if (idx < 0)
		return;

	CodecInfo::Get(codecList);
	avm::vector<AttributeInfo> encinfo = codecList[idx]->encoder_info;

	if (0 == encinfo.size())
	{
		cerr << indent << "There are no attributes for this codec." << endl;
		return;
	}

	cerr << indent << "These attributes are supported for this codec:" << 
		endl;

	avm::vector<AttributeInfo>::const_iterator it;
	for(it=encinfo.begin(); it!=encinfo.end(); it++)
	{
		ostringstream att;

		att << "\"" << it->GetName() << "\"";
		cerr << indent << "Attribute ";
		cerr.width(22);
		cerr << right << att.str() << ": ";

		switch(it->kind)
		{
		case AttributeInfo::Integer:
		{
			int status = avm::CodecGetAttr(*(codecList[idx]), it->GetName(), &defval);
			cerr << indent << "Type: integer (default value: ";
			if (status)
				cerr << "none";
			else
				cerr << defval;
			cerr << ")" << endl;
			break;
		}
		case AttributeInfo::Select:
		{
			avm::CodecGetAttr(*(codecList[idx]), it->GetName(), &defval);
			cerr << indent << "Type: select (default value: " << 
				(defval < (int)it->options.size() ? it->options[defval] : "unknown") <<
				")" << endl;
			cerr << indent << "=>Possible values: ";

			avm::vector<avm::string>::const_iterator sit;

			for (sit=(it->options).begin(); sit!=(it->options).end(); sit++)
			{
				cerr << sit->c_str() << " ";
			}

			cerr << endl;
			break;
		}
		case AttributeInfo::String:
			GetCodecAttr(*(codecList[idx]), it->GetName(), (const char **)&def_str);
			cerr << indent << "Type: string (default value: " << def_str <<
				")"<< endl;
			break;
		case AttributeInfo::Float:
			cerr << "Type: float" << endl;
			break;
		default:
			break;
		}
	}
}

void
list_freqtabs()
{
	int i = 0;

	cerr << "Supported frequency tables:" << endl << endl;

	while (chanlist_names[i].nr != -1)
	{
		cerr << chanlist_names[i].str << endl;
		i++;
	}
}

void
init_v4l()
{
	if (grabdevice)
		v4l = new v4l1if(0, grabdevice);
	else
		v4l = new v4l1if(0, V4L_DEVICE);
}

void
vcr_debug(const char *txt)
{
	if (vcr_debug_file)
	{
		fwrite(txt, sizeof(char), strlen(txt), vcr_debug_file);
		fflush(vcr_debug_file);
	}
}

void
usage()
{
	cerr << "vcr " << VERSION << ": non-interactive video recorder." << endl <<
		"Usage: " << prog_name << " [options] filename" << endl << endl <<
		"Options with an asterisk (*) support a parameter 'list', to list the " <<
		"options you can choose from." << endl << endl <<
		"Options:" << endl <<
		"  -a,  --codec-attribute a=b    set codec attribute" << endl <<
		" *-b,  --audiobitrate RATE      set mp3 compression bitrate" << endl <<
		" *-c,  --codec CODEC            set encoding codec" << endl <<
		"  -d,  --debug [DEBUG]          write debugging information to DEBUG" << endl <<
		"  -F,  --framerate RATE         set framerate (default=25)" << endl <<
		"  -f,  --config-file FILE       read FILE for configuration" << endl <<
		"  -g,  --grabdevice DEVICE      use DEVICE as capture device" << endl <<
		"  -h,  --help                   display help" << endl <<
		"  -k,  --keyframes NUM          set group of picture count" << endl <<
		"  -m,  --audiomode MODE         set audio-capture to mono or stereo" << endl <<
		" *-P,  --codec-preset PRESET    Use codec preset" << endl <<
		" *-p,  --preset STATION         tune v4l into STATION" << endl <<
		"  -q,  --quality QUAL           set quality of encoding percentage" << endl <<
		" *-r,  --resolution RES         set frame size ('-r list' for a list)" << endl <<
		"  -S,  --splitsize SIZE         split file in chunks of SIZE Megabytes" << endl <<
		" *-s,  --source                 v4l input source (default=Television)" << endl <<
		"  -t,  --rectime TIME           set recording time (see 'man vcr')" << endl <<
		"  -v,  --verbose                verbose output during capturing" << endl <<
		"  -w,  --window X,Y,W,H         crop frame size:(X,Y)=top left, (W,H)=size" << endl <<
		endl;
}

void
init_globals()
{
	input_source = NULL;
	want_debug = 0;
	rectime = 0;
	codec_name = NULL;
	rectime_str = NULL;
	codecID = -1;
	normID = -1;
	keyframes = 0;
	quality = 0;
	audio_freq = 0;
	audio_size = 0;
	audio_bitrate = -1; // 0 => no mp3 compression
	resolution = 0;
	fps = -1;
	verbose = 0;
	
	window.x = -1; // to force resize to full resolution when no window is specified
	
	preset = NULL;
	filename = NULL;
	attributes = NULL;
	attr_count = 0;
	config_file = NULL;
	error = NULL;
	norm = NULL;
	audio_mode = NULL;
	cap = NULL;
	freqtab = NULL;
	grabdevice = NULL;
	debugfile = NULL;
	window_str = NULL;
	codec_preset = NULL;
	init_sound = NULL;
	attributes_str = NULL;
}

void
init_debug()
{
	if (!debugfile)
	{
		char *homedir = get_homedir(NULL);
		char *dbfile = new char[strlen(homedir) + 255];

		strcpy(dbfile, homedir);
		free(homedir);
		strcat(dbfile, "/.vcrdebug");
		vcr_debug_file = fopen(dbfile, "a");
		delete[] dbfile;
	}
	else
		vcr_debug_file = fopen(debugfile, "a");

	if (!vcr_debug_file)
	{
		cerr << "Could not open debug file." << endl;
		end_program(1);
	}
	vcr_debug("VCR Debugging started!\n");
}

int
is_valid_audiobitrate(int kbps, enum Sound_Chans mode, enum Sound_Freqs freq)
{
	int i = 0;
	short foundit = 0;
	short m_index = (mode == Stereo ? 1 : 0);
	short f_index = (freq == F44 ? 1 : 0);

	for (i = 0; i < 6; i++)
	{
		if (kbps == (int)mp3_bitrates[f_index][m_index][i])
		{
			foundit = true;
			break;
		}
	}

	if (foundit)
		return (kbps * 1000) / 8;
	return -1;
}

int
is_valid_norm(const char *n)
{
	if (!n)
		return -1;

	int i = 0;

	while (norms[i].nr > -1)
	{
		if (!strcasecmp(norms[i].str, n))
		{
			return norms[i].nr;
		}
		i++;
	}
	return -1;
}

/* returns -1 if codec does not exist.
 * Otherwise, the codec's unique ID
 */
int
is_valid_codec(const char *cname)
{
	if (!cname)
		return -1;

	int
		found_codec = -1;
	avm::vector<const CodecInfo*> codecList;

	CodecInfo::Get(codecList);

	avm::vector<const CodecInfo*>::iterator it;

	for (it = codecList.begin(); it != codecList.end(); it++)
	{
		if ( (*it)->kind == CodecInfo::DShow_Dec)
			continue;

		if (!strcasecmp(cname, (*it)->GetName()))
		{
			found_codec = (*it)->fourcc;
			encoder_info = (*it)->encoder_info;
			break;
		}
	}

	return found_codec;
}

short
set_resolution(int width)
{
	int i = 0;

	while (restable[i].width != width && restable[i].res != WNONE)
		i++;

	if (restable[i].res == WNONE)
			return 0;
		
	videores = restable[i].res;
	resolution_width = restable[i].width;
	resolution_height = restable[i].height;
	return 1;
}

/* sndstring format: [/sound/device:][mixer_device_name:]volume
 * initialize_sound returns < 0 in case of failure, and sets global
 * error variable.
 */
short
initialize_sound(const char *sndstring)
{
	//no ":"'s: just the volume (of /dev/mixer:line)
	//1 ":" device (starts with '/'), else mixerdevname
	//2 ":" device:mixerdevname:volume
	char
		dev[strlen(sndstring)+20],
		mixdev[strlen(sndstring)+20];
	int
		temp;
	unsigned int
		volume;

	temp = sscanf(sndstring, "%[^:]:%[^:]:%u", dev, mixdev, &volume);

	if (temp < 1 || temp > 3)
		return 0;
	if (temp == 1) //only sound is specified.
	{
		volume = (unsigned int)atoi(dev);
		strcpy(dev, "/dev/mixer");
		strcpy(mixdev, "line");
	}
	else if (temp == 2) //sound device OR mixer name, and volume
	{
		volume = (unsigned int)atoi(mixdev);
		if (strlen(dev) && dev[0] == '/') //sound dev
		{
			strcpy(mixdev, "line");
		}
		else
		{
			strcpy(mixdev, dev);
			strcpy(dev, "/dev/mixer");
		}
	}

	if (!volume)
	{
		return set_error("Invalid recording device volume given. A sane value "\
			"would probably be in the range of 70..100.");
	}
	if (!strlen(dev) || !strlen(mixdev))
		return set_error("No (or empty) sound- or mixer device given.");

	if (volume < 50)
		vcr_warning("Recording device volume is very low!");
	
	ostringstream s;
	s << "Using mixer " << dev << ", mixer device " << mixdev <<
		"with volume " << volume << " to record audio from.";
	vcr_notice(s);

	int
		i,
		mixer = open(dev, O_RDWR);
	unsigned int
		available_devs,
		available_recdevs;
	short
		found_label = 0;

	if (mixer < 0)
		return set_error("Could not open sound device.");

	ioctl(mixer, MIXER_READ(SOUND_MIXER_DEVMASK), &available_devs);
	ioctl(mixer, MIXER_READ(SOUND_MIXER_RECMASK), &available_recdevs);

	static const char *mixer_labels[] = SOUND_DEVICE_LABELS;
	
	char
		*cropped_label = NULL;

	//if label's found, mixerdevice-ID == i
	for (i = 0; i < SOUND_MIXER_NRDEVICES; i++)
	{
		cropped_label = crop_whitespace(mixer_labels[i]);

		if (!strcasecmp(mixdev, cropped_label))	
		{
			found_label = 1;
			free(cropped_label);
			break;
		}
		free(cropped_label);
		cropped_label = NULL;
	}

	if (!found_label)
	{
		close(mixer);
		return set_error("Could not find mixer device.");
	}
		
	if (!(available_devs & (1 << i)))
	{
		close(mixer);
		return set_error("Selected mixer device is not a valid recording device.");
	}
	if (!(available_recdevs & (1 << i)))
	{
		ostringstream s;

		s << "WARNING: " << dev << ":" << mixdev << " is not a recording device.";
		vcr_notice(s);
	}
	
	unsigned int
		recdevs;

	//ioctl(mixer, MIXER_READ(SOUND_MIXER_RECSRC), &recdevs);
	recdevs = (1 << (unsigned int)i);
	ioctl(mixer, MIXER_WRITE(SOUND_MIXER_RECSRC), &recdevs);

	//now, set the volume.
	//TODO: UNMUTE
	unsigned int
		mixvolume = (volume & 0xff) + ((volume & 0xff) << 8);

	ioctl(mixer, MIXER_WRITE(i), &mixvolume);
	close(mixer);
	return 1;
}

/* Gets the 'channel' and 'fine' settings for the preset from the xawtv,
 * calculates the frequency using those 2 values, and sets the tuner.
 */
short
set_tuner_frequency(const char *preset)
{
	char
		*cname = cfg_get_str(preset, "channel");
	int
		fine = cfg_get_int(preset, "fine");
	
	if (!cname || !strcmp(cname, "none"))
		return 0;

	int
		channel = lookup_channel(cname);

	if (channel < 0)
		return 0;

	int freq = get_freq(channel);
	
	if (freq < 0)
		return 0;
	
	freq += fine;

	v4l->setFreq(freq);
	v4l->setChannelNorm(normID); //always do that after changing tuner/channel

	return 1;
}

int
get_codec_index(int fourcc)
{
	avm::vector<const CodecInfo*>::const_iterator it;
	avm::vector<const CodecInfo*> codecList;
	int i = 0;

	CodecInfo::Get(codecList);

	for (it = codecList.begin(); it != codecList.end(); it++)
	{
		if ((*it)->fourcc == (unsigned)fourcc)
			return i;
		i++;
	}

	return -1;
}

//TODO: It ignores fourcc currently..
short
set_attribute(int fourcc, const char *attr, const char *val)
{
	int retval = 0;

	avm::vector<AttributeInfo>::const_iterator it;
	int idx = get_codec_index(fourcc);
	if (idx < 0)
		return 0;

	avm::vector<const CodecInfo*> codecList;
	CodecInfo::Get(codecList);

	avm::vector<AttributeInfo> encinfo = codecList[idx]->encoder_info;

	for (it = encinfo.begin(); it != encinfo.end(); it++)
	{
		if (strcasecmp(attr, it->GetName()))
			continue; //wrong attribute.

		switch(it->kind)
		{
		case AttributeInfo::Integer:
			SetCodecAttr(*(codecList[idx]), it->GetName(), atoi(val));
			break;
		case AttributeInfo::String:
			SetCodecAttr(*(codecList[idx]), it->GetName(), val);
			break;
		case AttributeInfo::Select:
			SetCodecAttr(*(codecList[idx]), it->GetName(), val);
			break;
		case AttributeInfo::Float:
		default:
			break;
		}
		retval = 1;
		break;
	}
	return retval;
}

short
set_attributes(int fourcc)
{
	int i;

	for (i = 0; i < attr_count; i++)
	{
		if (!set_attribute(fourcc, attributes[i].name, attributes[i].val))
			return 0;
	}

	return 1;
}

short
set_freqtab(const char *f)
{
	int
		i = 0;
	short
		found_tab = 0;
	
	while (chanlist_names[i].nr != -1)
	{
		if (!strcmp(chanlist_names[i].str, f)) //freqtab found
		{
			chantab = int(chanlist_names[i].nr);
			chanlist = chanlists[chantab].list;
			chancount = chanlists[chantab].count;
			found_tab = 1;
			break;
		}
		i++;
	}

	return found_tab;
}

short
set_window_size()
{
	// check window
	int vidres = 0;
	
	while (restable[vidres].res != videores && restable[vidres].res != WNONE)
		vidres++;

	if (restable[vidres].res == videores)
	{
		if (window.x == -1)
		{
			char bla[100];
			sprintf(bla, "Setting window size to %i x %i\n",
				restable[vidres].width, restable[vidres].height);
			vcr_debug(bla);

			window.x = 0;
			window.y = 0;
			window.width = restable[vidres].width;
			window.height = restable[vidres].height;
		}
			
		if ( (window.x + window.width > restable[vidres].width)
			|| (window.y + window.height > restable[vidres].height)
			|| window.width == 0 || window.height == 0 )
		{
			cerr << "Invalid capture window size (" << window.width << " x " <<
				window.height << ") at offset (X,Y)=(" << window.x << ", " <<
				window.y << ")! Must be bigger than 0x0 and smaller than the " <<
				"original window (which is " << restable[vidres].width << " x " <<
				restable[vidres].height << "." << endl;
			return 0;
		}
	}
	else //should never happen!
	{
		cerr << "Unknown video resolution. But you shouldn't see this." << endl;
		return 0;	
	}

	return 1;
}

int
is_valid_preset(const char *p)
{
	if (!p)
		return 0;

	int
		found_preset = 0;
	char **list;

	for (list = cfg_list_sections(); *list != NULL; list++)
	{
		if (
			!strcmp(*list, "defaults") ||
			!strcmp(*list, "global") ||
			!strcmp(*list, "launch") ||
			strcasecmp(*list, p) ||
			cfg_get_str(p, "codec") //existing section, but it's a codec preset
			)
			continue;

		found_preset = 1;
		break;
	}

	vcr_debug("Found valid preset.\n");
	return found_preset;
}

int
is_valid_codec_preset(const char *p)
{
	if (!p)
		return 0;

	int
		found_preset = 0;
	char **list;
	for (list = cfg_list_sections(); *list != NULL; list++)
	{
		if (
			!strcmp(*list, "defaults") ||
			!strcmp(*list, "global") ||
			!strcmp(*list, "launch") ||
			strcasecmp(*list, p) ||
			!cfg_get_str(p, "codec")
			)
			continue;

		found_preset = 1;
		break;
	}

	vcr_debug("Found valid codec preset.\n");
	return found_preset;
}


/* returns -1 if inputsource does not exist.
 * Otherwise, the inputsource's ID (for v4l->setChannel())
 */
int
is_valid_inputsource(const char *src)
{
	if (!src) //default to television
		src = "Television";	

	int
		i,
	 	nrsources = v4l->capCapChannelC();
	int
		found_channel = -1;

	for (i = 0; i < nrsources; i++)
	{
		if (!strcasecmp(v4l->capChannelName(i), src))
		{
			found_channel = i;
		}
	}

	return found_channel;
}

/* read codec preset settings from config file */
void
read_codec_presets(const char *p)
{
	const char *tmpstr;
	int tmpint;
	float tmpfloat;

	//since we always read a new codec, reset the attributes now.
	clear_attributes();
	if (attributes_str)
	{
		free(attributes_str);
		attributes_str = NULL;
	}

	if ( (tmpstr = cfg_get_str(p, "codec")) )
	{
		if (codec_name)
			free(codec_name);
		codec_name = strdup(tmpstr);
	}

	if ( (tmpstr = cfg_get_str(p, "attributes")) )
	{
		if (attributes_str)
			free(attributes_str);
		attributes_str = strdup(tmpstr);
	}

	if ( (tmpint = cfg_get_int(p, "keyframes")) > -1 )
		keyframes = tmpint;
	
	if ((tmpint = cfg_get_int(p, "resolution")) > -1)
		resolution = tmpint;

	if ( (tmpint = cfg_get_int(p, "quality")) > -1 )
		quality = tmpint;

	if ((tmpstr = cfg_get_str(p, "audiomode")))
	{
		if (audio_mode)
			free(audio_mode);
		audio_mode = strdup(tmpstr);
	}

	if ( (tmpint = cfg_get_int(p, "audiofrequency")) > -1 )
		audio_freq = tmpint;

	if ( (tmpint = cfg_get_int(p, "audiosamplesize")) > -1 )
		audio_size = tmpint; // 8 or 16

	if ( (tmpfloat = cfg_get_float(p, "framerate")) > -1 )
		fps = tmpfloat;

	if ( (tmpint = cfg_get_int(p, "audiobitrate")) > -1 )
		audio_bitrate = tmpint;

	if (!window_str && (tmpstr = cfg_get_str(p, "window")))
	{
		if (window_str)
		{
			free(window_str);
			window_str = NULL;
		}
		window_str = strdup(tmpstr);
	}

	if ( (tmpint = cfg_get_int(p, "splitsize"))  ) //can be -1 !
		splitsize = tmpint;
	if (splitsize < -1)
		splitsize = -1;
}

/* parses argument to 'w' / 'window' */
short
parse_window_string()
{
	int winval[4];
	int count = 0;
	
	vcr_debug("Scanning window string\n");
	
	char 
		*my_str = window_str;

	while ( count < 4 && sscanf(my_str, "%i", &(winval[count]) ) )
	{
		bool cuted = false;
		char bla[strlen(my_str)+100];

		sprintf(bla, "	-> to value: %i\n", winval[count]);
		vcr_debug(bla);

		for (unsigned int i=0; i < strlen(my_str); i++)
		{
			if (tolower(my_str[i]) == ',')
			{
				unsigned int i2=0;
				unsigned int old_length = strlen(my_str);
				vcr_debug("  found _,_ - copy for next turn\n");
		
				i++; //don't copy the ","
				while (i+i2 < old_length+1) // +1 to copy post \0
				{
					my_str[i2] = my_str[i+i2];
					i2++;
				}
				i = strlen(my_str);
				cuted = true;
			}
		}
		count++;
		if (!cuted)
			my_str [0] = 0;
		sprintf(bla, "	%i - for next turn %s\n", count, my_str);
		vcr_debug(bla);
	}	

	if (count != 4 || strlen(my_str) > 0)
		return 0;
	
	free(my_str);
	window_str = NULL;
	my_str = NULL;
	
	window.x = winval [0];
	window.y = winval [1];
	window.width = winval [2];
	window.height = winval [3];
	
	vcr_debug("END\n");
	return 1;
}

/* Side effect: mangles argv */
void
parse_arguments(int argc, char *argv[])
{
	int
		c,
		long_index;
	static struct option long_options[] = 
		{
			{ "codec-attribute", 1, 0, 'a' },
			{ "audiobitrate", 1, 0, 'b' },
			{ "codec", 1, 0, 'c' },
			{ "debug", 0, 0, 'd' },
			{ "framerate", 1, 0, 'F' },
			{ "config-file", 1, 0, 'f' },
			{ "grabdevice", 1, 0, 'g' },
			{ "help", 0, 0, 'h' },
			{ "keyframes", 1, 0, 'k' },
			{ "audiomode", 1, 0, 'm' },
			{ "codec-preset", 1, 0, 'P' },
			{ "preset", 1, 0, 'p' },
			{ "quality", 1, 0, 'q' },
			{ "resolution", 1, 0, 'r' },
			{ "rectime", 1, 0, 't' },
			{ "splitsize", 1, 0, 'S' },
			{ "source", 1, 0, 's' },
			{ "verbose", 0, 0, 'v' },
			{ "window", 1, 0, 'w' },
			{ 0, 0, 0, 0 }
		};
		
	while(1)
	{
		c = getopt_long_only(argc, argv, "a:b:c:d::F:f:g:hk:m:P:p:q:r:s:S:t:vw:",
			long_options, &long_index);

		if (c == EOF)
			break;

		switch(c)
		{
		case ':':
		case '?':
			usage();
			end_program(1);
			break;
		case 'b': /* audiocompression */
			if (!strcasecmp(optarg, "list"))
			{
				list_audiobitrates();
				end_program(0);
			}
			audio_bitrate = atoi(optarg);	
			break;
		case 'a': /* codec attributes */
			if (codec_name && !strcasecmp(optarg, "list"))
			{
				int fourcc = is_valid_codec(codec_name);
				if (fourcc != -1)
				{
					list_attributes(fourcc);
					end_program(0);
				}
				else
				{
					cerr << "error: The codec name you have specified is invalid." <<
						endl;
					list_codecs();
					end_program(1);
				}
			}
			if (!add_attribute(optarg))
			{
				cerr << "error: invalid args to -a." << endl;
				end_program(1);
			}
			break;
		case 'c': /* codec */
			if (!strcasecmp(optarg, "list"))
			{
				list_codecs();
				end_program(0);
			}
			if (codec_name)
				free(codec_name);
			codec_name = strdup(optarg);
			break;
		case 'd': /* debug */
			if (optarg)
			{
				if (debugfile)
					free(debugfile);
				debugfile = strdup(optarg);
			}
			want_debug = 1;
			break;
		case 'F': /* framerate */
			fps = atoi(optarg);
			break;
		case 'f': /* config-file */
			if (config_file)
				free(config_file);
			config_file = strdup(optarg);
			break;
		case 'g': /* grabber device */
			if (grabdevice)
				free(grabdevice);
			grabdevice = strdup(optarg);
			break;
		case 'h': /* help */
			usage();
			end_program(0);
		case 'k': /* keyframes */
			keyframes = atoi(optarg);
			break;
		case 'm': /* audiomode */
			audio_mode = strdup(optarg);
			break;
		case 'P': /* codec preset */
			if (codec_preset)
				free(codec_preset);
			codec_preset = strdup(optarg);
			break;
		case 'p': /* tuner preset (taken from xawtv config file) */
			if (preset)
				free(preset);
			preset = strdup(optarg);
			break;
		case 'q': /* quality */
			quality = atoi(optarg);
			break;
 		case 'r': /* resolution */
			if (!strcasecmp(optarg, "list"))
			{
				list_resolutions();
				end_program(0);
			}
      resolution = atoi(optarg);
 			break;
		case 'S': /* splitsize */
			splitsize = atoi(optarg);
			break;
		case 's': /* input source */
			if (!strcasecmp(optarg, "list"))
			{
				list_sources();
				end_program(0);
			}
			if (input_source)
				free(input_source);
			input_source = strdup(optarg);
			break;
		case 't': /* recording time */
		{
			if (rectime_str)
				free(rectime_str);
			rectime_str = strdup(optarg);
		}
			break;
		case 'v': //verbose
			verbose = 1;
			break;
		case 'w': /* window */
			window_str = strdup(optarg);
			break;
 		break;
		default:
			usage();
			end_program(1);
		}
	}

	if (argc - optind > 1)
	{
		cerr << "Too many file arguments given." << endl;
		end_program(1);
	}	

	if (optind < argc) /* remaining argument = filename */
	{
			
		if (filename)
			free(filename);
		filename = strdup(argv[optind]);
	}
}

/* called after reading config file & parsing cmdline args */
void
verify_globals()
{
	int temp = 0;

	if (codec_preset)
	{
		if (!is_valid_codec_preset(codec_preset))
		{
			cerr << "Invalid codec preset given." << endl;
			list_codec_presets();
			end_program(1);
		}
		else
		{
			vcr_notice("Reading codec preset.");
			read_codec_presets(codec_preset);
		}
	}

	if (!quality)
		quality = 95;
	if (quality < 1 || quality > 100)
	{
		cerr << "Quality out of range (range is 1..100, default 95)" << endl;
		end_program(1);
	}

	if (!resolution)
		resolution = 384;

	if (!set_resolution(resolution))
	{
		cerr << "Invalid resolution given." << endl;
		list_resolutions();
		end_program(1);
	}

	//set channel *before* norm, and audio after norm/channel
	if ((temp = is_valid_inputsource(input_source)) < 0)
	{
			cerr << "Invalid input source given." << endl;
			list_sources();
			end_program(1);
	}
	else
		v4l->setChannel(temp);

	if (norm && (temp = is_valid_norm(norm)) < 0)
	{
		cerr << "Invalid norm given!" << endl;
		list_norms();
		end_program(1);
	}
	else if (norm)
		normID = temp;
	if (normID < 0)
		normID = 0; //default to PAL
	v4l->setChannelNorm(normID);

	if (!audio_freq)
		audio_freq = 44;
	switch(audio_freq)
	{
	case 44: soundfreq = F44; break;
	case 22: soundfreq = F22; break;
	case 11: soundfreq = F11; break;
	default:
		cerr << "Audio frequency not one of 44,22 or 11." << endl;
		end_program(1);
	}

	if (!audio_size)
		audio_size = 16;
	switch(audio_size)
	{
	case 8: samplesize = S8; break;
	case 16: samplesize = S16; break;
	default:
		cerr << "Audio sample size is neither 16 or 8." << endl;
		end_program(1);
	}

	if (audio_mode)
	{
		if (!strcasecmp(audio_mode, "stereo"))
		{
			v4l->setAudioMode(VIDEO_SOUND_STEREO);
			soundchan = Stereo; //Stereo
		}
		else if (!strcasecmp(audio_mode, "mono"))
		{
			v4l->setAudioMode(VIDEO_SOUND_MONO);
			soundchan = Mono; //Mono
		}
		else
		{
			cerr << "Invalid audiomode given. Should be either 'stereo' " << 
				"or 'mono'." << endl;
			end_program(1);
		}
	}
	else
	{
		v4l->setAudioMode(VIDEO_SOUND_MONO);
		soundchan = Mono;
	}
	v4l->setAudioMute(false);

	if (audio_bitrate == -1)
		audio_bitrate = 0;
	
	if (audio_bitrate) //mp3 compression, Khz+mono/stereo must have been set!
	{
		int
			tmprate = is_valid_audiobitrate(audio_bitrate, soundchan, soundfreq);

		if (tmprate < 0)
		{
			cerr << "Invalid mp3 bitrate given." << endl;
			list_audiobitrates();
			end_program(1);
		}
		else if (samplesize == S8)
		{
			cerr << "8 Bit mp3 compression is not suported." << endl;
			end_program(1);
		}
		else if (soundfreq == F11)
		{
			cerr << "11 Khz mp3 compression is not supported." << endl;
			end_program(1);
		}
		else
		{
			ostringstream s;
			s << "Setting mp3 bitrate to " << tmprate;
			vcr_notice(s);
			audio_bitrate = tmprate;
		}
	}

	if (vertical_flip < 0)
			vertical_flip = 1; //default: flip image vertically

	if (!keyframes)
		keyframes = 15;
	if (keyframes < 1 || keyframes > 30)
	{
		cerr << "Keyframes out of range (range is 1..30, default 15)" << endl;
		end_program(1);
	}

	if (fps < 0)
		fps = 25.0;
	if (fps > 50.0 || fps < 1.0)
	{
		cerr << "Invalid framerate set (range: 1.0-50.0)" << endl;
		end_program(1);
	}

	if (!codec_name)
	{
		codec_name = (char*)malloc(80 * sizeof(char));
		strcpy(codec_name, "DivX ;-) low-motion");
	}

	if ((temp = is_valid_codec(codec_name)) < 0)
	{
		cerr << "Invalid codec name given!" << endl;
		list_codecs();
		end_program(1);
	}
	else
	{
		ostringstream s;
		s << "Found codec ID: " << temp;
		vcr_notice(s);
		codecID = temp;
	}

	if (attributes_str) //attributes set in config file.
	{
		char **attrs = split_string(attributes_str);
		if (!attrs)
		{
			cerr << "Couldn't parse 'attributes' in config file." << endl;
			end_program(1);
		}

		int i = 0, success = 1;
		while (attrs[i])
		{
			if (!add_attribute(attrs[i]))
			{
				success = 0;
				cerr << "Couldn't add codec attribute '" << attrs[i] << 
					"' from config file." << endl;
				free(attrs[i]);
				break;
			}
			free(attrs[i]);
			i++;
		}
		free(attrs);
		if (!success)
			end_program(1);
	}

	if (attr_count && !set_attributes(codecID))
	{
		cerr << "Invalid attribute[s] set for this codec!" << endl;
		cerr << attr_count << endl;
		list_attributes(codecID);
		end_program(1);
	}

	if (!rectime_str || !set_rectime(rectime_str))
	{
		cerr << "No (or invalid) recording time given. Type 'vcr -h' for " <<
			"help, look for '-t'." << endl;
		end_program(1);
	}

	if (preset && !(temp = is_valid_preset(preset)))
	{
		cerr << "Unknown tuner preset given." << endl;
		list_presets();
		end_program(1);
	}

	if (preset && (!freqtab || !set_freqtab(freqtab)))
	{
		cerr << "Invalid (or no) frequency table given." << endl;
		list_freqtabs();
		end_program(1);
	}

	if (preset && !set_tuner_frequency(preset))
	{
		cerr << "Couldn't set tuner for chosen preset (maybe no channel" << 
			" is set in the .vcrrc config file?)" << endl;
		end_program(1);
	}

	if (!rectime)
	{
		cerr << "Recording time has not been set." << endl;
		end_program(1);
	}

	if (window_str && !parse_window_string())
	{
		cerr << "Invalid window parameter." << endl;
		end_program(1);
	}

	if (!set_window_size()) //it generates error on stderr itself.
	{
		end_program(1);
	}

	if (init_sound && initialize_sound(init_sound) < 0)
	{
		cerr << "Could not initialize sound recording device." << endl;
		show_error();
		end_program(1);
	}
}

short
add_attribute(const char *attr)
{
	int 
		attr_len = strlen(attr) + 1;
	char
		attrname[attr_len],
		attrval[attr_len];
	struct codec_attr
		new_attr;

	int val = sscanf(attr,"%[^=]=%s", attrname, attrval);
	if (val != 2)
		return 0;
	
	new_attr.name = strdup(attrname);
	new_attr.val = strdup(attrval);

	attributes = (struct codec_attr*)realloc(attributes, (attr_count + 1) *
		sizeof(struct codec_attr));
	attributes[attr_count++] = new_attr;	

	return 1;
}

void
clear_attributes()
{
	if (!attr_count)
		return;
	
	if (attributes)
		free(attributes);
	attributes = NULL;
	attr_count = 0;
}

short
parse_config_file()
{
	const char *tmpstr;
	int tmpint;
	float tmpfloat;

	if (!config_file)
	{
		char * homedir = get_homedir(NULL);
		if (homedir)
		{
			config_file = (char*)malloc((strlen(homedir) + 20) * sizeof(char));
			sprintf(config_file, "%s/.vcrrc", homedir);
			free(homedir);
		}
		else
			return set_error("Could not find your homedir to search for default config "\
				"file (~/.vcrrc).");
	}
	if (cfg_parse_file(config_file) < 0)
		return set_error("Couldn't parse config file (maybe ~/.vcrrc is missing?).");

	if (!codec_name && !attributes_str && !attr_count &&
		(tmpstr = cfg_get_str("defaults", "attributes")))
		attributes_str = strdup(tmpstr);

	if (!codec_name && (tmpstr = cfg_get_str("defaults", "codec")))
		codec_name = strdup(tmpstr);
	
	if (!keyframes && (tmpint = cfg_get_int("defaults", "keyframes")) > -1)
		keyframes = tmpint;
	
	if (!norm && (tmpstr = cfg_get_str("defaults", "norm")))
		norm = strdup(tmpstr);

	if (!quality && (tmpint = cfg_get_int("defaults", "quality")) > -1)
		quality = tmpint;
	
	if (!input_source && (tmpstr = cfg_get_str("defaults", "source")))
		input_source = strdup(tmpstr);
	
	if (!audio_mode && (tmpstr = cfg_get_str("defaults", "audiomode")))
		audio_mode = strdup(tmpstr);

	if (!grabdevice && (tmpstr = cfg_get_str("defaults", "grabdevice")))
		grabdevice = strdup(tmpstr);

	if (!audio_freq && (tmpint = cfg_get_int("defaults", "audiofrequency")) > -1)
		audio_freq = tmpint;

	if (!audio_size && (tmpint = cfg_get_int("defaults", "audiosamplesize")) > -1)
		audio_size = tmpint; // 8 or 16

	if (!resolution && (tmpint = cfg_get_int("defaults", "resolution")) > -1)
		resolution = tmpint;

	if (fps < 0 && (tmpfloat = cfg_get_float("defaults", "framerate")) > -1)
		fps = tmpfloat;

	if (!preset && (tmpstr = cfg_get_str("defaults", "channel")))
		preset = strdup(tmpstr);

	if ( (tmpint = cfg_get_int("defaults", "verbose")) > -1)
		verbose = 1;
	
	if (audio_bitrate < 0 &&
		(tmpint = cfg_get_int("defaults", "audiobitrate")) > -1)
	{
		audio_bitrate = tmpint;
	}

	if (vertical_flip < 0 &&
		(tmpint = cfg_get_int("defaults", "vflip")) > -1)
	{
		vertical_flip = !tmpint;
	}

	if (!rectime_str && (tmpstr = cfg_get_str("defaults", "rectime")))
		rectime_str = strdup(tmpstr);

	if (!freqtab && (tmpstr = cfg_get_str("defaults", "freqtab")))
		freqtab = strdup(tmpstr);

	if (!window_str && (tmpstr = cfg_get_str("defaults", "window")))
		window_str = strdup(tmpstr);

	if (!codec_preset && (tmpstr = cfg_get_str("defaults", "codec-preset")))
		codec_preset = strdup(tmpstr);

	if (!init_sound && (tmpstr = cfg_get_str("defaults", "init-sound")))
		init_sound = strdup(tmpstr);

	if (!sound_device && (tmpstr = cfg_get_str("defaults", "sound-device")))
		sound_device = strdup(tmpstr);

	if (splitsize < 0 &&
		(tmpint = cfg_get_int("defaults", "splitsize")) > -1)
	{
		splitsize = tmpint;
	}
	return 0;
}

char **
split_string(const char *tmpvals)
{
	unsigned int
		i = 0, curval_pos = 0, val_index = 0;
	char
		**values = NULL,
		*tmpval2 = NULL,
		prev = '\0';
	
	
	if (!tmpvals)
	{
		values = (char**)realloc(values, sizeof(char*));
		values[0] = NULL; //last element in the array.
		return values;
	}

	values = (char**)malloc(1 * sizeof(char*));
	values[val_index] = (char*)calloc(strlen(tmpvals)+1, sizeof(char));
	while (i < strlen(tmpvals))
	{
		char curr = tmpvals[i];
		if (prev == '\\') 
		{
			//previous read char was a backslash. That implies that this 
			//character is treated a 'special' character. Currently only
			//comma and backslash itself.
			switch(curr)
			{
			case '\\':
			case ',':
				values[val_index][curval_pos++] = curr;
				break;
			}
			prev = '\0'; //prevent special char interpr. in next loop ;)
		}
		else if (curr == '\\') 
		{
			prev = '\\';
		}
		else if (curr == ',') //keyword separator
		{
			tmpval2 = crop_whitespace(values[val_index]);
			free(values[val_index]);
			values[val_index] = tmpval2;
			curval_pos = 0;
			values = (char**)realloc(values, ((++val_index)+1) * sizeof(char*));
			values[val_index] = (char*)calloc(strlen(tmpvals)+1, sizeof(char));
			prev = ',';
		}
		else //normal character
		{
			values[val_index][curval_pos++] = curr;
			prev = curr;
		}
		i++;
	}

	tmpval2 = crop_whitespace(values[val_index]);
	free(values[val_index]);
	values[val_index] = tmpval2;
	values = (char**)realloc(values, ((++val_index)+1) * sizeof(char*));
	values[val_index] = NULL; //last element in the array.

	return values;
}

/* handle warnings. Currently simply displays warning on stderr. */
void
vcr_warning(const char *txt)
{
	cerr << "[WARNING] " << txt << endl;
}

void
vcr_notice(const char *txt)
{
	if (verbose)
		cout << "[NOTICE]  " << txt << endl;
}

void
vcr_notice(string& txt)
{
	vcr_notice(txt.c_str());
}

void
vcr_notice(ostringstream& txt)
{
	vcr_notice(txt.str().c_str());
}

/* sets global error var and returns -1 */
short
set_error(const char*txt)
{
	if (!txt)
		return -1;

	error = new char[strlen(txt)+1];
	strcpy(error, txt);
	return -1;
}

/* show global error on stderr */
void
show_error()
{
	cerr << error << endl;
}

int
set_rectime(const char *txt)
{
	if (!txt || !strlen(txt))
		return 0;

	char c = txt[strlen(txt)-1];
	int mult = 1;

	switch (c)
	{
	case 'H' :
	case 'h' : mult = 60 * 60; break;  // hours
	case 'M' :
	case 'm' : mult = 60;      break;  // minutes
	case 'S' :
	case 's' : mult = 1;       break;  // seconds
	default: 
		if (!isdigit((int)c))
		{
			cerr << "invalid arg to -t" << endl;
			end_program(1);
		}
		mult = 60; //default => minutes
	}
	float o;
	sscanf(txt, "%f", &o);
	rectime = (int) (o * mult);
	if (rectime < 0)
		rectime = -1; //unltd. time.
	
	return rectime;
}

int
main(int argc, char *argv[])
{
	int
		i;
	prog_name = strdup(argv[0]);

	cout << endl << 
		"-------------------------------------------------------------" << endl <<
		"VCR " << VERSION << ", (C) 2001 Bram Avontuur (bram@vcr.avontuur.org)" <<
		endl <<
#ifdef WANT_V4L_DEBUG
		"Configured with --with-debug" << endl <<
#else
		"" <<
#endif
		"-------------------------------------------------------------" << endl;

	init_globals();

	parse_arguments(argc, argv);

	if (want_debug)
		init_debug();

	if (parse_config_file() < 0)
	{
		cerr << error << endl;
		end_program(1);
	}

	/* these 'list' checks can only be performed after parsing the config
	 * file.
	 */
	if (codec_preset && !strcasecmp(codec_preset, "list"))
	{
		list_codec_presets();
		exit(0);
	}

	if (preset && !strcasecmp(preset, "list"))
	{
		list_presets();
		exit(0);
	}

	if (!filename)
	{
		cerr << "No filename argument found. Type 'vcr -h' for instructions." <<
			endl;
    end_program(1);
	}

	//initialize video4linux interface[s]
	init_v4l();

	if (!v4l)
	{
		cerr << "Couldn't open v4l device." << endl;
		end_program(1);
	}

	BITMAPINFOHEADER bih;
    bih.biCompression = 0xffffffff;
    // just to fill video_codecs list
    Creators::CreateVideoDecoder(bih, 0, 0);
	avm::vector<const CodecInfo*> codecList;
	CodecInfo::Get(codecList);

	verify_globals();

	//display recording info
	cout << " Filename: " << filename << endl;
	cout << "    Video: codec: " << codec_name << ", quality: " <<
		quality << endl;
	cout << "                  ";

	for (i = 0; i < attr_count; i++)
	{
		if (i)
			cout << ", ";
		cout << attributes[i].name << " = " << attributes[i].val;
	}
	if (!i)
		cout << "Default attributes will be used.";
	else
		cout << ".";

	cout << endl;
// todo: show video size.
	cout << "           time : ";
	if (rectime > 0)
	 cout << rectime << " s";
	else
		cout << "unlimited";
	cout << ", fps: " << fps << ", keyframes: " << keyframes << endl;
	cout << "           size : " << resolution_width << "x" <<
		resolution_height << " (WxH)";
	if (window.width != resolution_width || window.height != resolution_height)
	{
		cout << " downsized to " << window.width << "x" << window.height <<
			" at offset (" << window.x << "," << window.y << ") (x,y)";
	}
	cout << endl;

	cout << "    Audio: codec: ";
	if (audio_bitrate)
		cout << "MP3 @ " << (audio_bitrate * 8) / 1000 << " kbps";
	else
		cout << "none (uncompressed)";
	cout << endl;

	cout << "      Out: " << audio_freq << " kHz, " << audio_size <<
		" bits (" << (soundchan == Stereo ? "stereo" : "mono") << ")" << endl;

	cout << endl << "Starting to record...type q+enter to quit." << endl;

	signal(SIGINT, stop_record_handler);

	cap = new CaptureProcess(
		v4l,
		filename,
		(splitsize > -1 ? splitsize * 1024 * 1024 : -1),
		codecID,
		quality * 100,
		keyframes,
		soundfreq,
		samplesize,
		soundchan,
		videores,
		rectime,
		-1,
		fps,
		window,
		(audio_bitrate ? 0x55 : 0), //mp3 codec or uncompressed audio
		audio_bitrate,
		vertical_flip,
		sound_device);

	short ready = 0;
	fd_set readfds;
	struct timeval tv;

	int
		old_capdrop = 0,
		old_compdrop = 0;
	time_t
		start_time = time(0);

	if (start_time == (time_t)-1)
	{
		perror("time");
		end_program(1);
	}
	
	while (!ready)
	{
		int
			processed,
			cap_drop = 0,
			comp_drop = 0,
			finished;
    char str[256];

		cap->getState(processed, cap_drop, comp_drop, finished);

		if (verbose)
		{
			if (cap_drop > old_capdrop || comp_drop > old_compdrop)
			{
				time_t t = time(0);
				if (cap_drop > old_capdrop)
				{
					cerr << ctime(&t) << " Dropped " << (cap_drop - old_capdrop) << 
						" frame(s) in capture" << endl;
					old_capdrop = cap_drop;
				}
				else if (comp_drop > old_compdrop)
				{
					cerr << ctime(&t) << " Dropped " << comp_drop - old_compdrop << 
						" frame(s) in encoder" << endl;
					old_compdrop = comp_drop;
				}
			}
		}

		if (want_debug)
		{
    	sprintf(str, "Captured: %d frames, dropped in capture: %d, dropped "\
				"in encoder: %d\n", processed, cap_drop, comp_drop);
			vcr_debug(str);
		}

		tv.tv_sec = 0;
		tv.tv_usec = 0;
		FD_ZERO(&readfds);
		FD_SET(0, &readfds); //add stdin
		if (select(1, &readfds, NULL, NULL, &tv))
		{
			char input[81];
			ssize_t readbytes = read(0, (void*)input, 80);
			if (readbytes > 0)
			{
				char cmd = input[0];

				if (cmd == 'q')
					finished = 1;
			}
		}
		ready = finished;		
		if (ready)
		{
			cout << "Captured " << processed << " frames." << endl;
			
			if (processed)
			{
				cout <<
				"\tFrames dropped in capture: " << cap_drop << " (" <<
				cout.precision(4) << (double)(100 * cap_drop / processed) << "%)" <<
				endl << 
				"\tframes dropped in encoder: " << comp_drop << " (" <<
				cout.precision(4) << (double)(100 * comp_drop / processed) <<
				"%)" << endl;
			}
		}
		else
			sleep(1);
	}
	
	cout << "Exiting.." << endl;
	end_program(0);
}

// vim:ts=2:sw=2
