/* **********************************************************
 * Copyright 1998-2004 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 * **********************************************************/

/*
 * vm_assert.h --
 *
 *	The basic assertion facility for all VMware code.
 *
 *	For proper use, see
 *	http://vmweb.vmware.com/~mts/WebSite/guide/programming/asserts.html
 */

#ifndef _VM_ASSERT_H_
#define _VM_ASSERT_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMNIXMOD
#define INCLUDE_ALLOW_VMKERNEL
#define INCLUDE_ALLOW_VMK_MODULE
#define INCLUDE_ALLOW_DISTRIBUTE
#define INCLUDE_ALLOW_VMCORE
#include "includeCheck.h"

// XXX not necessary except some places include vm_assert.h improperly
#include "vm_basic_types.h"
#include "vm_basic_defs.h"


/*
 * XXX old file code
 */

#ifdef FILECODEINT
#error "Don't define FILECODEINT.  It is obsolete."
#endif
#ifdef FILECODE
#error "Don't define FILECODE.  It is obsolete."
#endif


/*
 * Panic and log functions
 */

EXTERN void Log(const char *fmt, ...) PRINTF_DECL(1, 2);
EXTERN void Warning(const char *fmt, ...) PRINTF_DECL(1, 2);
EXTERN NORETURN void Panic(const char *fmt, ...) PRINTF_DECL(1, 2);

EXTERN void LogThrottled(uint32 *count, const char *fmt, ...)
            PRINTF_DECL(2, 3);
EXTERN void WarningThrottled(uint32 *count, const char *fmt, ...)
            PRINTF_DECL(2, 3);

/* DB family:  messages which are parsed by logfile database system */
#define WarningDB Warning
#define LogDB Log
#define WarningThrottledDB WarningThrottled
#define LogThrottledDB LogThrottled


/*
 * Monitor source location
 *
 * To give the densest expansion of ASSERT() (and friends),
 * the monitor encodes source location in 32 bits.  This is
 * made possible by source rewriting (see make/mk/vmm-strings.pl).
 * The script converts the construct @__FILE__@ in the source
 * (see below) into a small constant, which can be converted
 * back to the file name with ASSERT_MONSRCFILE().
 *
 * -- edward
 */

typedef uint32 Assert_MonSrcLoc;

#define ASSERT_MONSRCFILETABLE(loc)	((uint32) (loc) >> 31)
#define ASSERT_MONSRCFILEINDEX(loc)	((loc) >> 16 & 0x7fff)
#define ASSERT_MONSRCLINE(loc)		LOWORD(loc)

#define ASSERT_NULL_MONSRCLOC	  0		// there is never line 0
#define ASSERT_ILLEGAL_MONSRCLOC  0xffffffff	// and never 32767 files

#ifdef VMM // {
#ifdef MONITOR_APP // {

#define ASSERT_MONSRCLOC()	ASSERT_NULL_MONSRCLOC

#else // } {

#ifdef VMCORE
#define ASSERT_MONSRCLOC()	(0 << 31 | @__FILE__@ << 16 | __LINE__)
#else
#define ASSERT_MONSRCLOC()	(1 << 31 | @__FILE__@ << 16 | __LINE__)
#endif

#define ASSERT_MONSRCFILE(loc) \
   (AssertMonSrcFiles[ASSERT_MONSRCFILETABLE(loc)] \
		     [ASSERT_MONSRCFILEINDEX(loc)])

extern char const * const * const AssertMonSrcFiles[];

#endif // }
#endif // }


/*
 * Stress testing: redefine ASSERT_IFNOT() to taste
 */

#ifndef ASSERT_IFNOT
   #ifdef __cplusplus
      #define ASSERT_IFNOT(cond, panic) (UNLIKELY(!(cond)) ? (panic) : (void)0)
   #else
      #define ASSERT_IFNOT(cond, panic) (UNLIKELY(!(cond)) ? (panic) : 0)
   #endif
#endif


/*
 * Assert, panic, and log macros
 *
 * Some of these are redefined below undef !VMX86_DEBUG.
 * ASSERT() is special cased because of interaction with Windows DDK.
 */

#if defined VMX86_DEBUG || defined ASSERT_ALWAYS_AVAILABLE
#undef ASSERT
#define ASSERT(cond) \
           ASSERT_IFNOT(cond, _ASSERT_PANIC(AssertAssert))
#endif
#define ASSERT_BUG(bug, cond) \
           ASSERT_IFNOT(cond, _ASSERT_PANIC_BUG(bug, AssertAssert))
#define ASSERT_BUG_DEBUGONLY(bug, cond) ASSERT_BUG(bug, cond)

#define PANIC()        _ASSERT_PANIC(AssertPanic)
#define PANIC_BUG(bug) _ASSERT_PANIC_BUG(bug, AssertPanic)

#define ASSERT_NOT_IMPLEMENTED(cond) \
           ASSERT_IFNOT(cond, NOT_IMPLEMENTED())
#define ASSERT_NOT_IMPLEMENTED_BUG(bug, cond) \
           ASSERT_IFNOT(cond, NOT_IMPLEMENTED_BUG(bug))

#define NOT_IMPLEMENTED()        _ASSERT_PANIC(AssertNotImplemented)
#define NOT_IMPLEMENTED_BUG(bug) _ASSERT_PANIC_BUG(bug, AssertNotImplemented)

#define NOT_REACHED()            _ASSERT_PANIC(AssertNotReached)
#define NOT_REACHED_BUG(bug)     _ASSERT_PANIC_BUG(bug, AssertNotReached)

#define ASSERT_MEM_ALLOC(cond) \
           ASSERT_IFNOT(cond, _ASSERT_PANIC(AssertMemAlloc))

#ifdef VMX86_DEVEL
   #define ASSERT_LENGTH(real, expected) \
              ASSERT_IFNOT((real) == (expected), \
                 Panic(AssertLengthFmt, __FILE__, __LINE__, real, expected))
#else
   #define ASSERT_LENGTH(real, expected) ASSERT((real) == (expected))
#endif

#ifdef VMX86_DEVEL
   #define ASSERT_DEVEL(cond) ASSERT(cond)
#else
   #define ASSERT_DEVEL(cond) ((void) 0)
#endif

#define ASSERT_NO_INTERRUPTS()  ASSERT(!INTERRUPTS_ENABLED())
#define ASSERT_HAS_INTERRUPTS() ASSERT(INTERRUPTS_ENABLED())

#define ASSERT_LOG_UNEXPECTED(bug, cond) \
           (UNLIKELY(!(cond)) ? LOG_UNEXPECTED(bug) : 0)
#ifdef VMX86_DEVEL
   #define LOG_UNEXPECTED(bug) \
              Warning(AssertUnexpectedFmt, __FILE__, __LINE__, bug)
#else
   #define LOG_UNEXPECTED(bug) \
              Log(AssertUnexpectedFmt, __FILE__, __LINE__, bug)
#endif

#define ASSERT_NOT_TESTED(cond) (UNLIKELY(!(cond)) ? NOT_TESTED() : 0)
#ifdef VMX86_DEVEL
   #define NOT_TESTED() Warning(AssertNotTestedFmt, __FILE__, __LINE__)
#else
   #define NOT_TESTED() Log(AssertNotTestedFmt, __FILE__, __LINE__)
#endif

#define NOT_TESTED_ONCE()                                               \
   do {                                                                 \
      static Bool alreadyPrinted = FALSE;                               \
      if (UNLIKELY(!alreadyPrinted)) {                                  \
	 alreadyPrinted = TRUE;                                         \
	 NOT_TESTED();                                                  \
      }                                                                 \
   } while (0)

#define NOT_TESTED_1024()                                               \
   do {                                                                 \
      static uint16 count = 0;                                          \
      if (UNLIKELY(count == 0)) { NOT_TESTED(); }                       \
      count = (count + 1) & 1023;                                       \
   } while (0)

#define NOT_TESTED_WARNING() NOT_TESTED()

#define LOG_ONCE(_s)                                                    \
   do {                                                                 \
      static Bool logged = FALSE;                                       \
      if (!logged) {                                                    \
	 Log _s;                                                        \
         logged = TRUE;                                                 \
      }                                                                 \
   } while (0)


/*
 * Redefine macros that are only in debug versions
 */

#if !defined VMX86_DEBUG && !defined ASSERT_ALWAYS_AVAILABLE // {

#undef  ASSERT
#define ASSERT(cond) ((void) 0)

#undef  ASSERT_BUG_DEBUGONLY
#define ASSERT_BUG_DEBUGONLY(bug, cond) ((void) 0)

#undef  ASSERT_LENGTH
#define ASSERT_LENGTH(real, expected) ((void) 0)

/*
 * Expand NOT_REACHED() as appropriate for each situation.
 *
 * Mainly, we want the compiler to infer the same control-flow
 * information as it would from Panic().  Otherwise, different
 * compilation options will lead to different control-flow-derived
 * errors, causing some make targets to fail while others succeed.
 *
 * VC++ has the __assume() built-in function which we don't trust
 * (see bug 43485); gcc has no such construct; we just panic in
 * userlevel code.  The monitor doesn't want to pay the size penalty
 * (measured at 212 bytes for the release vmm for a minimal infinite
 * loop; panic would cost even more) so it does without and lives
 * with the inconsistency.
 */

#ifdef VMM
#undef  NOT_REACHED
#define NOT_REACHED() ((void) 0)
#else
// keep debug definition
#endif

#undef  ASSERT_LOG_UNEXPECTED
#define ASSERT_LOG_UNEXPECTED(bug, cond) ((void) 0)

#undef LOG_UNEXPECTED
#define LOG_UNEXPECTED(bug) ((void) 0)

#undef  ASSERT_NOT_TESTED
#define ASSERT_NOT_TESTED(cond) ((void) 0)
#undef  NOT_TESTED
#define NOT_TESTED() ((void) 0)
#undef  NOT_TESTED_ONCE
#define NOT_TESTED_ONCE() ((void) 0)
#undef  NOT_TESTED_1024
#define NOT_TESTED_1024() ((void) 0)

#endif // !VMX86_DEBUG }


/*
 * Compile-time assertions
 *
 * The "cond" parameter must be a compile-time constant, and when
 * it is 0 (i.e., FALSE) the switch statement will have two 0 cases,
 * which is a compile-time error.
 * See MY_ASSERTS() below for how to use this outside a function.
 */

#define ASSERT_ON_COMPILE(cond) do { \
   switch (0) { \
   case 0: \
   case (cond): \
      ; \
   } \
} while (FALSE)

/*
 * To put an ASSERT_ON_COMPILE() outside a function, wrap it
 * in MY_ASSERTS().  The first parameter must be unique in
 * each .c file where it appears.  For example,
 *
 * MY_ASSERTS(FS3_INT,
 *    ASSERT_ON_COMPILE(sizeof(FS3_DiskLock) == 128);
 *    ASSERT_ON_COMPILE(sizeof(FS3_DiskLockReserved) == DISK_BLOCK_SIZE);
 *    ASSERT_ON_COMPILE(sizeof(FS3_DiskBlock) == DISK_BLOCK_SIZE);
 *    ASSERT_ON_COMPILE(sizeof(Hardware_DMIUUID) == 16);
 * )
 *
 * Caution: ASSERT() within MY_ASSERTS() is silently ignored.
 * The same goes for anything else not evaluated at compile time.
 */

#define MY_ASSERTS(name, assertions) \
   static INLINE void name(void) { \
      assertions \
   }


/*
 * Internal macros, functions, and strings
 *
 * The monitor wants to save space at call sites, so it has specialized
 * functions for each situation.  User level wants to save on implementation
 * so it uses generic functions.
 */

#if defined VMM && !defined MONITOR_APP // {

#define _ASSERT_PANIC(name)          ((name)(ASSERT_MONSRCLOC()))
#define _ASSERT_PANIC_BUG(bug, name) ((name##Bug)(ASSERT_MONSRCLOC(), bug))

EXTERN NORETURN void AssertPanic(Assert_MonSrcLoc loc);
EXTERN NORETURN void AssertAssert(Assert_MonSrcLoc loc);
EXTERN NORETURN void AssertNotImplemented(Assert_MonSrcLoc loc);
EXTERN NORETURN void AssertNotReached(Assert_MonSrcLoc loc);
EXTERN NORETURN void AssertPanicBug(Assert_MonSrcLoc loc, int bug);
EXTERN NORETURN void AssertAssertBug(Assert_MonSrcLoc loc, int bug);
EXTERN NORETURN void AssertNotImplementedBug(Assert_MonSrcLoc loc, int bug);
EXTERN NORETURN void AssertNotReachedBug(Assert_MonSrcLoc loc, int bug);

extern const char AssertLengthFmt[];
extern const char AssertUnexpectedFmt[];
extern const char AssertNotTestedFmt[];

#else // } {

#define _ASSERT_PANIC(name) \
           Panic(_##name##Fmt "\n", __FILE__, __LINE__)
#define _ASSERT_PANIC_BUG(bug, name) \
           Panic(_##name##Fmt " bugNr=%d\n", __FILE__, __LINE__, bug)

#define AssertLengthFmt     _AssertLengthFmt
#define AssertUnexpectedFmt _AssertUnexpectedFmt
#define AssertNotTestedFmt  _AssertNotTestedFmt

#endif // }

// these don't have newline so a bug can be tacked on
#define _AssertPanicFmt            "PANIC %s:%d"
#define _AssertAssertFmt           "ASSERT %s:%d"
#define _AssertNotImplementedFmt   "NOT_IMPLEMENTED %s:%d"
#define _AssertNotReachedFmt       "NOT_REACHED %s:%d"
#define _AssertMemAllocFmt         "MEM_ALLOC %s:%d"

// these are complete formats with newline
#define _AssertLengthFmt           "LENGTH %s:%d r=%#x e=%#x\n"
#define _AssertUnexpectedFmt       "UNEXPECTED %s:%d bugNr=%d\n"
#define _AssertNotTestedFmt        "NOT_TESTED %s:%d\n"


/*
 * Convenience macros and definitions. Can often be used instead of #ifdef.
 * XXX should be in vm_basic_defs.h
 */

#ifdef VMX86_DEBUG
#undef DEBUG_ONLY
#define vmx86_debug   1
#define DEBUG_ONLY(x) x
#else
#undef DEBUG_ONLY
#define vmx86_debug   0
#define DEBUG_ONLY(x)
#endif

#ifdef VMX86_LCH
#define vmx86_lch   1
#define LCH_ONLY(x) x
#else
#define vmx86_lch   0
#define LCH_ONLY(x)
#endif

#ifdef VMX86_STATS
#define vmx86_stats   1
#define STATS_ONLY(x) x
#else
#define vmx86_stats   0
#define STATS_ONLY(x)
#endif

#ifdef VMX86_DEVEL
#define vmx86_devel   1
#define DEVEL_ONLY(x) x
#else
#define vmx86_devel   0
#define DEVEL_ONLY(x)
#endif

#ifdef VMX86_LOG
#define vmx86_log     1
#define LOG_ONLY(x)   x
#else
#define vmx86_log     0
#define LOG_ONLY(x)
#endif

#ifdef VMX86_VMM_SERIAL_LOGGING
#define vmx86_vmm_serial_log     1
#define VMM_SERIAL_LOG_ONLY(x)   x
#else
#define vmx86_vmm_serial_log     0
#define VMM_SERIAL_LOG_ONLY(x)
#endif

#ifdef VMX86_SERVER
#define vmx86_server 1
#define SERVER_ONLY(x) x
#define HOSTED_ONLY(x)
#else
#define vmx86_server 0
#define SERVER_ONLY(x)
#define HOSTED_ONLY(x) x
#endif

#ifdef VMKERNEL
#define vmkernel 1
#define VMKERNEL_ONLY(x) x
#else
#define vmkernel 0
#define VMKERNEL_ONLY(x)
#endif

#ifdef VMX86_ENTERPRISE_DESKTOP
#define vmx86_wsee 1
#define WSEE_ONLY(x) x
#else
#define vmx86_wsee 0
#define WSEE_ONLY(x)
#endif

#ifdef WIN32
#define WIN32_ONLY(x) x
#define POSIX_ONLY(x)
#else
#define WIN32_ONLY(x)
#define POSIX_ONLY(x) x
#endif

#ifdef VMM
#define VMM_ONLY(x) x
#define USER_ONLY(x)
#else
#define VMM_ONLY(x)
#define USER_ONLY(x) x
#endif

#ifdef VMVISOR
#define vmvisor 1
#define VMVISOR_ONLY(x) x
#else
#define vmvisor 0
#define VMVISOR_ONLY(x)
#endif


#endif /* ifndef _VM_ASSERT_H_ */
