/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * phystrack.c --
 *
 *    track down the utilization of the physical pages
 *    
 *    N.B: used only for debugging, does not provide any 
 *         functionality
 */


#ifdef linux
/* Must come before any kernel header file --hpreg */
#   include "driver-config.h"

#   include <linux/string.h> /* memset() in the kernel */
#elif defined(WINNT_DDK)
#   include <string.h>
#else
#   error "Unknown platform"
#endif

#include "vmware.h"
#include "vmx86.h"
#include "hostif.h"


#define PHYSTRACK_PAGES_PER_ENTRY (32*1024)
#define PHYSTRACK_MAX_ENTRIES     (512) /* support 64 GB hosts, maximum for the current IA32 PAE */
#define BYTES_PER_ENTRY (PHYSTRACK_PAGES_PER_ENTRY/8)


typedef struct PhysTracker {
   int numVMs;
   void *dir[PHYSTRACK_MAX_ENTRIES];
} PhysTracker;

/* The global physTracker pointer used by PhysTrack_Init */
PhysTracker *physTracker = NULL;

/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Alloc --
 *
 *      Create new PhysTracker.
 *
 * Results:
 *      
 *      Creates new PhysTracker.
 *      
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
PhysTracker *
PhysTrack_Alloc(void)
{
   PhysTracker *tracker;

   /* allocate a new phystracker */
   tracker = HostIF_AllocKernelMem(sizeof *tracker, FALSE);
   if (tracker) {
      memset(tracker, 0, sizeof *tracker);
      tracker->numVMs = 1;
   } else {
      Warning("PhysTrack_Alloc failed\n");
   }

   return tracker;
} 


/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Init --
 *
 *      module initialization 
 *
 * Results:
 *      
 *      returns the PhysTracker. creates one if does not exist
 *      
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
PhysTracker *
PhysTrack_Init(void)
{
   /* allocate a new phystracker */
   if (physTracker == NULL) {
      physTracker = HostIF_AllocKernelMem(sizeof *physTracker, FALSE);
      if (physTracker) {
         memset(physTracker, 0, sizeof *physTracker);
      }
   }

   /* increment use count */
   if (physTracker) {
      physTracker->numVMs++;
   } else {
      Warning("PhysTrack_Init failed\n");
   }

   return physTracker;
} 



/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Cleanup --
 *
 *      module deallocation
 *
 * Results:
 *      
 *      reallocates all structures, including 'tracker'
 *      
 *      
 *
 * Side effects:
 *      tracker deallocated
 *
 *----------------------------------------------------------------------
 */
void
PhysTrack_Cleanup(PhysTracker *tracker) 
{ 
   int i, j;

   ASSERT(tracker);

   /* decrement use count */
   tracker->numVMs--;	

   /* deallocate phystracker if no more VMs */
   if (tracker->numVMs == 0) {
      for (i=0;i<PHYSTRACK_MAX_ENTRIES;i++) { 
	 if (tracker->dir[i]) { 
	    for (j = 0; j < BYTES_PER_ENTRY; j++) {
	       uint8 *bitvector = (uint8 *)tracker->dir[i];
	       if (bitvector[j] != 0) {
		  Warning("PhysTrack_Cleanup: pfns still locked\n");
		  ASSERT(0);
	       }
	    }
	    HostIF_FreePage(tracker->dir[i]);
	    tracker->dir[i] = NULL;
	 }
      }
      HostIF_FreeKernelMem(tracker);
      if (tracker == physTracker) {
         physTracker = NULL;
      }
   }
}


/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Add --
 *
 *      add a page to the core map tracking.
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      Fatal if the page is already tracked.
 *
 *----------------------------------------------------------------------
 */

void
PhysTrack_Add(PhysTracker *tracker,
              MPN mpn)
{
   unsigned int dir = mpn / PHYSTRACK_PAGES_PER_ENTRY;
   unsigned int off = mpn % PHYSTRACK_PAGES_PER_ENTRY;
   unsigned int pos = off / 8;
   unsigned int bit = 1 << (off % 8);
   uint8 *bitvector;

   ASSERT(tracker);
   ASSERT(dir < PHYSTRACK_MAX_ENTRIES);

   if (tracker->dir[dir] == NULL) { 
      // more efficient with page alloc
      ASSERT(BYTES_PER_ENTRY == PAGE_SIZE);
      tracker->dir[dir] = HostIF_AllocPage();
      if (tracker->dir[dir] == NULL) { 
         return;
      }
      memset(tracker->dir[dir], 0, BYTES_PER_ENTRY);
   }
   
   bitvector = tracker->dir[dir];
   
   if (bitvector[pos] & bit) {
      PANIC();
   }
   
   bitvector[pos] |= bit;
}



/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Remove --
 *
 *      remove a page from the core map tracking
 *
 * Results:
 *      
 *      void

 *
 * Side effects:
 *      Fatal if the page is not tracked
 *
 *----------------------------------------------------------------------
 */

void
PhysTrack_Remove(PhysTracker *tracker,
                 MPN mpn)
{
   unsigned int dir = mpn / PHYSTRACK_PAGES_PER_ENTRY;
   unsigned int off = mpn % PHYSTRACK_PAGES_PER_ENTRY;
   unsigned int pos = off / 8;
   unsigned int bit = 1 << (off % 8);
   uint8 *bitvector;

   ASSERT(tracker);
   ASSERT(dir < PHYSTRACK_MAX_ENTRIES);

   bitvector = tracker->dir[dir];
   if (bitvector == NULL) {
      PANIC();
   }
   if (!(bitvector[pos] & bit)) {
      PANIC();
   }

   bitvector[pos] &= ~bit;
}


/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Test --
 *
 *      tests whether a page is being tracked
 *
 * Results:
 *      
 *      TRUE if the page is tracked
 *      FALSE otherwise
 *      
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Bool
PhysTrack_Test(const PhysTracker *tracker,
               MPN mpn)
{
   unsigned int dir = mpn / PHYSTRACK_PAGES_PER_ENTRY;
   unsigned int off = mpn % PHYSTRACK_PAGES_PER_ENTRY;
   unsigned int pos = off / 8;
   unsigned int bit = 1 << (off % 8);
   uint8 *bitvector;

   ASSERT(tracker);
   /*
    * Can't just assert() here: this fuction is used to validate arguments 
    * passed from user mode.
    */ 
   if (dir >= PHYSTRACK_MAX_ENTRIES) {
      return FALSE;
   }

   bitvector = tracker->dir[dir];
   if (bitvector == NULL) { 
      return FALSE;
   }
   if (bitvector[pos] & bit) {
      return TRUE;
   } else {
      return FALSE;
   }
}


/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_GetNext --
 *
 *      Return next tracked page
 *
 * Results:
 *      
 *      MPN         when some tracked page was found
 *      INVALID_MPN otherwise
 *      
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

MPN
PhysTrack_GetNext(const PhysTracker *tracker,
                  MPN mpn)
{
   unsigned int dir;

   mpn++; /* We want the next MPN. */

   dir = mpn / PHYSTRACK_PAGES_PER_ENTRY;
   ASSERT(tracker);
   ASSERT(dir < PHYSTRACK_MAX_ENTRIES);
   for (; dir < PHYSTRACK_MAX_ENTRIES; dir++) {
      const MPN lastMpnInDir = (dir+1) * PHYSTRACK_PAGES_PER_ENTRY - 1;

      if (tracker->dir[dir] == NULL) {
	 continue;
      }
      /* 
       * Performance is not critical here so we can use
       * the generic test function PhysTrack_Test().
       */
      for (; mpn <= lastMpnInDir; mpn++) {
	 if (PhysTrack_Test(tracker, mpn)) {
	    return mpn;
	 }
      }
   }
   return INVALID_MPN;
}
