/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

#ifndef __DRIVER_H__
#define __DRIVER_H__

#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMCORE
#include "includeCheck.h"

#include <linux/sched.h>
#include <linux/miscdevice.h>

#include "machine.h"
#include "vmx86.h"
#include "vm_time.h"
#include "compat_version.h"
#include "compat_spinlock.h"
#include "compat_wait.h"
#include "driver_vmcore.h"
#include "vmx86.h"


/*
 * Per-instance driver state
 */

struct VMDriver;
struct VMHostMapping;


/* Range of memory pages --hpreg */
typedef struct MemRange {
   /* Next item in the list --hpreg */
   struct MemRange *next;

   /* Kernel virtual address of the beginning of the range --hpreg */
   VA kAddr;

   /* Physical address of the beginning of the range --hpreg */
   PA pAddr;

   /* Number of allocated pages --hpreg */
   unsigned long numPages;
} MemRange;
 
/* 16 pages (64KB) looks as a good limit for one allocation */
#define VMMON_MAX_LOWMEM_PAGES	16

typedef struct VMLinux {
   struct VMLinux *next;
   struct VMDriver *vm;

   /*
    * The semaphore protect accesses to size4Gb and pages4Gb
    * in mmap(). mmap() may happen only once, and all other
    * accesses except cleanup are read-only, and may happen
    * only after successful mmap.
    */
   struct semaphore lock4Gb;
   unsigned int size4Gb;
   unsigned int order4Gb;
   struct page *pages4Gb[VMMON_MAX_LOWMEM_PAGES];
   unsigned int sizeLocked;
   struct VMHostMapping *pagesLocked;
   VA baseLocked;

   enum vmEmulations version;
   int maySetUid;
   /*
    * LinuxDriverPoll() support
    */

   wait_queue_head_t pollQueue;
   volatile uint32 *pollTimeoutPtr;
   struct page *pollTimeoutPage;
   VmTimeType pollTime;
   struct VMLinux *pollForw;
   struct VMLinux **pollBack;
} VMLinux;

extern struct vm_operations_struct vmuser_locked_mops;
#define VMHOST_MAPPING_PD	(PAGE_SIZE / sizeof(void*))
#define VMHOST_MAPPING_PT	(PAGE_SIZE / sizeof(void*))
struct VMHostEntry {
	struct page *ent[VMHOST_MAPPING_PT];
};

struct VMHostMapping {
	struct page *ent[VMHOST_MAPPING_PD];
};



/*
 * Static driver state.
 */

#define VM_DEVICE_NAME_SIZE 32
#define LINUXLOG_BUFFER_SIZE  1024

typedef struct VMXLinuxState {
   int major;
   int minor;
   struct miscdevice misc;
   VmTimeStart startTime;	/* Used to compute kHz estimate */
   char deviceName[VM_DEVICE_NAME_SIZE];
   char buf[LINUXLOG_BUFFER_SIZE];
   VMLinux *head;

   /*
    * for LinuxDriverPoll()
    */

   struct timer_list pollTimer;
   wait_queue_head_t pollQueue;

   struct VMLinux *pollList;
#ifdef POLLSPINLOCK
   spinlock_t pollListLock;
#endif

   volatile int fastClockThread;
   unsigned fastClockRate;

   /*
    * Singly linked list of ranges of allocated memory pages whose physical
    * address is < 4 GB. The lock protect accesses to any field of any item in
    * the list --hpreg
    */
   MemRange oldList4Gb;
   spinlock_t oldLock4Gb;
} VMXLinuxState;

extern VMXLinuxState linuxState;

#endif
