#include "driver-config.h"
#include <linux/string.h> /* memset() in the kernel */
#include <linux/sched.h> /* jiffies from the kernel */

#include "vmware.h"
#include "machine.h"
#include "modulecall.h"
#include "vmx86.h"
#include "task.h"
#include "initblock.h"
#include "vm_asm.h"
#include "iocontrols.h"
#include "hostif.h"
#include "memtrack.h"
#include "modulecall_compat.h"

#define USERCALL_V45_FASTCLOCKRATE	(327) 


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_RunVM_V55  --
 *
 *      Main interaction between the module and the monitor:
 *
 *	   Run the monitor
 *	   Process module calls from the monitor
 *	   Make cross user calls to the main thread
 *	   Return to userlevel to process normal user calls
 *	   and to signal timeout or errors.
 *
 * Results:
 *      Positive: user call number.
 *	-1: error (currently not used).
 *
 * Side effects:
 *      Not really.
 *
 *----------------------------------------------------------------------
 */

int
Vmx86_RunVM_S1B1(VMDriver *vm, Vcpuid vcpuid)
{
   uint32 retval = MODULECALL_V55_USERRETURN;
   VMCrossPageS1B1 *crosspage = vm->crosspage[vcpuid];

   ASSERT(crosspage);

   /*
    * Check if we were interrupted by signal.
    */
   if (crosspage->moduleCallInterrupted) {
      crosspage->moduleCallInterrupted = FALSE;
      goto skipTaskSwitch;
   }
   for (;;) {

      crosspage->retval = retval;

      /*
       * Task_Switch changes the world to the monitor.
       * The monitor is waiting in the BackToHost routine.
       */
      UCTIMESTAMP(crosspage, SWITCHING_TO_MONITOR);
      Task_Switch_S1B1(vm, vcpuid);
      UCTIMESTAMP(crosspage, SWITCHED_TO_MODULE);

skipTaskSwitch:;

      retval = MODULECALL_V55_USERRETURN;

      if (crosspage->userCallType != MODULECALL_USERCALL_NONE) {
         /*
          * This is the main user call path.
          *
          * There are two kinds of user calls.  Normal ones
          * are handled by the calling VCPU thread itself.
          * We just return from here (back to userlevel)
          * in that case.
          *
          * Calls marked userCallCross are handled by the main
          * VMX thread.  In this case, the userCallRequest field
          * indicates to the VMX that this VCPU wants to make
          * a user call.  This field may be consulted by the VMX
          * at any time (specifically when the VMX is awakened by
          * another VCPU), so it must be set after the other
          * user call arguments.  The VMX is responsible for
          * resetting this field and awakening the VCPU when
          * the user call is complete, via the ACK_USER_CALL
          * and COMPLETE_USER_CALL ioctl.  The latter implies
          * the former.
          *
          * When and how to use ACK_USER_CALL and COMPLETE_USER_CALL
          * are at the discretion of the VMX.  In particular,
          * COMPLETE_USER_CALL does not imply that the requested
          * operation has fully completed, only that the VCPU can
          * continue.  See the comment in MonitorLoopHandleUserCall()
          * for use scenarios.
          *
          * See also comment at MonitorLoopUserCallPoll().
          *
          * -- edward
          */

         if (!crosspage->userCallCross) {
            ASSERT(!crosspage->userCallRestart);
            return crosspage->userCallType;
         }

	 if (!crosspage->userCallRestart) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	    crosspage->userCallRequest = crosspage->userCallType;
	    UCTIMESTAMP(crosspage, AWAKENING_VMX);
	    HostIF_UserCall(vm, vcpuid);
	 }

	 UCTIMESTAMP(crosspage, GOING_TO_SLEEP);
         if (HostIF_UserCallWait(vm, vcpuid, USERCALL_TIMEOUT)) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	 } else {
	    retval = MODULECALL_V55_USERTIMEOUT;
	 }
	 UCTIMESTAMP(crosspage, AWAKE);
      }

      switch (crosspage->moduleCallType) {

      case MODULECALL_V55_NONE:
         break;

      case MODULECALL_V55_INTR:    // Already done in task.c
         break;

      case MODULECALL_V55_GET_RECYCLED_PAGE: {
	 MPN32 mpn;
	 
	 retval = (Vmx86_AllocLockedPages(vm, &mpn, 1, TRUE) == 1) ? 
	    mpn : INVALID_MPN;
	 break;
      }

      case MODULECALL_V55_SEMAWAIT:
         retval = HostIF_SemaphoreWait(vm, vcpuid, crosspage->args[0],
	                               crosspage->args[1]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_V55_SEMASIGNAL:
         retval = HostIF_SemaphoreSignal(crosspage->args[0]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_V55_SEMAFORCEWAKEUP: {
         HostIF_SemaphoreForceWakeup(vm, (Vcpuid) crosspage->args[0]);
         break;
      }
      case MODULECALL_V55_IPI:
         retval = HostIF_IPI(vm, (VCPUSet)(uintptr_t)crosspage->args[0], TRUE);
         break;

      case MODULECALL_V55_RELEASE_ANON_PAGE: {
         MPN32 mpn = (MPN32)crosspage->args[0];
	 retval = Vmx86_FreeLockedPages(vm, &mpn, 1, TRUE);
         break;
      }

      case MODULECALL_V55_IS_ANON_PAGE: {
         MPN32 mpn = (MPN32)crosspage->args[0];
         retval = Vmx86_IsAnonPage(vm, mpn);
         break;
      }

      case MODULECALL_V55_SWITCH_TO_PEER: {
         crosspage->runVmm64 = !crosspage->runVmm64;
         break;
      }

      default:
         Warning("ModuleCall %d not supported\n", crosspage->moduleCallType);
      }

      /*
       * Give other threads and processes a chance.
       *
       * This is important when we are not preemptable while
       * in the driver and the monitor (e.g., Linux).
       */

      HostIF_Reschedule();
   }

   NOT_REACHED();
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_RunVM_V55  --
 *
 *      Main interaction between the module and the monitor:
 *
 *	   Run the monitor
 *	   Process module calls from the monitor
 *	   Make cross user calls to the main thread
 *	   Return to userlevel to process normal user calls
 *	   and to signal timeout or errors.
 *
 * Results:
 *      Positive: user call number.
 *	-1: error (currently not used).
 *
 * Side effects:
 *      Not really.
 *
 *----------------------------------------------------------------------
 */

int
Vmx86_RunVM_V55(VMDriver *vm, Vcpuid vcpuid)
{
   uint32 retval = MODULECALL_V55_USERRETURN;
   VMCrossPageV55 *crosspage = vm->crosspage[vcpuid];

   ASSERT(crosspage);

   /*
    * Check if we were interrupted by signal.
    */
   if (crosspage->moduleCallInterrupted) {
      crosspage->moduleCallInterrupted = FALSE;
      goto skipTaskSwitch;
   }
   for (;;) {

      crosspage->retval = retval;

      /*
       * Task_Switch changes the world to the monitor.
       * The monitor is waiting in the BackToHost routine.
       */
      UCTIMESTAMP(crosspage, SWITCHING_TO_MONITOR);
      Task_Switch_V55(vm, vcpuid);
      UCTIMESTAMP(crosspage, SWITCHED_TO_MODULE);

skipTaskSwitch:;

      retval = MODULECALL_V55_USERRETURN;

      if (crosspage->userCallType != MODULECALL_USERCALL_NONE) {
         /*
          * This is the main user call path.
          *
          * There are two kinds of user calls.  Normal ones
          * are handled by the calling VCPU thread itself.
          * We just return from here (back to userlevel)
          * in that case.
          *
          * Calls marked userCallCross are handled by the main
          * VMX thread.  In this case, the userCallRequest field
          * indicates to the VMX that this VCPU wants to make
          * a user call.  This field may be consulted by the VMX
          * at any time (specifically when the VMX is awakened by
          * another VCPU), so it must be set after the other
          * user call arguments.  The VMX is responsible for
          * resetting this field and awakening the VCPU when
          * the user call is complete, via the ACK_USER_CALL
          * and COMPLETE_USER_CALL ioctl.  The latter implies
          * the former.
          *
          * When and how to use ACK_USER_CALL and COMPLETE_USER_CALL
          * are at the discretion of the VMX.  In particular,
          * COMPLETE_USER_CALL does not imply that the requested
          * operation has fully completed, only that the VCPU can
          * continue.  See the comment in MonitorLoopHandleUserCall()
          * for use scenarios.
          *
          * See also comment at MonitorLoopUserCallPoll().
          *
          * -- edward
          */

         if (!crosspage->userCallCross) {
            ASSERT(!crosspage->userCallRestart);
            return crosspage->userCallType;
         }

	 if (!crosspage->userCallRestart) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	    crosspage->userCallRequest = crosspage->userCallType;
	    UCTIMESTAMP(crosspage, AWAKENING_VMX);
	    HostIF_UserCall(vm, vcpuid);
	 }

	 UCTIMESTAMP(crosspage, GOING_TO_SLEEP);
         if (HostIF_UserCallWait(vm, vcpuid, USERCALL_TIMEOUT)) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	 } else {
	    retval = MODULECALL_V55_USERTIMEOUT;
	 }
	 UCTIMESTAMP(crosspage, AWAKE);
      }

      switch (crosspage->moduleCallType) {

      case MODULECALL_V55_NONE:
         break;

      case MODULECALL_V55_INTR:    // Already done in task.c
         break;

      case MODULECALL_V55_GET_RECYCLED_PAGE: {
	 MPN32 mpn;
	 
	 retval = (Vmx86_AllocLockedPages(vm, &mpn, 1, TRUE) == 1) ? 
	    mpn : INVALID_MPN;
	 break;
      }

      case MODULECALL_V55_SEMAWAIT:
         retval = HostIF_SemaphoreWait(vm, vcpuid, crosspage->args[0],
	                               crosspage->args[1]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_V55_SEMASIGNAL:
         retval = HostIF_SemaphoreSignal(crosspage->args[0]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_V55_SEMAFORCEWAKEUP: {
         HostIF_SemaphoreForceWakeup(vm, (Vcpuid) crosspage->args[0]);
         break;
      }
      case MODULECALL_V55_IPI:
         retval = HostIF_IPI(vm, (VCPUSet)(uintptr_t)crosspage->args[0], TRUE);
         break;

      case MODULECALL_V55_RELEASE_ANON_PAGE: {
         MPN32 mpn = (MPN32)crosspage->args[0];
	 retval = Vmx86_FreeLockedPages(vm, &mpn, 1, TRUE);
         break;
      }

      case MODULECALL_V55_IS_ANON_PAGE: {
         MPN32 mpn = (MPN32)crosspage->args[0];
         retval = Vmx86_IsAnonPage(vm, mpn);
         break;
      }

      case MODULECALL_V55_SWITCH_TO_PEER: {
         crosspage->runVmm64 = !crosspage->runVmm64;
         break;
      }

      default:
         Warning("ModuleCall %d not supported\n", crosspage->moduleCallType);
      }

      /*
       * Give other threads and processes a chance.
       *
       * This is important when we are not preemptable while
       * in the driver and the monitor (e.g., Linux).
       */

      HostIF_Reschedule();
   }

   NOT_REACHED();
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_RunVM_V5  --
 *
 *      Main interaction between the module and the monitor:
 *
 *	   Run the monitor
 *	   Process module calls from the monitor
 *	   Make cross user calls to the main thread
 *	   Return to userlevel to process normal user calls
 *	   and to signal timeout or errors.
 *
 * Results:
 *      Positive: user call number.
 *	-1: error (currently not used).
 *
 * Side effects:
 *      Not really.
 *
 *----------------------------------------------------------------------
 */

int
Vmx86_RunVM_V5(VMDriver *vm, Vcpuid vcpuid)
{
   uint32 retval = MODULECALL_V5_USERRETURN;
   VMCrossPageV5 *crosspage = vm->crosspage[vcpuid];

   ASSERT(crosspage);

   /*
    * Check if we were interrupted by signal.
    */
   if (crosspage->moduleCallInterrupted) {
      crosspage->moduleCallInterrupted = FALSE;
      goto skipTaskSwitch;
   }
   for (;;) {

      crosspage->retval = retval;

      /*
       * Task_Switch changes the world to the monitor.
       * The monitor is waiting in the BackToHost routine.
       */
      UCTIMESTAMP(crosspage, SWITCHING_TO_MONITOR);
      Task_Switch_V5(vm, vcpuid);
      UCTIMESTAMP(crosspage, SWITCHED_TO_MODULE);

skipTaskSwitch:;

      retval = MODULECALL_V5_USERRETURN;

      if (crosspage->userCallType != MODULECALL_USERCALL_NONE) {
         /*
          * This is the main user call path.
          *
          * There are two kinds of user calls.  Normal ones
          * are handled by the calling VCPU thread itself.
          * We just return from here (back to userlevel)
          * in that case.
          *
          * Calls marked userCallCross are handled by the main
          * VMX thread.  In this case, the userCallRequest field
          * indicates to the VMX that this VCPU wants to make
          * a user call.  This field may be consulted by the VMX
          * at any time (specifically when the VMX is awakened by
          * another VCPU), so it must be set after the other
          * user call arguments.  The VMX is responsible for
          * resetting this field and awakening the VCPU when
          * the user call is complete, via the ACK_USER_CALL
          * and COMPLETE_USER_CALL ioctl.  The latter implies
          * the former.
          *
          * When and how to use ACK_USER_CALL and COMPLETE_USER_CALL
          * are at the discretion of the VMX.  In particular,
          * COMPLETE_USER_CALL does not imply that the requested
          * operation has fully completed, only that the VCPU can
          * continue.  See the comment in MonitorLoopHandleUserCall()
          * for use scenarios.
          *
          * See also comment at MonitorLoopUserCallPoll().
          *
          * -- edward
          */

         if (!crosspage->userCallCross) {
            ASSERT(!crosspage->userCallRestart);
            return crosspage->userCallType;
         }

	 if (!crosspage->userCallRestart) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	    crosspage->userCallRequest = crosspage->userCallType;
	    UCTIMESTAMP(crosspage, AWAKENING_VMX);
	    HostIF_UserCall(vm, vcpuid);
	 }

	 UCTIMESTAMP(crosspage, GOING_TO_SLEEP);
         if (HostIF_UserCallWait(vm, vcpuid, USERCALL_TIMEOUT)) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	 } else {
	    retval = MODULECALL_V5_USERTIMEOUT;
	 }
	 UCTIMESTAMP(crosspage, AWAKE);
      }

      switch (crosspage->moduleCallType) {

      case MODULECALL_V5_NONE:
         break;

      case MODULECALL_V5_INTR:    // Already done in task.c
         break;

      case MODULECALL_V5_GET_RECYCLED_PAGE: {
	 MPN32 mpn;
	 
	 retval = (Vmx86_AllocLockedPages(vm, &mpn, 1, TRUE) == 1) ? 
	    mpn : INVALID_MPN;
	 break;
      }

      case MODULECALL_V5_SEMAWAIT:
         retval = HostIF_SemaphoreWait(vm, vcpuid, crosspage->args[0], crosspage->args[1]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_V5_SEMASIGNAL:
         retval = HostIF_SemaphoreSignal(crosspage->args[0]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_V5_SEMAFORCEWAKEUP: {
         HostIF_SemaphoreForceWakeup(vm, (Vcpuid) crosspage->args[0]);
         break;
      }
      case MODULECALL_V5_IPI:
         retval = HostIF_IPI(vm, (VCPUSet)(uintptr_t)crosspage->args[0], TRUE);
         break;

      case MODULECALL_V5_RELEASE_ANON_PAGE: {
         MPN32 mpn = (MPN32)crosspage->args[0];
	 retval = Vmx86_FreeLockedPages(vm, &mpn, 1, TRUE);
         break;
      }

      case MODULECALL_V5_SWITCH_TO_PEER: {
         crosspage->runVmm64 = !crosspage->runVmm64;
         break;
      }

      default:
         Warning("ModuleCall %d not supported\n", crosspage->moduleCallType);
      }

      /*
       * Give other threads and processes a chance.
       *
       * This is important when we are not preemptable while
       * in the driver and the monitor (e.g., Linux).
       */

      HostIF_Reschedule();
   }

   NOT_REACHED();
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_RunVM_V45  --
 *
 *      Main interaction between the module and the monitor:
 *
 *	   Run the monitor
 *	   Process module calls from the monitor
 *	   Make cross user calls to the main thread
 *	   Return to userlevel to process normal user calls
 *	   and to signal timeout or errors.
 *
 * Results:
 *      Positive: user call number.
 *	-1: error (currently not used).
 *
 * Side effects:
 *      Not really.
 *
 *----------------------------------------------------------------------
 */

int
Vmx86_RunVM_V45(VMDriver *vm, Vcpuid vcpuid)
{
   uint32 retval = MODULECALL_V45_USERRETURN;
   VMCrossPageV45 *crosspage = vm->crosspage[vcpuid];

   ASSERT(crosspage);

   /*
    * Check if we were interrupted by signal.
    */
   if (crosspage->moduleCallInterrupted) {
      crosspage->moduleCallInterrupted = FALSE;
      goto skipTaskSwitch;
   }
   for (;;) {

      crosspage->retval = retval;

      /*
       * Task_Switch changes the world to the monitor.
       * The monitor is waiting in the BackToHost routine.
       */
      UCTIMESTAMP(crosspage, SWITCHING_TO_MONITOR);
      Task_Switch_V45(vm, vcpuid);
      UCTIMESTAMP(crosspage, SWITCHED_TO_MODULE);

skipTaskSwitch:;

      retval = MODULECALL_V45_USERRETURN;

      if (crosspage->userCallType != MODULECALL_USERCALL_NONE) {
         /*
          * This is the main user call path.
          *
          * There are two kinds of user calls.  Normal ones
          * are handled by the calling VCPU thread itself.
          * We just return from here (back to userlevel)
          * in that case.
          *
          * Calls marked userCallCross are handled by the main
          * VMX thread.  In this case, the userCallRequest field
          * indicates to the VMX that this VCPU wants to make
          * a user call.  This field may be consulted by the VMX
          * at any time (specifically when the VMX is awakened by
          * another VCPU), so it must be set after the other
          * user call arguments.  The VMX is responsible for
          * resetting this field and awakening the VCPU when
          * the user call is complete, via the ACK_USER_CALL
          * and COMPLETE_USER_CALL ioctl.  The latter implies
          * the former.
          *
          * When and how to use ACK_USER_CALL and COMPLETE_USER_CALL
          * are at the discretion of the VMX.  In particular,
          * COMPLETE_USER_CALL does not imply that the requested
          * operation has fully completed, only that the VCPU can
          * continue.  See the comment in MonitorLoopHandleUserCall()
          * for use scenarios.
          *
          * See also comment at MonitorLoopUserCallPoll().
          *
          * -- edward
          */

         if (!crosspage->userCallCross) {
            ASSERT(!crosspage->userCallRestart);
            return crosspage->userCallType;
         }

	 if (!crosspage->userCallRestart) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	    crosspage->userCallRequest = crosspage->userCallType;
	    UCTIMESTAMP(crosspage, AWAKENING_VMX);
	    HostIF_UserCall(vm, vcpuid);
	 }

	 UCTIMESTAMP(crosspage, GOING_TO_SLEEP);
         if (HostIF_UserCallWait(vm, vcpuid, USERCALL_TIMEOUT)) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	 } else {
	    retval = MODULECALL_V45_USERTIMEOUT;
	 }
	 UCTIMESTAMP(crosspage, AWAKE);
      }

      switch (crosspage->moduleCallType) {

      case MODULECALL_V45_NONE:
         break;

      case MODULECALL_V45_INTR:    // Already done in task.c
         break;

      case MODULECALL_V45_GET_RECYCLED_PAGE: {
	 MPN32 mpn;
	 
	 retval = (Vmx86_AllocLockedPages(vm, &mpn, 1, TRUE) == 1) ? 
	    mpn : INVALID_MPN;
	 break;
      }

      case MODULECALL_V45_SEMAWAIT:
         retval = HostIF_SemaphoreWait(vm, vcpuid, crosspage->args[0], USERCALL_TIMEOUT);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_V45_SEMASIGNAL:
         retval = HostIF_SemaphoreSignal(crosspage->args[0]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_V45_SEMAFORCEWAKEUP: {
         HostIF_SemaphoreForceWakeup(vm, (Vcpuid) crosspage->args[0]);
         break;
      }
      case MODULECALL_V45_IPI:
         retval = HostIF_IPI(vm, (VCPUSet)(uintptr_t)crosspage->args[0], TRUE);
         break;

      case MODULECALL_V45_RELEASE_ANON_PAGE: {
         MPN32 mpn = (MPN32)crosspage->args[0];
	 retval = Vmx86_FreeLockedPages(vm, &mpn, 1, TRUE);
         break;
      }

      case MODULECALL_V45_SET_FASTCLOCKRATE: {
         /*
          * We must pass this call up to userlevel too.  On Windows
          * hosts, we set the timer resolution at userlevel because
          * ExSetTimerResolution is not available to drivers in NT4,
          * and we also currently set up the event for fast wakeups at
          * userlevel.  On Linux hosts, Vmx86_SetHostClockRate can
          * fail, and we want to pop up a hint if it does.
          */
         crosspage->args[1] = Vmx86_SetHostClockRate(vm, crosspage->args[0]);
         return USERCALL_V45_FASTCLOCKRATE;
      }

      default:
         Warning("ModuleCall %d not supported\n", crosspage->moduleCallType);
      }

      /*
       * Give other threads and processes a chance.
       *
       * This is important when we are not preemptable while
       * in the driver and the monitor (e.g., Linux).
       */

      HostIF_Reschedule();
   }

   NOT_REACHED();
}

