/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * moduleloop.c --
 *
 *     Platform independent routines, private to VMCORE,  to
 *     support module calls and user calls in the module.
 *
 */

#ifdef linux
#   include "driver-config.h"
#   include <linux/kernel.h>
#endif
#include "vmware.h"
#include "machine.h"
#include "modulecall.h"
#include "vmx86.h"
#include "task.h"
#include "initblock.h"
#include "vm_basic_asm.h"
#include "iocontrols.h"
#ifdef USE_PERFCTRS_HOSTED
#   include "perfctr.h"
#endif
#include "hostif.h"
#include "memtrack.h"
#include "driver_vmcore.h"
#include "modulecall_compat.h"

/*
 *----------------------------------------------------------------------
 *
 * Vmx86_RunVM_TOT  --
 *
 *      Main interaction between the module and the monitor:
 *
 *	   Run the monitor
 *	   Process module calls from the monitor
 *	   Make cross user calls to the main thread
 *	   Return to userlevel to process normal user calls
 *	   and to signal timeout or errors.
 *
 * Results:
 *      Positive: user call number.
 *	-1: error (currently not used).
 *
 * Side effects:
 *      Not really.
 *
 *----------------------------------------------------------------------
 */

int
Vmx86_RunVM_TOT(VMDriver *vm, Vcpuid vcpuid)
{
   uint32 retval = MODULECALL_TOT_USERRETURN;
   VMCrossPageTOT *crosspage = vm->crosspage[vcpuid];

   ASSERT(crosspage);

   /*
    * Check if we were interrupted by signal.
    */
   if (crosspage->moduleCallInterrupted) {
      crosspage->moduleCallInterrupted = FALSE;
      goto skipTaskSwitch;
   }
   for (;;) {

      crosspage->retval = retval;

      /*
       * Task_Switch changes the world to the monitor.
       * The monitor is waiting in the BackToHost routine.
       */
      UCTIMESTAMP(crosspage, SWITCHING_TO_MONITOR);
      Task_Switch_TOT(vm, vcpuid);
      UCTIMESTAMP(crosspage, SWITCHED_TO_MODULE);

skipTaskSwitch:;

      retval = MODULECALL_TOT_USERRETURN;

      if (crosspage->userCallType != MODULECALL_USERCALL_NONE) {
         /*
          * This is the main user call path.
          *
          * There are two kinds of user calls.  Normal ones
          * are handled by the calling VCPU thread itself.
          * We just return from here (back to userlevel)
          * in that case.
          *
          * Calls marked userCallCross are handled by the main
          * VMX thread.  In this case, the userCallRequest field
          * indicates to the VMX that this VCPU wants to make
          * a user call.  This field may be consulted by the VMX
          * at any time (specifically when the VMX is awakened by
          * another VCPU), so it must be set after the other
          * user call arguments.  The VMX is responsible for
          * resetting this field and awakening the VCPU when
          * the user call is complete, via the ACK_USER_CALL
          * and COMPLETE_USER_CALL ioctl.  The latter implies
          * the former.
          *
          * When and how to use ACK_USER_CALL and COMPLETE_USER_CALL
          * are at the discretion of the VMX.  In particular,
          * COMPLETE_USER_CALL does not imply that the requested
          * operation has fully completed, only that the VCPU can
          * continue.  See the comment in MonitorLoopHandleUserCall()
          * for use scenarios.
          *
          * See also comment at MonitorLoopUserCallPoll().
          *
          * -- edward
          */

         if (!crosspage->userCallCross) {
            ASSERT(!crosspage->userCallRestart);
            return crosspage->userCallType;
         }

	 if (!crosspage->userCallRestart) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	    crosspage->userCallRequest = crosspage->userCallType;
	    UCTIMESTAMP(crosspage, AWAKENING_VMX);
	    HostIF_UserCall(vm, vcpuid);
	 }

	 UCTIMESTAMP(crosspage, GOING_TO_SLEEP);
         if (HostIF_UserCallWait(vm, vcpuid, USERCALL_TIMEOUT)) {
	    ASSERT(crosspage->userCallRequest == MODULECALL_USERCALL_NONE);
	 } else {
	    retval = MODULECALL_TOT_USERTIMEOUT;
	 }
	 UCTIMESTAMP(crosspage, AWAKE);
      }

      switch (crosspage->moduleCallType) {

      case MODULECALL_TOT_NONE:
         break;

      case MODULECALL_TOT_INTR:    // Already done in task.c
         break;

      case MODULECALL_TOT_GET_RECYCLED_PAGE: {
	 MPN32 mpn;
	 
	 retval = (Vmx86_AllocLockedPages(vm, &mpn, 1, TRUE) == 1) ? 
	    mpn : INVALID_MPN;
	 break;
      }

      case MODULECALL_TOT_SEMAWAIT:
         retval = HostIF_SemaphoreWait(vm, vcpuid, crosspage->args[0],
	                               crosspage->args[1]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_TOT_SEMASIGNAL:
         retval = HostIF_SemaphoreSignal(crosspage->args[0]);
	 if (retval == MX_WAITINTERRUPTED) {
	    crosspage->moduleCallInterrupted = TRUE;
	    return USERCALL_RESTART;
	 }
         break;
      case MODULECALL_TOT_SEMAFORCEWAKEUP: {
         HostIF_SemaphoreForceWakeup(vm, (Vcpuid) crosspage->args[0]);
         break;
      }
      case MODULECALL_TOT_IPI:
         retval = HostIF_IPI(vm, (VCPUSet)(uintptr_t)crosspage->args[0], TRUE);
         break;

      case MODULECALL_TOT_RELEASE_ANON_PAGE: {
         MPN32 mpn = (MPN32)crosspage->args[0];
	 retval = Vmx86_FreeLockedPages(vm, &mpn, 1, TRUE);
         break;
      }

      case MODULECALL_TOT_IS_ANON_PAGE: {
         MPN32 mpn = (MPN32)crosspage->args[0];
         retval = Vmx86_IsAnonPage(vm, mpn);
         break;
      }

      case MODULECALL_TOT_SWITCH_TO_PEER: {
         crosspage->runVmm64 = !crosspage->runVmm64;
         break;
      }

      default:
         Warning("ModuleCall %d not supported\n", crosspage->moduleCallType);
      }

      /*
       * Give other threads and processes a chance.
       *
       * This is important when we are not preemptable while
       * in the driver and the monitor (e.g., Linux).
       */

      HostIF_Reschedule();
   }

   NOT_REACHED();
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_AckUserCall
 *
 *      Acknowledge a user call.  The call need not have completed.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Yes.
 *
 *----------------------------------------------------------------------
 */

void
Vmx86_AckUserCall(VMDriver *vm, Vcpuid vcpuid)
{
   void *crosspage = vm->crosspage[vcpuid];
   switch (vm->vmVersion) {
      case VME_TOT:
         {
	    VMCrossPageTOT* cp = crosspage;
	    ASSERT(cp->userCallRequest != MODULECALL_USERCALL_NONE);
	    cp->userCallRequest = MODULECALL_USERCALL_NONE;
	 }
	 break;
      case VME_S1B1:
         {
	    VMCrossPageS1B1* cp = crosspage;
	    ASSERT(cp->userCallRequest != MODULECALL_USERCALL_NONE);
	    cp->userCallRequest = MODULECALL_USERCALL_NONE;
	 }
	 break;
      case VME_V55:
         {
	    VMCrossPageV55* cp = crosspage;
	    ASSERT(cp->userCallRequest != MODULECALL_USERCALL_NONE);
	    cp->userCallRequest = MODULECALL_USERCALL_NONE;
	 }
	 break;
      case VME_V5:
         {
	    VMCrossPageV5* cp = crosspage;
	    ASSERT(cp->userCallRequest != MODULECALL_USERCALL_NONE);
	    cp->userCallRequest = MODULECALL_USERCALL_NONE;
	 }
	 break;
      case VME_GSX32:
      case VME_V452:
      case VME_V45:
         {
	    VMCrossPageV45* cp = crosspage;
	    ASSERT(cp->userCallRequest != MODULECALL_USERCALL_NONE);
	    cp->userCallRequest = MODULECALL_USERCALL_NONE;
	 }
	 break;
      case VME_V4:
         {
	    VMCrossPageV4* cp = crosspage;
	    ASSERT(cp->userCallRequest != MODULECALL_USERCALL_NONE);
	    cp->userCallRequest = MODULECALL_USERCALL_NONE;
	 }
	 break;
      default:
         Warning("AckUserCall with unknown personality\n");
	 break;
   }
   HostIF_AckUserCall(vm, vcpuid);
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_CompleteUserCall --
 *
 *      Take actions on completion of a user call, which may or
 *      may not have been acknowledged.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      VCPU thread continues.
 *
 *----------------------------------------------------------------------
 */

void
Vmx86_CompleteUserCall(VMDriver *vm, Vcpuid vcpuid)
{
   void *crosspage = vm->crosspage[vcpuid];
   switch (vm->vmVersion) {
      case VME_TOT:
         {
	    VMCrossPageTOT* cp = crosspage;
	    if (cp->userCallRequest != MODULECALL_USERCALL_NONE) {
	       cp->userCallRequest = MODULECALL_USERCALL_NONE;
	       HostIF_AckUserCall(vm, vcpuid);
	    }
	 }
         break;
      case VME_S1B1:
         {
	    VMCrossPageS1B1* cp = crosspage;
	    if (cp->userCallRequest != MODULECALL_USERCALL_NONE) {
	       cp->userCallRequest = MODULECALL_USERCALL_NONE;
	       HostIF_AckUserCall(vm, vcpuid);
	    }
	 }
	 break;
      case VME_V55:
         {
	    VMCrossPageV55* cp = crosspage;
	    if (cp->userCallRequest != MODULECALL_USERCALL_NONE) {
	       cp->userCallRequest = MODULECALL_USERCALL_NONE;
	       HostIF_AckUserCall(vm, vcpuid);
	    }
	 }
	 break;
      case VME_V5:
         {
	    VMCrossPageV5* cp = crosspage;
	    if (cp->userCallRequest != MODULECALL_USERCALL_NONE) {
	       cp->userCallRequest = MODULECALL_USERCALL_NONE;
	       HostIF_AckUserCall(vm, vcpuid);
	    }
	 }
	 break;
      case VME_GSX32:
      case VME_V452:
      case VME_V45:
         {
	    VMCrossPageV45* cp = crosspage;
	    if (cp->userCallRequest != MODULECALL_USERCALL_NONE) {
	       cp->userCallRequest = MODULECALL_USERCALL_NONE;
	       HostIF_AckUserCall(vm, vcpuid);
	    }
	 }
	 break;
      case VME_V4:
         {
	    VMCrossPageV4* cp = crosspage;
	    if (cp->userCallRequest != MODULECALL_USERCALL_NONE) {
	       cp->userCallRequest = MODULECALL_USERCALL_NONE;
	       HostIF_AckUserCall(vm, vcpuid);
	    }
	 }
	 break;
      default:
         Warning("CompleteUserCall with unknown personality\n");
	 break;
   }
   HostIF_AwakenVcpu(vm, vcpuid);
}
