/*
 *----------------------------------------------------------------------
 *
 * CopyMemInfoV4 --
 *
 *      Convert memory management info to VMware2/3/4 format.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      info filled in.
 *
 *----------------------------------------------------------------------
 */

static void
CopyMemInfoV4(VMMemMgmtInfoV4 *info,
	      const VMDriver *vm)
{
   info->pageFaultRate = vm->memInfoCompat.pageFaultRate;
   info->lockFailRate = vm->memInfoCompat.lockFailRate;
   info->maxLocked = vm->memInfoCompat.maxLocked;
   info->currentLocked = vm->memInfo.locked;
   info->highWaterMark = vm->memInfoCompat.highWaterMark;
   info->waitingForMemoryTime = vm->memInfoCompat.waitingForMemoryTime;
   info->poweringOn = vm->memInfoCompat.poweringOn;
   info->hasFocus = vm->memInfoCompat.hasFocus;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetMemInfoV4 --
 *
 *      Return the info about all VMs.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      VMGetMemInfoArgsV4 is filled in.
 *
 *----------------------------------------------------------------------
 */

static Bool
GetMemInfoV4(VMDriver *curVM,
             int32 curVMOnly,
             VMGetMemInfoArgsV4 *outArgs,
             int outArgsLength)
{
   int outSize;
   int wantedVMs;

   if (curVMOnly) {
      wantedVMs = 1;
   } else {
      wantedVMs = vmCount;
   }

   outSize = VM_GET_MEM_INFO_SIZE_V4(wantedVMs);
   if (outSize > outArgsLength) {
      return FALSE;
   }

   outArgs->numVMs = wantedVMs;
   outArgs->numLockedPages = numLockedPages;
   outArgs->maxLockedPages = Vmx86LockedPageLimit(curVM);
   outArgs->callerIndex = -1;

   if (curVM != NULL) {
      if (wantedVMs == 1) {
         CopyMemInfoV4(outArgs->memMgmtInfo, curVM);
         outArgs->callerIndex = 0;
      } else {
         unsigned int i;
         VMDriver *vm;

         outArgs->callerIndex = -1;
         for (vm = vmDriverList, i = 0; vm != NULL && i < vmCount; vm = vm->nextDriver, i++) {
	    if (vm == curVM) {
	       outArgs->callerIndex = i;
	    }
            CopyMemInfoV4(outArgs->memMgmtInfo + i, vm);
         }
      }
   }

   return TRUE;
}


/*
 *-----------------------------------------------------------------------------
 *
 * Vmx86_GetMemInfoCopyV4 --
 *
 *    Return the information about all VMs by copying the data out to user
 *    memory.
 *
 *    On input, outArgs->numVMs indicates how much space has been allocated for
 *    the information. On output, it indicates how much space has been
 *    filled --hpreg
 *
 * Results:
 *    TRUE on success
 *    FALSE on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

Bool
Vmx86_GetMemInfoCopyV4(VMDriver *curVM,             // IN
                       VMGetMemInfoArgsV4 *outArgs) // IN/OUT
{
   int err;
   Bool status;
   VMGetMemInfoArgsV4 hdr;   
   VMGetMemInfoArgsV4 *tmp;
   int outSize;
   int32 curVMOnly;
   int wantedVMs;

   err = HostIF_CopyFromUser(&hdr, outArgs, sizeof(VMGetMemInfoArgsV4));
   if (err != 0) {
      return FALSE;
   }

   while (1) {
      HostIF_GlobalLock(8);
      curVMOnly = hdr.numVMs == 1;
      if (curVMOnly) {
	 wantedVMs = 1;
      } else {
	 wantedVMs = vmCount;
      }
      HostIF_GlobalUnlock(8);
      outSize = VM_GET_MEM_INFO_SIZE_V4(wantedVMs);
      tmp = HostIF_AllocKernelMem(outSize, TRUE);
      if (!tmp) {
	 return FALSE;
      }
      /* GetMemInfoV4 returns error only if buffer was too small */
      HostIF_GlobalLock(8);
      status = GetMemInfoV4(curVM, curVMOnly, tmp, outSize);
      HostIF_GlobalUnlock(8);
      if (status) {
	 break;
      }
      HostIF_FreeKernelMem(tmp);
   }
   err = HostIF_CopyToUser(outArgs, tmp, outSize);
   HostIF_FreeKernelMem(tmp);
   if (err) {
      return FALSE;
   }
   return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetMemInfoV4 --
 *
 *      Set the memory management information about this VM.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      vm->memInfo is filled in.
 *
 *----------------------------------------------------------------------
 */

void
Vmx86_SetMemInfoV4(VMDriver *curVM,
                   VMSetMemInfoArgsV4 *inArgs)
{
   HostIF_GlobalLock(9);

   if (inArgs->info.pageFaultRate >= 0) {
      curVM->memInfoCompat.pageFaultRate = inArgs->info.pageFaultRate;
   }
   if (inArgs->info.lockFailRate >= 0) {
      curVM->memInfoCompat.lockFailRate = inArgs->info.lockFailRate;
   }
   if (inArgs->info.maxLocked >= 0) {
      curVM->memInfoCompat.maxLocked = inArgs->info.maxLocked;
   }
   if (inArgs->info.highWaterMark >= 0) {
      curVM->memInfoCompat.highWaterMark = inArgs->info.highWaterMark;
   }
   if (inArgs->info.waitingForMemoryTime >= 0) {
      curVM->memInfoCompat.waitingForMemoryTime = inArgs->info.waitingForMemoryTime;
   }
   if (inArgs->info.poweringOn >= 0) {
      curVM->memInfoCompat.poweringOn = inArgs->info.poweringOn;
   }
   if (inArgs->info.hasFocus >= 0) {
      curVM->memInfoCompat.hasFocus = inArgs->info.hasFocus;
   }

   HostIF_GlobalUnlock(9);
}


/*
 *----------------------------------------------------------------------
 *
 * GetVMList 
 *
 *     Return the ids for all currently running VMs. Caller must hold
 *     global VM lock.
 *      
 *
 * Results:
 *      TRUE if the caller gave us enough space for the result.
 * 
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static Bool 
GetVMList(VMGetVMListResult *list, uint32 length)
{
   VMDriver *vm;
   int i = 0;

   if (vmCount * sizeof(uint32) + sizeof(VMGetVMListResult) > length) {
      return FALSE;
   }

   list->vmCount = vmCount;

   vm = vmDriverList;

   while (vm) { 
      list->vmUIDs[i++] = vm->id;
      vm = vm->nextDriver;
   }
  
   return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetVMListCopy
 *
 *     Return the ids for all currently running VMs by copying out each
 *     
 *      
 *
 * Results:
 *      TRUE if the caller gave us enough space for the result.
 * 
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Bool 
Vmx86_GetVMListCopy(VMGetVMListResult *list)
{
   Bool status;
   VMGetVMListResult hdr;
   VMGetVMListResult *info;
   uint32 length;

   if (HostIF_CopyFromUser(&hdr, list, sizeof(VMGetVMListResult))) {
      return FALSE;
   }

   HostIF_GlobalLock(15);

   if (vmCount > hdr.vmCount) {
      HostIF_GlobalUnlock(15);
      return FALSE;
   }

   length = sizeof(*info) + vmCount * sizeof(uint32);
   info = HostIF_AllocKernelMem(length, TRUE);
   if (!info) {
      HostIF_GlobalUnlock(15);
      return FALSE;
   }
   status = GetVMList(info, length);
   HostIF_GlobalUnlock(15);
   if (status) {
      if (HostIF_CopyToUser(list, info, length - sizeof(uint32))) {
         status = FALSE;
      }
   }
   HostIF_FreeKernelMem(info);
   return status;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetVMInfo
 *
 *     Set the static information for this VM such as its name and port.  
 *
 * Results:
 *     None.
 * 
 * Side effects:
 *     The VMDriver structure is updated with this info.
 *
 *----------------------------------------------------------------------
 */

void 
Vmx86_SetVMInfo(VMDriver *vm, VMSetVMInfoArgs *info)
{
   memcpy(vm->name, info->name, VM_NAME_LENGTH);
   memcpy(vm->userName, info->userName, VM_USER_NAME_LENGTH);
   vm->port = info->port;
   vm->state = info->state;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetVMInfo
 *
 *     Get the static information for this VM such as its name and port.  
 *
 * Results:
 *     TRUE if a VM with this uid was found, false otherwise.
 * 
 * Side effects:
 *     None.
 *
 *----------------------------------------------------------------------
 */

Bool Vmx86_GetVMInfo(uint32 vmUID, VMGetVMInfoResult *info)
{
   Bool status = FALSE;
   VMDriver *vm;

   HostIF_GlobalLock(16);

   vm = vmDriverList;

   while (vm) { 
      if (vm->id == vmUID) {
         memcpy(info->name, vm->name, VM_NAME_LENGTH);
         memcpy(info->userName, vm->userName, VM_USER_NAME_LENGTH);
         info->port = vm->port;
         info->vmUID = vm->id;
	 info->state = vm->state;
         status = TRUE;
         break;
      }
      vm = vm->nextDriver;
   }

   HostIF_GlobalUnlock(16);

   return status;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_ReleaseAnonPage --
 *
 *      Unlock a page.
 *
 * Results:
 *      PAGE_UNLOCK_NO_ERROR
 *
 * Side effects:
 *      Number of global and per-VM locked pages is decreased.
 *
 *----------------------------------------------------------------------
 */

int
Vmx86_ReleaseAnonPage(VMDriver *vm,
		      MPN mpn)
{
   int retval;

   HostIF_VMLock(vm, 100);
   retval = HostIF_ReleaseAnonPage(vm, mpn);
   HostIF_VMUnlock(vm, 100);

   if (PAGE_LOCK_SUCCESS(retval)) {
      Vmx86UnreserveFreePages(vm, 1);
   }
   
   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_GetRecycledPage --
 *
 *      gets a locked page from list of pages previosly relesed 
 *      by Vmx86_ReleaseAnonPage().
 *
 * Results:
 *	MPN of the page on success
 *      INVALID_MPN if the list is empty or lock failed. 
 *      
 *      
 *
 * Side effects:
 *      Number of global and per-VM locked pages is increased.
 *
 *----------------------------------------------------------------------
 */

MPN
Vmx86_GetRecycledPage(VMDriver *vm)
{
   VPN vpn;

   HostIF_VMLock(vm, 101);
   vpn = HostIF_GetRecycledPage(vm);
   HostIF_VMUnlock(vm, 101);

   if (vpn != INVALID_VPN) {
      MPN mpn = Vmx86_LockPage(vm, (void*)VPN_2_VA(vpn), FALSE);
      if (PAGE_LOCK_SUCCESS(mpn)) {
	 return mpn;
      }
      /* 
       * Return the page to recycled page list.
       * XXX It's ugly but the whole function will go away with anon apge recycling.
       */
      HostIF_VMLock(vm, 102);
      MemTrack_RecyclePage(vm->memtracker, MemTrack_LookupVPN(vm->memtracker, vpn));
      HostIF_VMUnlock(vm, 102);
   }
   return INVALID_MPN;
}
/*
 *-----------------------------------------------------------------------------
 *
 * Vmx86_GetMemInfoCopy --
 *
 *    Return the information about all VMs by copying the data out to user
 *    memory.
 *
 *    On input, outArgs->numVMs indicates how much space has been allocated for
 *    the information. On output, it indicates how much space has been
 *    filled --hpreg
 *
 * Results:
 *    TRUE on success
 *    FALSE on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */


static inline void
CopyMemInfoV45(VMMemMgmtInfoV45* out,	// OUT
	       const VMDriver* vm)	// IN
{
	out->minAllocation = vm->memInfo.minAllocation;
	out->maxAllocation = vm->memInfo.maxAllocation;
	out->shares        = vm->memInfo.shares;
	out->guestmemSize  = vm->memInfoCompat.guestmemSize;
	out->nonpaged      = vm->memInfo.nonpaged;
	out->paged         = vm->memInfo.paged;
	out->locked        = vm->memInfo.locked;
	out->usedPct       = vm->memInfo.usedPct;
	out->admitted      = vm->memInfo.admitted;
}

Bool
Vmx86_GetMemInfoCopyV45(VMDriver *curVM,           // IN
                        VMMemInfoArgsV45 *outArgs) // IN/OUT
{
   VMMemInfoArgsV45 *buf;
   VMDriver *vm;
   Bool ret = FALSE;

   ASSERT(curVM);

   buf = HostIF_AllocKernelMem(VM_GET_MEM_INFO_SIZE_V45(MAX_VMS), TRUE);
   if (!buf) {
      goto err0;
   }
   if (HostIF_CopyFromUser(buf, outArgs, VM_GET_MEM_INFO_SIZE_V45(1))) {
      goto err1;
   }

   HostIF_GlobalLock(8);

   /* Now that we have the lock, we can read vmCount --hpreg */
   if (buf->numVMs < vmCount) {
      HostIF_GlobalUnlock(8);
      goto err1;
   }

   buf->numLockedPages = numLockedPages;
   buf->lockedPageLimit = lockedPageLimit;
   buf->maxLockedPages = Vmx86LockedPageLimit(curVM);
   buf->globalMinAllocation = Vmx86CalculateGlobalMinAllocation(minVmMemPct);
   buf->minVmMemPct = minVmMemPct;

   for (vm = vmDriverList, buf->numVMs = 0;
        vm;
        vm = vm->nextDriver, buf->numVMs++) {
      ASSERT(buf->numVMs < vmCount);
      if (vm == curVM) {
         buf->callerIndex = buf->numVMs;
      }
      HostIF_VMLock(vm, 11);
      CopyMemInfoV45(buf->memInfo + buf->numVMs, vm);
      HostIF_VMUnlock(vm, 11);
   }
   ASSERT(buf->numVMs == vmCount);

   HostIF_GlobalUnlock(8);

   if (!HostIF_CopyToUser(outArgs, buf, VM_GET_MEM_INFO_SIZE_V45(buf->numVMs))) {
      ret = TRUE;
   }

err1:
   HostIF_FreeKernelMem(buf);
err0:
   return ret;
}


/*
 *----------------------------------------------------------------------
 *
 * Vmx86_SetMemInfo --
 *
 *      Set the memory management information about this VM and handles
 *      admission control. We allow vm to power on if there is room for
 *      the minimum allocation for all running vms in memory.  Note that 
 *      the hard memory limit can change dynamically in windows so we
 *      don't have guarantees due to admission control.  If memory checking
 *      is disabled, the vmx passes a non-zero valid argument to indicate
 *      that we should power on anyway.  
 *
 * Results:
 *      Returns global information about the memory state in args as well
 *      as a value indicating whether or not the virtual machine was
 *      started.
 *
 * Side effects:
 *      None
 *
 *----------------------------------------------------------------------
 */

void
Vmx86_SetMemInfoV45(VMDriver *curVM, 
		    VMMemInfoArgsV45 *args)
{
   unsigned globalMinAllocation;

   HostIF_GlobalLock(9);
   /* Update the overcommitment level and minimums for all vms. */
   minVmMemPct = args->minVmMemPct;
   Vmx86UpdateMinAllocations(minVmMemPct);
   globalMinAllocation = Vmx86CalculateGlobalMinAllocation(args->minVmMemPct);

   HostIF_VMLock(curVM, 12);

   curVM->memInfoCompat.guestmemSize  = args->memInfo->guestmemSize;
   curVM->memInfo.paged         = args->memInfo->paged;
   curVM->memInfo.nonpaged      = args->memInfo->nonpaged;
   curVM->memInfo.minAllocation = Vmx86MinAllocation(curVM, args->minVmMemPct);
   curVM->memInfo.maxAllocation = args->memInfo->paged + 
                                  args->memInfo->guestmemSize +
                                  args->memInfo->nonpaged;
   curVM->memInfo.shares        = args->memInfo->shares;
   curVM->memInfo.usedPct       = 100;
   // admission control
   if (globalMinAllocation + curVM->memInfo.minAllocation <= 
       Vmx86LockedPageLimit(curVM)) {
      curVM->memInfo.admitted = TRUE;
   } else {
      curVM->memInfo.admitted = FALSE;
   }
   /* Return global state to the caller. */
   CopyMemInfoV45(args->memInfo, curVM);
   args->numVMs = vmCount;
   args->numLockedPages = numLockedPages;
   args->maxLockedPages = Vmx86LockedPageLimit(curVM);
   args->lockedPageLimit = lockedPageLimit; 
   args->globalMinAllocation = globalMinAllocation;
   args->minVmMemPct = minVmMemPct;
   HostIF_VMUnlock(curVM, 12);
   HostIF_GlobalUnlock(9);
}


