/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriverDestructorOld4Gb --
 *
 *    Destructor for a range --hpreg
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static void
LinuxDriverDestructorOld4Gb(MemRange *that) // IN
{
   struct page *page;
   struct page const *pageEnd;
   int log2NumPages;

   ASSERT(that);

   /*
    * We must unreserve the pages first otherwise free_pages() will refuse to
    * do its job --hpreg
    */
   page = virt_to_page(that->kAddr);
   pageEnd = page + that->numPages;
   for (; page < pageEnd; page++) {
      compat_mem_map_unreserve(page);
   }

   log2NumPages = compat_get_order(that->numPages << PAGE_SHIFT);
   free_pages(that->kAddr, log2NumPages);

   kfree(that);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Open  --
 *
 *      called on open of /dev/vmmon or /dev/vmx86.$USER. Use count used
 *      to determine eventual deallocation of the module
 *
 * Side effects:
 *     Increment use count used to determine eventual deallocation of
 *     the module
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_InitV2(struct file* filp, VMLinux *vmLinux)
{
   VMDriver *vm;
   
   vm = vmLinux->vm;
   if (!vm) {
      vm = Vmx86_CreateVM((void *)filp, (uintptr_t)(current->pid), VME_V2);
      if (vm == NULL) {
         Warning("Vmx86_CreateVM failed\n");
         return -ENOMEM;
      }
      vmLinux->vm = vm;
   }
   return 0;
}


/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriverIoctlAlloc4Gb --
 *
 *    Allocate and lock 'numPages' pages of memory whose physical addresses
 *    are contiguous and < 4 GB --hpreg
 *
 * Results:
 *    0 on success: '*addr' is the physical address of the beginning of the
 *                  range
 *    < 0 on failure: the actual value determines the type of failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static int
LinuxDriverIoctlAlloc4Gb(unsigned long numPages, // IN
                         PA32 *addr)             // OUT
{
   MemRange *range;
   int log2NumPages;
   VA kAddr;
   struct page *page;
   struct page const *pageEnd;
 
   ASSERT(addr);

   if (numPages == 0) {
      return -EINVAL;
   }

   range = kmalloc(sizeof(*range), GFP_KERNEL);
   if (range == NULL) {
      return -ENOMEM;
   }

   log2NumPages = compat_get_order(numPages << PAGE_SHIFT);
   /*
    * We use the ISA DMA zone (i.e. pages whose physical addresses are < 16 MB)
    * by lack of better choice. We should really use GFP_32BIT or analog when
    * it becomes available, but do not do that in this branch, as there is code
    * to fix bug 16306 that relies the ISA DMA zone --hpreg
    */
   kAddr = __get_dma_pages(GFP_KERNEL, log2NumPages);
   if (kAddr == 0) {
      kfree(range);

      return -ENOMEM;
   }

   /*
    * We must reserve the pages otherwise remap_page_range() will refuse to do
    * its job --hpreg
    */
   page = virt_to_page(kAddr);
   pageEnd = page + numPages;
   for (; page < pageEnd; page++) {
      compat_mem_map_reserve(page);
   }

   range->kAddr = kAddr;
   range->pAddr = virt_to_phys((void *)kAddr);
   *addr = range->pAddr;
   range->numPages = numPages;

   /* Insert the range at the beginning of the list --hpreg */
   spin_lock(&linuxState.oldLock4Gb);
   range->next = linuxState.oldList4Gb.next;
   linuxState.oldList4Gb.next = range;
   spin_unlock(&linuxState.oldLock4Gb);

   return 0;
}


static INLINE void LinuxDriverUserPageLocked(VMDriver* vm, MPN mpn) {
   if (PAGE_LOCK_SUCCESS(mpn)) {
      vm->stats.userLockedPages++;
      if (vm->stats.maxUserLockedPages > vm->stats.userLockedPages) {
         vm->stats.maxUserLockedPages = vm->stats.userLockedPages;
      }
   }
}


static INLINE void LinuxDriverUserPageUnlocked(VMDriver* vm, MPN mpn) {
   if (PAGE_LOCK_SUCCESS(mpn)) {
      vm->stats.userLockedPages--;
   }
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriverEmulatedVersion --
 *
 *      retrieve default vmLinux version
 *
 * Results:
 *      int
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriverEmulatedVersion(void)
{
   VMLinux *vml;

   HostIF_GlobalLock(12);

   for (vml = linuxState.head; vml; vml = vml->next) {
      if (vml->version != VME_UNKNOWN) {
         break;
      }
   }

   HostIF_GlobalUnlock(12);

   if (vml) {
      return vml->version;
   }
   return VME_UNKNOWN;
}


