#include "initblock_compat.h"

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlTOT --
 *
 *      Main path for UserRPC from latest product
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlTOT(struct file *filp,
                     u_int iocmd,
		     unsigned long ioarg)
{
   if (iocmd < IOCTLCMD_NONE || iocmd >= IOCTLCMD_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   switch (iocmd) {
      case IOCTLCMD_VERSION:
         return VMMON_VERSION;
      default:
	 break;
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlS1B1 --
 *
 *      Main path for UserRPC from Server 1 beta 1
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlS1B1(struct file *filp,
                      u_int iocmd,
                      unsigned long ioarg)
{
   static const unsigned int map[IOCTLCMD_S1B1_LAST - IOCTLCMD_S1B1_NONE] = {
      IOCTLCMD_NONE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* VERSION */
      IOCTLCMD_CREATE_VM,
      IOCTLCMD_BIND_VM,
      IOCTLCMD_RELEASE_VM,
      IOCTLCMD_GET_NUM_VMS,
      IOCTLCMD_INIT_VM,
      IOCTLCMD_LATE_INIT_VM,
      IOCTLCMD_RUN_VM,
      IOCTLCMD_LOOK_UP_MPN,
      IOCTLCMD_LOCK_PAGE,
      IOCTLCMD_UNLOCK_PAGE,
      IOCTLCMD_APIC_BASE,
      IOCTLCMD_GET_HARD_LIMIT,
      IOCTLCMD_SET_HARD_LIMIT,
      IOCTLCMD_GET_MEM_INFO,
      IOCTLCMD_ADMIT,
      IOCTLCMD_SET_MEM_USAGE,
      IOCTLCMD_READMIT,
      IOCTLCMD_PAE_ENABLED,
      IOCTLCMD_HOST_X86_64,
      IOCTLCMD_GET_TOTAL_MEM_USAGE,
      IOCTLCMD_ACK_USER_CALL,
      IOCTLCMD_COMPLETE_USER_CALL,
      IOCTLCMD_GET_KHZ_ESTIMATE,
      IOCTLCMD_SET_HOST_CLOCK_RATE,
      IOCTLCMD_READ_PAGE,
      IOCTLCMD_WRITE_PAGE,
      IOCTLCMD_LOCK_PAGE_NEW,
      IOCTLCMD_UNLOCK_PAGE_BY_MPN,
      IOCTLCMD_MARK_LOCKEDVARANGE_CLEAN,
      IOCTLCMD_COW_SHARE,
      IOCTLCMD_COW_CHECK,
      IOCTLCMD_COW_UPDATE_HINT,
      IOCTLCMD_COW_COPY_PAGE,
      IOCTLCMD_COW_REMOVE_HINT,
      IOCTLCMD_COW_GET_ZERO_MPN,
      IOCTLCMD_ALLOC_LOCKED_PAGES,
      IOCTLCMD_FREE_LOCKED_PAGES,
      IOCTLCMD_GET_LOCKED_PAGES_LIST,
      IOCTLCMD_MAP_LOCKED_PAGES,
      IOCTLCMD_UNMAP_LOCKED_PAGES,
      IOCTLCMD_APIC_ID,
      IOCTLCMD_VT_CAPABLE_CPU,
      IOCTLCMD_VT_SUPPORTED_CPU,
      IOCTLCMD_SET_UID,
      IOCTLCMD_IS_MP_SAFE,
      IOCTLCMD_ALLOW_CORE_DUMP,
      IOCTLCMD_BROADCAST_IPI,
      IOCTLCMD_REGISTER_PASSTHROUGH_IO,
      IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
      IOCTLCMD_FREE_PASSTHROUGH_IO,
      IOCTLCMD_FREE_PASSTHROUGH_IRQ,
      IOCTLCMD_START_PASSTHROUGH,
      IOCTLCMD_STOP_PASSTHROUGH,
      IOCTLCMD_QUERY_PASSTHROUGH,
      IOCTLCMD_REGISTER_PERFCTR,
      IOCTLCMD_START_PERFCTR,
      IOCTLCMD_STOP_PERFCTR,
      IOCTLCMD_RELEASE_PERFCTR,
      IOCTLCMD_ALLOC_LOW_PAGES,
      IOCTLCMD_FREE_LOW_PAGES,
      IOCTLCMD_GET_ALL_CPUID,
      IOCTLCMD_SET_THREAD_AFFINITY,
      IOCTLCMD_GET_THREAD_AFFINITY,
      IOCTLCMD_SET_POLL_TIMEOUT_PTR,
      IOCTLCMD_GET_KERNEL_CLOCK_RATE,
   };
//   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval = -EINVAL;

   if (iocmd < IOCTLCMD_S1B1_NONE || iocmd >= IOCTLCMD_S1B1_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTLCMD_S1B1_VERSION:
         return VMMON_VERSION_S1B1;
      default:
         iocmd = map[iocmd - IOCTLCMD_S1B1_NONE];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV55 --
 *
 *      Main path for UserRPC from VMware Workstation 5.5
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV55(struct file *filp,
                     u_int iocmd,
		     unsigned long ioarg)
{
   switch (iocmd) {
      case IOCTLCMD_S1B1_VERSION:
         return VMMON_VERSION_V55;
      default:
	 break;
   }
   return LinuxDriver_IoctlS1B1(filp, iocmd, ioarg);
}


static inline int
initBlockV5_to_initBlock(InitBlock *dst, const InitBlockV5 *src) {
        unsigned int i;

        if (src->magicNumber != INIT_BLOCK_MAGIC_V5) {
                return -EINVAL;
        }
        dst->magicNumber = INIT_BLOCK_MAGIC;
        dst->numVCPUs = src->numVCPUs;
        for (i = 0; i < MAX_INITBLOCK_CPUS; i++) {
                dst->crosspage[i] = src->crosspage[i];
        }
        dst->vmInitFailurePeriod = 0;
        return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV5 --
 *
 *      Main path for UserRPC from latest product
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV5(struct file *filp,
                      u_int iocmd,
		      unsigned long ioarg)
{
   static const unsigned int v5map[IOCTLCMD_V5_LAST - IOCTLCMD_V5_NONE] = {
      IOCTLCMD_NONE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* VERSION */
      IOCTLCMD_NOT_IMPLEMENTED, /* CREATE_VM */
      IOCTLCMD_BIND_VM,
      IOCTLCMD_RELEASE_VM,
      IOCTLCMD_GET_NUM_VMS,
      IOCTLCMD_NOT_IMPLEMENTED, /* INIT_VM */
      IOCTLCMD_LATE_INIT_VM,
      IOCTLCMD_RUN_VM,
      IOCTLCMD_LOOK_UP_MPN,
      IOCTLCMD_LOCK_PAGE,
      IOCTLCMD_UNLOCK_PAGE,
      IOCTLCMD_APIC_BASE,
      IOCTLCMD_GET_HARD_LIMIT,
      IOCTLCMD_SET_HARD_LIMIT,
      IOCTLCMD_GET_MEM_INFO,
      IOCTLCMD_ADMIT,
      IOCTLCMD_SET_MEM_USAGE,
      IOCTLCMD_READMIT,
      IOCTLCMD_PAE_ENABLED,
      IOCTLCMD_HOST_X86_64,
      IOCTLCMD_GET_TOTAL_MEM_USAGE,
      IOCTLCMD_ACK_USER_CALL,
      IOCTLCMD_COMPLETE_USER_CALL,
      IOCTLCMD_GET_KHZ_ESTIMATE,
      IOCTLCMD_SET_HOST_CLOCK_RATE,
      IOCTLCMD_READ_PAGE,
      IOCTLCMD_WRITE_PAGE,
      IOCTLCMD_LOCK_PAGE_NEW,
      IOCTLCMD_UNLOCK_PAGE_BY_MPN,
      IOCTLCMD_MARK_LOCKEDVARANGE_CLEAN,
      IOCTLCMD_COW_SHARE,
      IOCTLCMD_COW_CHECK,
      IOCTLCMD_COW_UPDATE_HINT,
      IOCTLCMD_COW_COPY_PAGE,
      IOCTLCMD_COW_REMOVE_HINT,
      IOCTLCMD_ALLOC_LOCKED_PAGES,
      IOCTLCMD_FREE_LOCKED_PAGES,
      IOCTLCMD_GET_LOCKED_PAGES_LIST,
      IOCTLCMD_MAP_LOCKED_PAGES,
      IOCTLCMD_UNMAP_LOCKED_PAGES,
      IOCTLCMD_SET_UID,		// VMX86_DEVEL only
      IOCTLCMD_IS_MP_SAFE,
      IOCTLCMD_ALLOW_CORE_DUMP,
      IOCTLCMD_BROADCAST_IPI,	// SMP 2.2.8+ only
      IOCTLCMD_REGISTER_PASSTHROUGH_IO,
      IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
      IOCTLCMD_FREE_PASSTHROUGH_IO,
      IOCTLCMD_FREE_PASSTHROUGH_IRQ,
      IOCTLCMD_START_PASSTHROUGH,
      IOCTLCMD_STOP_PASSTHROUGH,
      IOCTLCMD_QUERY_PASSTHROUGH,
      IOCTLCMD_REGISTER_PERFCTR,
      IOCTLCMD_START_PERFCTR,
      IOCTLCMD_STOP_PERFCTR,
      IOCTLCMD_RELEASE_PERFCTR,
      IOCTLCMD_ALLOC_LOW_PAGES,
      IOCTLCMD_FREE_LOW_PAGES,
      IOCTLCMD_GET_ALL_CPUID,
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval = -EINVAL;

   if (iocmd < IOCTLCMD_V5_NONE || iocmd >= IOCTLCMD_V5_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTLCMD_V5_VERSION:
         return VMMON_VERSION_V5;

      case IOCTLCMD_V5_CREATE_VM: {
         WSLimitsInfo wsLimitsInfo;
         if (vmLinux->vm != NULL) {
            retval = -EINVAL;
            break;
         }
         if ((void *) ioarg != NULL) {
            retval = HostIF_CopyFromUser(&wsLimitsInfo, (char *) ioarg,
                                         sizeof wsLimitsInfo);
            if (retval != 0) {
               break;
            }
         } else {
            memset(&wsLimitsInfo, 0, sizeof wsLimitsInfo);
         }
         vmLinux->vm = Vmx86_CreateVM((void *)filp, current->pid,
                                      vmLinux->version);
         if (vmLinux->vm == NULL) {
            retval = -ENOMEM;
            break;
         }
         vmLinux->vm->memsizeV5 = wsLimitsInfo.memsize;
         retval = vmLinux->vm->id;
         break;
      }
      case IOCTLCMD_V5_INIT_VM: {
         InitBlock initParams;
         InitBlockV5 initParamsV5;

         if (vmLinux->vm == NULL) {
            retval = -EINVAL;
            break;
         }
         retval = HostIF_CopyFromUser(&initParamsV5, (char*)ioarg,
                                      sizeof initParamsV5);
         if (retval != 0) {
            break;
         }
         retval = initBlockV5_to_initBlock(&initParams, &initParamsV5);
         if (retval) {
            break;
         }
         if (Vmx86_InitVM(vmLinux->vm, &initParams)) {
            retval = -EINVAL;
            break;
         }
         break;
      }

      default:
         iocmd = v5map[iocmd - IOCTLCMD_V5_NONE];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}

static inline int
initBlockV4_to_initBlock(InitBlock *dst, const InitBlockV4 *src) {
	unsigned int i;

	if (src->magicNumber != INIT_BLOCK_MAGIC_V4) {
		return -EINVAL;
	}
	dst->magicNumber = INIT_BLOCK_MAGIC;
	dst->numVCPUs = src->numVCPUs;
	for (i = 0; i < MAX_INITBLOCK_CPUS; i++) {
		dst->crosspage[i] = src->crosspage[i];
	}
	dst->vmInitFailurePeriod = 0;
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX32 --
 *
 *      Main path for UserRPC from VMware GSX 3.2.0
 *
 * Results:
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
LinuxDriver_IoctlGSX32(struct file *filp,
                       u_int iocmd,
		       unsigned long ioarg)
{
   static const unsigned int gsx32map[IOCTLCMD_GSX32_LAST - IOCTLCMD_GSX32_NONE] = {
      IOCTLCMD_NONE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* VERSION */
      IOCTLCMD_NOT_IMPLEMENTED, /* CREATE_VM */
      IOCTLCMD_BIND_VM,
      IOCTLCMD_RELEASE_VM,
      IOCTLCMD_GET_NUM_VMS,
      IOCTLCMD_NOT_IMPLEMENTED, /* INIT_VM */
      IOCTLCMD_LATE_INIT_VM,
      IOCTLCMD_RUN_VM,
      IOCTLCMD_LOOK_UP_MPN,
      IOCTLCMD_LOCK_PAGE,
      IOCTLCMD_UNLOCK_PAGE,
      IOCTLCMD_APIC_BASE,
      IOCTLCMD_NOT_IMPLEMENTED, /* IOAPIC_BASE */
      IOCTLCMD_GET_HARD_LIMIT,
      IOCTLCMD_SET_HARD_LIMIT,
      IOCTLCMD_NOT_IMPLEMENTED, /* GET_MEM_INFO */
      IOCTLCMD_NOT_IMPLEMENTED, /* SET_MEM_INFO */
      IOCTLCMD_SET_MEM_USAGE,
      IOCTLCMD_PAE_ENABLED,
      IOCTLCMD_HOST_X86_64,
      IOCTLCMD_GET_TOTAL_MEM_USAGE,
      IOCTLCMD_ACK_USER_CALL,
      IOCTLCMD_COMPLETE_USER_CALL,
      IOCTLCMD_GET_KHZ_ESTIMATE,
      IOCTLCMD_READ_PAGE,
      IOCTLCMD_WRITE_PAGE,
      IOCTLCMD_LOCK_PAGE_NEW,
      IOCTLCMD_UNLOCK_PAGE_BY_MPN,
      IOCTLCMD_NOT_IMPLEMENTED, /* MARK_LOCKEDVARANGE_CLEAN */
      IOCTLCMD_ALLOC_LOCKED_PAGES,
      IOCTLCMD_FREE_LOCKED_PAGES,
      IOCTLCMD_GET_LOCKED_PAGES_LIST,
      IOCTLCMD_MAP_LOCKED_PAGES,
      IOCTLCMD_UNMAP_LOCKED_PAGES,
      IOCTLCMD_APIC_ID,
      IOCTLCMD_SET_UID,
      IOCTLCMD_IS_MP_SAFE,
      IOCTLCMD_ALLOW_CORE_DUMP,
      IOCTLCMD_BROADCAST_IPI,
      IOCTLCMD_REGISTER_PASSTHROUGH_IO,
      IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
      IOCTLCMD_FREE_PASSTHROUGH_IO,
      IOCTLCMD_FREE_PASSTHROUGH_IRQ,
      IOCTLCMD_START_PASSTHROUGH,
      IOCTLCMD_STOP_PASSTHROUGH,
      IOCTLCMD_QUERY_PASSTHROUGH,
      IOCTLCMD_REGISTER_PERFCTR,
      IOCTLCMD_START_PERFCTR,
      IOCTLCMD_STOP_PERFCTR,
      IOCTLCMD_RELEASE_PERFCTR,
      IOCTLCMD_ALLOC_LOW_PAGES,
      IOCTLCMD_FREE_LOW_PAGES,
      IOCTLCMD_GET_ALL_CPUID,
      IOCTLCMD_SET_THREAD_AFFINITY,
      IOCTLCMD_GET_THREAD_AFFINITY,
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval = -EINVAL;

   if (iocmd < IOCTLCMD_GSX32_NONE || iocmd >= IOCTLCMD_GSX32_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTLCMD_GSX32_VERSION:
         return VMMON_VERSION_GSX32;

      case IOCTLCMD_GSX32_CREATE_VM:
         return LinuxDriver_IoctlV5(filp, IOCTLCMD_V5_CREATE_VM, ioarg);

      case IOCTLCMD_GSX32_INIT_VM: {
         InitBlock initParams;
	 InitBlockV4 initParamsV4;

         if (vmLinux->vm == NULL) {
	    retval = -EINVAL;
	    break;
         }
         retval = HostIF_CopyFromUser(&initParamsV4, (char*)ioarg,
				      sizeof initParamsV4);
         if (retval != 0) {
	    break;
	 }
	 retval = initBlockV4_to_initBlock(&initParams, &initParamsV4);
	 if (retval) {
	    break;
	 }
	 if (Vmx86_InitVM(vmLinux->vm, &initParams)) {
	    retval = -EINVAL;
	    break;
         }
         break;
      }

      case IOCTLCMD_GSX32_IOAPIC_BASE: {
         MA ma; 

         if (vmLinux->vm == NULL) {
	    retval = -EINVAL;
            break;
         }
         ma = HostIF_IOAPICBaseV45();
         /* FIXME: Check whether MA was already 64 bit in GSX2/2.5 */
         retval = HostIF_CopyToUser((MA *)ioarg, &ma, sizeof ma);
         break;
      }

      case IOCTLCMD_GSX32_GET_MEM_INFO: {
         if (vmLinux->vm == NULL) {
            retval = -EINVAL;
            break;
         }
         if (!Vmx86_GetMemInfoCopyV45(vmLinux->vm, (VMMemInfoArgsV45 *) ioarg)) {
            retval = -EINVAL;
         }
         break;
      }

      case IOCTLCMD_GSX32_SET_MEM_INFO: {
         VMMemInfoArgsV45 args;

         if (vmLinux->vm == NULL) {
	    retval = -EINVAL;
            break;
         }
         retval = HostIF_CopyFromUser(&args, (void *)ioarg, sizeof args);
         if (retval != 0) {
	    break;
         }
         Vmx86_SetMemInfoV45(vmLinux->vm, &args); 
         retval = HostIF_CopyToUser((void *)ioarg, &args, sizeof args);
         break;
      }

      case IOCTLCMD_GSX32_MARK_LOCKEDVARANGE_CLEAN: {
         VARangeV45 var;

         retval = -EFAULT;
         if (HostIF_CopyFromUser(&var, (void *)ioarg, sizeof(VARangeV45)) == 0) {
            retval = HostIF_MarkLockedVARangeCleanV45(vmLinux->vm, var.addr, var.len);
         }
	 break;
      }

      default:
         iocmd = gsx32map[iocmd - IOCTLCMD_GSX32_NONE];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV452 --
 *
 *      Main path for UserRPC from VMware Workstation 4.5.2
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV452(struct file *filp,
                      u_int iocmd,
		      unsigned long ioarg)
{
   static const unsigned int v45map[IOCTLCMD_V45_LAST - IOCTLCMD_V45_NONE] = {
      IOCTLCMD_NONE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* VERSION */
      IOCTLCMD_NOT_IMPLEMENTED, /* CREATE_VM */
      IOCTLCMD_BIND_VM,
      IOCTLCMD_RELEASE_VM,
      IOCTLCMD_GET_NUM_VMS,
      IOCTLCMD_NOT_IMPLEMENTED, /* INIT_VM */
      IOCTLCMD_LATE_INIT_VM,
      IOCTLCMD_RUN_VM,
      IOCTLCMD_LOOK_UP_MPN,
      IOCTLCMD_LOCK_PAGE,
      IOCTLCMD_UNLOCK_PAGE,
      IOCTLCMD_APIC_BASE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTLCMD_GET_HARD_LIMIT,
      IOCTLCMD_SET_HARD_LIMIT,
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTLCMD_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTLCMD_SET_MEM_USAGE,
      IOCTLCMD_PAE_ENABLED,
      IOCTLCMD_HOST_X86_64,
      IOCTLCMD_GET_TOTAL_MEM_USAGE,
      IOCTLCMD_ACK_USER_CALL,
      IOCTLCMD_COMPLETE_USER_CALL,
      IOCTLCMD_GET_KHZ_ESTIMATE,
      IOCTLCMD_READ_PAGE,
      IOCTLCMD_WRITE_PAGE,
      IOCTLCMD_LOCK_PAGE_NEW,
      IOCTLCMD_UNLOCK_PAGE_BY_MPN,
      IOCTLCMD_NOT_IMPLEMENTED,	/* MARK_LOCKEDVARANGE_CLEAN */
      IOCTLCMD_ALLOC_LOCKED_PAGES,
      IOCTLCMD_FREE_LOCKED_PAGES,
      IOCTLCMD_GET_LOCKED_PAGES_LIST,
      IOCTLCMD_MAP_LOCKED_PAGES,
      IOCTLCMD_UNMAP_LOCKED_PAGES,
      IOCTLCMD_SET_UID,
      IOCTLCMD_IS_MP_SAFE,
      IOCTLCMD_ALLOW_CORE_DUMP,
      IOCTLCMD_BROADCAST_IPI,
      IOCTLCMD_REGISTER_PASSTHROUGH_IO,
      IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
      IOCTLCMD_FREE_PASSTHROUGH_IO,
      IOCTLCMD_FREE_PASSTHROUGH_IRQ,
      IOCTLCMD_START_PASSTHROUGH,
      IOCTLCMD_STOP_PASSTHROUGH,
      IOCTLCMD_QUERY_PASSTHROUGH,
      IOCTLCMD_REGISTER_PERFCTR,
      IOCTLCMD_START_PERFCTR,
      IOCTLCMD_STOP_PERFCTR,
      IOCTLCMD_RELEASE_PERFCTR,
      IOCTLCMD_ALLOC_LOW_PAGES,
      IOCTLCMD_FREE_LOW_PAGES,
      IOCTLCMD_GET_ALL_CPUID
   };
   int retval = -EINVAL;

   if (iocmd < IOCTLCMD_V45_NONE || iocmd >= IOCTLCMD_V45_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTLCMD_V45_VERSION:
         return VMMON_VERSION_V452;

      case IOCTLCMD_V45_CREATE_VM:
         return LinuxDriver_IoctlV5(filp, IOCTLCMD_V5_CREATE_VM, ioarg);

      case IOCTLCMD_V45_INIT_VM:
         return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_INIT_VM, ioarg);

      case IOCTLCMD_V45_IOAPIC_BASE:
         return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_IOAPIC_BASE, ioarg);

      case IOCTLCMD_V45_GET_MEM_INFO:
         return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_GET_MEM_INFO, ioarg);

      case IOCTLCMD_V45_SET_MEM_INFO:
         return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_SET_MEM_INFO, ioarg);

      case IOCTLCMD_V45_MARK_LOCKEDVARANGE_CLEAN:
         return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_MARK_LOCKEDVARANGE_CLEAN, ioarg);
	 
      default:
         iocmd = v45map[iocmd - IOCTLCMD_V45_NONE];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV45 --
 *
 *      Main path for UserRPC from VMware Workstation 4.5.1 and GSX 3.0.0
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
LinuxDriver_IoctlV45(struct file *filp,
                        u_int iocmd,
		        unsigned long ioarg)
{
   switch (iocmd) {
      case IOCTLCMD_V45_VERSION:
         return VMMON_VERSION_V45;
      default:
         return LinuxDriver_IoctlV452(filp, iocmd, ioarg);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV4 --
 *
 *      Main path for UserRPC from VMware Workstation 4.0
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV4(struct file *filp,
                    u_int iocmd,
		    unsigned long ioarg)
{
   static const unsigned int v4map[IOCTLCMD_V4_LAST - IOCTLCMD_V4_NONE] = {
      IOCTLCMD_NONE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* VERSION */
      IOCTLCMD_NOT_IMPLEMENTED, /* CREATE_VM */
      IOCTLCMD_BIND_VM,
      IOCTLCMD_RELEASE_VM,
      IOCTLCMD_GET_NUM_VMS,
      IOCTLCMD_NOT_IMPLEMENTED, /* INIT_VM */
      IOCTLCMD_LATE_INIT_VM,
      IOCTLCMD_RUN_VM,
      IOCTLCMD_LOOK_UP_MPN,
      IOCTLCMD_LOCK_PAGE,
      IOCTLCMD_UNLOCK_PAGE,
      IOCTLCMD_APIC_BASE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_STATS */
      IOCTLCMD_NOT_IMPLEMENTED,	/* SET_STATS */
      IOCTLCMD_GET_HARD_LIMIT,
      IOCTLCMD_SET_HARD_LIMIT,
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTLCMD_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTLCMD_PAE_ENABLED,
      IOCTLCMD_GET_TOTAL_MEM_USAGE,
      IOCTLCMD_ACK_USER_CALL,
      IOCTLCMD_COMPLETE_USER_CALL,
      IOCTLCMD_GET_KHZ_ESTIMATE,
      IOCTLCMD_SET_UID,
      IOCTLCMD_IS_MP_SAFE,
      IOCTLCMD_ALLOW_CORE_DUMP,
      IOCTLCMD_NOT_IMPLEMENTED,	/* CHECK_MEMORY */
      IOCTLCMD_BROADCAST_IPI,
      IOCTLCMD_REGISTER_PASSTHROUGH_IO,
      IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
      IOCTLCMD_FREE_PASSTHROUGH_IO,
      IOCTLCMD_FREE_PASSTHROUGH_IRQ,
      IOCTLCMD_START_PASSTHROUGH,
      IOCTLCMD_STOP_PASSTHROUGH,
      IOCTLCMD_QUERY_PASSTHROUGH,
      IOCTLCMD_REGISTER_PERFCTR,
      IOCTLCMD_START_PERFCTR,
      IOCTLCMD_STOP_PERFCTR,
      IOCTLCMD_RELEASE_PERFCTR,
      IOCTLCMD_ALLOC_LOW_PAGES,
      IOCTLCMD_FREE_LOW_PAGES,
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_SYSTEM_CLOCK_RATE */
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval = -EINVAL;

   if (iocmd < IOCTLCMD_V4_NONE || iocmd >= IOCTLCMD_V4_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTLCMD_V4_VERSION:
         return VMMON_VERSION_V4;

      case IOCTLCMD_V4_CREATE_VM:
         return LinuxDriver_IoctlV5(filp, IOCTLCMD_V5_CREATE_VM, ioarg);

      case IOCTLCMD_V4_INIT_VM:
         return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_INIT_VM, ioarg);

      case IOCTLCMD_V4_IOAPIC_BASE:
	 return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_IOAPIC_BASE, ioarg);

      case IOCTLCMD_V4_GET_STATS: {
         VMX86StatsV4 stats;

         if (vmLinux->vm == NULL) {
            break;
         }
	 stats = vmLinux->vm->stats;
	 stats.monitorLockedPages -= stats.userLockedPages;
         retval = HostIF_CopyToUser((void *)ioarg, &stats,
                                    sizeof stats);
         break;
      }

      case IOCTLCMD_V4_SET_STATS: {
         VMX86StatsV4 stats;

         if (vmLinux->vm == NULL) {
            break;
         }
         retval = HostIF_CopyFromUser(&stats, (void *)ioarg,
                                      sizeof stats);
	 if (retval == 0) {
	    vmLinux->vm->stats.maxMonitorLockedPages = stats.maxMonitorLockedPages;
	    vmLinux->vm->stats.maxUserLockedPages = stats.maxUserLockedPages;
	 }
         break;
      }

      case IOCTLCMD_V4_GET_MEM_INFO: {
         VMDriver *vm = vmLinux->vm;

         if (vm == NULL) {
            break;
         }
         if (!Vmx86_GetMemInfoCopyV4(vm, (VMGetMemInfoArgsV4 *) ioarg)) {
            break;
         }
         return 0;
      }

      case IOCTLCMD_V4_SET_MEM_INFO: {
         VMSetMemInfoArgsV4 args;
         VMDriver *vm = vmLinux->vm;

         if (vm == NULL) {
            break;
         }
         retval = HostIF_CopyFromUser(&args, (void *)ioarg, sizeof args);
         if (retval != 0) {
	    break;
         }
         Vmx86_SetMemInfoV4(vm, &args);
	 break;
      }

      case IOCTLCMD_V4_CHECK_MEMORY:
         if (vmLinux->vm == NULL) {
            break;
         }
         retval = HostIF_CheckMemory(vmLinux->vm);
         break;

      case IOCTLCMD_V4_GET_SYSTEM_CLOCK_RATE:
         retval = HZ;
         break;

      default:
         iocmd = v4map[iocmd - IOCTLCMD_V4_NONE];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX25 --
 *
 *      Main path for UserRPC from VMware GSX 2.5
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static inline int
initBlockG25_to_initBlock(InitBlock *dst, InitBlockG25 *src) {
	unsigned int i;

	if (src->magicNumber != INIT_BLOCK_MAGIC_V3) {
		return -EINVAL;
	}
	dst->magicNumber = INIT_BLOCK_MAGIC;
	dst->numVCPUs = src->numVCPUs;
	for (i = 0; i < MAX_INITBLOCK_CPUS; i++) {
		dst->crosspage[i] = src->crosspage[i];
	}
	dst->vmInitFailurePeriod = 0;
	return 0;
}

static inline void
initBlock_to_initBlockG25(InitBlockG25 *dst, InitBlock *src) {
	Vmx86_SetStartTime(&dst->st);
}

static int
LinuxDriver_IoctlGSX25(struct file *filp,
                       u_int iocmd,
		       unsigned long ioarg)
{
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval;
   static const unsigned int gsx25map[IOCTLCMD_GSX25_LAST - IOCTLCMD_GSX25_NONE] = {
      IOCTLCMD_NONE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* VERSION */
      IOCTLCMD_NOT_IMPLEMENTED, /* CREATE_VM */
      IOCTLCMD_BIND_VM,
      IOCTLCMD_RELEASE_VM,
      IOCTLCMD_GET_NUM_VMS,
      IOCTLCMD_NOT_IMPLEMENTED,	/* INIT_VM */
      IOCTLCMD_LATE_INIT_VM,
      IOCTLCMD_RUN_VM,
      IOCTLCMD_LOOK_UP_MPN,
      IOCTLCMD_LOCK_PAGE,
      IOCTLCMD_UNLOCK_PAGE,
      IOCTLCMD_APIC_BASE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_STATS */
      IOCTLCMD_NOT_IMPLEMENTED,	/* SET_STATS */
      IOCTLCMD_GET_HARD_LIMIT,
      IOCTLCMD_SET_HARD_LIMIT,
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTLCMD_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTLCMD_PAE_ENABLED,
      IOCTLCMD_GET_TOTAL_MEM_USAGE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_TSCS */
      IOCTLCMD_NOT_IMPLEMENTED,	/* SET_TSCS */
      IOCTLCMD_GET_KHZ_ESTIMATE,
      IOCTLCMD_SET_UID,
      IOCTLCMD_IS_MP_SAFE,
      IOCTLCMD_ALLOW_CORE_DUMP,
      IOCTLCMD_NOT_IMPLEMENTED,	/* CHECK_MEMORY */
      IOCTLCMD_BROADCAST_IPI,
      IOCTLCMD_REGISTER_PASSTHROUGH_IO,
      IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
      IOCTLCMD_FREE_PASSTHROUGH_IO,
      IOCTLCMD_FREE_PASSTHROUGH_IRQ,
      IOCTLCMD_START_PASSTHROUGH,
      IOCTLCMD_STOP_PASSTHROUGH,
      IOCTLCMD_QUERY_PASSTHROUGH,
      IOCTLCMD_REGISTER_PERFCTR,
      IOCTLCMD_START_PERFCTR,
      IOCTLCMD_STOP_PERFCTR,
      IOCTLCMD_RELEASE_PERFCTR,
      IOCTLCMD_ALLOC_LOW_PAGES,
      IOCTLCMD_FREE_LOW_PAGES,
   };

   if (iocmd < IOCTLCMD_GSX25_NONE || iocmd >= IOCTLCMD_GSX25_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   switch (iocmd) {
      case IOCTLCMD_GSX25_VERSION:
         return VMMON_VERSION_GSX25;
      case IOCTLCMD_GSX25_CREATE_VM:
         return LinuxDriver_IoctlV5(filp, IOCTLCMD_V5_CREATE_VM, ioarg);
      case IOCTLCMD_GSX25_INIT_VM: {
         InitBlockG25 initParamsG25;
         InitBlock initParams;

         if (vmLinux->vm == NULL) {
	    return -EINVAL;
         }
         retval = HostIF_CopyFromUser(&initParamsG25, (char*)ioarg,
				      sizeof initParamsG25);
         if (retval != 0) {
	    return retval;
         }
	 if (initBlockG25_to_initBlock(&initParams, &initParamsG25)) {
	    return -EINVAL;
	 }
         if (Vmx86_InitVM(vmLinux->vm, &initParams)) {
	    return -EINVAL;
         }
         initBlock_to_initBlockG25(&initParamsG25, &initParams); 
         return HostIF_CopyToUser((char*)ioarg, &initParamsG25,sizeof initParamsG25);
      }
      case IOCTLCMD_GSX25_SET_MEM_INFO:
         return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_SET_MEM_INFO, ioarg);
      case IOCTLCMD_GSX25_GET_MEM_INFO:
         return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_GET_MEM_INFO, ioarg);
      case IOCTLCMD_GSX25_SET_STATS:
         return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_SET_STATS, ioarg);
      case IOCTLCMD_GSX25_GET_STATS:
         return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_GET_STATS, ioarg);
      case IOCTLCMD_GSX25_CHECK_MEMORY:
         return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_CHECK_MEMORY, ioarg);
      case IOCTLCMD_GSX25_IOAPIC_BASE:
	 return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_IOAPIC_BASE, ioarg);
      default:
         iocmd = gsx25map[iocmd - IOCTLCMD_GSX25_NONE];
	 break;
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX251 --
 *
 *      Main path for UserRPC from VMware GSX 2.5.1
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
LinuxDriver_IoctlGSX251(struct file *filp,
                        u_int iocmd,
		        unsigned long ioarg)
{
   switch (iocmd) {
      case IOCTLCMD_GSX25_VERSION:
         return VMMON_VERSION_GSX251;
      case IOCTLCMD_GSX251_GET_ALL_CPUID:
         iocmd = IOCTLCMD_GET_ALL_CPUID;
	 break;
      default:
         return LinuxDriver_IoctlGSX25(filp, iocmd, ioarg);
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX2 --
 *
 *      Main path for UserRPC from VMware GSX 2.0
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlGSX2(struct file *filp,
                      u_int iocmd,
		      unsigned long ioarg)
{
   static const unsigned int gsx2map[IOCTLCMD_GSX2_LAST - IOCTLCMD_GSX2_NONE] = {
      IOCTLCMD_NONE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* VERSION */
      IOCTLCMD_NOT_IMPLEMENTED, /* CREATE_VM */
      IOCTLCMD_BIND_VM,
      IOCTLCMD_RELEASE_VM,
      IOCTLCMD_GET_NUM_VMS,
      IOCTLCMD_NOT_IMPLEMENTED,	/* INIT_VM */
      IOCTLCMD_LATE_INIT_VM,
      IOCTLCMD_RUN_VM,
      IOCTLCMD_LOOK_UP_MPN,
      IOCTLCMD_LOCK_PAGE,
      IOCTLCMD_UNLOCK_PAGE,
      IOCTLCMD_APIC_BASE,
      IOCTLCMD_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_STATS */
      IOCTLCMD_NOT_IMPLEMENTED,	/* SET_STATS */
      IOCTLCMD_GET_HARD_LIMIT,
      IOCTLCMD_SET_HARD_LIMIT,
      IOCTLCMD_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTLCMD_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTLCMD_PAE_ENABLED,
      IOCTLCMD_GET_TOTAL_MEM_USAGE,
      IOCTLCMD_SET_UID,
      IOCTLCMD_IS_MP_SAFE,
      IOCTLCMD_GET_MHZ_ESTIMATE,
      IOCTLCMD_ALLOW_CORE_DUMP,
      IOCTLCMD_NOT_IMPLEMENTED,	/* CHECK_MEMORY */
      IOCTLCMD_BROADCAST_IPI,
      IOCTLCMD_REGISTER_PASSTHROUGH_IO,
      IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
      IOCTLCMD_FREE_PASSTHROUGH_IO,
      IOCTLCMD_FREE_PASSTHROUGH_IRQ,
      IOCTLCMD_START_PASSTHROUGH,
      IOCTLCMD_STOP_PASSTHROUGH,
      IOCTLCMD_QUERY_PASSTHROUGH,
      IOCTLCMD_REGISTER_PERFCTR,
      IOCTLCMD_START_PERFCTR,
      IOCTLCMD_STOP_PERFCTR,
      IOCTLCMD_RELEASE_PERFCTR,
      IOCTLCMD_ALLOC_LOW_PAGES,
      IOCTLCMD_FREE_LOW_PAGES,
   };

   if (iocmd < IOCTLCMD_GSX2_NONE || iocmd >= IOCTLCMD_GSX2_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   switch (iocmd) {
   case IOCTLCMD_GSX2_VERSION:
      return VMMON_VERSION_GSX2;
   case IOCTLCMD_GSX2_CREATE_VM:
      return LinuxDriver_IoctlV5(filp, IOCTLCMD_V5_CREATE_VM, ioarg);
   case IOCTLCMD_GSX2_INIT_VM:
      return LinuxDriver_IoctlGSX25(filp, IOCTLCMD_GSX25_INIT_VM, ioarg);
   case IOCTLCMD_GSX2_SET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_SET_MEM_INFO, ioarg);
   case IOCTLCMD_GSX2_GET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_GET_MEM_INFO, ioarg);
   case IOCTLCMD_GSX2_SET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_SET_STATS, ioarg);
   case IOCTLCMD_GSX2_GET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_GET_STATS, ioarg);
   case IOCTLCMD_GSX2_CHECK_MEMORY:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_CHECK_MEMORY, ioarg);
   case IOCTLCMD_GSX2_IOAPIC_BASE:
      return LinuxDriver_IoctlGSX32(filp, IOCTLCMD_GSX32_IOAPIC_BASE, ioarg);
   default:
      iocmd = gsx2map[iocmd - IOCTLCMD_GSX2_NONE];
      return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV3 --
 *
 *      Main path for UserRPC from VMware 3.[01]
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static inline int
initBlock30_to_initBlock(InitBlock *dst, InitBlock30 *src) {
	if (src->magicNumber != INIT_BLOCK_MAGIC_V3) {
		return -EINVAL;
	}
	dst->magicNumber = INIT_BLOCK_MAGIC;
	dst->numVCPUs = 1;
	dst->crosspage[0] = src->crosspage;
	dst->vmInitFailurePeriod = 0;
	return 0;
}

static inline void
initBlock_to_initBlock30(InitBlock30 *dst, InitBlock *src) {
	Vmx86_SetStartTime(&dst->st);
}

static int
LinuxDriver_IoctlV3(struct file *filp,
                    u_int iocmd,
                    unsigned long ioarg)
{
   static const unsigned int v3map[] = {
      [IOCTLCMD_V3_BIND_VM]		= IOCTLCMD_BIND_VM,
      [IOCTLCMD_V3_RELEASE_VM]		= IOCTLCMD_RELEASE_VM,
      [IOCTLCMD_V3_LATE_INIT_VM]	= IOCTLCMD_LATE_INIT_VM,
      [IOCTLCMD_V3_SET_UID]		= IOCTLCMD_SET_UID,
      [IOCTLCMD_V3_GET_NUM_VMS]		= IOCTLCMD_GET_NUM_VMS,
      [IOCTLCMD_V3_GET_TOTAL_MEM_USAGE]	= IOCTLCMD_GET_TOTAL_MEM_USAGE,
      [IOCTLCMD_V3_SET_HARD_LIMIT]	= IOCTLCMD_SET_HARD_LIMIT,
      [IOCTLCMD_V3_GET_HARD_LIMIT]	= IOCTLCMD_GET_HARD_LIMIT,
      [IOCTLCMD_V3_PAE_ENABLED]		= IOCTLCMD_PAE_ENABLED,
      [IOCTLCMD_V3_IS_MP_SAFE]		= IOCTLCMD_IS_MP_SAFE,
      [IOCTLCMD_V3_ALLOW_CORE_DUMP]	= IOCTLCMD_ALLOW_CORE_DUMP,
      [IOCTLCMD_V3_BROADCAST_IPI]	= IOCTLCMD_BROADCAST_IPI,
      [IOCTLCMD_V3_GET_MHZ_ESTIMATE]	= IOCTLCMD_GET_MHZ_ESTIMATE,
      [IOCTLCMD_V3_REGISTER_PASSTHROUGH_IRQ]	= IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
      [IOCTLCMD_V3_REGISTER_PASSTHROUGH_IO]	= IOCTLCMD_REGISTER_PASSTHROUGH_IO,
      [IOCTLCMD_V3_FREE_PASSTHROUGH_IRQ]	= IOCTLCMD_FREE_PASSTHROUGH_IRQ,
      [IOCTLCMD_V3_FREE_PASSTHROUGH_IO]	= IOCTLCMD_FREE_PASSTHROUGH_IO,
      [IOCTLCMD_V3_START_PASSTHROUGH]	= IOCTLCMD_START_PASSTHROUGH,
      [IOCTLCMD_V3_STOP_PASSTHROUGH]	= IOCTLCMD_STOP_PASSTHROUGH,
      [IOCTLCMD_V3_QUERY_PASSTHROUGH]	= IOCTLCMD_QUERY_PASSTHROUGH,
      [IOCTLCMD_V3_REGISTER_PERFCTR]	= IOCTLCMD_REGISTER_PERFCTR,
      [IOCTLCMD_V3_START_PERFCTR]	= IOCTLCMD_START_PERFCTR,
      [IOCTLCMD_V3_STOP_PERFCTR]	= IOCTLCMD_STOP_PERFCTR,
      [IOCTLCMD_V3_RELEASE_PERFCTR]	= IOCTLCMD_RELEASE_PERFCTR,
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval;

   switch (iocmd) {
   case IOCTLCMD_V3_VERSION:
      return VMMON_VERSION_V3;
   case IOCTLCMD_V3_BIND_VM:
   case IOCTLCMD_V3_RELEASE_VM:
   case IOCTLCMD_V3_LATE_INIT_VM:
   case IOCTLCMD_V3_SET_UID:
   case IOCTLCMD_V3_GET_NUM_VMS:
   case IOCTLCMD_V3_GET_TOTAL_MEM_USAGE:
   case IOCTLCMD_V3_SET_HARD_LIMIT:
   case IOCTLCMD_V3_GET_HARD_LIMIT:
   case IOCTLCMD_V3_PAE_ENABLED:
   case IOCTLCMD_V3_IS_MP_SAFE:
   case IOCTLCMD_V3_ALLOW_CORE_DUMP:
   case IOCTLCMD_V3_BROADCAST_IPI:
   case IOCTLCMD_V3_GET_MHZ_ESTIMATE:
   case IOCTLCMD_V3_REGISTER_PASSTHROUGH_IRQ:
   case IOCTLCMD_V3_REGISTER_PASSTHROUGH_IO:
   case IOCTLCMD_V3_FREE_PASSTHROUGH_IRQ:
   case IOCTLCMD_V3_FREE_PASSTHROUGH_IO:
   case IOCTLCMD_V3_START_PASSTHROUGH:
   case IOCTLCMD_V3_STOP_PASSTHROUGH:
   case IOCTLCMD_V3_QUERY_PASSTHROUGH:
   case IOCTLCMD_V3_REGISTER_PERFCTR:
   case IOCTLCMD_V3_START_PERFCTR:
   case IOCTLCMD_V3_STOP_PERFCTR:
   case IOCTLCMD_V3_RELEASE_PERFCTR:
      iocmd = v3map[iocmd];
      break;
   case IOCTLCMD_V3_CREATE_VM:
      return LinuxDriver_IoctlV5(filp, IOCTLCMD_V5_CREATE_VM, ioarg);
   case IOCTLCMD_V3_INIT_VM: {
      InitBlock30 initParams30;
      InitBlock initParams;

      if (vmLinux->vm == NULL) {
	 return -EINVAL;
      }
      retval = HostIF_CopyFromUser(&initParams30, (char*)ioarg,
				   sizeof initParams30);
      if (retval != 0) {
	 return retval;
      }
      if (initBlock30_to_initBlock(&initParams, &initParams30)) {
      	 return -EINVAL;
      }
      if (Vmx86_InitVM(vmLinux->vm, &initParams)) {
	 return -EINVAL;
      }
      initBlock_to_initBlock30(&initParams30, &initParams); 
      return HostIF_CopyToUser((char*)ioarg,&initParams30,sizeof(initParams30));
   }
   case IOCTLCMD_V3_RUN_VM:
      ioarg = VCPUID_UP;
      iocmd = IOCTLCMD_RUN_VM;
      break;
   case IOCTLCMD_V3_LOOK_UP_MPN:
      retval = __LinuxDriver_Ioctl(filp, IOCTLCMD_LOOK_UP_MPN, ioarg);
      if (retval >= 0 && (retval & PAGE_LOCK_ERROR_BIT)) {
         return 0;
      }
      return retval;
   case IOCTLCMD_V3_LOCK_PAGE:
      retval = __LinuxDriver_Ioctl(filp, IOCTLCMD_LOCK_PAGE, ioarg);
      if (retval >= 0 && (retval & PAGE_LOCK_ERROR_BIT)) {
         return 0;
      }
      return retval;
   case IOCTLCMD_V3_UNLOCK_PAGE:
      retval = __LinuxDriver_Ioctl(filp, IOCTLCMD_UNLOCK_PAGE, ioarg);
      if (retval >= 0 && (retval & PAGE_LOCK_ERROR_BIT)) {
         return 0;
      }
      return retval;
   case IOCTLCMD_V3_APIC_BASE: {
      VMAPICInfo info;
      mm_segment_t oldfs;
      
      info.flags = ioarg;
      oldfs = get_fs();
      set_fs(KERNEL_DS);
      retval = __LinuxDriver_Ioctl(filp, IOCTLCMD_APIC_BASE, (unsigned long)&info);
      set_fs(oldfs);
      if (retval) {
         return retval;
      }
      return info.apicBase;
   }
   case IOCTLCMD_V3_IOAPIC_BASE: {
      /* It is simpler to copy code for retrieving IOAPICBase than
         writting wrapper around GSX2 call */
      if (vmLinux->vm == NULL) {
	 retval = -EINVAL;
	 break;
      }
      return HostIF_IOAPICBaseV45();
   }
   case IOCTLCMD_V3_SET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_SET_MEM_INFO, ioarg);
   case IOCTLCMD_V3_GET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_GET_MEM_INFO, ioarg);
   case IOCTLCMD_V3_SET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_SET_STATS, ioarg);
   case IOCTLCMD_V3_GET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_GET_STATS, ioarg);
   case IOCTLCMD_V3_CHECK_MEMORY:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_CHECK_MEMORY, ioarg);

   default:
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV32 --
 *
 *      Main path for UserRPC from VMware 3.2
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV32(struct file *filp,
                     u_int iocmd,
                     unsigned long ioarg)
{
   switch (iocmd) {
   case IOCTLCMD_V3_VERSION:
      return VMMON_VERSION_V32;
   case IOCTLCMD_V32_ALLOC_LOW_PAGES:
      return __LinuxDriver_Ioctl(filp, IOCTLCMD_ALLOC_LOW_PAGES, ioarg);
   case IOCTLCMD_V32_FREE_LOW_PAGES:
      return __LinuxDriver_Ioctl(filp, IOCTLCMD_FREE_LOW_PAGES, ioarg);
   }
   return LinuxDriver_IoctlV3(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV321 --
 *
 *      Main path for UserRPC from VMware 3.2.1
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV321(struct file *filp,
                     u_int iocmd,
                     unsigned long ioarg)
{
   switch (iocmd) {
   case IOCTLCMD_V3_VERSION:
      return VMMON_VERSION_V321;
   /*
    * There is new ioctl, IOCTLCMD_V321_ADD_WORLD_ARG, but it
    * is not handled by vmmon 3.2.1 code.
    */
   }
   return LinuxDriver_IoctlV32(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV2 --
 *
 *      Main path for UserRPC from VMware 2.x
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV2(VMLinux *vmLinux,
		    struct file *filp,
                    u_int iocmd,
		    unsigned long ioarg)
{
   VMDriver *vm = vmLinux->vm;

   switch (iocmd) {
   case IOCTLCMD_V2_INIT: {
      InitBlock20 initParams;
      InitBlock realParams;
      VmTimeStart stt;

      if (HostIF_CopyFromUser(&initParams,(char*)ioarg,sizeof(initParams))) {
         return -EFAULT;
      }
      if (initParams.magicNumber != INIT_BLOCK_MAGIC_V3) {
	 return -EINVAL;
      }
      realParams.magicNumber = INIT_BLOCK_MAGIC;
      realParams.numVCPUs = 1;
      realParams.crosspage[0] = initParams.crosspage;
      realParams.vmInitFailurePeriod = 0;
      if (Vmx86_InitVM(vm, &realParams)) {
	 return -EPERM;
      }
      initParams.smpMachine = __LinuxDriver_Ioctl(filp, IOCTLCMD_IS_MP_SAFE, 0);
      Vmx86_SetStartTime(&stt);
      initParams.st.count = stt.count;
      initParams.st.time = stt.time;
      initParams.st.mhzEstimate = Vmx86_GetkHzEstimate(&linuxState.startTime) / 1000;
      if (HostIF_CopyToUser((char*)ioarg,&initParams,sizeof(initParams))) {
         return -EFAULT;
      }
      return 0;
   }
   case IOCTLCMD_V2_LATE_INIT:
      iocmd = IOCTLCMD_LATE_INIT_VM;
      break;      
   case IOCTLCMD_V2_RUN:
      ioarg = VCPUID_UP;
      iocmd = IOCTLCMD_RUN_VM;
      break;
   case IOCTLCMD_V2_BEEP: { // XXX for buggy Linux
      uint8 byte;
      byte = inb(SPEAKER_PORT);
      if (ioarg >> 16) {
	 byte |= (SPEAKER_ENABLE_SPEAKER | SPEAKER_TIMER2_GATE);
      } else {
	 byte &= ~(SPEAKER_ENABLE_SPEAKER | SPEAKER_TIMER2_GATE);
      }
      outb(byte, SPEAKER_PORT);
      outb(0xb6, 0x43);
      outb((uint8) ioarg, 0x42); /* outb_p needs paravirt_ops */
      outb((uint8) (ioarg >> 8), 0x42);
      return 0;
   }

   // XXX Deprecated.  Now handled by devel_suid() for certain types of
   // open() calls.
   case IOCTLCMD_V2_SETUID:
      return -EPERM;
   case IOCTLCMD_V2_LOOKUPMPN:
      return LinuxDriver_IoctlV3(filp, IOCTLCMD_V3_LOOK_UP_MPN, ioarg);
   case IOCTLCMD_V2_LOCKPAGE:
      return LinuxDriver_IoctlV3(filp, IOCTLCMD_V3_LOCK_PAGE, ioarg);
   case IOCTLCMD_V2_UNLOCKPAGE:
      return LinuxDriver_IoctlV3(filp, IOCTLCMD_V3_UNLOCK_PAGE, ioarg);
   case IOCTLCMD_V2_GET_NUM_VMS:
      iocmd = IOCTLCMD_GET_NUM_VMS;
      break;
   case IOCTLCMD_V2_SET_HARD_LIMIT:
      iocmd = IOCTLCMD_SET_HARD_LIMIT;
      break;
   case IOCTLCMD_V2_GET_HARD_LIMIT:
      iocmd = IOCTLCMD_GET_HARD_LIMIT;
      break;
   case IOCTLCMD_V2_SET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_SET_MEM_INFO, ioarg);
   case IOCTLCMD_V2_GET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_GET_MEM_INFO, ioarg);
   case IOCTLCMD_V2_GET_VM_LIST:
      if (!Vmx86_GetVMListCopy((VMGetVMListResult *)ioarg)) {
         return -EINVAL;
      }
      return 0;
   case IOCTLCMD_V2_GET_VM_INFO: {
      VMGetVMInfoResult info;

      if (HostIF_CopyFromUser(&info, (void *)ioarg, sizeof(info))) {
         return -EFAULT;
      }
      if (!Vmx86_GetVMInfo(info.vmUID, &info)) {
         return -EINVAL;
      }
      if (HostIF_CopyToUser((void *)ioarg, &info, sizeof(info))) {
         return -EFAULT;
      }
      return 0;
   }
   case IOCTLCMD_V2_SET_VM_INFO: {
      VMSetVMInfoArgs info;

      if (HostIF_CopyFromUser(&info, (void *)ioarg, sizeof(info))) {
         return -EFAULT;
      }
      Vmx86_SetVMInfo(vm, &info);
      return 0;
   }
   case IOCTLCMD_V2_SET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_SET_STATS, ioarg);
   case IOCTLCMD_V2_GET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_GET_STATS, ioarg);
   case IOCTLCMD_V2_DECLARE_SLAVE:
      /* obsolete */
      return 0;
   case IOCTLCMD_V2_ISMPSAFE:
      iocmd = IOCTLCMD_IS_MP_SAFE;
      break;
   case IOCTLCMD_V2_APICBASE:
      /* v2.0 had only one APIC flag... */
      if (ioarg) {
         ioarg = APIC_FLAG_DISABLE_NMI;
      }
      return LinuxDriver_IoctlV3(filp, IOCTLCMD_V3_APIC_BASE, ioarg);
   case IOCTLCMD_V2_IOAPICBASE:
      return LinuxDriver_IoctlV3(filp, IOCTLCMD_V3_IOAPIC_BASE, 0);
   case IOCTLCMD_V2_CHECK_MEMORY: 
      return LinuxDriver_IoctlV4(filp, IOCTLCMD_V4_CHECK_MEMORY, ioarg);
   case IOCTLCMD_V2_REGISTER_PASSTHROUGH_IRQ:
      iocmd = IOCTLCMD_REGISTER_PASSTHROUGH_IRQ;
      break;
   case IOCTLCMD_V2_REGISTER_PASSTHROUGH_IO:
      iocmd = IOCTLCMD_REGISTER_PASSTHROUGH_IO;
      break;
   case IOCTLCMD_V2_FREE_PASSTHROUGH_IRQ:
      iocmd = IOCTLCMD_FREE_PASSTHROUGH_IRQ;
      break;
   case IOCTLCMD_V2_FREE_PASSTHROUGH_IO:
      iocmd = IOCTLCMD_FREE_PASSTHROUGH_IO;
      break;
   case IOCTLCMD_V2_START_PASSTHROUGH:
      iocmd = IOCTLCMD_START_PASSTHROUGH;
      break;
   case IOCTLCMD_V2_STOP_PASSTHROUGH:
      iocmd = IOCTLCMD_STOP_PASSTHROUGH;
      break;
   case IOCTLCMD_V2_QUERY_PASSTHROUGH:
      iocmd = IOCTLCMD_QUERY_PASSTHROUGH;
      break;
   case IOCTLCMD_V2_ALLOW_CORE_DUMP:
      iocmd = IOCTLCMD_ALLOW_CORE_DUMP;
      break;
   case IOCTLCMD_V2_BROADCAST_IPI:
      iocmd = IOCTLCMD_BROADCAST_IPI;
      break;
   case IOCTLCMD_V2_FREE_RESOURCES:
      return HostIF_FreeAllResources(vm);
   default:
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX1 --
 *
 *      Main path for UserRPC from VMware GSX 1.x
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlGSX1(struct file *filp,
                      u_int iocmd,
		      unsigned long ioarg)
{
   if (iocmd >= IOCTLCMD_V3_PAE_ENABLED) {
      iocmd += 2;
   }
   switch (iocmd) {
   case IOCTLCMD_V3_VERSION:
      return VMMON_VERSION_GSX1;
   }
   return LinuxDriver_IoctlV3(filp, iocmd, ioarg);
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Ioctl --
 *
 *      Main path for UserRPC
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int 
LinuxDriver_Ioctl( struct inode *inode, 
                   struct file *filp,
                   u_int iocmd, 
                   unsigned long ioarg )
{
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval;

   VMWare_SetVTracer(VTrace_Set);

   if (vmLinux->version == VME_UNKNOWN) {
      if (iocmd == IOCTLCMD_V2_GET_NUM_VMS || iocmd == IOCTLCMD_V2_ISMPSAFE || iocmd == IOCTLCMD_V2_SET_HARD_LIMIT) {
         vmLinux->version = VME_V2;
      } else if (iocmd == IOCTLCMD_VERSION) {
         /* VMware4.5 and newer use 2001, so force VM55 if default is not acceptable */
	 if (vmversion >= VME_V45) {
	    vmLinux->version = vmversion;
	 } else {
            vmLinux->version = VME_V55;
	 }
      } else if (iocmd == IOCTLCMD_V4_VERSION) {
      	 /* 201 cannot be VMware 4.5 or anything newer; use VMware4 as safe fallback */
      	 if (vmversion >= VME_V45) {
	    vmLinux->version = VME_V4;
	 } else {
	    vmLinux->version = vmversion;
	 }
      } else {
         Warning("Unexpected ioctl (%d) invoked, using default\n", iocmd);
         vmLinux->version = vmversion;
      }
   }
   switch (vmLinux->version) {
      case VME_V2:
         if (!vmLinux->vm) {
            retval = LinuxDriver_InitV2(filp, vmLinux);
	    if (retval) {
	       break;
	    }
         }
         retval = LinuxDriver_IoctlV2(vmLinux, filp, iocmd, ioarg);
	 break;
      case VME_GSX1:
         retval = LinuxDriver_IoctlGSX1(filp, iocmd, ioarg);
	 break;
      case VME_V3:
         retval = LinuxDriver_IoctlV3(filp, iocmd, ioarg);
	 break;
      case VME_V32:
         retval = LinuxDriver_IoctlV32(filp, iocmd, ioarg);
	 break;
      case VME_V321:
         retval = LinuxDriver_IoctlV321(filp, iocmd, ioarg);
	 break;
      case VME_GSX2:
         retval = LinuxDriver_IoctlGSX2(filp, iocmd, ioarg);
	 break;
      case VME_GSX25:
         retval = LinuxDriver_IoctlGSX25(filp, iocmd, ioarg);
	 break;
      case VME_GSX251:
         retval = LinuxDriver_IoctlGSX251(filp, iocmd, ioarg);
	 break;
      case VME_V4:
      	 retval = LinuxDriver_IoctlV4(filp, iocmd, ioarg);
	 break;
      case VME_V45:
      	 retval = LinuxDriver_IoctlV45(filp, iocmd, ioarg);
	 break;
      case VME_V452:
	 retval = LinuxDriver_IoctlV452(filp, iocmd, ioarg);
	 break;
      case VME_GSX32:
	 retval = LinuxDriver_IoctlGSX32(filp, iocmd, ioarg);
         break;
      case VME_V5:
	 retval = LinuxDriver_IoctlV5(filp, iocmd, ioarg);
	 break;
      case VME_V55:
	 retval = LinuxDriver_IoctlV55(filp, iocmd, ioarg);
	 break;
      case VME_S1B1:
	 retval = LinuxDriver_IoctlS1B1(filp, iocmd, ioarg);
	 break;
      case VME_TOT:
         retval = LinuxDriver_IoctlTOT(filp, iocmd, ioarg);
	 break;
      default:
	 retval = -EINVAL;
         Warning("Unexpected emulation (%u) selected\n", vmLinux->version);
	 break;
   }
   VMWare_SetVTracer(0);
   return retval;
}


