#include "initblock_compat.h"
#include "modulecall_compat.h"
#include <asm/fixmap.h>


/*
 *----------------------------------------------------------------------
 *
 * HostIFSystemCallResetMSR --
 *
 *      Reset physical sysenter & syscall MSR values on a processor.
 *
 *----------------------------------------------------------------------
 */

static void
HostIFSystemCallResetMSR(void *clientData) // IN
{
   const SystemCallRegisters *regs = ((const SystemCallRegisters *)clientData +
                                     HostIF_GetCurrentPCPU());

   if (regs->sysenterCS != SYSENTER_SENTINEL_CS) {
      HostIF_WRMSR(MSR_SYSENTER_CS, regs->sysenterCS);
   }
   if (regs->sysenterRIP != SYSENTER_SENTINEL_RIP) {
      HostIF_WRMSR(MSR_SYSENTER_EIP, regs->sysenterRIP);
   }
   if (regs->sysenterRSP != SYSENTER_SENTINEL_RSP) {
      HostIF_WRMSR(MSR_SYSENTER_ESP, regs->sysenterRSP);
   }

   if (CPUID_SyscallSupported() && regs->star != SYSCALL_SENTINEL_STAR) {
      HostIF_WRMSR(MSR_STAR, regs->star);
      if (regs->cstar != SYSCALL_SENTINEL_CSTAR) {
         HostIF_WRMSR(MSR_CSTAR, regs->cstar);
         HostIF_WRMSR(MSR_LSTAR, regs->lstar);
         HostIF_WRMSR(MSR_SFMASK, regs->sfmask);
      }
   }
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_SysenterReset --
 *
 *      Reset physical sysenter MSR values on all processors.
 *
 *----------------------------------------------------------------------
 */

void
HostIF_SystemCallReset(int numLogicalCPUs, SystemCallRegisters *regs)
{
   compat_preempt_disable();
   HostIFSystemCallResetMSR(regs);
   (void)compat_smp_call_function(HostIFSystemCallResetMSR, (void *)regs, 1, 1);
   compat_preempt_enable();
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_ReleaseAnonPage --
 *
 *      The page with given mpn is unlocked and released. The page
 *      is put into a cache of released pages. Released pages can be 
 *      recycled later by calling HostIF_GetRecycledPage.
 *
 * Results:
 *      PAGE_UNLOCK_NO_ERROR
 *
 *----------------------------------------------------------------------
 */

int
HostIF_ReleaseAnonPage(VMDriver *vm, MPN mpn)
{
   MemTrackEntry *e;

   e = MemTrack_LookupMPN(vm->memtracker, mpn);
   if (e != NULL) {
      ASSERT(e->mpn == mpn);

      HOST_UNLOCK_PFN(vm, mpn);
      e->mpn = 0;

      MemTrack_RecyclePage(vm->memtracker, e);
   }
   return PAGE_UNLOCK_NO_ERROR;
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_GetRecycledPage --
 *
 *      Returns the VPN of a previously released anonymous page, if
 *      one is available.
 *
 * Results:
 *      VPN of the page if one is available, INVALID_VPN otherwise
 *
 *----------------------------------------------------------------------
 */

VPN
HostIF_GetRecycledPage(VMDriver *vm)
{
   MemTrackEntry *e;

   e = MemTrack_GetRecycledPage(vm->memtracker);
   if (e != NULL) {
      return e->vpn;
   } else {
      return INVALID_VPN;
   }
}


/*
 *-----------------------------------------------------------------------------
 *
 * HostIF_UserToDriverPtr 
 *
 *    Convert a user virtual address into a kernel virtual address to access
 *    the same physical memory address --hpreg
 *
 * Results:
 *    The kernel virtual address on success
 *    NULL on failure
 *
 * Side effects:
 *    XXX Use 1 per-VM storage location. So for example, we leak kernel memory
 *        when we run a SMP VM on a host with more than 850 MB of RAM --hpreg
 *
 *-----------------------------------------------------------------------------
 */

void *
HostIF_UserToDriverPtr(VMDriver *vm, // IN
                       VA uvAddr)    // IN
{
   struct page *page;

   if (HostIFGetUserPage(vm->vmVersion, (void *)uvAddr, &page)) {
      return NULL;
   }

   HostIF_GlobalLock(17);
   if (vm->vmhost->crosspagePagesCount >= MAX_INITBLOCK_CPUS) {
      HostIF_GlobalUnlock(17);
      put_page(page);
      return NULL;
   }
   vm->vmhost->crosspagePages[vm->vmhost->crosspagePagesCount++] = page;
   HostIF_GlobalUnlock(17);

   return (char *)MapCrossPage(page) + (uvAddr & (PAGE_SIZE - 1));
}


/*
 *-----------------------------------------------------------------------------
 *
 * __mod_user_asm --
 *
 *    Mark page dirty. Does not check for kernel address or for broken
 *    write protect on i386.
 *
 * Results:
 *    err unmodified on success
 *    err set to errret on failure.
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

#define __mod_user_asm(addr, err, errret)	\
   asm volatile (				\
      "1:    lock; orb $0,%1\n"			\
      "2:\n"					\
      ".section .fixup,\"ax\"\n"		\
      "3:    movl %2,%0\n"			\
      "      jmp 2b\n"				\
      ".previous\n"				\
      ".section __ex_table,\"a\"\n"		\
      "      .align 4\n"			\
      "      .long 1b,3b\n"			\
      ".previous"				\
      : "=r"(err)				\
      : "m"(__m(addr)), "i"(errret), "0"(err))


/*
 *-----------------------------------------------------------------------------
 *
 * mod_user_check --
 *
 *    Mark page dirty.
 *
 * Results:
 *    0 on success
 *    -EFAULT on failure - address not mapped, or read-only, or not user address
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

#define mod_user_check(addr)				\
({							\
   long __mu_err = -EFAULT;				\
   void *__mu_addr = (addr);				\
   if (access_ok(VERIFY_WRITE, __mu_addr, 1)) {		\
      __mu_err = 0;					\
      __mod_user_asm(__mu_addr, __mu_err, -EFAULT);	\
   }							\
   __mu_err;						\
})


/*
 *-----------------------------------------------------------------------------
 *
 * HostIF_MarkPageDirty --
 *
 *    Mark page dirty.
 *
 * Results:
 *    0 on success
 *    != 0 on failure.
 *
 * Side effects:
 *    Complain loudly if global VM lock is held.
 *
 *-----------------------------------------------------------------------------
 */

int
HostIF_MarkPageDirty(VMDriver* vm, void *addr) // IN/OUT
{
#ifdef VM_X86_64
   /* XXX not implemented on x86-64. */
   return 0;
#elif defined(CONFIG_X86_4G) || defined(CONFIG_X86_UACCESS_INDIRECT)
   HostIFGetUserPage(vm->vmVersion, addr, NULL);
   return 0;
#else
   return mod_user_check(addr);
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_IsMPNLocked
 *
 *      Return entryPtr if MPN is locked into memory
 *
 * Results:
 *      0 if sucessful and non-zero or error.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static void *
FindFunc(void *arg, MemTrackEntry *entryPtr)
{
   MPN mpn = (MPN)(uintptr_t)arg;
   
   if (entryPtr->mpn == mpn) {
      return entryPtr;
   }
   return NULL;
}

int
HostIF_IsMPNLocked(VMDriver *vm, MPN mpn)
{
  MemTrackEntry *entry;

  entry = MemTrack_Scan(vm->memtracker, (void*)(uintptr_t)mpn, FindFunc);
  return (entry != NULL);
}


static void *
CheckFunc(void *arg,               // IN
          MemTrackEntry *entryPtr) // IN
{
   VMDriver *vm = (VMDriver *)arg;
   MPN mpn;
   
   if (entryPtr->mpn == 0) {
      return NULL;
   }

   mpn = PgtblVa2MPN(VPN_2_VA(entryPtr->vpn));
   if (mpn == INVALID_MPN) { 
      /* Unmapped */
      return entryPtr;
   }

   if (entryPtr->mpn != mpn) {
      Warning("CheckFunc vpn 0x%lx mpn 0x%lx and not 0x%lx \n", 
              (unsigned long)entryPtr->vpn, (unsigned long)mpn, (unsigned long)entryPtr->mpn);
      vm->checkFuncFailed = TRUE;
   }
    
   return NULL;
}


/*
 *-----------------------------------------------------------------------------
 *
 * HostIF_CheckMemory
 *
 *      Make sure memory locking is OK.
 *
 *      This function runs as part of the application process, as such
 *      PgtblVa2MPN is fine
 *
 * Results:
 *      None
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

VA
HostIF_CheckMemory(VMDriver *vm) // IN
{
   MemTrackEntry *entryPtr;
   
   vm->checkFuncFailed = FALSE;
   entryPtr = MemTrack_Scan(vm->memtracker, vm, CheckFunc);
   
   if (vm->checkFuncFailed) {
      return 1;
   }

   if (entryPtr) {
      return VPN_2_VA(entryPtr->vpn);
   }

   return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_MarkLockedVARangeCleanV45 --
 *
 *     Clear the dirty bit in the HW page tables for [VA, VA+len) if
 *     the page is already locked by the monitor/userlevel.
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     None.
 *
 *----------------------------------------------------------------------
 */

int
HostIF_MarkLockedVARangeCleanV45(const VMDriver *vm, VA va, unsigned len)
{
   struct mm_struct *mm = current->mm;
   VA end = va + len;

   if (vm->vmhost->lockedPages == NULL) {
      return 0;
   }
   if (compat_get_page_table_lock(mm)) {
      spin_lock(compat_get_page_table_lock(mm));
   }
   for (;va < end; va += PAGE_SIZE) {
      pte_t *pte;

      pte = PgtblVa2PTELocked(mm, va);
      if (pte != NULL) {
         MPN mpn;

         mpn = PgtblPte2MPN(pte);
         /* PgtblPte2Page does pte_present. */
         if (mpn != INVALID_MPN && pte_dirty(*pte)) {
            if (PhysTrack_Test(vm->vmhost->lockedPages, mpn)) {
               uint32 *p = (uint32 *)pte;
               *p &= ~(PTE_D);
            }
         }
         pte_unmap(pte);
      }
   }
   if (compat_get_page_table_lock(mm)) {
      spin_unlock(compat_get_page_table_lock(mm));
   }
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_IOAPICBaseV45 --
 *
 *      Lookup the IO-APIC physical address
 *
 * Results:
 *      
 *      Returned value is IO-APIC physical address
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
#define PCI_CONF_ADDR_REG_PORT   0xcf8
#define PCI_CONF_DATA_REG_PORT   0xcfc
#define VMWARE__PCI_DEVICE_ID_INTEL_82371SB 0x70008086
#define VMWARE__PCI_DEVICE_ID_INTEL_82371AB 0x71108086

MA
HostIF_IOAPICBaseV45(void)
{
#if defined(CONFIG_SMP) || defined(CONFIG_X86_UP_IOAPIC)
   int bus, slot, confword;
   uint32 id, apicbase;
   MPN mpn;

   bus = 0;
   for (slot = 0; slot < 32; slot++) {
      confword = 0x80000000 | (bus << 16) |  (slot << 11);
      OUT32(PCI_CONF_ADDR_REG_PORT, confword);
      id = IN32(PCI_CONF_DATA_REG_PORT);
      if ((id == VMWARE__PCI_DEVICE_ID_INTEL_82371SB) ||
	  (id == VMWARE__PCI_DEVICE_ID_INTEL_82371AB)) {
#ifdef VMX86_DEVEL
	 Log("IOAPIC: detected %s on PCI bus %d, slot %d\n",
	     (id == VMWARE__PCI_DEVICE_ID_INTEL_82371SB) ? "PIIX3" : "PIIX4",
	     bus, slot);
#endif
	 confword |= 20; /* register 80 */
	 OUT32(PCI_CONF_ADDR_REG_PORT, confword);
	 apicbase = IN32(PCI_CONF_DATA_REG_PORT);
	 apicbase = apicbase & 0x3f;
	 return (IOAPIC_DEFAULT_ADDRESS | (apicbase << 10));
      }
   }
   
   mpn = PgtblVa2MPN(__fix_to_virt(VMWARE__FIX_IO_APIC_BASE));
   if (mpn == INVALID_MPN) {
      return 0;
   }
   return MPN_2_MA(mpn);
#else
   return 0;
#endif
}


#define VM_CHECKSIZE(expected, structure)	\
   if (sizeof(structure) != expected) {		\
      asm volatile(".print \"Unexpected ABI change: Structure " #structure " should take %c0 bytes, but took %c1!\"; .abort" : : "i"(expected), "i"(sizeof(structure))); \
   }

#define prop_hex(key,digits,val)			\
   asm volatile(".section \".properties\"\n\t"		\
		".ascii \"" key " \"\n"			\
		"t1 = %a0\n\t"				\
		".rept %c1\n"				\
		"t2 = (t1 >> (%c1 * 4 - 4)) & 15\n\t"	\
		".if t2 < 10\n\t"			\
		".byte t2 + '0'\n\t"			\
		".else\n\t"				\
		".byte t2 + 'A' - 10\n\t"		\
		".endif\n"				\
		"t1 = t1 << 4\n\t"			\
		".endr\n\t"				\
		".byte 10\n\t"				\
		".previous" : : "i"(val), "i"(digits));

#define prop_entry(key, val)				\
   asm volatile(".section \".properties\"\n\t"		\
		".ascii \"" key " " val "\"\n\t"	\
		".byte 10\n\t"				\
		".previous")

#ifdef VMW_HAVE_COMPILE_H
/* Retrieve UTS_VERSION and UTS_MACHINE if there is such information available */
#include <linux/compile.h>
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 18)
#include <linux/utsrelease.h>
#endif

static void __attribute__((used))
properties(void)
{
  asm volatile(".section \".properties\", \"\"; .previous");
  prop_entry("UtsRelease", UTS_RELEASE);
#ifdef UTS_VERSION
  prop_entry("UtsVersion", UTS_VERSION);
#endif
#ifdef UTS_MACHINE
  prop_entry("UtsMachine", UTS_MACHINE);
#endif
#ifdef CONFIG_MODVERSIONS
  prop_entry("ModVersions", "yes");
#else
  prop_entry("ModVersions", "no");
#endif
#ifdef CONFIG_SMP
  prop_entry("SMP", "yes");
#else
  prop_entry("SMP", "no");
#endif
#ifdef VM_X86_64
  prop_hex("PageOffset", 16, PAGE_OFFSET);
#else
  prop_hex("PageOffset", 8, PAGE_OFFSET);
#endif
  prop_entry("Comment", "Extended vmmon module from Petr");
  
  /* Make sure that ABI did not change */
  VM_CHECKSIZE(   4, int32);
  VM_CHECKSIZE(   8, MA);
  VM_CHECKSIZE(   4, PA32);
  VM_CHECKSIZE(   8, PA);
  VM_CHECKSIZE(   4, LA32);
  VM_CHECKSIZE(   4, LPN32);
  VM_CHECKSIZE(  24, WSLimitsInfo);
  VM_CHECKSIZE(   8, VMMUnlockPageByMPN);
  VM_CHECKSIZE(   8, VMAPICInfo);
  VM_CHECKSIZE(  24, PerfCtrRegisterArgs);
  VM_CHECKSIZE(   8, CPUIDResult);
  VM_CHECKSIZE(  16, CPUIDRegs);
  VM_CHECKSIZE(  84, CPUIDSummary);
  VM_CHECKSIZE(  24, VmTimeStart20);
  VM_CHECKSIZE(  24, WSLimitsInfo);
  VM_CHECKSIZE(   8, PerfCtrInfo);
  VM_CHECKSIZE(  24, PerfCtrRegisterArgs);
  VM_CHECKSIZE(   8, VMAPICInfo);
  VM_CHECKSIZE( 132, VMSyncWriteTSCs);
  VM_CHECKSIZE(  12, LockedPageLimit);
  VM_CHECKSIZE(  16, VMMemCOWHotPage);
  VM_CHECKSIZE( 104, VMMemCOWInfo);
  VM_CHECKSIZE(   8, Gate);
  VM_CHECKSIZE(  16, Gate64);
  VM_CHECKSIZE(   6, DTR32);
  VM_CHECKSIZE(  10, DTR64);
  VM_CHECKSIZE(   8, DTRWords32);
  VM_CHECKSIZE(  16, DTRWords64);
  VM_CHECKSIZE(   8, Descriptor);
  VM_CHECKSIZE(   8, CallGate);
  VM_CHECKSIZE(   8, DescriptorUnion);
  VM_CHECKSIZE(   2, Reg16);
  VM_CHECKSIZE(   4, Reg32);
  VM_CHECKSIZE(   8, Reg64);
  VM_CHECKSIZE(   4, VM_PDE);
  VM_CHECKSIZE(   4, VM_PTE);
  VM_CHECKSIZE(   8, VM_PAE_PDE);
  VM_CHECKSIZE(   8, VM_PAE_PTE);
  VM_CHECKSIZE(   8, VM_PDPTE);
  VM_CHECKSIZE(   8, VM_L1E);
  VM_CHECKSIZE(   8, VM_L2E);
  VM_CHECKSIZE(   8, VM_L3E);
  VM_CHECKSIZE(   8, VM_L4E);
  VM_CHECKSIZE(   8, VMMPNList);
  VM_CHECKSIZE(  12, VMMPNListVA);
  VM_CHECKSIZE(  12, VARange);
  VM_CHECKSIZE(   8, VARangeV45);
  VM_CHECKSIZE(   8, VMMReadWritePage);
  VM_CHECKSIZE( 184, VMMemInfoArgs);
  VM_CHECKSIZE(  72, VMMemInfoArgsV45);
  VM_CHECKSIZE(  48, VMGetMemInfoArgsV4);
  VM_CHECKSIZE(  32, VMSetMemInfoArgsV4);
  VM_CHECKSIZE(  16, VMX86StatsV4);
  VM_CHECKSIZE(  40, VMMemMgmtInfo);
  VM_CHECKSIZE(  36, VMMemMgmtInfoV45);
  VM_CHECKSIZE(  32, VMMemMgmtInfoV4);
  VM_CHECKSIZE( 140, InitBlock);
  VM_CHECKSIZE( 136, InitBlockV5);
  VM_CHECKSIZE( 140, InitBlockV4);
  VM_CHECKSIZE( 164, InitBlockG25);
  VM_CHECKSIZE(  36, InitBlock30);
  VM_CHECKSIZE(  48, InitBlock20);
  VM_CHECKSIZE(   8, VMGetVMListResult);
  VM_CHECKSIZE( 108, VMGetVMInfoResult);
  VM_CHECKSIZE( 104, VMSetVMInfoArgs);
  VM_CHECKSIZE(   4, CReg);
  VM_CHECKSIZE(   2, Selector);
  VM_CHECKSIZE( 104, Task64);
  VM_CHECKSIZE( 104, Task);
  VM_CHECKSIZE(  44, Task16);
  VM_CHECKSIZE(   6, FarPtr);
  VM_CHECKSIZE(   4, FarPtr16);
  VM_CHECKSIZE(   6, FarPtr32);
  VM_CHECKSIZE(  40, x86ExcFrame64);
  VM_CHECKSIZE(  12, x86ExcFrame32);
  VM_CHECKSIZE(  24, x86ExcFrame32IL);
  VM_CHECKSIZE(   6, x86ExcFrame16);
  VM_CHECKSIZE(  36, x86ExcFrameV8086);
  VM_CHECKSIZE(  40, x86ExcFrameV8086WithErrorCode);
  VM_CHECKSIZE(   8, x86CallStack32);
  VM_CHECKSIZE(   4, x86CallStack16);
  VM_CHECKSIZE(  16, x86CallGateStack32);
  VM_CHECKSIZE(   8, x86CallGateStack16);
  VM_CHECKSIZE(   4, DebugControlRegister);
  VM_CHECKSIZE(  24, SysenterRegsV5);
  VM_CHECKSIZE(  32, SysenterStateV5);
  VM_CHECKSIZE(  56, SystemCallRegisters);
  VM_CHECKSIZE( 120, SystemCallState);
  VM_CHECKSIZE(  12, SMM32SegState);
  VM_CHECKSIZE(  16, SMMSegState);
  VM_CHECKSIZE( 512, SMMStateSaveMap);
  VM_CHECKSIZE( 108, struct FSAVEState);
  VM_CHECKSIZE( 512, struct FXSAVEState);
  VM_CHECKSIZE( 512, struct FXSAVEState64);
  VM_CHECKSIZE( 176, Context64);
  VM_CHECKSIZE(  92, Context64V45);
  VM_CHECKSIZE( 132, ContextInfo);
  VM_CHECKSIZE( 316, ContextInfo64);
  VM_CHECKSIZE( 136, ContextInfoV45);
  VM_CHECKSIZE( 178, ContextInfo64V45);
  VM_CHECKSIZE( 332, ContextInfo64V5);
  VM_CHECKSIZE(  96, LongModeSwitch);
  VM_CHECKSIZE(  80, LongModeSwitchV5);
  VM_CHECKSIZE( 660, VMCrossPageGSX1);
  VM_CHECKSIZE( 660, VMCrossPageV3);
  VM_CHECKSIZE(1488, VMCrossPageV32);
  VM_CHECKSIZE(1520, VMCrossPageV321);
  VM_CHECKSIZE(1501, VMCrossPageGSX2);
  VM_CHECKSIZE(1533, VMCrossPageGSX25);
  VM_CHECKSIZE(1577, VMCrossPageV4);
  VM_CHECKSIZE(1780, VMCrossPageV45);
  VM_CHECKSIZE(2368, VMCrossPageV5);
  VM_CHECKSIZE(2544, VMCrossPageV55);
  VM_CHECKSIZE(2544, VMCrossPageS1B1);
  VM_CHECKSIZE(2544, VMCrossPageTOT);
}
