/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 *  hostif.h - Platform dependent interface for supporting 
 *             the vmx86 device driver. 
 */


#ifndef _HOSTIF_H_
#define _HOSTIF_H_

#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMCORE
#include "includeCheck.h"

#include "vmx86.h"
#include "vcpuset.h"

#if !defined _WIN32
#include "cpuid_info.h"
#endif

/*
 * Host-specific definitions. 
 */
#if !__linux__ && !defined(WINNT_DDK) && !defined __APPLE__
#error "Only Linux or NT or Mac OS defined for now."
#endif


#ifdef linux
#   define IRQ_HOST_INTR1_BASE 0x20
#   define IRQ_HOST_INTR2_BASE 0x28

/* See arch/i386/kernel/irq.h in the kernel source --hpreg */
#   define IRQ0_TRAP_VECTOR 0x51
#elif WINNT_DDK
#   define IRQ_HOST_INTR1_BASE 0x30
#   define IRQ_HOST_INTR2_BASE 0x38
#elif defined __APPLE__
#   define IRQ_HOST_INTR1_BASE 0x30 /* XXXMACOS */
#   define IRQ_HOST_INTR2_BASE 0x38 /* XXXMACOS */
#else
#   error "Free host interrupt vectors are unknown."
#endif

EXTERN int   HostIF_Init(VMDriver *vm);
EXTERN MPN   HostIF_LookupUserMPN(VMDriver *vm, void *addr);
EXTERN void *HostIF_MapCrossPage(VMDriver *vm, void *addr, LA *linearAddr);
EXTERN void  HostIF_InitFP(VMDriver *vm);
#if defined __APPLE__
#define HostIF_InitEvent(_vm) do {} while (0)
#else
EXTERN void  HostIF_InitEvent(VMDriver *vm);
#endif

EXTERN void *HostIF_AllocKernelMem(size_t size, int nonPageable);
EXTERN void  HostIF_FreeKernelMem(void *ptr);

EXTERN void *HostIF_AllocPage(void);
EXTERN void  HostIF_FreePage(void *ptr);

EXTERN uint32 HostIF_CountPresentPages(VMMCountPresentPages const *cpp);

/*
 * XXX These are _not_ part of the platform-specific interface that the
 *     cross-platform code uses. Consequently, they should become private to
 *     their respective drivers. --hpreg
 */
#if !defined __APPLE__
EXTERN void *HostIF_AllocContigPages(VMDriver *vm, MPN* mpns, size_t numPages, 
                                     unsigned int order, MPN maxMPN);
EXTERN int   HostIF_FreeContigPages(VMDriver *vm, void *ptr);
EXTERN void  HostIF_InitGlobalLock(void);
#if defined _WIN32
EXTERN void HostIF_InitFastClock(void);
EXTERN void HostIF_StartFastClockThread(void);
EXTERN void HostIF_StopFastClockThread(void);
EXTERN void HostIF_SetUserCallHandle(VMDriver *vm, int h);
EXTERN int HostIF_SyncReadTSCs(uint64* TSCs, uintptr_t cpuSet);
EXTERN int HostIF_SyncWriteTSCs(const uint32* TSCs, uintptr_t cpuSet);
EXTERN void HostIF_SynchronizeTSCs(void);
EXTERN void HostIF_SetMemoryParams(const VA64* params, int count);
#endif
#if __linux__
void HostIF_GetAllCpuInfo(CPUIDResult *info);
EXTERN uint32 HostIF_BrokenCPUHelper(void);
EXTERN int HostIF_MarkLockedVARangeClean(const VMDriver *vm, VA uvAddr, 
                                         unsigned len, VA32 bv);
EXTERN void HostIF_PollListLock(int callerID);
EXTERN void HostIF_PollListUnlock(int callerID);
struct page;
EXTERN void *HostIF_MapUserMem(VA32 addr, size_t size, struct page **page);
EXTERN void HostIF_UnmapUserMem(struct page **page);
#endif
#endif

EXTERN int   HostIF_CopyFromUser(void *dst, const void *src, unsigned int len);
EXTERN int   HostIF_CopyToUser(void *dst, const void *src, unsigned int len);
EXTERN MPN   HostIF_LockPage(VMDriver *vm, void *addr, Bool allowMultipleMPNsPerVA);
EXTERN int   HostIF_UnlockPage(VMDriver *vm, void *addr);
EXTERN int   HostIF_UnlockPageByMPN(VMDriver *vm, MPN mpn, void* va);
EXTERN Bool  HostIF_IsLockedByMPN(VMDriver *vm, MPN mpn);
EXTERN int   HostIF_FreeAllResources(VMDriver *vm);
EXTERN uint64 HostIF_ReadUptime(void);
EXTERN uint64 HostIF_UptimeFrequency(void);
EXTERN unsigned int HostIF_EstimateLockedPageLimit(const VMDriver *vm, 
 						   unsigned int lockedPages);
EXTERN void  HostIF_WaitForFreePages(unsigned int timeoutMs);
EXTERN Bool  HostIF_IsAnonPage(VMDriver *vm, MPN mpn);
EXTERN void  HostIF_GlobalLock(int callerID);
EXTERN void  HostIF_GlobalUnlock(int callerID);
EXTERN Bool  HostIF_GetNUMAAnonPageDistribution(VMDriver *vm, int numNodes, 
                                                uint32 *perNodeCnt);
#ifdef VMX86_DEBUG
Bool HostIF_GlobalLockIsHeld(void);
#endif
EXTERN void  HostIF_VMLock(VMDriver *vm, int callerID);
EXTERN void  HostIF_VMUnlock(VMDriver *vm, int callerID);
#ifdef VMX86_DEBUG
Bool HostIF_VMLockIsHeld(VMDriver *vm);
#endif

EXTERN MA    HostIF_APICBase(VMDriver *vm, Bool setVMPtr, Bool probe);
EXTERN uint8 HostIF_APIC_ID(void);
EXTERN uint8 HostIF_GetInitialApicId(void);

EXTERN int   HostIF_SemaphoreWait(VMDriver *vm, Vcpuid vcpuid,
                                  int fd, int timeoutms);
EXTERN int   HostIF_SemaphoreSignal(int fd);
EXTERN void  HostIF_SemaphoreForceWakeup(VMDriver *vm, Vcpuid vcpuid);
EXTERN Bool  HostIF_IPI(VMDriver *vm, VCPUSet vcs, Bool all);

EXTERN void  HostIF_UserCall(VMDriver *vm, Vcpuid vcpuid);
EXTERN Bool  HostIF_UserCallWait(VMDriver *vm, Vcpuid vcpuid, int timeoutms);
EXTERN void  HostIF_AwakenVcpu(VMDriver *vm, Vcpuid vcpuid);

#if defined __APPLE__
/*
 * On Mac OS, MonitorLoopCrossUserCallPoll() unconditionnally does this for all
 * cross usercalls with VMMon_LowerCrossUserCallEvent() entirely in user mode.
 */
#define HostIF_AckUserCall(_vm, _vcpuid) do {} while (0)
#else
EXTERN void  HostIF_AckUserCall(VMDriver *vm, Vcpuid vcpuid);
#endif

EXTERN uint32 HostIF_GetCurrentPCPU(void);
EXTERN unsigned int HostIF_NumOnlineLogicalCPUs(void);

#if defined _WIN32 || defined __APPLE__
/*
 * Both the Windows and the Mac OS kernel can preempt the user thread running
 * in the driver's ioctl method.
 */
#define HostIF_Reschedule() do {} while (0)
#else
/*
 * Old versions of the Linux kernel cannot preempt the user thread running in
 * the driver's ioctl method.
 */
EXTERN void HostIF_Reschedule(void);
#endif

#if defined(USE_PERFCTRS_HOSTED) && defined(linux)
EXTERN MA HostIF_APICEnable(VMDriver *vm);
EXTERN Bool HostIF_APICSetPerfCtr(VMDriver *vm, int gate);
EXTERN void HostIF_APICAck(VMDriver *vm);
#endif 

EXTERN int
HostIF_AllocLockedPages(VMDriver *vm, MPN32 *pmpn, 
			unsigned int numPages, Bool kernelMPNBuffer);
EXTERN int
HostIF_FreeLockedPages(VMDriver *vm, const MPN32 *pmpn, 
		       unsigned int numPages, Bool kernelMPNBuffer);
EXTERN int 
HostIF_GetLockedPageList(VMDriver* vm, MPN32* mpns, unsigned int numPages);

EXTERN int HostIF_ReadPage(MPN mpn, void* buf, Bool kernelBuffer);
EXTERN int HostIF_WritePage(MPN mpn, const void* buf, Bool kernelBuffer);
#if defined __APPLE__
// There is no need for a fast clock lock on Mac OS.
#define HostIF_FastClockLock(_callerID) do {} while (0)
#define HostIF_FastClockUnlock(_callerID) do {} while (0)
#else
EXTERN void HostIF_FastClockLock(int callerID);
EXTERN void HostIF_FastClockUnlock(int callerID);
#endif
EXTERN int HostIF_SetFastClockRate(unsigned rate);
EXTERN MPN HostIF_COWAllocPage(void);
EXTERN void HostIF_COWFreePage(MPN mpn);

#define SYSENTER_SENTINEL_CS   ((Selector)~0U)
#define SYSENTER_SENTINEL_RIP  (~(uint64)0)
#define SYSENTER_SENTINEL_RSP  (~(uint64)0)

#define SYSCALL_SENTINEL_STAR   (~(uint64)0)
#define SYSCALL_SENTINEL_LSTAR  (~(uint64)0)
#define SYSCALL_SENTINEL_CSTAR  (~(uint64)0)
#define SYSCALL_SENTINEL_SFMASK (~(uint64)0)

EXTERN void
HostIF_SystemCallReset(int numLogicalCPUs, SystemCallRegisters *regs);
#ifdef linux
EXTERN void HostIF_GetAllCpuInfo(CPUIDResult *info);
EXTERN int HostIF_MarkLockedVARangeClean(const VMDriver *vm, VA uvAddr, 
                                         unsigned len, VA32 bv);
EXTERN void HostIF_PollListLock(int callerID);
EXTERN void HostIF_PollListUnlock(int callerID);
struct page;
EXTERN void *HostIF_MapUserMem(VA32 addr, size_t size, struct page **page);
EXTERN void HostIF_UnmapUserMem(struct page **page);
#endif
EXTERN int
HostIF_MapLockedPages(VMDriver *vm, VA32 va, const MPN32 *pmpn, 
                      unsigned int numPages);
EXTERN int
HostIF_UnmapLockedPages(VMDriver *vm, VA32 va, const MPN32 *pmpn, 
                        unsigned int numPages);
EXTERN void  HostIF_GetCPUID(int index, CPUIDRegs* regs);
EXTERN uint64 HostIF_RDMSR(int index);
EXTERN void  HostIF_WRMSR(int index, uint64 qword);
EXTERN void *HostIF_UserToDriverPtr(VMDriver *vm, VA addr);
EXTERN int   HostIF_MarkPageDirty(VMDriver *vm, void *addr);
EXTERN VA    HostIF_CheckMemory(VMDriver *vm);
EXTERN int   HostIF_IsMPNLocked(VMDriver *vm, MPN mpn);
EXTERN VPN   HostIF_GetRecycledPage(VMDriver *vm);
EXTERN int   HostIF_ReleaseAnonPage(VMDriver *vm, MPN mpn);
EXTERN int   HostIF_MarkLockedVARangeCleanV45(const VMDriver *vm, VA uvAddr, unsigned len);
EXTERN MA    HostIF_IOAPICBaseV45(void);

#endif // ifdef _HOSTIF_H_
