/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmciProcess.c --
 *
 *     VMCI Process code. 
 */

#ifdef linux
#   include "driver-config.h"
#   include <linux/string.h> /* memset() in the kernel */
#elif defined(WINNT_DDK)
#   include <string.h>
#elif !defined(__APPLE__)
#   error "Unknown platform"
#endif

#include "vmciProcess.h"
#include "vmciDriver.h"
#include "vmciDatagram.h"
#include "vmciSharedMem.h"
#include "vmware.h"
#include "circList.h"
#include "hostif.h"
#include "vmci_kernel_defs.h"
#include "vmciCommonInt.h"

#define LGPFX "VMCIProcess: "

static struct {
   ListItem *head;
   VMCILock lock;
} processList;


/*
 *----------------------------------------------------------------------
 *
 * VMCIProcess_Init --
 *
 *      Initialize the process module.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VMCIProcess_Init(void) {
   processList.head = NULL;
   VMCI_InitLock(&processList.lock);
   return VMCI_SUCCESS;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIProcess_Create --
 *
 *      Creates a new VMCI process.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VMCIProcess_Create(VMCIProcess **outProcess,  // IN
                   int eventHnd)              // IN
{
   VMCIProcess *process;
   int result;
   VMCILockFlags flags;

   process = VMCI_AllocKernelMem(sizeof *process,
                                 VMCI_MEMORY_NONPAGED);
   if (process == NULL) {
      Log(LGPFX "Failed to allocate memory for process.\n");
      result = VMCI_ERROR_NO_MEM;
      goto error;
   }

   process->pid = (VMCIId)(uintptr_t)process >> 1;

   process->sharedMemHandles = VMCIHandleArray_Create(0);
   if (process->sharedMemHandles == NULL) {
      result = VMCI_ERROR_NO_MEM;
      goto error;
   }

   VMCI_GrabLock(&processList.lock, &flags);
   LIST_QUEUE(&process->listItem, &processList.head);
   VMCI_ReleaseLock(&processList.lock, flags);

   *outProcess = process;
   return 0;

error:
   if (process) {
      if (process->sharedMemHandles) {
	 VMCIHandleArray_Destroy(process->sharedMemHandles);
      }
      VMCI_FreeKernelMem(process);
   }
   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIProcess_Destroy --
 *
 *      Destroys a vmci process.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
VMCIProcess_Destroy(VMCIProcess *process)
{
   VMCIHandle handle;
   VMCILockFlags flags;
   VMCIContext *hostCtx;

   /* Cleanup the datagrams still open by process. */
   ASSERT(process->sharedMemHandles);

   /* Clean up all shared memory regions still attached to by the process. */
   hostCtx = VMCIContext_Get(VMCI_HOST_CONTEXT_ID);
   ASSERT(hostCtx != NULL); // Host context must be around when executing this.
   handle = VMCIHandleArray_GetEntry(process->sharedMemHandles, 0);
   while (handle != VMCI_INVALID_HANDLE) {
      if (VMCISharedMem_Detach(hostCtx, process, handle) < VMCI_SUCCESS) {

	 /* 
	  * We rely on VMCISharedMem_Detach to remove the handle from the 
	  * list so if we fail we must remove it here instead.
	  */
	 VMCIHandleArray_RemoveEntry(process->sharedMemHandles, handle);
      }
      /* Detach removes the handle from the process' array. */
      handle = VMCIHandleArray_GetEntry(process->sharedMemHandles, 0);
   }

   /* Dequeue process. */
   VMCI_GrabLock(&processList.lock, &flags);
   LIST_DEL(&process->listItem, &processList.head);
   VMCI_ReleaseLock(&processList.lock, flags);

   VMCIHandleArray_Destroy(process->sharedMemHandles);
   VMCI_FreeKernelMem(process);
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIProcess_Get --
 *
 *      Get the process corresponding to the pid.
 *
 * Results:
 *      VMCI process on success, NULL otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

VMCIProcess *
VMCIProcess_Get(VMCIId processID)  // IN
{
   VMCIProcess *process = NULL;  
   ListItem *next;
   VMCILockFlags flags;

   VMCI_GrabLock(&processList.lock, &flags);
   if (LIST_EMPTY(processList.head)) {
      goto out;
   }

   LIST_SCAN(next, processList.head) {
      process = LIST_CONTAINER(next, VMCIProcess, listItem);
      if (process->pid == processID) {
         break;
      }
   }

out:
   VMCI_ReleaseLock(&processList.lock, flags);
   return (process && process->pid == processID) ? process : NULL;
}

