/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * x86paging.h --
 *
 *      Contains definitions for the x86 page table layout.
 */

#ifndef _X86PAGING_H_
#define _X86PAGING_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMNIXMOD
#define INCLUDE_ALLOW_VMK_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#define INCLUDE_ALLOW_DISTRIBUTE
#define INCLUDE_ALLOW_VMCORE
#include "includeCheck.h"

/*
 *   page tables
 */

#define PTE_P           0x00000001
#define PTE_RW          0x00000002
#define PTE_US          0x00000004
#define PTE_PWT         0x00000008
#define PTE_PCD         0x00000010 // careful: MMU_PTE_EX has same value.
#define PTE_A           0x00000020
#define PTE_D           0x00000040
#define PTE_PS          0x00000080
#define PTE_G           0x00000100

#define PTE_PROT_FLAGS  (PTE_P|PTE_RW|PTE_US)
#define PTE_FLAGS       (PTE_PROT_FLAGS|PTE_G)

#define PTE_PAGE_TABLE  (PTE_P|PTE_RW|PTE_US|PTE_A|PTE_D) 

/*
 * This is usable for 4M or 2M pde's (both PAE and non-PAE). Use the PTE
 * macros for large pages.
 */ 
#define PDE_PAGE_TABLE  (PTE_P | PTE_RW | PTE_US | PTE_A) 

#define PTE_PRESENT(_pte)   (((_pte) & PTE_P) != 0)
#define PTE_WRITEABLE(_pte) (((_pte) & PTE_RW) != 0)
#define PTE_ACCESS(_pte)    (((_pte) & PTE_A) != 0)
#define PTE_DIRTY(_pte)     (((_pte) & PTE_D) != 0)
#define PTE_USER(_pte)      (((_pte) & PTE_US) != 0)
#define PTE_GLOBAL(_pte)    (((_pte) & PTE_G) != 0)
#define PTE_LARGEPAGE(_pte) (((_pte) & PTE_PS) != 0)
#define PTE_UNCACHED(_pte)  (((_pte) & PTE_PCD) != 0)

#define PTE_AVAIL_MASK       0xe00
#define PTE_AVAIL_SHIFT      9
#define PTE_PFN_MASK         0xfffff000
#define PAE_PTE_PFN_MASK     CONST64U(0xffffff000)
#define LM_PTE_PFN_MASK      CONST64U(0xffffffffff000)
#define PTE_PFN_SHIFT        12
#define PAE_PTE_PFN_SHIFT    12
#define LM_PTE_PFN_SHIFT     12
#define PTE_2_PFN(_pte)      (((_pte) & PTE_PFN_MASK) >> PTE_PFN_SHIFT)
#define LM_PTE_2_PFN(_pte)   (((_pte) & LM_PTE_PFN_MASK) >> LM_PTE_PFN_SHIFT)
#define PTE_AVAIL(_pte)      (((_pte) & PTE_AVAIL_MASK) >> PTE_AVAIL_SHIFT)

#define PDE_PFN_MASK         0xffc00000
#define PAE_PDE_LARGE_PFN_MASK     0xfffe00000LL
#define LM_PDE_PFN_MASK      0xfffffffe00000LL
#define PDE_PFN_SHIFT        PTE_PFN_SHIFT

/* Use these only for 4M or 2M pde's, not the 4K ones. */

/* REMOVE : This macro is used by frobos only.
 * They should use LM_PTE_PFN_MASK instead.
 */
#define PDPTE_PFN_MASK       CONST64U(0xffffffffff000)

#define NUM_PAE_PDIRS        4
#define PDES_PER_PDIR        (PAGE_SIZE/sizeof(VM_PDE))
#define PTES_PER_PDIR        PDES_PER_PDIR
#define PAE_PDES_PER_PDIR    (PAGE_SIZE/sizeof(VM_PAE_PDE))
#define PTES_PER_PGTBL       (PAGE_SIZE/sizeof(VM_PTE))
#define PAE_PTES_PER_PGTBL   (PAGE_SIZE/sizeof(VM_PAE_PTE))
#define L1ES_PER_PGTBL       (PAGE_SIZE/sizeof(VM_L1E))

#define NONPAE_LARGE_PAGE      (PDES_PER_PDIR     * PAGE_SIZE)
#define LM_PAE_LARGE_PAGE      (PAE_PDES_PER_PDIR * PAGE_SIZE)

#define NONPAE_LARGE_PAGE_MASK (NONPAE_LARGE_PAGE - 1)
#define LM_PAE_LARGE_PAGE_MASK (LM_PAE_LARGE_PAGE - 1)

#define PGOFF_MASK            0x3ff
#define PAE_PGOFF_MASK        0x1ff
#define PDOFF_MASK            0x3ff
#define PAE_PDOFF_MASK        0x1ff
#define PDPTOFF_MASK          0x3
#define PAE_RANGE_MASK        0x7ff
#define LM_L4OFF_MASK         LM_L1OFF_MASK
#define LM_L3OFF_MASK         LM_L1OFF_MASK
#define LM_L2OFF_MASK         LM_L1OFF_MASK
#define LM_L1OFF_MASK         0x1ff

#define PDE_4M_RSVD_MASK      (1 << 21)
#define WIDE_PDE_2M_RSVD_MASK 0x1fe000

#define PFN_2_PDPTOFF(_a)     (((_a) >> 18) & PDPTOFF_MASK)
#define PFN_2_PDOFF(_a)       (((_a) >> 10) & PDOFF_MASK)
#define PAE_PFN_2_PDOFF(_a)   (((_a) >> 9) & PAE_PDOFF_MASK)
#define PFN_2_PGOFF(_a)       ((_a) & PGOFF_MASK)
#define PAE_PFN_2_PGOFF(_a)   ((_a) & PAE_PGOFF_MASK)
#define PFN_2_RANGE(_pfn)     PFN_2_PDOFF(_pfn)
#define PAE_PFN_2_RANGE(_pfn) (((_pfn) >> 9) & PAE_RANGE_MASK)

#define RANGE_2_LA(_r)        ((_r) << 22)
#define PAE_RANGE_2_LA(_r)    ((_r) << 21)
#define RANGE_2_PDPTOFF(_r)   (((_r) >> 9) & 0x3)
#define RANGE_2_PDOFF(_r)     ((_r))
#define PAE_RANGE_2_PDOFF(_r) ((_r) & 0x1ff)

#define LA_2_PDPTOFF(_a)      (((_a) >> 30) & PDPTOFF_MASK)
#define LA_2_PDOFF(_a)        (((_a) >> 22) & PDOFF_MASK)
#define PAE_LA_2_PDOFF(_a)    (((_a) >> 21) & PAE_PDOFF_MASK)
#define LA_2_PGOFF(_a)        (PFN_2_PGOFF(VA_2_VPN(_a)))
#define PAE_LA_2_PGOFF(_a)    (PAE_PFN_2_PGOFF(VA_2_VPN(_a)))
#define LA_2_RANGE(_a)        LA_2_PDOFF((_a))
#define PAE_LA_2_RANGE(_a)    (((_a) >> 21) & PAE_RANGE_MASK)
#define PAE_LA_2_PDPTOFF(_a)  (((_a) >> 30) & 0x3)

#define LM_LA_2_L4OFF(_a)    (((_a) >> 39) & LM_L4OFF_MASK)
#define LM_LA_2_L3OFF(_a)    (((_a) >> 30) & LM_L3OFF_MASK)
#define LM_LA_2_L2OFF(_a)    (((_a) >> 21) & LM_L2OFF_MASK)
#define LM_LA_2_L1OFF(_a)    (((_a) >> 12) & LM_L1OFF_MASK)

#define LM_LPN_2_L4OFF(_a)   (((_a) >> (39 - LM_PTE_PFN_SHIFT)) & LM_L4OFF_MASK)
#define LM_LPN_2_L3OFF(_a)   (((_a) >> (30 - LM_PTE_PFN_SHIFT)) & LM_L3OFF_MASK)
#define LM_LPN_2_L2OFF(_a)   (((_a) >> (21 - LM_PTE_PFN_SHIFT)) & LM_L2OFF_MASK)
#define LM_LPN_2_L1OFF(_a)   (((_a) >> (12 - LM_PTE_PFN_SHIFT)) & LM_L1OFF_MASK)

/* This following macro only work for flat model were VA==LA */
#define VA_2_PDOFF(_a)        ((_a) >> 22)
#define VA_2_PGOFF(_a)        (PFN_2_PGOFF(VA_2_VPN(_a)))

#define PTE_2_PA(_a)          (PPN_2_PA(PTE_2_PFN(_a)))

#define PDE_FLAGS(_pde)       ((_pde) & 0xfff)  
#define PAE_PDE_FLAGS(_pde)   ((_pde) & 0xfff)  

/* Error code flags */
#define PF_P            0x1
#define PF_RW           0x2
#define PF_US           0x4
#define PF_RSVD         0x8
#define PF_ID           0x10
/*
 * Macros for LM page table used by x86_64.
 */
#define LM_OFF_MASK       0x1ff

#define PFN_2_L4OFF(_a)   (((_a) >> 27) & LM_OFF_MASK)
#define PFN_2_L3OFF(_a)   (((_a) >> 18) & LM_OFF_MASK)
#define PFN_2_L2OFF(_a)   (((_a) >> 9) & LM_OFF_MASK)
#define PFN_2_L1OFF(_a)    ((_a) & LM_OFF_MASK)

#define LM_PTE_SHIFT          12
#define LM_A1_SHIFT           52
#define LM_A2_SHIFT           9

#define LM_A1_MASK            CONST64(0x7ff)
#define LM_A2_MASK            CONST64(0x7)
#define LM_FLAGS_MASK         CONST64(0x80000000000001ff)
#define LM_CR3_FLAGS_MASK     CONST64(0x18)
#define PTE_NX                CONST64(0x8000000000000000)

#define LM_MAKE_CR3(_mpfn, _flags) \
                   (((uint64)(_mpfn) << LM_PTE_SHIFT) | \
                   ((_flags) & LM_CR3_FLAGS_MASK))

#define LM_MAKE_PTE(_mpfn, _a1, _a2, _flags) \
                   (((uint64)(_mpfn) << LM_PTE_SHIFT) | \
                   ((((uint64)_a1) & LM_A1_MASK) << LM_A1_SHIFT)\
                   | (((_a2) & LM_A2_MASK) << LM_A2_SHIFT) \
                   | ((uint64)(_flags) & LM_FLAGS_MASK))

#define LM_MAKE_PDE(_pfn, _a1, _a2, _flags) LM_MAKE_PTE(_pfn, _a1, _a2, _flags)
#define LM_MAKE_L4E(_pfn, _a1, _a2, _flags) LM_MAKE_PTE(_pfn, _a1, _a2, _flags)
#define LM_MAKE_L3E(_pfn, _a1, _a2, _flags) LM_MAKE_PTE(_pfn, _a1, _a2, _flags)
#define LM_MAKE_L2E(_pfn, _a1, _a2, _flags) LM_MAKE_PTE(_pfn, _a1, _a2, _flags)
#define MAKE_PTE(_mpfn, _avail, _flags) \
                        (((_mpfn) << PTE_PFN_SHIFT) \
                         | (((_avail) << PTE_AVAIL_SHIFT) & PTE_AVAIL_MASK) \
                         | (_flags))

// PAE_MAKE_PTE and PAE_MAKE_PDE should be removed 
#define PAE_MAKE_PTE(_mpfn, _avail, _flags) \
     ((((VM_PAE_PTE) (_mpfn)) << PAE_PTE_PFN_SHIFT) \
      | (((_avail) << PTE_AVAIL_SHIFT) & PTE_AVAIL_MASK) \
      | (_flags))

#define MAKE_PDE(_mpfn, _avail, _flags) MAKE_PTE(_mpfn, _avail, _flags)
#define PAE_MAKE_PDE(_mpfn, _avail, _flags) PAE_MAKE_PTE(_mpfn, _avail, _flags)
#define MAKE_PDPTE(_mpfn, _avail, _flags) LM_MAKE_PTE(_mpfn, 0, _avail, _flags)
#define PDPTE_FLAGS (PTE_P)

/* 
 *---------------------------------------------------------------------------
 * IsGoodMPN --
 *    Return TRUE if "mpn" looks plausible. We could make this
 *    test stricter in the non-PAE case. We could even take into
 *    account the amount of memory on the host.
 *---------------------------------------------------------------------------
 */

static INLINE Bool
IsGoodMPN(MPN mpn)
{
   return mpn <= MAX_MPN;
}


/*
 * x86-64 architecture requires implementations supporting less than
 * full 64-bit VAs to ensure that all virtual addresses are in canonical
 * form. An address is in canonical form if the address bits from the
 * most significant implemented bit up to bit 63 are all ones or all
 * zeros. If this is not the case, the processor generates #GP/#SS. Our
 * VCPU implements 48 bits of virtual address space.
 */

#define VA64_IMPL_BITS             48
#define VA64_CANONICAL_MASK        ~((CONST64U(1) << (VA64_IMPL_BITS - 1)) - 1)
#define VA64_CANONICAL_HOLE_START   (CONST64U(1) << (VA64_IMPL_BITS - 1))
#define VA64_CANONICAL_HOLE_LEN  VA64_CANONICAL_MASK - VA64_CANONICAL_HOLE_START

#endif /* _X86PAGING_H_ */
