/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmciContext.c --
 *
 *     Platform independent routines for VMCI calls.
 */

#ifdef linux
#   include "driver-config.h"
#   include <linux/string.h> /* memset() in the kernel */
#elif defined(WINNT_DDK)
#   include <string.h>
#elif !defined(__APPLE__)
#   error "Unknown platform"
#endif

#include "vmciContext.h"
#include "vmci_infrastructure.h"
#include "vmciDatagram.h"
#include "vmciSharedMem.h"
#include "vmciDriver.h"
#include "vmciResource.h"
#include "vmciDsInt.h"
#include "vmciGroup.h"
#include "vmware.h"
#include "circList.h"
#include "hostif.h"
#include "vmci_kernel_defs.h"
#include "vmciCommonInt.h"

#define LGPFX "VMCIContext: "

static int VMCIContextGetID(VMCIId contextID, void* clientData,
                            VMCICall *call);
static int VMCIContextVectorsInformHandler(VMCIId contextID,
                                           void* clientData,
                                           VMCICall *call);

static int64 VMCIContextDriverHandler(void *vmciObj,
                                      VMCIObjType vmciObjType,
                                      VMCICall *call);

/*
 * List of current VMCI contexts.
 */

static struct {
   ListItem *head;
   VMCILock lock;
} contextList;


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_Init --
 *
 *      Initializes the VMCI context module.
*
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VMCIContext_Init(void)
{
   contextList.head = NULL;
   VMCI_InitLock(&contextList.lock);

   VMCI_RegisterHandler(VMCI_VECTORS_INFORM,
                        VMCIContextVectorsInformHandler,
                        0 , NULL);
   VMCI_RegisterHandler(VMCI_GET_CONTEXT_ID, VMCIContextGetID, 
                        0 , NULL);

   VMCI_RegisterDriverHandler(VMCI_GET_CONTEXT_ID,
                              VMCIContextDriverHandler);

   return VMCI_SUCCESS;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_Exit --
 *
 *      Cleans up the contexts module.
*
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
VMCIContext_Exit(void)
{
}

/*
 *-------------------------------------------------------------------------
 *
 *  VMCIContextGetID --
 *
 *     Hypercall handler returning context id.
 *
 *  Result:
 *     Context id.
 *     
 *-------------------------------------------------------------------------
 */

static int 
VMCIContextGetID(VMCIId contextID,  // IN:
                 void* clientData,       // IN: Currently unused
                 VMCICall *call)    // IN:
{
   if (call == NULL || call->h.size != VMCI_CALL_HEADERSIZE) {
      return VMCI_ERROR_INVALID_ARGS;
   }
   return contextID;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_InitContext --
 *
 *      Allocates and initializes a VMCI context. 
 *
 * Results:
 *      Returns 0 on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VMCIContext_InitContext(VMCIId cid,               // IN
                        uintptr_t eventHnd,            // IN
                        VMCIContext **outContext)      // OUT
{
   VMCILockFlags flags;
   VMCIContext *context;
   int result;

   context = VMCI_AllocKernelMem(sizeof *context, VMCI_MEMORY_NONPAGED);
   if (context == NULL) {
      Log(LGPFX "Failed to allocate memory for VMCI context.\n");
      return VMCI_ERROR_NO_MEM;
   }
   context->datagramArray = NULL;
   context->sharedMemArray = NULL;
   context->groupArray = NULL;
   context->guestCallQueue = NULL;
   context->pendingGuestCalls = 0;
   VMCI_InitLock(&context->guestCallLock);

   /* 
    * If a new cid has been requested, return one. If cid has been 
    * specified, verify that it isnt already in use. 
    */
   context->cid = cid;
   if (context->cid == VMCI_INVALID_ID) {
       context->cid = (unsigned)(uintptr_t)context >> 1;
   } else if (VMCIContext_Get(context->cid) != NULL) {
      Log(LGPFX "CID already exists.\n");
      result = VMCI_ERROR_DUPLICATE_ENTRY;
      goto error;
   }
   
   context->datagramArray = VMCIHandleArray_Create(0);
   if (context->datagramArray == NULL) {
      result = VMCI_ERROR_NO_MEM;
      goto error;
   }

   context->sharedMemArray = VMCIHandleArray_Create(0);
   if (context->sharedMemArray == NULL) {
      result = VMCI_ERROR_NO_MEM;
      goto error;
   }

   context->groupArray = VMCIHandleArray_Create(0);
   if (context->groupArray == NULL) {
      result = VMCI_ERROR_NO_MEM;
      goto error;
   }

   context->numSupportedGuestCalls = 0;
   context->supportedGuestCalls = NULL;

   /* Inititialize host-specific VMCI context. */
   VMCIHost_InitContext(&context->hostContext, eventHnd);

   VMCI_GrabLock(&contextList.lock, &flags);
   LIST_QUEUE(&context->listItem, &contextList.head);
   VMCI_ReleaseLock(&contextList.lock, flags);

   *outContext = context;
   return VMCI_SUCCESS;

error:
   if (context->datagramArray) {
      VMCIHandleArray_Destroy(context->datagramArray);
   }
   if (context->sharedMemArray) {
      VMCIHandleArray_Destroy(context->sharedMemArray);
   }
   if (context->groupArray) {
      VMCIHandleArray_Destroy(context->groupArray);
   }
   VMCI_FreeKernelMem(context);
   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_ReleaseContext --
 *
 *      Cleans up a VMCI context.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
VMCIContext_ReleaseContext(VMCIContext *context)   // IN
{
   ListItem *tempGuestCallQueue;
   ListItem *curr;
   ListItem *next;
   CallEntry *gcEntry;
   VMCIHandle tempHandle;
   VMCILockFlags flags;

   /*
    * Cleanup all datagram resources owned by context. Ideally these would
    * be removed already but we maintain this list to make sure no resources
    * are leaked. It is updated by the VMCIDatagramCreate/DestroyHnd functions.
    */
   ASSERT(context->datagramArray);
   tempHandle = VMCIHandleArray_RemoveTail(context->datagramArray);
   while (tempHandle != VMCI_INVALID_HANDLE) {
      VMCIResource_Remove(tempHandle, VMCI_RESOURCE_TYPE_DATAGRAM);
      tempHandle = VMCIHandleArray_RemoveTail(context->datagramArray);
   }

   /*
    * Cleanup all sharedmem resources attached to context. Ideally these would
    * be detached already but we maintain this list to make sure no resources
    * are leaked. The array is updated by the VMCISharedMemCreate/Attach and 
    * VMCISharedMem_Detach functions.
    */
   ASSERT(context->sharedMemArray);
   tempHandle = VMCIHandleArray_GetEntry(context->sharedMemArray, 0);
   while (tempHandle != VMCI_INVALID_HANDLE) {
      if (VMCISharedMem_Detach(context, NULL, tempHandle) < VMCI_SUCCESS) {

	 /* 
	  * We rely on VMCISharedMem_Detach to remove the handle from the 
	  * list so if we fail we must remove it here instead.
	  */
	 VMCIHandleArray_RemoveEntry(context->sharedMemArray, tempHandle);
      }
      /* Detach removes the handle from the context's array. */
      tempHandle = VMCIHandleArray_GetEntry(context->sharedMemArray, 0);
   }

   /*
    * Check that the context has been removed from all the groups it was a
    * member of. If not, remove it from the group.
    */
   ASSERT(context->groupArray);
   tempHandle = VMCIHandleArray_RemoveTail(context->groupArray);
   while (tempHandle != VMCI_INVALID_HANDLE) {
      Log(LGPFX"Removing context 0x%x from group 0x%"FMT64"x during release.\n",
          context->cid, tempHandle);
      VMCIGroup_RemoveMember(tempHandle, VMCI_MAKE_HANDLE(context->cid,
                                                      VMCI_CONTEXT_RESOURCE_ID));
      tempHandle = VMCIHandleArray_RemoveTail(context->groupArray);
   }

   /* Dequeue VMCI context. */
   VMCI_GrabLock(&contextList.lock, &flags);
   LIST_DEL(&context->listItem, &contextList.head);
   VMCI_ReleaseLock(&contextList.lock, flags);

   /*
    * Flush guest call queue for this VM. We need to free the guestcalls
    * while not holding the guestCallLock, since the calls themselves
    * are allocated in NonPagedPool on Windows, so we cannot access them
    * while holding a spinlock.
    */
   VMCI_GrabLock(&context->guestCallLock, &flags);
   tempGuestCallQueue = context->guestCallQueue;
   context->guestCallQueue = NULL;
   VMCI_ReleaseLock(&context->guestCallLock, flags);

   LIST_SCAN_SAFE(curr, next, tempGuestCallQueue) {
      gcEntry = LIST_CONTAINER(curr, CallEntry, listItem);
      LIST_DEL(curr, &tempGuestCallQueue);
      ASSERT(gcEntry && gcEntry->call);
      VMCI_FreeKernelMem(gcEntry->call);
      VMCI_FreeKernelMem(gcEntry);
   }

   VMCIHandleArray_Destroy(context->datagramArray);
   VMCIHandleArray_Destroy(context->sharedMemArray);
   VMCIHandleArray_Destroy(context->groupArray);
   if (context->supportedGuestCalls != NULL) {
      VMCI_FreeKernelMem(context->supportedGuestCalls);
   }
   VMCI_FreeKernelMem(context);
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCIContext_SupportsGuestCall --
 *
 *      Check whether context supports guestcall vector.
 *
 * Results:
 *      TRUE if vector is supported by context, FALSE otherwise or if there
 *      is no information on supported guestcalls on context.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

Bool
VMCIContext_SupportsGuestCall(VMCIContext *context, // IN
                              VMCI_Vector vector)   // IN
{
   int i;

   if (context->supportedGuestCalls == NULL) {
      return FALSE;
   }

   for (i = 0; i < context->numSupportedGuestCalls; i++) {
      if (context->supportedGuestCalls[i] == vector) {
         return TRUE;
      }
   }

   return FALSE;
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCIContextVectorsInformHandler --
 *
 *      Called by guest contexts to let us know which guestcalls they support.
 *      This information is used to fail unsupported guestcalls (in 
 *      VMCIContext_Send). The guestcalls are also shown to the various API,
 *      which can fail the inform if the context does not support a required
 *      guestcalls.
 *
 * Results:
 *      VMCI_SUCCESS if successful, appropriate VMCI_ERROR_* otherwise.
 *
 * Side effects:
 *      Guestcall information is stored in the context struct.
 *
 *-----------------------------------------------------------------------------
 */

static int
VMCIContextVectorsInformHandler(VMCIId contextID, // IN
                                void* unused,          // IN
                                VMCICall *call)   // IN
{
   VMCIContext *context;
   VMCIVectorsMsg *msg;
   uint32 size;
   VMCI_Vector *guestCalls;
   Bool supported;

   if (!call || (call->h.size < sizeof(VMCIVectorsHdr))) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   context = VMCIContext_Get(contextID);
   if (context == NULL) {
      return VMCI_ERROR_NO_RESOURCES;
   }

   msg = (VMCIVectorsMsg *)call->args;
   size = msg->numVectors * sizeof *context->supportedGuestCalls;

   if (call->h.size != (size + sizeof(VMCIVectorsHdr))) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   guestCalls = VMCI_AllocKernelMem(size, 0);
   if (guestCalls == NULL) {
      return VMCI_ERROR_NO_MEM;
   }

   memcpy(guestCalls, msg->vectors, size);

   if (context->supportedGuestCalls != NULL) {
      VMCI_FreeKernelMem(context->supportedGuestCalls);
   }

   context->supportedGuestCalls = guestCalls;
   context->numSupportedGuestCalls = msg->numVectors;

   supported = VMCI_CheckGuestCalls(context);
   supported &= VMCIContext_CheckGuestCalls(context);
   supported &= VMCIDatagram_CheckGuestCalls(context);
   supported &= VMCIResource_CheckGuestCalls(context);
   supported &= VMCISharedMem_CheckGuestCalls(context);

   if (supported) {
      return VMCI_SUCCESS;
   } else {
      return VMCI_ERROR_NO_GUESTCALL;
   }
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_SendCall --
 *
 *      Queues a VMCI guest call for the appropriate target VM 
 *      context.
 *
 * Results:
 *      0 on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VMCIContext_SendCall(VMCIId cid,          // IN: Target VM
                     VMCICall *guestCall, // IN:
                     Bool mustCopy)            // IN: TRUE if must copy.
{
   CallEntry *gcEntry;
   VMCIContext *context;
   VMCILockFlags flags;

   if (guestCall == NULL || guestCall->h.size < VMCI_CALL_HEADERSIZE) {
      Log(LGPFX "Invalid call parameters received\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (guestCall->h.size > VMCI_CALL_STANDARD_SIZE) {
      Log(LGPFX"Guestcall too large.\n");
      return VMCI_ERROR_NO_MEM;
   }

   /* Get the target VM's VMCI context. */
   context = VMCIContext_Get(cid);
   if (context == NULL) {
      Log(LGPFX "Invalid cid\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   /* Verify that target context supports the guestcall. */
   if (!VMCIContext_SupportsGuestCall(context, guestCall->h.vector)) {
      Log(LGPFX"Context %d does not support guestcall %d\n",
          cid, guestCall->h.vector);
      return VMCI_ERROR_INVALID_VECTOR;
   }

   /* Allocate guest call entry and add it to the target VM's queue. */
   gcEntry = VMCI_AllocKernelMem(sizeof *gcEntry, VMCI_MEMORY_NONPAGED);
   if (gcEntry == NULL) {
      Log(LGPFX "Failed to allocate memory for guest call.\n");
      return VMCI_ERROR_NO_MEM;
   }
   if (mustCopy) {
      gcEntry->call = VMCI_AllocKernelMem(guestCall->h.size, 0);
      if (gcEntry->call == NULL) {
         Log(LGPFX "Failed to allocate memory for guest call.\n");
         VMCI_FreeKernelMem(gcEntry);
         return VMCI_ERROR_NO_MEM;
      }
      memcpy(gcEntry->call, guestCall, guestCall->h.size);
   } else {
      gcEntry->call = guestCall;
   }

   VMCI_GrabLock(&context->guestCallLock, &flags);
   if (context->pendingGuestCalls >= MAX_QUEUED_GUESTCALLS_PER_VM) {
      VMCI_ReleaseLock(&context->guestCallLock, flags);
      if (mustCopy) {
         VMCI_FreeKernelMem(gcEntry->call);
      }
      VMCI_FreeKernelMem(gcEntry);
      Log(LGPFX "Too many queued guest calls\n");
      return VMCI_ERROR_NO_RESOURCES;
   }

   LIST_QUEUE(&gcEntry->listItem, &context->guestCallQueue);
   context->pendingGuestCalls++;
   VMCIHost_SignalCall(&context->hostContext);
   VMCI_ReleaseLock(&context->guestCallLock, flags);

   return VMCI_SUCCESS;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_Get --
 *
 *      Retrieves VMCI context corresponding to the given cid.
 *
 * Results:
 *      VMCI context on success, NULL otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

VMCIContext *
VMCIContext_Get(VMCIId cid)  // IN
{
   VMCIContext *context = NULL;  
   ListItem *next;
   VMCILockFlags flags;

   VMCI_GrabLock(&contextList.lock, &flags);
   if (LIST_EMPTY(contextList.head)) {
      goto out;
   }

   LIST_SCAN(next, contextList.head) {
      context = LIST_CONTAINER(next, VMCIContext, listItem);
      if (context->cid == cid) {
         break;
      }
   }

out:
   VMCI_ReleaseLock(&contextList.lock, flags);
   return (context && context->cid == cid) ? context : NULL;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_ReadCall --
 *
 *      Dequeues the next guest call and returns it to user level.
 *
 * Results:
 *      0 on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VMCIContext_ReadCall(VMCIContext *context, // IN
                     void *buf,            // OUT: Userlevel buffer
                     size_t bufSize)       // IN: size of buf
{
   CallEntry *gcEntry;
   ListItem *listItem;
   int result;
   VMCILockFlags flags;

   /* Dequeue the next guestcall entry. */
   VMCI_GrabLock(&context->guestCallLock, &flags);
   if (context->pendingGuestCalls == 0) {
      VMCIHost_ClearCall(&context->hostContext);
      VMCI_ReleaseLock(&context->guestCallLock, flags);
      Log(LGPFX "No guest calls present\n");
      return VMCI_ERROR_NO_GUESTCALL;
   }

   listItem = LIST_FIRST(context->guestCallQueue);
   ASSERT (listItem != NULL);

   gcEntry = LIST_CONTAINER(listItem, CallEntry, listItem);
   ASSERT(gcEntry->call);

   /* Check size of the userland buffer. */
   if (bufSize < gcEntry->call->h.size) {
      VMCI_ReleaseLock(&context->guestCallLock, flags);
      Log(LGPFX "Userland buffer is too small.\n");
      return VMCI_ERROR_NO_MEM;
   }
   
   LIST_DEL(listItem, &context->guestCallQueue);
   context->pendingGuestCalls--;
   if (context->pendingGuestCalls == 0) {
      VMCIHost_ClearCall(&context->hostContext);
   }
   VMCI_ReleaseLock(&context->guestCallLock, flags);

   result = HostIF_CopyToUser(buf, gcEntry->call,
			      gcEntry->call->h.size);
   VMCI_FreeKernelMem(gcEntry->call);
   VMCI_FreeKernelMem(gcEntry);

   return result == 0 ? VMCI_SUCCESS : VMCI_ERROR_NO_MEM;
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCIContextDriverHandler --
 *
 *      Parses the driver call and calls the appropriate function.
 *
 * Results:
 *      Value of called function.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static int64
VMCIContextDriverHandler(void *vmciObj,           // IN:
                         VMCIObjType vmciObjType, // IN:
                         VMCICall *call)          // IN:
{
   int64 retval;
   ASSERT(call);

   switch(call->h.vector) {
   case VMCI_GET_CONTEXT_ID:
      retval = VMCI_HOST_CONTEXT_ID;
      break;

   default:
      retval = VMCI_ERROR_INVALID_VECTOR;
      break;
   }

   return retval;
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCIContext_CheckGuestCalls --
 *
 *      Make sure that context supports the guestcalls we need.
 *
 * Results:
 *      TRUE if context is supported, FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

Bool
VMCIContext_CheckGuestCalls(VMCIContext *context)
{
   /* This part of the API does not need any guestcalls. */
   return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_GetId --
 *
 *      Retrieves cid of given VMCI context.
 *
 * Results:
 *      VMCIId of context on success, VMCI_INVALID_ID otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

VMCIId
VMCIContext_GetId(VMCIContext *context) // IN:
{
   if (!context) {
      return VMCI_INVALID_ID;
   }
   ASSERT(context->cid != VMCI_INVALID_ID);
   return context->cid;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_AddGroupEntry --
 *
 *      Wrapper to call VMCIHandleArray_AppendEntry().
 *
 * Results:
 *      VMCI_SUCCESS on success, error code otherwise.
 *
 * Side effects:
 *      As in VMCIHandleArray_AppendEntry().
 *
 *----------------------------------------------------------------------
 */

int
VMCIContext_AddGroupEntry(VMCIContext *context, // IN:
                          VMCIHandle entryHandle) // IN:
{
   if (!context) {
      return VMCI_ERROR_INVALID_ARGS;
   }
   VMCIHandleArray_AppendEntry(&context->groupArray, entryHandle);
   return VMCI_SUCCESS;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIContext_RemoveGroupEntry --
 *
 *      Wrapper to call VMCIHandleArray_RemoveEntry().
 *
 * Results:
 *      Return value from VMCIHandleArray_RemoveEntry().
 *
 * Side effects:
 *      As in VMCIHandleArray_RemoveEntry().
 *
 *----------------------------------------------------------------------
 */

VMCIHandle
VMCIContext_RemoveGroupEntry(VMCIContext *context, // IN:
                             VMCIHandle entryHandle) // IN:
{
   if (!context) {
      return VMCI_INVALID_HANDLE;
   }
   return VMCIHandleArray_RemoveEntry(context->groupArray, entryHandle);
}
