/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmciDatagram.c --
 *
 *    This file implements the VMCI Simple Datagram API on the host. 
 *    
 *    XXX: This is currently part of vmmon but should probably be moved into
 *         a separate host driver. 
 *
 */

#ifdef linux
#   include "driver-config.h"
#   include <linux/string.h> /* memset() in the kernel */
#elif defined(WINNT_DDK)
#   include <string.h>
#elif !defined (__APPLE__)
#   error "Unknown platform"
#endif

#include "vmci_defs.h"
#include "vmci_kernel_defs.h"
#include "vmciCommonInt.h"
#include "vmciContext.h"
#include "vmciProcess.h"
#include "vmci_infrastructure.h"
#include "vmciResource.h"
#include "vmciGroup.h"
#include "vmciDatagram.h"
#include "vmciDriver.h"
#include "vmciDsInt.h"
#include "vmware.h"
#include "hostif.h"

#define LGPFX "VMCIDatagram: "

/*
 * DatagramEntry describes the datagram entity. It is used for datagram
 * entities created both on the host and in guests. The notifyCB and notifyData
 * fields are used only for datagram entities created on the host.
 */
typedef struct DatagramEntry {
   VMCIResource     resource;
   uint32         flags;
   VMCIDatagramCB   notifyCB;
   void           *notifyData;
} DatagramEntry;

static VMCIHandle DatagramMakeHnd(VMCIId contextID, VMCIId resourceID,
                                uint32 flags, VMCIDatagramCB notifyCB, 
                                void *notifyData);
static void DatagramFreeCB(void *resource);
static int DatagramCreateHnd(VMCIId contextID, void* clientData,
                             VMCICall *call);
static int DatagramDestroyHnd(VMCIId contextID, void* clientData,
                              VMCICall *call);
static int DatagramSend(VMCIId contextID, void* clientData,
                        VMCICall *call);
static int64 DatagramDriverHandler(void *ctObj, 
                                   VMCIObjType ctObjType, 
                                   VMCICall *call);
static void DatagramProcessNotifyCB(void *notifyData, uint32 msgSize,
                                    VMCIDatagramMsg *msg);
static int DatagramProcessSend(VMCIDatagramProcess *dgmProc,
                               VMCICall *dgmProcCall,
                               Bool mustCopy);

/* Struct used to represent the datagram API. */
static struct DatagramAPI {
   VMCIResource resource; 
   VMCIHandle resourceHandle;
   VMCIHandle groupHandle;
} datagramAPI;


/*
 *------------------------------------------------------------------------------
 *
 *  DatagramMakeHnd --
 * 
 *  Result:
 *     handle if successful, VMCI_INVALID_HANDLE if not.
 *     
 *------------------------------------------------------------------------------
 */

static VMCIHandle
DatagramMakeHnd(VMCIId contextID,  // IN:
                VMCIId resourceID, // IN:
                uint32 flags,           // IN:
                VMCIDatagramCB notifyCB,  // IN:
                void *notifyData)       // IN:
{
   int result;
   VMCIHandle handle;
   VMCIContext *context;
   DatagramEntry *entry;
   VMCIResourcePrivilegeType validPriv = VMCI_PRIV_DG_SEND;

   if (notifyCB == NULL) {
      Log(LGPFX "Client callback needed when creating datagram.\n");
      return VMCI_INVALID_HANDLE;
   }

   handle = VMCI_MAKE_HANDLE(contextID, resourceID);
   
   /* Check that context has the privilege to create datagram entries. */
   result = 
      VMCIResource_CheckClientPrivilege(datagramAPI.resourceHandle, 
                                        VMCI_MAKE_HANDLE(contextID, 
                                             VMCI_CONTEXT_RESOURCE_ID),
                                        VMCI_PRIV_DG_CREATE);
   if (result != VMCI_SUCCESS_ACCESS_GRANTED) {
      Log(LGPFX "Create: Context 0x%x does not have the privilege to create "
	  "datagram handles.\n", contextID);
      return VMCI_INVALID_HANDLE;
   }

   entry = VMCI_AllocKernelMem(sizeof *entry, VMCI_MEMORY_NONPAGED);
   if (entry == NULL) {
      Log(LGPFX "Create: Failed allocating memory for datagram entry.\n");
      return VMCI_INVALID_HANDLE;
   }

   entry->flags = flags;
   entry->notifyCB = notifyCB;
   entry->notifyData = notifyData;

   /* Add handle to context's datagramArray. */
   context = VMCIContext_Get(contextID);
   if (context == NULL) {
      VMCI_FreeKernelMem(entry);
      return VMCI_INVALID_HANDLE;
   }

   ASSERT(context->datagramArray);

   VMCIHandleArray_AppendEntry(&context->datagramArray, handle);

   /* Make datagram resource live. */
   result = VMCIResource_Add(&entry->resource, VMCI_RESOURCE_TYPE_DATAGRAM,
                             handle, 
                             VMCI_MAKE_HANDLE(contextID, VMCI_CONTEXT_RESOURCE_ID),
                             1, &validPriv, DatagramFreeCB, entry);
   if (result != VMCI_SUCCESS) {
      VMCI_FreeKernelMem(entry);
      handle = VMCI_INVALID_HANDLE;
   }

   return handle;
}


/*------------------------------ Helper functions ----------------------------*/

/*
 *------------------------------------------------------------------------------
 *
 *  DatagramFreeCB --
 *     Callback to free datagram structure when resource is no longer used,
 *     ie. the reference count reached 0.
 * 
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static void
DatagramFreeCB(void *resource)
{
   DatagramEntry *entry;
   ASSERT(resource);

   entry = RESOURCE_CONTAINER(resource, DatagramEntry, resource);
   if (entry) {
      VMCIResource *entryResource = &entry->resource;
      if (entryResource->registrationCount) {
         /* Remove all discovery service registrations for this resource. */
         VMCIDs_UnregisterResource(entryResource);
      }
      ASSERT(!entryResource->registrationCount);
   }
   VMCI_FreeKernelMem(entry);
}


/*
 *------------------------------------------------------------------------------
 *
 *  CTDatagramGuestNotifyCB --
 *     Callback to send a datagram notification to the guest.
 * 
 *  Result:
 *     None.
 *
 *  Side effects:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static void
CTDatagramGuestNotifyCB(void *notifyData,    // IN
			uint32 msgSize,      // IN
			VMCIDatagramMsg *msg)  // IN
{
   int result;
   VMCIId contextID = (VMCIId)(size_t)notifyData;
   unsigned gcSize = VMCI_CALL_HEADERSIZE + msgSize;
   VMCICall *guestcall = VMCI_AllocKernelMem(gcSize, 0);
   if (!guestcall) {
      Log(LGPFX "Failed to allocate guest datagram of size %d bytes.\n", 
	  gcSize);
      return;
   }
   
   guestcall->h.vector = VMCI_DATAGRAM_RECEIVE;
   guestcall->h.size = gcSize;
   memcpy(guestcall->args, msg, msgSize);
   result = VMCIContext_SendCall(contextID, guestcall, FALSE);
   if (result < VMCI_SUCCESS) {
      /*
       * This is where we should handle fallback to old guestcall vectors
       * (when we change the API in the future), if result ==
       * VMCI_ERROR_INVALID_VECTOR. For now, we just give up since this is
       * version 1 of the API and there are no fallbacks.
       */

      /* Since the guestcall was not queued, we need to free it. */
      VMCI_FreeKernelMem(guestcall);
   } else {
      Log(LGPFX "Sent guest call vector %d of size %d.\n", 
          VMCI_DATAGRAM_RECEIVE, (int)gcSize);
      /* guestcall is freed when guest call is read by context. */
   }
}

/*---------------------------- Hypercall functions ---------------------------*/


/*
 *------------------------------------------------------------------------------
 *
 *  DatagramCreateHnd --
 *
 *     Hypercall handler to create datagram handle.
 *
 *  Result:
 *     
 *     
 *------------------------------------------------------------------------------
 */

static int
DatagramCreateHnd(VMCIId contextID,  // IN:
                  void* clientData,       // IN: 
                  VMCICall *call)    // IN:
{
   VMCIHandle handle;
   VMCIDatagramCreateHdr *createCall;
   
   ASSERT(call);
   ASSERT_ON_COMPILE(sizeof(VMCIDatagramCreateHdr) == 16);

   createCall = (VMCIDatagramCreateHdr *)call;
   if (createCall->hdr.size != sizeof *createCall) {
      Log(LGPFX "Create: Invalid args.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (createCall->resourceID == VMCI_INVALID_ID) {
      Log(LGPFX "Create: Invalid resource id.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   handle = DatagramMakeHnd(contextID, createCall->resourceID, 
                           createCall->flags,
                           CTDatagramGuestNotifyCB,
                           (void *)(size_t)contextID);
   if (handle == VMCI_INVALID_HANDLE) {
      return VMCI_ERROR_NO_RESOURCES;
   }

   return VMCI_SUCCESS;
}


/*
 *------------------------------------------------------------------------------
 *
 *  DatagramDestroyHnd --
 *
 *     Hypercall handler to destroy datagram handle. 
 *
 *  Result:
 *     
 *     
 *------------------------------------------------------------------------------
 */

static int 
DatagramDestroyHnd(VMCIId contextID,  // IN: 
                   void* clientData,       // IN: Currently unused
                   VMCICall *call)    // IN
{
   int result;
   VMCIContext *context;
   VMCIDatagramDestroyHdr *destroyCall;

   ASSERT(call);
   ASSERT_ON_COMPILE(sizeof(VMCIDatagramDestroyHdr) == 16);

   destroyCall = (VMCIDatagramDestroyHdr *)call;
   if (destroyCall->hdr.size != sizeof *destroyCall) {
      Log(LGPFX "Destroy: Invalid args.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   /*
    * For now we don't allow entities other than the creator to destroy the
    * datagram handle.
    */
   result = 
      VMCIResource_CheckClientPrivilege(destroyCall->handle,
                                        VMCI_MAKE_HANDLE(contextID,
                                             VMCI_CONTEXT_RESOURCE_ID),
                                        VMCI_PRIV_DESTROY_RESOURCE);
   if (result != VMCI_SUCCESS_ACCESS_GRANTED) {
      Log(LGPFX"Disallowing destruction because non-owner VM %#x.\n",
          contextID);
      return result;
   }
   
   VMCIResource_Remove(destroyCall->handle, VMCI_RESOURCE_TYPE_DATAGRAM);

   /*
    * Remove owner context's reference to datagram entry. Owner context is the
    * context which has the VMCI_PRIV_DESTROY_RESOURCE privilege.
    */
   context = VMCIContext_Get(contextID);
   if (context) {
      ASSERT(context->datagramArray);
      VMCIHandleArray_RemoveEntry(context->datagramArray, 
				destroyCall->handle);
   }

   return VMCI_SUCCESS;
}


/*
 *------------------------------------------------------------------------------
 *
 *  DatagramSend --
 *
 *     Hypercall handler to send a datagram from the src handle to the dst
 *     handle.
 *
 *  Result:
 *     
 *     
 *------------------------------------------------------------------------------
 */

static int 
DatagramSend(VMCIId contextID,  // IN: src context ID
             void* clientData,       // IN: Currently unused
             VMCICall *call)    // IN:
{
   int retval = 0;
   VMCIResource *resource;
   DatagramEntry *srcEntry = NULL, *dstEntry = NULL;
   VMCIDatagramSendHdr *sendCall;

   ASSERT(call);
   ASSERT_ON_COMPILE(sizeof(VMCIDatagramSendHdr) == 32);
   
   sendCall = (VMCIDatagramSendHdr *)call;
   if (sendCall->hdr.size < sizeof *sendCall ||
       sendCall->hdr.size > VMCI_CALL_STANDARD_SIZE) {
      Log(LGPFX "Send: Invalid args.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   Log(LGPFX"Send: call size %d, header size %d.\n"
       "Sending from handle 0x%"FMT64"x to handle 0x%"FMT64"x.\n",
       sendCall->hdr.size, (uint32)sizeof *sendCall,
       sendCall->srcHandle, sendCall->dstHandle);

   if (sendCall->hdr.size != (sizeof *sendCall + sendCall->payloadSize)) {
      Log(LGPFX"Mismatch between call size %d and header + payload size "
	  "%d.\n",
	  sendCall->hdr.size, (uint32)sizeof *sendCall + sendCall->payloadSize);
      retval = VMCI_ERROR_INVALID_ARGS;
      goto release;
   }
   
   /* Check if the sender is the owner/creator of the src datagram entry. */
   retval =
      VMCIResource_CheckClientPrivilege(sendCall->srcHandle,
                                        VMCI_MAKE_HANDLE(contextID,
                                             VMCI_CONTEXT_RESOURCE_ID),
                                        VMCI_PRIV_DESTROY_RESOURCE);
   if (retval != VMCI_SUCCESS_ACCESS_GRANTED) {
      Log(LGPFX"Sender context 0x%x is not owner of src datagram entry with "
	  "handle 0x%"FMT64"x.\n", contextID, sendCall->srcHandle);
      retval = VMCI_ERROR_NO_ACCESS;
      goto release;
   }

   resource = VMCIResource_Get(sendCall->srcHandle, VMCI_RESOURCE_TYPE_DATAGRAM);
   if (resource == NULL) {
      Log(LGPFX"Sending from invalid handle 0x%"FMT64"x.\n",
	  sendCall->srcHandle);
      retval = VMCI_ERROR_INVALID_ARGS;
      goto release;
   }
   srcEntry = RESOURCE_CONTAINER(resource, DatagramEntry, resource);

   resource = VMCIResource_Get(sendCall->dstHandle, VMCI_RESOURCE_TYPE_DATAGRAM);
   if (resource == NULL) {
      Log(LGPFX"Sending to invalid destination datagram handle 0x%"FMT64"x.\n",
	  sendCall->dstHandle);
      retval = VMCI_ERROR_INVALID_ARGS;
      goto release;
   }
   dstEntry = RESOURCE_CONTAINER(resource, DatagramEntry, resource);

   /* Check if the context has privilege to send to destination. */
   retval = VMCIResource_CheckClientPrivilege(sendCall->dstHandle,
                                              VMCI_MAKE_HANDLE(contextID,
                                                   VMCI_CONTEXT_RESOURCE_ID),
                                              VMCI_PRIV_DG_SEND);
   if (retval != VMCI_SUCCESS_ACCESS_GRANTED) {
      Log(LGPFX "Send: Context 0x%x does not have the privilege to send to "
	  "datagram handle 0x%"FMT64"x.\n", contextID, sendCall->dstHandle);
      retval = VMCI_ERROR_NO_ACCESS;
      goto release;
   }

   if (sendCall->payloadSize > 0) {
      ASSERT(dstEntry->notifyCB);
      dstEntry->notifyCB(dstEntry->notifyData,
			 call->h.size - VMCI_CALL_HEADERSIZE,
			 (VMCIDatagramMsg *)call->args);
   }
   retval = sendCall->payloadSize;
   
  release:
   if (dstEntry) { 
      VMCIResource_Release(&dstEntry->resource);
   }
   if (srcEntry) {
      VMCIResource_Release(&srcEntry->resource);
   }
   return retval;
}


/*---------------------------- Driver call functions ---------------------------*/


/*
 *------------------------------------------------------------------------------
 *
 * DatagramDriverHandler --
 *
 *      Parses the driver call and calls the appropriate function.
 *
 * Results:
 *      Value of called function.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

static int64
DatagramDriverHandler(void *ctObj,         // IN:
                      VMCIObjType ctObjType, // IN: 
                      VMCICall *call) // IN
{
   VMCIDatagramProcess *dgmProc;
   int64 retval;

   if (!ctObj || ctObjType != VMCIOBJ_DATAGRAM_PROCESS || !call) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   dgmProc = (VMCIDatagramProcess *) ctObj; 

   switch (call->h.vector) {
   case VMCI_DATAGRAM_CREATE_HND: {
      VMCIHandle handle;
      VMCIDatagramCreateHdr *createCall = (VMCIDatagramCreateHdr *)call;
      if (createCall->hdr.size != sizeof *createCall) {
         retval = VMCI_ERROR_INVALID_ARGS;
         break;
      }
      
      handle = VMCIDatagram_CreateHnd(createCall->resourceID, createCall->flags,
                                      DatagramProcessNotifyCB, (void *) dgmProc);
      
      if (handle != VMCI_INVALID_HANDLE) {
         ASSERT(dgmProc->handle == VMCI_INVALID_HANDLE);
         dgmProc->handle = handle;
      }
      retval = handle;
      break;
   }
   case VMCI_DATAGRAM_DESTROY_HND: {
      VMCIDatagramDestroyHdr *destroyCall = (VMCIDatagramDestroyHdr *)call;
      
      if (destroyCall->hdr.size != sizeof *destroyCall) {
         retval = VMCI_ERROR_INVALID_ARGS;
         break;
      }
      
      if (dgmProc->handle != destroyCall->handle) {
         /* not ours */
         retval = VMCI_ERROR_NO_HANDLE;
         break;
      }
      
      VMCIDatagram_DestroyHnd(destroyCall->handle);
      dgmProc->handle = VMCI_INVALID_HANDLE;
      retval = VMCI_SUCCESS;
      break;
   }
   case VMCI_DATAGRAM_SEND: {
      uint32 callSize;
      VMCIDatagramSendHdr *sendCall = (VMCIDatagramSendHdr *)call;
      callSize = sizeof *sendCall + sendCall->payloadSize;
      
      if (call->h.size != callSize) {
         retval = VMCI_ERROR_INVALID_ARGS;
         break;
      }
      
      retval = VMCI_Dispatch(VMCI_HOST_CONTEXT_ID, call);
      break;
   }
   default:
      retval = VMCI_ERROR_INVALID_VECTOR;
      break;
   }
   return retval;
}

/*
 *----------------------------------------------------------------------
 *
 * VMCIDatagramProcess_Create --
 *
 *      Creates a new VMCIDatagramProcess object.
 *
 * Results:
 *      0 on success, appropriate error code otherwise.
 *
 * Side effects:
 *      Memory is allocated, deallocated in the destroy routine.
 *
 *----------------------------------------------------------------------
 */

int
VMCIDatagramProcess_Create(VMCIDatagramProcess **outDgmProc, // OUT:
                           const uintptr_t eventHnd)         // IN:
{
   VMCIDatagramProcess *dgmProc;
   
   ASSERT(outDgmProc);
   *outDgmProc = NULL;

   dgmProc = VMCI_AllocKernelMem(sizeof *dgmProc,
                                          VMCI_MEMORY_NONPAGED);
   if (!dgmProc) {
      Log(LGPFX "Failed to allocate memory for datagram fd.\n");
      return VMCI_ERROR_NO_MEM;
   }

   /* Initialize state */
   VMCI_InitLock(&dgmProc->callLock);
   dgmProc->handle = VMCI_INVALID_HANDLE;
   VMCIHost_InitContext(&dgmProc->host, eventHnd);
   dgmProc->pendingCalls = 0;
   dgmProc->callQueue = NULL;

   *outDgmProc = dgmProc;
   return VMCI_SUCCESS;
}

/*
 *----------------------------------------------------------------------
 *
 * VMCIDatagramProcess_Destroy --
 *
 *      Destroys a VMCIDatagramProcess object.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Memory allocated in create routine is deallocated here.
 *
 *----------------------------------------------------------------------
 */

void
VMCIDatagramProcess_Destroy(VMCIDatagramProcess *dgmProc) // IN:
{
   ListItem *curr, *next;
   CallEntry *pcEntry;
   VMCILockFlags flags;
   ListItem *tempCallQueue;

   if (!dgmProc) {
      return;
   }

   if (dgmProc->handle != VMCI_INVALID_HANDLE) {
      VMCIDatagram_DestroyHnd(dgmProc->handle);
      dgmProc->handle = VMCI_INVALID_HANDLE;
   }

   /*
    * Flush dgmProc's call queue. We need to free the guestcalls while
    * not holding the callLock, since the calls themselves are allocated
    * in NonPagedPool on Windows, so we cannot access them while holding
    * a spinlock.
    */
   VMCI_GrabLock(&dgmProc->callLock, &flags);
   tempCallQueue = dgmProc->callQueue;
   dgmProc->callQueue = NULL;
   VMCI_ReleaseLock(&dgmProc->callLock, flags);

   LIST_SCAN_SAFE(curr, next, tempCallQueue) {
      pcEntry = LIST_CONTAINER(curr, CallEntry, listItem);
      LIST_DEL(curr, &tempCallQueue);
      ASSERT(pcEntry && pcEntry->call);
      VMCI_FreeKernelMem(pcEntry->call);
      VMCI_FreeKernelMem(pcEntry);
   }

   VMCI_FreeKernelMem(dgmProc);
}

/*
 *------------------------------------------------------------------------------
 *
 *  DatagramProcessNotifyCB --
 *     Callback to send a notificaton to a host vmci process.
 * 
 *  Result:
 *     None.
 *
 *  Side effects:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static void
DatagramProcessNotifyCB(void *notifyData,   // IN:
                        uint32 msgSize,     // IN:
                        VMCIDatagramMsg *msg) // IN:
{
   VMCIDatagramProcess *dgmProc = (VMCIDatagramProcess *) notifyData;
   unsigned dgmSize = VMCI_CALL_HEADERSIZE + msgSize ;
   VMCICall *dgmProcCall = VMCI_AllocKernelMem(dgmSize, 0);
   if (!dgmProcCall) {
      Log(LGPFX "Failed to allocate process call structure of size %d bytes.\n", 
            dgmSize);
      return;
   }

   dgmProcCall->h.vector = VMCI_DATAGRAM_RECEIVE;
   dgmProcCall->h.size = dgmSize;
   memcpy(dgmProcCall->args, msg, msgSize);
   DatagramProcessSend(dgmProc, dgmProcCall, FALSE);
   Log(LGPFX "Sent process call with vector %d and size %d.\n", 
       VMCI_DATAGRAM_RECEIVE, dgmSize);
   /* dgmProcCall is freed when user reads call.. */
}

/*
 *----------------------------------------------------------------------
 *
 * DatagramProcessSend --
 *
 *      Queues a VMCI datagram call for the appropriate target 
 *      process.
 *
 * Results:
 *      0 on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
DatagramProcessSend(VMCIDatagramProcess *dgmProc, // IN: Target datagram device handle
                    VMCICall *dgmProcCall,   // IN:
                    Bool mustCopy)                // IN: TRUE if must copy.
{
   CallEntry *dcEntry;
   VMCILockFlags flags;

   if (!dgmProc || !dgmProcCall ||
       dgmProcCall->h.size < VMCI_CALL_HEADERSIZE) {
      Log(LGPFX "Invalid call parameters received\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (dgmProcCall->h.size > VMCI_CALL_STANDARD_SIZE) {
      Log(LGPFX"Datagram guestcall too large.\n");
      return VMCI_ERROR_NO_MEM;
   }

   /* Allocate datagram call entry and add it to the target fd's queue. */
   dcEntry = VMCI_AllocKernelMem(sizeof *dcEntry, VMCI_MEMORY_NONPAGED);
   if (dcEntry == NULL) {
      Log(LGPFX "Failed to allocate memory for process call.\n");
      return VMCI_ERROR_NO_MEM;
   }
   if (mustCopy) {
      dcEntry->call = VMCI_AllocKernelMem(dgmProcCall->h.size, 0);
      if (dcEntry->call == NULL) {
         Log(LGPFX "Failed to allocate memory for process call.\n");
         VMCI_FreeKernelMem(dcEntry);
         return VMCI_ERROR_NO_MEM;
      }
      memcpy(dcEntry->call, dgmProcCall, dgmProcCall->h.size);
   } else {
      dcEntry->call = dgmProcCall;
   }

   VMCI_GrabLock(&dgmProc->callLock, &flags);
   if (dgmProc->pendingCalls >= VMCI_MAX_NUM_CALLS) {
      VMCI_ReleaseLock(&dgmProc->callLock, flags);
      if (mustCopy) {
         VMCI_FreeKernelMem(dcEntry->call);
      }
      VMCI_FreeKernelMem(dcEntry);
      Log(LGPFX "Too many queued datagram calls\n");
      return VMCI_ERROR_NO_RESOURCES;
   }

   LIST_QUEUE(&dcEntry->listItem, &dgmProc->callQueue);
   dgmProc->pendingCalls++;
   VMCIHost_SignalCall(&dgmProc->host);
   VMCI_ReleaseLock(&dgmProc->callLock, flags);

   return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * VMCIDatagramProcess_ReadCall --
 *
 *      Dequeues the next guest call and returns it to user level.
 *
 * Results:
 *      0 on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VMCIDatagramProcess_ReadCall(VMCIDatagramProcess *dgmProc, // IN:
                             void *buf,                    // OUT: Userlevel buffer
                             size_t bufSize)               // IN: size of buf
{
   CallEntry *dcEntry;
   ListItem *listItem;
   int result;
   VMCILockFlags flags;

   ASSERT(dgmProc);

   /* Dequeue the next dgmProc call entry. */
   VMCI_GrabLock(&dgmProc->callLock, &flags);
   if (dgmProc->pendingCalls == 0) {
      VMCIHost_ClearCall(&dgmProc->host);
      VMCI_ReleaseLock(&dgmProc->callLock, flags);
      Log(LGPFX "No calls pending.\n");
      return VMCI_ERROR_NO_MORE_CALLS;
   }

   listItem = LIST_FIRST(dgmProc->callQueue);
   ASSERT (listItem != NULL);

   dcEntry = LIST_CONTAINER(listItem, CallEntry, listItem);
   ASSERT(dcEntry->call);

   /* Check the size of the userland buffer. */
   if (bufSize < dcEntry->call->h.size) {
      VMCI_ReleaseLock(&dgmProc->callLock, flags);
      Log(LGPFX "Userland buffer is too small.\n");
      return VMCI_ERROR_NO_MEM;
   }
   
   LIST_DEL(listItem, &dgmProc->callQueue);
   dgmProc->pendingCalls--;
   if (dgmProc->pendingCalls == 0) {
      VMCIHost_ClearCall(&dgmProc->host);
   }
   VMCI_ReleaseLock(&dgmProc->callLock, flags);

   result = HostIF_CopyToUser((void *)buf, dcEntry->call,
			      dcEntry->call->h.size);
   VMCI_FreeKernelMem(dcEntry->call);
   VMCI_FreeKernelMem(dcEntry);

   return result == 0 ? VMCI_SUCCESS : VMCI_ERROR_NO_MEM;
}

/*------------------------------ Global functions ----------------------------*/

/*
 *------------------------------------------------------------------------------
 *
 *  VMCIDatagram_Init --
 *
 *     Initialize Datagram API, ie. register the API functions with their
 *     corresponding vectors.
 *
 *  Result:
 *     None.
 *     
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

int
VMCIDatagram_Init(void)
{
   int result;
   VMCIResourcePrivilegeType priv = VMCI_PRIV_DG_CREATE;

   datagramAPI.resourceHandle =
      VMCI_MAKE_HANDLE(VMCI_HOST_CONTEXT_ID, VMCIResource_GetID());

   /* Resource used to represent the datagram API. */
   result = VMCIResource_Add(&datagramAPI.resource, VMCI_RESOURCE_TYPE_API,
                             datagramAPI.resourceHandle,
                             VMCI_MAKE_HANDLE(VMCI_HOST_CONTEXT_ID,
                                  VMCI_CONTEXT_RESOURCE_ID),
                             1, &priv, NULL, NULL);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"Failed adding Datagram API resource.\n");
      return result;
   }

   /*
    * Create a datagram api group.By adding this group as a client to the 
    * datagramAPIResource with the VMCI_PRIV_DG_CREATE we can give contexts access
    * to the datagram api by making them members of this group.
    */
   datagramAPI.groupHandle = VMCIGroup_Create();
   if (datagramAPI.groupHandle == VMCI_INVALID_HANDLE) {
      Log(LGPFX"Failed creating Datagram API group.\n");
      VMCIResource_Remove(datagramAPI.resourceHandle, VMCI_RESOURCE_TYPE_API);
      return VMCI_ERROR_NO_HANDLE;
   }

   /* Add group as client of datagram API with the right privilege. */
   result = VMCIResource_AddClientPrivileges(datagramAPI.resourceHandle,
                                             datagramAPI.groupHandle,
                                             1, &priv, 0, NULL);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"Failed making Datagram API group a client of Datagram API "
	  "resource.\n");
      VMCIGroup_Destroy(datagramAPI.groupHandle);
      VMCIResource_Remove(datagramAPI.resourceHandle, VMCI_RESOURCE_TYPE_API);
      return result;
   }

   /* Register hypercall handlers. */
   VMCI_RegisterHandler(VMCI_DATAGRAM_CREATE_HND,
                        DatagramCreateHnd, 
                        VMCI_IMMEDIATE_DISPATCH, NULL);
   VMCI_RegisterHandler(VMCI_DATAGRAM_DESTROY_HND, 
                        DatagramDestroyHnd,
                        VMCI_IMMEDIATE_DISPATCH, NULL);
   VMCI_RegisterHandler(VMCI_DATAGRAM_SEND, 
                        DatagramSend,
                        VMCI_IMMEDIATE_DISPATCH, NULL);

   /* Register driver call handlers. */
   VMCI_RegisterDriverHandler(VMCI_DATAGRAM_CREATE_HND,
                              DatagramDriverHandler);
   VMCI_RegisterDriverHandler(VMCI_DATAGRAM_DESTROY_HND,
                              DatagramDriverHandler);
   VMCI_RegisterDriverHandler(VMCI_DATAGRAM_SEND,
                              DatagramDriverHandler);

   return VMCI_SUCCESS;
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIDatagram_Exit --
 *
 *     Cleanup Datagram API.
 *
 *  Result:
 *     None.
 *     
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

void
VMCIDatagram_Exit(void)
{
   /* Remove group as client of datagramAPI resource. */
   VMCIResource_RemoveAllClientPrivileges(datagramAPI.resourceHandle,
                                          datagramAPI.groupHandle);
   
   VMCIGroup_Destroy(datagramAPI.groupHandle);

   VMCIResource_Remove(datagramAPI.resourceHandle, VMCI_RESOURCE_TYPE_API);
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIDatagram_AddContext --
 *
 *     Adds the context as a member of the datagram API group. This makes it 
 *     possible for the context to use the datagram API.
 *
 *  Result:
 *     None.
 *     
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

void
VMCIDatagram_AddContext(VMCIId contextID)
{
   VMCIContext *context = VMCIContext_Get(contextID);
   if (context != NULL) {
      VMCIGroup_AddMember(datagramAPI.groupHandle,
                          VMCI_MAKE_HANDLE(contextID, VMCI_CONTEXT_RESOURCE_ID),
                          FALSE);
      VMCIHandleArray_AppendEntry(&context->groupArray, datagramAPI.groupHandle);
   }
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIDatagram_RemoveContext --
 *
 *     Removes the context as a member of the Datagram API, disallowing the 
 *     context access to the Datagram API functions.
 *
 *  Result:
 *     None.
 *     
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

void
VMCIDatagram_RemoveContext(VMCIId contextID)
{
   VMCIContext *context = VMCIContext_Get(contextID);
   if (context != NULL) {
      VMCIHandleArray_RemoveEntry(context->groupArray, datagramAPI.groupHandle);
      VMCIGroup_RemoveMember(datagramAPI.groupHandle,
                             VMCI_MAKE_HANDLE(contextID, VMCI_CONTEXT_RESOURCE_ID));
   }
}


/*
 *------------------------------------------------------------------------------
 *
 * VMCIDatagram_CreateHnd --
 *
 *      Creates a datagram endpoint and returns a handle to it.
 *
 * Results:
 *      Returns handle if success, negative errno value otherwise.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

VMCIHandle
VMCIDatagram_CreateHnd(VMCIId resourceID, // IN: Optional, generated if VMCI_INVALID_ID
                       uint32 flags,           // IN:
                       VMCIDatagramCB notifyCB,  // IN:
                       void *notifyData)       // IN:
{
   /* 
    * If the caller specifies a resource ID, it is the caller's responsibility to
    * make sure it is unused. If not this function will fail.
    */      
   if (resourceID == VMCI_INVALID_ID) {
      resourceID = VMCIResource_GetID();
   }

   return DatagramMakeHnd(VMCI_HOST_CONTEXT_ID, resourceID, flags, 
                          notifyCB, notifyData);
}

		      
/*
 *------------------------------------------------------------------------------
 *
 * VMCIDatagram_DestroyHnd --
 *
 *      Destroys a handle.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

void
VMCIDatagram_DestroyHnd(VMCIHandle handle)       // IN
{
   VMCIDatagramDestroyHdr destroyCall =
      {{VMCI_DATAGRAM_DESTROY_HND, sizeof destroyCall}, handle};
   VMCI_Dispatch(VMCI_HOST_CONTEXT_ID, (VMCICall *)&destroyCall);
}


/*
 *------------------------------------------------------------------------------
 *
 * VMCIDatagram_Send --
 *
 *      Sends the payload to the destination datagram handle.
 *
 * Results:
 *      Returns number of bytes sent if success, or error code if failure.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

int
VMCIDatagram_Send(VMCIHandle dstHandle,      // IN
                  VMCIHandle srcHandle,      // IN
                  uint32 payloadSize,      // IN
                  void *payload)           // IN
{
   uint32 retval = VMCI_SUCCESS;
   VMCIDatagramSendHdr *sendCall;
   uint32 callSize = sizeof *sendCall + payloadSize;

   if (callSize > VMCI_CALL_STANDARD_SIZE) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   sendCall = VMCI_AllocKernelMem(callSize, 0);
   if (sendCall == NULL) {
      return VMCI_ERROR_NO_MEM;
   }
   sendCall->hdr.vector = VMCI_DATAGRAM_SEND;
   sendCall->hdr.size = callSize;
   sendCall->dstHandle = dstHandle;
   sendCall->srcHandle = srcHandle;
   sendCall->payloadSize = payloadSize;
   if (payloadSize > 0) {
      memcpy((char *)sendCall + sizeof *sendCall, payload, payloadSize);
   }
   retval = VMCI_Dispatch(VMCI_HOST_CONTEXT_ID, (VMCICall *)sendCall);
   VMCI_FreeKernelMem(sendCall);

   return retval;
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCIDatagram_CheckGuestCalls --
 *
 *      Make sure that context supports the guestcalls we need.
 *
 * Results:
 *      TRUE if context is supported, FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

Bool
VMCIDatagram_CheckGuestCalls(VMCIContext *context)
{
   /* Context must support VMCI_DATAGRAM_RECEIVE. There are no fallbacks. */
   return VMCIContext_SupportsGuestCall(context, VMCI_DATAGRAM_RECEIVE);
}

