/* **********************************************************
 * Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 *
 * rateconv.h --
 *
 *     Parameters and functions for linear rate conversion of 64 bit
 *     counters:
 *
 *       y = ((x * mult) >> shift) + add.
 *
 */

#ifndef _VM_RATECONV_H_
#define _VM_RATECONV_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMNIXMOD
#define INCLUDE_ALLOW_VMKERNEL
#define INCLUDE_ALLOW_VMK_MODULE
#define INCLUDE_ALLOW_DISTRIBUTE
#define INCLUDE_ALLOW_VMCORE
#define INCLUDE_ALLOW_VMMON
#include "includeCheck.h"

#include "vm_basic_types.h"
#include "vm_basic_asm.h"
#include "vm_atomic.h"


typedef struct RateConv_Params {
   uint32 mult;
   uint32 shift;
   int64 add;
} RateConv_Params;

typedef struct RateConv_Ratio {
   uint32 mult;
   uint32 shift;
} RateConv_Ratio;

#define RATE_CONV_IDENTITY { 1, 0, 0 }  // out = in

Bool RateConv_ComputeParams(uint64 inHz, uint64 inBase,
		            uint64 outHz, uint64 outBase,
		            RateConv_Params *conv);
void RateConv_LogParams(const char *prefix,
			uint64 inHz, uint64 inBase,
			uint64 outHz, uint64 outBase,
			const RateConv_Params *conv);
Bool RateConv_ComputeRatio(uint64 inHz, uint64 outHz,
                           RateConv_Ratio *ratio);


/*
 *----------------------------------------------------------------------
 *
 * RateConv_IsIdentity --
 *
 *      Test whether rate conversion parameters represent the identity
 *      function.  By convention, mult == 1 implies that shift == 0
 *      and add == 0 as well, so we test only mult for speed.  This is
 *      not a serious limitation, since mult should generally be
 *      normalized to have bit 31 != 0.
 *
 *      The commented-out C inline is logically equivalent to the
 *      macro, but the macro generates slightly better code in gcc
 *      3.0.3.
 *
 *----------------------------------------------------------------------
 */

// static INLINE Bool
// RateConv_IsIdentity(const RateConv_Params* conv)
// {
//    return conv->mult == 1;
// }

#define RateConv_IsIdentity(conv) \
   ((conv)->mult == 1)


/*
 *----------------------------------------------------------------------
 *
 * RateConv_Unsigned --
 *
 *      Apply rate conversion to an unsigned argument:
 *       y = ((x * mult) >> shift) + add.
 *
 *      The commented-out C inline is logically equivalent to the
 *      macro, but the macro generates slightly better code in gcc
 *      3.0.3.
 *
 *----------------------------------------------------------------------
 */

// static INLINE uint64
// RateConv_Unsigned(const RateConv_Params *conv, uint64 x)
// {
//    return Mul64x3264(x, conv->mult, conv->shift) + conv->add;
// }

#define RateConv_Unsigned(conv, x) \
   (Mul64x3264(x, (conv)->mult, (conv)->shift) + (conv)->add)

#define RateConv_UnsignedAtomic(conv, x) \
   (Mul64x3264(x, (conv)->mult, (conv)->shift) + \
    Atomic_Read64((Atomic_uint64 *) &(conv)->add))


/*
 *----------------------------------------------------------------------
 *
 * RateConv_Signed --
 *
 *      Apply rate conversion to an signed argument:
 *       y = ((x * mult) >> shift) + add.
 *
 *      The commented-out C inline is logically equivalent to the
 *      macro, but the macro generates slightly better code in gcc
 *      3.0.3.
 *
 *----------------------------------------------------------------------
 */

// static INLINE int64
// RateConv_Signed(const RateConv_Params *conv, int64 x)
// {
//    return Muls64x32s64(x, conv->mult, conv->shift) + conv->add;
// }

#define RateConv_Signed(conv, x) \
   (Muls64x32s64(x, (conv)->mult, (conv)->shift) + (conv)->add)

#define RateConv_SignedAtomic(conv, x) \
   (Muls64x32s64(x, (conv)->mult, (conv)->shift) + \
    (int64) Atomic_Read64((Atomic_uint64 *) &(conv)->add))


#endif // _VM_RATECONV_H_
