#include "initblock_compat.h"
#include "x86vt_compat.h"

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlTOT --
 *
 *      Main path for UserRPC from latest product
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlTOT(struct file *filp,
                     u_int iocmd,
		     unsigned long ioarg)
{
   if (iocmd < IOCTL_VMX86_FIRST || iocmd >= IOCTL_VMX86_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   switch (iocmd) {
      case IOCTL_VMX86_VERSION:
         return VMMON_VERSION;
      default:
	 break;
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV6 --
 *
 *      Main path for UserRPC from Server 1 beta 1
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV6(struct file *filp,
                      u_int iocmd,
                      unsigned long ioarg)
{
   static const unsigned int map[IOCTL_VMX86_V6_LAST - IOCTL_VMX86_V6_VERSION] = {
      IOCTL_VMX86_VERSION,
      IOCTL_VMX86_CREATE_VM,
      IOCTL_VMX86_RELEASE_VM,
      IOCTL_VMX86_GET_NUM_VMS,
      IOCTL_VMX86_INIT_VM,
      IOCTL_VMX86_LATE_INIT_VM,
      IOCTL_VMX86_RUN_VM,
      IOCTL_VMX86_LOOK_UP_MPN,
      IOCTL_VMX86_LOCK_PAGE,
      IOCTL_VMX86_UNLOCK_PAGE,
      IOCTL_VMX86_APIC_BASE,
      IOCTL_VMX86_SET_HARD_LIMIT,
      IOCTL_VMX86_GET_MEM_INFO,
      IOCTL_VMX86_ADMIT,
      IOCTL_VMX86_SET_MEM_USAGE,
      IOCTL_VMX86_READMIT,
      IOCTL_VMX86_PAE_ENABLED,
      IOCTL_VMX86_HOST_X86_64,
      IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      IOCTL_VMX86_COMPLETE_USER_CALL,
      IOCTL_VMX86_GET_KHZ_ESTIMATE,
      IOCTL_VMX86_SET_HOST_CLOCK_RATE,
      IOCTL_VMX86_READ_PAGE,
      IOCTL_VMX86_WRITE_PAGE,
      IOCTL_VMX86_LOCK_PAGE_NEW,
      IOCTL_VMX86_UNLOCK_PAGE_BY_MPN,
      IOCTL_VMX86_MARK_LOCKEDVARANGE_CLEAN,
      IOCTL_VMX86_COW_SHARE,
      IOCTL_VMX86_COW_CHECK,
      IOCTL_VMX86_COW_UPDATE_HINT,
      IOCTL_VMX86_COW_COPY_PAGE,
      IOCTL_VMX86_COW_REMOVE_HINT,
      IOCTL_VMX86_COW_GET_ZERO_MPN,
      IOCTL_VMX86_ALLOC_LOCKED_PAGES,
      IOCTL_VMX86_FREE_LOCKED_PAGES,
      IOCTL_VMX86_GET_LOCKED_PAGES_LIST,
      IOCTL_VMX86_APIC_ID,
      IOCTL_VMX86_SVM_ENABLED_CPU,
      IOCTL_VMX86_VT_ENABLED_CPU,
      IOCTL_VMX86_VT_SUPPORTED_CPU,
      IOCTL_VMX86_BROKEN_CPU_HELPER,
      IOCTLCMD_VMCI_INIT_CONTEXT,
      IOCTLCMD_VMCI_CREATE_PROCESS,
      IOCTLCMD_VMCI_CREATE_DATAGRAM_PROCESS,
      IOCTLCMD_VMCI_HYPERCALL,
      IOCTLCMD_VMCI_READ_CALL,
      IOCTLCMD_VMCI_SHAREDMEM_CREATE,
      IOCTLCMD_VMCI_SHAREDMEM_ATTACH,
      IOCTLCMD_VMCI_SHAREDMEM_QUERY,
      IOCTLCMD_VMCI_SHAREDMEM_DETACH,
      IOCTL_VMX86_COUNT_PRESENT_PAGES,
      IOCTL_VMX86_INIT_NUMA_INFO,
      IOCTL_VMX86_GET_NUMA_MEM_STATS,
      IOCTL_VMX86_SET_UID,
      IOCTL_VMX86_ALLOW_CORE_DUMP,
      IOCTL_VMX86_BROADCAST_IPI,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      IOCTL_VMX86_START_PASSTHROUGH,
      IOCTL_VMX86_STOP_PASSTHROUGH,
      IOCTL_VMX86_QUERY_PASSTHROUGH,
      IOCTL_VMX86_REGISTER_PERFCTR,
      IOCTL_VMX86_START_PERFCTR,
      IOCTL_VMX86_STOP_PERFCTR,
      IOCTL_VMX86_RELEASE_PERFCTR,
      IOCTL_VMX86_GET_ALL_CPUID,
      IOCTL_VMX86_SET_THREAD_AFFINITY,
      IOCTL_VMX86_GET_THREAD_AFFINITY,
      IOCTL_VMX86_GET_KERNEL_CLOCK_RATE,
      IOCTL_VMX86_ACK_USER_CALL,
      IOCTL_VMX86_SET_POLL_TIMEOUT_PTR,
      IOCTL_VMX86_IS_MP_SAFE,
      IOCTL_VMX86_FAST_MON_SWITCH_START,
      IOCTL_VMX86_FAST_MON_SWITCH_IS_DONE,
      IOCTL_VMX86_FAST_MON_SWITCH_TRANSFER,
      IOCTL_VMX86_FAST_MON_SWITCH_END,
   };
   int retval = -EINVAL;

   if (iocmd < IOCTL_VMX86_V6_VERSION || iocmd >= IOCTL_VMX86_V6_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTL_VMX86_V6_VERSION:
         return VMMON_VERSION_V6;

      default:
         iocmd = map[iocmd - IOCTL_VMX86_V6_VERSION];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlS1B1 --
 *
 *      Main path for UserRPC from Server 1 beta 1
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlS1B1(struct file *filp,
                      u_int iocmd,
                      unsigned long ioarg)
{
   static const unsigned int map[IOCTL_VMX86_S1B1_LAST - IOCTL_VMX86_S1B1_VERSION] = {
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VERSION */
      IOCTL_VMX86_CREATE_VM,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* BIND_VM */
      IOCTL_VMX86_RELEASE_VM,
      IOCTL_VMX86_GET_NUM_VMS,
      IOCTL_VMX86_INIT_VM,
      IOCTL_VMX86_LATE_INIT_VM,
      IOCTL_VMX86_RUN_VM,
      IOCTL_VMX86_LOOK_UP_MPN,
      IOCTL_VMX86_LOCK_PAGE,
      IOCTL_VMX86_UNLOCK_PAGE,
      IOCTL_VMX86_APIC_BASE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_HARD_LIMIT */
      IOCTL_VMX86_SET_HARD_LIMIT,
      IOCTL_VMX86_GET_MEM_INFO,
      IOCTL_VMX86_ADMIT,
      IOCTL_VMX86_SET_MEM_USAGE,
      IOCTL_VMX86_READMIT,
      IOCTL_VMX86_PAE_ENABLED,
      IOCTL_VMX86_HOST_X86_64,
      IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ACK_USER_CALL */
      IOCTL_VMX86_COMPLETE_USER_CALL,
      IOCTL_VMX86_GET_KHZ_ESTIMATE,
      IOCTL_VMX86_SET_HOST_CLOCK_RATE,
      IOCTL_VMX86_READ_PAGE,
      IOCTL_VMX86_WRITE_PAGE,
      IOCTL_VMX86_LOCK_PAGE_NEW,
      IOCTL_VMX86_UNLOCK_PAGE_BY_MPN,
      IOCTL_VMX86_MARK_LOCKEDVARANGE_CLEAN,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* COW_SHARE */
      IOCTL_VMX86_COW_CHECK,
      IOCTL_VMX86_COW_UPDATE_HINT,
      IOCTL_VMX86_COW_COPY_PAGE,
      IOCTL_VMX86_COW_REMOVE_HINT,
      IOCTL_VMX86_COW_GET_ZERO_MPN,
      IOCTL_VMX86_ALLOC_LOCKED_PAGES,
      IOCTL_VMX86_FREE_LOCKED_PAGES,
      IOCTL_VMX86_GET_LOCKED_PAGES_LIST,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* MAP_LOCKED_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* UNMAP_LOCKED_PAGES */
      IOCTL_VMX86_APIC_ID,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VT_CAPABLE_CPU */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VT_SUPPORTED_CPU */
      IOCTL_VMX86_SET_UID,
      IOCTL_VMX86_IS_MP_SAFE,
      IOCTL_VMX86_ALLOW_CORE_DUMP,
      IOCTL_VMX86_BROADCAST_IPI,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      IOCTL_VMX86_START_PASSTHROUGH,
      IOCTL_VMX86_STOP_PASSTHROUGH,
      IOCTL_VMX86_QUERY_PASSTHROUGH,
      IOCTL_VMX86_REGISTER_PERFCTR,
      IOCTL_VMX86_START_PERFCTR,
      IOCTL_VMX86_STOP_PERFCTR,
      IOCTL_VMX86_RELEASE_PERFCTR,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ALLOC_LOW_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* FREE_LOW_PAGES */
      IOCTL_VMX86_GET_ALL_CPUID,
      IOCTL_VMX86_SET_THREAD_AFFINITY,
      IOCTL_VMX86_GET_THREAD_AFFINITY,
      IOCTL_VMX86_SET_POLL_TIMEOUT_PTR,
      IOCTL_VMX86_GET_KERNEL_CLOCK_RATE,
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval = -EINVAL;

   if (iocmd < IOCTL_VMX86_S1B1_VERSION || iocmd >= IOCTL_VMX86_S1B1_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTL_VMX86_S1B1_VERSION:
         return VMMON_VERSION_S1B1;

      case IOCTL_VMX86_S1B1_BIND_VM:
         if (vmLinux->vm != NULL) {
            retval = -EINVAL;
            break;
         }
         vmLinux->vm = Vmx86_BindVM(ioarg);
         if (vmLinux->vm == NULL) {
            retval = -EINVAL;
         }
         break;

      case IOCTL_VMX86_S1B1_GET_HARD_LIMIT: {
         retval = Vmx86_GetLockedPagesLimit(vmLinux->vm);
         break;
      }

      case IOCTL_VMX86_S1B1_VT_CAPABLE_CPU:
         retval = VT_EnabledCPU();
         break;
     
      case IOCTL_VMX86_S1B1_MAP_LOCKED_PAGES:
      case IOCTL_VMX86_S1B1_UNMAP_LOCKED_PAGES:
         {
            VMMPNListVA req;
	 
   	    retval = HostIF_CopyFromUser(&req, (void*)ioarg, sizeof req);
    	    if (retval) {
	       break;
	    }
	    if (!vmLinux->vm) {
	       retval = -EINVAL;
	       break;
	    }
            if (iocmd == IOCTL_VMX86_S1B1_MAP_LOCKED_PAGES) {
               retval = HostIF_MapLockedPages(vmLinux->vm, req.va, 
                                              (MPN32*)(VA)req.mpn32List, 
                                              req.mpnCount);
            } else {
               retval = HostIF_UnmapLockedPages(vmLinux->vm, req.va, 
                                                (MPN32*)(VA)req.mpn32List, 
                                                req.mpnCount);
            }
         }
         break;

      case IOCTL_VMX86_S1B1_ALLOC_LOW_PAGES:
         {
            uint32 numPages;
            PA32 addr;

            retval = HostIF_CopyFromUser(&numPages, (uint32*)ioarg, sizeof(numPages));
            if (retval < 0) {
               break;
            }

            retval = LinuxDriverIoctlAlloc4Gb(numPages, &addr);
            if (retval < 0) {
               break;
            }

            retval = HostIF_CopyToUser((PA32*)ioarg, &addr, sizeof(addr));
         }
         break;

      case IOCTL_VMX86_S1B1_FREE_LOW_PAGES:
         retval = 0;
         // Release happens on module unload, not sooner!
         break;
	 
      case IOCTL_VMX86_S1B1_VT_SUPPORTED_CPU:
         retval = VT_SupportedCPUS1B1();
         break;

      case IOCTL_VMX86_S1B1_COW_SHARE:
      {
         COWShareInfo info;

         retval = HostIF_CopyFromUser(&info, (void*)ioarg, sizeof info);
         if (retval) {
            break;
         }
         Vmx86_COWSharePagesS1B1(vmLinux->vm, &info);
         retval = HostIF_CopyToUser((void*)ioarg, &info, sizeof info);
         break;
      }

      case IOCTL_VMX86_S1B1_ACK_USER_CALL:
      {
         Vcpuid vcpuid = (Vcpuid) ioarg;

         if (vcpuid >= vmLinux->vm->numVCPUs) {
            retval = -EINVAL;
            break;
         }
         Vmx86_AckUserCallS1B1(vmLinux->vm, vcpuid);
	 retval = 0;
         break;
      }

      default:
         iocmd = map[iocmd - IOCTL_VMX86_S1B1_VERSION];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV55 --
 *
 *      Main path for UserRPC from VMware Workstation 5.5
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV55(struct file *filp,
                     u_int iocmd,
		     unsigned long ioarg)
{
   switch (iocmd) {
      case IOCTL_VMX86_S1B1_VERSION:
         return VMMON_VERSION_V55;
      default:
	 break;
   }
   return LinuxDriver_IoctlS1B1(filp, iocmd, ioarg);
}


static inline int
initBlockV5_to_initBlock(InitBlock *dst, const InitBlockV5 *src) {
        unsigned int i;

        if (src->magicNumber != INIT_BLOCK_MAGIC_V5) {
                return -EINVAL;
        }
        dst->magicNumber = INIT_BLOCK_MAGIC;
        dst->numVCPUs = src->numVCPUs;
        for (i = 0; i < MAX_INITBLOCK_CPUS; i++) {
                dst->crosspage[i] = src->crosspage[i];
        }
        dst->vmInitFailurePeriod = 0;
        return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV5 --
 *
 *      Main path for UserRPC from latest product
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV5(struct file *filp,
                      u_int iocmd,
		      unsigned long ioarg)
{
   static const unsigned int v5map[IOCTL_VMX86_V5_LAST - IOCTL_VMX86_V5_VERSION] = {
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VERSION */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CREATE_VM */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* BIND_VM */
      IOCTL_VMX86_RELEASE_VM,
      IOCTL_VMX86_GET_NUM_VMS,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* INIT_VM */
      IOCTL_VMX86_LATE_INIT_VM,
      IOCTL_VMX86_RUN_VM,
      IOCTL_VMX86_LOOK_UP_MPN,
      IOCTL_VMX86_LOCK_PAGE,
      IOCTL_VMX86_UNLOCK_PAGE,
      IOCTL_VMX86_APIC_BASE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_HARD_LIMIT */
      IOCTL_VMX86_SET_HARD_LIMIT,
      IOCTL_VMX86_GET_MEM_INFO,
      IOCTL_VMX86_ADMIT,
      IOCTL_VMX86_SET_MEM_USAGE,
      IOCTL_VMX86_READMIT,
      IOCTL_VMX86_PAE_ENABLED,
      IOCTL_VMX86_HOST_X86_64,
      IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ACK_USER_CALL */
      IOCTL_VMX86_COMPLETE_USER_CALL,
      IOCTL_VMX86_GET_KHZ_ESTIMATE,
      IOCTL_VMX86_SET_HOST_CLOCK_RATE,
      IOCTL_VMX86_READ_PAGE,
      IOCTL_VMX86_WRITE_PAGE,
      IOCTL_VMX86_LOCK_PAGE_NEW,
      IOCTL_VMX86_UNLOCK_PAGE_BY_MPN,
      IOCTL_VMX86_MARK_LOCKEDVARANGE_CLEAN,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* COW_SHARE */
      IOCTL_VMX86_COW_CHECK,
      IOCTL_VMX86_COW_UPDATE_HINT,
      IOCTL_VMX86_COW_COPY_PAGE,
      IOCTL_VMX86_COW_REMOVE_HINT,
      IOCTL_VMX86_ALLOC_LOCKED_PAGES,
      IOCTL_VMX86_FREE_LOCKED_PAGES,
      IOCTL_VMX86_GET_LOCKED_PAGES_LIST,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* MAP_LOCKED_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* UNMAP_LOCKED_PAGES */
      IOCTL_VMX86_SET_UID,		// VMX86_DEVEL only
      IOCTL_VMX86_IS_MP_SAFE,
      IOCTL_VMX86_ALLOW_CORE_DUMP,
      IOCTL_VMX86_BROADCAST_IPI,	// SMP 2.2.8+ only
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      IOCTL_VMX86_START_PASSTHROUGH,
      IOCTL_VMX86_STOP_PASSTHROUGH,
      IOCTL_VMX86_QUERY_PASSTHROUGH,
      IOCTL_VMX86_REGISTER_PERFCTR,
      IOCTL_VMX86_START_PERFCTR,
      IOCTL_VMX86_STOP_PERFCTR,
      IOCTL_VMX86_RELEASE_PERFCTR,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ALLOC_LOW_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* FREE_LOW_PAGES */
      IOCTL_VMX86_GET_ALL_CPUID,
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval = -EINVAL;

   if (iocmd < IOCTL_VMX86_V5_VERSION || iocmd >= IOCTL_VMX86_V5_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTL_VMX86_V5_VERSION:
         return VMMON_VERSION_V5;

      case IOCTL_VMX86_V5_BIND_VM:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_BIND_VM, ioarg);

      case IOCTL_VMX86_V5_GET_HARD_LIMIT:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_GET_HARD_LIMIT, ioarg);

      case IOCTL_VMX86_V5_ALLOC_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ALLOC_LOW_PAGES, ioarg);

      case IOCTL_VMX86_V5_FREE_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_FREE_LOW_PAGES, ioarg);

      case IOCTL_VMX86_V5_MAP_LOCKED_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_MAP_LOCKED_PAGES, ioarg);

      case IOCTL_VMX86_V5_UNMAP_LOCKED_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_UNMAP_LOCKED_PAGES, ioarg);

      case IOCTL_VMX86_V5_COW_SHARE:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_COW_SHARE, ioarg);

      case IOCTL_VMX86_V5_ACK_USER_CALL:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ACK_USER_CALL, ioarg);

      case IOCTL_VMX86_V5_CREATE_VM: {
         WSLimitsInfo wsLimitsInfo;
         if (vmLinux->vm != NULL) {
            retval = -EINVAL;
            break;
         }
         if ((void *) ioarg != NULL) {
            retval = HostIF_CopyFromUser(&wsLimitsInfo, (char *) ioarg,
                                         sizeof wsLimitsInfo);
            if (retval != 0) {
               break;
            }
         } else {
            memset(&wsLimitsInfo, 0, sizeof wsLimitsInfo);
         }
         vmLinux->vm = Vmx86_CreateVM((void *)filp, current->pid,
                                      vmLinux->version);
         if (vmLinux->vm == NULL) {
            retval = -ENOMEM;
            break;
         }
         /* Convert size from megabytes to pages... */
         vmLinux->vm->memInfo.mainMemSize = wsLimitsInfo.memsize * 256;
         retval = vmLinux->vm->id;
         break;
      }
      case IOCTL_VMX86_V5_INIT_VM: {
         InitBlock initParams;
         InitBlockV5 initParamsV5;

         if (vmLinux->vm == NULL) {
            retval = -EINVAL;
            break;
         }
         retval = HostIF_CopyFromUser(&initParamsV5, (char*)ioarg,
                                      sizeof initParamsV5);
         if (retval != 0) {
            break;
         }
         retval = initBlockV5_to_initBlock(&initParams, &initParamsV5);
         if (retval) {
            break;
         }
         if (Vmx86_InitVM(vmLinux->vm, &initParams)) {
            retval = -EINVAL;
            break;
         }
         break;
      }

      default:
         iocmd = v5map[iocmd - IOCTL_VMX86_V5_VERSION];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}

static inline int
initBlockV4_to_initBlock(InitBlock *dst, const InitBlockV4 *src) {
	unsigned int i;

	if (src->magicNumber != INIT_BLOCK_MAGIC_V4) {
		return -EINVAL;
	}
	dst->magicNumber = INIT_BLOCK_MAGIC;
	dst->numVCPUs = src->numVCPUs;
	for (i = 0; i < MAX_INITBLOCK_CPUS; i++) {
		dst->crosspage[i] = src->crosspage[i];
	}
	dst->vmInitFailurePeriod = 0;
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX32 --
 *
 *      Main path for UserRPC from VMware GSX 3.2.0
 *
 * Results:
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
LinuxDriver_IoctlGSX32(struct file *filp,
                       u_int iocmd,
		       unsigned long ioarg)
{
   static const unsigned int gsx32map[IOCTL_VMX86_GSX32_LAST - IOCTL_VMX86_GSX32_VERSION] = {
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VERSION */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CREATE_VM */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* BIND_VM */
      IOCTL_VMX86_RELEASE_VM,
      IOCTL_VMX86_GET_NUM_VMS,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* INIT_VM */
      IOCTL_VMX86_LATE_INIT_VM,
      IOCTL_VMX86_RUN_VM,
      IOCTL_VMX86_LOOK_UP_MPN,
      IOCTL_VMX86_LOCK_PAGE,
      IOCTL_VMX86_UNLOCK_PAGE,
      IOCTL_VMX86_APIC_BASE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_HARD_LIMIT */
      IOCTL_VMX86_SET_HARD_LIMIT,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTL_VMX86_SET_MEM_USAGE,
      IOCTL_VMX86_PAE_ENABLED,
      IOCTL_VMX86_HOST_X86_64,
      IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ACK_USER_CALL */
      IOCTL_VMX86_COMPLETE_USER_CALL,
      IOCTL_VMX86_GET_KHZ_ESTIMATE,
      IOCTL_VMX86_READ_PAGE,
      IOCTL_VMX86_WRITE_PAGE,
      IOCTL_VMX86_LOCK_PAGE_NEW,
      IOCTL_VMX86_UNLOCK_PAGE_BY_MPN,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* MARK_LOCKEDVARANGE_CLEAN */
      IOCTL_VMX86_ALLOC_LOCKED_PAGES,
      IOCTL_VMX86_FREE_LOCKED_PAGES,
      IOCTL_VMX86_GET_LOCKED_PAGES_LIST,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* MAP_LOCKED_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* UNMAP_LOCKED_PAGES */
      IOCTL_VMX86_APIC_ID,
      IOCTL_VMX86_SET_UID,
      IOCTL_VMX86_IS_MP_SAFE,
      IOCTL_VMX86_ALLOW_CORE_DUMP,
      IOCTL_VMX86_BROADCAST_IPI,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      IOCTL_VMX86_START_PASSTHROUGH,
      IOCTL_VMX86_STOP_PASSTHROUGH,
      IOCTL_VMX86_QUERY_PASSTHROUGH,
      IOCTL_VMX86_REGISTER_PERFCTR,
      IOCTL_VMX86_START_PERFCTR,
      IOCTL_VMX86_STOP_PERFCTR,
      IOCTL_VMX86_RELEASE_PERFCTR,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ALLOC_LOW_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* FREE_LOW_PAGES */
      IOCTL_VMX86_GET_ALL_CPUID,
      IOCTL_VMX86_SET_THREAD_AFFINITY,
      IOCTL_VMX86_GET_THREAD_AFFINITY,
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval = -EINVAL;

   if (iocmd < IOCTL_VMX86_GSX32_VERSION || iocmd >= IOCTL_VMX86_GSX32_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTL_VMX86_GSX32_VERSION:
         return VMMON_VERSION_GSX32;

      case IOCTL_VMX86_GSX32_CREATE_VM:
         return LinuxDriver_IoctlV5(filp, IOCTL_VMX86_V5_CREATE_VM, ioarg);

      case IOCTL_VMX86_GSX32_BIND_VM:
	 return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_BIND_VM, ioarg);

      case IOCTL_VMX86_GSX32_GET_HARD_LIMIT:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_GET_HARD_LIMIT, ioarg);

      case IOCTL_VMX86_GSX32_ALLOC_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ALLOC_LOW_PAGES, ioarg);

      case IOCTL_VMX86_GSX32_FREE_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_FREE_LOW_PAGES, ioarg);

      case IOCTL_VMX86_GSX32_MAP_LOCKED_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_MAP_LOCKED_PAGES, ioarg);

      case IOCTL_VMX86_GSX32_UNMAP_LOCKED_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_UNMAP_LOCKED_PAGES, ioarg);

      case IOCTL_VMX86_GSX32_ACK_USER_CALL:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ACK_USER_CALL, ioarg);

      case IOCTL_VMX86_GSX32_INIT_VM: {
         InitBlock initParams;
	 InitBlockV4 initParamsV4;

         if (vmLinux->vm == NULL) {
	    retval = -EINVAL;
	    break;
         }
         retval = HostIF_CopyFromUser(&initParamsV4, (char*)ioarg,
				      sizeof initParamsV4);
         if (retval != 0) {
	    break;
	 }
	 retval = initBlockV4_to_initBlock(&initParams, &initParamsV4);
	 if (retval) {
	    break;
	 }
	 if (Vmx86_InitVM(vmLinux->vm, &initParams)) {
	    retval = -EINVAL;
	    break;
         }
         break;
      }

      case IOCTL_VMX86_GSX32_IOAPIC_BASE: {
         MA ma; 

         if (vmLinux->vm == NULL) {
	    retval = -EINVAL;
            break;
         }
         ma = HostIF_IOAPICBaseV45();
         /* FIXME: Check whether MA was already 64 bit in GSX2/2.5 */
         retval = HostIF_CopyToUser((MA *)ioarg, &ma, sizeof ma);
         break;
      }

      case IOCTL_VMX86_GSX32_GET_MEM_INFO: {
         if (vmLinux->vm == NULL) {
            retval = -EINVAL;
            break;
         }
         if (!Vmx86_GetMemInfoCopyV45(vmLinux->vm, (VMMemInfoArgsV45 *) ioarg)) {
            retval = -EINVAL;
         }
         break;
      }

      case IOCTL_VMX86_GSX32_SET_MEM_INFO: {
         VMMemInfoArgsV45 args;

         if (vmLinux->vm == NULL) {
	    retval = -EINVAL;
            break;
         }
         retval = HostIF_CopyFromUser(&args, (void *)ioarg, sizeof args);
         if (retval != 0) {
	    break;
         }
         Vmx86_SetMemInfoV45(vmLinux->vm, &args); 
         retval = HostIF_CopyToUser((void *)ioarg, &args, sizeof args);
         break;
      }

      case IOCTL_VMX86_GSX32_MARK_LOCKEDVARANGE_CLEAN: {
         VARangeV45 var;

         retval = -EFAULT;
         if (HostIF_CopyFromUser(&var, (void *)ioarg, sizeof(VARangeV45)) == 0) {
            retval = HostIF_MarkLockedVARangeCleanV45(vmLinux->vm, var.addr, var.len);
         }
	 break;
      }

      default:
         iocmd = gsx32map[iocmd - IOCTL_VMX86_GSX32_VERSION];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV452 --
 *
 *      Main path for UserRPC from VMware Workstation 4.5.2
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV452(struct file *filp,
                      u_int iocmd,
		      unsigned long ioarg)
{
   static const unsigned int v45map[IOCTL_VMX86_V45_LAST - IOCTL_VMX86_V45_VERSION] = {
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VERSION */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CREATE_VM */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* BIND_VM */
      IOCTL_VMX86_RELEASE_VM,
      IOCTL_VMX86_GET_NUM_VMS,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* INIT_VM */
      IOCTL_VMX86_LATE_INIT_VM,
      IOCTL_VMX86_RUN_VM,
      IOCTL_VMX86_LOOK_UP_MPN,
      IOCTL_VMX86_LOCK_PAGE,
      IOCTL_VMX86_UNLOCK_PAGE,
      IOCTL_VMX86_APIC_BASE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_HARD_LIMIT */
      IOCTL_VMX86_SET_HARD_LIMIT,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTL_VMX86_SET_MEM_USAGE,
      IOCTL_VMX86_PAE_ENABLED,
      IOCTL_VMX86_HOST_X86_64,
      IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ACK_USER_CALL */
      IOCTL_VMX86_COMPLETE_USER_CALL,
      IOCTL_VMX86_GET_KHZ_ESTIMATE,
      IOCTL_VMX86_READ_PAGE,
      IOCTL_VMX86_WRITE_PAGE,
      IOCTL_VMX86_LOCK_PAGE_NEW,
      IOCTL_VMX86_UNLOCK_PAGE_BY_MPN,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* MARK_LOCKEDVARANGE_CLEAN */
      IOCTL_VMX86_ALLOC_LOCKED_PAGES,
      IOCTL_VMX86_FREE_LOCKED_PAGES,
      IOCTL_VMX86_GET_LOCKED_PAGES_LIST,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* MAP_LOCKED_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* UNMAP_LOCKED_PAGES */
      IOCTL_VMX86_SET_UID,
      IOCTL_VMX86_IS_MP_SAFE,
      IOCTL_VMX86_ALLOW_CORE_DUMP,
      IOCTL_VMX86_BROADCAST_IPI,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      IOCTL_VMX86_START_PASSTHROUGH,
      IOCTL_VMX86_STOP_PASSTHROUGH,
      IOCTL_VMX86_QUERY_PASSTHROUGH,
      IOCTL_VMX86_REGISTER_PERFCTR,
      IOCTL_VMX86_START_PERFCTR,
      IOCTL_VMX86_STOP_PERFCTR,
      IOCTL_VMX86_RELEASE_PERFCTR,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ALLOC_LOW_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* FREE_LOW_PAGES */
      IOCTL_VMX86_GET_ALL_CPUID
   };
   int retval = -EINVAL;

   if (iocmd < IOCTL_VMX86_V45_VERSION || iocmd >= IOCTL_VMX86_V45_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTL_VMX86_V45_VERSION:
         return VMMON_VERSION_V452;

      case IOCTL_VMX86_V45_BIND_VM:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_BIND_VM, ioarg);

      case IOCTL_VMX86_V45_GET_HARD_LIMIT:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_GET_HARD_LIMIT, ioarg);

      case IOCTL_VMX86_V45_ALLOC_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ALLOC_LOW_PAGES, ioarg);

      case IOCTL_VMX86_V45_FREE_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_FREE_LOW_PAGES, ioarg);

      case IOCTL_VMX86_V45_MAP_LOCKED_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_MAP_LOCKED_PAGES, ioarg);

      case IOCTL_VMX86_V45_UNMAP_LOCKED_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_UNMAP_LOCKED_PAGES, ioarg);

      case IOCTL_VMX86_V45_ACK_USER_CALL:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ACK_USER_CALL, ioarg);

      case IOCTL_VMX86_V45_CREATE_VM:
         return LinuxDriver_IoctlV5(filp, IOCTL_VMX86_V5_CREATE_VM, ioarg);

      case IOCTL_VMX86_V45_INIT_VM:
         return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_INIT_VM, ioarg);

      case IOCTL_VMX86_V45_IOAPIC_BASE:
         return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_IOAPIC_BASE, ioarg);

      case IOCTL_VMX86_V45_GET_MEM_INFO:
         return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_GET_MEM_INFO, ioarg);

      case IOCTL_VMX86_V45_SET_MEM_INFO:
         return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_SET_MEM_INFO, ioarg);

      case IOCTL_VMX86_V45_MARK_LOCKEDVARANGE_CLEAN:
         return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_MARK_LOCKEDVARANGE_CLEAN, ioarg);
	 
      default:
         iocmd = v45map[iocmd - IOCTL_VMX86_V45_VERSION];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV45 --
 *
 *      Main path for UserRPC from VMware Workstation 4.5.1 and GSX 3.0.0
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
LinuxDriver_IoctlV45(struct file *filp,
                        u_int iocmd,
		        unsigned long ioarg)
{
   switch (iocmd) {
      case IOCTL_VMX86_V45_VERSION:
         return VMMON_VERSION_V45;
      default:
         return LinuxDriver_IoctlV452(filp, iocmd, ioarg);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV4 --
 *
 *      Main path for UserRPC from VMware Workstation 4.0
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV4(struct file *filp,
                    u_int iocmd,
		    unsigned long ioarg)
{
   static const unsigned int v4map[IOCTL_VMX86_V4_LAST - IOCTL_VMX86_V4_VERSION] = {
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VERSION */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CREATE_VM */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* BIND_VM */
      IOCTL_VMX86_RELEASE_VM,
      IOCTL_VMX86_GET_NUM_VMS,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* INIT_VM */
      IOCTL_VMX86_LATE_INIT_VM,
      IOCTL_VMX86_RUN_VM,
      IOCTL_VMX86_LOOK_UP_MPN,
      IOCTL_VMX86_LOCK_PAGE,
      IOCTL_VMX86_UNLOCK_PAGE,
      IOCTL_VMX86_APIC_BASE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_STATS */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_STATS */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_HARD_LIMIT */
      IOCTL_VMX86_SET_HARD_LIMIT,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTL_VMX86_PAE_ENABLED,
      IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ACK_USER_CALL */
      IOCTL_VMX86_COMPLETE_USER_CALL,
      IOCTL_VMX86_GET_KHZ_ESTIMATE,
      IOCTL_VMX86_SET_UID,
      IOCTL_VMX86_IS_MP_SAFE,
      IOCTL_VMX86_ALLOW_CORE_DUMP,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CHECK_MEMORY */
      IOCTL_VMX86_BROADCAST_IPI,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      IOCTL_VMX86_START_PASSTHROUGH,
      IOCTL_VMX86_STOP_PASSTHROUGH,
      IOCTL_VMX86_QUERY_PASSTHROUGH,
      IOCTL_VMX86_REGISTER_PERFCTR,
      IOCTL_VMX86_START_PERFCTR,
      IOCTL_VMX86_STOP_PERFCTR,
      IOCTL_VMX86_RELEASE_PERFCTR,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ALLOC_LOW_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* FREE_LOW_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_SYSTEM_CLOCK_RATE */
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval = -EINVAL;

   if (iocmd < IOCTL_VMX86_V4_VERSION || iocmd >= IOCTL_VMX86_V4_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return retval;
   }
   switch (iocmd) {
      case IOCTL_VMX86_V4_VERSION:
         return VMMON_VERSION_V4;

      case IOCTL_VMX86_V4_BIND_VM:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_BIND_VM, ioarg);

      case IOCTL_VMX86_V4_GET_HARD_LIMIT:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_GET_HARD_LIMIT, ioarg);

      case IOCTL_VMX86_V4_ALLOC_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ALLOC_LOW_PAGES, ioarg);

      case IOCTL_VMX86_V4_FREE_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_FREE_LOW_PAGES, ioarg);

      case IOCTL_VMX86_V4_ACK_USER_CALL:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ACK_USER_CALL, ioarg);

      case IOCTL_VMX86_V4_CREATE_VM:
         return LinuxDriver_IoctlV5(filp, IOCTL_VMX86_V5_CREATE_VM, ioarg);

      case IOCTL_VMX86_V4_INIT_VM:
         return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_INIT_VM, ioarg);

      case IOCTL_VMX86_V4_IOAPIC_BASE:
	 return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_IOAPIC_BASE, ioarg);

      case IOCTL_VMX86_V4_GET_STATS: {
         VMX86StatsV4 stats;

         if (vmLinux->vm == NULL) {
            break;
         }
	 stats = vmLinux->vm->stats;
	 stats.monitorLockedPages -= stats.userLockedPages;
         retval = HostIF_CopyToUser((void *)ioarg, &stats,
                                    sizeof stats);
         break;
      }

      case IOCTL_VMX86_V4_SET_STATS: {
         VMX86StatsV4 stats;

         if (vmLinux->vm == NULL) {
            break;
         }
         retval = HostIF_CopyFromUser(&stats, (void *)ioarg,
                                      sizeof stats);
	 if (retval == 0) {
	    vmLinux->vm->stats.maxMonitorLockedPages = stats.maxMonitorLockedPages;
	    vmLinux->vm->stats.maxUserLockedPages = stats.maxUserLockedPages;
	 }
         break;
      }

      case IOCTL_VMX86_V4_GET_MEM_INFO: {
         VMDriver *vm = vmLinux->vm;

         if (vm == NULL) {
            break;
         }
         if (!Vmx86_GetMemInfoCopyV4(vm, (VMGetMemInfoArgsV4 *) ioarg)) {
            break;
         }
         return 0;
      }

      case IOCTL_VMX86_V4_SET_MEM_INFO: {
         VMSetMemInfoArgsV4 args;
         VMDriver *vm = vmLinux->vm;

         if (vm == NULL) {
            break;
         }
         retval = HostIF_CopyFromUser(&args, (void *)ioarg, sizeof args);
         if (retval != 0) {
	    break;
         }
         Vmx86_SetMemInfoV4(vm, &args);
	 break;
      }

      case IOCTL_VMX86_V4_CHECK_MEMORY:
         if (vmLinux->vm == NULL) {
            break;
         }
         retval = HostIF_CheckMemory(vmLinux->vm);
         break;

      case IOCTL_VMX86_V4_GET_SYSTEM_CLOCK_RATE:
         retval = HZ;
         break;

      default:
         iocmd = v4map[iocmd - IOCTL_VMX86_V4_VERSION];
         return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX25 --
 *
 *      Main path for UserRPC from VMware GSX 2.5
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static inline int
initBlockG25_to_initBlock(InitBlock *dst, InitBlockG25 *src) {
	unsigned int i;

	if (src->magicNumber != INIT_BLOCK_MAGIC_V3) {
		return -EINVAL;
	}
	dst->magicNumber = INIT_BLOCK_MAGIC;
	dst->numVCPUs = src->numVCPUs;
	for (i = 0; i < MAX_INITBLOCK_CPUS; i++) {
		dst->crosspage[i] = src->crosspage[i];
	}
	dst->vmInitFailurePeriod = 0;
	return 0;
}

static inline void
initBlock_to_initBlockG25(InitBlockG25 *dst, InitBlock *src) {
	Vmx86_SetStartTime(&dst->st);
}

static int
LinuxDriver_IoctlGSX25(struct file *filp,
                       u_int iocmd,
		       unsigned long ioarg)
{
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval;
   static const unsigned int gsx25map[IOCTL_VMX86_GSX25_LAST - IOCTL_VMX86_GSX25_VERSION] = {
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VERSION */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CREATE_VM */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* BIND_VM */
      IOCTL_VMX86_RELEASE_VM,
      IOCTL_VMX86_GET_NUM_VMS,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* INIT_VM */
      IOCTL_VMX86_LATE_INIT_VM,
      IOCTL_VMX86_RUN_VM,
      IOCTL_VMX86_LOOK_UP_MPN,
      IOCTL_VMX86_LOCK_PAGE,
      IOCTL_VMX86_UNLOCK_PAGE,
      IOCTL_VMX86_APIC_BASE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_STATS */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_STATS */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_HARD_LIMIT */
      IOCTL_VMX86_SET_HARD_LIMIT,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTL_VMX86_PAE_ENABLED,
      IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_TSCS */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_TSCS */
      IOCTL_VMX86_GET_KHZ_ESTIMATE,
      IOCTL_VMX86_SET_UID,
      IOCTL_VMX86_IS_MP_SAFE,
      IOCTL_VMX86_ALLOW_CORE_DUMP,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CHECK_MEMORY */
      IOCTL_VMX86_BROADCAST_IPI,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      IOCTL_VMX86_START_PASSTHROUGH,
      IOCTL_VMX86_STOP_PASSTHROUGH,
      IOCTL_VMX86_QUERY_PASSTHROUGH,
      IOCTL_VMX86_REGISTER_PERFCTR,
      IOCTL_VMX86_START_PERFCTR,
      IOCTL_VMX86_STOP_PERFCTR,
      IOCTL_VMX86_RELEASE_PERFCTR,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ALLOC_LOW_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* FREE_LOW_PAGES */
   };

   if (iocmd < IOCTL_VMX86_GSX25_VERSION || iocmd >= IOCTL_VMX86_GSX25_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   switch (iocmd) {
      case IOCTL_VMX86_GSX25_VERSION:
         return VMMON_VERSION_GSX25;
      case IOCTL_VMX86_GSX25_BIND_VM:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_BIND_VM, ioarg);
      case IOCTL_VMX86_GSX25_GET_HARD_LIMIT:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_GET_HARD_LIMIT, ioarg);
      case IOCTL_VMX86_GSX25_ALLOC_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ALLOC_LOW_PAGES, ioarg);
      case IOCTL_VMX86_GSX25_FREE_LOW_PAGES:
         return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_FREE_LOW_PAGES, ioarg);
      case IOCTL_VMX86_GSX25_CREATE_VM:
         return LinuxDriver_IoctlV5(filp, IOCTL_VMX86_V5_CREATE_VM, ioarg);
      case IOCTL_VMX86_GSX25_INIT_VM: {
         InitBlockG25 initParamsG25;
         InitBlock initParams;

         if (vmLinux->vm == NULL) {
	    return -EINVAL;
         }
         retval = HostIF_CopyFromUser(&initParamsG25, (char*)ioarg,
				      sizeof initParamsG25);
         if (retval != 0) {
	    return retval;
         }
	 if (initBlockG25_to_initBlock(&initParams, &initParamsG25)) {
	    return -EINVAL;
	 }
         if (Vmx86_InitVM(vmLinux->vm, &initParams)) {
	    return -EINVAL;
         }
         initBlock_to_initBlockG25(&initParamsG25, &initParams); 
         return HostIF_CopyToUser((char*)ioarg, &initParamsG25,sizeof initParamsG25);
      }
      case IOCTL_VMX86_GSX25_SET_MEM_INFO:
         return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_SET_MEM_INFO, ioarg);
      case IOCTL_VMX86_GSX25_GET_MEM_INFO:
         return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_GET_MEM_INFO, ioarg);
      case IOCTL_VMX86_GSX25_SET_STATS:
         return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_SET_STATS, ioarg);
      case IOCTL_VMX86_GSX25_GET_STATS:
         return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_GET_STATS, ioarg);
      case IOCTL_VMX86_GSX25_CHECK_MEMORY:
         return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_CHECK_MEMORY, ioarg);
      case IOCTL_VMX86_GSX25_IOAPIC_BASE:
	 return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_IOAPIC_BASE, ioarg);
      default:
         iocmd = gsx25map[iocmd - IOCTL_VMX86_GSX25_VERSION];
	 break;
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX251 --
 *
 *      Main path for UserRPC from VMware GSX 2.5.1
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
LinuxDriver_IoctlGSX251(struct file *filp,
                        u_int iocmd,
		        unsigned long ioarg)
{
   switch (iocmd) {
      case IOCTL_VMX86_GSX25_VERSION:
         return VMMON_VERSION_GSX251;
      case IOCTL_VMX86_GSX251_GET_ALL_CPUID:
         iocmd = IOCTL_VMX86_GET_ALL_CPUID;
	 break;
      default:
         return LinuxDriver_IoctlGSX25(filp, iocmd, ioarg);
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX2 --
 *
 *      Main path for UserRPC from VMware GSX 2.0
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlGSX2(struct file *filp,
                      u_int iocmd,
		      unsigned long ioarg)
{
   int retval;
   static const unsigned int gsx2map[IOCTL_VMX86_GSX2_LAST - IOCTL_VMX86_GSX2_VERSION] = {
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* VERSION */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CREATE_VM */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* BIND_VM */
      IOCTL_VMX86_RELEASE_VM,
      IOCTL_VMX86_GET_NUM_VMS,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* INIT_VM */
      IOCTL_VMX86_LATE_INIT_VM,
      IOCTL_VMX86_RUN_VM,
      IOCTL_VMX86_LOOK_UP_MPN,
      IOCTL_VMX86_LOCK_PAGE,
      IOCTL_VMX86_UNLOCK_PAGE,
      IOCTL_VMX86_APIC_BASE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* IOAPIC_BASE */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_STATS */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_STATS */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_HARD_LIMIT */
      IOCTL_VMX86_SET_HARD_LIMIT,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_MEM_INFO */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* SET_MEM_INFO */
      IOCTL_VMX86_PAE_ENABLED,
      IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      IOCTL_VMX86_SET_UID,
      IOCTL_VMX86_IS_MP_SAFE,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* GET_MHZ_ESTIMATE */
      IOCTL_VMX86_ALLOW_CORE_DUMP,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* CHECK_MEMORY */
      IOCTL_VMX86_BROADCAST_IPI,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      IOCTL_VMX86_START_PASSTHROUGH,
      IOCTL_VMX86_STOP_PASSTHROUGH,
      IOCTL_VMX86_QUERY_PASSTHROUGH,
      IOCTL_VMX86_REGISTER_PERFCTR,
      IOCTL_VMX86_START_PERFCTR,
      IOCTL_VMX86_STOP_PERFCTR,
      IOCTL_VMX86_RELEASE_PERFCTR,
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* ALLOC_LOW_PAGES */
      IOCTL_VMX86_NOT_IMPLEMENTED,	/* FREE_LOW_PAGES */
   };

   if (iocmd < IOCTL_VMX86_GSX2_VERSION || iocmd >= IOCTL_VMX86_GSX2_LAST) {
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   switch (iocmd) {
   case IOCTL_VMX86_GSX2_VERSION:
      return VMMON_VERSION_GSX2;
   case IOCTL_VMX86_GSX2_BIND_VM:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_BIND_VM, ioarg);
   case IOCTL_VMX86_GSX2_GET_HARD_LIMIT:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_GET_HARD_LIMIT, ioarg);
   case IOCTL_VMX86_GSX2_ALLOC_LOW_PAGES:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ALLOC_LOW_PAGES, ioarg);
   case IOCTL_VMX86_GSX2_FREE_LOW_PAGES:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_FREE_LOW_PAGES, ioarg);
   case IOCTL_VMX86_GSX2_CREATE_VM:
      return LinuxDriver_IoctlV5(filp, IOCTL_VMX86_V5_CREATE_VM, ioarg);
   case IOCTL_VMX86_GSX2_INIT_VM:
      return LinuxDriver_IoctlGSX25(filp, IOCTL_VMX86_GSX25_INIT_VM, ioarg);
   case IOCTL_VMX86_GSX2_SET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_SET_MEM_INFO, ioarg);
   case IOCTL_VMX86_GSX2_GET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_GET_MEM_INFO, ioarg);
   case IOCTL_VMX86_GSX2_SET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_SET_STATS, ioarg);
   case IOCTL_VMX86_GSX2_GET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_GET_STATS, ioarg);
   case IOCTL_VMX86_GSX2_CHECK_MEMORY:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_CHECK_MEMORY, ioarg);
   case IOCTL_VMX86_GSX2_IOAPIC_BASE:
      return LinuxDriver_IoctlGSX32(filp, IOCTL_VMX86_GSX32_IOAPIC_BASE, ioarg);
   case IOCTL_VMX86_GSX2_GET_MHZ_ESTIMATE:
      retval = Vmx86_GetkHzEstimate(&linuxState.startTime);
      if (retval >= 0) {
         retval /= 1000;
      }
      break;

   default:
      iocmd = gsx2map[iocmd - IOCTL_VMX86_GSX2_VERSION];
      return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
   }
   return retval;
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV3 --
 *
 *      Main path for UserRPC from VMware 3.[01]
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static inline int
initBlock30_to_initBlock(InitBlock *dst, InitBlock30 *src) {
	if (src->magicNumber != INIT_BLOCK_MAGIC_V3) {
		return -EINVAL;
	}
	dst->magicNumber = INIT_BLOCK_MAGIC;
	dst->numVCPUs = 1;
	dst->crosspage[0] = src->crosspage;
	dst->vmInitFailurePeriod = 0;
	return 0;
}

static inline void
initBlock_to_initBlock30(InitBlock30 *dst, InitBlock *src) {
	Vmx86_SetStartTime(&dst->st);
}

static int
LinuxDriver_IoctlV3(struct file *filp,
                    u_int iocmd,
                    unsigned long ioarg)
{
   static const unsigned int v3map[] = {
      [IOCTL_VMX86_V3_RELEASE_VM]		= IOCTL_VMX86_RELEASE_VM,
      [IOCTL_VMX86_V3_LATE_INIT_VM]		= IOCTL_VMX86_LATE_INIT_VM,
      [IOCTL_VMX86_V3_SET_UID]			= IOCTL_VMX86_SET_UID,
      [IOCTL_VMX86_V3_GET_NUM_VMS]		= IOCTL_VMX86_GET_NUM_VMS,
      [IOCTL_VMX86_V3_GET_TOTAL_MEM_USAGE]	= IOCTL_VMX86_GET_TOTAL_MEM_USAGE,
      [IOCTL_VMX86_V3_SET_HARD_LIMIT]		= IOCTL_VMX86_SET_HARD_LIMIT,
      [IOCTL_VMX86_V3_PAE_ENABLED]		= IOCTL_VMX86_PAE_ENABLED,
      [IOCTL_VMX86_V3_IS_MP_SAFE]		= IOCTL_VMX86_IS_MP_SAFE,
      [IOCTL_VMX86_V3_ALLOW_CORE_DUMP]		= IOCTL_VMX86_ALLOW_CORE_DUMP,
      [IOCTL_VMX86_V3_BROADCAST_IPI]		= IOCTL_VMX86_BROADCAST_IPI,
      [IOCTL_VMX86_V3_REGISTER_PASSTHROUGH_IRQ]	= IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ,
      [IOCTL_VMX86_V3_REGISTER_PASSTHROUGH_IO]	= IOCTL_VMX86_REGISTER_PASSTHROUGH_IO,
      [IOCTL_VMX86_V3_FREE_PASSTHROUGH_IRQ]	= IOCTL_VMX86_FREE_PASSTHROUGH_IRQ,
      [IOCTL_VMX86_V3_FREE_PASSTHROUGH_IO]	= IOCTL_VMX86_FREE_PASSTHROUGH_IO,
      [IOCTL_VMX86_V3_START_PASSTHROUGH]	= IOCTL_VMX86_START_PASSTHROUGH,
      [IOCTL_VMX86_V3_STOP_PASSTHROUGH]		= IOCTL_VMX86_STOP_PASSTHROUGH,
      [IOCTL_VMX86_V3_QUERY_PASSTHROUGH]	= IOCTL_VMX86_QUERY_PASSTHROUGH,
      [IOCTL_VMX86_V3_REGISTER_PERFCTR]		= IOCTL_VMX86_REGISTER_PERFCTR,
      [IOCTL_VMX86_V3_START_PERFCTR]		= IOCTL_VMX86_START_PERFCTR,
      [IOCTL_VMX86_V3_STOP_PERFCTR]		= IOCTL_VMX86_STOP_PERFCTR,
      [IOCTL_VMX86_V3_RELEASE_PERFCTR]		= IOCTL_VMX86_RELEASE_PERFCTR,
   };
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval;

   switch (iocmd) {
   case IOCTL_VMX86_V3_VERSION:
      return VMMON_VERSION_V3;
   case IOCTL_VMX86_V3_RELEASE_VM:
   case IOCTL_VMX86_V3_LATE_INIT_VM:
   case IOCTL_VMX86_V3_SET_UID:
   case IOCTL_VMX86_V3_GET_NUM_VMS:
   case IOCTL_VMX86_V3_GET_TOTAL_MEM_USAGE:
   case IOCTL_VMX86_V3_SET_HARD_LIMIT:
   case IOCTL_VMX86_V3_PAE_ENABLED:
   case IOCTL_VMX86_V3_IS_MP_SAFE:
   case IOCTL_VMX86_V3_ALLOW_CORE_DUMP:
   case IOCTL_VMX86_V3_BROADCAST_IPI:
   case IOCTL_VMX86_V3_REGISTER_PASSTHROUGH_IRQ:
   case IOCTL_VMX86_V3_REGISTER_PASSTHROUGH_IO:
   case IOCTL_VMX86_V3_FREE_PASSTHROUGH_IRQ:
   case IOCTL_VMX86_V3_FREE_PASSTHROUGH_IO:
   case IOCTL_VMX86_V3_START_PASSTHROUGH:
   case IOCTL_VMX86_V3_STOP_PASSTHROUGH:
   case IOCTL_VMX86_V3_QUERY_PASSTHROUGH:
   case IOCTL_VMX86_V3_REGISTER_PERFCTR:
   case IOCTL_VMX86_V3_START_PERFCTR:
   case IOCTL_VMX86_V3_STOP_PERFCTR:
   case IOCTL_VMX86_V3_RELEASE_PERFCTR:
      iocmd = v3map[iocmd];
      break;
   case IOCTL_VMX86_V3_BIND_VM:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_BIND_VM, ioarg);
   case IOCTL_VMX86_V3_GET_HARD_LIMIT:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_GET_HARD_LIMIT, ioarg);
   case IOCTL_VMX86_V3_CREATE_VM:
      return LinuxDriver_IoctlV5(filp, IOCTL_VMX86_V5_CREATE_VM, ioarg);
   case IOCTL_VMX86_V3_INIT_VM: {
      InitBlock30 initParams30;
      InitBlock initParams;

      if (vmLinux->vm == NULL) {
	 return -EINVAL;
      }
      retval = HostIF_CopyFromUser(&initParams30, (char*)ioarg,
				   sizeof initParams30);
      if (retval != 0) {
	 return retval;
      }
      if (initBlock30_to_initBlock(&initParams, &initParams30)) {
      	 return -EINVAL;
      }
      if (Vmx86_InitVM(vmLinux->vm, &initParams)) {
	 return -EINVAL;
      }
      initBlock_to_initBlock30(&initParams30, &initParams); 
      return HostIF_CopyToUser((char*)ioarg,&initParams30,sizeof(initParams30));
   }
   case IOCTL_VMX86_V3_RUN_VM:
      ioarg = VCPUID_UP;
      iocmd = IOCTL_VMX86_RUN_VM;
      break;
   case IOCTL_VMX86_V3_LOOK_UP_MPN:
      retval = __LinuxDriver_Ioctl(filp, IOCTL_VMX86_LOOK_UP_MPN, ioarg);
      if (retval >= 0 && (retval & PAGE_LOCK_ERROR_BIT)) {
         return 0;
      }
      return retval;
   case IOCTL_VMX86_V3_LOCK_PAGE:
      retval = __LinuxDriver_Ioctl(filp, IOCTL_VMX86_LOCK_PAGE, ioarg);
      if (retval >= 0 && (retval & PAGE_LOCK_ERROR_BIT)) {
         return 0;
      }
      return retval;
   case IOCTL_VMX86_V3_UNLOCK_PAGE:
      retval = __LinuxDriver_Ioctl(filp, IOCTL_VMX86_UNLOCK_PAGE, ioarg);
      if (retval >= 0 && (retval & PAGE_LOCK_ERROR_BIT)) {
         return 0;
      }
      return retval;
   case IOCTL_VMX86_V3_APIC_BASE: {
      VMAPICInfo info;
      mm_segment_t oldfs;
      
      info.flags = ioarg;
      oldfs = get_fs();
      set_fs(KERNEL_DS);
      retval = __LinuxDriver_Ioctl(filp, IOCTL_VMX86_APIC_BASE, (unsigned long)&info);
      set_fs(oldfs);
      if (retval) {
         return retval;
      }
      return info.apicBase;
   }
   case IOCTL_VMX86_V3_IOAPIC_BASE: {
      /* It is simpler to copy code for retrieving IOAPICBase than
         writting wrapper around GSX2 call */
      if (vmLinux->vm == NULL) {
	 retval = -EINVAL;
	 break;
      }
      return HostIF_IOAPICBaseV45();
   }
   case IOCTL_VMX86_V3_SET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_SET_MEM_INFO, ioarg);
   case IOCTL_VMX86_V3_GET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_GET_MEM_INFO, ioarg);
   case IOCTL_VMX86_V3_SET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_SET_STATS, ioarg);
   case IOCTL_VMX86_V3_GET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_GET_STATS, ioarg);
   case IOCTL_VMX86_V3_CHECK_MEMORY:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_CHECK_MEMORY, ioarg);
   case IOCTL_VMX86_V3_GET_MHZ_ESTIMATE:
      return LinuxDriver_IoctlGSX2(filp, IOCTL_VMX86_GSX2_GET_MHZ_ESTIMATE, ioarg);

   default:
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV32 --
 *
 *      Main path for UserRPC from VMware 3.2
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV32(struct file *filp,
                     u_int iocmd,
                     unsigned long ioarg)
{
   switch (iocmd) {
   case IOCTL_VMX86_V3_VERSION:
      return VMMON_VERSION_V32;
   case IOCTL_VMX86_V32_ALLOC_LOW_PAGES:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_ALLOC_LOW_PAGES, ioarg);
   case IOCTL_VMX86_V32_FREE_LOW_PAGES:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_FREE_LOW_PAGES, ioarg);
   }
   return LinuxDriver_IoctlV3(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV321 --
 *
 *      Main path for UserRPC from VMware 3.2.1
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV321(struct file *filp,
                     u_int iocmd,
                     unsigned long ioarg)
{
   switch (iocmd) {
   case IOCTL_VMX86_V3_VERSION:
      return VMMON_VERSION_V321;
   /*
    * There is new ioctl, IOCTL_VMX86_V321_ADD_WORLD_ARG, but it
    * is not handled by vmmon 3.2.1 code.
    */
   }
   return LinuxDriver_IoctlV32(filp, iocmd, ioarg);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlV2 --
 *
 *      Main path for UserRPC from VMware 2.x
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlV2(VMLinux *vmLinux,
		    struct file *filp,
                    u_int iocmd,
		    unsigned long ioarg)
{
   VMDriver *vm = vmLinux->vm;

   switch (iocmd) {
   case IOCTL_VMX86_V2_INIT: {
      InitBlock20 initParams;
      InitBlock realParams;
      VmTimeStart stt;

      if (HostIF_CopyFromUser(&initParams,(char*)ioarg,sizeof(initParams))) {
         return -EFAULT;
      }
      if (initParams.magicNumber != INIT_BLOCK_MAGIC_V3) {
	 return -EINVAL;
      }
      realParams.magicNumber = INIT_BLOCK_MAGIC;
      realParams.numVCPUs = 1;
      realParams.crosspage[0] = initParams.crosspage;
      realParams.vmInitFailurePeriod = 0;
      if (Vmx86_InitVM(vm, &realParams)) {
	 return -EPERM;
      }
      initParams.smpMachine = __LinuxDriver_Ioctl(filp, IOCTL_VMX86_IS_MP_SAFE, 0);
      Vmx86_SetStartTime(&stt);
      initParams.st.count = stt.count;
      initParams.st.time = stt.time;
      initParams.st.mhzEstimate = Vmx86_GetkHzEstimate(&linuxState.startTime) / 1000;
      if (HostIF_CopyToUser((char*)ioarg,&initParams,sizeof(initParams))) {
         return -EFAULT;
      }
      return 0;
   }
   case IOCTL_VMX86_V2_LATE_INIT:
      iocmd = IOCTL_VMX86_LATE_INIT_VM;
      break;      
   case IOCTL_VMX86_V2_RUN:
      ioarg = VCPUID_UP;
      iocmd = IOCTL_VMX86_RUN_VM;
      break;
   case IOCTL_VMX86_V2_BEEP: { // XXX for buggy Linux
      uint8 byte;
      byte = inb(SPEAKER_PORT);
      if (ioarg >> 16) {
	 byte |= (SPEAKER_ENABLE_SPEAKER | SPEAKER_TIMER2_GATE);
      } else {
	 byte &= ~(SPEAKER_ENABLE_SPEAKER | SPEAKER_TIMER2_GATE);
      }
      outb(byte, SPEAKER_PORT);
      outb(0xb6, 0x43);
      outb((uint8) ioarg, 0x42); /* outb_p needs paravirt_ops */
      outb((uint8) (ioarg >> 8), 0x42);
      return 0;
   }

   // XXX Deprecated.  Now handled by devel_suid() for certain types of
   // open() calls.
   case IOCTL_VMX86_V2_SETUID:
      return -EPERM;
   case IOCTL_VMX86_V2_LOOKUPMPN:
      return LinuxDriver_IoctlV3(filp, IOCTL_VMX86_V3_LOOK_UP_MPN, ioarg);
   case IOCTL_VMX86_V2_LOCKPAGE:
      return LinuxDriver_IoctlV3(filp, IOCTL_VMX86_V3_LOCK_PAGE, ioarg);
   case IOCTL_VMX86_V2_UNLOCKPAGE:
      return LinuxDriver_IoctlV3(filp, IOCTL_VMX86_V3_UNLOCK_PAGE, ioarg);
   case IOCTL_VMX86_V2_GET_NUM_VMS:
      iocmd = IOCTL_VMX86_GET_NUM_VMS;
      break;
   case IOCTL_VMX86_V2_SET_HARD_LIMIT:
      iocmd = IOCTL_VMX86_SET_HARD_LIMIT;
      break;
   case IOCTL_VMX86_V2_GET_HARD_LIMIT:
      return LinuxDriver_IoctlS1B1(filp, IOCTL_VMX86_S1B1_GET_HARD_LIMIT, ioarg);
   case IOCTL_VMX86_V2_SET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_SET_MEM_INFO, ioarg);
   case IOCTL_VMX86_V2_GET_MEM_INFO:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_GET_MEM_INFO, ioarg);
   case IOCTL_VMX86_V2_GET_VM_LIST:
      if (!Vmx86_GetVMListCopy((VMGetVMListResult *)ioarg)) {
         return -EINVAL;
      }
      return 0;
   case IOCTL_VMX86_V2_GET_VM_INFO: {
      VMGetVMInfoResult info;

      if (HostIF_CopyFromUser(&info, (void *)ioarg, sizeof(info))) {
         return -EFAULT;
      }
      if (!Vmx86_GetVMInfo(info.vmUID, &info)) {
         return -EINVAL;
      }
      if (HostIF_CopyToUser((void *)ioarg, &info, sizeof(info))) {
         return -EFAULT;
      }
      return 0;
   }
   case IOCTL_VMX86_V2_SET_VM_INFO: {
      VMSetVMInfoArgs info;

      if (HostIF_CopyFromUser(&info, (void *)ioarg, sizeof(info))) {
         return -EFAULT;
      }
      Vmx86_SetVMInfo(vm, &info);
      return 0;
   }
   case IOCTL_VMX86_V2_SET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_SET_STATS, ioarg);
   case IOCTL_VMX86_V2_GET_STATS:
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_GET_STATS, ioarg);
   case IOCTL_VMX86_V2_DECLARE_SLAVE:
      /* obsolete */
      return 0;
   case IOCTL_VMX86_V2_ISMPSAFE:
      iocmd = IOCTL_VMX86_IS_MP_SAFE;
      break;
   case IOCTL_VMX86_V2_APICBASE:
      /* v2.0 had only one APIC flag... */
      if (ioarg) {
         ioarg = APIC_FLAG_DISABLE_NMI;
      }
      return LinuxDriver_IoctlV3(filp, IOCTL_VMX86_V3_APIC_BASE, ioarg);
   case IOCTL_VMX86_V2_IOAPICBASE:
      return LinuxDriver_IoctlV3(filp, IOCTL_VMX86_V3_IOAPIC_BASE, 0);
   case IOCTL_VMX86_V2_CHECK_MEMORY: 
      return LinuxDriver_IoctlV4(filp, IOCTL_VMX86_V4_CHECK_MEMORY, ioarg);
   case IOCTL_VMX86_V2_REGISTER_PASSTHROUGH_IRQ:
      iocmd = IOCTL_VMX86_REGISTER_PASSTHROUGH_IRQ;
      break;
   case IOCTL_VMX86_V2_REGISTER_PASSTHROUGH_IO:
      iocmd = IOCTL_VMX86_REGISTER_PASSTHROUGH_IO;
      break;
   case IOCTL_VMX86_V2_FREE_PASSTHROUGH_IRQ:
      iocmd = IOCTL_VMX86_FREE_PASSTHROUGH_IRQ;
      break;
   case IOCTL_VMX86_V2_FREE_PASSTHROUGH_IO:
      iocmd = IOCTL_VMX86_FREE_PASSTHROUGH_IO;
      break;
   case IOCTL_VMX86_V2_START_PASSTHROUGH:
      iocmd = IOCTL_VMX86_START_PASSTHROUGH;
      break;
   case IOCTL_VMX86_V2_STOP_PASSTHROUGH:
      iocmd = IOCTL_VMX86_STOP_PASSTHROUGH;
      break;
   case IOCTL_VMX86_V2_QUERY_PASSTHROUGH:
      iocmd = IOCTL_VMX86_QUERY_PASSTHROUGH;
      break;
   case IOCTL_VMX86_V2_ALLOW_CORE_DUMP:
      iocmd = IOCTL_VMX86_ALLOW_CORE_DUMP;
      break;
   case IOCTL_VMX86_V2_BROADCAST_IPI:
      iocmd = IOCTL_VMX86_BROADCAST_IPI;
      break;
   case IOCTL_VMX86_V2_FREE_RESOURCES:
      return HostIF_FreeAllResources(vm);
   default:
      Warning("Unknown ioctl %d\n", iocmd);
      return -EINVAL;
   }
   return __LinuxDriver_Ioctl(filp, iocmd, ioarg);
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_IoctlGSX1 --
 *
 *      Main path for UserRPC from VMware GSX 1.x
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_IoctlGSX1(struct file *filp,
                      u_int iocmd,
		      unsigned long ioarg)
{
   if (iocmd >= IOCTL_VMX86_V3_PAE_ENABLED) {
      iocmd += 2;
   }
   switch (iocmd) {
   case IOCTL_VMX86_V3_VERSION:
      return VMMON_VERSION_GSX1;
   }
   return LinuxDriver_IoctlV3(filp, iocmd, ioarg);
}

/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Ioctl --
 *
 *      Main path for UserRPC
 *
 * Results:
 *
 *
 *
 *
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int 
LinuxDriver_Ioctl( struct inode *inode, 
                   struct file *filp,
                   u_int iocmd, 
                   unsigned long ioarg )
{
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   int retval;

   VMWare_SetVTracer(VTrace_Set);

   if (vmLinux->version == VME_UNKNOWN) {
      if (iocmd == IOCTL_VMX86_V2_GET_NUM_VMS || iocmd == IOCTL_VMX86_V2_ISMPSAFE || iocmd == IOCTL_VMX86_V2_SET_HARD_LIMIT) {
         vmLinux->version = VME_V2;
      } else if (iocmd == IOCTL_VMX86_VERSION) {
         /* VMware4.5 and newer use 2001, so force VM55 if default is not acceptable */
	 if (vmversion >= VME_V45) {
	    vmLinux->version = vmversion;
	 } else {
            vmLinux->version = VME_V55;
	 }
      } else if (iocmd == IOCTL_VMX86_V4_VERSION) {
      	 /* 201 cannot be VMware 4.5 or anything newer; use VMware4 as safe fallback */
      	 if (vmversion >= VME_V45) {
	    vmLinux->version = VME_V4;
	 } else {
	    vmLinux->version = vmversion;
	 }
      } else {
         Warning("Unexpected ioctl (%d) invoked, using default\n", iocmd);
         vmLinux->version = vmversion;
      }
   }
   switch (vmLinux->version) {
      case VME_V2:
         if (!vmLinux->vm) {
            retval = LinuxDriver_InitV2(filp, vmLinux);
	    if (retval) {
	       break;
	    }
         }
         retval = LinuxDriver_IoctlV2(vmLinux, filp, iocmd, ioarg);
	 break;
      case VME_GSX1:
         retval = LinuxDriver_IoctlGSX1(filp, iocmd, ioarg);
	 break;
      case VME_V3:
         retval = LinuxDriver_IoctlV3(filp, iocmd, ioarg);
	 break;
      case VME_V32:
         retval = LinuxDriver_IoctlV32(filp, iocmd, ioarg);
	 break;
      case VME_V321:
         retval = LinuxDriver_IoctlV321(filp, iocmd, ioarg);
	 break;
      case VME_GSX2:
         retval = LinuxDriver_IoctlGSX2(filp, iocmd, ioarg);
	 break;
      case VME_GSX25:
         retval = LinuxDriver_IoctlGSX25(filp, iocmd, ioarg);
	 break;
      case VME_GSX251:
         retval = LinuxDriver_IoctlGSX251(filp, iocmd, ioarg);
	 break;
      case VME_V4:
      	 retval = LinuxDriver_IoctlV4(filp, iocmd, ioarg);
	 break;
      case VME_V45:
      	 retval = LinuxDriver_IoctlV45(filp, iocmd, ioarg);
	 break;
      case VME_V452:
	 retval = LinuxDriver_IoctlV452(filp, iocmd, ioarg);
	 break;
      case VME_GSX32:
	 retval = LinuxDriver_IoctlGSX32(filp, iocmd, ioarg);
         break;
      case VME_V5:
	 retval = LinuxDriver_IoctlV5(filp, iocmd, ioarg);
	 break;
      case VME_V55:
	 retval = LinuxDriver_IoctlV55(filp, iocmd, ioarg);
	 break;
      case VME_S1B1:
	 retval = LinuxDriver_IoctlS1B1(filp, iocmd, ioarg);
	 break;
      case VME_V6:
         retval = LinuxDriver_IoctlV6(filp, iocmd, ioarg);
	 break;
      case VME_TOT:
         retval = LinuxDriver_IoctlTOT(filp, iocmd, ioarg);
	 break;
      default:
	 retval = -EINVAL;
         Warning("Unexpected emulation (%u) selected\n", vmLinux->version);
	 break;
   }
   VMWare_SetVTracer(0);
   return retval;
}


