/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmciDriver.c --
 *
 *     VMCI initialization and ioctl handling.
 */

#ifdef linux
#   include "driver-config.h"
#   include <linux/string.h> /* memset() in the kernel */
#elif defined(WINNT_DDK)
#   include <string.h>
#elif !defined(__APPLE__)
#   error "Unknown platform"
#endif

#include "vmciResource.h"
#include "vmciContext.h"
#include "vmciProcess.h"
#include "vmciDatagram.h"
#include "vmciDsInt.h"
#include "vmciSharedMem.h"
#include "vmciGroup.h"
#include "vmci_infrastructure.h"
#include "vmci_defs.h"
#include "vmci_call_defs.h"
#include "vmciDriver.h"
#include "vmware.h"
#include "hostif.h"
#include "vmciCommonInt.h"

#define LGPFX "VMCI: "

#ifdef VMX86_DEBUG
static int HandleTestCall(VMCIId contextID, void* unused,
                          VMCICall *call);
static int64 HandleTestDriverCall(void *vmciObj, VMCIObjType vmciObjType,
                                  VMCICall *call);

VMCIHandle vmGroupHandle;
#endif // VMX86_DEBUG

static int  CTVectorsQueryHandler(VMCIId contextID, void* unused,
                                  VMCICall *call);

/* All contexts are members of this group handle. */
static VMCIHandle ctPublicGroupHandle;

static VMCIContext *hostContext;

/* Array of all the context hypercall functions indexed by their vector. */
VMCIHandlerEntry vmciVector[VMCI_VECTOR_MAX];

/* Array of all the process hypercall functions indexed by their vector. */
VMCIDriverCallHandler driverCallHandler[VMCI_VECTOR_MAX];

/*
 *----------------------------------------------------------------------
 *
 * VMCI_Init --
 *
 *      Initializes VMCI. This registers core hypercalls.
 *
 * Results:
 *      VMCI_SUCCESS if successful, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VMCI_Init(void)
{
   int result;

   VMCI_RegisterHandler(VMCI_VECTORS_QUERY, CTVectorsQueryHandler, 0, NULL);

   result = VMCIResource_Init();
   if (result < VMCI_SUCCESS) {
      Log(LGPFX"Failed to initialize CTResource: %d\n", result);
      goto errorExit;
   }

   result = VMCIProcess_Init();
   if (result < VMCI_SUCCESS) {
      Log(LGPFX"Failed to initialize CTProcess: %d\n", result);
      goto resourceExit;
   }

   result = VMCIContext_Init();
   if (result < VMCI_SUCCESS) {
      Log(LGPFX"Failed to initialize CTContext: %d\n", result);
      goto resourceExit;
   }

   result = VMCIDatagram_Init();
   if (result < VMCI_SUCCESS) {
      Log(LGPFX"Failed to initialize CTDatagram: %d\n", result);
      goto contextExit;
   }

   result = VMCISharedMem_Init();
   if (result < VMCI_SUCCESS) {
      Log(LGPFX"Failed to initialize CTSharedMem: %d\n", result);
      goto datagramExit;
   }

   result = VMCIContext_InitContext(VMCI_HOST_CONTEXT_ID, -1, &hostContext);
   if (result < VMCI_SUCCESS) {
      Log(LGPFX"Failed to initialize CTContext: %d\n", result);
      goto sharedMemExit;
   }

   /*
    * Give host context access to the datagram API.This must be done
    * before VMCIDs_Init(), which uses the datagram API.
    */
   VMCIDatagram_AddContext(VMCI_HOST_CONTEXT_ID);

   /* This needs to be after init. of the host context */
   if (!VMCIDs_Init()) {
      result = VMCI_ERROR_GENERIC;
      Log(LGPFX"Failed to initialize Discovery Service.\n");
      goto hostContextExit;
   }

   /* Give host context access to the shared mem API. */
   VMCISharedMem_AddContext(VMCI_HOST_CONTEXT_ID);

   /* Give host context access to the DS API. */
   VMCIDs_AddContext(VMCI_HOST_CONTEXT_ID);

#ifdef VMX86_DEBUG
   /* Create a group for VM contexts. */
   vmGroupHandle = VMCIGroup_Create();

   /* Register VM group in DS. */
   VMCIDs_Register("group.vm", vmGroupHandle, VMCI_HOST_CONTEXT_ID);

   VMCI_RegisterHandler(VMCI_TEST_CALL, HandleTestCall, 
                        0, NULL);

   VMCI_RegisterDriverHandler(VMCI_TEST_CALL, HandleTestDriverCall);
#endif // VMX86_DEBUG

   /* Create the public group handle under a well known name. */
   ctPublicGroupHandle = VMCIGroup_Create();
   VMCIDs_Register(VMCI_PUBLIC_GROUP_NAME, ctPublicGroupHandle,
                   VMCI_HOST_CONTEXT_ID);
   VMCIPublicGroup_AddContext(VMCI_HOST_CONTEXT_ID);
      
   Log(LGPFX"Driver initialized.\n");
   return VMCI_SUCCESS;

  hostContextExit:
   VMCIDatagram_RemoveContext(VMCI_HOST_CONTEXT_ID);
   VMCIContext_ReleaseContext(hostContext);
  sharedMemExit:
   VMCISharedMem_Exit();
  datagramExit:
   VMCIDatagram_Exit();
  contextExit:
   VMCIContext_Exit();
  resourceExit:
   VMCIResource_Exit();
  errorExit:
   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCI_Cleanup --
 *
 *      Cleanup the  VMCI module. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
VMCI_Cleanup(void)
{
#ifdef VMX86_DEBUG
   /* Destroy VM context group. */
   VMCIDs_Unregister("group.vm", VMCI_HOST_CONTEXT_ID);
   VMCIGroup_Destroy(vmGroupHandle);
#endif // VMX86_DEBUG
   VMCIPublicGroup_RemoveContext(hostContext->cid);
   /* Unregister & destroy the public group handle. */
   VMCIDs_Unregister(VMCI_PUBLIC_GROUP_NAME, VMCI_HOST_CONTEXT_ID);
   VMCIGroup_Destroy(ctPublicGroupHandle);

   /* Revoke host context access to SM, DS and datagram API. */
   VMCISharedMem_RemoveContext(hostContext->cid);
   VMCIDs_RemoveContext(hostContext->cid);
   VMCIDatagram_RemoveContext(hostContext->cid);

   VMCIDs_Exit();
   VMCISharedMem_Exit();
   VMCIContext_ReleaseContext(hostContext);
   VMCIDatagram_Exit();
   VMCIContext_Exit();
   VMCIResource_Exit();
}


/*
 *-------------------------------------------------------------------------
 *
 *  VMCI_GetResourceID --
 *
 *     Return resource ID. XXX should keep account to know when id is free
 *     to use again.
 *
 *  Result:
 *     Context id.
 *     
 *-------------------------------------------------------------------------
 */

VMCIId
VMCI_GetResourceID(void)
{
   static VMCIId resourceID = 0;
   return ++resourceID;
}


#ifdef VMX86_DEBUG
/*
 *-------------------------------------------------------------------------
 *
 *  HandleTestCall --
 *
 *     VMmon part of the VMCI test suite.
 *     
 *  Result:
 *     VMCI_SUCCESS on success, appropriate error code otherwise.  
 *   
 *  Side effects:
 *     None.
 *     
 *-------------------------------------------------------------------------
 */

static int 
HandleTestCall(VMCIId contextID, // IN
               void *unused,          // IN
               VMCICall *call)   // IN
{
   int result = -1;
   VMCIContext *context;
   VMCIHandle contextHandle;
   int command;

   if (call->h.size != (sizeof(VMCITestCallHdr))) {
      Log(LGPFX"Test driver call, invalid size (%d != %d)\n",
          call->h.size, (uint32)sizeof(VMCITestCallHdr));
      return -1;
   }

   context = VMCIContext_Get(VMCI_HOST_CONTEXT_ID);
   if (context == NULL) {
      Log(LGPFX"Invalid host context\n");
      return -1;
   }

   command = ((VMCITestCallHdr*)call)->command;
   contextHandle = VMCI_MAKE_HANDLE(VMCI_HOST_CONTEXT_ID, VMCI_CONTEXT_RESOURCE_ID);

   switch(command) {
   case VMCI_TEST_INIT: {
      /*
       * Add the host context to the VM group, so we can use the VM group
       * to test group access privileges. VM contexts are automatically
       * added to this group, but we need to add the host context manually,
       * since it is not normally in the VM group.
       */
      result = VMCIGroup_AddMember(vmGroupHandle, contextHandle, TRUE);
      if (result == VMCI_SUCCESS) {
         VMCIHandleArray_AppendEntry(&context->groupArray, vmGroupHandle);
      }
      break;
   }
   case VMCI_TEST_CLEANUP: {
      /*
       * Testing is finished. Remove the host context from the VM group.
       * Failure to do so will cause the VMmon module to Panic when unloaded.
       */
      result = VMCIGroup_RemoveMember(vmGroupHandle, contextHandle);
      if (result == VMCI_SUCCESS) {
         VMCIHandleArray_RemoveEntry(context->groupArray, vmGroupHandle);
      }
      break;
   }
   case VMCI_TEST_GUESTCALL: {
      // TODO: Make VMCI_TEST_CALL guestcall to caller.
      result = VMCI_SUCCESS;
      break;
   }
   default:
      Log(LGPFX"Unknown command: %d\n", command);
      result = -1;
      break;
   }

   return result;
}


/*
 *-----------------------------------------------------------------------------
 *
 * HandleTestDriverCall --
 *
 *      Host driver part of the VMCI test suite
 *
 * Results:
 *      VMCI_SUCCESS on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static int64
HandleTestDriverCall(void *vmciObj,           // IN:
                     VMCIObjType vmciObjType, // IN:
                     VMCICall *call)          // IN:
{
   /* 
    * TODO: Passthrough to hypercall handler for now. Might want
    * to implement "drivercall" tests here in the future.
    */
   return VMCI_Dispatch(VMCI_HOST_CONTEXT_ID, call);
}

#endif // VMX86_DEBUG

/*
 *-------------------------------------------------------------------------
 *
 *  VMCIPublicGroup_AddContext --
 *
 *    Adds a context to the public group handle.
 *     
 *  Result:
 *    None. 
 *   
 *  Side effects:
 *    None.
 *     
 *-------------------------------------------------------------------------
 */

void
VMCIPublicGroup_AddContext(VMCIId contextID)
{
   VMCIContext *context = VMCIContext_Get(contextID);
   if (context) {
      VMCIGroup_AddMember(ctPublicGroupHandle,
                          VMCI_MAKE_HANDLE(contextID, VMCI_CONTEXT_RESOURCE_ID),
                          TRUE);
      VMCIHandleArray_AppendEntry(&context->groupArray, ctPublicGroupHandle);
   }
}

/*
 *-------------------------------------------------------------------------
 *
 *  VMCIPublicGroup_RemoveContext --
 *
 *    Removes a context to the public group handle.
 *     
 *  Result:
 *    Returns the result from VMCIGroup_RemoveMember. 
 *   
 *  Side effects:
 *    None.
 *     
 *-------------------------------------------------------------------------
 */

int
VMCIPublicGroup_RemoveContext(VMCIId contextID)
{
   int rv = VMCI_ERROR_INVALID_ARGS;
   VMCIContext *context = VMCIContext_Get(contextID);
   if (context) {
      VMCIHandleArray_RemoveEntry(context->groupArray, ctPublicGroupHandle);
      rv = VMCIGroup_RemoveMember(ctPublicGroupHandle,
                                  VMCI_MAKE_HANDLE(contextID,
                                                 VMCI_CONTEXT_RESOURCE_ID));
   }
   return rv;
}


/*
 *-----------------------------------------------------------------------------
 *
 * CTVectorsQueryHandler --
 *
 *      Check that the vectors in the call are supported by this host.
 *
 * Results:
 *      A bitmask with bits set for the calls that are supported by this host.
 *      On error, returns a negative error code, which means that there can be
 *      no more than 31 vectors in the query structure.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static int
CTVectorsQueryHandler(VMCIId contextID, // IN
                      void* unused,          // IN
                      VMCICall *call)   // IN)
{
   VMCIVectorsMsg *msg = (VMCIVectorsMsg *)call->args;
   uint32 callSize;
   int i, result;

   if (call->h.size < sizeof(VMCIVectorsHdr)) {
      Log(LGPFX"Query: Invalid call size %d.\n", call->h.size);
      return VMCI_ERROR_INVALID_ARGS;
   }

   callSize = sizeof(VMCIVectorsHdr)
              + msg->numVectors * sizeof *msg->vectors;

   if (callSize != call->h.size) {
      Log(LGPFX"Query: Invalid call size. Is %d, should be %d.\n",
          call->h.size, callSize);
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (msg->numVectors > 31) {
      Log(LGPFX"Query: Too many vectors.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   result = 0;

   for (i = 0; i < msg->numVectors; i++) {
      uint32 vector = msg->vectors[i];
      if ((vector < VMCI_VECTOR_MAX) &&
          (vmciVector[vector].handler != NULL)) {
         result |= (1 << i);
      }
   }

   return result;
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCI_CheckGuestCalls --
 *
 *      Make sure that context supports the guestcalls we need.
 *
 * Results:
 *      TRUE if context is supported, FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

Bool
VMCI_CheckGuestCalls(VMCIContext *context)
{
   /* This part of the API does not need any guestcalls. */
   return TRUE;
}

