/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmciHashtable.c --
 *
 *     Implementation of the VMCI Hashtable. 
 *     TODO: Look into what is takes to use lib/misc/hash.c instead of 
 *     our own implementation.
 */

#ifdef linux
#   include "driver-config.h"
#   include <linux/string.h> /* memset() in the kernel */
#elif defined(WINNT_DDK)
#   include <string.h>
#elif !defined(__APPLE__)
#   error "Unknown platform"
#endif

#define LGPFX "VMCIHashTable: "

#include "vmci_kernel_defs.h"
#include "vmciHashtable.h"
#include "vmci_infrastructure.h"
#include "vmware.h"

static int HashTableUnlinkEntry(VMCIHashTable *table, VMCIHashEntry *entry);

/*
 *------------------------------------------------------------------------------
 *
 *  VMCIHashTable_Create --
 *     XXX Factor out the hashtable code to be shared amongst host and guest.
 * 
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

VMCIHashTable *
VMCIHashTable_Create(int size)
{
   VMCIHashTable *table = VMCI_AllocKernelMem(sizeof *table,
                                              VMCI_MEMORY_NONPAGED);
   if (table == NULL) {
      return NULL;
   }

   table->entries = VMCI_AllocKernelMem(sizeof *table->entries * size,
                                        VMCI_MEMORY_NONPAGED);
   if (table->entries == NULL) {
      VMCI_FreeKernelMem(table);
      return NULL;
   }
   memset(table->entries, 0, sizeof *table->entries * size);
   table->size = size;
   VMCI_InitLock(&table->lock);   

   return table;
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIHashTable_Destroy --
 *     This function should be called at module exit time.
 *     We rely on the module ref count to insure that no one is accessing any
 *     hash table entries at this point in time. Hence we should be able to just
 *     remove all entries from the hash table.
 * 
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

void
VMCIHashTable_Destroy(VMCIHashTable *table)
{
   VMCILockFlags flags;
   DEBUG_ONLY(int i;)
   DEBUG_ONLY(int leakingEntries = 0;)

   ASSERT(table);

   VMCI_GrabLock(&table->lock, &flags);
#ifdef VMX86_DEBUG
   for (i = 0; i < table->size; i++) {
      VMCIHashEntry *head = table->entries[i];
      while (head) {
	 leakingEntries++;
	 head = head->next;
      }
   }
   if (leakingEntries) {
      Log(LGPFX"Leaking %d hash table entries for table %p.\n",
	  leakingEntries, table);
   }
#endif // VMX86_DEBUG
   VMCI_FreeKernelMem(table->entries);
   table->entries = NULL;
   VMCI_ReleaseLock(&table->lock, flags);
   VMCI_FreeKernelMem(table);
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIHashTable_InitEntry --
 *     Initializes a hash entry;
 * 
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */
void
VMCIHashTable_InitEntry(VMCIHashEntry *entry,    // IN
                        VMCIHandle handle)         // IN
{
   ASSERT(entry);
   entry->handle = handle;
   entry->refCount = 0;
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIHashTable_AddEntry --
 *     XXX Factor out the hashtable code to be shared amongst host and guest.
 * 
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

int
VMCIHashTable_AddEntry(VMCIHashTable *table,  // IN
                       VMCIHashEntry *entry)  // IN
{
   int idx;
   VMCILockFlags flags;

   ASSERT(entry);

   if (VMCIHashTable_EntryExists(table, entry->handle)) {
      Log(LGPFX "Entry's handle 0x%"FMT64"x already exists.\n", entry->handle);
      return VMCI_ERROR_DUPLICATE_ENTRY;
   }

   idx = VMCI_Hash(entry->handle, table->size);
   ASSERT(idx < table->size);

   /* New entry is added to top/front of hash bucket. */
   VMCI_GrabLock(&table->lock, &flags);
   entry->refCount++;
   entry->next = table->entries[idx];
   table->entries[idx] = entry;
   VMCI_ReleaseLock(&table->lock, flags);

   return VMCI_SUCCESS;
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIHashTable_RemoveEntry --
 *     XXX Factor out the hashtable code to shared amongst API and perhaps 
 *     host and guest.
 *
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

int
VMCIHashTable_RemoveEntry(VMCIHashTable *table, // IN
                          VMCIHashEntry *entry) // IN
{
   int result;
   VMCILockFlags flags;

   ASSERT(table);
   VMCI_GrabLock(&table->lock, &flags);
   
   /* First unlink the entry. */
   result = HashTableUnlinkEntry(table, entry);
   if (result != VMCI_SUCCESS) {
      /* We failed to find the entry. */
      goto done;
   }

   /* Decrement refcount and check if this is last reference. */
   entry->refCount--;
   if (entry->refCount == 0) {
      result = VMCI_SUCCESS_ENTRY_DEAD;
      goto done;
   }
   
  done:
   VMCI_ReleaseLock(&table->lock, flags);
   
   return result;
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIHashTable_GetEntry --
 *     XXX Factor out the hashtable code to shared amongst API and perhaps 
 *     host and guest.
 *
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

VMCIHashEntry *
VMCIHashTable_GetEntry(VMCIHashTable *table,  // IN
                       VMCIHandle handle)       // IN
{
   VMCIHashEntry *cur = NULL;
   int idx;
   VMCILockFlags flags;

   if (handle == VMCI_INVALID_HANDLE) {
     return NULL;
   }

   ASSERT(table);
   idx = VMCI_Hash(handle, table->size);
   
   VMCI_GrabLock(&table->lock, &flags);
   cur = table->entries[idx];
   while (TRUE) {
      if (cur == NULL) {
	 break;
      }

      if (cur->handle == handle) {
	 cur->refCount++;
	 break;
      }
      cur = cur->next;
   }
   VMCI_ReleaseLock(&table->lock, flags);

   return cur;
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIHashTable_ReleaseEntry --
 *     XXX Factor out the hashtable code to shared amongst API and perhaps 
 *     host and guest.
 *
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

int
VMCIHashTable_ReleaseEntry(VMCIHashTable *table,  // IN
                           VMCIHashEntry *entry)  // IN
{
   VMCILockFlags flags;
   int result = VMCI_SUCCESS;

   ASSERT(table);
   VMCI_GrabLock(&table->lock, &flags);
   entry->refCount--;

   /* Check if this is last reference and report if so. */
   if (entry->refCount == 0) { 

      /*
       * Remove entry from hash table if not already removed. This could have
       * happened already because VMCIHashTable_RemoveEntry was called to unlink
       * it. We ignore if it is not found. Datagram handles will often have
       * RemoveEntry called, whereas SharedMemory regions rely on ReleaseEntry
       * to unlink the entry, since the creator does not call RemoveEntry when
       * it detaches.
       */

      HashTableUnlinkEntry(table, entry);
      result = VMCI_SUCCESS_ENTRY_DEAD;
   }
   VMCI_ReleaseLock(&table->lock, flags);

   return result;
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCIHashTable_EntryExists --
 *     XXX Factor out the hashtable code to shared amongst API and perhaps 
 *     host and guest.
 *
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

Bool
VMCIHashTable_EntryExists(VMCIHashTable *table,  // IN
                          VMCIHandle handle)       // IN
{
   VMCILockFlags flags;
   Bool exists = FALSE;
   VMCIHashEntry *entry;
   int idx;
   
   ASSERT(table);
   idx = VMCI_Hash(handle, table->size);

   VMCI_GrabLock(&table->lock, &flags);
   entry = table->entries[idx];
   while (entry) {
      if (entry->handle == handle) {
	 exists = TRUE;
	 break;
      }
      entry = entry->next;
   }
   VMCI_ReleaseLock(&table->lock, flags);
   return exists;
}


/*
 *------------------------------------------------------------------------------
 *
 *  HashTableUnlinkEntry --
 *     XXX Factor out the hashtable code to shared amongst API and perhaps 
 *     host and guest.
 *     Assumes caller holds table lock.
 *
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static int
HashTableUnlinkEntry(VMCIHashTable *table, // IN
                     VMCIHashEntry *entry) // IN 
{
   int result;
   VMCIHashEntry *prev, *cur;
   int idx;

   idx = VMCI_Hash(entry->handle, table->size);

   prev = NULL;
   cur = table->entries[idx];
   while (TRUE) {
      if (cur == NULL) {
	 result = VMCI_ERROR_NOT_FOUND;
	 break;
      }
      if (cur->handle == entry->handle) {
	 ASSERT(cur == entry);

	 /* Remove entry and break. */
	 if (prev) {
	    prev->next = cur->next;
	 } else {
	    table->entries[idx] = cur->next;
	 }
	 cur->next = NULL;
	 result = VMCI_SUCCESS;
	 break;
      }
      prev = cur;
      cur = cur->next;
   }
   return result;
}
