/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmciSharedMem.c --
 *
 *    This file implements the VMCI Shared Memory API on the host. 
 *    
 *    XXX: This is currently part of vmmon but should probably be moved into
 *         a separate host driver.
 *
 */

#ifdef linux
#   include "driver-config.h"
#   include <linux/string.h> /* memset() in the kernel */
#elif defined(WINNT_DDK)
#   include <string.h>
#elif !defined(__APPLE__)
#   error "Unknown platform"
#endif

#include "vmci_kernel_defs.h"
#include "vmci_defs.h"
#include "vmciContext.h"
#include "vmci_infrastructure.h"
#include "vmciSharedMem.h"
#include "vmciGroup.h"
#include "vmciResource.h"
#include "vmciDsInt.h"
#include "vmware.h"
#include "vmciCommonInt.h"

#define LGPFX "VMCISharedMem: "

typedef struct SharedMemEntry {
   VMCIResource    resource;
   char          pageFileName[VMCI_PATH_MAX];
   uint32        size;
   VMCIHandle      handle;
} SharedMemEntry;

static void SharedMemFreeCB(void *resource);
static SharedMemEntry *SharedMemGetEntry(VMCIHandle handle);
static int SharedMemReleaseEntry(SharedMemEntry *entry);
static int SharedMemCreate(VMCIContext *context, 
                           VMCIProcess *process, const char* pageFileName,
                           uint32 size, VMCIHandle *handle);
static int SharedMemAttach(VMCIContext *context,
                           VMCIProcess *process, VMCIHandle handle,
                           char* pageFileName, uint32 *size);
static int SharedMemQuery(VMCIContext *context,
                          VMCIProcess *process, VMCIHandle handle,
                          char* pageFileName, uint32 *size);

#ifdef VMX86_DEVEL
static void DumpSharedMemState(VMCIContext *context,
                               VMCIProcess *process);
#endif

/* Struct used to represent the SharedMem API. */
static struct SharedMemAPI {
   VMCIResource resource; 
   VMCIHandle resourceHandle;
   VMCIHandle groupHandle;
} sharedMemAPI;


/*
 *-------------------------------------------------------------------------
 *
 * VMCISharedMem_Init --
 *
 *     Initialize Simple Shared Memory API.
 *
 * Result:
 *     None.
 *
 * Side effects:
 *     None.
 *
 *-------------------------------------------------------------------------
 */

int
VMCISharedMem_Init(void)
{
   int result;
   VMCIResourcePrivilegeType priv = VMCI_PRIV_SM_CREATE;

   sharedMemAPI.resourceHandle =
      VMCI_MAKE_HANDLE(VMCI_HOST_CONTEXT_ID, VMCIResource_GetID());

   /* Resource used to represent the sharedmem API. */
   result = VMCIResource_Add(&sharedMemAPI.resource, VMCI_RESOURCE_TYPE_API,
                             sharedMemAPI.resourceHandle,
                             VMCI_MAKE_HANDLE(VMCI_HOST_CONTEXT_ID,
                                            VMCI_CONTEXT_RESOURCE_ID),
                             1, &priv, NULL, NULL);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"Failed adding SharedMem API resource.\n");
      return result;
   }

   /*
    * Create a sharedmem api group.By adding this group as a client to the 
    * sharedMemAPI.resource with the VMCI_PRIV_SM_CREATE privilege we can
    * give contexts access to the sharedmem api by making them members of
    * this group.
    */
   sharedMemAPI.groupHandle = VMCIGroup_Create();
   if (sharedMemAPI.groupHandle == VMCI_INVALID_HANDLE) {
      Log(LGPFX"Failed creating SharedMem API group.\n");
      VMCIResource_Remove(sharedMemAPI.resourceHandle, VMCI_RESOURCE_TYPE_API);
      return VMCI_ERROR_NO_HANDLE;
   }

   /* Add group as client of sharedMem API with the right privilege. */
   result = VMCIResource_AddClientPrivileges(sharedMemAPI.resourceHandle,
                                             sharedMemAPI.groupHandle,
                                             1, &priv, 0, NULL);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"Failed making SharedMem API group a client of SharedMem API "
	  "resource.\n");
      VMCIGroup_Destroy(sharedMemAPI.groupHandle);
      VMCIResource_Remove(sharedMemAPI.resourceHandle, VMCI_RESOURCE_TYPE_API);
      return result;
   }

   return VMCI_SUCCESS;
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCISharedMem_Exit --
 *
 *     Cleanup SharedMem API.
 *
 *  Result:
 *     None.
 *     
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

void
VMCISharedMem_Exit(void)
{
   /* Remove group as client of sharedMemAPI resource. */
   VMCIResource_RemoveAllClientPrivileges(sharedMemAPI.resourceHandle,
                                          sharedMemAPI.groupHandle);
   
   VMCIGroup_Destroy(sharedMemAPI.groupHandle);

   VMCIResource_Remove(sharedMemAPI.resourceHandle, VMCI_RESOURCE_TYPE_API);
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCISharedMem_AddContext --
 *
 *     Adds the context as a member of the shared mem API group. This makes it 
 *     possible for the context to use the shared mem API.
 *
 *  Result:
 *     None.
 *     
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

void
VMCISharedMem_AddContext(VMCIId contextID)
{
   VMCIContext *context = VMCIContext_Get(contextID);
   if (context != NULL) {
      VMCIGroup_AddMember(sharedMemAPI.groupHandle,
                          VMCI_MAKE_HANDLE(contextID, VMCI_CONTEXT_RESOURCE_ID),
                          FALSE);
      VMCIHandleArray_AppendEntry(&context->groupArray, sharedMemAPI.groupHandle);
   }
}


/*
 *------------------------------------------------------------------------------
 *
 *  VMCISharedMem_RemoveContext --
 *
 *     Removes the context as a member of the SharedMem API, disallowing the 
 *     context access to the SharedMem API functions.
 *
 *  Result:
 *     None.
 *     
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------------------
 */

void
VMCISharedMem_RemoveContext(VMCIId contextID)
{
   VMCIContext *context = VMCIContext_Get(contextID);
   if (context != NULL) {
      VMCIHandleArray_RemoveEntry(context->groupArray, sharedMemAPI.groupHandle);
      VMCIGroup_RemoveMember(sharedMemAPI.groupHandle,
                             VMCI_MAKE_HANDLE(contextID,
                                            VMCI_CONTEXT_RESOURCE_ID));
   }
}


/*
 *------------------------------------------------------------------------------
 *
 *  SharedMemFreeCB --
 *     Callback to free sharedmem structure when resource is no longer used,
 *     ie. the reference count reached 0.
 * 
 *  Result:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static void
SharedMemFreeCB(void *resource)
{
   SharedMemEntry *entry;
   ASSERT(resource);

   entry = RESOURCE_CONTAINER(resource, SharedMemEntry, resource);
   if (entry) {
      VMCIResource *entryResource = &entry->resource;
      if (entryResource->registrationCount) {
         /* Remove all discovery service registrations for this resource. */
         VMCIDs_UnregisterResource(entryResource);
      }
      ASSERT(!entryResource->registrationCount);
   }
   VMCI_FreeKernelMem(entry);
}


/*
 *----------------------------------------------------------------------
 *
 * SharedMemGetEntry --
 *
 *     Look up Vmci shared memory resource handle. The resource's
 *     refCount is incremented if successfully found.
 *
 * Results:
 *      Shared memory entry on success, NULL if the handle is not found.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

static SharedMemEntry *
SharedMemGetEntry(VMCIHandle handle)     // IN
{
   VMCIResource *resource = VMCIResource_Get(handle, VMCI_RESOURCE_TYPE_SHAREDMEM);

   if (resource == NULL) {
      Log(LGPFX"Cannot find resource with handle 0x%"FMT64"x.\n", handle);
      return NULL;
   }
   return RESOURCE_CONTAINER(resource, SharedMemEntry, resource);
}


/*
 *----------------------------------------------------------------------
 *
 * SharedMemReleaseEntry --
 *
 *     Releases a sharedmem entry. This decrements the ref count and
 *     the SharedMemFreeCB could be called as a side effect.
 *
 * Results:
 *      void.
 *
 * Side effects:
 *      Ref count is decremented and entry could be freed.
 *
 *---------------------------------------------------------------------- 
 */

static int
SharedMemReleaseEntry(SharedMemEntry *entry)     // IN
{
   return VMCIResource_Release(&entry->resource);
}


/*
 *-------------------------------------------------------------------------
 *
 * SharedMemCreate --
 *
 *     Registers creation of a shared memory region.
 *
 * Results:
 *      VMCI_SUCCESS on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *-------------------------------------------------------------------------
 */

static int 
SharedMemCreate(VMCIContext *context,     // IN
                VMCIProcess *process,     // IN
                const char* pageFileName, // IN
                uint32 size,              // IN
                VMCIHandle *hnd)            // IN/OUT
{
   int result;
   VMCIHandle handle;
   VMCIHandle contextHnd;
   VMCIHandleArray **smArrayPtr;
   SharedMemEntry *entry;
   VMCIResourcePrivilegeType validPrivs[1] = {VMCI_PRIV_SM_ATTACH};

   ASSERT(context != NULL && hnd != NULL);
   contextHnd = VMCI_MAKE_HANDLE(context->cid, VMCI_CONTEXT_RESOURCE_ID);

   result =
       VMCIResource_CheckClientPrivilege(sharedMemAPI.resourceHandle, 
				       contextHnd, VMCI_PRIV_SM_CREATE);
   if (result != VMCI_SUCCESS_ACCESS_GRANTED) {
      Log(LGPFX "Create: Context 0x%x does not have the privilege to create "
	  "shared memory regions.\n", context->cid);
      return result;
   }

   if (pageFileName == NULL) {
      Log(LGPFX "Invalid page file name.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   /* 
    * If called from a context the caller must supply a valid handle which
    * matches the context id making the call. If called from a process
    * the handle will be created here. 
    */
   if (process) {
      ASSERT(context->cid == VMCI_HOST_CONTEXT_ID);

      handle = VMCI_MAKE_HANDLE(VMCI_HOST_CONTEXT_ID, VMCIResource_GetID());
      smArrayPtr = &process->sharedMemHandles;
   } else {
      if (*hnd == VMCI_INVALID_HANDLE ||
	  context->cid != VMCI_HANDLE_TO_CONTEXT_ID(*hnd)) {
	 Log(LGPFX "Invalid handle 0x%"FMT64"x for context 0x%x.\n", 
	     *hnd, context->cid);
	 return VMCI_ERROR_INVALID_ARGS;
      }
      handle = *hnd;
      smArrayPtr = &context->sharedMemArray;
   }

   if (VMCIHandleArray_HasEntry(*smArrayPtr, handle)) {
      Log(LGPFX "Shared memory region 0x%"FMT64"x already exists.\n", handle);
      return VMCI_ERROR_REGION_ALREADY_SHARED;
   }

   entry = VMCI_AllocKernelMem(sizeof *entry, VMCI_MEMORY_NONPAGED);
   if (!entry) {
      Log(LGPFX "Cannot allocate memory for shared memory state.\n");
      return VMCI_ERROR_NO_MEM;
   }

   memcpy(entry->pageFileName, pageFileName, sizeof entry->pageFileName);
   entry->size = size;
   entry->handle = handle;

   /* 
    * Place new sm region's handle in either process or context sm handle array.
    */
   VMCIHandleArray_AppendEntry(smArrayPtr, handle);

   /* Add sm resource to resource table. This also sets the ref count to 1.*/
   result = VMCIResource_Add(&entry->resource, VMCI_RESOURCE_TYPE_SHAREDMEM,
                             handle, contextHnd,
                             1, validPrivs, SharedMemFreeCB, entry);
   if (result != VMCI_SUCCESS) {
      VMCIHandleArray_RemoveEntry(*smArrayPtr, handle);
      VMCI_FreeKernelMem(entry);
      return result;
   }
   
   *hnd = handle;
   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * SharedMemQuery --
 *
 *      Returns information about shared memory region. 
 *
 * Results:
 *      VMCI_SUCCESS on success, error code otherwise.
 *      Returns entry, page file name and number of pages for the 
 *      shared memory region corresponding to the given handle.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int 
SharedMemQuery(VMCIContext *context,  // IN:
               VMCIProcess *process,  // IN:
               VMCIHandle handle,     // IN: VMCI handle 
               char *pageFileName,    // OUT: page file backing region
               uint32 *size)          // OUT: size of shared region in bytes
{
   SharedMemEntry *entry;
   VMCIHandle contextHnd;
   Bool queryAllowed;

   ASSERT(context != NULL);
   ASSERT(pageFileName != NULL);
   ASSERT(size != NULL);

   contextHnd = VMCI_MAKE_HANDLE(context->cid, VMCI_CONTEXT_RESOURCE_ID);

   /* 
    * First check access privileges. We currently allow queries for resources 
    * either owned by the querying context or when a context has the 
    * VMCI_PRIV_SM_ATTACH privilege to the resource.
    */
   queryAllowed = context->cid == VMCI_HANDLE_TO_CONTEXT_ID(handle) ||
      VMCIResource_CheckClientPrivilege(handle, contextHnd, VMCI_PRIV_SM_ATTACH) ==
      VMCI_SUCCESS_ACCESS_GRANTED;

   if (!queryAllowed) {
      Log(LGPFX "Query: Context 0x%x does not have the privilege to query "
	  "shared memory region 0x%"FMT64"x.\n", context->cid, handle);
      return VMCI_ERROR_NO_ACCESS;
   }

   /* Get hold of entry. This also increments its ref count if entry is found. */
   entry = SharedMemGetEntry(handle);
   if (entry == NULL) {
      return VMCI_ERROR_NOT_FOUND;
   }

   /* Copy data out for caller. */
   memcpy(pageFileName, entry->pageFileName, VMCI_PATH_MAX);
   *size = entry->size;

   /* Release entry and decrement ref count. */
   SharedMemReleaseEntry(entry);

   return VMCI_SUCCESS;   
}


/*
 *----------------------------------------------------------------------
 *
 * SharedMemAttach --
 *
 *      Attach to existing shared memory region. Also returns 
 *      information about shared memory region.
 *
 * Results:
 *      VMCI_SUCCESS on success, error code otherwise.
 *      Returns page file name and number of pages for the 
 *      shared memory region corresponding to the given handle.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int 
SharedMemAttach(VMCIContext *context,  // IN:
                VMCIProcess *process,  // IN
                VMCIHandle handle,       // IN: handle we want to attach to
                char *pageFileName,    // OUT: page file backing region
                uint32 *size)          // OUT: size of shared region in bytes
{
   SharedMemEntry *entry;
   int result;
   VMCIHandleArray **smArrayPtr;
   VMCIHandle contextHnd;

   ASSERT(context != NULL);
   ASSERT(pageFileName != NULL);
   ASSERT(size != NULL);
   contextHnd = VMCI_MAKE_HANDLE(context->cid, VMCI_CONTEXT_RESOURCE_ID);

   /* First check access privileges. */
   result = 
      VMCIResource_CheckClientPrivilege(handle, contextHnd, VMCI_PRIV_SM_ATTACH);
   if (result != VMCI_SUCCESS_ACCESS_GRANTED) {
      Log(LGPFX "Attach: Context 0x%x does not have the privilege to attach "
	  "to shared memory region 0x%"FMT64"x.\n", context->cid, handle);
      return result;

   }

   smArrayPtr = process ? &process->sharedMemHandles : &context->sharedMemArray;
   if (VMCIHandleArray_HasEntry(*smArrayPtr, handle)) {
      Log(LGPFX "%s has already attached to region %"FMT64"d\n", 
	  process ? "Process" : "Context", handle);
      return VMCI_ERROR_REGION_ALREADY_SHARED;
   }

   /* Get hold of entry. This also increments its ref count if entry is found. */
   entry = SharedMemGetEntry(handle);
   if (entry == NULL) {
      return VMCI_ERROR_NOT_FOUND;
   }

   /* Place handle into either context's or process' sm region array. */
   VMCIHandleArray_AppendEntry(smArrayPtr, handle);

   /* Copy region's data out to caller. */
   memcpy(pageFileName, entry->pageFileName, VMCI_PATH_MAX);
   *size = entry->size;

   /* When detaching we decrement the ref count that was inc'ed here. */
   return VMCI_SUCCESS;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCISharedMem_Detach --
 *
 *      Detach from shared memory region.
 *
 * Results:
 *      VMCI_SUCCESS on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int 
VMCISharedMem_Detach(VMCIContext *context,  // IN
                     VMCIProcess *process,  // IN
                     VMCIHandle handle)       // IN
{
   SharedMemEntry *entry;
   VMCIHandleArray *smArray;

   ASSERT(context != NULL);

   smArray = process ? process->sharedMemHandles: context->sharedMemArray;
   if (!VMCIHandleArray_HasEntry(smArray, handle)) {
      Log(LGPFX "%s not attached to shared region %"FMT64"d\n",
	  process ? "Process" : "Context", handle);
      return VMCI_ERROR_NO_ACCESS;
   }

   entry = SharedMemGetEntry(handle);
   if (entry == NULL) {
      Log(LGPFX "Shared memory region does not exist.\n");
      return VMCI_ERROR_NOT_FOUND;
   }

   /* Remove handle from either context's or process' sm array. */ 
   VMCIHandleArray_RemoveEntry(smArray, handle);
  
   /* 
    * Release the entry twice. Once for the Get above and once for the previous
    * attach. The entry might be freed if this is the last reference.
    */
   SharedMemReleaseEntry(entry);
   return SharedMemReleaseEntry(entry) == VMCI_SUCCESS_ENTRY_DEAD ? 
                                          VMCI_SUCCESS_LAST_DETACH : VMCI_SUCCESS;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCISharedMem_IOCTLHandler --
 *
 *     Handles requests to create, attach, detach and query shared
 *     memory regions.
 *     
 * Results:
 *      VMCI_SUCCESS on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

int
VMCISharedMem_IOCTLHandler(VMCIId contextID,             // IN
                           VMCIProcess *process,         // IN
                           unsigned int iocmd,           // IN
                           VMCISharedMemInfo *memInfo)   // OUT
{
   int result = VMCI_SUCCESS;
   VMCIContext *context;

   context = VMCIContext_Get(contextID);
   if (context == NULL) {
      Log(LGPFX "No context with id 0x%x.\n", contextID);
      return VMCI_ERROR_INVALID_ARGS;
   }

   switch (iocmd) { 
   case IOCTLCMD_VMCI_SHAREDMEM_CREATE:
      Log(LGPFX "SharedMemCreate: handle = %"FMT64"d, "
          "size = %d, name = %s\n", memInfo->handle, memInfo->size, 
          memInfo->pageFileName);
      result = SharedMemCreate(context, process, memInfo->pageFileName,
                               memInfo->size, &memInfo->handle);
      break;

   case IOCTLCMD_VMCI_SHAREDMEM_ATTACH:
      result = SharedMemAttach(context, process, memInfo->handle,
                               memInfo->pageFileName, &memInfo->size);
      break;

   case IOCTLCMD_VMCI_SHAREDMEM_QUERY:
      result = SharedMemQuery(context, process, memInfo->handle,
                              memInfo->pageFileName, &memInfo->size);
      break;

   case IOCTLCMD_VMCI_SHAREDMEM_DETACH:
      result = VMCISharedMem_Detach(context, process, memInfo->handle);
      break;

   default:
      NOT_REACHED();
      result = VMCI_ERROR_INVALID_ARGS;  
      break;
   }

   memInfo->result = result;

#ifdef VMX86_DEVEL
   DumpSharedMemState(context, process);
#endif

   return result;
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCISharedMem_CheckGuestCalls --
 *
 *      Make sure that context supports the guestcalls we need.
 *
 * Results:
 *      TRUE if context is supported, FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

Bool
VMCISharedMem_CheckGuestCalls(VMCIContext *context)
{
   /* This part of the API does not need any guestcalls. */
   return TRUE;
}


#ifdef VMX86_DEVEL

/*
 *----------------------------------------------------------------------
 *
 * DumpSharedMemState --
 *
 *     Logs information about shared memory state.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *---------------------------------------------------------------------- 
 */

static void 
DumpSharedMemState(VMCIContext *context,  // IN
                   VMCIProcess *process)  // IN
{
   int i;
   VMCIHandleArray *smArray;

   ASSERT(context != NULL);
   smArray = process ? process->sharedMemHandles : context->sharedMemArray;

   Log(LGPFX "Dumping shared memory state\n");

   
   for (i = 0; i < VMCIHandleArray_GetSize(smArray); i++) {
      VMCIHandle handle = VMCIHandleArray_GetEntry(smArray, i);
      SharedMemEntry *entry = SharedMemGetEntry(handle);
      if (entry != NULL) {
	 Log(LGPFX "Handle %"FMT64"d, name %s, size = %d\n",
	     entry->handle, entry->pageFileName, entry->size);
	 if (handle != entry->handle) {
	    Log(LGPFX "Entry handle 0x%"FMT64"x mismatch with array handle "
		"%"FMT64"d.\n", entry->handle, handle);
	 }
	 SharedMemReleaseEntry(entry);
      } else {
	 Log(LGPFX "No entry for handle %"FMT64"d.\n", handle);
      }
   }
}
#endif

