/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmci_infrastructure.h -- 
 * 
 *      This file implements the VMCI infrastructure.
 *      
 *      XXX VMCI_TODO: This should probably be moved to a better
 *                     location in a C file accessible in vmmon,
 *                     vmkernel and the guest driver.
 *
 */ 
 
#ifndef _VMCI_INFRASTRUCTURE_H_
#define _VMCI_INFRASTRUCTURE_H_

#include "vmware.h"
#include "vmci_defs.h"
#include "vmci_call_defs.h"

typedef int (*VMCICallHandler)(VMCIId contextID, void *clientData, 
                               VMCICall *call);
#define VMCI_IMMEDIATE_DISPATCH  0x1

typedef struct VMCIHandlerEntry {
   VMCICallHandler handler;
   void *clientData;
} VMCIHandlerEntry;

typedef enum {
   VMCIOBJ_VMX_VM = 10,
   VMCIOBJ_CONTEXT,
   VMCIOBJ_PROCESS,
   VMCIOBJ_DATAGRAM_PROCESS,
   VMCIOBJ_NOT_SET,
} VMCIObjType;

/* XXX Merge hypercall and driver call handlers. */
typedef int64 (*VMCIDriverCallHandler)(void *vmciObj,
                                       VMCIObjType vmciObjType,
                                       VMCICall *call);

/* Guestcalls currently support a maximum of 8 uint64 arguments. */
#define VMCI_GUESTCALL_MAX_ARGS_SIZE 64

EXTERN VMCIHandlerEntry vmciVector[VMCI_VECTOR_MAX];

EXTERN VMCIDriverCallHandler driverCallHandler[VMCI_VECTOR_MAX];

/*
 *-------------------------------------------------------------------------
 *
 *  VMCI_RegisterHandler --
 *
 *     Function used to register a command and the associated 
 *     function for handling hypercalls or guestcalls with the given 
 *     vector.
 *
 *  Result:
 *     None.
 *     
 *  Side effects:
 *     None.
 *
 *-------------------------------------------------------------------------
 */

static INLINE void
VMCI_RegisterHandler(VMCI_Vector vector,        // IN
                     VMCICallHandler handler,   // IN
                     unsigned int flags,        // IN
                     void *clientData)          // IN
{
   VMCIHandlerEntry *entry;

   ASSERT(vector < VMCI_VECTOR_MAX);
   
   entry = &vmciVector[vector];
   entry->handler = handler;
   entry->clientData = clientData;
}


/*
 *-------------------------------------------------------------------------
 *
 *  VMCI_Dispatch --
 *
 *     Function used by device or device driver to call into the VMCI 
 *     framework code when it has received a hypercall or guestcall.
 *
 *  Result:
 *     Return value of function in data is returned to other side,
 *     guest or host.
 *     
 *  Side effects:
 *     None.
 *
 *-------------------------------------------------------------------------
 */

static INLINE int
VMCI_Dispatch(VMCIId contextID,  // IN: identifies caller
              VMCICall *call)    // IN: The call data
{
   VMCIHandlerEntry *entry;

   if (call == NULL) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   if (call->h.vector >= VMCI_VECTOR_MAX) {
      return VMCI_ERROR_INVALID_VECTOR;
   }

   entry = &vmciVector[call->h.vector];
   if (entry->handler == NULL) {
      return VMCI_ERROR_INVALID_VECTOR;
   }

   return entry->handler(contextID, entry->clientData, call);
}


/*
 *-------------------------------------------------------------------------
 *
 *  VMCI_GuestCallGetSize --
 *
 *     Returns the size of the guest call including the size of the argument
 *     buffer rounded up to a multiple of 8.
 *
 *  Result:
 *     Returns guest call size.
 *     
 *  Side effects:
 *     None.
 *
 *-------------------------------------------------------------------------
 */

static INLINE unsigned
VMCI_GuestCallGetSize(VMCICall *call)
{
   return ((call->h.size + 7) & 0xfffffff8);
}


/*
 *-------------------------------------------------------------------------
 *
 *  VMCI_Hash --
 *
 *     Hash function used by the Simple Datagram API. Based on the djb2 
 *     hash function by Dan Bernstein.
 * 
 *  Result:
 *     Returns guest call size.
 *     
 *  Side effects:
 *     None.
 *
 *-------------------------------------------------------------------------
 */

static INLINE int
VMCI_Hash(VMCIHandle handle, // IN
          unsigned size)     // IN
{
   int i;
   int hash = 5381;

   for (i = 0; i < sizeof handle; i++) {
      hash = ((hash << 5) + hash) + (uint8)(handle >> (i*8));
   }
   return hash & (size -1);
}

 
/*
 *-------------------------------------------------------------------------
 *
 *  VMCI_RegisterDriverHandler --
 *
 *     Function used to register a handler for a driver call.
 *
 *  Result:
 *     None.
 *
 *  Side effects:
 *     None.
 *
 *-------------------------------------------------------------------------
 */

static INLINE void
VMCI_RegisterDriverHandler(VMCI_Vector vector,             // IN
                           VMCIDriverCallHandler handler)  // IN
{
   ASSERT(vector < VMCI_VECTOR_MAX);
   
   driverCallHandler[vector] = handler;
}


/*
 *-------------------------------------------------------------------------
 *
 *  VMCI_DispatchDriverCall --
 *
 *     Function used by device or device driver to call into the VMCI 
 *     framework code when it has received a driver call.
 *
 *  Result:
 *     Return value of function in data is returned to other side,
 *     guest or host.
 *     
 *  Side effects:
 *     None.
 *
 *-------------------------------------------------------------------------
 */

static INLINE int64
VMCI_DispatchDriverCall(void *vmciObj,           // IN: identifies caller
                        VMCIObjType vmciObjType, // IN: type of object
                        VMCICall *driverCall)    // IN: 
{
   VMCIDriverCallHandler handler;
   ASSERT(driverCall);

   if (driverCall->h.vector >= VMCI_VECTOR_MAX) {
      return VMCI_ERROR_INVALID_VECTOR;
   }

   if (driverCall->h.size < VMCI_CALL_HEADERSIZE) {
      return VMCI_ERROR_INVALID_ARGS;
   }
   
   handler = driverCallHandler[driverCall->h.vector];
   if (handler == NULL) {
      return VMCI_ERROR_INVALID_VECTOR;
   }

   return handler(vmciObj, vmciObjType, driverCall);
}


#endif // _VMCI_INFRASTRUCTURE_H_
