/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriverDestructorOld4Gb --
 *
 *    Destructor for a range --hpreg
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static void
LinuxDriverDestructorOld4Gb(MemRange *that) // IN
{
   struct page *page;
   struct page const *pageEnd;
   int log2NumPages;

   ASSERT(that);

   /*
    * We must unreserve the pages first otherwise free_pages() will refuse to
    * do its job --hpreg
    */
   page = virt_to_page(that->kAddr);
   pageEnd = page + that->numPages;
   for (; page < pageEnd; page++) {
      compat_mem_map_unreserve(page);
   }

   log2NumPages = compat_get_order(that->numPages << PAGE_SHIFT);
   free_pages(that->kAddr, log2NumPages);

   kfree(that);
}


static void
LinuxDriverFreeOld4Gb(void)
{
   MemRange *head;
   MemRange *first;

   for (head = &linuxState.oldList4Gb; (first = head->next) != head; ) {
      /* Unlink the first item at the beginning of the list --hpreg */
      head->next = first->next;
      LinuxDriverDestructorOld4Gb(first);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_InitV2  --
 *
 *      called on open of /dev/vmmon or /dev/vmx86.$USER. Use count used
 *      to determine eventual deallocation of the module
 *
 * Side effects:
 *     Increment use count used to determine eventual deallocation of
 *     the module
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriver_InitV2(struct file* filp, VMLinux *vmLinux)
{
   VMDriver *vm;
   
   vm = vmLinux->vm;
   if (!vm) {
      vm = Vmx86_CreateVM((void *)filp, (uintptr_t)(current->pid), VME_V2);
      if (vm == NULL) {
         Warning("Vmx86_CreateVM failed\n");
         return -ENOMEM;
      }
      vmLinux->vm = vm;
   }
   return 0;
}


/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriverIoctlAlloc4Gb --
 *
 *    Allocate and lock 'numPages' pages of memory whose physical addresses
 *    are contiguous and < 4 GB --hpreg
 *
 * Results:
 *    0 on success: '*addr' is the physical address of the beginning of the
 *                  range
 *    < 0 on failure: the actual value determines the type of failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static int
LinuxDriverIoctlAlloc4Gb(unsigned long numPages, // IN
                         PA32 *addr)             // OUT
{
   MemRange *range;
   int log2NumPages;
   VA kAddr;
   struct page *page;
   struct page const *pageEnd;
 
   ASSERT(addr);

   if (numPages == 0) {
      return -EINVAL;
   }

   range = kmalloc(sizeof(*range), GFP_KERNEL);
   if (range == NULL) {
      return -ENOMEM;
   }

   log2NumPages = compat_get_order(numPages << PAGE_SHIFT);
   /*
    * We use the ISA DMA zone (i.e. pages whose physical addresses are < 16 MB)
    * by lack of better choice. We should really use GFP_32BIT or analog when
    * it becomes available, but do not do that in this branch, as there is code
    * to fix bug 16306 that relies the ISA DMA zone --hpreg
    */
   kAddr = __get_dma_pages(GFP_KERNEL, log2NumPages);
   if (kAddr == 0) {
      kfree(range);

      return -ENOMEM;
   }

   /*
    * We must reserve the pages otherwise remap_page_range() will refuse to do
    * its job --hpreg
    */
   page = virt_to_page(kAddr);
   pageEnd = page + numPages;
   for (; page < pageEnd; page++) {
      compat_mem_map_reserve(page);
   }

   range->kAddr = kAddr;
   range->pAddr = virt_to_phys((void *)kAddr);
   *addr = range->pAddr;
   range->numPages = numPages;

   /* Insert the range at the beginning of the list --hpreg */
   spin_lock(&linuxState.oldLock4Gb);
   range->next = linuxState.oldList4Gb.next;
   linuxState.oldList4Gb.next = range;
   spin_unlock(&linuxState.oldLock4Gb);

   return 0;
}


static INLINE void LinuxDriverUserPageLocked(VMDriver* vm, MPN mpn) {
   if (PAGE_LOCK_SUCCESS(mpn)) {
      vm->stats.userLockedPages++;
      if (vm->stats.maxUserLockedPages > vm->stats.userLockedPages) {
         vm->stats.maxUserLockedPages = vm->stats.userLockedPages;
      }
   }
}


static INLINE void LinuxDriverUserPageUnlocked(VMDriver* vm, MPN mpn) {
   if (PAGE_LOCK_SUCCESS(mpn)) {
      vm->stats.userLockedPages--;
   }
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriverEmulatedVersion --
 *
 *      retrieve default vmLinux version
 *
 * Results:
 *      int
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int
LinuxDriverEmulatedVersion(void)
{
   VMLinux *vml;

   HostIF_GlobalLock(12);

   for (vml = linuxState.head; vml; vml = vml->next) {
      if (vml->version != VME_UNKNOWN) {
         break;
      }
   }

   HostIF_GlobalUnlock(12);

   if (vml) {
      return vml->version;
   }
   return VME_UNKNOWN;
}


/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriverDestructorLocked --
 *
 *    Deallocate all locked memory.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static void
LinuxDriverDestructorLocked(VMLinux *vmLinux) // IN
{
   unsigned int pg;

   if (!vmLinux->sizeLocked) {
      return;
   }
   for (pg = (vmLinux->sizeLocked + VMHOST_MAPPING_PT - 1) / VMHOST_MAPPING_PT; pg-- > 0; ) {
      __free_page(vmLinux->pagesLocked->ent[pg]);
   }
   HostIF_FreePage(vmLinux->pagesLocked);
   vmLinux->sizeLocked = 0;
}


/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriverLockedNoPage --
 *
 *      Callback for returning allocated page for memory mapping
 *
 * Results:
 *
 *      Page or page address on success, NULL or 0 on failure.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

#ifdef VMW_NOPAGE_261
static struct page *LinuxDriverLockedNoPage(struct vm_area_struct *vma, //IN
				            unsigned long address,      //IN
				            int *type)		        //OUT: Fault type
#elif defined(KERNEL_2_4_0)
static struct page *LinuxDriverLockedNoPage(struct vm_area_struct *vma, //IN
				            unsigned long address,      //IN
				            int unused)		        //nothing
#else
static unsigned long LinuxDriverLockedNoPage(struct vm_area_struct *vma, //IN
				             unsigned long address,      //IN
				             int unused)	        //nothing
#endif
{
   VMLinux *vmLinux = (VMLinux *) vma->vm_file->private_data;
   unsigned long pg;
   struct page* pgt;
   struct VMHostEntry* vmhe;
   struct page* result;
	
   pg = ((address - vma->vm_start) >> PAGE_SHIFT) + compat_vm_pgoff(vma);
   if (pg >= vmLinux->sizeLocked) {
      printk(KERN_DEBUG "vmmon: Something went wrong: entry %08lX out of range (>=%08X) for mapping on filp %p\n", pg, vmLinux->sizeLocked, vmLinux);
      return NOPAGE_SIGBUS;
   }
   if (!vmLinux->vm || !vmLinux->vm->vmhost) {
      printk(KERN_DEBUG "vmmon: Something went wrong: no vm or vmhost for mapping on filp %p\n", vmLinux);
      return NOPAGE_SIGBUS;
   }
   pgt = vmLinux->pagesLocked->ent[pg / VMHOST_MAPPING_PT];
   if (!pgt) {
      printk(KERN_DEBUG "vmmon: Something went wrong: missing entry %08lX from mapping on filp %p\n", pg, vmLinux);
      return NOPAGE_SIGBUS;
   }
   vmhe = kmap(pgt);
   result = vmhe->ent[pg % VMHOST_MAPPING_PT];
   kunmap(pgt);
   if (!result) {
      printk(KERN_DEBUG "vmmon: Something went wrong: attempt to access non-existing entry %08lX in mapping on filp %p\n", pg, vmLinux);
      return NOPAGE_SIGBUS;
   }
   if (!PhysTrack_Test(vmLinux->vm->vmhost->AWEPages, page_to_pfn(result))) {
      printk(KERN_DEBUG "vmmon: MPN %08lX not tracked! Someone released it before removing it from VA first!\n", pg);
      return NOPAGE_SIGBUS;
   }
   get_page(result);
#ifdef KERNEL_2_4_0
#ifdef VMW_NOPAGE_261
   *type = VM_FAULT_MINOR;
#endif
   return result;
#else
   return page_address(result);
#endif
}


/*
 *-----------------------------------------------------------------------------
 *
 * LinuxDriverAllocLockedMem --
 *
 *      Create mapping for locked memory areas.
 *
 * Results:
 *
 *      0       on success,
 *      -EINVAL on invalid arguments or
 *      -ENOMEM on out of memory
 *
 * Side effects:
 *      Pages for mapping are allocated.
 *
 *-----------------------------------------------------------------------------
 */

static int LinuxDriverAllocLockedMem(struct file *filp,
				     VMLinux *vmLinux,
                                     VMHost *vmHost,
                                     struct vm_area_struct *vma,
				     unsigned long size) {
   unsigned long i;
   struct VMHostMapping *map;
   unsigned long pdes;

   if (size > VMHOST_MAPPING_PD * VMHOST_MAPPING_PT) {
      return -EINVAL;
   }
restart:
   down(&vmLinux->lock4Gb);
   if (vmLinux->sizeLocked == 0) {
      up(&vmLinux->lock4Gb);
      /* First mapping must start at zero */
      if (compat_vm_pgoff(vma) != 0) {
         return -EINVAL;
      }
      map = HostIF_AllocPage();
      if (!map) {
         return -ENOMEM;
      }
      pdes = (size + VMHOST_MAPPING_PT - 1) / VMHOST_MAPPING_PT;
      for (i = 0; i < pdes; i++) {
         struct page* pg;
         void* addr;

         pg = alloc_page(GFP_HIGHUSER);
         if (!pg) {
            while (i-- > 0) {
               __free_page(map->ent[i]);
            }
            HostIF_FreePage(map);
	    return -ENOMEM;
         }
         addr = kmap(pg);
         memset(addr, 0, PAGE_SIZE);
         kunmap(pg);
         map->ent[i] = pg;
      }
      down(&vmLinux->lock4Gb);
      if (vmLinux->sizeLocked != 0) {
         up(&vmLinux->lock4Gb);
      
         while (i-- > 0) {
            __free_page(map->ent[i]);
         }
         HostIF_FreePage(map);
	 goto restart;
      }
      vmLinux->sizeLocked = size;
      vmLinux->pagesLocked = map;
      vmLinux->baseLocked = vma->vm_start;
   } else {
      /* Subsequent mappings must be fully contained in first one */
      if (compat_vm_pgoff(vma) != (vma->vm_start - vmLinux->baseLocked) >> PAGE_SHIFT ||
          compat_vm_pgoff(vma) >= vmLinux->sizeLocked ||
          size > vmLinux->sizeLocked ||
	  compat_vm_pgoff(vma) + size > vmLinux->sizeLocked) {
	 up(&vmLinux->lock4Gb);
	 return -EINVAL;
      }
      /* Add some checks for illegal mmaps */
   }
   up(&vmLinux->lock4Gb);
   vma->vm_ops = &vmuser_locked_mops;
   return 0;
}
